/* Copyright (c) 2015  Gerald Knizia
 * 
 * This file is part of the IboView program (see: http://www.iboview.org)
 * 
 * IboView is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, version 3.
 * 
 * IboView is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with IboView (LICENSE). If not, see http://www.gnu.org/licenses/
 * 
 * Please see IboView documentation in README.txt for:
 * -- A list of included external software and their licenses. The included
 *    external software's copyright is not touched by this agreement.
 * -- Notes on re-distribution and contributions to/further development of
 *    the IboView software
 */

/* IrAmrr.cpp v20200522 EST [mirage, Gerald Knizia] */
#include <stddef.h> // for size_t
#include "IrAmrr.h" // for cart_index_t, IR_RP, and assert

namespace ir {

// List of labels of solid harmonic Gaussians Slm(x,y,z) e^{-zeta r^2}
// in the order of spherical components 'c' (0 <= c < 2*l+1) for which this IR core
// was configured. Use pSlcLabel(l,c) function to access.
// (if needed, change order in SolidHarmonics.py and rebuild kernel routines)
static char const * s_pSlcLabels[49] = {
   "1s", "2px", "2py", "2pz", "3d0", "3d2-", "3d1+", "3d2+",
   "3d1-", "4f1+", "4f1-", "4f0", "4f3+", "4f2-", "4f3-", "4f2+",
   "5g0", "5g2-", "5g1+", "5g4+", "5g1-", "5g2+", "5g4-", "5g3+",
   "5g3-", "6h1+", "6h1-", "6h2+", "6h3+", "6h4-", "6h3-", "6h4+",
   "6h5-", "6h0", "6h5+", "6h2-", "7i6+", "7i2-", "7i5+", "7i4+",
   "7i5-", "7i2+", "7i6-", "7i3+", "7i4-", "7i0", "7i3-", "7i1-",
   "7i1+"
};

// Table of spherical component indices 'c' for Slc at which a Slm with
// given 'm' can be found (see m2c function)
static unsigned int s_data_m2c[49] = {
   0, 1, 2, 0, 1, 4, 0, 2,
   3, 5, 4, 1, 2, 0, 6, 3,
   6, 8, 1, 4, 0, 2, 5, 7,
   3, 7, 4, 5, 10, 1, 8, 0,
   2, 3, 6, 9, 6, 4, 8, 10,
   1, 11, 9, 12, 5, 7, 3, 2,
   0
};

// Table of real 'm' indices (for real solid harmonics in Helgaker-
// Joergensen-Olsen convention) to which a given Slc corresponds (see c2m
// function)
static signed int s_data_c2m[49] = {
   0, 1, -1, 0, 0, -2, 1, 2,
   -1, 1, -1, 0, 3, -2, -3, 2,
   0, -2, 1, 4, -1, 2, -4, 3,
   -3, 1, -1, 2, 3, -4, -3, 4,
   -5, 0, 5, -2, 6, -2, 5, 4,
   -5, 2, -6, 3, -4, 0, -3, -1,
   1
};

// Given l and c, return a descriptive label of the solid harmonic
// Gaussian Slc(x,y,z) exp(-zeta r^2)
char const *pSlcLabel(unsigned l, unsigned c)
{
   assert_rt(c < 2*l+1 && l <= unsigned(FSlcX::MAX_AngMom));
   assert(iSlcX(l,c) <= sizeof(s_data_c2m)/sizeof(s_data_c2m[0]));
   return s_pSlcLabels[iSlcX(l,c)];
}

// Given l and c, returns the 'm' component of the real solid harmonic
// S^l_m(R) (with m\in\{-l,...,+l\}) which is represented by the solid
// harmonic component S^l_c(R), where 'c' (with c\in\{0,1,...,2l\}) denotes
// a spherical component ('c') of the order decided at generation-time of IR
int Sl_c2m(unsigned l, unsigned c)
{
   assert_rt(c < 2*l+1 && l <= unsigned(FSlcX::MAX_AngMom));
   assert(iSlcX(l,c) <= sizeof(s_data_c2m)/sizeof(s_data_c2m[0]));
   return s_data_c2m[iSlcX(l,c)];
}

// Reverse of c2m: returns the component 'c' (with c\in\{0,1,..,2l\})
// for a given m\in\{-l,..,l\}
unsigned Sl_m2c(unsigned l, int m)
{
   assert_rt(-signed(l) <= m && m <= signed(l) && l <= unsigned(FSlcX::MAX_AngMom));
   assert(iSlmX(l,m) <= sizeof(s_data_m2c)/sizeof(s_data_m2c[0]));
   return s_data_m2c[iSlmX(l,m)];
}

// Constants for direct expansion of solid harmonics in terms of cartesians.
// These ones are good for L=0..6
static double const sd0 = .5;                      // = 0.5;
static double const sd1 = 1.7320508075688772935;   // = sqrt(3);
static double const sd2 = .86602540378443864676;   // = (0.5)*sqrt(3);
static double const sd3 = .61237243569579452455;   // = (0.25)*sqrt(6);
static double const sd4 = 2.4494897427831780982;   // = sqrt(6);
static double const sd5 = 1.5;                     // = 1.5;
static double const sd6 = .790569415042094833;     // = (0.25)*sqrt(10);
static double const sd7 = 2.371708245126284499;    // = (0.75)*sqrt(10);
static double const sd8 = 3.8729833462074168852;   // = sqrt(15);
static double const sd9 = 1.9364916731037084426;   // = (0.5)*sqrt(15);
static double const sda = .375;                    // = 0.375;
static double const sdb = .75;                     // = 0.75;
static double const sdc = 3.;                      // = 3;
static double const sdd = 1.1180339887498948482;   // = (0.5)*sqrt(5);
static double const sde = 6.7082039324993690892;   // = 3*sqrt(5);
static double const sdf = 3.162277660168379332;    // = sqrt(10);
static double const sd10 = .73950997288745200532;  // = (0.125)*sqrt(35);
static double const sd11 = 4.4370598373247120319;  // = (0.75)*sqrt(35);
static double const sd12 = .5590169943749474241;   // = (0.25)*sqrt(5);
static double const sd13 = 3.3541019662496845446;  // = (1.5)*sqrt(5);
static double const sd14 = 2.9580398915498080213;  // = (0.5)*sqrt(35);
static double const sd15 = 2.0916500663351888699;  // = (0.25)*sqrt(70);
static double const sd16 = 6.2749501990055666098;  // = (0.75)*sqrt(70);
static double const sd17 = .48412291827592711065;  // = (0.125)*sqrt(15);
static double const sd18 = .96824583655185422129;  // = (0.25)*sqrt(15);
static double const sd19 = 5.8094750193111253278;  // = (1.5)*sqrt(15);
static double const sd1a = 2.5617376914898995958;  // = (0.25)*sqrt(105);
static double const sd1b = 5.1234753829797991916;  // = (0.5)*sqrt(105);
static double const sd1c = .52291251658379721749;  // = (0.0625)*sqrt(70);
static double const sd1d = 1.045825033167594435;   // = (0.125)*sqrt(70);
static double const sd1e = 4.1833001326703777399;  // = (0.5)*sqrt(70);
static double const sd1f = 1.5687375497513916525;  // = (0.1875)*sqrt(70);
static double const sd20 = 12.54990039801113322;   // = (1.5)*sqrt(70);
static double const sd21 = 8.8741196746494240639;  // = (1.5)*sqrt(35);
static double const sd22 = 2.218529918662356016;   // = (0.375)*sqrt(35);
static double const sd23 = 13.311179511974136096;  // = (2.25)*sqrt(35);
static double const sd24 = 3.507803800100570049;   // = (0.9375)*sqrt(14);
static double const sd25 = 7.015607600201140098;   // = (1.875)*sqrt(14);
static double const sd26 = .7015607600201140098;   // = (0.1875)*sqrt(14);
static double const sd27 = 1.875;                  // = 1.875;
static double const sd28 = 3.75;                   // = 3.75;
static double const sd29 = 5.;                     // = 5;
static double const sd2a = 10.246950765959598383;  // = sqrt(105);
static double const sd2b = .67169328938139615748;  // = (0.03125)*sqrt(462);
static double const sd2c = 10.075399340720942362;  // = (0.46875)*sqrt(462);
static double const sd2d = .90571104663683991086;  // = (0.0625)*sqrt(210);
static double const sd2e = 1.8114220932736798217;  // = (0.125)*sqrt(210);
static double const sd2f = 14.491376746189438574;  // = sqrt(210);
static double const sd30 = 2.3268138086232856118;  // = (0.1875)*sqrt(154);
static double const sd31 = 23.268138086232856118;  // = (1.875)*sqrt(154);
static double const sd32 = 11.634069043116428059;  // = (0.9375)*sqrt(154);
static double const sd33 = .49607837082461073572;  // = (0.1875)*sqrt(7);
static double const sd34 = 2.4803918541230536786;  // = (0.9375)*sqrt(7);
static double const sd35 = 4.9607837082461073572;  // = (1.875)*sqrt(7);
static double const sd36 = 29.764702249476644143;  // = (11.25)*sqrt(7);
static double const sd37 = .45285552331841995543;  // = (0.03125)*sqrt(210);
static double const sd38 = 7.2456883730947192869;  // = (0.5)*sqrt(210);
static double const sd39 = 4.0301597362883769449;  // = (0.1875)*sqrt(462);
static double const sd3a = 13.43386578762792315;   // = (0.625)*sqrt(462);
static double const sd3b = 2.7171331399105197326;  // = (0.1875)*sqrt(210);
static double const sd3c = 5.4342662798210394651;  // = (0.375)*sqrt(210);
static double const sd3d = 8.1513994197315591977;  // = (0.5625)*sqrt(210);
static double const sd3e = 21.737065119284157861;  // = (1.5)*sqrt(210);
static double const sd3f = 1.9843134832984429429;  // = (0.75)*sqrt(7);
static double const sd40 = 19.843134832984429429;  // = (7.5)*sqrt(7);
static double const sd41 = .3125;                  // = 0.3125;
static double const sd42 = .9375;                  // = 0.9375;
static double const sd43 = 5.625;                  // = 5.625;
static double const sd44 = 11.25;                  // = 11.25;
static double const sd45 = 7.5;                    // = 7.5;
static double const sd46 = 2.8641098093474000041;  // = (0.625)*sqrt(21);
static double const sd47 = 5.7282196186948000082;  // = (1.25)*sqrt(21);
static double const sd48 = 11.456439237389600016;  // = (2.5)*sqrt(21);
static double const sd49 = 4.5825756949558400066;  // = sqrt(21);

// Constants for incremental expansion of solid harmonics in terms of x/y/z/rsq
// and other solid harmonics. These ones are good for L=0..6
static double const si0 = .5;                      // = 0.5;
static double const si1 = 1.5;                     // = 1.5;
static double const si2 = 1.7320508075688772935;   // = sqrt(3);
static double const si3 = .86602540378443864676;   // = (0.5)*sqrt(3);
static double const si4 = .61237243569579452455;   // = (0.25)*sqrt(6);
static double const si5 = 3.0618621784789726227;   // = (1.25)*sqrt(6);
static double const si6 = .66666666666666666667;   // = 2./3.;
static double const si7 = 1.6666666666666666667;   // = 5./3.;
static double const si8 = 1.581138830084189666;    // = (0.5)*sqrt(10);
static double const si9 = .91287092917527685576;   // = (1./6.)*sqrt(30);
static double const si10 = 3.8729833462074168852;  // = sqrt(15);
static double const si11 = 2.2360679774997896964;  // = sqrt(5);
static double const si12 = .75;                    // = 0.75;
static double const si13 = 1.75;                   // = 1.75;
static double const si14 = 1.1180339887498948482;  // = (0.5)*sqrt(5);
static double const si15 = 7.8262379212492639374;  // = (3.5)*sqrt(5);
static double const si16 = 1.8073922282301278797;  // = (7./15.)*sqrt(15);
static double const si17 = 1.2649110640673517328;  // = (0.4)*sqrt(10);
static double const si18 = .9354143466934853464;   // = (0.25)*sqrt(14);
static double const si19 = .6454972243679028142;   // = (1./6.)*sqrt(15);
static double const si20 = 2.0207259421636901758;  // = (7./6.)*sqrt(3);
static double const si21 = 2.6457513110645905905;  // = sqrt(7);
static double const si22 = .790569415042094833;    // = (0.25)*sqrt(10);
static double const si23 = 1.8371173070873835736;  // = (0.75)*sqrt(6);
static double const si24 = .75592894601845445443;  // = (2./7.)*sqrt(7);
static double const si25 = 1.9639610121239314314;  // = (3./7.)*sqrt(21);
static double const si26 = 2.25;                   // = 2.25;
static double const si27 = .66143782776614764763;  // = (0.25)*sqrt(7);
static double const si28 = .9486832980505137996;   // = (0.3)*sqrt(10);
static double const si29 = .8;                     // = 0.8;
static double const si30 = 1.8;                    // = 1.8;
static double const si31 = 2.9277002188455995381;  // = (2./7.)*sqrt(105);
static double const si32 = .95742710775633810997;  // = (1./6.)*sqrt(33);
static double const si33 = .81009258730098252887;  // = (0.125)*sqrt(42);
static double const si34 = 1.9445436482630056921;  // = (1.375)*sqrt(2);
static double const si35 = 3.3166247903553998491;  // = sqrt(11);
static double const si36 = .67082039324993690892;  // = (0.3)*sqrt(5);
static double const si37 = 2.459674775249768666;   // = (1.1)*sqrt(5);
static double const si38 = .76980035891950101935;  // = (4./9.)*sqrt(3);
static double const si39 = 2.1169509870286278032;  // = (11./9.)*sqrt(3);
static double const si40 = .83333333333333333333;  // = 5./6.;
static double const si41 = 1.8333333333333333333;  // = 11./6.;
static double const si42 = .82807867121082506136;  // = (2./35.)*sqrt(210);
static double const si43 = 1.8593393604027364705;  // = (11./35.)*sqrt(35);
static double const si44 = 1.2247448713915890491;  // = (0.5)*sqrt(6);
static double const si45 = 4.8989794855663561964;  // = 2*sqrt(6);
static double const si46 = 4.743416490252568998;   // = (1.5)*sqrt(10);
static double const si47 = 5.25;                   // = 5.25;
static double const si48 = 13.416407864998738178;  // = 6*sqrt(5);
static double const si49 = 2.5298221281347034656;  // = (0.8)*sqrt(10);
static double const si50 = 6.324555320336758664;   // = 2*sqrt(10);
static double const si51 = 4.4370598373247120319;  // = (0.75)*sqrt(35);
static double const si52 = 1.2909944487358056284;  // = (1./3.)*sqrt(15);
static double const si53 = 12.54990039801113322;   // = (1.5)*sqrt(70);
static double const si54 = 8.7142125289666879916;  // = (2.25)*sqrt(15);
static double const si55 = .96824583655185422129;  // = (0.25)*sqrt(15);
static double const si56 = 1.5118578920369089089;  // = (4./7.)*sqrt(7);
static double const si57 = 1.3228756555322952953;  // = (0.5)*sqrt(7);
static double const si58 = 28.237275895525049744;  // = (3.375)*sqrt(70);
static double const si59 = 3.1374750995027833049;  // = (0.375)*sqrt(70);
static double const si60 = 1.6;                    // = 1.6;
static double const si61 = 2.4;                    // = 2.4;
static double const si62 = 5.8554004376911990761;  // = (4./7.)*sqrt(105);
static double const si63 = 20.493901531919196766;  // = 2*sqrt(105);
static double const si64 = 1.6201851746019650577;  // = (0.25)*sqrt(42);
static double const si65 = 10.86853255964207893;   // = (0.75)*sqrt(210);
static double const si66 = 1.3416407864998738178;  // = (0.6)*sqrt(5);
static double const si67 = 1.5396007178390020387;  // = (8./9.)*sqrt(3);
static double const si68 = 9.6609178307929590491;  // = (2./3.)*sqrt(210);
static double const si69 = 1.6561573424216501227;  // = (4./35.)*sqrt(210);
static double const si70 = 3.9279220242478628628;  // = (6./7.)*sqrt(21);
static double const si71 = 3.6742346141747671473;  // = (1.5)*sqrt(6);
static double const si72 = 3.5;                    // = 3.5;
static double const si73 = 4.5;                    // = 4.5;
static double const si74 = 6.7082039324993690892;  // = 3*sqrt(5);
static double const si75 = 14.230249470757706994;  // = (4.5)*sqrt(10);
static double const si76 = 18.973665961010275992;  // = 6*sqrt(10);
static double const si77 = 1.8708286933869706928;  // = (0.5)*sqrt(14);
static double const si78 = 17.748239349298848128;  // = 3*sqrt(35);
static double const si79 = 4.4721359549995793928;  // = 2*sqrt(5);
static double const si80 = 4.0414518843273803516;  // = (7./3.)*sqrt(3);
static double const si81 = 3.162277660168379332;   // = sqrt(10);
static double const si82 = 26.142637586900063975;  // = (6.75)*sqrt(15);
static double const si83 = 2.9047375096555626639;  // = (0.75)*sqrt(15);
static double const si84 = 19.364916731037084426;  // = 5*sqrt(15);
static double const si85 = 1.9364916731037084426;  // = (0.5)*sqrt(15);
static double const si86 = 7.7459666924148337704;  // = 2*sqrt(15);
static double const si87 = 23.237900077244501311;  // = 6*sqrt(15);
static double const si88 = 11.710800875382398152;  // = (8./7.)*sqrt(105);
static double const si89 = 3.0237157840738178177;  // = (8./7.)*sqrt(7);
static double const si90 = 6.2749501990055666098;  // = (0.75)*sqrt(70);
static double const si91 = 50.199601592044532879;  // = 6*sqrt(70);
static double const si92 = 53.244718047896544383;  // = 9*sqrt(35);
static double const si93 = 1.8973665961010275992;  // = (0.6)*sqrt(10);
static double const si94 = 16.837458240482736235;  // = (4.5)*sqrt(14);
static double const si95 = 9.6;                    // = 9.6;
static double const si96 = 3.6;                    // = 3.6;
static double const si97 = 4.8;                    // = 4.8;
static double const si98 = 3.2;                    // = 3.2;
static double const si99 = 16.8;                   // = 16.8;
static double const si100 = 30.74085229787879515;  // = 3*sqrt(105);
static double const si101 = 61.481704595757590299; // = 6*sqrt(105);
static double const si102 = 1.9148542155126762199; // = (1./3.)*sqrt(33);
static double const si103 = 3.2403703492039301155; // = (0.5)*sqrt(42);
static double const si104 = 59.776929078031434117; // = (4.125)*sqrt(210);
static double const si105 = 5.4342662798210394651; // = (0.375)*sqrt(210);
static double const si106 = 3.8890872965260113842; // = (2.75)*sqrt(2);
static double const si107 = 76.079727917494552512; // = (5.25)*sqrt(210);
static double const si108 = 21.737065119284157861; // = (1.5)*sqrt(210);
static double const si109 = 2.6832815729997476357; // = (1.2)*sqrt(5);
static double const si110 = 4.9193495504995373321; // = (2.2)*sqrt(5);
static double const si111 = 130.96468989769723423; // = (49.5)*sqrt(7);
static double const si112 = 11.905880899790657657; // = (4.5)*sqrt(7);
static double const si113 = 3.0792014356780040774; // = (16./9.)*sqrt(3);
static double const si114 = 38.643671323171836197; // = (8./3.)*sqrt(210);
static double const si115 = 4.2339019740572556064; // = (22./9.)*sqrt(3);
static double const si116 = 19.321835661585918098; // = (4./3.)*sqrt(210);
static double const si117 = 86.948260477136631442; // = 6*sqrt(210);
static double const si118 = 3.3333333333333333333; // = 10./3.;
static double const si119 = 3.6666666666666666667; // = 11./3.;
static double const si120 = 2.5;                   // = 2.5;
static double const si121 = 27.5;                  // = 27.5;
static double const si122 = 15.711688096991451451; // = (24./7.)*sqrt(21);
static double const si123 = 3.3123146848433002454; // = (8./35.)*sqrt(210);
static double const si124 = 11.783766072743588588; // = (18./7.)*sqrt(21);
static double const si125 = 3.718678720805472941;  // = (22./35.)*sqrt(35);
static double const si126 = 7.8558440484957257256; // = (12./7.)*sqrt(21);
static double const si127 = 35.351298218230765765; // = (54./7.)*sqrt(21);
static double const si128 = 17.428425057933375983; // = (4.5)*sqrt(15);
static double const si129 = 11.618950038622250656; // = 3*sqrt(15);
static double const si130 = 92.951600308978005244; // = 24*sqrt(15);
static double const si131 = 69.713700231733503933; // = 18*sqrt(15);
static double const si132 = 4.5355736761107267266; // = (12./7.)*sqrt(7);
static double const si133 = 5.8918830363717942942; // = (9./7.)*sqrt(21);
static double const si134 = 18.82485059701669983;  // = (2.25)*sqrt(70);
static double const si135 = 3.9686269665968858858; // = (1.5)*sqrt(7);
static double const si136 = 37.649701194033399659; // = (4.5)*sqrt(70);
static double const si137 = 84.187291202413681176; // = (22.5)*sqrt(14);
static double const si138 = 2.8460498941515413988; // = (0.9)*sqrt(10);
static double const si139 = 5.4;                   // = 5.4;
static double const si140 = 14.4;                  // = 14.4;
static double const si141 = 2.8722813232690143299; // = (0.5)*sqrt(33);
static double const si142 = 80.603194725767538897; // = (3.75)*sqrt(462);
static double const si143 = 4.8605555238058951732; // = (0.75)*sqrt(42);
static double const si144 = 32.605597678926236791; // = (2.25)*sqrt(210);
static double const si145 = 347.79304190854652577; // = 24*sqrt(210);
static double const si146 = 173.89652095427326288; // = 12*sqrt(210);
static double const si147 = 279.21765703479427341; // = (22.5)*sqrt(154);
static double const si148 = 4.0249223594996214535; // = (1.8)*sqrt(5);
static double const si149 = 47.623523599162630629; // = 18*sqrt(7);
static double const si150 = 238.11761799581315315; // = 90*sqrt(7);
static double const si151 = 119.55385815606286823; // = (8.25)*sqrt(210);
static double const si152 = 5.8336309447890170763; // = (4.125)*sqrt(2);
static double const si153 = 43.474130238568315721; // = 3*sqrt(210);
static double const si154 = 4.6188021535170061161; // = (8./3.)*sqrt(3);
static double const si155 = 57.965506984757754295; // = 4*sqrt(210);
static double const si156 = 195.63358607355742075; // = (13.5)*sqrt(210);
static double const si157 = 6.3508529610858834096; // = (11./3.)*sqrt(3);
static double const si158 = 65.211195357852473582; // = (4.5)*sqrt(210);
static double const si159 = 260.84478143140989433; // = 18*sqrt(210);
static double const si160 = 71.435285398743945944; // = 27*sqrt(7);
static double const si161 = 23.811761799581315315; // = 9*sqrt(7);
static double const si162 = 7.5;                   // = 7.5;
static double const si163 = 82.5;                  // = 82.5;
static double const si164 = 5.5;                   // = 5.5;
static double const si165 = 68.738635424337600099; // = 15*sqrt(21);
static double const si166 = 4.9684720272649503681; // = (12./35.)*sqrt(210);
static double const si167 = 70.70259643646153153;  // = (108./7.)*sqrt(21);
static double const si168 = 5.5780180812082094116; // = (33./35.)*sqrt(35);
static double const si169 = 78.558440484957257256; // = (120./7.)*sqrt(21);
static double const si170 = 106.0538946546922973;  // = (162./7.)*sqrt(21);
static double const si171 = 137.4772708486752002;  // = 30*sqrt(21);
static double const si172 = 31.423376193982902902; // = (48./7.)*sqrt(21);

static inline double sqr(double x) { IR_SUPPRESS_UNUSED_WARNING(sqr); return x*x; }
static inline double cube(double x) { IR_SUPPRESS_UNUSED_WARNING(cube); return x*x*x; }
static inline double pow4(double x) { IR_SUPPRESS_UNUSED_WARNING(pow4); double xx = x*x; return xx*xx; }

// Calculate Slc(x,y,z) for all l,m with l <= L.
// Output is addressed as pOut[iSlcX(l,c)]
void EvalSlcX_Deriv0(double *IR_RP pOut, double x, double y, double z, unsigned L)
{
   // Slm: [S(0,+0)]
   pOut[0] = 1;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 0) return;

   // Slm: [S(1,+1), S(1,-1), S(1,+0)]
   pOut[1] = x;
   pOut[2] = y;
   pOut[3] = z;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 1) return;

   double const xx = x*x;
   double const yy = y*y;
   double const zz = z*z;
   double const rsq = xx + yy + zz;
   // Slm: [S(2,+0), S(2,-2), S(2,+1), S(2,+2), S(2,-1)]
   pOut[4] = -rsq*si0 + si1*zz;
   pOut[5] = si2*x*y;
   pOut[6] = si2*x*z;
   pOut[7] = si3*(xx - yy);
   pOut[8] = si2*y*z;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 2) return;

   // Slm: [S(3,+1), S(3,-1), S(3,+0), S(3,+3), S(3,-2), S(3,-3), S(3,+2)]
   pOut[9] = x*(-rsq*si4 + si5*zz);
   pOut[10] = y*(-rsq*si4 + si5*zz);
   pOut[11] = z*(pOut[4]*si7 - rsq*si6);
   pOut[12] = x*(pOut[7]*si9 - si8*yy);
   pOut[13] = si10*x*y*z;
   pOut[14] = y*(pOut[7]*si9 + si8*xx);
   pOut[15] = pOut[7]*si11*z;
   // 1.29 steps per component, 0.00 kb stack
   if (L == 3) return;

   // Slm: [S(4,+0), S(4,-2), S(4,+1), S(4,+4), S(4,-1), S(4,+2), S(4,-4), S(4,+3), S(4,-3)]
   pOut[16] = pOut[11]*si13*z - pOut[4]*rsq*si12;
   pOut[17] = x*y*(-rsq*si14 + si15*zz);
   pOut[18] = z*(pOut[9]*si16 - rsq*si17*x);
   pOut[19] = si18*(pOut[12]*x - pOut[14]*y);
   pOut[20] = z*(pOut[10]*si16 - rsq*si17*y);
   pOut[21] = pOut[15]*si20*z - pOut[7]*rsq*si19;
   pOut[22] = si18*(pOut[12]*y + pOut[14]*x);
   pOut[23] = pOut[12]*si21*z;
   pOut[24] = pOut[14]*si21*z;
   // 1.89 steps per component, 0.00 kb stack
   if (L == 4) return;

   // Slm: [S(5,+1), S(5,-1), S(5,+2), S(5,+3), S(5,-4), S(5,-3), S(5,+4), S(5,-5), S(5,+0), S(5,+5), S(5,-2)]
   pOut[25] = pOut[18]*si23*z - pOut[9]*rsq*si22;
   pOut[26] = -pOut[10]*rsq*si22 + pOut[20]*si23*z;
   pOut[27] = -pOut[15]*rsq*si24 + pOut[21]*si25*z;
   pOut[28] = -pOut[12]*rsq*si27 + pOut[23]*si26*z;
   pOut[29] = 3*pOut[22]*z;
   pOut[30] = -pOut[14]*rsq*si27 + pOut[24]*si26*z;
   pOut[31] = 3*pOut[19]*z;
   pOut[32] = si28*(pOut[19]*y + pOut[22]*x);
   pOut[33] = -pOut[11]*rsq*si29 + pOut[16]*si30*z;
   pOut[34] = si28*(pOut[19]*x - pOut[22]*y);
   pOut[35] = z*(pOut[17]*si25 - rsq*si31*x*y);
   // 2.55 steps per component, 0.00 kb stack
   if (L == 5) return;

   // Slm: [S(6,+6), S(6,-2), S(6,+5), S(6,+4), S(6,-5), S(6,+2), S(6,-6), S(6,+3), S(6,-4), S(6,+0), S(6,-3), S(6,-1), S(6,+1)]
   pOut[36] = si32*(-pOut[32]*y + pOut[34]*x);
   pOut[37] = -pOut[17]*rsq*si33 + pOut[35]*si34*z;
   pOut[38] = pOut[34]*si35*z;
   pOut[39] = -pOut[19]*rsq*si36 + pOut[31]*si37*z;
   pOut[40] = pOut[32]*si35*z;
   pOut[41] = -pOut[21]*rsq*si33 + pOut[27]*si34*z;
   pOut[42] = si32*(pOut[32]*x + pOut[34]*y);
   pOut[43] = -pOut[23]*rsq*si38 + pOut[28]*si39*z;
   pOut[44] = -pOut[22]*rsq*si36 + pOut[29]*si37*z;
   pOut[45] = -pOut[16]*rsq*si40 + pOut[33]*si41*z;
   pOut[46] = -pOut[24]*rsq*si38 + pOut[30]*si39*z;
   pOut[47] = -pOut[20]*rsq*si42 + pOut[26]*si43*z;
   pOut[48] = -pOut[18]*rsq*si42 + pOut[25]*si43*z;
   // 3.15 steps per component, 0.00 kb stack
   if (L == 6) return;

   assert(0);
}

// Calculate Slc(x,y,z) and 1st derivatives for all l,m with l <= L.
// Output is addressed as pOut[iSlcX(l,c)*NCOMP_Deriv1 + iComp] with iComp \in {0,1,...,3} indexed by FSlcX::COMP__*
// This function evaluates derivative components:
//    [1, d/dx, d/dy, d/dz] Slc(x,y,z)
void EvalSlcX_Deriv1(double *IR_RP pOut, double x, double y, double z, unsigned L)
{
   // Slm: [S(0,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[0] = 1;
   pOut[1] = 0;
   pOut[2] = 0;
   pOut[3] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 0) return;

   // Slm: [S(1,+1), S(1,-1), S(1,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[4] = x;
   pOut[5] = 1;
   pOut[6] = 0;
   pOut[7] = 0;
   pOut[8] = y;
   pOut[9] = 0;
   pOut[10] = 1;
   pOut[11] = 0;
   pOut[12] = z;
   pOut[13] = 0;
   pOut[14] = 0;
   pOut[15] = 1;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 1) return;

   double const xx = x*x;
   double const yy = y*y;
   double const zz = z*z;
   double const rsq = xx + yy + zz;
   // Slm: [S(2,+0), S(2,-2), S(2,+1), S(2,+2), S(2,-1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[16] = -rsq*si0 + si1*zz;
   pOut[17] = -x;
   pOut[18] = -y;
   pOut[19] = 2*z;
   pOut[20] = si2*x*y;
   pOut[21] = si2*y;
   pOut[22] = si2*x;
   pOut[23] = 0;
   pOut[24] = si2*x*z;
   pOut[25] = si2*z;
   pOut[26] = 0;
   pOut[27] = si2*x;
   pOut[28] = si3*(xx - yy);
   pOut[29] = si2*x;
   pOut[30] = -si2*y;
   pOut[31] = 0;
   pOut[32] = si2*y*z;
   pOut[33] = 0;
   pOut[34] = si2*z;
   pOut[35] = si2*y;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 2) return;

   // Slm: [S(3,+1), S(3,-1), S(3,+0), S(3,+3), S(3,-2), S(3,-3), S(3,+2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[36] = x*(-rsq*si4 + si5*zz);
   pOut[37] = -rsq*si4 - si44*xx + si5*zz;
   pOut[38] = -si44*x*y;
   pOut[39] = si45*x*z;
   pOut[40] = y*(-rsq*si4 + si5*zz);
   pOut[41] = -si44*x*y;
   pOut[42] = -rsq*si4 - si44*yy + si5*zz;
   pOut[43] = si45*y*z;
   pOut[44] = z*(pOut[16]*si7 - rsq*si6);
   pOut[45] = -3*x*z;
   pOut[46] = -3*y*z;
   pOut[47] = pOut[16]*si7 - rsq*si6 + 2*zz;
   pOut[48] = x*(pOut[28]*si9 - si8*yy);
   pOut[49] = pOut[28]*si9 + si8*xx - si8*yy;
   pOut[50] = -si46*x*y;
   pOut[51] = 0;
   pOut[52] = si10*x*y*z;
   pOut[53] = si10*y*z;
   pOut[54] = si10*x*z;
   pOut[55] = si10*x*y;
   pOut[56] = y*(pOut[28]*si9 + si8*xx);
   pOut[57] = si46*x*y;
   pOut[58] = pOut[28]*si9 + si8*xx - si8*yy;
   pOut[59] = 0;
   pOut[60] = pOut[28]*si11*z;
   pOut[61] = si10*x*z;
   pOut[62] = -si10*y*z;
   pOut[63] = pOut[28]*si11;
   // 1.07 steps per component, 0.00 kb stack
   if (L == 3) return;

   // Slm: [S(4,+0), S(4,-2), S(4,+1), S(4,+4), S(4,-1), S(4,+2), S(4,-4), S(4,+3), S(4,-3)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[64] = -pOut[16]*rsq*si12 + pOut[44]*si13*z;
   pOut[65] = x*(-pOut[16]*si1 + rsq*si12 - si47*zz);
   pOut[66] = y*(-pOut[16]*si1 + rsq*si12 - si47*zz);
   pOut[67] = -pOut[16]*si1*z + pOut[44]*si13 + pOut[47]*si13*z - rsq*si1*z;
   pOut[68] = x*y*(-rsq*si14 + si15*zz);
   pOut[69] = y*(-rsq*si14 - si11*xx + si15*zz);
   pOut[70] = x*(-rsq*si14 - si11*yy + si15*zz);
   pOut[71] = si48*x*y*z;
   pOut[72] = z*(pOut[36]*si16 - rsq*si17*x);
   pOut[73] = z*(pOut[37]*si16 - rsq*si17 - si49*xx);
   pOut[74] = -si46*x*y*z;
   pOut[75] = pOut[36]*si16 - rsq*si17*x + si50*x*zz;
   pOut[76] = si18*(pOut[48]*x - pOut[56]*y);
   pOut[77] = pOut[48]*si18 + pOut[49]*si18*x - si51*x*yy;
   pOut[78] = -pOut[56]*si18 - pOut[58]*si18*y - si51*xx*y;
   pOut[79] = 0;
   pOut[80] = z*(pOut[40]*si16 - rsq*si17*y);
   pOut[81] = -si46*x*y*z;
   pOut[82] = z*(pOut[42]*si16 - rsq*si17 - si49*yy);
   pOut[83] = pOut[40]*si16 - rsq*si17*y + si50*y*zz;
   pOut[84] = -pOut[28]*rsq*si19 + pOut[60]*si20*z;
   pOut[85] = x*(-pOut[28]*si52 - rsq*si14 + si15*zz);
   pOut[86] = y*(-pOut[28]*si52 + rsq*si14 - si15*zz);
   pOut[87] = -pOut[28]*si52*z + pOut[60]*si20 + pOut[63]*si20*z;
   pOut[88] = si18*(pOut[48]*y + pOut[56]*x);
   pOut[89] = pOut[49]*si18*y + pOut[56]*si18 + si51*xx*y;
   pOut[90] = pOut[48]*si18 + pOut[58]*si18*x - si51*x*yy;
   pOut[91] = 0;
   pOut[92] = pOut[48]*si21*z;
   pOut[93] = pOut[49]*si21*z;
   pOut[94] = -si53*x*y*z;
   pOut[95] = pOut[48]*si21;
   pOut[96] = pOut[56]*si21*z;
   pOut[97] = si53*x*y*z;
   pOut[98] = pOut[58]*si21*z;
   pOut[99] = pOut[56]*si21;
   // 1.39 steps per component, 0.00 kb stack
   if (L == 4) return;

   // Slm: [S(5,+1), S(5,-1), S(5,+2), S(5,+3), S(5,-4), S(5,-3), S(5,+4), S(5,-5), S(5,+0), S(5,+5), S(5,-2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[100] = -pOut[36]*rsq*si22 + pOut[72]*si23*z;
   pOut[101] = -pOut[36]*si8*x - pOut[37]*rsq*si22 + pOut[73]*si23*z;
   pOut[102] = y*(-pOut[36]*si8 + rsq*si55*x - si54*x*zz);
   pOut[103] = -pOut[36]*si8*z + pOut[72]*si23 + pOut[75]*si23*z - rsq*si10*x*z;
   pOut[104] = -pOut[40]*rsq*si22 + pOut[80]*si23*z;
   pOut[105] = x*(-pOut[40]*si8 + rsq*si55*y - si54*y*zz);
   pOut[106] = -pOut[40]*si8*y - pOut[42]*rsq*si22 + pOut[82]*si23*z;
   pOut[107] = -pOut[40]*si8*z + pOut[80]*si23 + pOut[83]*si23*z - rsq*si10*y*z;
   pOut[108] = -pOut[60]*rsq*si24 + pOut[84]*si25*z;
   pOut[109] = -pOut[60]*si56*x + pOut[85]*si25*z - rsq*si31*x*z;
   pOut[110] = -pOut[60]*si56*y + pOut[86]*si25*z + rsq*si31*y*z;
   pOut[111] = -pOut[60]*si56*z - pOut[63]*rsq*si24 + pOut[84]*si25 + pOut[87]*si25*z;
   pOut[112] = -pOut[48]*rsq*si27 + pOut[92]*si26*z;
   pOut[113] = -pOut[48]*si57*x - pOut[49]*rsq*si27 + pOut[93]*si26*z;
   pOut[114] = y*(-pOut[48]*si57 + rsq*si59*x - si58*x*zz);
   pOut[115] = -pOut[48]*si57*z + pOut[92]*si26 + pOut[95]*si26*z;
   pOut[116] = 3*pOut[88]*z;
   pOut[117] = 3*pOut[89]*z;
   pOut[118] = 3*pOut[90]*z;
   pOut[119] = 3*pOut[88];
   pOut[120] = -pOut[56]*rsq*si27 + pOut[96]*si26*z;
   pOut[121] = x*(-pOut[56]*si57 - rsq*si59*y + si58*y*zz);
   pOut[122] = -pOut[56]*si57*y - pOut[58]*rsq*si27 + pOut[98]*si26*z;
   pOut[123] = -pOut[56]*si57*z + pOut[96]*si26 + pOut[99]*si26*z;
   pOut[124] = 3*pOut[76]*z;
   pOut[125] = 3*pOut[77]*z;
   pOut[126] = 3*pOut[78]*z;
   pOut[127] = 3*pOut[76];
   pOut[128] = si28*(pOut[76]*y + pOut[88]*x);
   pOut[129] = si28*(pOut[77]*y + pOut[88] + pOut[89]*x);
   pOut[130] = si28*(pOut[76] + pOut[78]*y + pOut[90]*x);
   pOut[131] = 0;
   pOut[132] = -pOut[44]*rsq*si29 + pOut[64]*si30*z;
   pOut[133] = -pOut[44]*si60*x + pOut[65]*si30*z + rsq*si61*x*z;
   pOut[134] = -pOut[44]*si60*y + pOut[66]*si30*z + rsq*si61*y*z;
   pOut[135] = -pOut[44]*si60*z - pOut[47]*rsq*si29 + pOut[64]*si30 + pOut[67]*si30*z;
   pOut[136] = si28*(pOut[76]*x - pOut[88]*y);
   pOut[137] = si28*(pOut[76] + pOut[77]*x - pOut[89]*y);
   pOut[138] = si28*(pOut[78]*x - pOut[88] - pOut[90]*y);
   pOut[139] = 0;
   pOut[140] = z*(pOut[68]*si25 - rsq*si31*x*y);
   pOut[141] = z*(pOut[69]*si25 - rsq*si31*y - si62*xx*y);
   pOut[142] = z*(pOut[70]*si25 - rsq*si31*x - si62*x*yy);
   pOut[143] = pOut[68]*si25 - rsq*si31*x*y + si63*x*y*zz;
   // 1.98 steps per component, 0.00 kb stack
   if (L == 5) return;

   // Slm: [S(6,+6), S(6,-2), S(6,+5), S(6,+4), S(6,-5), S(6,+2), S(6,-6), S(6,+3), S(6,-4), S(6,+0), S(6,-3), S(6,-1), S(6,+1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z]]
   pOut[144] = si32*(-pOut[128]*y + pOut[136]*x);
   pOut[145] = si32*(-pOut[129]*y + pOut[136] + pOut[137]*x);
   pOut[146] = si32*(-pOut[128] - pOut[130]*y + pOut[138]*x);
   pOut[147] = 0;
   pOut[148] = pOut[140]*si34*z - pOut[68]*rsq*si33;
   pOut[149] = pOut[141]*si34*z - pOut[68]*si64*x - pOut[69]*rsq*si33;
   pOut[150] = pOut[142]*si34*z - pOut[68]*si64*y - pOut[70]*rsq*si33;
   pOut[151] = pOut[140]*si34 + pOut[143]*si34*z - pOut[68]*si64*z - rsq*si65*x*y*z;
   pOut[152] = pOut[136]*si35*z;
   pOut[153] = pOut[137]*si35*z;
   pOut[154] = pOut[138]*si35*z;
   pOut[155] = pOut[136]*si35;
   pOut[156] = pOut[124]*si37*z - pOut[76]*rsq*si36;
   pOut[157] = pOut[125]*si37*z - pOut[76]*si66*x - pOut[77]*rsq*si36;
   pOut[158] = pOut[126]*si37*z - pOut[76]*si66*y - pOut[78]*rsq*si36;
   pOut[159] = pOut[124]*si37 + pOut[127]*si37*z - pOut[76]*si66*z;
   pOut[160] = pOut[128]*si35*z;
   pOut[161] = pOut[129]*si35*z;
   pOut[162] = pOut[130]*si35*z;
   pOut[163] = pOut[128]*si35;
   pOut[164] = pOut[108]*si34*z - pOut[84]*rsq*si33;
   pOut[165] = pOut[109]*si34*z - pOut[84]*si64*x - pOut[85]*rsq*si33;
   pOut[166] = pOut[110]*si34*z - pOut[84]*si64*y - pOut[86]*rsq*si33;
   pOut[167] = pOut[108]*si34 + pOut[111]*si34*z - pOut[84]*si64*z - pOut[87]*rsq*si33;
   pOut[168] = si32*(pOut[128]*x + pOut[136]*y);
   pOut[169] = si32*(pOut[128] + pOut[129]*x + pOut[137]*y);
   pOut[170] = si32*(pOut[130]*x + pOut[136] + pOut[138]*y);
   pOut[171] = 0;
   pOut[172] = pOut[112]*si39*z - pOut[92]*rsq*si38;
   pOut[173] = pOut[113]*si39*z - pOut[92]*si67*x - pOut[93]*rsq*si38;
   pOut[174] = pOut[114]*si39*z - pOut[92]*si67*y + rsq*si68*x*y*z;
   pOut[175] = pOut[112]*si39 + pOut[115]*si39*z - pOut[92]*si67*z - pOut[95]*rsq*si38;
   pOut[176] = pOut[116]*si37*z - pOut[88]*rsq*si36;
   pOut[177] = pOut[117]*si37*z - pOut[88]*si66*x - pOut[89]*rsq*si36;
   pOut[178] = pOut[118]*si37*z - pOut[88]*si66*y - pOut[90]*rsq*si36;
   pOut[179] = pOut[116]*si37 + pOut[119]*si37*z - pOut[88]*si66*z;
   pOut[180] = pOut[132]*si41*z - pOut[64]*rsq*si40;
   pOut[181] = pOut[133]*si41*z - pOut[64]*si7*x - pOut[65]*rsq*si40;
   pOut[182] = pOut[134]*si41*z - pOut[64]*si7*y - pOut[66]*rsq*si40;
   pOut[183] = pOut[132]*si41 + pOut[135]*si41*z - pOut[64]*si7*z - pOut[67]*rsq*si40;
   pOut[184] = pOut[120]*si39*z - pOut[96]*rsq*si38;
   pOut[185] = pOut[121]*si39*z - pOut[96]*si67*x - rsq*si68*x*y*z;
   pOut[186] = pOut[122]*si39*z - pOut[96]*si67*y - pOut[98]*rsq*si38;
   pOut[187] = pOut[120]*si39 + pOut[123]*si39*z - pOut[96]*si67*z - pOut[99]*rsq*si38;
   pOut[188] = pOut[104]*si43*z - pOut[80]*rsq*si42;
   pOut[189] = pOut[105]*si43*z - pOut[80]*si69*x + rsq*si70*x*y*z;
   pOut[190] = pOut[106]*si43*z - pOut[80]*si69*y - pOut[82]*rsq*si42;
   pOut[191] = pOut[104]*si43 + pOut[107]*si43*z - pOut[80]*si69*z - pOut[83]*rsq*si42;
   pOut[192] = pOut[100]*si43*z - pOut[72]*rsq*si42;
   pOut[193] = pOut[101]*si43*z - pOut[72]*si69*x - pOut[73]*rsq*si42;
   pOut[194] = pOut[102]*si43*z - pOut[72]*si69*y + rsq*si70*x*y*z;
   pOut[195] = pOut[100]*si43 + pOut[103]*si43*z - pOut[72]*si69*z - pOut[75]*rsq*si42;
   // 2.63 steps per component, 0.00 kb stack
   if (L == 6) return;

   assert(0);
}

// Calculate Slc(x,y,z) and 1st+2nd derivatives for all l,m with l <= L.
// Output is addressed as pOut[iSlcX(l,c)*NCOMP_Deriv2 + iComp] with iComp \in {0,1,...,9} indexed by FSlcX::COMP__*
// This function evaluates derivative components:
//    [1, d/dx, d/dy, d/dz, d^2/dxx, d^2/dyy, d^2/dzz, d^2/dxy, d^2/dxz, d^2/dyz] Slc(x,y,z)
void EvalSlcX_Deriv2(double *IR_RP pOut, double x, double y, double z, unsigned L)
{
   // Slm: [S(0,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[0] = 1;
   pOut[1] = 0;
   pOut[2] = 0;
   pOut[3] = 0;
   pOut[4] = 0;
   pOut[5] = 0;
   pOut[6] = 0;
   pOut[7] = 0;
   pOut[8] = 0;
   pOut[9] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 0) return;

   // Slm: [S(1,+1), S(1,-1), S(1,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[10] = x;
   pOut[11] = 1;
   pOut[12] = 0;
   pOut[13] = 0;
   pOut[14] = 0;
   pOut[15] = 0;
   pOut[16] = 0;
   pOut[17] = 0;
   pOut[18] = 0;
   pOut[19] = 0;
   pOut[20] = y;
   pOut[21] = 0;
   pOut[22] = 1;
   pOut[23] = 0;
   pOut[24] = 0;
   pOut[25] = 0;
   pOut[26] = 0;
   pOut[27] = 0;
   pOut[28] = 0;
   pOut[29] = 0;
   pOut[30] = z;
   pOut[31] = 0;
   pOut[32] = 0;
   pOut[33] = 1;
   pOut[34] = 0;
   pOut[35] = 0;
   pOut[36] = 0;
   pOut[37] = 0;
   pOut[38] = 0;
   pOut[39] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 1) return;

   double const xx = x*x;
   double const yy = y*y;
   double const zz = z*z;
   double const rsq = xx + yy + zz;
   // Slm: [S(2,+0), S(2,-2), S(2,+1), S(2,+2), S(2,-1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[40] = -rsq*si0 + si1*zz;
   pOut[41] = -x;
   pOut[42] = -y;
   pOut[43] = 2*z;
   pOut[44] = -1;
   pOut[45] = -1;
   pOut[46] = 2;
   pOut[47] = 0;
   pOut[48] = 0;
   pOut[49] = 0;
   pOut[50] = si2*x*y;
   pOut[51] = si2*y;
   pOut[52] = si2*x;
   pOut[53] = 0;
   pOut[54] = 0;
   pOut[55] = 0;
   pOut[56] = 0;
   pOut[57] = si2;
   pOut[58] = 0;
   pOut[59] = 0;
   pOut[60] = si2*x*z;
   pOut[61] = si2*z;
   pOut[62] = 0;
   pOut[63] = si2*x;
   pOut[64] = 0;
   pOut[65] = 0;
   pOut[66] = 0;
   pOut[67] = 0;
   pOut[68] = si2;
   pOut[69] = 0;
   pOut[70] = si3*(xx - yy);
   pOut[71] = si2*x;
   pOut[72] = -si2*y;
   pOut[73] = 0;
   pOut[74] = si2;
   pOut[75] = -si2;
   pOut[76] = 0;
   pOut[77] = 0;
   pOut[78] = 0;
   pOut[79] = 0;
   pOut[80] = si2*y*z;
   pOut[81] = 0;
   pOut[82] = si2*z;
   pOut[83] = si2*y;
   pOut[84] = 0;
   pOut[85] = 0;
   pOut[86] = 0;
   pOut[87] = 0;
   pOut[88] = 0;
   pOut[89] = si2;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 2) return;

   // Slm: [S(3,+1), S(3,-1), S(3,+0), S(3,+3), S(3,-2), S(3,-3), S(3,+2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[90] = x*(-rsq*si4 + si5*zz);
   pOut[91] = -rsq*si4 - si44*xx + si5*zz;
   pOut[92] = -si44*x*y;
   pOut[93] = si45*x*z;
   pOut[94] = -si71*x;
   pOut[95] = -si44*x;
   pOut[96] = si45*x;
   pOut[97] = -si44*y;
   pOut[98] = si45*z;
   pOut[99] = 0;
   pOut[100] = y*(-rsq*si4 + si5*zz);
   pOut[101] = -si44*x*y;
   pOut[102] = -rsq*si4 - si44*yy + si5*zz;
   pOut[103] = si45*y*z;
   pOut[104] = -si44*y;
   pOut[105] = -si71*y;
   pOut[106] = si45*y;
   pOut[107] = -si44*x;
   pOut[108] = 0;
   pOut[109] = si45*z;
   pOut[110] = z*(pOut[40]*si7 - rsq*si6);
   pOut[111] = -3*x*z;
   pOut[112] = -3*y*z;
   pOut[113] = pOut[40]*si7 - rsq*si6 + 2*zz;
   pOut[114] = -3*z;
   pOut[115] = -3*z;
   pOut[116] = 6*z;
   pOut[117] = 0;
   pOut[118] = -3*x;
   pOut[119] = -3*y;
   pOut[120] = x*(pOut[70]*si9 - si8*yy);
   pOut[121] = pOut[70]*si9 + si8*xx - si8*yy;
   pOut[122] = -si46*x*y;
   pOut[123] = 0;
   pOut[124] = si46*x;
   pOut[125] = -si46*x;
   pOut[126] = 0;
   pOut[127] = -si46*y;
   pOut[128] = 0;
   pOut[129] = 0;
   pOut[130] = si10*x*y*z;
   pOut[131] = si10*y*z;
   pOut[132] = si10*x*z;
   pOut[133] = si10*x*y;
   pOut[134] = 0;
   pOut[135] = 0;
   pOut[136] = 0;
   pOut[137] = si10*z;
   pOut[138] = si10*y;
   pOut[139] = si10*x;
   pOut[140] = y*(pOut[70]*si9 + si8*xx);
   pOut[141] = si46*x*y;
   pOut[142] = pOut[70]*si9 + si8*xx - si8*yy;
   pOut[143] = 0;
   pOut[144] = si46*y;
   pOut[145] = -si46*y;
   pOut[146] = 0;
   pOut[147] = si46*x;
   pOut[148] = 0;
   pOut[149] = 0;
   pOut[150] = pOut[70]*si11*z;
   pOut[151] = si10*x*z;
   pOut[152] = -si10*y*z;
   pOut[153] = pOut[70]*si11;
   pOut[154] = si10*z;
   pOut[155] = -si10*z;
   pOut[156] = 0;
   pOut[157] = 0;
   pOut[158] = si10*x;
   pOut[159] = -si10*y;
   // 1.03 steps per component, 0.00 kb stack
   if (L == 3) return;

   // Slm: [S(4,+0), S(4,-2), S(4,+1), S(4,+4), S(4,-1), S(4,+2), S(4,-4), S(4,+3), S(4,-3)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[160] = pOut[110]*si13*z - pOut[40]*rsq*si12;
   pOut[161] = x*(-pOut[40]*si1 + rsq*si12 - si47*zz);
   pOut[162] = y*(-pOut[40]*si1 + rsq*si12 - si47*zz);
   pOut[163] = pOut[110]*si13 + pOut[113]*si13*z - pOut[40]*si1*z - rsq*si1*z;
   pOut[164] = -pOut[40]*si1 + rsq*si12 - si47*zz + 3*xx;
   pOut[165] = -pOut[40]*si1 + rsq*si12 - si47*zz + 3*yy;
   pOut[166] = pOut[113]*si72 - pOut[40]*si1 - rsq*si1 + si73*zz;
   pOut[167] = 3*x*y;
   pOut[168] = -12*x*z;
   pOut[169] = -12*y*z;
   pOut[170] = x*y*(-rsq*si14 + si15*zz);
   pOut[171] = y*(-rsq*si14 - si11*xx + si15*zz);
   pOut[172] = x*(-rsq*si14 - si11*yy + si15*zz);
   pOut[173] = si48*x*y*z;
   pOut[174] = -si74*x*y;
   pOut[175] = -si74*x*y;
   pOut[176] = si48*x*y;
   pOut[177] = -rsq*si14 - si11*xx - si11*yy + si15*zz;
   pOut[178] = si48*y*z;
   pOut[179] = si48*x*z;
   pOut[180] = z*(pOut[90]*si16 - rsq*si17*x);
   pOut[181] = z*(pOut[91]*si16 - rsq*si17 - si49*xx);
   pOut[182] = -si46*x*y*z;
   pOut[183] = pOut[90]*si16 - rsq*si17*x + si50*x*zz;
   pOut[184] = -si75*x*z;
   pOut[185] = -si46*x*z;
   pOut[186] = si76*x*z;
   pOut[187] = -si46*y*z;
   pOut[188] = pOut[91]*si16 - rsq*si17 - si49*xx + si50*zz;
   pOut[189] = -si46*x*y;
   pOut[190] = si18*(pOut[120]*x - pOut[140]*y);
   pOut[191] = pOut[120]*si18 + pOut[121]*si18*x - si51*x*yy;
   pOut[192] = -pOut[140]*si18 - pOut[142]*si18*y - si51*xx*y;
   pOut[193] = 0;
   pOut[194] = pOut[121]*si77 + si51*xx - si51*yy;
   pOut[195] = -pOut[142]*si77 - si51*xx + si51*yy;
   pOut[196] = 0;
   pOut[197] = -si78*x*y;
   pOut[198] = 0;
   pOut[199] = 0;
   pOut[200] = z*(pOut[100]*si16 - rsq*si17*y);
   pOut[201] = -si46*x*y*z;
   pOut[202] = z*(pOut[102]*si16 - rsq*si17 - si49*yy);
   pOut[203] = pOut[100]*si16 - rsq*si17*y + si50*y*zz;
   pOut[204] = -si46*y*z;
   pOut[205] = -si75*y*z;
   pOut[206] = si76*y*z;
   pOut[207] = -si46*x*z;
   pOut[208] = -si46*x*y;
   pOut[209] = pOut[102]*si16 - rsq*si17 - si49*yy + si50*zz;
   pOut[210] = pOut[150]*si20*z - pOut[70]*rsq*si19;
   pOut[211] = x*(-pOut[70]*si52 - rsq*si14 + si15*zz);
   pOut[212] = y*(-pOut[70]*si52 + rsq*si14 - si15*zz);
   pOut[213] = pOut[150]*si20 + pOut[153]*si20*z - pOut[70]*si52*z;
   pOut[214] = -pOut[70]*si52 - rsq*si14 + si15*zz - si79*xx;
   pOut[215] = -pOut[70]*si52 + rsq*si14 - si15*zz + si79*yy;
   pOut[216] = pOut[153]*si80 - pOut[70]*si52;
   pOut[217] = 0;
   pOut[218] = si48*x*z;
   pOut[219] = -si48*y*z;
   pOut[220] = si18*(pOut[120]*y + pOut[140]*x);
   pOut[221] = pOut[121]*si18*y + pOut[140]*si18 + si51*xx*y;
   pOut[222] = pOut[120]*si18 + pOut[142]*si18*x - si51*x*yy;
   pOut[223] = 0;
   pOut[224] = si78*x*y;
   pOut[225] = -si78*x*y;
   pOut[226] = 0;
   pOut[227] = pOut[121]*si18 + pOut[142]*si18 + si51*xx - si51*yy;
   pOut[228] = 0;
   pOut[229] = 0;
   pOut[230] = pOut[120]*si21*z;
   pOut[231] = pOut[121]*si21*z;
   pOut[232] = -si53*x*y*z;
   pOut[233] = pOut[120]*si21;
   pOut[234] = si53*x*z;
   pOut[235] = -si53*x*z;
   pOut[236] = 0;
   pOut[237] = -si53*y*z;
   pOut[238] = pOut[121]*si21;
   pOut[239] = -si53*x*y;
   pOut[240] = pOut[140]*si21*z;
   pOut[241] = si53*x*y*z;
   pOut[242] = pOut[142]*si21*z;
   pOut[243] = pOut[140]*si21;
   pOut[244] = si53*y*z;
   pOut[245] = -si53*y*z;
   pOut[246] = 0;
   pOut[247] = si53*x*z;
   pOut[248] = si53*x*y;
   pOut[249] = pOut[142]*si21;
   // 1.16 steps per component, 0.00 kb stack
   if (L == 4) return;

   // Slm: [S(5,+1), S(5,-1), S(5,+2), S(5,+3), S(5,-4), S(5,-3), S(5,+4), S(5,-5), S(5,+0), S(5,+5), S(5,-2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[250] = pOut[180]*si23*z - pOut[90]*rsq*si22;
   pOut[251] = pOut[181]*si23*z - pOut[90]*si8*x - pOut[91]*rsq*si22;
   pOut[252] = y*(-pOut[90]*si8 + rsq*si55*x - si54*x*zz);
   pOut[253] = pOut[180]*si23 + pOut[183]*si23*z - pOut[90]*si8*z - rsq*si10*x*z;
   pOut[254] = -pOut[90]*si8 - pOut[91]*si81*x + rsq*si83*x - si82*x*zz;
   pOut[255] = -pOut[90]*si8 + rsq*si55*x + si10*x*yy - si54*x*zz;
   pOut[256] = pOut[183]*si71 - pOut[90]*si8 - rsq*si10*x + si84*x*zz;
   pOut[257] = y*(-pOut[91]*si8 + rsq*si55 - si54*zz + si85*xx);
   pOut[258] = pOut[181]*si23 + pOut[188]*si23*z - pOut[91]*si8*z - rsq*si10*z - si86*xx*z;
   pOut[259] = -si87*x*y*z;
   pOut[260] = -pOut[100]*rsq*si22 + pOut[200]*si23*z;
   pOut[261] = x*(-pOut[100]*si8 + rsq*si55*y - si54*y*zz);
   pOut[262] = -pOut[100]*si8*y - pOut[102]*rsq*si22 + pOut[202]*si23*z;
   pOut[263] = -pOut[100]*si8*z + pOut[200]*si23 + pOut[203]*si23*z - rsq*si10*y*z;
   pOut[264] = -pOut[100]*si8 + rsq*si55*y + si10*xx*y - si54*y*zz;
   pOut[265] = -pOut[100]*si8 - pOut[102]*si81*y + rsq*si83*y - si82*y*zz;
   pOut[266] = -pOut[100]*si8 + pOut[203]*si71 - rsq*si10*y + si84*y*zz;
   pOut[267] = x*(-pOut[102]*si8 + rsq*si55 - si54*zz + si85*yy);
   pOut[268] = -si87*x*y*z;
   pOut[269] = -pOut[102]*si8*z + pOut[202]*si23 + pOut[209]*si23*z - rsq*si10*z - si86*yy*z;
   pOut[270] = -pOut[150]*rsq*si24 + pOut[210]*si25*z;
   pOut[271] = -pOut[150]*si56*x + pOut[211]*si25*z - rsq*si31*x*z;
   pOut[272] = -pOut[150]*si56*y + pOut[212]*si25*z + rsq*si31*y*z;
   pOut[273] = -pOut[150]*si56*z - pOut[153]*rsq*si24 + pOut[210]*si25 + pOut[213]*si25*z;
   pOut[274] = -pOut[150]*si56 + pOut[214]*si25*z - rsq*si31*z - si88*xx*z;
   pOut[275] = -pOut[150]*si56 + pOut[215]*si25*z + rsq*si31*z + si88*yy*z;
   pOut[276] = -pOut[150]*si56 - pOut[153]*si89*z + pOut[213]*si70 + pOut[216]*si25*z;
   pOut[277] = 0;
   pOut[278] = -pOut[153]*si56*x + pOut[211]*si25 - rsq*si31*x + si63*x*zz;
   pOut[279] = -pOut[153]*si56*y + pOut[212]*si25 + rsq*si31*y - si63*y*zz;
   pOut[280] = -pOut[120]*rsq*si27 + pOut[230]*si26*z;
   pOut[281] = -pOut[120]*si57*x - pOut[121]*rsq*si27 + pOut[231]*si26*z;
   pOut[282] = y*(-pOut[120]*si57 + rsq*si59*x - si58*x*zz);
   pOut[283] = -pOut[120]*si57*z + pOut[230]*si26 + pOut[233]*si26*z;
   pOut[284] = -pOut[120]*si57 - pOut[121]*si21*x - rsq*si59*x + si58*x*zz;
   pOut[285] = -pOut[120]*si57 + rsq*si59*x + si53*x*yy - si58*x*zz;
   pOut[286] = -pOut[120]*si57 + pOut[233]*si73;
   pOut[287] = y*(-pOut[121]*si57 + rsq*si59 - si58*zz + si90*xx);
   pOut[288] = -pOut[121]*si57*z + pOut[231]*si26 + pOut[238]*si26*z;
   pOut[289] = -si91*x*y*z;
   pOut[290] = 3*pOut[220]*z;
   pOut[291] = 3*pOut[221]*z;
   pOut[292] = 3*pOut[222]*z;
   pOut[293] = 3*pOut[220];
   pOut[294] = si92*x*y*z;
   pOut[295] = -si92*x*y*z;
   pOut[296] = 0;
   pOut[297] = 3*pOut[227]*z;
   pOut[298] = 3*pOut[221];
   pOut[299] = 3*pOut[222];
   pOut[300] = -pOut[140]*rsq*si27 + pOut[240]*si26*z;
   pOut[301] = x*(-pOut[140]*si57 - rsq*si59*y + si58*y*zz);
   pOut[302] = -pOut[140]*si57*y - pOut[142]*rsq*si27 + pOut[242]*si26*z;
   pOut[303] = -pOut[140]*si57*z + pOut[240]*si26 + pOut[243]*si26*z;
   pOut[304] = -pOut[140]*si57 - rsq*si59*y - si53*xx*y + si58*y*zz;
   pOut[305] = -pOut[140]*si57 - pOut[142]*si21*y + rsq*si59*y - si58*y*zz;
   pOut[306] = -pOut[140]*si57 + pOut[243]*si73;
   pOut[307] = x*(-pOut[142]*si57 - rsq*si59 + si58*zz - si90*yy);
   pOut[308] = si91*x*y*z;
   pOut[309] = -pOut[142]*si57*z + pOut[242]*si26 + pOut[249]*si26*z;
   pOut[310] = 3*pOut[190]*z;
   pOut[311] = 3*pOut[191]*z;
   pOut[312] = 3*pOut[192]*z;
   pOut[313] = 3*pOut[190];
   pOut[314] = 3*pOut[194]*z;
   pOut[315] = 3*pOut[195]*z;
   pOut[316] = 0;
   pOut[317] = -si92*x*y*z;
   pOut[318] = 3*pOut[191];
   pOut[319] = 3*pOut[192];
   pOut[320] = si28*(pOut[190]*y + pOut[220]*x);
   pOut[321] = si28*(pOut[191]*y + pOut[220] + pOut[221]*x);
   pOut[322] = si28*(pOut[190] + pOut[192]*y + pOut[222]*x);
   pOut[323] = 0;
   pOut[324] = pOut[194]*si28*y + pOut[221]*si93 + si94*xx*y;
   pOut[325] = pOut[192]*si93 + pOut[195]*si28*y - si94*xx*y;
   pOut[326] = 0;
   pOut[327] = pOut[191]*si28 + pOut[222]*si28 + pOut[227]*si28*x - si94*x*yy;
   pOut[328] = 0;
   pOut[329] = 0;
   pOut[330] = -pOut[110]*rsq*si29 + pOut[160]*si30*z;
   pOut[331] = -pOut[110]*si60*x + pOut[161]*si30*z + rsq*si61*x*z;
   pOut[332] = -pOut[110]*si60*y + pOut[162]*si30*z + rsq*si61*y*z;
   pOut[333] = -pOut[110]*si60*z - pOut[113]*rsq*si29 + pOut[160]*si30 + pOut[163]*si30*z;
   pOut[334] = -pOut[110]*si60 + pOut[164]*si30*z + rsq*si61*z + si95*xx*z;
   pOut[335] = -pOut[110]*si60 + pOut[165]*si30*z + rsq*si61*z + si95*yy*z;
   pOut[336] = -pOut[110]*si60 - pOut[113]*si98*z + pOut[163]*si96 + pOut[166]*si30*z - rsq*si97*z;
   pOut[337] = 15*x*y*z;
   pOut[338] = -pOut[113]*si60*x + pOut[161]*si30 + rsq*si61*x - si99*x*zz;
   pOut[339] = -pOut[113]*si60*y + pOut[162]*si30 + rsq*si61*y - si99*y*zz;
   pOut[340] = si28*(pOut[190]*x - pOut[220]*y);
   pOut[341] = si28*(pOut[190] + pOut[191]*x - pOut[221]*y);
   pOut[342] = si28*(pOut[192]*x - pOut[220] - pOut[222]*y);
   pOut[343] = 0;
   pOut[344] = pOut[191]*si93 + pOut[194]*si28*x - si94*x*yy;
   pOut[345] = pOut[195]*si28*x - pOut[222]*si93 + si94*x*yy;
   pOut[346] = 0;
   pOut[347] = pOut[192]*si28 - pOut[221]*si28 - pOut[227]*si28*y - si94*xx*y;
   pOut[348] = 0;
   pOut[349] = 0;
   pOut[350] = z*(pOut[170]*si25 - rsq*si31*x*y);
   pOut[351] = z*(pOut[171]*si25 - rsq*si31*y - si62*xx*y);
   pOut[352] = z*(pOut[172]*si25 - rsq*si31*x - si62*x*yy);
   pOut[353] = pOut[170]*si25 - rsq*si31*x*y + si63*x*y*zz;
   pOut[354] = -si100*x*y*z;
   pOut[355] = -si100*x*y*z;
   pOut[356] = si101*x*y*z;
   pOut[357] = z*(pOut[177]*si25 - rsq*si31 - si62*xx - si62*yy);
   pOut[358] = pOut[171]*si25 - rsq*si31*y - si62*xx*y + si63*y*zz;
   pOut[359] = pOut[172]*si25 - rsq*si31*x - si62*x*yy + si63*x*zz;
   // 1.52 steps per component, 0.00 kb stack
   if (L == 5) return;

   // Slm: [S(6,+6), S(6,-2), S(6,+5), S(6,+4), S(6,-5), S(6,+2), S(6,-6), S(6,+3), S(6,-4), S(6,+0), S(6,-3), S(6,-1), S(6,+1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz]]
   pOut[360] = si32*(-pOut[320]*y + pOut[340]*x);
   pOut[361] = si32*(-pOut[321]*y + pOut[340] + pOut[341]*x);
   pOut[362] = si32*(-pOut[320] - pOut[322]*y + pOut[342]*x);
   pOut[363] = 0;
   pOut[364] = -pOut[324]*si32*y + pOut[341]*si102 + pOut[344]*si32*x;
   pOut[365] = -pOut[322]*si102 - pOut[325]*si32*y + pOut[345]*si32*x;
   pOut[366] = 0;
   pOut[367] = si32*(-pOut[321] - pOut[327]*y + pOut[342] + pOut[347]*x);
   pOut[368] = 0;
   pOut[369] = 0;
   pOut[370] = -pOut[170]*rsq*si33 + pOut[350]*si34*z;
   pOut[371] = -pOut[170]*si64*x - pOut[171]*rsq*si33 + pOut[351]*si34*z;
   pOut[372] = -pOut[170]*si64*y - pOut[172]*rsq*si33 + pOut[352]*si34*z;
   pOut[373] = -pOut[170]*si64*z + pOut[350]*si34 + pOut[353]*si34*z - rsq*si65*x*y*z;
   pOut[374] = -pOut[170]*si64 - pOut[171]*si103*x + rsq*si105*x*y - si104*x*y*zz;
   pOut[375] = -pOut[170]*si64 - pOut[172]*si103*y + rsq*si105*x*y - si104*x*y*zz;
   pOut[376] = -pOut[170]*si64 + pOut[353]*si106 - rsq*si65*x*y + si107*x*y*zz;
   pOut[377] = -pOut[171]*si64*y - pOut[172]*si64*x - pOut[177]*rsq*si33 + pOut[357]*si34*z;
   pOut[378] = -pOut[171]*si64*z + pOut[351]*si34 + pOut[358]*si34*z - rsq*si65*y*z - si108*xx*y*z;
   pOut[379] = -pOut[172]*si64*z + pOut[352]*si34 + pOut[359]*si34*z - rsq*si65*x*z - si108*x*yy*z;
   pOut[380] = pOut[340]*si35*z;
   pOut[381] = pOut[341]*si35*z;
   pOut[382] = pOut[342]*si35*z;
   pOut[383] = pOut[340]*si35;
   pOut[384] = pOut[344]*si35*z;
   pOut[385] = pOut[345]*si35*z;
   pOut[386] = 0;
   pOut[387] = pOut[347]*si35*z;
   pOut[388] = pOut[341]*si35;
   pOut[389] = pOut[342]*si35;
   pOut[390] = -pOut[190]*rsq*si36 + pOut[310]*si37*z;
   pOut[391] = -pOut[190]*si66*x - pOut[191]*rsq*si36 + pOut[311]*si37*z;
   pOut[392] = -pOut[190]*si66*y - pOut[192]*rsq*si36 + pOut[312]*si37*z;
   pOut[393] = -pOut[190]*si66*z + pOut[310]*si37 + pOut[313]*si37*z;
   pOut[394] = -pOut[190]*si66 - pOut[191]*si109*x - pOut[194]*rsq*si36 + pOut[314]*si37*z;
   pOut[395] = -pOut[190]*si66 - pOut[192]*si109*y - pOut[195]*rsq*si36 + pOut[315]*si37*z;
   pOut[396] = -pOut[190]*si66 + pOut[313]*si110;
   pOut[397] = -pOut[191]*si66*y - pOut[192]*si66*x + rsq*si112*x*y - si111*x*y*zz;
   pOut[398] = -pOut[191]*si66*z + pOut[311]*si37 + pOut[318]*si37*z;
   pOut[399] = -pOut[192]*si66*z + pOut[312]*si37 + pOut[319]*si37*z;
   pOut[400] = pOut[320]*si35*z;
   pOut[401] = pOut[321]*si35*z;
   pOut[402] = pOut[322]*si35*z;
   pOut[403] = pOut[320]*si35;
   pOut[404] = pOut[324]*si35*z;
   pOut[405] = pOut[325]*si35*z;
   pOut[406] = 0;
   pOut[407] = pOut[327]*si35*z;
   pOut[408] = pOut[321]*si35;
   pOut[409] = pOut[322]*si35;
   pOut[410] = -pOut[210]*rsq*si33 + pOut[270]*si34*z;
   pOut[411] = -pOut[210]*si64*x - pOut[211]*rsq*si33 + pOut[271]*si34*z;
   pOut[412] = -pOut[210]*si64*y - pOut[212]*rsq*si33 + pOut[272]*si34*z;
   pOut[413] = -pOut[210]*si64*z - pOut[213]*rsq*si33 + pOut[270]*si34 + pOut[273]*si34*z;
   pOut[414] = -pOut[210]*si64 - pOut[211]*si103*x - pOut[214]*rsq*si33 + pOut[274]*si34*z;
   pOut[415] = -pOut[210]*si64 - pOut[212]*si103*y - pOut[215]*rsq*si33 + pOut[275]*si34*z;
   pOut[416] = -pOut[210]*si64 - pOut[213]*si103*z - pOut[216]*rsq*si33 + pOut[273]*si106 + pOut[276]*si34*z;
   pOut[417] = -si64*(pOut[211]*y + pOut[212]*x);
   pOut[418] = -pOut[211]*si64*z - pOut[213]*si64*x + pOut[271]*si34 + pOut[278]*si34*z - rsq*si65*x*z;
   pOut[419] = -pOut[212]*si64*z - pOut[213]*si64*y + pOut[272]*si34 + pOut[279]*si34*z + rsq*si65*y*z;
   pOut[420] = si32*(pOut[320]*x + pOut[340]*y);
   pOut[421] = si32*(pOut[320] + pOut[321]*x + pOut[341]*y);
   pOut[422] = si32*(pOut[322]*x + pOut[340] + pOut[342]*y);
   pOut[423] = 0;
   pOut[424] = pOut[321]*si102 + pOut[324]*si32*x + pOut[344]*si32*y;
   pOut[425] = pOut[325]*si32*x + pOut[342]*si102 + pOut[345]*si32*y;
   pOut[426] = 0;
   pOut[427] = si32*(pOut[322] + pOut[327]*x + pOut[341] + pOut[347]*y);
   pOut[428] = 0;
   pOut[429] = 0;
   pOut[430] = -pOut[230]*rsq*si38 + pOut[280]*si39*z;
   pOut[431] = -pOut[230]*si67*x - pOut[231]*rsq*si38 + pOut[281]*si39*z;
   pOut[432] = -pOut[230]*si67*y + pOut[282]*si39*z + rsq*si68*x*y*z;
   pOut[433] = -pOut[230]*si67*z - pOut[233]*rsq*si38 + pOut[280]*si39 + pOut[283]*si39*z;
   pOut[434] = -pOut[230]*si67 - pOut[231]*si113*x + pOut[284]*si39*z - rsq*si68*x*z;
   pOut[435] = -pOut[230]*si67 + pOut[285]*si39*z + rsq*si68*x*z + si114*x*yy*z;
   pOut[436] = -pOut[230]*si67 - pOut[233]*si113*z + pOut[283]*si115 + pOut[286]*si39*z;
   pOut[437] = -pOut[231]*si67*y + pOut[287]*si39*z + rsq*si68*y*z + si116*xx*y*z;
   pOut[438] = -pOut[231]*si67*z - pOut[233]*si67*x - pOut[238]*rsq*si38 + pOut[281]*si39 + pOut[288]*si39*z;
   pOut[439] = -pOut[233]*si67*y + pOut[282]*si39 + rsq*si68*x*y - si117*x*y*zz;
   pOut[440] = -pOut[220]*rsq*si36 + pOut[290]*si37*z;
   pOut[441] = -pOut[220]*si66*x - pOut[221]*rsq*si36 + pOut[291]*si37*z;
   pOut[442] = -pOut[220]*si66*y - pOut[222]*rsq*si36 + pOut[292]*si37*z;
   pOut[443] = -pOut[220]*si66*z + pOut[290]*si37 + pOut[293]*si37*z;
   pOut[444] = -pOut[220]*si66 - pOut[221]*si109*x - rsq*si112*x*y + si111*x*y*zz;
   pOut[445] = -pOut[220]*si66 - pOut[222]*si109*y + rsq*si112*x*y - si111*x*y*zz;
   pOut[446] = -pOut[220]*si66 + pOut[293]*si110;
   pOut[447] = -pOut[221]*si66*y - pOut[222]*si66*x - pOut[227]*rsq*si36 + pOut[297]*si37*z;
   pOut[448] = -pOut[221]*si66*z + pOut[291]*si37 + pOut[298]*si37*z;
   pOut[449] = -pOut[222]*si66*z + pOut[292]*si37 + pOut[299]*si37*z;
   pOut[450] = -pOut[160]*rsq*si40 + pOut[330]*si41*z;
   pOut[451] = -pOut[160]*si7*x - pOut[161]*rsq*si40 + pOut[331]*si41*z;
   pOut[452] = -pOut[160]*si7*y - pOut[162]*rsq*si40 + pOut[332]*si41*z;
   pOut[453] = -pOut[160]*si7*z - pOut[163]*rsq*si40 + pOut[330]*si41 + pOut[333]*si41*z;
   pOut[454] = -pOut[160]*si7 - pOut[161]*si118*x - pOut[164]*rsq*si40 + pOut[334]*si41*z;
   pOut[455] = -pOut[160]*si7 - pOut[162]*si118*y - pOut[165]*rsq*si40 + pOut[335]*si41*z;
   pOut[456] = -pOut[160]*si7 - pOut[163]*si118*z - pOut[166]*rsq*si40 + pOut[333]*si119 + pOut[336]*si41*z;
   pOut[457] = -pOut[161]*si7*y - pOut[162]*si7*x - rsq*si120*x*y + si121*x*y*zz;
   pOut[458] = -pOut[161]*si7*z - pOut[163]*si7*x + pOut[331]*si41 + pOut[338]*si41*z + 10*rsq*x*z;
   pOut[459] = -pOut[162]*si7*z - pOut[163]*si7*y + pOut[332]*si41 + pOut[339]*si41*z + 10*rsq*y*z;
   pOut[460] = -pOut[240]*rsq*si38 + pOut[300]*si39*z;
   pOut[461] = -pOut[240]*si67*x + pOut[301]*si39*z - rsq*si68*x*y*z;
   pOut[462] = -pOut[240]*si67*y - pOut[242]*rsq*si38 + pOut[302]*si39*z;
   pOut[463] = -pOut[240]*si67*z - pOut[243]*rsq*si38 + pOut[300]*si39 + pOut[303]*si39*z;
   pOut[464] = -pOut[240]*si67 + pOut[304]*si39*z - rsq*si68*y*z - si114*xx*y*z;
   pOut[465] = -pOut[240]*si67 - pOut[242]*si113*y + pOut[305]*si39*z + rsq*si68*y*z;
   pOut[466] = -pOut[240]*si67 - pOut[243]*si113*z + pOut[303]*si115 + pOut[306]*si39*z;
   pOut[467] = -pOut[242]*si67*x + pOut[307]*si39*z - rsq*si68*x*z - si116*x*yy*z;
   pOut[468] = -pOut[243]*si67*x + pOut[301]*si39 - rsq*si68*x*y + si117*x*y*zz;
   pOut[469] = -pOut[242]*si67*z - pOut[243]*si67*y - pOut[249]*rsq*si38 + pOut[302]*si39 + pOut[309]*si39*z;
   pOut[470] = -pOut[200]*rsq*si42 + pOut[260]*si43*z;
   pOut[471] = -pOut[200]*si69*x + pOut[261]*si43*z + rsq*si70*x*y*z;
   pOut[472] = -pOut[200]*si69*y - pOut[202]*rsq*si42 + pOut[262]*si43*z;
   pOut[473] = -pOut[200]*si69*z - pOut[203]*rsq*si42 + pOut[260]*si43 + pOut[263]*si43*z;
   pOut[474] = -pOut[200]*si69 + pOut[264]*si43*z + rsq*si70*y*z + si122*xx*y*z;
   pOut[475] = -pOut[200]*si69 - pOut[202]*si123*y + pOut[265]*si43*z + rsq*si124*y*z;
   pOut[476] = -pOut[200]*si69 - pOut[203]*si123*z + pOut[263]*si125 + pOut[266]*si43*z - rsq*si122*y*z;
   pOut[477] = -pOut[202]*si69*x + pOut[267]*si43*z + rsq*si70*x*z + si126*x*yy*z;
   pOut[478] = -pOut[203]*si69*x + pOut[261]*si43 + rsq*si70*x*y - si127*x*y*zz;
   pOut[479] = -pOut[202]*si69*z - pOut[203]*si69*y - pOut[209]*rsq*si42 + pOut[262]*si43 + pOut[269]*si43*z;
   pOut[480] = -pOut[180]*rsq*si42 + pOut[250]*si43*z;
   pOut[481] = -pOut[180]*si69*x - pOut[181]*rsq*si42 + pOut[251]*si43*z;
   pOut[482] = -pOut[180]*si69*y + pOut[252]*si43*z + rsq*si70*x*y*z;
   pOut[483] = -pOut[180]*si69*z - pOut[183]*rsq*si42 + pOut[250]*si43 + pOut[253]*si43*z;
   pOut[484] = -pOut[180]*si69 - pOut[181]*si123*x + pOut[254]*si43*z + rsq*si124*x*z;
   pOut[485] = -pOut[180]*si69 + pOut[255]*si43*z + rsq*si70*x*z + si122*x*yy*z;
   pOut[486] = -pOut[180]*si69 - pOut[183]*si123*z + pOut[253]*si125 + pOut[256]*si43*z - rsq*si122*x*z;
   pOut[487] = -pOut[181]*si69*y + pOut[257]*si43*z + rsq*si70*y*z + si126*xx*y*z;
   pOut[488] = -pOut[181]*si69*z - pOut[183]*si69*x - pOut[188]*rsq*si42 + pOut[251]*si43 + pOut[258]*si43*z;
   pOut[489] = -pOut[183]*si69*y + pOut[252]*si43 + rsq*si70*x*y - si127*x*y*zz;
   // 2.12 steps per component, 0.00 kb stack
   if (L == 6) return;

   assert(0);
}

// Calculate Slc(x,y,z) and 1st+2nd+3rd derivatives for all l,m with l <= L.
// Output is addressed as pOut[iSlcX(l,c)*NCOMP_Deriv3 + iComp] with iComp \in {0,1,...,19} indexed by FSlcX::COMP__*
// This function evaluates derivative components:
//    [1, d/dx, d/dy, d/dz, d^2/dxx, d^2/dyy, d^2/dzz, d^2/dxy, d^2/dxz, d^2/dyz, d^3/dxxx, d^3/dyyy, d^3/dzzz, d^3/dxxy, d^3/dxxz, d^3/dxyy, d^3/dyyz, d^3/dxzz, d^3/dyzz, d^3/dxyz] Slc(x,y,z)
void EvalSlcX_Deriv3(double *IR_RP pOut, double x, double y, double z, unsigned L)
{
   // Slm: [S(0,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[0] = 1;
   pOut[1] = 0;
   pOut[2] = 0;
   pOut[3] = 0;
   pOut[4] = 0;
   pOut[5] = 0;
   pOut[6] = 0;
   pOut[7] = 0;
   pOut[8] = 0;
   pOut[9] = 0;
   pOut[10] = 0;
   pOut[11] = 0;
   pOut[12] = 0;
   pOut[13] = 0;
   pOut[14] = 0;
   pOut[15] = 0;
   pOut[16] = 0;
   pOut[17] = 0;
   pOut[18] = 0;
   pOut[19] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 0) return;

   // Slm: [S(1,+1), S(1,-1), S(1,+0)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[20] = x;
   pOut[21] = 1;
   pOut[22] = 0;
   pOut[23] = 0;
   pOut[24] = 0;
   pOut[25] = 0;
   pOut[26] = 0;
   pOut[27] = 0;
   pOut[28] = 0;
   pOut[29] = 0;
   pOut[30] = 0;
   pOut[31] = 0;
   pOut[32] = 0;
   pOut[33] = 0;
   pOut[34] = 0;
   pOut[35] = 0;
   pOut[36] = 0;
   pOut[37] = 0;
   pOut[38] = 0;
   pOut[39] = 0;
   pOut[40] = y;
   pOut[41] = 0;
   pOut[42] = 1;
   pOut[43] = 0;
   pOut[44] = 0;
   pOut[45] = 0;
   pOut[46] = 0;
   pOut[47] = 0;
   pOut[48] = 0;
   pOut[49] = 0;
   pOut[50] = 0;
   pOut[51] = 0;
   pOut[52] = 0;
   pOut[53] = 0;
   pOut[54] = 0;
   pOut[55] = 0;
   pOut[56] = 0;
   pOut[57] = 0;
   pOut[58] = 0;
   pOut[59] = 0;
   pOut[60] = z;
   pOut[61] = 0;
   pOut[62] = 0;
   pOut[63] = 1;
   pOut[64] = 0;
   pOut[65] = 0;
   pOut[66] = 0;
   pOut[67] = 0;
   pOut[68] = 0;
   pOut[69] = 0;
   pOut[70] = 0;
   pOut[71] = 0;
   pOut[72] = 0;
   pOut[73] = 0;
   pOut[74] = 0;
   pOut[75] = 0;
   pOut[76] = 0;
   pOut[77] = 0;
   pOut[78] = 0;
   pOut[79] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 1) return;

   double const xx = x*x;
   double const yy = y*y;
   double const zz = z*z;
   double const rsq = xx + yy + zz;
   // Slm: [S(2,+0), S(2,-2), S(2,+1), S(2,+2), S(2,-1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[80] = -rsq*si0 + si1*zz;
   pOut[81] = -x;
   pOut[82] = -y;
   pOut[83] = 2*z;
   pOut[84] = -1;
   pOut[85] = -1;
   pOut[86] = 2;
   pOut[87] = 0;
   pOut[88] = 0;
   pOut[89] = 0;
   pOut[90] = 0;
   pOut[91] = 0;
   pOut[92] = 0;
   pOut[93] = 0;
   pOut[94] = 0;
   pOut[95] = 0;
   pOut[96] = 0;
   pOut[97] = 0;
   pOut[98] = 0;
   pOut[99] = 0;
   pOut[100] = si2*x*y;
   pOut[101] = si2*y;
   pOut[102] = si2*x;
   pOut[103] = 0;
   pOut[104] = 0;
   pOut[105] = 0;
   pOut[106] = 0;
   pOut[107] = si2;
   pOut[108] = 0;
   pOut[109] = 0;
   pOut[110] = 0;
   pOut[111] = 0;
   pOut[112] = 0;
   pOut[113] = 0;
   pOut[114] = 0;
   pOut[115] = 0;
   pOut[116] = 0;
   pOut[117] = 0;
   pOut[118] = 0;
   pOut[119] = 0;
   pOut[120] = si2*x*z;
   pOut[121] = si2*z;
   pOut[122] = 0;
   pOut[123] = si2*x;
   pOut[124] = 0;
   pOut[125] = 0;
   pOut[126] = 0;
   pOut[127] = 0;
   pOut[128] = si2;
   pOut[129] = 0;
   pOut[130] = 0;
   pOut[131] = 0;
   pOut[132] = 0;
   pOut[133] = 0;
   pOut[134] = 0;
   pOut[135] = 0;
   pOut[136] = 0;
   pOut[137] = 0;
   pOut[138] = 0;
   pOut[139] = 0;
   pOut[140] = si3*(xx - yy);
   pOut[141] = si2*x;
   pOut[142] = -si2*y;
   pOut[143] = 0;
   pOut[144] = si2;
   pOut[145] = -si2;
   pOut[146] = 0;
   pOut[147] = 0;
   pOut[148] = 0;
   pOut[149] = 0;
   pOut[150] = 0;
   pOut[151] = 0;
   pOut[152] = 0;
   pOut[153] = 0;
   pOut[154] = 0;
   pOut[155] = 0;
   pOut[156] = 0;
   pOut[157] = 0;
   pOut[158] = 0;
   pOut[159] = 0;
   pOut[160] = si2*y*z;
   pOut[161] = 0;
   pOut[162] = si2*z;
   pOut[163] = si2*y;
   pOut[164] = 0;
   pOut[165] = 0;
   pOut[166] = 0;
   pOut[167] = 0;
   pOut[168] = 0;
   pOut[169] = si2;
   pOut[170] = 0;
   pOut[171] = 0;
   pOut[172] = 0;
   pOut[173] = 0;
   pOut[174] = 0;
   pOut[175] = 0;
   pOut[176] = 0;
   pOut[177] = 0;
   pOut[178] = 0;
   pOut[179] = 0;
   // 1.00 steps per component, 0.00 kb stack
   if (L == 2) return;

   // Slm: [S(3,+1), S(3,-1), S(3,+0), S(3,+3), S(3,-2), S(3,-3), S(3,+2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[180] = x*(-rsq*si4 + si5*zz);
   pOut[181] = -rsq*si4 - si44*xx + si5*zz;
   pOut[182] = -si44*x*y;
   pOut[183] = si45*x*z;
   pOut[184] = -si71*x;
   pOut[185] = -si44*x;
   pOut[186] = si45*x;
   pOut[187] = -si44*y;
   pOut[188] = si45*z;
   pOut[189] = 0;
   pOut[190] = -si71;
   pOut[191] = 0;
   pOut[192] = 0;
   pOut[193] = 0;
   pOut[194] = 0;
   pOut[195] = -si44;
   pOut[196] = 0;
   pOut[197] = si45;
   pOut[198] = 0;
   pOut[199] = 0;
   pOut[200] = y*(-rsq*si4 + si5*zz);
   pOut[201] = -si44*x*y;
   pOut[202] = -rsq*si4 - si44*yy + si5*zz;
   pOut[203] = si45*y*z;
   pOut[204] = -si44*y;
   pOut[205] = -si71*y;
   pOut[206] = si45*y;
   pOut[207] = -si44*x;
   pOut[208] = 0;
   pOut[209] = si45*z;
   pOut[210] = 0;
   pOut[211] = -si71;
   pOut[212] = 0;
   pOut[213] = -si44;
   pOut[214] = 0;
   pOut[215] = 0;
   pOut[216] = 0;
   pOut[217] = 0;
   pOut[218] = si45;
   pOut[219] = 0;
   pOut[220] = z*(pOut[80]*si7 - rsq*si6);
   pOut[221] = -3*x*z;
   pOut[222] = -3*y*z;
   pOut[223] = pOut[80]*si7 - rsq*si6 + 2*zz;
   pOut[224] = -3*z;
   pOut[225] = -3*z;
   pOut[226] = 6*z;
   pOut[227] = 0;
   pOut[228] = -3*x;
   pOut[229] = -3*y;
   pOut[230] = 0;
   pOut[231] = 0;
   pOut[232] = 6;
   pOut[233] = 0;
   pOut[234] = -3;
   pOut[235] = 0;
   pOut[236] = -3;
   pOut[237] = 0;
   pOut[238] = 0;
   pOut[239] = 0;
   pOut[240] = x*(pOut[140]*si9 - si8*yy);
   pOut[241] = pOut[140]*si9 + si8*xx - si8*yy;
   pOut[242] = -si46*x*y;
   pOut[243] = 0;
   pOut[244] = si46*x;
   pOut[245] = -si46*x;
   pOut[246] = 0;
   pOut[247] = -si46*y;
   pOut[248] = 0;
   pOut[249] = 0;
   pOut[250] = si46;
   pOut[251] = 0;
   pOut[252] = 0;
   pOut[253] = 0;
   pOut[254] = 0;
   pOut[255] = -si46;
   pOut[256] = 0;
   pOut[257] = 0;
   pOut[258] = 0;
   pOut[259] = 0;
   pOut[260] = si10*x*y*z;
   pOut[261] = si10*y*z;
   pOut[262] = si10*x*z;
   pOut[263] = si10*x*y;
   pOut[264] = 0;
   pOut[265] = 0;
   pOut[266] = 0;
   pOut[267] = si10*z;
   pOut[268] = si10*y;
   pOut[269] = si10*x;
   pOut[270] = 0;
   pOut[271] = 0;
   pOut[272] = 0;
   pOut[273] = 0;
   pOut[274] = 0;
   pOut[275] = 0;
   pOut[276] = 0;
   pOut[277] = 0;
   pOut[278] = 0;
   pOut[279] = si10;
   pOut[280] = y*(pOut[140]*si9 + si8*xx);
   pOut[281] = si46*x*y;
   pOut[282] = pOut[140]*si9 + si8*xx - si8*yy;
   pOut[283] = 0;
   pOut[284] = si46*y;
   pOut[285] = -si46*y;
   pOut[286] = 0;
   pOut[287] = si46*x;
   pOut[288] = 0;
   pOut[289] = 0;
   pOut[290] = 0;
   pOut[291] = -si46;
   pOut[292] = 0;
   pOut[293] = si46;
   pOut[294] = 0;
   pOut[295] = 0;
   pOut[296] = 0;
   pOut[297] = 0;
   pOut[298] = 0;
   pOut[299] = 0;
   pOut[300] = pOut[140]*si11*z;
   pOut[301] = si10*x*z;
   pOut[302] = -si10*y*z;
   pOut[303] = pOut[140]*si11;
   pOut[304] = si10*z;
   pOut[305] = -si10*z;
   pOut[306] = 0;
   pOut[307] = 0;
   pOut[308] = si10*x;
   pOut[309] = -si10*y;
   pOut[310] = 0;
   pOut[311] = 0;
   pOut[312] = 0;
   pOut[313] = 0;
   pOut[314] = si10;
   pOut[315] = 0;
   pOut[316] = -si10;
   pOut[317] = 0;
   pOut[318] = 0;
   pOut[319] = 0;
   // 1.01 steps per component, 0.00 kb stack
   if (L == 3) return;

   // Slm: [S(4,+0), S(4,-2), S(4,+1), S(4,+4), S(4,-1), S(4,+2), S(4,-4), S(4,+3), S(4,-3)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[320] = pOut[220]*si13*z - pOut[80]*rsq*si12;
   pOut[321] = x*(-pOut[80]*si1 + rsq*si12 - si47*zz);
   pOut[322] = y*(-pOut[80]*si1 + rsq*si12 - si47*zz);
   pOut[323] = pOut[220]*si13 + pOut[223]*si13*z - pOut[80]*si1*z - rsq*si1*z;
   pOut[324] = -pOut[80]*si1 + rsq*si12 - si47*zz + 3*xx;
   pOut[325] = -pOut[80]*si1 + rsq*si12 - si47*zz + 3*yy;
   pOut[326] = pOut[223]*si72 - pOut[80]*si1 - rsq*si1 + si73*zz;
   pOut[327] = 3*x*y;
   pOut[328] = -12*x*z;
   pOut[329] = -12*y*z;
   pOut[330] = 9*x;
   pOut[331] = 9*y;
   pOut[332] = 24*z;
   pOut[333] = 3*y;
   pOut[334] = -12*z;
   pOut[335] = 3*x;
   pOut[336] = -12*z;
   pOut[337] = -12*x;
   pOut[338] = -12*y;
   pOut[339] = 0;
   pOut[340] = x*y*(-rsq*si14 + si15*zz);
   pOut[341] = y*(-rsq*si14 - si11*xx + si15*zz);
   pOut[342] = x*(-rsq*si14 - si11*yy + si15*zz);
   pOut[343] = si48*x*y*z;
   pOut[344] = -si74*x*y;
   pOut[345] = -si74*x*y;
   pOut[346] = si48*x*y;
   pOut[347] = -rsq*si14 - si11*xx - si11*yy + si15*zz;
   pOut[348] = si48*y*z;
   pOut[349] = si48*x*z;
   pOut[350] = -si74*y;
   pOut[351] = -si74*x;
   pOut[352] = 0;
   pOut[353] = -si74*x;
   pOut[354] = 0;
   pOut[355] = -si74*y;
   pOut[356] = 0;
   pOut[357] = si48*y;
   pOut[358] = si48*x;
   pOut[359] = si48*z;
   pOut[360] = z*(pOut[180]*si16 - rsq*si17*x);
   pOut[361] = z*(pOut[181]*si16 - rsq*si17 - si49*xx);
   pOut[362] = -si46*x*y*z;
   pOut[363] = pOut[180]*si16 - rsq*si17*x + si50*x*zz;
   pOut[364] = -si75*x*z;
   pOut[365] = -si46*x*z;
   pOut[366] = si76*x*z;
   pOut[367] = -si46*y*z;
   pOut[368] = pOut[181]*si16 - rsq*si17 - si49*xx + si50*zz;
   pOut[369] = -si46*x*y;
   pOut[370] = -si75*z;
   pOut[371] = 0;
   pOut[372] = si76*x;
   pOut[373] = 0;
   pOut[374] = -si75*x;
   pOut[375] = -si46*z;
   pOut[376] = -si46*x;
   pOut[377] = si76*z;
   pOut[378] = 0;
   pOut[379] = -si46*y;
   pOut[380] = si18*(pOut[240]*x - pOut[280]*y);
   pOut[381] = pOut[240]*si18 + pOut[241]*si18*x - si51*x*yy;
   pOut[382] = -pOut[280]*si18 - pOut[282]*si18*y - si51*xx*y;
   pOut[383] = 0;
   pOut[384] = pOut[241]*si77 + si51*xx - si51*yy;
   pOut[385] = -pOut[282]*si77 - si51*xx + si51*yy;
   pOut[386] = 0;
   pOut[387] = -si78*x*y;
   pOut[388] = 0;
   pOut[389] = 0;
   pOut[390] = si78*x;
   pOut[391] = si78*y;
   pOut[392] = 0;
   pOut[393] = -si78*y;
   pOut[394] = 0;
   pOut[395] = -si78*x;
   pOut[396] = 0;
   pOut[397] = 0;
   pOut[398] = 0;
   pOut[399] = 0;
   pOut[400] = z*(pOut[200]*si16 - rsq*si17*y);
   pOut[401] = -si46*x*y*z;
   pOut[402] = z*(pOut[202]*si16 - rsq*si17 - si49*yy);
   pOut[403] = pOut[200]*si16 - rsq*si17*y + si50*y*zz;
   pOut[404] = -si46*y*z;
   pOut[405] = -si75*y*z;
   pOut[406] = si76*y*z;
   pOut[407] = -si46*x*z;
   pOut[408] = -si46*x*y;
   pOut[409] = pOut[202]*si16 - rsq*si17 - si49*yy + si50*zz;
   pOut[410] = 0;
   pOut[411] = -si75*z;
   pOut[412] = si76*y;
   pOut[413] = -si46*z;
   pOut[414] = -si46*y;
   pOut[415] = 0;
   pOut[416] = -si75*y;
   pOut[417] = 0;
   pOut[418] = si76*z;
   pOut[419] = -si46*x;
   pOut[420] = -pOut[140]*rsq*si19 + pOut[300]*si20*z;
   pOut[421] = x*(-pOut[140]*si52 - rsq*si14 + si15*zz);
   pOut[422] = y*(-pOut[140]*si52 + rsq*si14 - si15*zz);
   pOut[423] = -pOut[140]*si52*z + pOut[300]*si20 + pOut[303]*si20*z;
   pOut[424] = -pOut[140]*si52 - rsq*si14 + si15*zz - si79*xx;
   pOut[425] = -pOut[140]*si52 + rsq*si14 - si15*zz + si79*yy;
   pOut[426] = -pOut[140]*si52 + pOut[303]*si80;
   pOut[427] = 0;
   pOut[428] = si48*x*z;
   pOut[429] = -si48*y*z;
   pOut[430] = -si48*x;
   pOut[431] = si48*y;
   pOut[432] = 0;
   pOut[433] = 0;
   pOut[434] = si48*z;
   pOut[435] = 0;
   pOut[436] = -si48*z;
   pOut[437] = si48*x;
   pOut[438] = -si48*y;
   pOut[439] = 0;
   pOut[440] = si18*(pOut[240]*y + pOut[280]*x);
   pOut[441] = pOut[241]*si18*y + pOut[280]*si18 + si51*xx*y;
   pOut[442] = pOut[240]*si18 + pOut[282]*si18*x - si51*x*yy;
   pOut[443] = 0;
   pOut[444] = si78*x*y;
   pOut[445] = -si78*x*y;
   pOut[446] = 0;
   pOut[447] = pOut[241]*si18 + pOut[282]*si18 + si51*xx - si51*yy;
   pOut[448] = 0;
   pOut[449] = 0;
   pOut[450] = si78*y;
   pOut[451] = -si78*x;
   pOut[452] = 0;
   pOut[453] = si78*x;
   pOut[454] = 0;
   pOut[455] = -si78*y;
   pOut[456] = 0;
   pOut[457] = 0;
   pOut[458] = 0;
   pOut[459] = 0;
   pOut[460] = pOut[240]*si21*z;
   pOut[461] = pOut[241]*si21*z;
   pOut[462] = -si53*x*y*z;
   pOut[463] = pOut[240]*si21;
   pOut[464] = si53*x*z;
   pOut[465] = -si53*x*z;
   pOut[466] = 0;
   pOut[467] = -si53*y*z;
   pOut[468] = pOut[241]*si21;
   pOut[469] = -si53*x*y;
   pOut[470] = si53*z;
   pOut[471] = 0;
   pOut[472] = 0;
   pOut[473] = 0;
   pOut[474] = si53*x;
   pOut[475] = -si53*z;
   pOut[476] = -si53*x;
   pOut[477] = 0;
   pOut[478] = 0;
   pOut[479] = -si53*y;
   pOut[480] = pOut[280]*si21*z;
   pOut[481] = si53*x*y*z;
   pOut[482] = pOut[282]*si21*z;
   pOut[483] = pOut[280]*si21;
   pOut[484] = si53*y*z;
   pOut[485] = -si53*y*z;
   pOut[486] = 0;
   pOut[487] = si53*x*z;
   pOut[488] = si53*x*y;
   pOut[489] = pOut[282]*si21;
   pOut[490] = 0;
   pOut[491] = -si53*z;
   pOut[492] = 0;
   pOut[493] = si53*z;
   pOut[494] = si53*y;
   pOut[495] = 0;
   pOut[496] = -si53*y;
   pOut[497] = 0;
   pOut[498] = 0;
   pOut[499] = si53*x;
   // 1.08 steps per component, 0.00 kb stack
   if (L == 4) return;

   // Slm: [S(5,+1), S(5,-1), S(5,+2), S(5,+3), S(5,-4), S(5,-3), S(5,+4), S(5,-5), S(5,+0), S(5,+5), S(5,-2)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[500] = -pOut[180]*rsq*si22 + pOut[360]*si23*z;
   pOut[501] = -pOut[180]*si8*x - pOut[181]*rsq*si22 + pOut[361]*si23*z;
   pOut[502] = y*(-pOut[180]*si8 + rsq*si55*x - si54*x*zz);
   pOut[503] = -pOut[180]*si8*z + pOut[360]*si23 + pOut[363]*si23*z - rsq*si10*x*z;
   pOut[504] = -pOut[180]*si8 - pOut[181]*si81*x + rsq*si83*x - si82*x*zz;
   pOut[505] = -pOut[180]*si8 + rsq*si55*x + si10*x*yy - si54*x*zz;
   pOut[506] = -pOut[180]*si8 + pOut[363]*si71 - rsq*si10*x + si84*x*zz;
   pOut[507] = y*(-pOut[181]*si8 + rsq*si55 - si54*zz + si85*xx);
   pOut[508] = -pOut[181]*si8*z + pOut[361]*si23 + pOut[368]*si23*z - rsq*si10*z - si86*xx*z;
   pOut[509] = -si87*x*y*z;
   pOut[510] = -pOut[181]*si46 + rsq*si83 + si128*xx - si82*zz;
   pOut[511] = si129*x*y;
   pOut[512] = si130*x*z;
   pOut[513] = si129*x*y;
   pOut[514] = -si131*x*z;
   pOut[515] = -pOut[181]*si8 + rsq*si55 + si10*yy - si54*zz + si85*xx;
   pOut[516] = -si87*x*z;
   pOut[517] = -pOut[181]*si8 + pOut[368]*si71 - rsq*si10 + si84*zz - si86*xx;
   pOut[518] = -si87*x*y;
   pOut[519] = -si87*y*z;
   pOut[520] = -pOut[200]*rsq*si22 + pOut[400]*si23*z;
   pOut[521] = x*(-pOut[200]*si8 + rsq*si55*y - si54*y*zz);
   pOut[522] = -pOut[200]*si8*y - pOut[202]*rsq*si22 + pOut[402]*si23*z;
   pOut[523] = -pOut[200]*si8*z + pOut[400]*si23 + pOut[403]*si23*z - rsq*si10*y*z;
   pOut[524] = -pOut[200]*si8 + rsq*si55*y + si10*xx*y - si54*y*zz;
   pOut[525] = -pOut[200]*si8 - pOut[202]*si81*y + rsq*si83*y - si82*y*zz;
   pOut[526] = -pOut[200]*si8 + pOut[403]*si71 - rsq*si10*y + si84*y*zz;
   pOut[527] = x*(-pOut[202]*si8 + rsq*si55 - si54*zz + si85*yy);
   pOut[528] = -si87*x*y*z;
   pOut[529] = -pOut[202]*si8*z + pOut[402]*si23 + pOut[409]*si23*z - rsq*si10*z - si86*yy*z;
   pOut[530] = si129*x*y;
   pOut[531] = -pOut[202]*si46 + rsq*si83 + si128*yy - si82*zz;
   pOut[532] = si130*y*z;
   pOut[533] = -pOut[202]*si8 + rsq*si55 + si10*xx - si54*zz + si85*yy;
   pOut[534] = -si87*y*z;
   pOut[535] = si129*x*y;
   pOut[536] = -si131*y*z;
   pOut[537] = -si87*x*y;
   pOut[538] = -pOut[202]*si8 + pOut[409]*si71 - rsq*si10 + si84*zz - si86*yy;
   pOut[539] = -si87*x*z;
   pOut[540] = -pOut[300]*rsq*si24 + pOut[420]*si25*z;
   pOut[541] = -pOut[300]*si56*x + pOut[421]*si25*z - rsq*si31*x*z;
   pOut[542] = -pOut[300]*si56*y + pOut[422]*si25*z + rsq*si31*y*z;
   pOut[543] = -pOut[300]*si56*z - pOut[303]*rsq*si24 + pOut[420]*si25 + pOut[423]*si25*z;
   pOut[544] = -pOut[300]*si56 + pOut[424]*si25*z - rsq*si31*z - si88*xx*z;
   pOut[545] = -pOut[300]*si56 + pOut[425]*si25*z + rsq*si31*z + si88*yy*z;
   pOut[546] = -pOut[300]*si56 - pOut[303]*si89*z + pOut[423]*si70 + pOut[426]*si25*z;
   pOut[547] = 0;
   pOut[548] = -pOut[303]*si56*x + pOut[421]*si25 - rsq*si31*x + si63*x*zz;
   pOut[549] = -pOut[303]*si56*y + pOut[422]*si25 + rsq*si31*y - si63*y*zz;
   pOut[550] = -si101*x*z;
   pOut[551] = si101*y*z;
   pOut[552] = -pOut[303]*si132 + pOut[426]*si133;
   pOut[553] = 0;
   pOut[554] = -pOut[303]*si56 + pOut[424]*si25 - rsq*si31 + si63*zz - si88*xx;
   pOut[555] = 0;
   pOut[556] = -pOut[303]*si56 + pOut[425]*si25 + rsq*si31 - si63*zz + si88*yy;
   pOut[557] = si101*x*z;
   pOut[558] = -si101*y*z;
   pOut[559] = 0;
   pOut[560] = -pOut[240]*rsq*si27 + pOut[460]*si26*z;
   pOut[561] = -pOut[240]*si57*x - pOut[241]*rsq*si27 + pOut[461]*si26*z;
   pOut[562] = y*(-pOut[240]*si57 + rsq*si59*x - si58*x*zz);
   pOut[563] = -pOut[240]*si57*z + pOut[460]*si26 + pOut[463]*si26*z;
   pOut[564] = -pOut[240]*si57 - pOut[241]*si21*x - rsq*si59*x + si58*x*zz;
   pOut[565] = -pOut[240]*si57 + rsq*si59*x + si53*x*yy - si58*x*zz;
   pOut[566] = -pOut[240]*si57 + pOut[463]*si73;
   pOut[567] = y*(-pOut[241]*si57 + rsq*si59 - si58*zz + si90*xx);
   pOut[568] = -pOut[241]*si57*z + pOut[461]*si26 + pOut[468]*si26*z;
   pOut[569] = -si91*x*y*z;
   pOut[570] = -pOut[241]*si135 - rsq*si59 - si134*xx + si58*zz;
   pOut[571] = si136*x*y;
   pOut[572] = 0;
   pOut[573] = si53*x*y;
   pOut[574] = si91*x*z;
   pOut[575] = -pOut[241]*si57 + rsq*si59 + si53*yy - si58*zz + si90*xx;
   pOut[576] = -si91*x*z;
   pOut[577] = -pOut[241]*si57 + pOut[468]*si73;
   pOut[578] = -si91*x*y;
   pOut[579] = -si91*y*z;
   pOut[580] = 3*pOut[440]*z;
   pOut[581] = 3*pOut[441]*z;
   pOut[582] = 3*pOut[442]*z;
   pOut[583] = 3*pOut[440];
   pOut[584] = si92*x*y*z;
   pOut[585] = -si92*x*y*z;
   pOut[586] = 0;
   pOut[587] = 3*pOut[447]*z;
   pOut[588] = 3*pOut[441];
   pOut[589] = 3*pOut[442];
   pOut[590] = si92*y*z;
   pOut[591] = -si92*x*z;
   pOut[592] = 0;
   pOut[593] = si92*x*z;
   pOut[594] = si92*x*y;
   pOut[595] = -si92*y*z;
   pOut[596] = -si92*x*y;
   pOut[597] = 0;
   pOut[598] = 0;
   pOut[599] = 3*pOut[447];
   pOut[600] = -pOut[280]*rsq*si27 + pOut[480]*si26*z;
   pOut[601] = x*(-pOut[280]*si57 - rsq*si59*y + si58*y*zz);
   pOut[602] = -pOut[280]*si57*y - pOut[282]*rsq*si27 + pOut[482]*si26*z;
   pOut[603] = -pOut[280]*si57*z + pOut[480]*si26 + pOut[483]*si26*z;
   pOut[604] = -pOut[280]*si57 - rsq*si59*y - si53*xx*y + si58*y*zz;
   pOut[605] = -pOut[280]*si57 - pOut[282]*si21*y + rsq*si59*y - si58*y*zz;
   pOut[606] = -pOut[280]*si57 + pOut[483]*si73;
   pOut[607] = x*(-pOut[282]*si57 - rsq*si59 + si58*zz - si90*yy);
   pOut[608] = si91*x*y*z;
   pOut[609] = -pOut[282]*si57*z + pOut[482]*si26 + pOut[489]*si26*z;
   pOut[610] = -si136*x*y;
   pOut[611] = -pOut[282]*si135 + rsq*si59 + si134*yy - si58*zz;
   pOut[612] = 0;
   pOut[613] = -pOut[282]*si57 - rsq*si59 - si53*xx + si58*zz - si90*yy;
   pOut[614] = si91*y*z;
   pOut[615] = -si53*x*y;
   pOut[616] = -si91*y*z;
   pOut[617] = si91*x*y;
   pOut[618] = -pOut[282]*si57 + pOut[489]*si73;
   pOut[619] = si91*x*z;
   pOut[620] = 3*pOut[380]*z;
   pOut[621] = 3*pOut[381]*z;
   pOut[622] = 3*pOut[382]*z;
   pOut[623] = 3*pOut[380];
   pOut[624] = 3*pOut[384]*z;
   pOut[625] = 3*pOut[385]*z;
   pOut[626] = 0;
   pOut[627] = -si92*x*y*z;
   pOut[628] = 3*pOut[381];
   pOut[629] = 3*pOut[382];
   pOut[630] = si92*x*z;
   pOut[631] = si92*y*z;
   pOut[632] = 0;
   pOut[633] = -si92*y*z;
   pOut[634] = 3*pOut[384];
   pOut[635] = -si92*x*z;
   pOut[636] = 3*pOut[385];
   pOut[637] = 0;
   pOut[638] = 0;
   pOut[639] = -si92*x*y;
   pOut[640] = si28*(pOut[380]*y + pOut[440]*x);
   pOut[641] = si28*(pOut[381]*y + pOut[440] + pOut[441]*x);
   pOut[642] = si28*(pOut[380] + pOut[382]*y + pOut[442]*x);
   pOut[643] = 0;
   pOut[644] = pOut[384]*si28*y + pOut[441]*si93 + si94*xx*y;
   pOut[645] = pOut[382]*si93 + pOut[385]*si28*y - si94*xx*y;
   pOut[646] = 0;
   pOut[647] = pOut[381]*si28 + pOut[442]*si28 + pOut[447]*si28*x - si94*x*yy;
   pOut[648] = 0;
   pOut[649] = 0;
   pOut[650] = si137*x*y;
   pOut[651] = pOut[385]*si138 - si94*xx + si94*yy;
   pOut[652] = 0;
   pOut[653] = pOut[384]*si28 + pOut[447]*si93 + si94*xx - si94*yy;
   pOut[654] = 0;
   pOut[655] = -si137*x*y;
   pOut[656] = 0;
   pOut[657] = 0;
   pOut[658] = 0;
   pOut[659] = 0;
   pOut[660] = -pOut[220]*rsq*si29 + pOut[320]*si30*z;
   pOut[661] = -pOut[220]*si60*x + pOut[321]*si30*z + rsq*si61*x*z;
   pOut[662] = -pOut[220]*si60*y + pOut[322]*si30*z + rsq*si61*y*z;
   pOut[663] = -pOut[220]*si60*z - pOut[223]*rsq*si29 + pOut[320]*si30 + pOut[323]*si30*z;
   pOut[664] = -pOut[220]*si60 + pOut[324]*si30*z + rsq*si61*z + si95*xx*z;
   pOut[665] = -pOut[220]*si60 + pOut[325]*si30*z + rsq*si61*z + si95*yy*z;
   pOut[666] = -pOut[220]*si60 - pOut[223]*si98*z + pOut[323]*si96 + pOut[326]*si30*z - rsq*si97*z;
   pOut[667] = 15*x*y*z;
   pOut[668] = -pOut[223]*si60*x + pOut[321]*si30 + rsq*si61*x - si99*x*zz;
   pOut[669] = -pOut[223]*si60*y + pOut[322]*si30 + rsq*si61*y - si99*y*zz;
   pOut[670] = 45*x*z;
   pOut[671] = 45*y*z;
   pOut[672] = -pOut[223]*si97 + pOut[326]*si139 - rsq*si97 + si140*zz;
   pOut[673] = 15*y*z;
   pOut[674] = -pOut[223]*si60 + pOut[324]*si30 + rsq*si61 + si95*xx - si99*zz;
   pOut[675] = 15*x*z;
   pOut[676] = -pOut[223]*si60 + pOut[325]*si30 + rsq*si61 + si95*yy - si99*zz;
   pOut[677] = -60*x*z;
   pOut[678] = -60*y*z;
   pOut[679] = 15*x*y;
   pOut[680] = si28*(pOut[380]*x - pOut[440]*y);
   pOut[681] = si28*(pOut[380] + pOut[381]*x - pOut[441]*y);
   pOut[682] = si28*(pOut[382]*x - pOut[440] - pOut[442]*y);
   pOut[683] = 0;
   pOut[684] = pOut[381]*si93 + pOut[384]*si28*x - si94*x*yy;
   pOut[685] = pOut[385]*si28*x - pOut[442]*si93 + si94*x*yy;
   pOut[686] = 0;
   pOut[687] = pOut[382]*si28 - pOut[441]*si28 - pOut[447]*si28*y - si94*xx*y;
   pOut[688] = 0;
   pOut[689] = 0;
   pOut[690] = pOut[384]*si138 + si94*xx - si94*yy;
   pOut[691] = si137*x*y;
   pOut[692] = 0;
   pOut[693] = -si137*x*y;
   pOut[694] = 0;
   pOut[695] = pOut[385]*si28 - pOut[447]*si93 - si94*xx + si94*yy;
   pOut[696] = 0;
   pOut[697] = 0;
   pOut[698] = 0;
   pOut[699] = 0;
   pOut[700] = z*(pOut[340]*si25 - rsq*si31*x*y);
   pOut[701] = z*(pOut[341]*si25 - rsq*si31*y - si62*xx*y);
   pOut[702] = z*(pOut[342]*si25 - rsq*si31*x - si62*x*yy);
   pOut[703] = pOut[340]*si25 - rsq*si31*x*y + si63*x*y*zz;
   pOut[704] = -si100*x*y*z;
   pOut[705] = -si100*x*y*z;
   pOut[706] = si101*x*y*z;
   pOut[707] = z*(pOut[347]*si25 - rsq*si31 - si62*xx - si62*yy);
   pOut[708] = pOut[341]*si25 - rsq*si31*y - si62*xx*y + si63*y*zz;
   pOut[709] = pOut[342]*si25 - rsq*si31*x - si62*x*yy + si63*x*zz;
   pOut[710] = -si100*y*z;
   pOut[711] = -si100*x*z;
   pOut[712] = si101*x*y;
   pOut[713] = -si100*x*z;
   pOut[714] = -si100*x*y;
   pOut[715] = -si100*y*z;
   pOut[716] = -si100*x*y;
   pOut[717] = si101*y*z;
   pOut[718] = si101*x*z;
   pOut[719] = pOut[347]*si25 - rsq*si31 - si62*xx - si62*yy + si63*zz;
   // 1.26 steps per component, 0.00 kb stack
   if (L == 5) return;

   // Slm: [S(6,+6), S(6,-2), S(6,+5), S(6,+4), S(6,-5), S(6,+2), S(6,-6), S(6,+3), S(6,-4), S(6,+0), S(6,-3), S(6,-1), S(6,+1)]
   // Derivs: [1, D[.,x], D[.,y], D[.,z], D[.,xx], D[.,yy], D[.,zz], D[.,xy], D[.,xz], D[.,yz], D[.,xxx], D[.,yyy], D[.,zzz], D[.,xxy], D[.,xxz], D[.,xyy], D[.,yyz], D[.,xzz], D[.,yzz], D[.,xyz]]
   pOut[720] = si32*(-pOut[640]*y + pOut[680]*x);
   pOut[721] = si32*(-pOut[641]*y + pOut[680] + pOut[681]*x);
   pOut[722] = si32*(-pOut[640] - pOut[642]*y + pOut[682]*x);
   pOut[723] = 0;
   pOut[724] = -pOut[644]*si32*y + pOut[681]*si102 + pOut[684]*si32*x;
   pOut[725] = -pOut[642]*si102 - pOut[645]*si32*y + pOut[685]*si32*x;
   pOut[726] = 0;
   pOut[727] = si32*(-pOut[641] - pOut[647]*y + pOut[682] + pOut[687]*x);
   pOut[728] = 0;
   pOut[729] = 0;
   pOut[730] = pOut[684]*si141 + pOut[690]*si32*x - si142*x*yy;
   pOut[731] = -pOut[645]*si141 - pOut[651]*si32*y + si142*xx*y;
   pOut[732] = 0;
   pOut[733] = -pOut[644]*si32 - pOut[653]*si32*y + pOut[687]*si102 - si142*xx*y;
   pOut[734] = 0;
   pOut[735] = -pOut[647]*si102 + pOut[685]*si32 + pOut[695]*si32*x + si142*x*yy;
   pOut[736] = 0;
   pOut[737] = 0;
   pOut[738] = 0;
   pOut[739] = 0;
   pOut[740] = -pOut[340]*rsq*si33 + pOut[700]*si34*z;
   pOut[741] = -pOut[340]*si64*x - pOut[341]*rsq*si33 + pOut[701]*si34*z;
   pOut[742] = -pOut[340]*si64*y - pOut[342]*rsq*si33 + pOut[702]*si34*z;
   pOut[743] = -pOut[340]*si64*z + pOut[700]*si34 + pOut[703]*si34*z - rsq*si65*x*y*z;
   pOut[744] = -pOut[340]*si64 - pOut[341]*si103*x + rsq*si105*x*y - si104*x*y*zz;
   pOut[745] = -pOut[340]*si64 - pOut[342]*si103*y + rsq*si105*x*y - si104*x*y*zz;
   pOut[746] = -pOut[340]*si64 + pOut[703]*si106 - rsq*si65*x*y + si107*x*y*zz;
   pOut[747] = -pOut[341]*si64*y - pOut[342]*si64*x - pOut[347]*rsq*si33 + pOut[707]*si34*z;
   pOut[748] = -pOut[341]*si64*z + pOut[701]*si34 + pOut[708]*si34*z - rsq*si65*y*z - si108*xx*y*z;
   pOut[749] = -pOut[342]*si64*z + pOut[702]*si34 + pOut[709]*si34*z - rsq*si65*x*z - si108*x*yy*z;
   pOut[750] = -pOut[341]*si143 + rsq*si105*y - si104*y*zz + si144*xx*y;
   pOut[751] = -pOut[342]*si143 + rsq*si105*x - si104*x*zz + si144*x*yy;
   pOut[752] = si145*x*y*z;
   pOut[753] = -pOut[342]*si64 - pOut[347]*si103*x + rsq*si105*x - si104*x*zz + si65*x*yy;
   pOut[754] = -si146*x*y*z;
   pOut[755] = -pOut[341]*si64 - pOut[347]*si103*y + rsq*si105*y - si104*y*zz + si65*xx*y;
   pOut[756] = -si146*x*y*z;
   pOut[757] = -pOut[341]*si64 + pOut[708]*si106 - rsq*si65*y + si107*y*zz - si108*xx*y;
   pOut[758] = -pOut[342]*si64 + pOut[709]*si106 - rsq*si65*x + si107*x*zz - si108*x*yy;
   pOut[759] = -pOut[347]*si64*z + pOut[707]*si34 + pOut[719]*si34*z - rsq*si65*z - si108*xx*z - si108*yy*z;
   pOut[760] = pOut[680]*si35*z;
   pOut[761] = pOut[681]*si35*z;
   pOut[762] = pOut[682]*si35*z;
   pOut[763] = pOut[680]*si35;
   pOut[764] = pOut[684]*si35*z;
   pOut[765] = pOut[685]*si35*z;
   pOut[766] = 0;
   pOut[767] = pOut[687]*si35*z;
   pOut[768] = pOut[681]*si35;
   pOut[769] = pOut[682]*si35;
   pOut[770] = pOut[690]*si35*z;
   pOut[771] = si147*x*y*z;
   pOut[772] = 0;
   pOut[773] = -si147*x*y*z;
   pOut[774] = pOut[684]*si35;
   pOut[775] = pOut[695]*si35*z;
   pOut[776] = pOut[685]*si35;
   pOut[777] = 0;
   pOut[778] = 0;
   pOut[779] = pOut[687]*si35;
   pOut[780] = -pOut[380]*rsq*si36 + pOut[620]*si37*z;
   pOut[781] = -pOut[380]*si66*x - pOut[381]*rsq*si36 + pOut[621]*si37*z;
   pOut[782] = -pOut[380]*si66*y - pOut[382]*rsq*si36 + pOut[622]*si37*z;
   pOut[783] = -pOut[380]*si66*z + pOut[620]*si37 + pOut[623]*si37*z;
   pOut[784] = -pOut[380]*si66 - pOut[381]*si109*x - pOut[384]*rsq*si36 + pOut[624]*si37*z;
   pOut[785] = -pOut[380]*si66 - pOut[382]*si109*y - pOut[385]*rsq*si36 + pOut[625]*si37*z;
   pOut[786] = -pOut[380]*si66 + pOut[623]*si110;
   pOut[787] = -pOut[381]*si66*y - pOut[382]*si66*x + rsq*si112*x*y - si111*x*y*zz;
   pOut[788] = -pOut[381]*si66*z + pOut[621]*si37 + pOut[628]*si37*z;
   pOut[789] = -pOut[382]*si66*z + pOut[622]*si37 + pOut[629]*si37*z;
   pOut[790] = -pOut[381]*si148 - pOut[384]*si148*x - rsq*si112*x + si111*x*zz;
   pOut[791] = -pOut[382]*si148 - pOut[385]*si148*y - rsq*si112*y + si111*y*zz;
   pOut[792] = 0;
   pOut[793] = -pOut[382]*si66 - pOut[384]*si66*y + rsq*si112*y - si111*y*zz + si149*xx*y;
   pOut[794] = -pOut[384]*si66*z + pOut[624]*si37 + pOut[634]*si37*z;
   pOut[795] = -pOut[381]*si66 - pOut[385]*si66*x + rsq*si112*x - si111*x*zz + si149*x*yy;
   pOut[796] = -pOut[385]*si66*z + pOut[625]*si37 + pOut[636]*si37*z;
   pOut[797] = -pOut[381]*si66 + pOut[628]*si110;
   pOut[798] = -pOut[382]*si66 + pOut[629]*si110;
   pOut[799] = -si150*x*y*z;
   pOut[800] = pOut[640]*si35*z;
   pOut[801] = pOut[641]*si35*z;
   pOut[802] = pOut[642]*si35*z;
   pOut[803] = pOut[640]*si35;
   pOut[804] = pOut[644]*si35*z;
   pOut[805] = pOut[645]*si35*z;
   pOut[806] = 0;
   pOut[807] = pOut[647]*si35*z;
   pOut[808] = pOut[641]*si35;
   pOut[809] = pOut[642]*si35;
   pOut[810] = si147*x*y*z;
   pOut[811] = pOut[651]*si35*z;
   pOut[812] = 0;
   pOut[813] = pOut[653]*si35*z;
   pOut[814] = pOut[644]*si35;
   pOut[815] = -si147*x*y*z;
   pOut[816] = pOut[645]*si35;
   pOut[817] = 0;
   pOut[818] = 0;
   pOut[819] = pOut[647]*si35;
   pOut[820] = -pOut[420]*rsq*si33 + pOut[540]*si34*z;
   pOut[821] = -pOut[420]*si64*x - pOut[421]*rsq*si33 + pOut[541]*si34*z;
   pOut[822] = -pOut[420]*si64*y - pOut[422]*rsq*si33 + pOut[542]*si34*z;
   pOut[823] = -pOut[420]*si64*z - pOut[423]*rsq*si33 + pOut[540]*si34 + pOut[543]*si34*z;
   pOut[824] = -pOut[420]*si64 - pOut[421]*si103*x - pOut[424]*rsq*si33 + pOut[544]*si34*z;
   pOut[825] = -pOut[420]*si64 - pOut[422]*si103*y - pOut[425]*rsq*si33 + pOut[545]*si34*z;
   pOut[826] = -pOut[420]*si64 - pOut[423]*si103*z - pOut[426]*rsq*si33 + pOut[543]*si106 + pOut[546]*si34*z;
   pOut[827] = -si64*(pOut[421]*y + pOut[422]*x);
   pOut[828] = -pOut[421]*si64*z - pOut[423]*si64*x + pOut[541]*si34 + pOut[548]*si34*z - rsq*si65*x*z;
   pOut[829] = -pOut[422]*si64*z - pOut[423]*si64*y + pOut[542]*si34 + pOut[549]*si34*z + rsq*si65*y*z;
   pOut[830] = -pOut[421]*si143 - pOut[424]*si143*x + rsq*si65*x - si151*x*zz;
   pOut[831] = -pOut[422]*si143 - pOut[425]*si143*y - rsq*si65*y + si151*y*zz;
   pOut[832] = -pOut[423]*si143 - pOut[426]*si143*z + pOut[546]*si152 + pOut[552]*si34*z;
   pOut[833] = -si64*(pOut[422] + pOut[424]*y);
   pOut[834] = -pOut[423]*si64 - pOut[424]*si64*z + pOut[544]*si34 + pOut[554]*si34*z - rsq*si65*z - si153*xx*z;
   pOut[835] = -si64*(pOut[421] + pOut[425]*x);
   pOut[836] = -pOut[423]*si64 - pOut[425]*si64*z + pOut[545]*si34 + pOut[556]*si34*z + rsq*si65*z + si153*yy*z;
   pOut[837] = -pOut[421]*si64 - pOut[426]*si64*x + pOut[548]*si106 - rsq*si65*x + si107*x*zz;
   pOut[838] = -pOut[422]*si64 - pOut[426]*si64*y + pOut[549]*si106 + rsq*si65*y - si107*y*zz;
   pOut[839] = 0;
   pOut[840] = si32*(pOut[640]*x + pOut[680]*y);
   pOut[841] = si32*(pOut[640] + pOut[641]*x + pOut[681]*y);
   pOut[842] = si32*(pOut[642]*x + pOut[680] + pOut[682]*y);
   pOut[843] = 0;
   pOut[844] = pOut[641]*si102 + pOut[644]*si32*x + pOut[684]*si32*y;
   pOut[845] = pOut[645]*si32*x + pOut[682]*si102 + pOut[685]*si32*y;
   pOut[846] = 0;
   pOut[847] = si32*(pOut[642] + pOut[647]*x + pOut[681] + pOut[687]*y);
   pOut[848] = 0;
   pOut[849] = 0;
   pOut[850] = pOut[644]*si141 + pOut[690]*si32*y + si142*xx*y;
   pOut[851] = pOut[651]*si32*x + pOut[685]*si141 + si142*x*yy;
   pOut[852] = 0;
   pOut[853] = pOut[647]*si102 + pOut[653]*si32*x + pOut[684]*si32 - si142*x*yy;
   pOut[854] = 0;
   pOut[855] = pOut[645]*si32 + pOut[687]*si102 + pOut[695]*si32*y - si142*xx*y;
   pOut[856] = 0;
   pOut[857] = 0;
   pOut[858] = 0;
   pOut[859] = 0;
   pOut[860] = -pOut[460]*rsq*si38 + pOut[560]*si39*z;
   pOut[861] = -pOut[460]*si67*x - pOut[461]*rsq*si38 + pOut[561]*si39*z;
   pOut[862] = -pOut[460]*si67*y + pOut[562]*si39*z + rsq*si68*x*y*z;
   pOut[863] = -pOut[460]*si67*z - pOut[463]*rsq*si38 + pOut[560]*si39 + pOut[563]*si39*z;
   pOut[864] = -pOut[460]*si67 - pOut[461]*si113*x + pOut[564]*si39*z - rsq*si68*x*z;
   pOut[865] = -pOut[460]*si67 + pOut[565]*si39*z + rsq*si68*x*z + si114*x*yy*z;
   pOut[866] = -pOut[460]*si67 - pOut[463]*si113*z + pOut[563]*si115 + pOut[566]*si39*z;
   pOut[867] = -pOut[461]*si67*y + pOut[567]*si39*z + rsq*si68*y*z + si116*xx*y*z;
   pOut[868] = -pOut[461]*si67*z - pOut[463]*si67*x - pOut[468]*rsq*si38 + pOut[561]*si39 + pOut[568]*si39*z;
   pOut[869] = -pOut[463]*si67*y + pOut[562]*si39 + rsq*si68*x*y - si117*x*y*zz;
   pOut[870] = -pOut[461]*si154 + pOut[570]*si39*z - rsq*si68*z - si155*xx*z;
   pOut[871] = si156*x*y*z;
   pOut[872] = -pOut[463]*si154 + pOut[566]*si157;
   pOut[873] = si158*x*y*z;
   pOut[874] = -pOut[463]*si67 - pOut[468]*si113*x + pOut[564]*si39 - rsq*si68*x + si117*x*zz;
   pOut[875] = -pOut[461]*si67 + pOut[575]*si39*z + rsq*si68*z + si114*yy*z + si116*xx*z;
   pOut[876] = -pOut[463]*si67 + pOut[565]*si39 + rsq*si68*x + si114*x*yy - si117*x*zz;
   pOut[877] = -pOut[461]*si67 - pOut[468]*si113*z + pOut[568]*si115 + pOut[577]*si39*z;
   pOut[878] = -si159*x*y*z;
   pOut[879] = -pOut[468]*si67*y + pOut[567]*si39 + rsq*si68*y + si116*xx*y - si117*y*zz;
   pOut[880] = -pOut[440]*rsq*si36 + pOut[580]*si37*z;
   pOut[881] = -pOut[440]*si66*x - pOut[441]*rsq*si36 + pOut[581]*si37*z;
   pOut[882] = -pOut[440]*si66*y - pOut[442]*rsq*si36 + pOut[582]*si37*z;
   pOut[883] = -pOut[440]*si66*z + pOut[580]*si37 + pOut[583]*si37*z;
   pOut[884] = -pOut[440]*si66 - pOut[441]*si109*x - rsq*si112*x*y + si111*x*y*zz;
   pOut[885] = -pOut[440]*si66 - pOut[442]*si109*y + rsq*si112*x*y - si111*x*y*zz;
   pOut[886] = -pOut[440]*si66 + pOut[583]*si110;
   pOut[887] = -pOut[441]*si66*y - pOut[442]*si66*x - pOut[447]*rsq*si36 + pOut[587]*si37*z;
   pOut[888] = -pOut[441]*si66*z + pOut[581]*si37 + pOut[588]*si37*z;
   pOut[889] = -pOut[442]*si66*z + pOut[582]*si37 + pOut[589]*si37*z;
   pOut[890] = -pOut[441]*si148 - rsq*si112*y + si111*y*zz - si160*xx*y;
   pOut[891] = -pOut[442]*si148 + rsq*si112*x - si111*x*zz + si160*x*yy;
   pOut[892] = 0;
   pOut[893] = -pOut[442]*si66 - pOut[447]*si109*x - rsq*si112*x + si111*x*zz - si161*x*yy;
   pOut[894] = si150*x*y*z;
   pOut[895] = -pOut[441]*si66 - pOut[447]*si109*y + rsq*si112*y - si111*y*zz + si161*xx*y;
   pOut[896] = -si150*x*y*z;
   pOut[897] = -pOut[441]*si66 + pOut[588]*si110;
   pOut[898] = -pOut[442]*si66 + pOut[589]*si110;
   pOut[899] = -pOut[447]*si66*z + pOut[587]*si37 + pOut[599]*si37*z;
   pOut[900] = -pOut[320]*rsq*si40 + pOut[660]*si41*z;
   pOut[901] = -pOut[320]*si7*x - pOut[321]*rsq*si40 + pOut[661]*si41*z;
   pOut[902] = -pOut[320]*si7*y - pOut[322]*rsq*si40 + pOut[662]*si41*z;
   pOut[903] = -pOut[320]*si7*z - pOut[323]*rsq*si40 + pOut[660]*si41 + pOut[663]*si41*z;
   pOut[904] = -pOut[320]*si7 - pOut[321]*si118*x - pOut[324]*rsq*si40 + pOut[664]*si41*z;
   pOut[905] = -pOut[320]*si7 - pOut[322]*si118*y - pOut[325]*rsq*si40 + pOut[665]*si41*z;
   pOut[906] = -pOut[320]*si7 - pOut[323]*si118*z - pOut[326]*rsq*si40 + pOut[663]*si119 + pOut[666]*si41*z;
   pOut[907] = -pOut[321]*si7*y - pOut[322]*si7*x - rsq*si120*x*y + si121*x*y*zz;
   pOut[908] = -pOut[321]*si7*z - pOut[323]*si7*x + pOut[661]*si41 + pOut[668]*si41*z + 10*rsq*x*z;
   pOut[909] = -pOut[322]*si7*z - pOut[323]*si7*y + pOut[662]*si41 + pOut[669]*si41*z + 10*rsq*y*z;
   pOut[910] = -5*pOut[321] - 5*pOut[324]*x - rsq*si162*x + si163*x*zz;
   pOut[911] = -5*pOut[322] - 5*pOut[325]*y - rsq*si162*y + si163*y*zz;
   pOut[912] = -5*pOut[323] - 5*pOut[326]*z + pOut[666]*si164 + pOut[672]*si41*z - 20*rsq*z;
   pOut[913] = -pOut[322]*si7 - pOut[324]*si7*y - rsq*si120*y + si121*y*zz - 10*xx*y;
   pOut[914] = -pOut[323]*si7 - pOut[324]*si7*z + pOut[664]*si41 + pOut[674]*si41*z + 10*rsq*z + 40*xx*z;
   pOut[915] = -pOut[321]*si7 - pOut[325]*si7*x - rsq*si120*x + si121*x*zz - 10*x*yy;
   pOut[916] = -pOut[323]*si7 - pOut[325]*si7*z + pOut[665]*si41 + pOut[676]*si41*z + 10*rsq*z + 40*yy*z;
   pOut[917] = -pOut[321]*si7 - pOut[326]*si7*x + pOut[668]*si119 + 10*rsq*x - 70*x*zz;
   pOut[918] = -pOut[322]*si7 - pOut[326]*si7*y + pOut[669]*si119 + 10*rsq*y - 70*y*zz;
   pOut[919] = 90*x*y*z;
   pOut[920] = -pOut[480]*rsq*si38 + pOut[600]*si39*z;
   pOut[921] = -pOut[480]*si67*x + pOut[601]*si39*z - rsq*si68*x*y*z;
   pOut[922] = -pOut[480]*si67*y - pOut[482]*rsq*si38 + pOut[602]*si39*z;
   pOut[923] = -pOut[480]*si67*z - pOut[483]*rsq*si38 + pOut[600]*si39 + pOut[603]*si39*z;
   pOut[924] = -pOut[480]*si67 + pOut[604]*si39*z - rsq*si68*y*z - si114*xx*y*z;
   pOut[925] = -pOut[480]*si67 - pOut[482]*si113*y + pOut[605]*si39*z + rsq*si68*y*z;
   pOut[926] = -pOut[480]*si67 - pOut[483]*si113*z + pOut[603]*si115 + pOut[606]*si39*z;
   pOut[927] = -pOut[482]*si67*x + pOut[607]*si39*z - rsq*si68*x*z - si116*x*yy*z;
   pOut[928] = -pOut[483]*si67*x + pOut[601]*si39 - rsq*si68*x*y + si117*x*y*zz;
   pOut[929] = -pOut[482]*si67*z - pOut[483]*si67*y - pOut[489]*rsq*si38 + pOut[602]*si39 + pOut[609]*si39*z;
   pOut[930] = -si156*x*y*z;
   pOut[931] = -pOut[482]*si154 + pOut[611]*si39*z + rsq*si68*z + si155*yy*z;
   pOut[932] = -pOut[483]*si154 + pOut[606]*si157;
   pOut[933] = -pOut[482]*si67 + pOut[613]*si39*z - rsq*si68*z - si114*xx*z - si116*yy*z;
   pOut[934] = -pOut[483]*si67 + pOut[604]*si39 - rsq*si68*y - si114*xx*y + si117*y*zz;
   pOut[935] = -si158*x*y*z;
   pOut[936] = -pOut[483]*si67 - pOut[489]*si113*y + pOut[605]*si39 + rsq*si68*y - si117*y*zz;
   pOut[937] = si159*x*y*z;
   pOut[938] = -pOut[482]*si67 - pOut[489]*si113*z + pOut[609]*si115 + pOut[618]*si39*z;
   pOut[939] = -pOut[489]*si67*x + pOut[607]*si39 - rsq*si68*x - si116*x*yy + si117*x*zz;
   pOut[940] = -pOut[400]*rsq*si42 + pOut[520]*si43*z;
   pOut[941] = -pOut[400]*si69*x + pOut[521]*si43*z + rsq*si70*x*y*z;
   pOut[942] = -pOut[400]*si69*y - pOut[402]*rsq*si42 + pOut[522]*si43*z;
   pOut[943] = -pOut[400]*si69*z - pOut[403]*rsq*si42 + pOut[520]*si43 + pOut[523]*si43*z;
   pOut[944] = -pOut[400]*si69 + pOut[524]*si43*z + rsq*si70*y*z + si122*xx*y*z;
   pOut[945] = -pOut[400]*si69 - pOut[402]*si123*y + pOut[525]*si43*z + rsq*si124*y*z;
   pOut[946] = -pOut[400]*si69 - pOut[403]*si123*z + pOut[523]*si125 + pOut[526]*si43*z - rsq*si122*y*z;
   pOut[947] = -pOut[402]*si69*x + pOut[527]*si43*z + rsq*si70*x*z + si126*x*yy*z;
   pOut[948] = -pOut[403]*si69*x + pOut[521]*si43 + rsq*si70*x*y - si127*x*y*zz;
   pOut[949] = -pOut[402]*si69*z - pOut[403]*si69*y - pOut[409]*rsq*si42 + pOut[522]*si43 + pOut[529]*si43*z;
   pOut[950] = si165*x*y*z;
   pOut[951] = -pOut[402]*si166 + pOut[531]*si43*z + rsq*si124*z + si167*yy*z;
   pOut[952] = -pOut[403]*si166 + pOut[526]*si168 - rsq*si122*y + si169*y*zz;
   pOut[953] = -pOut[402]*si69 + pOut[533]*si43*z + rsq*si70*z + si122*xx*z + si126*yy*z;
   pOut[954] = -pOut[403]*si69 + pOut[524]*si43 + rsq*si70*y + si122*xx*y - si127*y*zz;
   pOut[955] = si165*x*y*z;
   pOut[956] = -pOut[403]*si69 - pOut[409]*si123*y + pOut[525]*si43 + rsq*si124*y - si170*y*zz;
   pOut[957] = -si171*x*y*z;
   pOut[958] = -pOut[402]*si69 - pOut[409]*si123*z + pOut[529]*si125 + pOut[538]*si43*z - rsq*si122*z - si172*yy*z;
   pOut[959] = -pOut[409]*si69*x + pOut[527]*si43 + rsq*si70*x + si126*x*yy - si127*x*zz;
   pOut[960] = -pOut[360]*rsq*si42 + pOut[500]*si43*z;
   pOut[961] = -pOut[360]*si69*x - pOut[361]*rsq*si42 + pOut[501]*si43*z;
   pOut[962] = -pOut[360]*si69*y + pOut[502]*si43*z + rsq*si70*x*y*z;
   pOut[963] = -pOut[360]*si69*z - pOut[363]*rsq*si42 + pOut[500]*si43 + pOut[503]*si43*z;
   pOut[964] = -pOut[360]*si69 - pOut[361]*si123*x + pOut[504]*si43*z + rsq*si124*x*z;
   pOut[965] = -pOut[360]*si69 + pOut[505]*si43*z + rsq*si70*x*z + si122*x*yy*z;
   pOut[966] = -pOut[360]*si69 - pOut[363]*si123*z + pOut[503]*si125 + pOut[506]*si43*z - rsq*si122*x*z;
   pOut[967] = -pOut[361]*si69*y + pOut[507]*si43*z + rsq*si70*y*z + si126*xx*y*z;
   pOut[968] = -pOut[361]*si69*z - pOut[363]*si69*x - pOut[368]*rsq*si42 + pOut[501]*si43 + pOut[508]*si43*z;
   pOut[969] = -pOut[363]*si69*y + pOut[502]*si43 + rsq*si70*x*y - si127*x*y*zz;
   pOut[970] = -pOut[361]*si166 + pOut[510]*si43*z + rsq*si124*z + si167*xx*z;
   pOut[971] = si165*x*y*z;
   pOut[972] = -pOut[363]*si166 + pOut[506]*si168 - rsq*si122*x + si169*x*zz;
   pOut[973] = si165*x*y*z;
   pOut[974] = -pOut[363]*si69 - pOut[368]*si123*x + pOut[504]*si43 + rsq*si124*x - si170*x*zz;
   pOut[975] = -pOut[361]*si69 + pOut[515]*si43*z + rsq*si70*z + si122*yy*z + si126*xx*z;
   pOut[976] = -pOut[363]*si69 + pOut[505]*si43 + rsq*si70*x + si122*x*yy - si127*x*zz;
   pOut[977] = -pOut[361]*si69 - pOut[368]*si123*z + pOut[508]*si125 + pOut[517]*si43*z - rsq*si122*z - si172*xx*z;
   pOut[978] = -si171*x*y*z;
   pOut[979] = -pOut[368]*si69*y + pOut[507]*si43 + rsq*si70*y + si126*xx*y - si127*y*zz;
   // 1.66 steps per component, 0.00 kb stack
   if (L == 6) return;

   assert(0);
}

// Calculate (a0|00)^m for a = 0..lab, from [0]^m and (a0|00)^(m+1) for a = 0..(lab-1).
// Inputs:
// - pAm1: [nCartX(lab-1)]-array containing (a0|00)^(m+1) elements
// - Im: the single base integral (00|00)^m at target m level
// - PmA is (P-A) (make as (B-A) * (beta/zeta) for numerical stability! see doi.org/10.1021/ct200239p appendix C)
// - PmQ is (P-Q), InvZeta is 1/zeta = 1/(alpha+beta). See PCCP 6 5119 (2004) eq. 11.
// Output:
// - pOut: a single [CartX(lab)]-array containing the (a0|00)^m integrals at the target m
void OsrrA_StepM(double *IR_RP pOut, double const *IR_RP pAm1, double Im, unsigned lab, double PmAx, double PmAy, double PmAz, double PmQx, double PmQy, double PmQz, double rho, double InvZeta)
{
   // lab == 0
   pOut[0] = Im;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 0) return;

   // lab == 1
   double riz = -rho*InvZeta;
   double rPmQx = riz*PmQx, rPmQy = riz*PmQy, rPmQz = riz*PmQz;
   pOut[1] = Im*PmAx + pAm1[0]*rPmQx;
   pOut[2] = Im*PmAy + pAm1[0]*rPmQy;
   pOut[3] = Im*PmAz + pAm1[0]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 1) return;

   // lab == 2
   double i2z = .5*InvZeta;
   pOut[4] = PmAx*pOut[1] + i2z*(Im + pAm1[0]*riz) + pAm1[1]*rPmQx;
   pOut[5] = PmAy*pOut[2] + i2z*(Im + pAm1[0]*riz) + pAm1[2]*rPmQy;
   pOut[6] = PmAz*pOut[3] + i2z*(Im + pAm1[0]*riz) + pAm1[3]*rPmQz;
   pOut[7] = PmAy*pOut[1] + pAm1[1]*rPmQy;
   pOut[8] = PmAz*pOut[1] + pAm1[1]*rPmQz;
   pOut[9] = PmAz*pOut[2] + pAm1[2]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 2) return;

   // lab == 3
   pOut[10] = PmAx*pOut[4] + 2*i2z*(pAm1[1]*riz + pOut[1]) + pAm1[4]*rPmQx;
   pOut[11] = PmAy*pOut[5] + 2*i2z*(pAm1[2]*riz + pOut[2]) + pAm1[5]*rPmQy;
   pOut[12] = PmAz*pOut[6] + 2*i2z*(pAm1[3]*riz + pOut[3]) + pAm1[6]*rPmQz;
   pOut[13] = PmAx*pOut[5] + pAm1[5]*rPmQx;
   pOut[14] = PmAx*pOut[6] + pAm1[6]*rPmQx;
   pOut[15] = PmAy*pOut[4] + pAm1[4]*rPmQy;
   pOut[16] = PmAy*pOut[6] + pAm1[6]*rPmQy;
   pOut[17] = PmAz*pOut[4] + pAm1[4]*rPmQz;
   pOut[18] = PmAz*pOut[5] + pAm1[5]*rPmQz;
   pOut[19] = PmAz*pOut[7] + pAm1[7]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 3) return;

   // lab == 4
   pOut[20] = PmAx*pOut[10] + 3*i2z*(pAm1[4]*riz + pOut[4]) + pAm1[10]*rPmQx;
   pOut[21] = PmAy*pOut[11] + 3*i2z*(pAm1[5]*riz + pOut[5]) + pAm1[11]*rPmQy;
   pOut[22] = PmAz*pOut[12] + 3*i2z*(pAm1[6]*riz + pOut[6]) + pAm1[12]*rPmQz;
   pOut[23] = PmAy*pOut[10] + pAm1[10]*rPmQy;
   pOut[24] = PmAx*pOut[11] + pAm1[11]*rPmQx;
   pOut[25] = PmAz*pOut[10] + pAm1[10]*rPmQz;
   pOut[26] = PmAx*pOut[12] + pAm1[12]*rPmQx;
   pOut[27] = PmAz*pOut[11] + pAm1[11]*rPmQz;
   pOut[28] = PmAy*pOut[12] + pAm1[12]*rPmQy;
   pOut[29] = PmAx*pOut[13] + i2z*(pAm1[5]*riz + pOut[5]) + pAm1[13]*rPmQx;
   pOut[30] = PmAx*pOut[14] + i2z*(pAm1[6]*riz + pOut[6]) + pAm1[14]*rPmQx;
   pOut[31] = PmAy*pOut[16] + i2z*(pAm1[6]*riz + pOut[6]) + pAm1[16]*rPmQy;
   pOut[32] = PmAy*pOut[14] + pAm1[14]*rPmQy;
   pOut[33] = PmAz*pOut[13] + pAm1[13]*rPmQz;
   pOut[34] = PmAz*pOut[15] + pAm1[15]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 4) return;

   // lab == 5
   pOut[35] = PmAx*pOut[20] + 4*i2z*(pAm1[10]*riz + pOut[10]) + pAm1[20]*rPmQx;
   pOut[36] = PmAy*pOut[21] + 4*i2z*(pAm1[11]*riz + pOut[11]) + pAm1[21]*rPmQy;
   pOut[37] = PmAz*pOut[22] + 4*i2z*(pAm1[12]*riz + pOut[12]) + pAm1[22]*rPmQz;
   pOut[38] = PmAx*pOut[21] + pAm1[21]*rPmQx;
   pOut[39] = PmAx*pOut[22] + pAm1[22]*rPmQx;
   pOut[40] = PmAy*pOut[20] + pAm1[20]*rPmQy;
   pOut[41] = PmAy*pOut[22] + pAm1[22]*rPmQy;
   pOut[42] = PmAz*pOut[20] + pAm1[20]*rPmQz;
   pOut[43] = PmAz*pOut[21] + pAm1[21]*rPmQz;
   pOut[44] = PmAy*pOut[23] + i2z*(pAm1[10]*riz + pOut[10]) + pAm1[23]*rPmQy;
   pOut[45] = PmAz*pOut[25] + i2z*(pAm1[10]*riz + pOut[10]) + pAm1[25]*rPmQz;
   pOut[46] = PmAx*pOut[24] + i2z*(pAm1[11]*riz + pOut[11]) + pAm1[24]*rPmQx;
   pOut[47] = PmAz*pOut[27] + i2z*(pAm1[11]*riz + pOut[11]) + pAm1[27]*rPmQz;
   pOut[48] = PmAx*pOut[26] + i2z*(pAm1[12]*riz + pOut[12]) + pAm1[26]*rPmQx;
   pOut[49] = PmAy*pOut[28] + i2z*(pAm1[12]*riz + pOut[12]) + pAm1[28]*rPmQy;
   pOut[50] = PmAz*pOut[23] + pAm1[23]*rPmQz;
   pOut[51] = PmAz*pOut[24] + pAm1[24]*rPmQz;
   pOut[52] = PmAy*pOut[26] + pAm1[26]*rPmQy;
   pOut[53] = PmAx*pOut[31] + pAm1[31]*rPmQx;
   pOut[54] = PmAy*pOut[30] + pAm1[30]*rPmQy;
   pOut[55] = PmAz*pOut[29] + pAm1[29]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 5) return;

   // lab == 6
   pOut[56] = PmAx*pOut[35] + 5*i2z*(pAm1[20]*riz + pOut[20]) + pAm1[35]*rPmQx;
   pOut[57] = PmAy*pOut[36] + 5*i2z*(pAm1[21]*riz + pOut[21]) + pAm1[36]*rPmQy;
   pOut[58] = PmAz*pOut[37] + 5*i2z*(pAm1[22]*riz + pOut[22]) + pAm1[37]*rPmQz;
   pOut[59] = PmAy*pOut[35] + pAm1[35]*rPmQy;
   pOut[60] = PmAx*pOut[36] + pAm1[36]*rPmQx;
   pOut[61] = PmAz*pOut[35] + pAm1[35]*rPmQz;
   pOut[62] = PmAx*pOut[37] + pAm1[37]*rPmQx;
   pOut[63] = PmAz*pOut[36] + pAm1[36]*rPmQz;
   pOut[64] = PmAy*pOut[37] + pAm1[37]*rPmQy;
   pOut[65] = PmAy*pOut[40] + i2z*(pAm1[20]*riz + pOut[20]) + pAm1[40]*rPmQy;
   pOut[66] = PmAz*pOut[42] + i2z*(pAm1[20]*riz + pOut[20]) + pAm1[42]*rPmQz;
   pOut[67] = PmAx*pOut[38] + i2z*(pAm1[21]*riz + pOut[21]) + pAm1[38]*rPmQx;
   pOut[68] = PmAx*pOut[39] + i2z*(pAm1[22]*riz + pOut[22]) + pAm1[39]*rPmQx;
   pOut[69] = PmAz*pOut[43] + i2z*(pAm1[21]*riz + pOut[21]) + pAm1[43]*rPmQz;
   pOut[70] = PmAy*pOut[41] + i2z*(pAm1[22]*riz + pOut[22]) + pAm1[41]*rPmQy;
   pOut[71] = PmAy*pOut[44] + 2*i2z*(pAm1[23]*riz + pOut[23]) + pAm1[44]*rPmQy;
   pOut[72] = PmAz*pOut[45] + 2*i2z*(pAm1[25]*riz + pOut[25]) + pAm1[45]*rPmQz;
   pOut[73] = PmAz*pOut[47] + 2*i2z*(pAm1[27]*riz + pOut[27]) + pAm1[47]*rPmQz;
   pOut[74] = PmAy*pOut[39] + pAm1[39]*rPmQy;
   pOut[75] = PmAz*pOut[38] + pAm1[38]*rPmQz;
   pOut[76] = PmAz*pOut[40] + pAm1[40]*rPmQz;
   pOut[77] = PmAy*pOut[45] + pAm1[45]*rPmQy;
   pOut[78] = PmAx*pOut[47] + pAm1[47]*rPmQx;
   pOut[79] = PmAz*pOut[44] + pAm1[44]*rPmQz;
   pOut[80] = PmAx*pOut[49] + pAm1[49]*rPmQx;
   pOut[81] = PmAz*pOut[46] + pAm1[46]*rPmQz;
   pOut[82] = PmAy*pOut[48] + pAm1[48]*rPmQy;
   pOut[83] = PmAx*pOut[53] + i2z*(pAm1[31]*riz + pOut[31]) + pAm1[53]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 6) return;

   // lab == 7
   pOut[84] = PmAx*pOut[56] + 6*i2z*(pAm1[35]*riz + pOut[35]) + pAm1[56]*rPmQx;
   pOut[85] = PmAy*pOut[57] + 6*i2z*(pAm1[36]*riz + pOut[36]) + pAm1[57]*rPmQy;
   pOut[86] = PmAz*pOut[58] + 6*i2z*(pAm1[37]*riz + pOut[37]) + pAm1[58]*rPmQz;
   pOut[87] = PmAx*pOut[57] + pAm1[57]*rPmQx;
   pOut[88] = PmAx*pOut[58] + pAm1[58]*rPmQx;
   pOut[89] = PmAy*pOut[56] + pAm1[56]*rPmQy;
   pOut[90] = PmAy*pOut[58] + pAm1[58]*rPmQy;
   pOut[91] = PmAz*pOut[56] + pAm1[56]*rPmQz;
   pOut[92] = PmAz*pOut[57] + pAm1[57]*rPmQz;
   pOut[93] = PmAy*pOut[59] + i2z*(pAm1[35]*riz + pOut[35]) + pAm1[59]*rPmQy;
   pOut[94] = PmAz*pOut[61] + i2z*(pAm1[35]*riz + pOut[35]) + pAm1[61]*rPmQz;
   pOut[95] = PmAx*pOut[60] + i2z*(pAm1[36]*riz + pOut[36]) + pAm1[60]*rPmQx;
   pOut[96] = PmAz*pOut[63] + i2z*(pAm1[36]*riz + pOut[36]) + pAm1[63]*rPmQz;
   pOut[97] = PmAx*pOut[62] + i2z*(pAm1[37]*riz + pOut[37]) + pAm1[62]*rPmQx;
   pOut[98] = PmAy*pOut[64] + i2z*(pAm1[37]*riz + pOut[37]) + pAm1[64]*rPmQy;
   pOut[99] = PmAx*pOut[67] + 2*i2z*(pAm1[38]*riz + pOut[38]) + pAm1[67]*rPmQx;
   pOut[100] = PmAx*pOut[68] + 2*i2z*(pAm1[39]*riz + pOut[39]) + pAm1[68]*rPmQx;
   pOut[101] = PmAy*pOut[65] + 2*i2z*(pAm1[40]*riz + pOut[40]) + pAm1[65]*rPmQy;
   pOut[102] = PmAy*pOut[70] + 2*i2z*(pAm1[41]*riz + pOut[41]) + pAm1[70]*rPmQy;
   pOut[103] = PmAz*pOut[66] + 2*i2z*(pAm1[42]*riz + pOut[42]) + pAm1[66]*rPmQz;
   pOut[104] = PmAz*pOut[69] + 2*i2z*(pAm1[43]*riz + pOut[43]) + pAm1[69]*rPmQz;
   pOut[105] = PmAz*pOut[59] + pAm1[59]*rPmQz;
   pOut[106] = PmAz*pOut[60] + pAm1[60]*rPmQz;
   pOut[107] = PmAy*pOut[62] + pAm1[62]*rPmQy;
   pOut[108] = PmAx*pOut[69] + pAm1[69]*rPmQx;
   pOut[109] = PmAx*pOut[70] + pAm1[70]*rPmQx;
   pOut[110] = PmAy*pOut[66] + pAm1[66]*rPmQy;
   pOut[111] = PmAy*pOut[68] + pAm1[68]*rPmQy;
   pOut[112] = PmAz*pOut[65] + pAm1[65]*rPmQz;
   pOut[113] = PmAz*pOut[67] + pAm1[67]*rPmQz;
   pOut[114] = PmAz*pOut[71] + pAm1[71]*rPmQz;
   pOut[115] = PmAy*pOut[72] + pAm1[72]*rPmQy;
   pOut[116] = PmAx*pOut[73] + pAm1[73]*rPmQx;
   pOut[117] = PmAy*pOut[77] + i2z*(pAm1[45]*riz + pOut[45]) + pAm1[77]*rPmQy;
   pOut[118] = PmAx*pOut[78] + i2z*(pAm1[47]*riz + pOut[47]) + pAm1[78]*rPmQx;
   pOut[119] = PmAx*pOut[80] + i2z*(pAm1[49]*riz + pOut[49]) + pAm1[80]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 7) return;

   // lab == 8
   pOut[120] = PmAx*pOut[84] + 7*i2z*(pAm1[56]*riz + pOut[56]) + pAm1[84]*rPmQx;
   pOut[121] = PmAy*pOut[85] + 7*i2z*(pAm1[57]*riz + pOut[57]) + pAm1[85]*rPmQy;
   pOut[122] = PmAz*pOut[86] + 7*i2z*(pAm1[58]*riz + pOut[58]) + pAm1[86]*rPmQz;
   pOut[123] = PmAy*pOut[84] + pAm1[84]*rPmQy;
   pOut[124] = PmAx*pOut[85] + pAm1[85]*rPmQx;
   pOut[125] = PmAz*pOut[84] + pAm1[84]*rPmQz;
   pOut[126] = PmAx*pOut[86] + pAm1[86]*rPmQx;
   pOut[127] = PmAz*pOut[85] + pAm1[85]*rPmQz;
   pOut[128] = PmAy*pOut[86] + pAm1[86]*rPmQy;
   pOut[129] = PmAy*pOut[89] + i2z*(pAm1[56]*riz + pOut[56]) + pAm1[89]*rPmQy;
   pOut[130] = PmAz*pOut[91] + i2z*(pAm1[56]*riz + pOut[56]) + pAm1[91]*rPmQz;
   pOut[131] = PmAx*pOut[87] + i2z*(pAm1[57]*riz + pOut[57]) + pAm1[87]*rPmQx;
   pOut[132] = PmAx*pOut[88] + i2z*(pAm1[58]*riz + pOut[58]) + pAm1[88]*rPmQx;
   pOut[133] = PmAz*pOut[92] + i2z*(pAm1[57]*riz + pOut[57]) + pAm1[92]*rPmQz;
   pOut[134] = PmAy*pOut[90] + i2z*(pAm1[58]*riz + pOut[58]) + pAm1[90]*rPmQy;
   pOut[135] = PmAy*pOut[93] + 2*i2z*(pAm1[59]*riz + pOut[59]) + pAm1[93]*rPmQy;
   pOut[136] = PmAx*pOut[95] + 2*i2z*(pAm1[60]*riz + pOut[60]) + pAm1[95]*rPmQx;
   pOut[137] = PmAz*pOut[94] + 2*i2z*(pAm1[61]*riz + pOut[61]) + pAm1[94]*rPmQz;
   pOut[138] = PmAx*pOut[97] + 2*i2z*(pAm1[62]*riz + pOut[62]) + pAm1[97]*rPmQx;
   pOut[139] = PmAz*pOut[96] + 2*i2z*(pAm1[63]*riz + pOut[63]) + pAm1[96]*rPmQz;
   pOut[140] = PmAy*pOut[98] + 2*i2z*(pAm1[64]*riz + pOut[64]) + pAm1[98]*rPmQy;
   pOut[141] = PmAx*pOut[99] + 3*i2z*(pAm1[67]*riz + pOut[67]) + pAm1[99]*rPmQx;
   pOut[142] = PmAx*pOut[100] + 3*i2z*(pAm1[68]*riz + pOut[68]) + pAm1[100]*rPmQx;
   pOut[143] = PmAy*pOut[102] + 3*i2z*(pAm1[70]*riz + pOut[70]) + pAm1[102]*rPmQy;
   pOut[144] = PmAy*pOut[88] + pAm1[88]*rPmQy;
   pOut[145] = PmAz*pOut[87] + pAm1[87]*rPmQz;
   pOut[146] = PmAz*pOut[89] + pAm1[89]*rPmQz;
   pOut[147] = PmAy*pOut[94] + pAm1[94]*rPmQy;
   pOut[148] = PmAx*pOut[96] + pAm1[96]*rPmQx;
   pOut[149] = PmAz*pOut[93] + pAm1[93]*rPmQz;
   pOut[150] = PmAx*pOut[98] + pAm1[98]*rPmQx;
   pOut[151] = PmAz*pOut[95] + pAm1[95]*rPmQz;
   pOut[152] = PmAy*pOut[97] + pAm1[97]*rPmQy;
   pOut[153] = PmAy*pOut[100] + pAm1[100]*rPmQy;
   pOut[154] = PmAx*pOut[102] + pAm1[102]*rPmQx;
   pOut[155] = PmAz*pOut[99] + pAm1[99]*rPmQz;
   pOut[156] = PmAx*pOut[104] + pAm1[104]*rPmQx;
   pOut[157] = PmAz*pOut[101] + pAm1[101]*rPmQz;
   pOut[158] = PmAy*pOut[103] + pAm1[103]*rPmQy;
   pOut[159] = PmAy*pOut[110] + i2z*(pAm1[66]*riz + pOut[66]) + pAm1[110]*rPmQy;
   pOut[160] = PmAx*pOut[108] + i2z*(pAm1[69]*riz + pOut[69]) + pAm1[108]*rPmQx;
   pOut[161] = PmAx*pOut[109] + i2z*(pAm1[70]*riz + pOut[70]) + pAm1[109]*rPmQx;
   pOut[162] = PmAz*pOut[114] + i2z*(pAm1[71]*riz + pOut[71]) + pAm1[114]*rPmQz;
   pOut[163] = PmAy*pOut[115] + i2z*(pAm1[72]*riz + pOut[72]) + pAm1[115]*rPmQy;
   pOut[164] = PmAx*pOut[116] + i2z*(pAm1[73]*riz + pOut[73]) + pAm1[116]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 8) return;

   // lab == 9
   pOut[165] = PmAx*pOut[120] + 8*i2z*(pAm1[84]*riz + pOut[84]) + pAm1[120]*rPmQx;
   pOut[166] = PmAy*pOut[121] + 8*i2z*(pAm1[85]*riz + pOut[85]) + pAm1[121]*rPmQy;
   pOut[167] = PmAz*pOut[122] + 8*i2z*(pAm1[86]*riz + pOut[86]) + pAm1[122]*rPmQz;
   pOut[168] = PmAx*pOut[121] + pAm1[121]*rPmQx;
   pOut[169] = PmAx*pOut[122] + pAm1[122]*rPmQx;
   pOut[170] = PmAy*pOut[120] + pAm1[120]*rPmQy;
   pOut[171] = PmAy*pOut[122] + pAm1[122]*rPmQy;
   pOut[172] = PmAz*pOut[120] + pAm1[120]*rPmQz;
   pOut[173] = PmAz*pOut[121] + pAm1[121]*rPmQz;
   pOut[174] = PmAy*pOut[123] + i2z*(pAm1[84]*riz + pOut[84]) + pAm1[123]*rPmQy;
   pOut[175] = PmAz*pOut[125] + i2z*(pAm1[84]*riz + pOut[84]) + pAm1[125]*rPmQz;
   pOut[176] = PmAx*pOut[124] + i2z*(pAm1[85]*riz + pOut[85]) + pAm1[124]*rPmQx;
   pOut[177] = PmAz*pOut[127] + i2z*(pAm1[85]*riz + pOut[85]) + pAm1[127]*rPmQz;
   pOut[178] = PmAx*pOut[126] + i2z*(pAm1[86]*riz + pOut[86]) + pAm1[126]*rPmQx;
   pOut[179] = PmAy*pOut[128] + i2z*(pAm1[86]*riz + pOut[86]) + pAm1[128]*rPmQy;
   pOut[180] = PmAx*pOut[131] + 2*i2z*(pAm1[87]*riz + pOut[87]) + pAm1[131]*rPmQx;
   pOut[181] = PmAx*pOut[132] + 2*i2z*(pAm1[88]*riz + pOut[88]) + pAm1[132]*rPmQx;
   pOut[182] = PmAy*pOut[129] + 2*i2z*(pAm1[89]*riz + pOut[89]) + pAm1[129]*rPmQy;
   pOut[183] = PmAy*pOut[134] + 2*i2z*(pAm1[90]*riz + pOut[90]) + pAm1[134]*rPmQy;
   pOut[184] = PmAz*pOut[130] + 2*i2z*(pAm1[91]*riz + pOut[91]) + pAm1[130]*rPmQz;
   pOut[185] = PmAz*pOut[133] + 2*i2z*(pAm1[92]*riz + pOut[92]) + pAm1[133]*rPmQz;
   pOut[186] = PmAy*pOut[135] + 3*i2z*(pAm1[93]*riz + pOut[93]) + pAm1[135]*rPmQy;
   pOut[187] = PmAz*pOut[137] + 3*i2z*(pAm1[94]*riz + pOut[94]) + pAm1[137]*rPmQz;
   pOut[188] = PmAx*pOut[136] + 3*i2z*(pAm1[95]*riz + pOut[95]) + pAm1[136]*rPmQx;
   pOut[189] = PmAz*pOut[139] + 3*i2z*(pAm1[96]*riz + pOut[96]) + pAm1[139]*rPmQz;
   pOut[190] = PmAx*pOut[138] + 3*i2z*(pAm1[97]*riz + pOut[97]) + pAm1[138]*rPmQx;
   pOut[191] = PmAy*pOut[140] + 3*i2z*(pAm1[98]*riz + pOut[98]) + pAm1[140]*rPmQy;
   pOut[192] = PmAz*pOut[123] + pAm1[123]*rPmQz;
   pOut[193] = PmAz*pOut[124] + pAm1[124]*rPmQz;
   pOut[194] = PmAy*pOut[126] + pAm1[126]*rPmQy;
   pOut[195] = PmAx*pOut[133] + pAm1[133]*rPmQx;
   pOut[196] = PmAx*pOut[134] + pAm1[134]*rPmQx;
   pOut[197] = PmAy*pOut[130] + pAm1[130]*rPmQy;
   pOut[198] = PmAy*pOut[132] + pAm1[132]*rPmQy;
   pOut[199] = PmAz*pOut[129] + pAm1[129]*rPmQz;
   pOut[200] = PmAz*pOut[131] + pAm1[131]*rPmQz;
   pOut[201] = PmAz*pOut[135] + pAm1[135]*rPmQz;
   pOut[202] = PmAy*pOut[137] + pAm1[137]*rPmQy;
   pOut[203] = PmAz*pOut[136] + pAm1[136]*rPmQz;
   pOut[204] = PmAy*pOut[138] + pAm1[138]*rPmQy;
   pOut[205] = PmAx*pOut[139] + pAm1[139]*rPmQx;
   pOut[206] = PmAx*pOut[140] + pAm1[140]*rPmQx;
   pOut[207] = PmAx*pOut[143] + pAm1[143]*rPmQx;
   pOut[208] = PmAy*pOut[142] + pAm1[142]*rPmQy;
   pOut[209] = PmAz*pOut[141] + pAm1[141]*rPmQz;
   pOut[210] = PmAy*pOut[147] + i2z*(pAm1[94]*riz + pOut[94]) + pAm1[147]*rPmQy;
   pOut[211] = PmAx*pOut[148] + i2z*(pAm1[96]*riz + pOut[96]) + pAm1[148]*rPmQx;
   pOut[212] = PmAx*pOut[150] + i2z*(pAm1[98]*riz + pOut[98]) + pAm1[150]*rPmQx;
   pOut[213] = PmAz*pOut[155] + i2z*(pAm1[99]*riz + pOut[99]) + pAm1[155]*rPmQz;
   pOut[214] = PmAy*pOut[153] + i2z*(pAm1[100]*riz + pOut[100]) + pAm1[153]*rPmQy;
   pOut[215] = PmAz*pOut[157] + i2z*(pAm1[101]*riz + pOut[101]) + pAm1[157]*rPmQz;
   pOut[216] = PmAx*pOut[154] + i2z*(pAm1[102]*riz + pOut[102]) + pAm1[154]*rPmQx;
   pOut[217] = PmAy*pOut[158] + i2z*(pAm1[103]*riz + pOut[103]) + pAm1[158]*rPmQy;
   pOut[218] = PmAx*pOut[156] + i2z*(pAm1[104]*riz + pOut[104]) + pAm1[156]*rPmQx;
   pOut[219] = PmAz*pOut[162] + 2*i2z*(pAm1[114]*riz + pOut[114]) + pAm1[162]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 9) return;

   // lab == 10
   pOut[220] = PmAx*pOut[165] + 9*i2z*(pAm1[120]*riz + pOut[120]) + pAm1[165]*rPmQx;
   pOut[221] = PmAy*pOut[166] + 9*i2z*(pAm1[121]*riz + pOut[121]) + pAm1[166]*rPmQy;
   pOut[222] = PmAz*pOut[167] + 9*i2z*(pAm1[122]*riz + pOut[122]) + pAm1[167]*rPmQz;
   pOut[223] = PmAy*pOut[165] + pAm1[165]*rPmQy;
   pOut[224] = PmAx*pOut[166] + pAm1[166]*rPmQx;
   pOut[225] = PmAz*pOut[165] + pAm1[165]*rPmQz;
   pOut[226] = PmAx*pOut[167] + pAm1[167]*rPmQx;
   pOut[227] = PmAz*pOut[166] + pAm1[166]*rPmQz;
   pOut[228] = PmAy*pOut[167] + pAm1[167]*rPmQy;
   pOut[229] = PmAy*pOut[170] + i2z*(pAm1[120]*riz + pOut[120]) + pAm1[170]*rPmQy;
   pOut[230] = PmAz*pOut[172] + i2z*(pAm1[120]*riz + pOut[120]) + pAm1[172]*rPmQz;
   pOut[231] = PmAx*pOut[168] + i2z*(pAm1[121]*riz + pOut[121]) + pAm1[168]*rPmQx;
   pOut[232] = PmAx*pOut[169] + i2z*(pAm1[122]*riz + pOut[122]) + pAm1[169]*rPmQx;
   pOut[233] = PmAz*pOut[173] + i2z*(pAm1[121]*riz + pOut[121]) + pAm1[173]*rPmQz;
   pOut[234] = PmAy*pOut[171] + i2z*(pAm1[122]*riz + pOut[122]) + pAm1[171]*rPmQy;
   pOut[235] = PmAy*pOut[174] + 2*i2z*(pAm1[123]*riz + pOut[123]) + pAm1[174]*rPmQy;
   pOut[236] = PmAx*pOut[176] + 2*i2z*(pAm1[124]*riz + pOut[124]) + pAm1[176]*rPmQx;
   pOut[237] = PmAz*pOut[175] + 2*i2z*(pAm1[125]*riz + pOut[125]) + pAm1[175]*rPmQz;
   pOut[238] = PmAx*pOut[178] + 2*i2z*(pAm1[126]*riz + pOut[126]) + pAm1[178]*rPmQx;
   pOut[239] = PmAz*pOut[177] + 2*i2z*(pAm1[127]*riz + pOut[127]) + pAm1[177]*rPmQz;
   pOut[240] = PmAy*pOut[179] + 2*i2z*(pAm1[128]*riz + pOut[128]) + pAm1[179]*rPmQy;
   pOut[241] = PmAy*pOut[182] + 3*i2z*(pAm1[129]*riz + pOut[129]) + pAm1[182]*rPmQy;
   pOut[242] = PmAz*pOut[184] + 3*i2z*(pAm1[130]*riz + pOut[130]) + pAm1[184]*rPmQz;
   pOut[243] = PmAx*pOut[180] + 3*i2z*(pAm1[131]*riz + pOut[131]) + pAm1[180]*rPmQx;
   pOut[244] = PmAx*pOut[181] + 3*i2z*(pAm1[132]*riz + pOut[132]) + pAm1[181]*rPmQx;
   pOut[245] = PmAz*pOut[185] + 3*i2z*(pAm1[133]*riz + pOut[133]) + pAm1[185]*rPmQz;
   pOut[246] = PmAy*pOut[183] + 3*i2z*(pAm1[134]*riz + pOut[134]) + pAm1[183]*rPmQy;
   pOut[247] = PmAy*pOut[186] + 4*i2z*(pAm1[135]*riz + pOut[135]) + pAm1[186]*rPmQy;
   pOut[248] = PmAz*pOut[187] + 4*i2z*(pAm1[137]*riz + pOut[137]) + pAm1[187]*rPmQz;
   pOut[249] = PmAz*pOut[189] + 4*i2z*(pAm1[139]*riz + pOut[139]) + pAm1[189]*rPmQz;
   pOut[250] = PmAy*pOut[169] + pAm1[169]*rPmQy;
   pOut[251] = PmAz*pOut[168] + pAm1[168]*rPmQz;
   pOut[252] = PmAz*pOut[170] + pAm1[170]*rPmQz;
   pOut[253] = PmAy*pOut[175] + pAm1[175]*rPmQy;
   pOut[254] = PmAx*pOut[177] + pAm1[177]*rPmQx;
   pOut[255] = PmAz*pOut[174] + pAm1[174]*rPmQz;
   pOut[256] = PmAx*pOut[179] + pAm1[179]*rPmQx;
   pOut[257] = PmAz*pOut[176] + pAm1[176]*rPmQz;
   pOut[258] = PmAy*pOut[178] + pAm1[178]*rPmQy;
   pOut[259] = PmAy*pOut[181] + pAm1[181]*rPmQy;
   pOut[260] = PmAx*pOut[183] + pAm1[183]*rPmQx;
   pOut[261] = PmAz*pOut[180] + pAm1[180]*rPmQz;
   pOut[262] = PmAx*pOut[185] + pAm1[185]*rPmQx;
   pOut[263] = PmAz*pOut[182] + pAm1[182]*rPmQz;
   pOut[264] = PmAy*pOut[184] + pAm1[184]*rPmQy;
   pOut[265] = PmAy*pOut[187] + pAm1[187]*rPmQy;
   pOut[266] = PmAx*pOut[189] + pAm1[189]*rPmQx;
   pOut[267] = PmAz*pOut[186] + pAm1[186]*rPmQz;
   pOut[268] = PmAx*pOut[191] + pAm1[191]*rPmQx;
   pOut[269] = PmAz*pOut[188] + pAm1[188]*rPmQz;
   pOut[270] = PmAy*pOut[190] + pAm1[190]*rPmQy;
   pOut[271] = PmAy*pOut[197] + i2z*(pAm1[130]*riz + pOut[130]) + pAm1[197]*rPmQy;
   pOut[272] = PmAx*pOut[195] + i2z*(pAm1[133]*riz + pOut[133]) + pAm1[195]*rPmQx;
   pOut[273] = PmAx*pOut[196] + i2z*(pAm1[134]*riz + pOut[134]) + pAm1[196]*rPmQx;
   pOut[274] = PmAz*pOut[201] + i2z*(pAm1[135]*riz + pOut[135]) + pAm1[201]*rPmQz;
   pOut[275] = PmAz*pOut[203] + i2z*(pAm1[136]*riz + pOut[136]) + pAm1[203]*rPmQz;
   pOut[276] = PmAy*pOut[202] + i2z*(pAm1[137]*riz + pOut[137]) + pAm1[202]*rPmQy;
   pOut[277] = PmAy*pOut[204] + i2z*(pAm1[138]*riz + pOut[138]) + pAm1[204]*rPmQy;
   pOut[278] = PmAx*pOut[205] + i2z*(pAm1[139]*riz + pOut[139]) + pAm1[205]*rPmQx;
   pOut[279] = PmAx*pOut[206] + i2z*(pAm1[140]*riz + pOut[140]) + pAm1[206]*rPmQx;
   pOut[280] = PmAz*pOut[209] + i2z*(pAm1[141]*riz + pOut[141]) + pAm1[209]*rPmQz;
   pOut[281] = PmAy*pOut[208] + i2z*(pAm1[142]*riz + pOut[142]) + pAm1[208]*rPmQy;
   pOut[282] = PmAx*pOut[207] + i2z*(pAm1[143]*riz + pOut[143]) + pAm1[207]*rPmQx;
   pOut[283] = PmAy*pOut[214] + 2*i2z*(pAm1[153]*riz + pOut[153]) + pAm1[214]*rPmQy;
   pOut[284] = PmAz*pOut[213] + 2*i2z*(pAm1[155]*riz + pOut[155]) + pAm1[213]*rPmQz;
   pOut[285] = PmAz*pOut[215] + 2*i2z*(pAm1[157]*riz + pOut[157]) + pAm1[215]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 10) return;

   // lab == 11
   pOut[286] = PmAx*pOut[220] + 10*i2z*(pAm1[165]*riz + pOut[165]) + pAm1[220]*rPmQx;
   pOut[287] = PmAy*pOut[221] + 10*i2z*(pAm1[166]*riz + pOut[166]) + pAm1[221]*rPmQy;
   pOut[288] = PmAz*pOut[222] + 10*i2z*(pAm1[167]*riz + pOut[167]) + pAm1[222]*rPmQz;
   pOut[289] = PmAx*pOut[221] + pAm1[221]*rPmQx;
   pOut[290] = PmAx*pOut[222] + pAm1[222]*rPmQx;
   pOut[291] = PmAy*pOut[220] + pAm1[220]*rPmQy;
   pOut[292] = PmAy*pOut[222] + pAm1[222]*rPmQy;
   pOut[293] = PmAz*pOut[220] + pAm1[220]*rPmQz;
   pOut[294] = PmAz*pOut[221] + pAm1[221]*rPmQz;
   pOut[295] = PmAy*pOut[223] + i2z*(pAm1[165]*riz + pOut[165]) + pAm1[223]*rPmQy;
   pOut[296] = PmAz*pOut[225] + i2z*(pAm1[165]*riz + pOut[165]) + pAm1[225]*rPmQz;
   pOut[297] = PmAx*pOut[224] + i2z*(pAm1[166]*riz + pOut[166]) + pAm1[224]*rPmQx;
   pOut[298] = PmAz*pOut[227] + i2z*(pAm1[166]*riz + pOut[166]) + pAm1[227]*rPmQz;
   pOut[299] = PmAx*pOut[226] + i2z*(pAm1[167]*riz + pOut[167]) + pAm1[226]*rPmQx;
   pOut[300] = PmAy*pOut[228] + i2z*(pAm1[167]*riz + pOut[167]) + pAm1[228]*rPmQy;
   pOut[301] = PmAx*pOut[231] + 2*i2z*(pAm1[168]*riz + pOut[168]) + pAm1[231]*rPmQx;
   pOut[302] = PmAx*pOut[232] + 2*i2z*(pAm1[169]*riz + pOut[169]) + pAm1[232]*rPmQx;
   pOut[303] = PmAy*pOut[229] + 2*i2z*(pAm1[170]*riz + pOut[170]) + pAm1[229]*rPmQy;
   pOut[304] = PmAy*pOut[234] + 2*i2z*(pAm1[171]*riz + pOut[171]) + pAm1[234]*rPmQy;
   pOut[305] = PmAz*pOut[230] + 2*i2z*(pAm1[172]*riz + pOut[172]) + pAm1[230]*rPmQz;
   pOut[306] = PmAz*pOut[233] + 2*i2z*(pAm1[173]*riz + pOut[173]) + pAm1[233]*rPmQz;
   pOut[307] = PmAy*pOut[235] + 3*i2z*(pAm1[174]*riz + pOut[174]) + pAm1[235]*rPmQy;
   pOut[308] = PmAz*pOut[237] + 3*i2z*(pAm1[175]*riz + pOut[175]) + pAm1[237]*rPmQz;
   pOut[309] = PmAx*pOut[236] + 3*i2z*(pAm1[176]*riz + pOut[176]) + pAm1[236]*rPmQx;
   pOut[310] = PmAz*pOut[239] + 3*i2z*(pAm1[177]*riz + pOut[177]) + pAm1[239]*rPmQz;
   pOut[311] = PmAx*pOut[238] + 3*i2z*(pAm1[178]*riz + pOut[178]) + pAm1[238]*rPmQx;
   pOut[312] = PmAy*pOut[240] + 3*i2z*(pAm1[179]*riz + pOut[179]) + pAm1[240]*rPmQy;
   pOut[313] = PmAx*pOut[243] + 4*i2z*(pAm1[180]*riz + pOut[180]) + pAm1[243]*rPmQx;
   pOut[314] = PmAx*pOut[244] + 4*i2z*(pAm1[181]*riz + pOut[181]) + pAm1[244]*rPmQx;
   pOut[315] = PmAy*pOut[241] + 4*i2z*(pAm1[182]*riz + pOut[182]) + pAm1[241]*rPmQy;
   pOut[316] = PmAy*pOut[246] + 4*i2z*(pAm1[183]*riz + pOut[183]) + pAm1[246]*rPmQy;
   pOut[317] = PmAz*pOut[242] + 4*i2z*(pAm1[184]*riz + pOut[184]) + pAm1[242]*rPmQz;
   pOut[318] = PmAz*pOut[245] + 4*i2z*(pAm1[185]*riz + pOut[185]) + pAm1[245]*rPmQz;
   pOut[319] = PmAz*pOut[223] + pAm1[223]*rPmQz;
   pOut[320] = PmAz*pOut[224] + pAm1[224]*rPmQz;
   pOut[321] = PmAy*pOut[226] + pAm1[226]*rPmQy;
   pOut[322] = PmAx*pOut[233] + pAm1[233]*rPmQx;
   pOut[323] = PmAx*pOut[234] + pAm1[234]*rPmQx;
   pOut[324] = PmAy*pOut[230] + pAm1[230]*rPmQy;
   pOut[325] = PmAy*pOut[232] + pAm1[232]*rPmQy;
   pOut[326] = PmAz*pOut[229] + pAm1[229]*rPmQz;
   pOut[327] = PmAz*pOut[231] + pAm1[231]*rPmQz;
   pOut[328] = PmAz*pOut[235] + pAm1[235]*rPmQz;
   pOut[329] = PmAy*pOut[237] + pAm1[237]*rPmQy;
   pOut[330] = PmAz*pOut[236] + pAm1[236]*rPmQz;
   pOut[331] = PmAy*pOut[238] + pAm1[238]*rPmQy;
   pOut[332] = PmAx*pOut[239] + pAm1[239]*rPmQx;
   pOut[333] = PmAx*pOut[240] + pAm1[240]*rPmQx;
   pOut[334] = PmAx*pOut[245] + pAm1[245]*rPmQx;
   pOut[335] = PmAx*pOut[246] + pAm1[246]*rPmQx;
   pOut[336] = PmAy*pOut[242] + pAm1[242]*rPmQy;
   pOut[337] = PmAy*pOut[244] + pAm1[244]*rPmQy;
   pOut[338] = PmAz*pOut[241] + pAm1[241]*rPmQz;
   pOut[339] = PmAz*pOut[243] + pAm1[243]*rPmQz;
   pOut[340] = PmAz*pOut[247] + pAm1[247]*rPmQz;
   pOut[341] = PmAy*pOut[248] + pAm1[248]*rPmQy;
   pOut[342] = PmAx*pOut[249] + pAm1[249]*rPmQx;
   pOut[343] = PmAy*pOut[253] + i2z*(pAm1[175]*riz + pOut[175]) + pAm1[253]*rPmQy;
   pOut[344] = PmAx*pOut[254] + i2z*(pAm1[177]*riz + pOut[177]) + pAm1[254]*rPmQx;
   pOut[345] = PmAx*pOut[256] + i2z*(pAm1[179]*riz + pOut[179]) + pAm1[256]*rPmQx;
   pOut[346] = PmAz*pOut[261] + i2z*(pAm1[180]*riz + pOut[180]) + pAm1[261]*rPmQz;
   pOut[347] = PmAy*pOut[259] + i2z*(pAm1[181]*riz + pOut[181]) + pAm1[259]*rPmQy;
   pOut[348] = PmAz*pOut[263] + i2z*(pAm1[182]*riz + pOut[182]) + pAm1[263]*rPmQz;
   pOut[349] = PmAx*pOut[260] + i2z*(pAm1[183]*riz + pOut[183]) + pAm1[260]*rPmQx;
   pOut[350] = PmAy*pOut[264] + i2z*(pAm1[184]*riz + pOut[184]) + pAm1[264]*rPmQy;
   pOut[351] = PmAx*pOut[262] + i2z*(pAm1[185]*riz + pOut[185]) + pAm1[262]*rPmQx;
   pOut[352] = PmAz*pOut[267] + i2z*(pAm1[186]*riz + pOut[186]) + pAm1[267]*rPmQz;
   pOut[353] = PmAy*pOut[265] + i2z*(pAm1[187]*riz + pOut[187]) + pAm1[265]*rPmQy;
   pOut[354] = PmAz*pOut[269] + i2z*(pAm1[188]*riz + pOut[188]) + pAm1[269]*rPmQz;
   pOut[355] = PmAx*pOut[266] + i2z*(pAm1[189]*riz + pOut[189]) + pAm1[266]*rPmQx;
   pOut[356] = PmAy*pOut[270] + i2z*(pAm1[190]*riz + pOut[190]) + pAm1[270]*rPmQy;
   pOut[357] = PmAx*pOut[268] + i2z*(pAm1[191]*riz + pOut[191]) + pAm1[268]*rPmQx;
   pOut[358] = PmAz*pOut[274] + 2*i2z*(pAm1[201]*riz + pOut[201]) + pAm1[274]*rPmQz;
   pOut[359] = PmAz*pOut[275] + 2*i2z*(pAm1[203]*riz + pOut[203]) + pAm1[275]*rPmQz;
   pOut[360] = PmAy*pOut[277] + 2*i2z*(pAm1[204]*riz + pOut[204]) + pAm1[277]*rPmQy;
   pOut[361] = PmAx*pOut[282] + 2*i2z*(pAm1[207]*riz + pOut[207]) + pAm1[282]*rPmQx;
   pOut[362] = PmAy*pOut[281] + 2*i2z*(pAm1[208]*riz + pOut[208]) + pAm1[281]*rPmQy;
   pOut[363] = PmAz*pOut[280] + 2*i2z*(pAm1[209]*riz + pOut[209]) + pAm1[280]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 11) return;

   // lab == 12
   pOut[364] = PmAx*pOut[286] + 11*i2z*(pAm1[220]*riz + pOut[220]) + pAm1[286]*rPmQx;
   pOut[365] = PmAy*pOut[287] + 11*i2z*(pAm1[221]*riz + pOut[221]) + pAm1[287]*rPmQy;
   pOut[366] = PmAz*pOut[288] + 11*i2z*(pAm1[222]*riz + pOut[222]) + pAm1[288]*rPmQz;
   pOut[367] = PmAy*pOut[286] + pAm1[286]*rPmQy;
   pOut[368] = PmAx*pOut[287] + pAm1[287]*rPmQx;
   pOut[369] = PmAz*pOut[286] + pAm1[286]*rPmQz;
   pOut[370] = PmAx*pOut[288] + pAm1[288]*rPmQx;
   pOut[371] = PmAz*pOut[287] + pAm1[287]*rPmQz;
   pOut[372] = PmAy*pOut[288] + pAm1[288]*rPmQy;
   pOut[373] = PmAy*pOut[291] + i2z*(pAm1[220]*riz + pOut[220]) + pAm1[291]*rPmQy;
   pOut[374] = PmAz*pOut[293] + i2z*(pAm1[220]*riz + pOut[220]) + pAm1[293]*rPmQz;
   pOut[375] = PmAx*pOut[289] + i2z*(pAm1[221]*riz + pOut[221]) + pAm1[289]*rPmQx;
   pOut[376] = PmAx*pOut[290] + i2z*(pAm1[222]*riz + pOut[222]) + pAm1[290]*rPmQx;
   pOut[377] = PmAz*pOut[294] + i2z*(pAm1[221]*riz + pOut[221]) + pAm1[294]*rPmQz;
   pOut[378] = PmAy*pOut[292] + i2z*(pAm1[222]*riz + pOut[222]) + pAm1[292]*rPmQy;
   pOut[379] = PmAy*pOut[295] + 2*i2z*(pAm1[223]*riz + pOut[223]) + pAm1[295]*rPmQy;
   pOut[380] = PmAx*pOut[297] + 2*i2z*(pAm1[224]*riz + pOut[224]) + pAm1[297]*rPmQx;
   pOut[381] = PmAz*pOut[296] + 2*i2z*(pAm1[225]*riz + pOut[225]) + pAm1[296]*rPmQz;
   pOut[382] = PmAx*pOut[299] + 2*i2z*(pAm1[226]*riz + pOut[226]) + pAm1[299]*rPmQx;
   pOut[383] = PmAz*pOut[298] + 2*i2z*(pAm1[227]*riz + pOut[227]) + pAm1[298]*rPmQz;
   pOut[384] = PmAy*pOut[300] + 2*i2z*(pAm1[228]*riz + pOut[228]) + pAm1[300]*rPmQy;
   pOut[385] = PmAy*pOut[303] + 3*i2z*(pAm1[229]*riz + pOut[229]) + pAm1[303]*rPmQy;
   pOut[386] = PmAz*pOut[305] + 3*i2z*(pAm1[230]*riz + pOut[230]) + pAm1[305]*rPmQz;
   pOut[387] = PmAx*pOut[301] + 3*i2z*(pAm1[231]*riz + pOut[231]) + pAm1[301]*rPmQx;
   pOut[388] = PmAx*pOut[302] + 3*i2z*(pAm1[232]*riz + pOut[232]) + pAm1[302]*rPmQx;
   pOut[389] = PmAz*pOut[306] + 3*i2z*(pAm1[233]*riz + pOut[233]) + pAm1[306]*rPmQz;
   pOut[390] = PmAy*pOut[304] + 3*i2z*(pAm1[234]*riz + pOut[234]) + pAm1[304]*rPmQy;
   pOut[391] = PmAy*pOut[307] + 4*i2z*(pAm1[235]*riz + pOut[235]) + pAm1[307]*rPmQy;
   pOut[392] = PmAx*pOut[309] + 4*i2z*(pAm1[236]*riz + pOut[236]) + pAm1[309]*rPmQx;
   pOut[393] = PmAz*pOut[308] + 4*i2z*(pAm1[237]*riz + pOut[237]) + pAm1[308]*rPmQz;
   pOut[394] = PmAx*pOut[311] + 4*i2z*(pAm1[238]*riz + pOut[238]) + pAm1[311]*rPmQx;
   pOut[395] = PmAz*pOut[310] + 4*i2z*(pAm1[239]*riz + pOut[239]) + pAm1[310]*rPmQz;
   pOut[396] = PmAy*pOut[312] + 4*i2z*(pAm1[240]*riz + pOut[240]) + pAm1[312]*rPmQy;
   pOut[397] = PmAx*pOut[313] + 5*i2z*(pAm1[243]*riz + pOut[243]) + pAm1[313]*rPmQx;
   pOut[398] = PmAx*pOut[314] + 5*i2z*(pAm1[244]*riz + pOut[244]) + pAm1[314]*rPmQx;
   pOut[399] = PmAy*pOut[316] + 5*i2z*(pAm1[246]*riz + pOut[246]) + pAm1[316]*rPmQy;
   pOut[400] = PmAy*pOut[290] + pAm1[290]*rPmQy;
   pOut[401] = PmAz*pOut[289] + pAm1[289]*rPmQz;
   pOut[402] = PmAz*pOut[291] + pAm1[291]*rPmQz;
   pOut[403] = PmAy*pOut[296] + pAm1[296]*rPmQy;
   pOut[404] = PmAx*pOut[298] + pAm1[298]*rPmQx;
   pOut[405] = PmAz*pOut[295] + pAm1[295]*rPmQz;
   pOut[406] = PmAx*pOut[300] + pAm1[300]*rPmQx;
   pOut[407] = PmAz*pOut[297] + pAm1[297]*rPmQz;
   pOut[408] = PmAy*pOut[299] + pAm1[299]*rPmQy;
   pOut[409] = PmAy*pOut[302] + pAm1[302]*rPmQy;
   pOut[410] = PmAx*pOut[304] + pAm1[304]*rPmQx;
   pOut[411] = PmAz*pOut[301] + pAm1[301]*rPmQz;
   pOut[412] = PmAx*pOut[306] + pAm1[306]*rPmQx;
   pOut[413] = PmAz*pOut[303] + pAm1[303]*rPmQz;
   pOut[414] = PmAy*pOut[305] + pAm1[305]*rPmQy;
   pOut[415] = PmAy*pOut[308] + pAm1[308]*rPmQy;
   pOut[416] = PmAx*pOut[310] + pAm1[310]*rPmQx;
   pOut[417] = PmAz*pOut[307] + pAm1[307]*rPmQz;
   pOut[418] = PmAx*pOut[312] + pAm1[312]*rPmQx;
   pOut[419] = PmAz*pOut[309] + pAm1[309]*rPmQz;
   pOut[420] = PmAy*pOut[311] + pAm1[311]*rPmQy;
   pOut[421] = PmAy*pOut[314] + pAm1[314]*rPmQy;
   pOut[422] = PmAx*pOut[316] + pAm1[316]*rPmQx;
   pOut[423] = PmAz*pOut[313] + pAm1[313]*rPmQz;
   pOut[424] = PmAx*pOut[318] + pAm1[318]*rPmQx;
   pOut[425] = PmAz*pOut[315] + pAm1[315]*rPmQz;
   pOut[426] = PmAy*pOut[317] + pAm1[317]*rPmQy;
   pOut[427] = PmAy*pOut[324] + i2z*(pAm1[230]*riz + pOut[230]) + pAm1[324]*rPmQy;
   pOut[428] = PmAx*pOut[322] + i2z*(pAm1[233]*riz + pOut[233]) + pAm1[322]*rPmQx;
   pOut[429] = PmAx*pOut[323] + i2z*(pAm1[234]*riz + pOut[234]) + pAm1[323]*rPmQx;
   pOut[430] = PmAz*pOut[328] + i2z*(pAm1[235]*riz + pOut[235]) + pAm1[328]*rPmQz;
   pOut[431] = PmAz*pOut[330] + i2z*(pAm1[236]*riz + pOut[236]) + pAm1[330]*rPmQz;
   pOut[432] = PmAy*pOut[329] + i2z*(pAm1[237]*riz + pOut[237]) + pAm1[329]*rPmQy;
   pOut[433] = PmAy*pOut[331] + i2z*(pAm1[238]*riz + pOut[238]) + pAm1[331]*rPmQy;
   pOut[434] = PmAx*pOut[332] + i2z*(pAm1[239]*riz + pOut[239]) + pAm1[332]*rPmQx;
   pOut[435] = PmAx*pOut[333] + i2z*(pAm1[240]*riz + pOut[240]) + pAm1[333]*rPmQx;
   pOut[436] = PmAz*pOut[338] + i2z*(pAm1[241]*riz + pOut[241]) + pAm1[338]*rPmQz;
   pOut[437] = PmAy*pOut[336] + i2z*(pAm1[242]*riz + pOut[242]) + pAm1[336]*rPmQy;
   pOut[438] = PmAz*pOut[339] + i2z*(pAm1[243]*riz + pOut[243]) + pAm1[339]*rPmQz;
   pOut[439] = PmAy*pOut[337] + i2z*(pAm1[244]*riz + pOut[244]) + pAm1[337]*rPmQy;
   pOut[440] = PmAx*pOut[334] + i2z*(pAm1[245]*riz + pOut[245]) + pAm1[334]*rPmQx;
   pOut[441] = PmAx*pOut[335] + i2z*(pAm1[246]*riz + pOut[246]) + pAm1[335]*rPmQx;
   pOut[442] = PmAz*pOut[340] + i2z*(pAm1[247]*riz + pOut[247]) + pAm1[340]*rPmQz;
   pOut[443] = PmAy*pOut[341] + i2z*(pAm1[248]*riz + pOut[248]) + pAm1[341]*rPmQy;
   pOut[444] = PmAx*pOut[342] + i2z*(pAm1[249]*riz + pOut[249]) + pAm1[342]*rPmQx;
   pOut[445] = PmAy*pOut[347] + 2*i2z*(pAm1[259]*riz + pOut[259]) + pAm1[347]*rPmQy;
   pOut[446] = PmAz*pOut[346] + 2*i2z*(pAm1[261]*riz + pOut[261]) + pAm1[346]*rPmQz;
   pOut[447] = PmAz*pOut[348] + 2*i2z*(pAm1[263]*riz + pOut[263]) + pAm1[348]*rPmQz;
   pOut[448] = PmAy*pOut[353] + 2*i2z*(pAm1[265]*riz + pOut[265]) + pAm1[353]*rPmQy;
   pOut[449] = PmAx*pOut[355] + 2*i2z*(pAm1[266]*riz + pOut[266]) + pAm1[355]*rPmQx;
   pOut[450] = PmAz*pOut[352] + 2*i2z*(pAm1[267]*riz + pOut[267]) + pAm1[352]*rPmQz;
   pOut[451] = PmAx*pOut[357] + 2*i2z*(pAm1[268]*riz + pOut[268]) + pAm1[357]*rPmQx;
   pOut[452] = PmAz*pOut[354] + 2*i2z*(pAm1[269]*riz + pOut[269]) + pAm1[354]*rPmQz;
   pOut[453] = PmAy*pOut[356] + 2*i2z*(pAm1[270]*riz + pOut[270]) + pAm1[356]*rPmQy;
   pOut[454] = PmAx*pOut[361] + 3*i2z*(pAm1[282]*riz + pOut[282]) + pAm1[361]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 12) return;

   // lab == 13
   pOut[455] = PmAx*pOut[364] + 12*i2z*(pAm1[286]*riz + pOut[286]) + pAm1[364]*rPmQx;
   pOut[456] = PmAy*pOut[365] + 12*i2z*(pAm1[287]*riz + pOut[287]) + pAm1[365]*rPmQy;
   pOut[457] = PmAz*pOut[366] + 12*i2z*(pAm1[288]*riz + pOut[288]) + pAm1[366]*rPmQz;
   pOut[458] = PmAx*pOut[365] + pAm1[365]*rPmQx;
   pOut[459] = PmAx*pOut[366] + pAm1[366]*rPmQx;
   pOut[460] = PmAy*pOut[364] + pAm1[364]*rPmQy;
   pOut[461] = PmAy*pOut[366] + pAm1[366]*rPmQy;
   pOut[462] = PmAz*pOut[364] + pAm1[364]*rPmQz;
   pOut[463] = PmAz*pOut[365] + pAm1[365]*rPmQz;
   pOut[464] = PmAy*pOut[367] + i2z*(pAm1[286]*riz + pOut[286]) + pAm1[367]*rPmQy;
   pOut[465] = PmAz*pOut[369] + i2z*(pAm1[286]*riz + pOut[286]) + pAm1[369]*rPmQz;
   pOut[466] = PmAx*pOut[368] + i2z*(pAm1[287]*riz + pOut[287]) + pAm1[368]*rPmQx;
   pOut[467] = PmAz*pOut[371] + i2z*(pAm1[287]*riz + pOut[287]) + pAm1[371]*rPmQz;
   pOut[468] = PmAx*pOut[370] + i2z*(pAm1[288]*riz + pOut[288]) + pAm1[370]*rPmQx;
   pOut[469] = PmAy*pOut[372] + i2z*(pAm1[288]*riz + pOut[288]) + pAm1[372]*rPmQy;
   pOut[470] = PmAx*pOut[375] + 2*i2z*(pAm1[289]*riz + pOut[289]) + pAm1[375]*rPmQx;
   pOut[471] = PmAx*pOut[376] + 2*i2z*(pAm1[290]*riz + pOut[290]) + pAm1[376]*rPmQx;
   pOut[472] = PmAy*pOut[373] + 2*i2z*(pAm1[291]*riz + pOut[291]) + pAm1[373]*rPmQy;
   pOut[473] = PmAy*pOut[378] + 2*i2z*(pAm1[292]*riz + pOut[292]) + pAm1[378]*rPmQy;
   pOut[474] = PmAz*pOut[374] + 2*i2z*(pAm1[293]*riz + pOut[293]) + pAm1[374]*rPmQz;
   pOut[475] = PmAz*pOut[377] + 2*i2z*(pAm1[294]*riz + pOut[294]) + pAm1[377]*rPmQz;
   pOut[476] = PmAy*pOut[379] + 3*i2z*(pAm1[295]*riz + pOut[295]) + pAm1[379]*rPmQy;
   pOut[477] = PmAz*pOut[381] + 3*i2z*(pAm1[296]*riz + pOut[296]) + pAm1[381]*rPmQz;
   pOut[478] = PmAx*pOut[380] + 3*i2z*(pAm1[297]*riz + pOut[297]) + pAm1[380]*rPmQx;
   pOut[479] = PmAz*pOut[383] + 3*i2z*(pAm1[298]*riz + pOut[298]) + pAm1[383]*rPmQz;
   pOut[480] = PmAx*pOut[382] + 3*i2z*(pAm1[299]*riz + pOut[299]) + pAm1[382]*rPmQx;
   pOut[481] = PmAy*pOut[384] + 3*i2z*(pAm1[300]*riz + pOut[300]) + pAm1[384]*rPmQy;
   pOut[482] = PmAx*pOut[387] + 4*i2z*(pAm1[301]*riz + pOut[301]) + pAm1[387]*rPmQx;
   pOut[483] = PmAx*pOut[388] + 4*i2z*(pAm1[302]*riz + pOut[302]) + pAm1[388]*rPmQx;
   pOut[484] = PmAy*pOut[385] + 4*i2z*(pAm1[303]*riz + pOut[303]) + pAm1[385]*rPmQy;
   pOut[485] = PmAy*pOut[390] + 4*i2z*(pAm1[304]*riz + pOut[304]) + pAm1[390]*rPmQy;
   pOut[486] = PmAz*pOut[386] + 4*i2z*(pAm1[305]*riz + pOut[305]) + pAm1[386]*rPmQz;
   pOut[487] = PmAz*pOut[389] + 4*i2z*(pAm1[306]*riz + pOut[306]) + pAm1[389]*rPmQz;
   pOut[488] = PmAy*pOut[391] + 5*i2z*(pAm1[307]*riz + pOut[307]) + pAm1[391]*rPmQy;
   pOut[489] = PmAz*pOut[393] + 5*i2z*(pAm1[308]*riz + pOut[308]) + pAm1[393]*rPmQz;
   pOut[490] = PmAx*pOut[392] + 5*i2z*(pAm1[309]*riz + pOut[309]) + pAm1[392]*rPmQx;
   pOut[491] = PmAz*pOut[395] + 5*i2z*(pAm1[310]*riz + pOut[310]) + pAm1[395]*rPmQz;
   pOut[492] = PmAx*pOut[394] + 5*i2z*(pAm1[311]*riz + pOut[311]) + pAm1[394]*rPmQx;
   pOut[493] = PmAy*pOut[396] + 5*i2z*(pAm1[312]*riz + pOut[312]) + pAm1[396]*rPmQy;
   pOut[494] = PmAz*pOut[367] + pAm1[367]*rPmQz;
   pOut[495] = PmAz*pOut[368] + pAm1[368]*rPmQz;
   pOut[496] = PmAy*pOut[370] + pAm1[370]*rPmQy;
   pOut[497] = PmAx*pOut[377] + pAm1[377]*rPmQx;
   pOut[498] = PmAx*pOut[378] + pAm1[378]*rPmQx;
   pOut[499] = PmAy*pOut[374] + pAm1[374]*rPmQy;
   pOut[500] = PmAy*pOut[376] + pAm1[376]*rPmQy;
   pOut[501] = PmAz*pOut[373] + pAm1[373]*rPmQz;
   pOut[502] = PmAz*pOut[375] + pAm1[375]*rPmQz;
   pOut[503] = PmAz*pOut[379] + pAm1[379]*rPmQz;
   pOut[504] = PmAy*pOut[381] + pAm1[381]*rPmQy;
   pOut[505] = PmAz*pOut[380] + pAm1[380]*rPmQz;
   pOut[506] = PmAy*pOut[382] + pAm1[382]*rPmQy;
   pOut[507] = PmAx*pOut[383] + pAm1[383]*rPmQx;
   pOut[508] = PmAx*pOut[384] + pAm1[384]*rPmQx;
   pOut[509] = PmAx*pOut[389] + pAm1[389]*rPmQx;
   pOut[510] = PmAx*pOut[390] + pAm1[390]*rPmQx;
   pOut[511] = PmAy*pOut[386] + pAm1[386]*rPmQy;
   pOut[512] = PmAy*pOut[388] + pAm1[388]*rPmQy;
   pOut[513] = PmAz*pOut[385] + pAm1[385]*rPmQz;
   pOut[514] = PmAz*pOut[387] + pAm1[387]*rPmQz;
   pOut[515] = PmAz*pOut[391] + pAm1[391]*rPmQz;
   pOut[516] = PmAy*pOut[393] + pAm1[393]*rPmQy;
   pOut[517] = PmAz*pOut[392] + pAm1[392]*rPmQz;
   pOut[518] = PmAy*pOut[394] + pAm1[394]*rPmQy;
   pOut[519] = PmAx*pOut[395] + pAm1[395]*rPmQx;
   pOut[520] = PmAx*pOut[396] + pAm1[396]*rPmQx;
   pOut[521] = PmAx*pOut[399] + pAm1[399]*rPmQx;
   pOut[522] = PmAy*pOut[398] + pAm1[398]*rPmQy;
   pOut[523] = PmAz*pOut[397] + pAm1[397]*rPmQz;
   pOut[524] = PmAy*pOut[403] + i2z*(pAm1[296]*riz + pOut[296]) + pAm1[403]*rPmQy;
   pOut[525] = PmAx*pOut[404] + i2z*(pAm1[298]*riz + pOut[298]) + pAm1[404]*rPmQx;
   pOut[526] = PmAx*pOut[406] + i2z*(pAm1[300]*riz + pOut[300]) + pAm1[406]*rPmQx;
   pOut[527] = PmAz*pOut[411] + i2z*(pAm1[301]*riz + pOut[301]) + pAm1[411]*rPmQz;
   pOut[528] = PmAy*pOut[409] + i2z*(pAm1[302]*riz + pOut[302]) + pAm1[409]*rPmQy;
   pOut[529] = PmAz*pOut[413] + i2z*(pAm1[303]*riz + pOut[303]) + pAm1[413]*rPmQz;
   pOut[530] = PmAx*pOut[410] + i2z*(pAm1[304]*riz + pOut[304]) + pAm1[410]*rPmQx;
   pOut[531] = PmAy*pOut[414] + i2z*(pAm1[305]*riz + pOut[305]) + pAm1[414]*rPmQy;
   pOut[532] = PmAx*pOut[412] + i2z*(pAm1[306]*riz + pOut[306]) + pAm1[412]*rPmQx;
   pOut[533] = PmAz*pOut[417] + i2z*(pAm1[307]*riz + pOut[307]) + pAm1[417]*rPmQz;
   pOut[534] = PmAy*pOut[415] + i2z*(pAm1[308]*riz + pOut[308]) + pAm1[415]*rPmQy;
   pOut[535] = PmAz*pOut[419] + i2z*(pAm1[309]*riz + pOut[309]) + pAm1[419]*rPmQz;
   pOut[536] = PmAx*pOut[416] + i2z*(pAm1[310]*riz + pOut[310]) + pAm1[416]*rPmQx;
   pOut[537] = PmAy*pOut[420] + i2z*(pAm1[311]*riz + pOut[311]) + pAm1[420]*rPmQy;
   pOut[538] = PmAx*pOut[418] + i2z*(pAm1[312]*riz + pOut[312]) + pAm1[418]*rPmQx;
   pOut[539] = PmAz*pOut[423] + i2z*(pAm1[313]*riz + pOut[313]) + pAm1[423]*rPmQz;
   pOut[540] = PmAy*pOut[421] + i2z*(pAm1[314]*riz + pOut[314]) + pAm1[421]*rPmQy;
   pOut[541] = PmAz*pOut[425] + i2z*(pAm1[315]*riz + pOut[315]) + pAm1[425]*rPmQz;
   pOut[542] = PmAx*pOut[422] + i2z*(pAm1[316]*riz + pOut[316]) + pAm1[422]*rPmQx;
   pOut[543] = PmAy*pOut[426] + i2z*(pAm1[317]*riz + pOut[317]) + pAm1[426]*rPmQy;
   pOut[544] = PmAx*pOut[424] + i2z*(pAm1[318]*riz + pOut[318]) + pAm1[424]*rPmQx;
   pOut[545] = PmAz*pOut[430] + 2*i2z*(pAm1[328]*riz + pOut[328]) + pAm1[430]*rPmQz;
   pOut[546] = PmAz*pOut[431] + 2*i2z*(pAm1[330]*riz + pOut[330]) + pAm1[431]*rPmQz;
   pOut[547] = PmAy*pOut[433] + 2*i2z*(pAm1[331]*riz + pOut[331]) + pAm1[433]*rPmQy;
   pOut[548] = PmAx*pOut[440] + 2*i2z*(pAm1[334]*riz + pOut[334]) + pAm1[440]*rPmQx;
   pOut[549] = PmAx*pOut[441] + 2*i2z*(pAm1[335]*riz + pOut[335]) + pAm1[441]*rPmQx;
   pOut[550] = PmAy*pOut[437] + 2*i2z*(pAm1[336]*riz + pOut[336]) + pAm1[437]*rPmQy;
   pOut[551] = PmAy*pOut[439] + 2*i2z*(pAm1[337]*riz + pOut[337]) + pAm1[439]*rPmQy;
   pOut[552] = PmAz*pOut[436] + 2*i2z*(pAm1[338]*riz + pOut[338]) + pAm1[436]*rPmQz;
   pOut[553] = PmAz*pOut[438] + 2*i2z*(pAm1[339]*riz + pOut[339]) + pAm1[438]*rPmQz;
   pOut[554] = PmAz*pOut[442] + 2*i2z*(pAm1[340]*riz + pOut[340]) + pAm1[442]*rPmQz;
   pOut[555] = PmAy*pOut[443] + 2*i2z*(pAm1[341]*riz + pOut[341]) + pAm1[443]*rPmQy;
   pOut[556] = PmAx*pOut[444] + 2*i2z*(pAm1[342]*riz + pOut[342]) + pAm1[444]*rPmQx;
   pOut[557] = PmAy*pOut[448] + 3*i2z*(pAm1[353]*riz + pOut[353]) + pAm1[448]*rPmQy;
   pOut[558] = PmAx*pOut[449] + 3*i2z*(pAm1[355]*riz + pOut[355]) + pAm1[449]*rPmQx;
   pOut[559] = PmAx*pOut[451] + 3*i2z*(pAm1[357]*riz + pOut[357]) + pAm1[451]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 13) return;

   // lab == 14
   pOut[560] = PmAx*pOut[455] + 13*i2z*(pAm1[364]*riz + pOut[364]) + pAm1[455]*rPmQx;
   pOut[561] = PmAy*pOut[456] + 13*i2z*(pAm1[365]*riz + pOut[365]) + pAm1[456]*rPmQy;
   pOut[562] = PmAz*pOut[457] + 13*i2z*(pAm1[366]*riz + pOut[366]) + pAm1[457]*rPmQz;
   pOut[563] = PmAy*pOut[455] + pAm1[455]*rPmQy;
   pOut[564] = PmAx*pOut[456] + pAm1[456]*rPmQx;
   pOut[565] = PmAz*pOut[455] + pAm1[455]*rPmQz;
   pOut[566] = PmAx*pOut[457] + pAm1[457]*rPmQx;
   pOut[567] = PmAz*pOut[456] + pAm1[456]*rPmQz;
   pOut[568] = PmAy*pOut[457] + pAm1[457]*rPmQy;
   pOut[569] = PmAy*pOut[460] + i2z*(pAm1[364]*riz + pOut[364]) + pAm1[460]*rPmQy;
   pOut[570] = PmAz*pOut[462] + i2z*(pAm1[364]*riz + pOut[364]) + pAm1[462]*rPmQz;
   pOut[571] = PmAx*pOut[458] + i2z*(pAm1[365]*riz + pOut[365]) + pAm1[458]*rPmQx;
   pOut[572] = PmAx*pOut[459] + i2z*(pAm1[366]*riz + pOut[366]) + pAm1[459]*rPmQx;
   pOut[573] = PmAz*pOut[463] + i2z*(pAm1[365]*riz + pOut[365]) + pAm1[463]*rPmQz;
   pOut[574] = PmAy*pOut[461] + i2z*(pAm1[366]*riz + pOut[366]) + pAm1[461]*rPmQy;
   pOut[575] = PmAy*pOut[464] + 2*i2z*(pAm1[367]*riz + pOut[367]) + pAm1[464]*rPmQy;
   pOut[576] = PmAx*pOut[466] + 2*i2z*(pAm1[368]*riz + pOut[368]) + pAm1[466]*rPmQx;
   pOut[577] = PmAz*pOut[465] + 2*i2z*(pAm1[369]*riz + pOut[369]) + pAm1[465]*rPmQz;
   pOut[578] = PmAx*pOut[468] + 2*i2z*(pAm1[370]*riz + pOut[370]) + pAm1[468]*rPmQx;
   pOut[579] = PmAz*pOut[467] + 2*i2z*(pAm1[371]*riz + pOut[371]) + pAm1[467]*rPmQz;
   pOut[580] = PmAy*pOut[469] + 2*i2z*(pAm1[372]*riz + pOut[372]) + pAm1[469]*rPmQy;
   pOut[581] = PmAy*pOut[472] + 3*i2z*(pAm1[373]*riz + pOut[373]) + pAm1[472]*rPmQy;
   pOut[582] = PmAz*pOut[474] + 3*i2z*(pAm1[374]*riz + pOut[374]) + pAm1[474]*rPmQz;
   pOut[583] = PmAx*pOut[470] + 3*i2z*(pAm1[375]*riz + pOut[375]) + pAm1[470]*rPmQx;
   pOut[584] = PmAx*pOut[471] + 3*i2z*(pAm1[376]*riz + pOut[376]) + pAm1[471]*rPmQx;
   pOut[585] = PmAz*pOut[475] + 3*i2z*(pAm1[377]*riz + pOut[377]) + pAm1[475]*rPmQz;
   pOut[586] = PmAy*pOut[473] + 3*i2z*(pAm1[378]*riz + pOut[378]) + pAm1[473]*rPmQy;
   pOut[587] = PmAy*pOut[476] + 4*i2z*(pAm1[379]*riz + pOut[379]) + pAm1[476]*rPmQy;
   pOut[588] = PmAx*pOut[478] + 4*i2z*(pAm1[380]*riz + pOut[380]) + pAm1[478]*rPmQx;
   pOut[589] = PmAz*pOut[477] + 4*i2z*(pAm1[381]*riz + pOut[381]) + pAm1[477]*rPmQz;
   pOut[590] = PmAx*pOut[480] + 4*i2z*(pAm1[382]*riz + pOut[382]) + pAm1[480]*rPmQx;
   pOut[591] = PmAz*pOut[479] + 4*i2z*(pAm1[383]*riz + pOut[383]) + pAm1[479]*rPmQz;
   pOut[592] = PmAy*pOut[481] + 4*i2z*(pAm1[384]*riz + pOut[384]) + pAm1[481]*rPmQy;
   pOut[593] = PmAy*pOut[484] + 5*i2z*(pAm1[385]*riz + pOut[385]) + pAm1[484]*rPmQy;
   pOut[594] = PmAz*pOut[486] + 5*i2z*(pAm1[386]*riz + pOut[386]) + pAm1[486]*rPmQz;
   pOut[595] = PmAx*pOut[482] + 5*i2z*(pAm1[387]*riz + pOut[387]) + pAm1[482]*rPmQx;
   pOut[596] = PmAx*pOut[483] + 5*i2z*(pAm1[388]*riz + pOut[388]) + pAm1[483]*rPmQx;
   pOut[597] = PmAz*pOut[487] + 5*i2z*(pAm1[389]*riz + pOut[389]) + pAm1[487]*rPmQz;
   pOut[598] = PmAy*pOut[485] + 5*i2z*(pAm1[390]*riz + pOut[390]) + pAm1[485]*rPmQy;
   pOut[599] = PmAy*pOut[488] + 6*i2z*(pAm1[391]*riz + pOut[391]) + pAm1[488]*rPmQy;
   pOut[600] = PmAz*pOut[489] + 6*i2z*(pAm1[393]*riz + pOut[393]) + pAm1[489]*rPmQz;
   pOut[601] = PmAz*pOut[491] + 6*i2z*(pAm1[395]*riz + pOut[395]) + pAm1[491]*rPmQz;
   pOut[602] = PmAy*pOut[459] + pAm1[459]*rPmQy;
   pOut[603] = PmAz*pOut[458] + pAm1[458]*rPmQz;
   pOut[604] = PmAz*pOut[460] + pAm1[460]*rPmQz;
   pOut[605] = PmAy*pOut[465] + pAm1[465]*rPmQy;
   pOut[606] = PmAx*pOut[467] + pAm1[467]*rPmQx;
   pOut[607] = PmAz*pOut[464] + pAm1[464]*rPmQz;
   pOut[608] = PmAx*pOut[469] + pAm1[469]*rPmQx;
   pOut[609] = PmAz*pOut[466] + pAm1[466]*rPmQz;
   pOut[610] = PmAy*pOut[468] + pAm1[468]*rPmQy;
   pOut[611] = PmAy*pOut[471] + pAm1[471]*rPmQy;
   pOut[612] = PmAx*pOut[473] + pAm1[473]*rPmQx;
   pOut[613] = PmAz*pOut[470] + pAm1[470]*rPmQz;
   pOut[614] = PmAx*pOut[475] + pAm1[475]*rPmQx;
   pOut[615] = PmAz*pOut[472] + pAm1[472]*rPmQz;
   pOut[616] = PmAy*pOut[474] + pAm1[474]*rPmQy;
   pOut[617] = PmAy*pOut[477] + pAm1[477]*rPmQy;
   pOut[618] = PmAx*pOut[479] + pAm1[479]*rPmQx;
   pOut[619] = PmAz*pOut[476] + pAm1[476]*rPmQz;
   pOut[620] = PmAx*pOut[481] + pAm1[481]*rPmQx;
   pOut[621] = PmAz*pOut[478] + pAm1[478]*rPmQz;
   pOut[622] = PmAy*pOut[480] + pAm1[480]*rPmQy;
   pOut[623] = PmAy*pOut[483] + pAm1[483]*rPmQy;
   pOut[624] = PmAx*pOut[485] + pAm1[485]*rPmQx;
   pOut[625] = PmAz*pOut[482] + pAm1[482]*rPmQz;
   pOut[626] = PmAx*pOut[487] + pAm1[487]*rPmQx;
   pOut[627] = PmAz*pOut[484] + pAm1[484]*rPmQz;
   pOut[628] = PmAy*pOut[486] + pAm1[486]*rPmQy;
   pOut[629] = PmAy*pOut[489] + pAm1[489]*rPmQy;
   pOut[630] = PmAx*pOut[491] + pAm1[491]*rPmQx;
   pOut[631] = PmAz*pOut[488] + pAm1[488]*rPmQz;
   pOut[632] = PmAx*pOut[493] + pAm1[493]*rPmQx;
   pOut[633] = PmAz*pOut[490] + pAm1[490]*rPmQz;
   pOut[634] = PmAy*pOut[492] + pAm1[492]*rPmQy;
   pOut[635] = PmAy*pOut[499] + i2z*(pAm1[374]*riz + pOut[374]) + pAm1[499]*rPmQy;
   pOut[636] = PmAx*pOut[497] + i2z*(pAm1[377]*riz + pOut[377]) + pAm1[497]*rPmQx;
   pOut[637] = PmAx*pOut[498] + i2z*(pAm1[378]*riz + pOut[378]) + pAm1[498]*rPmQx;
   pOut[638] = PmAz*pOut[503] + i2z*(pAm1[379]*riz + pOut[379]) + pAm1[503]*rPmQz;
   pOut[639] = PmAz*pOut[505] + i2z*(pAm1[380]*riz + pOut[380]) + pAm1[505]*rPmQz;
   pOut[640] = PmAy*pOut[504] + i2z*(pAm1[381]*riz + pOut[381]) + pAm1[504]*rPmQy;
   pOut[641] = PmAy*pOut[506] + i2z*(pAm1[382]*riz + pOut[382]) + pAm1[506]*rPmQy;
   pOut[642] = PmAx*pOut[507] + i2z*(pAm1[383]*riz + pOut[383]) + pAm1[507]*rPmQx;
   pOut[643] = PmAx*pOut[508] + i2z*(pAm1[384]*riz + pOut[384]) + pAm1[508]*rPmQx;
   pOut[644] = PmAz*pOut[513] + i2z*(pAm1[385]*riz + pOut[385]) + pAm1[513]*rPmQz;
   pOut[645] = PmAy*pOut[511] + i2z*(pAm1[386]*riz + pOut[386]) + pAm1[511]*rPmQy;
   pOut[646] = PmAz*pOut[514] + i2z*(pAm1[387]*riz + pOut[387]) + pAm1[514]*rPmQz;
   pOut[647] = PmAy*pOut[512] + i2z*(pAm1[388]*riz + pOut[388]) + pAm1[512]*rPmQy;
   pOut[648] = PmAx*pOut[509] + i2z*(pAm1[389]*riz + pOut[389]) + pAm1[509]*rPmQx;
   pOut[649] = PmAx*pOut[510] + i2z*(pAm1[390]*riz + pOut[390]) + pAm1[510]*rPmQx;
   pOut[650] = PmAz*pOut[515] + i2z*(pAm1[391]*riz + pOut[391]) + pAm1[515]*rPmQz;
   pOut[651] = PmAz*pOut[517] + i2z*(pAm1[392]*riz + pOut[392]) + pAm1[517]*rPmQz;
   pOut[652] = PmAy*pOut[516] + i2z*(pAm1[393]*riz + pOut[393]) + pAm1[516]*rPmQy;
   pOut[653] = PmAy*pOut[518] + i2z*(pAm1[394]*riz + pOut[394]) + pAm1[518]*rPmQy;
   pOut[654] = PmAx*pOut[519] + i2z*(pAm1[395]*riz + pOut[395]) + pAm1[519]*rPmQx;
   pOut[655] = PmAx*pOut[520] + i2z*(pAm1[396]*riz + pOut[396]) + pAm1[520]*rPmQx;
   pOut[656] = PmAz*pOut[523] + i2z*(pAm1[397]*riz + pOut[397]) + pAm1[523]*rPmQz;
   pOut[657] = PmAy*pOut[522] + i2z*(pAm1[398]*riz + pOut[398]) + pAm1[522]*rPmQy;
   pOut[658] = PmAx*pOut[521] + i2z*(pAm1[399]*riz + pOut[399]) + pAm1[521]*rPmQx;
   pOut[659] = PmAy*pOut[528] + 2*i2z*(pAm1[409]*riz + pOut[409]) + pAm1[528]*rPmQy;
   pOut[660] = PmAz*pOut[527] + 2*i2z*(pAm1[411]*riz + pOut[411]) + pAm1[527]*rPmQz;
   pOut[661] = PmAz*pOut[529] + 2*i2z*(pAm1[413]*riz + pOut[413]) + pAm1[529]*rPmQz;
   pOut[662] = PmAy*pOut[534] + 2*i2z*(pAm1[415]*riz + pOut[415]) + pAm1[534]*rPmQy;
   pOut[663] = PmAx*pOut[536] + 2*i2z*(pAm1[416]*riz + pOut[416]) + pAm1[536]*rPmQx;
   pOut[664] = PmAz*pOut[533] + 2*i2z*(pAm1[417]*riz + pOut[417]) + pAm1[533]*rPmQz;
   pOut[665] = PmAx*pOut[538] + 2*i2z*(pAm1[418]*riz + pOut[418]) + pAm1[538]*rPmQx;
   pOut[666] = PmAz*pOut[535] + 2*i2z*(pAm1[419]*riz + pOut[419]) + pAm1[535]*rPmQz;
   pOut[667] = PmAy*pOut[537] + 2*i2z*(pAm1[420]*riz + pOut[420]) + pAm1[537]*rPmQy;
   pOut[668] = PmAy*pOut[540] + 2*i2z*(pAm1[421]*riz + pOut[421]) + pAm1[540]*rPmQy;
   pOut[669] = PmAx*pOut[542] + 2*i2z*(pAm1[422]*riz + pOut[422]) + pAm1[542]*rPmQx;
   pOut[670] = PmAz*pOut[539] + 2*i2z*(pAm1[423]*riz + pOut[423]) + pAm1[539]*rPmQz;
   pOut[671] = PmAx*pOut[544] + 2*i2z*(pAm1[424]*riz + pOut[424]) + pAm1[544]*rPmQx;
   pOut[672] = PmAz*pOut[541] + 2*i2z*(pAm1[425]*riz + pOut[425]) + pAm1[541]*rPmQz;
   pOut[673] = PmAy*pOut[543] + 2*i2z*(pAm1[426]*riz + pOut[426]) + pAm1[543]*rPmQy;
   pOut[674] = PmAy*pOut[550] + 3*i2z*(pAm1[437]*riz + pOut[437]) + pAm1[550]*rPmQy;
   pOut[675] = PmAx*pOut[548] + 3*i2z*(pAm1[440]*riz + pOut[440]) + pAm1[548]*rPmQx;
   pOut[676] = PmAx*pOut[549] + 3*i2z*(pAm1[441]*riz + pOut[441]) + pAm1[549]*rPmQx;
   pOut[677] = PmAz*pOut[554] + 3*i2z*(pAm1[442]*riz + pOut[442]) + pAm1[554]*rPmQz;
   pOut[678] = PmAy*pOut[555] + 3*i2z*(pAm1[443]*riz + pOut[443]) + pAm1[555]*rPmQy;
   pOut[679] = PmAx*pOut[556] + 3*i2z*(pAm1[444]*riz + pOut[444]) + pAm1[556]*rPmQx;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 14) return;
   // If you arrive here we ran out of angular momenta. Regenerate with larger Lab.
   assert(0);
}

// Calculate (a0|00)^0 from [0]^m, for a = 0..lab, m = 0..lab (both inclusive; see notes for (a0|00)^m with m!=0).
// Inputs:
// - pIm: input scalar integrals I[m] = (00|00)^m for m = 0..lab (inclusive!)
// - PmA is (P-A) (make as (B-A) * (beta/zeta) for numerical stability! see doi.org/10.1021/ct200239p appendix C)
// - PmQ is (P-Q), InvZeta is 1/zeta = 1/(alpha+beta). See PCCP 6 5119 (2004) eq. 11.
// Output:
// - pOut: a single [CartX(lab)]-array containing the (a0|00)^m integrals at the target m
// Notes:
// - This routine can also make 3-center integrals (a0|0)^m from [0]^m = (00|0)^m
//   (use with delta=0, and therefore Q=C and Scd=1)
// - To make a single set of (a0|00)^n with non-zero 'n' with this routine, provide
//   (00|00)^m base integrals with m=n...(n+lab) instead of m=0..lab, ...by simply
//   offsetting the input 'pIm' pointer by 'n' before calling;
//   (e.g., in the 3c base case, one can make (a0|0)^m with m=lc by offsetting pIm by lc)
// - The effect of this routine can also be achieved by stacking OsrrA_StepM calls. This one
//   here makes less intermediates, tough, and is easier to use if only a single target m is needed.
//   If (a0|00)^m with multiple m=0..lcd are needed, the easiest way to achieve it is to first call.
//   this routine to assemble (a0|00)^m, and with m=lcd, then lcd times OsrrA_StepM to make the m=(lcd-1),...,0 sets.
void OsrrA(double *IR_RP pOut, double const *IR_RP pIm, unsigned lab, double PmAx, double PmAy, double PmAz, double PmQx, double PmQy, double PmQz, double rho, double InvZeta)
{
   // m + lab == 0
   pOut[0] = pIm[0];
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 0) return;

   // m + lab == 1
   double riz = -rho*InvZeta;
   double rPmQx = riz*PmQx, rPmQy = riz*PmQy, rPmQz = riz*PmQz;
   pOut[1] = PmAx*pIm[0] + pIm[1]*rPmQx;
   pOut[2] = PmAy*pIm[0] + pIm[1]*rPmQy;
   pOut[3] = PmAz*pIm[0] + pIm[1]*rPmQz;
   // 1.00 steps per component, 0.00 kb stack
   if (lab == 1) return;

   // m + lab == 2
   double i2z = .5*InvZeta;
   double r_100_1 = PmAx*pIm[1] + pIm[2]*rPmQx;
   double r_010_1 = PmAy*pIm[1] + pIm[2]*rPmQy;
   double r_001_1 = PmAz*pIm[1] + pIm[2]*rPmQz;
   pOut[4] = PmAx*pOut[1] + i2z*(pIm[0] + pIm[1]*riz) + rPmQx*r_100_1;
   pOut[5] = PmAy*pOut[2] + i2z*(pIm[0] + pIm[1]*riz) + rPmQy*r_010_1;
   pOut[6] = PmAz*pOut[3] + i2z*(pIm[0] + pIm[1]*riz) + rPmQz*r_001_1;
   pOut[7] = PmAy*pOut[1] + rPmQy*r_100_1;
   pOut[8] = PmAz*pOut[1] + rPmQz*r_100_1;
   pOut[9] = PmAz*pOut[2] + rPmQz*r_010_1;
   // 1.50 steps per component, 0.02 kb stack
   if (lab == 2) return;

   // m + lab == 3
   double r_100_2 = PmAx*pIm[2] + pIm[3]*rPmQx;
   double r_010_2 = PmAy*pIm[2] + pIm[3]*rPmQy;
   double r_001_2 = PmAz*pIm[2] + pIm[3]*rPmQz;
   double r_200_1 = PmAx*r_100_1 + i2z*(pIm[1] + pIm[2]*riz) + rPmQx*r_100_2;
   double r_020_1 = PmAy*r_010_1 + i2z*(pIm[1] + pIm[2]*riz) + rPmQy*r_010_2;
   double r_002_1 = PmAz*r_001_1 + i2z*(pIm[1] + pIm[2]*riz) + rPmQz*r_001_2;
   double r_110_1 = PmAy*r_100_1 + rPmQy*r_100_2;
   pOut[10] = PmAx*pOut[4] + 2*i2z*(pOut[1] + r_100_1*riz) + rPmQx*r_200_1;
   pOut[11] = PmAy*pOut[5] + 2*i2z*(pOut[2] + r_010_1*riz) + rPmQy*r_020_1;
   pOut[12] = PmAz*pOut[6] + 2*i2z*(pOut[3] + r_001_1*riz) + rPmQz*r_002_1;
   pOut[13] = PmAx*pOut[5] + rPmQx*r_020_1;
   pOut[14] = PmAx*pOut[6] + rPmQx*r_002_1;
   pOut[15] = PmAy*pOut[4] + rPmQy*r_200_1;
   pOut[16] = PmAy*pOut[6] + rPmQy*r_002_1;
   pOut[17] = PmAz*pOut[4] + rPmQz*r_200_1;
   pOut[18] = PmAz*pOut[5] + rPmQz*r_020_1;
   pOut[19] = PmAz*pOut[7] + rPmQz*r_110_1;
   // 1.70 steps per component, 0.05 kb stack
   if (lab == 3) return;

   // m + lab == 4
   double r_100_3 = PmAx*pIm[3] + pIm[4]*rPmQx;
   double r_010_3 = PmAy*pIm[3] + pIm[4]*rPmQy;
   double r_001_3 = PmAz*pIm[3] + pIm[4]*rPmQz;
   double r_200_2 = PmAx*r_100_2 + i2z*(pIm[2] + pIm[3]*riz) + rPmQx*r_100_3;
   double r_020_2 = PmAy*r_010_2 + i2z*(pIm[2] + pIm[3]*riz) + rPmQy*r_010_3;
   double r_002_2 = PmAz*r_001_2 + i2z*(pIm[2] + pIm[3]*riz) + rPmQz*r_001_3;
   double r_300_1 = PmAx*r_200_1 + 2*i2z*(r_100_1 + r_100_2*riz) + rPmQx*r_200_2;
   double r_030_1 = PmAy*r_020_1 + 2*i2z*(r_010_1 + r_010_2*riz) + rPmQy*r_020_2;
   double r_003_1 = PmAz*r_002_1 + 2*i2z*(r_001_1 + r_001_2*riz) + rPmQz*r_002_2;
   double r_120_1 = PmAx*r_020_1 + rPmQx*r_020_2;
   double r_102_1 = PmAx*r_002_1 + rPmQx*r_002_2;
   double r_210_1 = PmAy*r_200_1 + rPmQy*r_200_2;
   double r_012_1 = PmAy*r_002_1 + rPmQy*r_002_2;
   pOut[20] = PmAx*pOut[10] + 3*i2z*(pOut[4] + r_200_1*riz) + rPmQx*r_300_1;
   pOut[21] = PmAy*pOut[11] + 3*i2z*(pOut[5] + r_020_1*riz) + rPmQy*r_030_1;
   pOut[22] = PmAz*pOut[12] + 3*i2z*(pOut[6] + r_002_1*riz) + rPmQz*r_003_1;
   pOut[23] = PmAy*pOut[10] + rPmQy*r_300_1;
   pOut[24] = PmAx*pOut[11] + rPmQx*r_030_1;
   pOut[25] = PmAz*pOut[10] + rPmQz*r_300_1;
   pOut[26] = PmAx*pOut[12] + rPmQx*r_003_1;
   pOut[27] = PmAz*pOut[11] + rPmQz*r_030_1;
   pOut[28] = PmAy*pOut[12] + rPmQy*r_003_1;
   pOut[29] = PmAx*pOut[13] + i2z*(pOut[5] + r_020_1*riz) + rPmQx*r_120_1;
   pOut[30] = PmAx*pOut[14] + i2z*(pOut[6] + r_002_1*riz) + rPmQx*r_102_1;
   pOut[31] = PmAy*pOut[16] + i2z*(pOut[6] + r_002_1*riz) + rPmQy*r_012_1;
   pOut[32] = PmAy*pOut[14] + rPmQy*r_102_1;
   pOut[33] = PmAz*pOut[13] + rPmQz*r_120_1;
   pOut[34] = PmAz*pOut[15] + rPmQz*r_210_1;
   // 1.87 steps per component, 0.10 kb stack
   if (lab == 4) return;

   // m + lab == 5
   double r_100_4 = PmAx*pIm[4] + pIm[5]*rPmQx;
   double r_010_4 = PmAy*pIm[4] + pIm[5]*rPmQy;
   double r_001_4 = PmAz*pIm[4] + pIm[5]*rPmQz;
   double r_200_3 = PmAx*r_100_3 + i2z*(pIm[3] + pIm[4]*riz) + rPmQx*r_100_4;
   double r_020_3 = PmAy*r_010_3 + i2z*(pIm[3] + pIm[4]*riz) + rPmQy*r_010_4;
   double r_002_3 = PmAz*r_001_3 + i2z*(pIm[3] + pIm[4]*riz) + rPmQz*r_001_4;
   double r_300_2 = PmAx*r_200_2 + 2*i2z*(r_100_2 + r_100_3*riz) + rPmQx*r_200_3;
   double r_030_2 = PmAy*r_020_2 + 2*i2z*(r_010_2 + r_010_3*riz) + rPmQy*r_020_3;
   double r_003_2 = PmAz*r_002_2 + 2*i2z*(r_001_2 + r_001_3*riz) + rPmQz*r_002_3;
   double r_120_2 = PmAx*r_020_2 + rPmQx*r_020_3;
   double r_102_2 = PmAx*r_002_2 + rPmQx*r_002_3;
   double r_012_2 = PmAy*r_002_2 + rPmQy*r_002_3;
   double r_400_1 = PmAx*r_300_1 + 3*i2z*(r_200_1 + r_200_2*riz) + rPmQx*r_300_2;
   double r_040_1 = PmAy*r_030_1 + 3*i2z*(r_020_1 + r_020_2*riz) + rPmQy*r_030_2;
   double r_004_1 = PmAz*r_003_1 + 3*i2z*(r_002_1 + r_002_2*riz) + rPmQz*r_003_2;
   double r_310_1 = PmAy*r_300_1 + rPmQy*r_300_2;
   double r_130_1 = PmAx*r_030_1 + rPmQx*r_030_2;
   double r_301_1 = PmAz*r_300_1 + rPmQz*r_300_2;
   double r_103_1 = PmAx*r_003_1 + rPmQx*r_003_2;
   double r_031_1 = PmAz*r_030_1 + rPmQz*r_030_2;
   double r_013_1 = PmAy*r_003_1 + rPmQy*r_003_2;
   double r_220_1 = PmAx*r_120_1 + i2z*(r_020_1 + r_020_2*riz) + rPmQx*r_120_2;
   double r_202_1 = PmAx*r_102_1 + i2z*(r_002_1 + r_002_2*riz) + rPmQx*r_102_2;
   double r_022_1 = PmAy*r_012_1 + i2z*(r_002_1 + r_002_2*riz) + rPmQy*r_012_2;
   pOut[35] = PmAx*pOut[20] + 4*i2z*(pOut[10] + r_300_1*riz) + rPmQx*r_400_1;
   pOut[36] = PmAy*pOut[21] + 4*i2z*(pOut[11] + r_030_1*riz) + rPmQy*r_040_1;
   pOut[37] = PmAz*pOut[22] + 4*i2z*(pOut[12] + r_003_1*riz) + rPmQz*r_004_1;
   pOut[38] = PmAx*pOut[21] + rPmQx*r_040_1;
   pOut[39] = PmAx*pOut[22] + rPmQx*r_004_1;
   pOut[40] = PmAy*pOut[20] + rPmQy*r_400_1;
   pOut[41] = PmAy*pOut[22] + rPmQy*r_004_1;
   pOut[42] = PmAz*pOut[20] + rPmQz*r_400_1;
   pOut[43] = PmAz*pOut[21] + rPmQz*r_040_1;
   pOut[44] = PmAy*pOut[23] + i2z*(pOut[10] + r_300_1*riz) + rPmQy*r_310_1;
   pOut[45] = PmAz*pOut[25] + i2z*(pOut[10] + r_300_1*riz) + rPmQz*r_301_1;
   pOut[46] = PmAx*pOut[24] + i2z*(pOut[11] + r_030_1*riz) + rPmQx*r_130_1;
   pOut[47] = PmAz*pOut[27] + i2z*(pOut[11] + r_030_1*riz) + rPmQz*r_031_1;
   pOut[48] = PmAx*pOut[26] + i2z*(pOut[12] + r_003_1*riz) + rPmQx*r_103_1;
   pOut[49] = PmAy*pOut[28] + i2z*(pOut[12] + r_003_1*riz) + rPmQy*r_013_1;
   pOut[50] = PmAz*pOut[23] + rPmQz*r_310_1;
   pOut[51] = PmAz*pOut[24] + rPmQz*r_130_1;
   pOut[52] = PmAy*pOut[26] + rPmQy*r_103_1;
   pOut[53] = PmAx*pOut[31] + rPmQx*r_022_1;
   pOut[54] = PmAy*pOut[30] + rPmQy*r_202_1;
   pOut[55] = PmAz*pOut[29] + rPmQz*r_220_1;
   // 2.14 steps per component, 0.19 kb stack
   if (lab == 5) return;

   // m + lab == 6
   double r_100_5 = PmAx*pIm[5] + pIm[6]*rPmQx;
   double r_010_5 = PmAy*pIm[5] + pIm[6]*rPmQy;
   double r_001_5 = PmAz*pIm[5] + pIm[6]*rPmQz;
   double r_200_4 = PmAx*r_100_4 + i2z*(pIm[4] + pIm[5]*riz) + rPmQx*r_100_5;
   double r_020_4 = PmAy*r_010_4 + i2z*(pIm[4] + pIm[5]*riz) + rPmQy*r_010_5;
   double r_002_4 = PmAz*r_001_4 + i2z*(pIm[4] + pIm[5]*riz) + rPmQz*r_001_5;
   double r_300_3 = PmAx*r_200_3 + 2*i2z*(r_100_3 + r_100_4*riz) + rPmQx*r_200_4;
   double r_030_3 = PmAy*r_020_3 + 2*i2z*(r_010_3 + r_010_4*riz) + rPmQy*r_020_4;
   double r_003_3 = PmAz*r_002_3 + 2*i2z*(r_001_3 + r_001_4*riz) + rPmQz*r_002_4;
   double r_012_3 = PmAy*r_002_3 + rPmQy*r_002_4;
   double r_400_2 = PmAx*r_300_2 + 3*i2z*(r_200_2 + r_200_3*riz) + rPmQx*r_300_3;
   double r_040_2 = PmAy*r_030_2 + 3*i2z*(r_020_2 + r_020_3*riz) + rPmQy*r_030_3;
   double r_004_2 = PmAz*r_003_2 + 3*i2z*(r_002_2 + r_002_3*riz) + rPmQz*r_003_3;
   double r_310_2 = PmAy*r_300_2 + rPmQy*r_300_3;
   double r_130_2 = PmAx*r_030_2 + rPmQx*r_030_3;
   double r_301_2 = PmAz*r_300_2 + rPmQz*r_300_3;
   double r_103_2 = PmAx*r_003_2 + rPmQx*r_003_3;
   double r_031_2 = PmAz*r_030_2 + rPmQz*r_030_3;
   double r_013_2 = PmAy*r_003_2 + rPmQy*r_003_3;
   double r_022_2 = PmAy*r_012_2 + i2z*(r_002_2 + r_002_3*riz) + rPmQy*r_012_3;
   double r_500_1 = PmAx*r_400_1 + 4*i2z*(r_300_1 + r_300_2*riz) + rPmQx*r_400_2;
   double r_050_1 = PmAy*r_040_1 + 4*i2z*(r_030_1 + r_030_2*riz) + rPmQy*r_040_2;
   double r_005_1 = PmAz*r_004_1 + 4*i2z*(r_003_1 + r_003_2*riz) + rPmQz*r_004_2;
   double r_140_1 = PmAx*r_040_1 + rPmQx*r_040_2;
   double r_104_1 = PmAx*r_004_1 + rPmQx*r_004_2;
   double r_410_1 = PmAy*r_400_1 + rPmQy*r_400_2;
   double r_014_1 = PmAy*r_004_1 + rPmQy*r_004_2;
   double r_401_1 = PmAz*r_400_1 + rPmQz*r_400_2;
   double r_041_1 = PmAz*r_040_1 + rPmQz*r_040_2;
   double r_320_1 = PmAy*r_310_1 + i2z*(r_300_1 + r_300_2*riz) + rPmQy*r_310_2;
   double r_302_1 = PmAz*r_301_1 + i2z*(r_300_1 + r_300_2*riz) + rPmQz*r_301_2;
   double r_230_1 = PmAx*r_130_1 + i2z*(r_030_1 + r_030_2*riz) + rPmQx*r_130_2;
   double r_032_1 = PmAz*r_031_1 + i2z*(r_030_1 + r_030_2*riz) + rPmQz*r_031_2;
   double r_203_1 = PmAx*r_103_1 + i2z*(r_003_1 + r_003_2*riz) + rPmQx*r_103_2;
   double r_023_1 = PmAy*r_013_1 + i2z*(r_003_1 + r_003_2*riz) + rPmQy*r_013_2;
   double r_122_1 = PmAx*r_022_1 + rPmQx*r_022_2;
   pOut[56] = PmAx*pOut[35] + 5*i2z*(pOut[20] + r_400_1*riz) + rPmQx*r_500_1;
   pOut[57] = PmAy*pOut[36] + 5*i2z*(pOut[21] + r_040_1*riz) + rPmQy*r_050_1;
   pOut[58] = PmAz*pOut[37] + 5*i2z*(pOut[22] + r_004_1*riz) + rPmQz*r_005_1;
   pOut[59] = PmAy*pOut[35] + rPmQy*r_500_1;
   pOut[60] = PmAx*pOut[36] + rPmQx*r_050_1;
   pOut[61] = PmAz*pOut[35] + rPmQz*r_500_1;
   pOut[62] = PmAx*pOut[37] + rPmQx*r_005_1;
   pOut[63] = PmAz*pOut[36] + rPmQz*r_050_1;
   pOut[64] = PmAy*pOut[37] + rPmQy*r_005_1;
   pOut[65] = PmAy*pOut[40] + i2z*(pOut[20] + r_400_1*riz) + rPmQy*r_410_1;
   pOut[66] = PmAz*pOut[42] + i2z*(pOut[20] + r_400_1*riz) + rPmQz*r_401_1;
   pOut[67] = PmAx*pOut[38] + i2z*(pOut[21] + r_040_1*riz) + rPmQx*r_140_1;
   pOut[68] = PmAx*pOut[39] + i2z*(pOut[22] + r_004_1*riz) + rPmQx*r_104_1;
   pOut[69] = PmAz*pOut[43] + i2z*(pOut[21] + r_040_1*riz) + rPmQz*r_041_1;
   pOut[70] = PmAy*pOut[41] + i2z*(pOut[22] + r_004_1*riz) + rPmQy*r_014_1;
   pOut[71] = PmAy*pOut[44] + 2*i2z*(pOut[23] + r_310_1*riz) + rPmQy*r_320_1;
   pOut[72] = PmAz*pOut[45] + 2*i2z*(pOut[25] + r_301_1*riz) + rPmQz*r_302_1;
   pOut[73] = PmAz*pOut[47] + 2*i2z*(pOut[27] + r_031_1*riz) + rPmQz*r_032_1;
   pOut[74] = PmAy*pOut[39] + rPmQy*r_104_1;
   pOut[75] = PmAz*pOut[38] + rPmQz*r_140_1;
   pOut[76] = PmAz*pOut[40] + rPmQz*r_410_1;
   pOut[77] = PmAy*pOut[45] + rPmQy*r_302_1;
   pOut[78] = PmAx*pOut[47] + rPmQx*r_032_1;
   pOut[79] = PmAz*pOut[44] + rPmQz*r_320_1;
   pOut[80] = PmAx*pOut[49] + rPmQx*r_023_1;
   pOut[81] = PmAz*pOut[46] + rPmQz*r_230_1;
   pOut[82] = PmAy*pOut[48] + rPmQy*r_203_1;
   pOut[83] = PmAx*pOut[53] + i2z*(pOut[31] + r_022_1*riz) + rPmQx*r_122_1;
   // 2.29 steps per component, 0.28 kb stack
   if (lab == 6) return;

   // m + lab == 7
   double r_100_6 = PmAx*pIm[6] + pIm[7]*rPmQx;
   double r_010_6 = PmAy*pIm[6] + pIm[7]*rPmQy;
   double r_001_6 = PmAz*pIm[6] + pIm[7]*rPmQz;
   double r_200_5 = PmAx*r_100_5 + i2z*(pIm[5] + pIm[6]*riz) + rPmQx*r_100_6;
   double r_020_5 = PmAy*r_010_5 + i2z*(pIm[5] + pIm[6]*riz) + rPmQy*r_010_6;
   double r_002_5 = PmAz*r_001_5 + i2z*(pIm[5] + pIm[6]*riz) + rPmQz*r_001_6;
   double r_300_4 = PmAx*r_200_4 + 2*i2z*(r_100_4 + r_100_5*riz) + rPmQx*r_200_5;
   double r_030_4 = PmAy*r_020_4 + 2*i2z*(r_010_4 + r_010_5*riz) + rPmQy*r_020_5;
   double r_003_4 = PmAz*r_002_4 + 2*i2z*(r_001_4 + r_001_5*riz) + rPmQz*r_002_5;
   double r_400_3 = PmAx*r_300_3 + 3*i2z*(r_200_3 + r_200_4*riz) + rPmQx*r_300_4;
   double r_040_3 = PmAy*r_030_3 + 3*i2z*(r_020_3 + r_020_4*riz) + rPmQy*r_030_4;
   double r_004_3 = PmAz*r_003_3 + 3*i2z*(r_002_3 + r_002_4*riz) + rPmQz*r_003_4;
   double r_310_3 = PmAy*r_300_3 + rPmQy*r_300_4;
   double r_301_3 = PmAz*r_300_3 + rPmQz*r_300_4;
   double r_031_3 = PmAz*r_030_3 + rPmQz*r_030_4;
   double r_013_3 = PmAy*r_003_3 + rPmQy*r_003_4;
   double r_500_2 = PmAx*r_400_2 + 4*i2z*(r_300_2 + r_300_3*riz) + rPmQx*r_400_3;
   double r_050_2 = PmAy*r_040_2 + 4*i2z*(r_030_2 + r_030_3*riz) + rPmQy*r_040_3;
   double r_005_2 = PmAz*r_004_2 + 4*i2z*(r_003_2 + r_003_3*riz) + rPmQz*r_004_3;
   double r_140_2 = PmAx*r_040_2 + rPmQx*r_040_3;
   double r_104_2 = PmAx*r_004_2 + rPmQx*r_004_3;
   double r_410_2 = PmAy*r_400_2 + rPmQy*r_400_3;
   double r_014_2 = PmAy*r_004_2 + rPmQy*r_004_3;
   double r_401_2 = PmAz*r_400_2 + rPmQz*r_400_3;
   double r_041_2 = PmAz*r_040_2 + rPmQz*r_040_3;
   double r_320_2 = PmAy*r_310_2 + i2z*(r_300_2 + r_300_3*riz) + rPmQy*r_310_3;
   double r_302_2 = PmAz*r_301_2 + i2z*(r_300_2 + r_300_3*riz) + rPmQz*r_301_3;
   double r_032_2 = PmAz*r_031_2 + i2z*(r_030_2 + r_030_3*riz) + rPmQz*r_031_3;
   double r_023_2 = PmAy*r_013_2 + i2z*(r_003_2 + r_003_3*riz) + rPmQy*r_013_3;
   double r_600_1 = PmAx*r_500_1 + 5*i2z*(r_400_1 + r_400_2*riz) + rPmQx*r_500_2;
   double r_060_1 = PmAy*r_050_1 + 5*i2z*(r_040_1 + r_040_2*riz) + rPmQy*r_050_2;
   double r_006_1 = PmAz*r_005_1 + 5*i2z*(r_004_1 + r_004_2*riz) + rPmQz*r_005_2;
   double r_510_1 = PmAy*r_500_1 + rPmQy*r_500_2;
   double r_150_1 = PmAx*r_050_1 + rPmQx*r_050_2;
   double r_501_1 = PmAz*r_500_1 + rPmQz*r_500_2;
   double r_105_1 = PmAx*r_005_1 + rPmQx*r_005_2;
   double r_051_1 = PmAz*r_050_1 + rPmQz*r_050_2;
   double r_015_1 = PmAy*r_005_1 + rPmQy*r_005_2;
   double r_420_1 = PmAy*r_410_1 + i2z*(r_400_1 + r_400_2*riz) + rPmQy*r_410_2;
   double r_402_1 = PmAz*r_401_1 + i2z*(r_400_1 + r_400_2*riz) + rPmQz*r_401_2;
   double r_240_1 = PmAx*r_140_1 + i2z*(r_040_1 + r_040_2*riz) + rPmQx*r_140_2;
   double r_204_1 = PmAx*r_104_1 + i2z*(r_004_1 + r_004_2*riz) + rPmQx*r_104_2;
   double r_042_1 = PmAz*r_041_1 + i2z*(r_040_1 + r_040_2*riz) + rPmQz*r_041_2;
   double r_024_1 = PmAy*r_014_1 + i2z*(r_004_1 + r_004_2*riz) + rPmQy*r_014_2;
   double r_330_1 = PmAy*r_320_1 + 2*i2z*(r_310_1 + r_310_2*riz) + rPmQy*r_320_2;
   double r_303_1 = PmAz*r_302_1 + 2*i2z*(r_301_1 + r_301_2*riz) + rPmQz*r_302_2;
   double r_033_1 = PmAz*r_032_1 + 2*i2z*(r_031_1 + r_031_2*riz) + rPmQz*r_032_2;
   double r_312_1 = PmAy*r_302_1 + rPmQy*r_302_2;
   double r_132_1 = PmAx*r_032_1 + rPmQx*r_032_2;
   double r_123_1 = PmAx*r_023_1 + rPmQx*r_023_2;
   pOut[84] = PmAx*pOut[56] + 6*i2z*(pOut[35] + r_500_1*riz) + rPmQx*r_600_1;
   pOut[85] = PmAy*pOut[57] + 6*i2z*(pOut[36] + r_050_1*riz) + rPmQy*r_060_1;
   pOut[86] = PmAz*pOut[58] + 6*i2z*(pOut[37] + r_005_1*riz) + rPmQz*r_006_1;
   pOut[87] = PmAx*pOut[57] + rPmQx*r_060_1;
   pOut[88] = PmAx*pOut[58] + rPmQx*r_006_1;
   pOut[89] = PmAy*pOut[56] + rPmQy*r_600_1;
   pOut[90] = PmAy*pOut[58] + rPmQy*r_006_1;
   pOut[91] = PmAz*pOut[56] + rPmQz*r_600_1;
   pOut[92] = PmAz*pOut[57] + rPmQz*r_060_1;
   pOut[93] = PmAy*pOut[59] + i2z*(pOut[35] + r_500_1*riz) + rPmQy*r_510_1;
   pOut[94] = PmAz*pOut[61] + i2z*(pOut[35] + r_500_1*riz) + rPmQz*r_501_1;
   pOut[95] = PmAx*pOut[60] + i2z*(pOut[36] + r_050_1*riz) + rPmQx*r_150_1;
   pOut[96] = PmAz*pOut[63] + i2z*(pOut[36] + r_050_1*riz) + rPmQz*r_051_1;
   pOut[97] = PmAx*pOut[62] + i2z*(pOut[37] + r_005_1*riz) + rPmQx*r_105_1;
   pOut[98] = PmAy*pOut[64] + i2z*(pOut[37] + r_005_1*riz) + rPmQy*r_015_1;
   pOut[99] = PmAx*pOut[67] + 2*i2z*(pOut[38] + r_140_1*riz) + rPmQx*r_240_1;
   pOut[100] = PmAx*pOut[68] + 2*i2z*(pOut[39] + r_104_1*riz) + rPmQx*r_204_1;
   pOut[101] = PmAy*pOut[65] + 2*i2z*(pOut[40] + r_410_1*riz) + rPmQy*r_420_1;
   pOut[102] = PmAy*pOut[70] + 2*i2z*(pOut[41] + r_014_1*riz) + rPmQy*r_024_1;
   pOut[103] = PmAz*pOut[66] + 2*i2z*(pOut[42] + r_401_1*riz) + rPmQz*r_402_1;
   pOut[104] = PmAz*pOut[69] + 2*i2z*(pOut[43] + r_041_1*riz) + rPmQz*r_042_1;
   pOut[105] = PmAz*pOut[59] + rPmQz*r_510_1;
   pOut[106] = PmAz*pOut[60] + rPmQz*r_150_1;
   pOut[107] = PmAy*pOut[62] + rPmQy*r_105_1;
   pOut[108] = PmAx*pOut[69] + rPmQx*r_042_1;
   pOut[109] = PmAx*pOut[70] + rPmQx*r_024_1;
   pOut[110] = PmAy*pOut[66] + rPmQy*r_402_1;
   pOut[111] = PmAy*pOut[68] + rPmQy*r_204_1;
   pOut[112] = PmAz*pOut[65] + rPmQz*r_420_1;
   pOut[113] = PmAz*pOut[67] + rPmQz*r_240_1;
   pOut[114] = PmAz*pOut[71] + rPmQz*r_330_1;
   pOut[115] = PmAy*pOut[72] + rPmQy*r_303_1;
   pOut[116] = PmAx*pOut[73] + rPmQx*r_033_1;
   pOut[117] = PmAy*pOut[77] + i2z*(pOut[45] + r_302_1*riz) + rPmQy*r_312_1;
   pOut[118] = PmAx*pOut[78] + i2z*(pOut[47] + r_032_1*riz) + rPmQx*r_132_1;
   pOut[119] = PmAx*pOut[80] + i2z*(pOut[49] + r_023_1*riz) + rPmQx*r_123_1;
   // 2.39 steps per component, 0.39 kb stack
   if (lab == 7) return;

   // m + lab == 8
   double r_100_7 = PmAx*pIm[7] + pIm[8]*rPmQx;
   double r_010_7 = PmAy*pIm[7] + pIm[8]*rPmQy;
   double r_001_7 = PmAz*pIm[7] + pIm[8]*rPmQz;
   double r_200_6 = PmAx*r_100_6 + i2z*(pIm[6] + pIm[7]*riz) + rPmQx*r_100_7;
   double r_020_6 = PmAy*r_010_6 + i2z*(pIm[6] + pIm[7]*riz) + rPmQy*r_010_7;
   double r_002_6 = PmAz*r_001_6 + i2z*(pIm[6] + pIm[7]*riz) + rPmQz*r_001_7;
   double r_300_5 = PmAx*r_200_5 + 2*i2z*(r_100_5 + r_100_6*riz) + rPmQx*r_200_6;
   double r_030_5 = PmAy*r_020_5 + 2*i2z*(r_010_5 + r_010_6*riz) + rPmQy*r_020_6;
   double r_003_5 = PmAz*r_002_5 + 2*i2z*(r_001_5 + r_001_6*riz) + rPmQz*r_002_6;
   double r_400_4 = PmAx*r_300_4 + 3*i2z*(r_200_4 + r_200_5*riz) + rPmQx*r_300_5;
   double r_040_4 = PmAy*r_030_4 + 3*i2z*(r_020_4 + r_020_5*riz) + rPmQy*r_030_5;
   double r_004_4 = PmAz*r_003_4 + 3*i2z*(r_002_4 + r_002_5*riz) + rPmQz*r_003_5;
   double r_310_4 = PmAy*r_300_4 + rPmQy*r_300_5;
   double r_301_4 = PmAz*r_300_4 + rPmQz*r_300_5;
   double r_031_4 = PmAz*r_030_4 + rPmQz*r_030_5;
   double r_500_3 = PmAx*r_400_3 + 4*i2z*(r_300_3 + r_300_4*riz) + rPmQx*r_400_4;
   double r_050_3 = PmAy*r_040_3 + 4*i2z*(r_030_3 + r_030_4*riz) + rPmQy*r_040_4;
   double r_005_3 = PmAz*r_004_3 + 4*i2z*(r_003_3 + r_003_4*riz) + rPmQz*r_004_4;
   double r_140_3 = PmAx*r_040_3 + rPmQx*r_040_4;
   double r_104_3 = PmAx*r_004_3 + rPmQx*r_004_4;
   double r_410_3 = PmAy*r_400_3 + rPmQy*r_400_4;
   double r_014_3 = PmAy*r_004_3 + rPmQy*r_004_4;
   double r_401_3 = PmAz*r_400_3 + rPmQz*r_400_4;
   double r_041_3 = PmAz*r_040_3 + rPmQz*r_040_4;
   double r_320_3 = PmAy*r_310_3 + i2z*(r_300_3 + r_300_4*riz) + rPmQy*r_310_4;
   double r_302_3 = PmAz*r_301_3 + i2z*(r_300_3 + r_300_4*riz) + rPmQz*r_301_4;
   double r_032_3 = PmAz*r_031_3 + i2z*(r_030_3 + r_030_4*riz) + rPmQz*r_031_4;
   double r_600_2 = PmAx*r_500_2 + 5*i2z*(r_400_2 + r_400_3*riz) + rPmQx*r_500_3;
   double r_060_2 = PmAy*r_050_2 + 5*i2z*(r_040_2 + r_040_3*riz) + rPmQy*r_050_3;
   double r_006_2 = PmAz*r_005_2 + 5*i2z*(r_004_2 + r_004_3*riz) + rPmQz*r_005_3;
   double r_510_2 = PmAy*r_500_2 + rPmQy*r_500_3;
   double r_150_2 = PmAx*r_050_2 + rPmQx*r_050_3;
   double r_501_2 = PmAz*r_500_2 + rPmQz*r_500_3;
   double r_105_2 = PmAx*r_005_2 + rPmQx*r_005_3;
   double r_051_2 = PmAz*r_050_2 + rPmQz*r_050_3;
   double r_015_2 = PmAy*r_005_2 + rPmQy*r_005_3;
   double r_420_2 = PmAy*r_410_2 + i2z*(r_400_2 + r_400_3*riz) + rPmQy*r_410_3;
   double r_402_2 = PmAz*r_401_2 + i2z*(r_400_2 + r_400_3*riz) + rPmQz*r_401_3;
   double r_240_2 = PmAx*r_140_2 + i2z*(r_040_2 + r_040_3*riz) + rPmQx*r_140_3;
   double r_204_2 = PmAx*r_104_2 + i2z*(r_004_2 + r_004_3*riz) + rPmQx*r_104_3;
   double r_042_2 = PmAz*r_041_2 + i2z*(r_040_2 + r_040_3*riz) + rPmQz*r_041_3;
   double r_024_2 = PmAy*r_014_2 + i2z*(r_004_2 + r_004_3*riz) + rPmQy*r_014_3;
   double r_330_2 = PmAy*r_320_2 + 2*i2z*(r_310_2 + r_310_3*riz) + rPmQy*r_320_3;
   double r_303_2 = PmAz*r_302_2 + 2*i2z*(r_301_2 + r_301_3*riz) + rPmQz*r_302_3;
   double r_033_2 = PmAz*r_032_2 + 2*i2z*(r_031_2 + r_031_3*riz) + rPmQz*r_032_3;
   double r_700_1 = PmAx*r_600_1 + 6*i2z*(r_500_1 + r_500_2*riz) + rPmQx*r_600_2;
   double r_070_1 = PmAy*r_060_1 + 6*i2z*(r_050_1 + r_050_2*riz) + rPmQy*r_060_2;
   double r_007_1 = PmAz*r_006_1 + 6*i2z*(r_005_1 + r_005_2*riz) + rPmQz*r_006_2;
   double r_160_1 = PmAx*r_060_1 + rPmQx*r_060_2;
   double r_106_1 = PmAx*r_006_1 + rPmQx*r_006_2;
   double r_610_1 = PmAy*r_600_1 + rPmQy*r_600_2;
   double r_016_1 = PmAy*r_006_1 + rPmQy*r_006_2;
   double r_601_1 = PmAz*r_600_1 + rPmQz*r_600_2;
   double r_061_1 = PmAz*r_060_1 + rPmQz*r_060_2;
   double r_520_1 = PmAy*r_510_1 + i2z*(r_500_1 + r_500_2*riz) + rPmQy*r_510_2;
   double r_502_1 = PmAz*r_501_1 + i2z*(r_500_1 + r_500_2*riz) + rPmQz*r_501_2;
   double r_250_1 = PmAx*r_150_1 + i2z*(r_050_1 + r_050_2*riz) + rPmQx*r_150_2;
   double r_052_1 = PmAz*r_051_1 + i2z*(r_050_1 + r_050_2*riz) + rPmQz*r_051_2;
   double r_205_1 = PmAx*r_105_1 + i2z*(r_005_1 + r_005_2*riz) + rPmQx*r_105_2;
   double r_025_1 = PmAy*r_015_1 + i2z*(r_005_1 + r_005_2*riz) + rPmQy*r_015_2;
   double r_340_1 = PmAx*r_240_1 + 2*i2z*(r_140_1 + r_140_2*riz) + rPmQx*r_240_2;
   double r_304_1 = PmAx*r_204_1 + 2*i2z*(r_104_1 + r_104_2*riz) + rPmQx*r_204_2;
   double r_430_1 = PmAy*r_420_1 + 2*i2z*(r_410_1 + r_410_2*riz) + rPmQy*r_420_2;
   double r_034_1 = PmAy*r_024_1 + 2*i2z*(r_014_1 + r_014_2*riz) + rPmQy*r_024_2;
   double r_403_1 = PmAz*r_402_1 + 2*i2z*(r_401_1 + r_401_2*riz) + rPmQz*r_402_2;
   double r_043_1 = PmAz*r_042_1 + 2*i2z*(r_041_1 + r_041_2*riz) + rPmQz*r_042_2;
   double r_142_1 = PmAx*r_042_1 + rPmQx*r_042_2;
   double r_124_1 = PmAx*r_024_1 + rPmQx*r_024_2;
   double r_412_1 = PmAy*r_402_1 + rPmQy*r_402_2;
   double r_331_1 = PmAz*r_330_1 + rPmQz*r_330_2;
   double r_313_1 = PmAy*r_303_1 + rPmQy*r_303_2;
   double r_133_1 = PmAx*r_033_1 + rPmQx*r_033_2;
   pOut[120] = PmAx*pOut[84] + 7*i2z*(pOut[56] + r_600_1*riz) + rPmQx*r_700_1;
   pOut[121] = PmAy*pOut[85] + 7*i2z*(pOut[57] + r_060_1*riz) + rPmQy*r_070_1;
   pOut[122] = PmAz*pOut[86] + 7*i2z*(pOut[58] + r_006_1*riz) + rPmQz*r_007_1;
   pOut[123] = PmAy*pOut[84] + rPmQy*r_700_1;
   pOut[124] = PmAx*pOut[85] + rPmQx*r_070_1;
   pOut[125] = PmAz*pOut[84] + rPmQz*r_700_1;
   pOut[126] = PmAx*pOut[86] + rPmQx*r_007_1;
   pOut[127] = PmAz*pOut[85] + rPmQz*r_070_1;
   pOut[128] = PmAy*pOut[86] + rPmQy*r_007_1;
   pOut[129] = PmAy*pOut[89] + i2z*(pOut[56] + r_600_1*riz) + rPmQy*r_610_1;
   pOut[130] = PmAz*pOut[91] + i2z*(pOut[56] + r_600_1*riz) + rPmQz*r_601_1;
   pOut[131] = PmAx*pOut[87] + i2z*(pOut[57] + r_060_1*riz) + rPmQx*r_160_1;
   pOut[132] = PmAx*pOut[88] + i2z*(pOut[58] + r_006_1*riz) + rPmQx*r_106_1;
   pOut[133] = PmAz*pOut[92] + i2z*(pOut[57] + r_060_1*riz) + rPmQz*r_061_1;
   pOut[134] = PmAy*pOut[90] + i2z*(pOut[58] + r_006_1*riz) + rPmQy*r_016_1;
   pOut[135] = PmAy*pOut[93] + 2*i2z*(pOut[59] + r_510_1*riz) + rPmQy*r_520_1;
   pOut[136] = PmAx*pOut[95] + 2*i2z*(pOut[60] + r_150_1*riz) + rPmQx*r_250_1;
   pOut[137] = PmAz*pOut[94] + 2*i2z*(pOut[61] + r_501_1*riz) + rPmQz*r_502_1;
   pOut[138] = PmAx*pOut[97] + 2*i2z*(pOut[62] + r_105_1*riz) + rPmQx*r_205_1;
   pOut[139] = PmAz*pOut[96] + 2*i2z*(pOut[63] + r_051_1*riz) + rPmQz*r_052_1;
   pOut[140] = PmAy*pOut[98] + 2*i2z*(pOut[64] + r_015_1*riz) + rPmQy*r_025_1;
   pOut[141] = PmAx*pOut[99] + 3*i2z*(pOut[67] + r_240_1*riz) + rPmQx*r_340_1;
   pOut[142] = PmAx*pOut[100] + 3*i2z*(pOut[68] + r_204_1*riz) + rPmQx*r_304_1;
   pOut[143] = PmAy*pOut[102] + 3*i2z*(pOut[70] + r_024_1*riz) + rPmQy*r_034_1;
   pOut[144] = PmAy*pOut[88] + rPmQy*r_106_1;
   pOut[145] = PmAz*pOut[87] + rPmQz*r_160_1;
   pOut[146] = PmAz*pOut[89] + rPmQz*r_610_1;
   pOut[147] = PmAy*pOut[94] + rPmQy*r_502_1;
   pOut[148] = PmAx*pOut[96] + rPmQx*r_052_1;
   pOut[149] = PmAz*pOut[93] + rPmQz*r_520_1;
   pOut[150] = PmAx*pOut[98] + rPmQx*r_025_1;
   pOut[151] = PmAz*pOut[95] + rPmQz*r_250_1;
   pOut[152] = PmAy*pOut[97] + rPmQy*r_205_1;
   pOut[153] = PmAy*pOut[100] + rPmQy*r_304_1;
   pOut[154] = PmAx*pOut[102] + rPmQx*r_034_1;
   pOut[155] = PmAz*pOut[99] + rPmQz*r_340_1;
   pOut[156] = PmAx*pOut[104] + rPmQx*r_043_1;
   pOut[157] = PmAz*pOut[101] + rPmQz*r_430_1;
   pOut[158] = PmAy*pOut[103] + rPmQy*r_403_1;
   pOut[159] = PmAy*pOut[110] + i2z*(pOut[66] + r_402_1*riz) + rPmQy*r_412_1;
   pOut[160] = PmAx*pOut[108] + i2z*(pOut[69] + r_042_1*riz) + rPmQx*r_142_1;
   pOut[161] = PmAx*pOut[109] + i2z*(pOut[70] + r_024_1*riz) + rPmQx*r_124_1;
   pOut[162] = PmAz*pOut[114] + i2z*(pOut[71] + r_330_1*riz) + rPmQz*r_331_1;
   pOut[163] = PmAy*pOut[115] + i2z*(pOut[72] + r_303_1*riz) + rPmQy*r_313_1;
   pOut[164] = PmAx*pOut[116] + i2z*(pOut[73] + r_033_1*riz) + rPmQx*r_133_1;
   // 2.60 steps per component, 0.56 kb stack
   if (lab == 8) return;

   // m + lab == 9
   double r_100_8 = PmAx*pIm[8] + pIm[9]*rPmQx;
   double r_010_8 = PmAy*pIm[8] + pIm[9]*rPmQy;
   double r_001_8 = PmAz*pIm[8] + pIm[9]*rPmQz;
   double r_200_7 = PmAx*r_100_7 + i2z*(pIm[7] + pIm[8]*riz) + rPmQx*r_100_8;
   double r_020_7 = PmAy*r_010_7 + i2z*(pIm[7] + pIm[8]*riz) + rPmQy*r_010_8;
   double r_002_7 = PmAz*r_001_7 + i2z*(pIm[7] + pIm[8]*riz) + rPmQz*r_001_8;
   double r_300_6 = PmAx*r_200_6 + 2*i2z*(r_100_6 + r_100_7*riz) + rPmQx*r_200_7;
   double r_030_6 = PmAy*r_020_6 + 2*i2z*(r_010_6 + r_010_7*riz) + rPmQy*r_020_7;
   double r_003_6 = PmAz*r_002_6 + 2*i2z*(r_001_6 + r_001_7*riz) + rPmQz*r_002_7;
   double r_400_5 = PmAx*r_300_5 + 3*i2z*(r_200_5 + r_200_6*riz) + rPmQx*r_300_6;
   double r_040_5 = PmAy*r_030_5 + 3*i2z*(r_020_5 + r_020_6*riz) + rPmQy*r_030_6;
   double r_004_5 = PmAz*r_003_5 + 3*i2z*(r_002_5 + r_002_6*riz) + rPmQz*r_003_6;
   double r_310_5 = PmAy*r_300_5 + rPmQy*r_300_6;
   double r_500_4 = PmAx*r_400_4 + 4*i2z*(r_300_4 + r_300_5*riz) + rPmQx*r_400_5;
   double r_050_4 = PmAy*r_040_4 + 4*i2z*(r_030_4 + r_030_5*riz) + rPmQy*r_040_5;
   double r_005_4 = PmAz*r_004_4 + 4*i2z*(r_003_4 + r_003_5*riz) + rPmQz*r_004_5;
   double r_140_4 = PmAx*r_040_4 + rPmQx*r_040_5;
   double r_104_4 = PmAx*r_004_4 + rPmQx*r_004_5;
   double r_410_4 = PmAy*r_400_4 + rPmQy*r_400_5;
   double r_014_4 = PmAy*r_004_4 + rPmQy*r_004_5;
   double r_401_4 = PmAz*r_400_4 + rPmQz*r_400_5;
   double r_041_4 = PmAz*r_040_4 + rPmQz*r_040_5;
   double r_320_4 = PmAy*r_310_4 + i2z*(r_300_4 + r_300_5*riz) + rPmQy*r_310_5;
   double r_600_3 = PmAx*r_500_3 + 5*i2z*(r_400_3 + r_400_4*riz) + rPmQx*r_500_4;
   double r_060_3 = PmAy*r_050_3 + 5*i2z*(r_040_3 + r_040_4*riz) + rPmQy*r_050_4;
   double r_006_3 = PmAz*r_005_3 + 5*i2z*(r_004_3 + r_004_4*riz) + rPmQz*r_005_4;
   double r_510_3 = PmAy*r_500_3 + rPmQy*r_500_4;
   double r_150_3 = PmAx*r_050_3 + rPmQx*r_050_4;
   double r_501_3 = PmAz*r_500_3 + rPmQz*r_500_4;
   double r_105_3 = PmAx*r_005_3 + rPmQx*r_005_4;
   double r_051_3 = PmAz*r_050_3 + rPmQz*r_050_4;
   double r_015_3 = PmAy*r_005_3 + rPmQy*r_005_4;
   double r_420_3 = PmAy*r_410_3 + i2z*(r_400_3 + r_400_4*riz) + rPmQy*r_410_4;
   double r_402_3 = PmAz*r_401_3 + i2z*(r_400_3 + r_400_4*riz) + rPmQz*r_401_4;
   double r_240_3 = PmAx*r_140_3 + i2z*(r_040_3 + r_040_4*riz) + rPmQx*r_140_4;
   double r_204_3 = PmAx*r_104_3 + i2z*(r_004_3 + r_004_4*riz) + rPmQx*r_104_4;
   double r_042_3 = PmAz*r_041_3 + i2z*(r_040_3 + r_040_4*riz) + rPmQz*r_041_4;
   double r_024_3 = PmAy*r_014_3 + i2z*(r_004_3 + r_004_4*riz) + rPmQy*r_014_4;
   double r_330_3 = PmAy*r_320_3 + 2*i2z*(r_310_3 + r_310_4*riz) + rPmQy*r_320_4;
   double r_700_2 = PmAx*r_600_2 + 6*i2z*(r_500_2 + r_500_3*riz) + rPmQx*r_600_3;
   double r_070_2 = PmAy*r_060_2 + 6*i2z*(r_050_2 + r_050_3*riz) + rPmQy*r_060_3;
   double r_007_2 = PmAz*r_006_2 + 6*i2z*(r_005_2 + r_005_3*riz) + rPmQz*r_006_3;
   double r_160_2 = PmAx*r_060_2 + rPmQx*r_060_3;
   double r_106_2 = PmAx*r_006_2 + rPmQx*r_006_3;
   double r_610_2 = PmAy*r_600_2 + rPmQy*r_600_3;
   double r_016_2 = PmAy*r_006_2 + rPmQy*r_006_3;
   double r_601_2 = PmAz*r_600_2 + rPmQz*r_600_3;
   double r_061_2 = PmAz*r_060_2 + rPmQz*r_060_3;
   double r_520_2 = PmAy*r_510_2 + i2z*(r_500_2 + r_500_3*riz) + rPmQy*r_510_3;
   double r_502_2 = PmAz*r_501_2 + i2z*(r_500_2 + r_500_3*riz) + rPmQz*r_501_3;
   double r_250_2 = PmAx*r_150_2 + i2z*(r_050_2 + r_050_3*riz) + rPmQx*r_150_3;
   double r_052_2 = PmAz*r_051_2 + i2z*(r_050_2 + r_050_3*riz) + rPmQz*r_051_3;
   double r_205_2 = PmAx*r_105_2 + i2z*(r_005_2 + r_005_3*riz) + rPmQx*r_105_3;
   double r_025_2 = PmAy*r_015_2 + i2z*(r_005_2 + r_005_3*riz) + rPmQy*r_015_3;
   double r_340_2 = PmAx*r_240_2 + 2*i2z*(r_140_2 + r_140_3*riz) + rPmQx*r_240_3;
   double r_304_2 = PmAx*r_204_2 + 2*i2z*(r_104_2 + r_104_3*riz) + rPmQx*r_204_3;
   double r_430_2 = PmAy*r_420_2 + 2*i2z*(r_410_2 + r_410_3*riz) + rPmQy*r_420_3;
   double r_034_2 = PmAy*r_024_2 + 2*i2z*(r_014_2 + r_014_3*riz) + rPmQy*r_024_3;
   double r_403_2 = PmAz*r_402_2 + 2*i2z*(r_401_2 + r_401_3*riz) + rPmQz*r_402_3;
   double r_043_2 = PmAz*r_042_2 + 2*i2z*(r_041_2 + r_041_3*riz) + rPmQz*r_042_3;
   double r_331_2 = PmAz*r_330_2 + rPmQz*r_330_3;
   double r_800_1 = PmAx*r_700_1 + 7*i2z*(r_600_1 + r_600_2*riz) + rPmQx*r_700_2;
   double r_080_1 = PmAy*r_070_1 + 7*i2z*(r_060_1 + r_060_2*riz) + rPmQy*r_070_2;
   double r_008_1 = PmAz*r_007_1 + 7*i2z*(r_006_1 + r_006_2*riz) + rPmQz*r_007_2;
   double r_710_1 = PmAy*r_700_1 + rPmQy*r_700_2;
   double r_170_1 = PmAx*r_070_1 + rPmQx*r_070_2;
   double r_701_1 = PmAz*r_700_1 + rPmQz*r_700_2;
   double r_107_1 = PmAx*r_007_1 + rPmQx*r_007_2;
   double r_071_1 = PmAz*r_070_1 + rPmQz*r_070_2;
   double r_017_1 = PmAy*r_007_1 + rPmQy*r_007_2;
   double r_620_1 = PmAy*r_610_1 + i2z*(r_600_1 + r_600_2*riz) + rPmQy*r_610_2;
   double r_602_1 = PmAz*r_601_1 + i2z*(r_600_1 + r_600_2*riz) + rPmQz*r_601_2;
   double r_260_1 = PmAx*r_160_1 + i2z*(r_060_1 + r_060_2*riz) + rPmQx*r_160_2;
   double r_206_1 = PmAx*r_106_1 + i2z*(r_006_1 + r_006_2*riz) + rPmQx*r_106_2;
   double r_062_1 = PmAz*r_061_1 + i2z*(r_060_1 + r_060_2*riz) + rPmQz*r_061_2;
   double r_026_1 = PmAy*r_016_1 + i2z*(r_006_1 + r_006_2*riz) + rPmQy*r_016_2;
   double r_530_1 = PmAy*r_520_1 + 2*i2z*(r_510_1 + r_510_2*riz) + rPmQy*r_520_2;
   double r_350_1 = PmAx*r_250_1 + 2*i2z*(r_150_1 + r_150_2*riz) + rPmQx*r_250_2;
   double r_503_1 = PmAz*r_502_1 + 2*i2z*(r_501_1 + r_501_2*riz) + rPmQz*r_502_2;
   double r_305_1 = PmAx*r_205_1 + 2*i2z*(r_105_1 + r_105_2*riz) + rPmQx*r_205_2;
   double r_053_1 = PmAz*r_052_1 + 2*i2z*(r_051_1 + r_051_2*riz) + rPmQz*r_052_2;
   double r_035_1 = PmAy*r_025_1 + 2*i2z*(r_015_1 + r_015_2*riz) + rPmQy*r_025_2;
   double r_440_1 = PmAx*r_340_1 + 3*i2z*(r_240_1 + r_240_2*riz) + rPmQx*r_340_2;
   double r_404_1 = PmAx*r_304_1 + 3*i2z*(r_204_1 + r_204_2*riz) + rPmQx*r_304_2;
   double r_044_1 = PmAy*r_034_1 + 3*i2z*(r_024_1 + r_024_2*riz) + rPmQy*r_034_2;
   double r_512_1 = PmAy*r_502_1 + rPmQy*r_502_2;
   double r_152_1 = PmAx*r_052_1 + rPmQx*r_052_2;
   double r_125_1 = PmAx*r_025_1 + rPmQx*r_025_2;
   double r_314_1 = PmAy*r_304_1 + rPmQy*r_304_2;
   double r_134_1 = PmAx*r_034_1 + rPmQx*r_034_2;
   double r_341_1 = PmAz*r_340_1 + rPmQz*r_340_2;
   double r_143_1 = PmAx*r_043_1 + rPmQx*r_043_2;
   double r_431_1 = PmAz*r_430_1 + rPmQz*r_430_2;
   double r_413_1 = PmAy*r_403_1 + rPmQy*r_403_2;
   double r_332_1 = PmAz*r_331_1 + i2z*(r_330_1 + r_330_2*riz) + rPmQz*r_331_2;
   pOut[165] = PmAx*pOut[120] + 8*i2z*(pOut[84] + r_700_1*riz) + rPmQx*r_800_1;
   pOut[166] = PmAy*pOut[121] + 8*i2z*(pOut[85] + r_070_1*riz) + rPmQy*r_080_1;
   pOut[167] = PmAz*pOut[122] + 8*i2z*(pOut[86] + r_007_1*riz) + rPmQz*r_008_1;
   pOut[168] = PmAx*pOut[121] + rPmQx*r_080_1;
   pOut[169] = PmAx*pOut[122] + rPmQx*r_008_1;
   pOut[170] = PmAy*pOut[120] + rPmQy*r_800_1;
   pOut[171] = PmAy*pOut[122] + rPmQy*r_008_1;
   pOut[172] = PmAz*pOut[120] + rPmQz*r_800_1;
   pOut[173] = PmAz*pOut[121] + rPmQz*r_080_1;
   pOut[174] = PmAy*pOut[123] + i2z*(pOut[84] + r_700_1*riz) + rPmQy*r_710_1;
   pOut[175] = PmAz*pOut[125] + i2z*(pOut[84] + r_700_1*riz) + rPmQz*r_701_1;
   pOut[176] = PmAx*pOut[124] + i2z*(pOut[85] + r_070_1*riz) + rPmQx*r_170_1;
   pOut[177] = PmAz*pOut[127] + i2z*(pOut[85] + r_070_1*riz) + rPmQz*r_071_1;
   pOut[178] = PmAx*pOut[126] + i2z*(pOut[86] + r_007_1*riz) + rPmQx*r_107_1;
   pOut[179] = PmAy*pOut[128] + i2z*(pOut[86] + r_007_1*riz) + rPmQy*r_017_1;
   pOut[180] = PmAx*pOut[131] + 2*i2z*(pOut[87] + r_160_1*riz) + rPmQx*r_260_1;
   pOut[181] = PmAx*pOut[132] + 2*i2z*(pOut[88] + r_106_1*riz) + rPmQx*r_206_1;
   pOut[182] = PmAy*pOut[129] + 2*i2z*(pOut[89] + r_610_1*riz) + rPmQy*r_620_1;
   pOut[183] = PmAy*pOut[134] + 2*i2z*(pOut[90] + r_016_1*riz) + rPmQy*r_026_1;
   pOut[184] = PmAz*pOut[130] + 2*i2z*(pOut[91] + r_601_1*riz) + rPmQz*r_602_1;
   pOut[185] = PmAz*pOut[133] + 2*i2z*(pOut[92] + r_061_1*riz) + rPmQz*r_062_1;
   pOut[186] = PmAy*pOut[135] + 3*i2z*(pOut[93] + r_520_1*riz) + rPmQy*r_530_1;
   pOut[187] = PmAz*pOut[137] + 3*i2z*(pOut[94] + r_502_1*riz) + rPmQz*r_503_1;
   pOut[188] = PmAx*pOut[136] + 3*i2z*(pOut[95] + r_250_1*riz) + rPmQx*r_350_1;
   pOut[189] = PmAz*pOut[139] + 3*i2z*(pOut[96] + r_052_1*riz) + rPmQz*r_053_1;
   pOut[190] = PmAx*pOut[138] + 3*i2z*(pOut[97] + r_205_1*riz) + rPmQx*r_305_1;
   pOut[191] = PmAy*pOut[140] + 3*i2z*(pOut[98] + r_025_1*riz) + rPmQy*r_035_1;
   pOut[192] = PmAz*pOut[123] + rPmQz*r_710_1;
   pOut[193] = PmAz*pOut[124] + rPmQz*r_170_1;
   pOut[194] = PmAy*pOut[126] + rPmQy*r_107_1;
   pOut[195] = PmAx*pOut[133] + rPmQx*r_062_1;
   pOut[196] = PmAx*pOut[134] + rPmQx*r_026_1;
   pOut[197] = PmAy*pOut[130] + rPmQy*r_602_1;
   pOut[198] = PmAy*pOut[132] + rPmQy*r_206_1;
   pOut[199] = PmAz*pOut[129] + rPmQz*r_620_1;
   pOut[200] = PmAz*pOut[131] + rPmQz*r_260_1;
   pOut[201] = PmAz*pOut[135] + rPmQz*r_530_1;
   pOut[202] = PmAy*pOut[137] + rPmQy*r_503_1;
   pOut[203] = PmAz*pOut[136] + rPmQz*r_350_1;
   pOut[204] = PmAy*pOut[138] + rPmQy*r_305_1;
   pOut[205] = PmAx*pOut[139] + rPmQx*r_053_1;
   pOut[206] = PmAx*pOut[140] + rPmQx*r_035_1;
   pOut[207] = PmAx*pOut[143] + rPmQx*r_044_1;
   pOut[208] = PmAy*pOut[142] + rPmQy*r_404_1;
   pOut[209] = PmAz*pOut[141] + rPmQz*r_440_1;
   pOut[210] = PmAy*pOut[147] + i2z*(pOut[94] + r_502_1*riz) + rPmQy*r_512_1;
   pOut[211] = PmAx*pOut[148] + i2z*(pOut[96] + r_052_1*riz) + rPmQx*r_152_1;
   pOut[212] = PmAx*pOut[150] + i2z*(pOut[98] + r_025_1*riz) + rPmQx*r_125_1;
   pOut[213] = PmAz*pOut[155] + i2z*(pOut[99] + r_340_1*riz) + rPmQz*r_341_1;
   pOut[214] = PmAy*pOut[153] + i2z*(pOut[100] + r_304_1*riz) + rPmQy*r_314_1;
   pOut[215] = PmAz*pOut[157] + i2z*(pOut[101] + r_430_1*riz) + rPmQz*r_431_1;
   pOut[216] = PmAx*pOut[154] + i2z*(pOut[102] + r_034_1*riz) + rPmQx*r_134_1;
   pOut[217] = PmAy*pOut[158] + i2z*(pOut[103] + r_403_1*riz) + rPmQy*r_413_1;
   pOut[218] = PmAx*pOut[156] + i2z*(pOut[104] + r_043_1*riz) + rPmQx*r_143_1;
   pOut[219] = PmAz*pOut[162] + 2*i2z*(pOut[114] + r_331_1*riz) + rPmQz*r_332_1;
   // 2.73 steps per component, 0.74 kb stack
   if (lab == 9) return;

   // m + lab == 10
   double r_100_9 = PmAx*pIm[9] + pIm[10]*rPmQx;
   double r_010_9 = PmAy*pIm[9] + pIm[10]*rPmQy;
   double r_001_9 = PmAz*pIm[9] + pIm[10]*rPmQz;
   double r_200_8 = PmAx*r_100_8 + i2z*(pIm[8] + pIm[9]*riz) + rPmQx*r_100_9;
   double r_020_8 = PmAy*r_010_8 + i2z*(pIm[8] + pIm[9]*riz) + rPmQy*r_010_9;
   double r_002_8 = PmAz*r_001_8 + i2z*(pIm[8] + pIm[9]*riz) + rPmQz*r_001_9;
   double r_300_7 = PmAx*r_200_7 + 2*i2z*(r_100_7 + r_100_8*riz) + rPmQx*r_200_8;
   double r_030_7 = PmAy*r_020_7 + 2*i2z*(r_010_7 + r_010_8*riz) + rPmQy*r_020_8;
   double r_003_7 = PmAz*r_002_7 + 2*i2z*(r_001_7 + r_001_8*riz) + rPmQz*r_002_8;
   double r_400_6 = PmAx*r_300_6 + 3*i2z*(r_200_6 + r_200_7*riz) + rPmQx*r_300_7;
   double r_040_6 = PmAy*r_030_6 + 3*i2z*(r_020_6 + r_020_7*riz) + rPmQy*r_030_7;
   double r_004_6 = PmAz*r_003_6 + 3*i2z*(r_002_6 + r_002_7*riz) + rPmQz*r_003_7;
   double r_500_5 = PmAx*r_400_5 + 4*i2z*(r_300_5 + r_300_6*riz) + rPmQx*r_400_6;
   double r_050_5 = PmAy*r_040_5 + 4*i2z*(r_030_5 + r_030_6*riz) + rPmQy*r_040_6;
   double r_005_5 = PmAz*r_004_5 + 4*i2z*(r_003_5 + r_003_6*riz) + rPmQz*r_004_6;
   double r_140_5 = PmAx*r_040_5 + rPmQx*r_040_6;
   double r_104_5 = PmAx*r_004_5 + rPmQx*r_004_6;
   double r_410_5 = PmAy*r_400_5 + rPmQy*r_400_6;
   double r_014_5 = PmAy*r_004_5 + rPmQy*r_004_6;
   double r_600_4 = PmAx*r_500_4 + 5*i2z*(r_400_4 + r_400_5*riz) + rPmQx*r_500_5;
   double r_060_4 = PmAy*r_050_4 + 5*i2z*(r_040_4 + r_040_5*riz) + rPmQy*r_050_5;
   double r_006_4 = PmAz*r_005_4 + 5*i2z*(r_004_4 + r_004_5*riz) + rPmQz*r_005_5;
   double r_510_4 = PmAy*r_500_4 + rPmQy*r_500_5;
   double r_150_4 = PmAx*r_050_4 + rPmQx*r_050_5;
   double r_501_4 = PmAz*r_500_4 + rPmQz*r_500_5;
   double r_105_4 = PmAx*r_005_4 + rPmQx*r_005_5;
   double r_051_4 = PmAz*r_050_4 + rPmQz*r_050_5;
   double r_015_4 = PmAy*r_005_4 + rPmQy*r_005_5;
   double r_420_4 = PmAy*r_410_4 + i2z*(r_400_4 + r_400_5*riz) + rPmQy*r_410_5;
   double r_240_4 = PmAx*r_140_4 + i2z*(r_040_4 + r_040_5*riz) + rPmQx*r_140_5;
   double r_204_4 = PmAx*r_104_4 + i2z*(r_004_4 + r_004_5*riz) + rPmQx*r_104_5;
   double r_024_4 = PmAy*r_014_4 + i2z*(r_004_4 + r_004_5*riz) + rPmQy*r_014_5;
   double r_700_3 = PmAx*r_600_3 + 6*i2z*(r_500_3 + r_500_4*riz) + rPmQx*r_600_4;
   double r_070_3 = PmAy*r_060_3 + 6*i2z*(r_050_3 + r_050_4*riz) + rPmQy*r_060_4;
   double r_007_3 = PmAz*r_006_3 + 6*i2z*(r_005_3 + r_005_4*riz) + rPmQz*r_006_4;
   double r_160_3 = PmAx*r_060_3 + rPmQx*r_060_4;
   double r_106_3 = PmAx*r_006_3 + rPmQx*r_006_4;
   double r_610_3 = PmAy*r_600_3 + rPmQy*r_600_4;
   double r_016_3 = PmAy*r_006_3 + rPmQy*r_006_4;
   double r_601_3 = PmAz*r_600_3 + rPmQz*r_600_4;
   double r_061_3 = PmAz*r_060_3 + rPmQz*r_060_4;
   double r_520_3 = PmAy*r_510_3 + i2z*(r_500_3 + r_500_4*riz) + rPmQy*r_510_4;
   double r_502_3 = PmAz*r_501_3 + i2z*(r_500_3 + r_500_4*riz) + rPmQz*r_501_4;
   double r_250_3 = PmAx*r_150_3 + i2z*(r_050_3 + r_050_4*riz) + rPmQx*r_150_4;
   double r_052_3 = PmAz*r_051_3 + i2z*(r_050_3 + r_050_4*riz) + rPmQz*r_051_4;
   double r_205_3 = PmAx*r_105_3 + i2z*(r_005_3 + r_005_4*riz) + rPmQx*r_105_4;
   double r_025_3 = PmAy*r_015_3 + i2z*(r_005_3 + r_005_4*riz) + rPmQy*r_015_4;
   double r_340_3 = PmAx*r_240_3 + 2*i2z*(r_140_3 + r_140_4*riz) + rPmQx*r_240_4;
   double r_304_3 = PmAx*r_204_3 + 2*i2z*(r_104_3 + r_104_4*riz) + rPmQx*r_204_4;
   double r_430_3 = PmAy*r_420_3 + 2*i2z*(r_410_3 + r_410_4*riz) + rPmQy*r_420_4;
   double r_034_3 = PmAy*r_024_3 + 2*i2z*(r_014_3 + r_014_4*riz) + rPmQy*r_024_4;
   double r_800_2 = PmAx*r_700_2 + 7*i2z*(r_600_2 + r_600_3*riz) + rPmQx*r_700_3;
   double r_080_2 = PmAy*r_070_2 + 7*i2z*(r_060_2 + r_060_3*riz) + rPmQy*r_070_3;
   double r_008_2 = PmAz*r_007_2 + 7*i2z*(r_006_2 + r_006_3*riz) + rPmQz*r_007_3;
   double r_710_2 = PmAy*r_700_2 + rPmQy*r_700_3;
   double r_170_2 = PmAx*r_070_2 + rPmQx*r_070_3;
   double r_701_2 = PmAz*r_700_2 + rPmQz*r_700_3;
   double r_107_2 = PmAx*r_007_2 + rPmQx*r_007_3;
   double r_071_2 = PmAz*r_070_2 + rPmQz*r_070_3;
   double r_017_2 = PmAy*r_007_2 + rPmQy*r_007_3;
   double r_620_2 = PmAy*r_610_2 + i2z*(r_600_2 + r_600_3*riz) + rPmQy*r_610_3;
   double r_602_2 = PmAz*r_601_2 + i2z*(r_600_2 + r_600_3*riz) + rPmQz*r_601_3;
   double r_260_2 = PmAx*r_160_2 + i2z*(r_060_2 + r_060_3*riz) + rPmQx*r_160_3;
   double r_206_2 = PmAx*r_106_2 + i2z*(r_006_2 + r_006_3*riz) + rPmQx*r_106_3;
   double r_062_2 = PmAz*r_061_2 + i2z*(r_060_2 + r_060_3*riz) + rPmQz*r_061_3;
   double r_026_2 = PmAy*r_016_2 + i2z*(r_006_2 + r_006_3*riz) + rPmQy*r_016_3;
   double r_530_2 = PmAy*r_520_2 + 2*i2z*(r_510_2 + r_510_3*riz) + rPmQy*r_520_3;
   double r_350_2 = PmAx*r_250_2 + 2*i2z*(r_150_2 + r_150_3*riz) + rPmQx*r_250_3;
   double r_503_2 = PmAz*r_502_2 + 2*i2z*(r_501_2 + r_501_3*riz) + rPmQz*r_502_3;
   double r_305_2 = PmAx*r_205_2 + 2*i2z*(r_105_2 + r_105_3*riz) + rPmQx*r_205_3;
   double r_053_2 = PmAz*r_052_2 + 2*i2z*(r_051_2 + r_051_3*riz) + rPmQz*r_052_3;
   double r_035_2 = PmAy*r_025_2 + 2*i2z*(r_015_2 + r_015_3*riz) + rPmQy*r_025_3;
   double r_440_2 = PmAx*r_340_2 + 3*i2z*(r_240_2 + r_240_3*riz) + rPmQx*r_340_3;
   double r_404_2 = PmAx*r_304_2 + 3*i2z*(r_204_2 + r_204_3*riz) + rPmQx*r_304_3;
   double r_044_2 = PmAy*r_034_2 + 3*i2z*(r_024_2 + r_024_3*riz) + rPmQy*r_034_3;
   double r_314_2 = PmAy*r_304_2 + rPmQy*r_304_3;
   double r_341_2 = PmAz*r_340_2 + rPmQz*r_340_3;
   double r_431_2 = PmAz*r_430_2 + rPmQz*r_430_3;
   double r_900_1 = PmAx*r_800_1 + 8*i2z*(r_700_1 + r_700_2*riz) + rPmQx*r_800_2;
   double r_090_1 = PmAy*r_080_1 + 8*i2z*(r_070_1 + r_070_2*riz) + rPmQy*r_080_2;
   double r_009_1 = PmAz*r_008_1 + 8*i2z*(r_007_1 + r_007_2*riz) + rPmQz*r_008_2;
   double r_180_1 = PmAx*r_080_1 + rPmQx*r_080_2;
   double r_108_1 = PmAx*r_008_1 + rPmQx*r_008_2;
   double r_810_1 = PmAy*r_800_1 + rPmQy*r_800_2;
   double r_018_1 = PmAy*r_008_1 + rPmQy*r_008_2;
   double r_801_1 = PmAz*r_800_1 + rPmQz*r_800_2;
   double r_081_1 = PmAz*r_080_1 + rPmQz*r_080_2;
   double r_720_1 = PmAy*r_710_1 + i2z*(r_700_1 + r_700_2*riz) + rPmQy*r_710_2;
   double r_702_1 = PmAz*r_701_1 + i2z*(r_700_1 + r_700_2*riz) + rPmQz*r_701_2;
   double r_270_1 = PmAx*r_170_1 + i2z*(r_070_1 + r_070_2*riz) + rPmQx*r_170_2;
   double r_072_1 = PmAz*r_071_1 + i2z*(r_070_1 + r_070_2*riz) + rPmQz*r_071_2;
   double r_207_1 = PmAx*r_107_1 + i2z*(r_007_1 + r_007_2*riz) + rPmQx*r_107_2;
   double r_027_1 = PmAy*r_017_1 + i2z*(r_007_1 + r_007_2*riz) + rPmQy*r_017_2;
   double r_360_1 = PmAx*r_260_1 + 2*i2z*(r_160_1 + r_160_2*riz) + rPmQx*r_260_2;
   double r_306_1 = PmAx*r_206_1 + 2*i2z*(r_106_1 + r_106_2*riz) + rPmQx*r_206_2;
   double r_630_1 = PmAy*r_620_1 + 2*i2z*(r_610_1 + r_610_2*riz) + rPmQy*r_620_2;
   double r_036_1 = PmAy*r_026_1 + 2*i2z*(r_016_1 + r_016_2*riz) + rPmQy*r_026_2;
   double r_603_1 = PmAz*r_602_1 + 2*i2z*(r_601_1 + r_601_2*riz) + rPmQz*r_602_2;
   double r_063_1 = PmAz*r_062_1 + 2*i2z*(r_061_1 + r_061_2*riz) + rPmQz*r_062_2;
   double r_540_1 = PmAy*r_530_1 + 3*i2z*(r_520_1 + r_520_2*riz) + rPmQy*r_530_2;
   double r_504_1 = PmAz*r_503_1 + 3*i2z*(r_502_1 + r_502_2*riz) + rPmQz*r_503_2;
   double r_450_1 = PmAx*r_350_1 + 3*i2z*(r_250_1 + r_250_2*riz) + rPmQx*r_350_2;
   double r_054_1 = PmAz*r_053_1 + 3*i2z*(r_052_1 + r_052_2*riz) + rPmQz*r_053_2;
   double r_405_1 = PmAx*r_305_1 + 3*i2z*(r_205_1 + r_205_2*riz) + rPmQx*r_305_2;
   double r_045_1 = PmAy*r_035_1 + 3*i2z*(r_025_1 + r_025_2*riz) + rPmQy*r_035_2;
   double r_162_1 = PmAx*r_062_1 + rPmQx*r_062_2;
   double r_126_1 = PmAx*r_026_1 + rPmQx*r_026_2;
   double r_612_1 = PmAy*r_602_1 + rPmQy*r_602_2;
   double r_531_1 = PmAz*r_530_1 + rPmQz*r_530_2;
   double r_513_1 = PmAy*r_503_1 + rPmQy*r_503_2;
   double r_351_1 = PmAz*r_350_1 + rPmQz*r_350_2;
   double r_315_1 = PmAy*r_305_1 + rPmQy*r_305_2;
   double r_153_1 = PmAx*r_053_1 + rPmQx*r_053_2;
   double r_135_1 = PmAx*r_035_1 + rPmQx*r_035_2;
   double r_144_1 = PmAx*r_044_1 + rPmQx*r_044_2;
   double r_414_1 = PmAy*r_404_1 + rPmQy*r_404_2;
   double r_441_1 = PmAz*r_440_1 + rPmQz*r_440_2;
   double r_342_1 = PmAz*r_341_1 + i2z*(r_340_1 + r_340_2*riz) + rPmQz*r_341_2;
   double r_324_1 = PmAy*r_314_1 + i2z*(r_304_1 + r_304_2*riz) + rPmQy*r_314_2;
   double r_432_1 = PmAz*r_431_1 + i2z*(r_430_1 + r_430_2*riz) + rPmQz*r_431_2;
   pOut[220] = PmAx*pOut[165] + 9*i2z*(pOut[120] + r_800_1*riz) + rPmQx*r_900_1;
   pOut[221] = PmAy*pOut[166] + 9*i2z*(pOut[121] + r_080_1*riz) + rPmQy*r_090_1;
   pOut[222] = PmAz*pOut[167] + 9*i2z*(pOut[122] + r_008_1*riz) + rPmQz*r_009_1;
   pOut[223] = PmAy*pOut[165] + rPmQy*r_900_1;
   pOut[224] = PmAx*pOut[166] + rPmQx*r_090_1;
   pOut[225] = PmAz*pOut[165] + rPmQz*r_900_1;
   pOut[226] = PmAx*pOut[167] + rPmQx*r_009_1;
   pOut[227] = PmAz*pOut[166] + rPmQz*r_090_1;
   pOut[228] = PmAy*pOut[167] + rPmQy*r_009_1;
   pOut[229] = PmAy*pOut[170] + i2z*(pOut[120] + r_800_1*riz) + rPmQy*r_810_1;
   pOut[230] = PmAz*pOut[172] + i2z*(pOut[120] + r_800_1*riz) + rPmQz*r_801_1;
   pOut[231] = PmAx*pOut[168] + i2z*(pOut[121] + r_080_1*riz) + rPmQx*r_180_1;
   pOut[232] = PmAx*pOut[169] + i2z*(pOut[122] + r_008_1*riz) + rPmQx*r_108_1;
   pOut[233] = PmAz*pOut[173] + i2z*(pOut[121] + r_080_1*riz) + rPmQz*r_081_1;
   pOut[234] = PmAy*pOut[171] + i2z*(pOut[122] + r_008_1*riz) + rPmQy*r_018_1;
   pOut[235] = PmAy*pOut[174] + 2*i2z*(pOut[123] + r_710_1*riz) + rPmQy*r_720_1;
   pOut[236] = PmAx*pOut[176] + 2*i2z*(pOut[124] + r_170_1*riz) + rPmQx*r_270_1;
   pOut[237] = PmAz*pOut[175] + 2*i2z*(pOut[125] + r_701_1*riz) + rPmQz*r_702_1;
   pOut[238] = PmAx*pOut[178] + 2*i2z*(pOut[126] + r_107_1*riz) + rPmQx*r_207_1;
   pOut[239] = PmAz*pOut[177] + 2*i2z*(pOut[127] + r_071_1*riz) + rPmQz*r_072_1;
   pOut[240] = PmAy*pOut[179] + 2*i2z*(pOut[128] + r_017_1*riz) + rPmQy*r_027_1;
   pOut[241] = PmAy*pOut[182] + 3*i2z*(pOut[129] + r_620_1*riz) + rPmQy*r_630_1;
   pOut[242] = PmAz*pOut[184] + 3*i2z*(pOut[130] + r_602_1*riz) + rPmQz*r_603_1;
   pOut[243] = PmAx*pOut[180] + 3*i2z*(pOut[131] + r_260_1*riz) + rPmQx*r_360_1;
   pOut[244] = PmAx*pOut[181] + 3*i2z*(pOut[132] + r_206_1*riz) + rPmQx*r_306_1;
   pOut[245] = PmAz*pOut[185] + 3*i2z*(pOut[133] + r_062_1*riz) + rPmQz*r_063_1;
   pOut[246] = PmAy*pOut[183] + 3*i2z*(pOut[134] + r_026_1*riz) + rPmQy*r_036_1;
   pOut[247] = PmAy*pOut[186] + 4*i2z*(pOut[135] + r_530_1*riz) + rPmQy*r_540_1;
   pOut[248] = PmAz*pOut[187] + 4*i2z*(pOut[137] + r_503_1*riz) + rPmQz*r_504_1;
   pOut[249] = PmAz*pOut[189] + 4*i2z*(pOut[139] + r_053_1*riz) + rPmQz*r_054_1;
   pOut[250] = PmAy*pOut[169] + rPmQy*r_108_1;
   pOut[251] = PmAz*pOut[168] + rPmQz*r_180_1;
   pOut[252] = PmAz*pOut[170] + rPmQz*r_810_1;
   pOut[253] = PmAy*pOut[175] + rPmQy*r_702_1;
   pOut[254] = PmAx*pOut[177] + rPmQx*r_072_1;
   pOut[255] = PmAz*pOut[174] + rPmQz*r_720_1;
   pOut[256] = PmAx*pOut[179] + rPmQx*r_027_1;
   pOut[257] = PmAz*pOut[176] + rPmQz*r_270_1;
   pOut[258] = PmAy*pOut[178] + rPmQy*r_207_1;
   pOut[259] = PmAy*pOut[181] + rPmQy*r_306_1;
   pOut[260] = PmAx*pOut[183] + rPmQx*r_036_1;
   pOut[261] = PmAz*pOut[180] + rPmQz*r_360_1;
   pOut[262] = PmAx*pOut[185] + rPmQx*r_063_1;
   pOut[263] = PmAz*pOut[182] + rPmQz*r_630_1;
   pOut[264] = PmAy*pOut[184] + rPmQy*r_603_1;
   pOut[265] = PmAy*pOut[187] + rPmQy*r_504_1;
   pOut[266] = PmAx*pOut[189] + rPmQx*r_054_1;
   pOut[267] = PmAz*pOut[186] + rPmQz*r_540_1;
   pOut[268] = PmAx*pOut[191] + rPmQx*r_045_1;
   pOut[269] = PmAz*pOut[188] + rPmQz*r_450_1;
   pOut[270] = PmAy*pOut[190] + rPmQy*r_405_1;
   pOut[271] = PmAy*pOut[197] + i2z*(pOut[130] + r_602_1*riz) + rPmQy*r_612_1;
   pOut[272] = PmAx*pOut[195] + i2z*(pOut[133] + r_062_1*riz) + rPmQx*r_162_1;
   pOut[273] = PmAx*pOut[196] + i2z*(pOut[134] + r_026_1*riz) + rPmQx*r_126_1;
   pOut[274] = PmAz*pOut[201] + i2z*(pOut[135] + r_530_1*riz) + rPmQz*r_531_1;
   pOut[275] = PmAz*pOut[203] + i2z*(pOut[136] + r_350_1*riz) + rPmQz*r_351_1;
   pOut[276] = PmAy*pOut[202] + i2z*(pOut[137] + r_503_1*riz) + rPmQy*r_513_1;
   pOut[277] = PmAy*pOut[204] + i2z*(pOut[138] + r_305_1*riz) + rPmQy*r_315_1;
   pOut[278] = PmAx*pOut[205] + i2z*(pOut[139] + r_053_1*riz) + rPmQx*r_153_1;
   pOut[279] = PmAx*pOut[206] + i2z*(pOut[140] + r_035_1*riz) + rPmQx*r_135_1;
   pOut[280] = PmAz*pOut[209] + i2z*(pOut[141] + r_440_1*riz) + rPmQz*r_441_1;
   pOut[281] = PmAy*pOut[208] + i2z*(pOut[142] + r_404_1*riz) + rPmQy*r_414_1;
   pOut[282] = PmAx*pOut[207] + i2z*(pOut[143] + r_044_1*riz) + rPmQx*r_144_1;
   pOut[283] = PmAy*pOut[214] + 2*i2z*(pOut[153] + r_314_1*riz) + rPmQy*r_324_1;
   pOut[284] = PmAz*pOut[213] + 2*i2z*(pOut[155] + r_341_1*riz) + rPmQz*r_342_1;
   pOut[285] = PmAz*pOut[215] + 2*i2z*(pOut[157] + r_431_1*riz) + rPmQz*r_432_1;
   // 2.82 steps per component, 0.94 kb stack
   if (lab == 10) return;

   // m + lab == 11
   double r_100_a = PmAx*pIm[10] + pIm[11]*rPmQx;
   double r_010_a = PmAy*pIm[10] + pIm[11]*rPmQy;
   double r_001_a = PmAz*pIm[10] + pIm[11]*rPmQz;
   double r_200_9 = PmAx*r_100_9 + i2z*(pIm[10]*riz + pIm[9]) + rPmQx*r_100_a;
   double r_020_9 = PmAy*r_010_9 + i2z*(pIm[10]*riz + pIm[9]) + rPmQy*r_010_a;
   double r_002_9 = PmAz*r_001_9 + i2z*(pIm[10]*riz + pIm[9]) + rPmQz*r_001_a;
   double r_300_8 = PmAx*r_200_8 + 2*i2z*(r_100_8 + r_100_9*riz) + rPmQx*r_200_9;
   double r_030_8 = PmAy*r_020_8 + 2*i2z*(r_010_8 + r_010_9*riz) + rPmQy*r_020_9;
   double r_003_8 = PmAz*r_002_8 + 2*i2z*(r_001_8 + r_001_9*riz) + rPmQz*r_002_9;
   double r_400_7 = PmAx*r_300_7 + 3*i2z*(r_200_7 + r_200_8*riz) + rPmQx*r_300_8;
   double r_040_7 = PmAy*r_030_7 + 3*i2z*(r_020_7 + r_020_8*riz) + rPmQy*r_030_8;
   double r_004_7 = PmAz*r_003_7 + 3*i2z*(r_002_7 + r_002_8*riz) + rPmQz*r_003_8;
   double r_500_6 = PmAx*r_400_6 + 4*i2z*(r_300_6 + r_300_7*riz) + rPmQx*r_400_7;
   double r_050_6 = PmAy*r_040_6 + 4*i2z*(r_030_6 + r_030_7*riz) + rPmQy*r_040_7;
   double r_005_6 = PmAz*r_004_6 + 4*i2z*(r_003_6 + r_003_7*riz) + rPmQz*r_004_7;
   double r_140_6 = PmAx*r_040_6 + rPmQx*r_040_7;
   double r_104_6 = PmAx*r_004_6 + rPmQx*r_004_7;
   double r_014_6 = PmAy*r_004_6 + rPmQy*r_004_7;
   double r_600_5 = PmAx*r_500_5 + 5*i2z*(r_400_5 + r_400_6*riz) + rPmQx*r_500_6;
   double r_060_5 = PmAy*r_050_5 + 5*i2z*(r_040_5 + r_040_6*riz) + rPmQy*r_050_6;
   double r_006_5 = PmAz*r_005_5 + 5*i2z*(r_004_5 + r_004_6*riz) + rPmQz*r_005_6;
   double r_510_5 = PmAy*r_500_5 + rPmQy*r_500_6;
   double r_150_5 = PmAx*r_050_5 + rPmQx*r_050_6;
   double r_501_5 = PmAz*r_500_5 + rPmQz*r_500_6;
   double r_105_5 = PmAx*r_005_5 + rPmQx*r_005_6;
   double r_051_5 = PmAz*r_050_5 + rPmQz*r_050_6;
   double r_015_5 = PmAy*r_005_5 + rPmQy*r_005_6;
   double r_240_5 = PmAx*r_140_5 + i2z*(r_040_5 + r_040_6*riz) + rPmQx*r_140_6;
   double r_204_5 = PmAx*r_104_5 + i2z*(r_004_5 + r_004_6*riz) + rPmQx*r_104_6;
   double r_024_5 = PmAy*r_014_5 + i2z*(r_004_5 + r_004_6*riz) + rPmQy*r_014_6;
   double r_700_4 = PmAx*r_600_4 + 6*i2z*(r_500_4 + r_500_5*riz) + rPmQx*r_600_5;
   double r_070_4 = PmAy*r_060_4 + 6*i2z*(r_050_4 + r_050_5*riz) + rPmQy*r_060_5;
   double r_007_4 = PmAz*r_006_4 + 6*i2z*(r_005_4 + r_005_5*riz) + rPmQz*r_006_5;
   double r_160_4 = PmAx*r_060_4 + rPmQx*r_060_5;
   double r_106_4 = PmAx*r_006_4 + rPmQx*r_006_5;
   double r_610_4 = PmAy*r_600_4 + rPmQy*r_600_5;
   double r_016_4 = PmAy*r_006_4 + rPmQy*r_006_5;
   double r_601_4 = PmAz*r_600_4 + rPmQz*r_600_5;
   double r_061_4 = PmAz*r_060_4 + rPmQz*r_060_5;
   double r_520_4 = PmAy*r_510_4 + i2z*(r_500_4 + r_500_5*riz) + rPmQy*r_510_5;
   double r_502_4 = PmAz*r_501_4 + i2z*(r_500_4 + r_500_5*riz) + rPmQz*r_501_5;
   double r_250_4 = PmAx*r_150_4 + i2z*(r_050_4 + r_050_5*riz) + rPmQx*r_150_5;
   double r_052_4 = PmAz*r_051_4 + i2z*(r_050_4 + r_050_5*riz) + rPmQz*r_051_5;
   double r_205_4 = PmAx*r_105_4 + i2z*(r_005_4 + r_005_5*riz) + rPmQx*r_105_5;
   double r_025_4 = PmAy*r_015_4 + i2z*(r_005_4 + r_005_5*riz) + rPmQy*r_015_5;
   double r_340_4 = PmAx*r_240_4 + 2*i2z*(r_140_4 + r_140_5*riz) + rPmQx*r_240_5;
   double r_304_4 = PmAx*r_204_4 + 2*i2z*(r_104_4 + r_104_5*riz) + rPmQx*r_204_5;
   double r_034_4 = PmAy*r_024_4 + 2*i2z*(r_014_4 + r_014_5*riz) + rPmQy*r_024_5;
   double r_800_3 = PmAx*r_700_3 + 7*i2z*(r_600_3 + r_600_4*riz) + rPmQx*r_700_4;
   double r_080_3 = PmAy*r_070_3 + 7*i2z*(r_060_3 + r_060_4*riz) + rPmQy*r_070_4;
   double r_008_3 = PmAz*r_007_3 + 7*i2z*(r_006_3 + r_006_4*riz) + rPmQz*r_007_4;
   double r_710_3 = PmAy*r_700_3 + rPmQy*r_700_4;
   double r_170_3 = PmAx*r_070_3 + rPmQx*r_070_4;
   double r_701_3 = PmAz*r_700_3 + rPmQz*r_700_4;
   double r_107_3 = PmAx*r_007_3 + rPmQx*r_007_4;
   double r_071_3 = PmAz*r_070_3 + rPmQz*r_070_4;
   double r_017_3 = PmAy*r_007_3 + rPmQy*r_007_4;
   double r_620_3 = PmAy*r_610_3 + i2z*(r_600_3 + r_600_4*riz) + rPmQy*r_610_4;
   double r_602_3 = PmAz*r_601_3 + i2z*(r_600_3 + r_600_4*riz) + rPmQz*r_601_4;
   double r_260_3 = PmAx*r_160_3 + i2z*(r_060_3 + r_060_4*riz) + rPmQx*r_160_4;
   double r_206_3 = PmAx*r_106_3 + i2z*(r_006_3 + r_006_4*riz) + rPmQx*r_106_4;
   double r_062_3 = PmAz*r_061_3 + i2z*(r_060_3 + r_060_4*riz) + rPmQz*r_061_4;
   double r_026_3 = PmAy*r_016_3 + i2z*(r_006_3 + r_006_4*riz) + rPmQy*r_016_4;
   double r_530_3 = PmAy*r_520_3 + 2*i2z*(r_510_3 + r_510_4*riz) + rPmQy*r_520_4;
   double r_350_3 = PmAx*r_250_3 + 2*i2z*(r_150_3 + r_150_4*riz) + rPmQx*r_250_4;
   double r_503_3 = PmAz*r_502_3 + 2*i2z*(r_501_3 + r_501_4*riz) + rPmQz*r_502_4;
   double r_305_3 = PmAx*r_205_3 + 2*i2z*(r_105_3 + r_105_4*riz) + rPmQx*r_205_4;
   double r_053_3 = PmAz*r_052_3 + 2*i2z*(r_051_3 + r_051_4*riz) + rPmQz*r_052_4;
   double r_035_3 = PmAy*r_025_3 + 2*i2z*(r_015_3 + r_015_4*riz) + rPmQy*r_025_4;
   double r_440_3 = PmAx*r_340_3 + 3*i2z*(r_240_3 + r_240_4*riz) + rPmQx*r_340_4;
   double r_404_3 = PmAx*r_304_3 + 3*i2z*(r_204_3 + r_204_4*riz) + rPmQx*r_304_4;
   double r_044_3 = PmAy*r_034_3 + 3*i2z*(r_024_3 + r_024_4*riz) + rPmQy*r_034_4;
   double r_900_2 = PmAx*r_800_2 + 8*i2z*(r_700_2 + r_700_3*riz) + rPmQx*r_800_3;
   double r_090_2 = PmAy*r_080_2 + 8*i2z*(r_070_2 + r_070_3*riz) + rPmQy*r_080_3;
   double r_009_2 = PmAz*r_008_2 + 8*i2z*(r_007_2 + r_007_3*riz) + rPmQz*r_008_3;
   double r_180_2 = PmAx*r_080_2 + rPmQx*r_080_3;
   double r_108_2 = PmAx*r_008_2 + rPmQx*r_008_3;
   double r_810_2 = PmAy*r_800_2 + rPmQy*r_800_3;
   double r_018_2 = PmAy*r_008_2 + rPmQy*r_008_3;
   double r_801_2 = PmAz*r_800_2 + rPmQz*r_800_3;
   double r_081_2 = PmAz*r_080_2 + rPmQz*r_080_3;
   double r_720_2 = PmAy*r_710_2 + i2z*(r_700_2 + r_700_3*riz) + rPmQy*r_710_3;
   double r_702_2 = PmAz*r_701_2 + i2z*(r_700_2 + r_700_3*riz) + rPmQz*r_701_3;
   double r_270_2 = PmAx*r_170_2 + i2z*(r_070_2 + r_070_3*riz) + rPmQx*r_170_3;
   double r_072_2 = PmAz*r_071_2 + i2z*(r_070_2 + r_070_3*riz) + rPmQz*r_071_3;
   double r_207_2 = PmAx*r_107_2 + i2z*(r_007_2 + r_007_3*riz) + rPmQx*r_107_3;
   double r_027_2 = PmAy*r_017_2 + i2z*(r_007_2 + r_007_3*riz) + rPmQy*r_017_3;
   double r_360_2 = PmAx*r_260_2 + 2*i2z*(r_160_2 + r_160_3*riz) + rPmQx*r_260_3;
   double r_306_2 = PmAx*r_206_2 + 2*i2z*(r_106_2 + r_106_3*riz) + rPmQx*r_206_3;
   double r_630_2 = PmAy*r_620_2 + 2*i2z*(r_610_2 + r_610_3*riz) + rPmQy*r_620_3;
   double r_036_2 = PmAy*r_026_2 + 2*i2z*(r_016_2 + r_016_3*riz) + rPmQy*r_026_3;
   double r_603_2 = PmAz*r_602_2 + 2*i2z*(r_601_2 + r_601_3*riz) + rPmQz*r_602_3;
   double r_063_2 = PmAz*r_062_2 + 2*i2z*(r_061_2 + r_061_3*riz) + rPmQz*r_062_3;
   double r_540_2 = PmAy*r_530_2 + 3*i2z*(r_520_2 + r_520_3*riz) + rPmQy*r_530_3;
   double r_504_2 = PmAz*r_503_2 + 3*i2z*(r_502_2 + r_502_3*riz) + rPmQz*r_503_3;
   double r_450_2 = PmAx*r_350_2 + 3*i2z*(r_250_2 + r_250_3*riz) + rPmQx*r_350_3;
   double r_054_2 = PmAz*r_053_2 + 3*i2z*(r_052_2 + r_052_3*riz) + rPmQz*r_053_3;
   double r_405_2 = PmAx*r_305_2 + 3*i2z*(r_205_2 + r_205_3*riz) + rPmQx*r_305_3;
   double r_045_2 = PmAy*r_035_2 + 3*i2z*(r_025_2 + r_025_3*riz) + rPmQy*r_035_3;
   double r_531_2 = PmAz*r_530_2 + rPmQz*r_530_3;
   double r_351_2 = PmAz*r_350_2 + rPmQz*r_350_3;
   double r_315_2 = PmAy*r_305_2 + rPmQy*r_305_3;
   double r_144_2 = PmAx*r_044_2 + rPmQx*r_044_3;
   double r_414_2 = PmAy*r_404_2 + rPmQy*r_404_3;
   double r_441_2 = PmAz*r_440_2 + rPmQz*r_440_3;
   double r_a00_1 = PmAx*r_900_1 + 9*i2z*(r_800_1 + r_800_2*riz) + rPmQx*r_900_2;
   double r_0a0_1 = PmAy*r_090_1 + 9*i2z*(r_080_1 + r_080_2*riz) + rPmQy*r_090_2;
   double r_00a_1 = PmAz*r_009_1 + 9*i2z*(r_008_1 + r_008_2*riz) + rPmQz*r_009_2;
   double r_910_1 = PmAy*r_900_1 + rPmQy*r_900_2;
   double r_190_1 = PmAx*r_090_1 + rPmQx*r_090_2;
   double r_901_1 = PmAz*r_900_1 + rPmQz*r_900_2;
   double r_109_1 = PmAx*r_009_1 + rPmQx*r_009_2;
   double r_091_1 = PmAz*r_090_1 + rPmQz*r_090_2;
   double r_019_1 = PmAy*r_009_1 + rPmQy*r_009_2;
   double r_820_1 = PmAy*r_810_1 + i2z*(r_800_1 + r_800_2*riz) + rPmQy*r_810_2;
   double r_802_1 = PmAz*r_801_1 + i2z*(r_800_1 + r_800_2*riz) + rPmQz*r_801_2;
   double r_280_1 = PmAx*r_180_1 + i2z*(r_080_1 + r_080_2*riz) + rPmQx*r_180_2;
   double r_208_1 = PmAx*r_108_1 + i2z*(r_008_1 + r_008_2*riz) + rPmQx*r_108_2;
   double r_082_1 = PmAz*r_081_1 + i2z*(r_080_1 + r_080_2*riz) + rPmQz*r_081_2;
   double r_028_1 = PmAy*r_018_1 + i2z*(r_008_1 + r_008_2*riz) + rPmQy*r_018_2;
   double r_730_1 = PmAy*r_720_1 + 2*i2z*(r_710_1 + r_710_2*riz) + rPmQy*r_720_2;
   double r_370_1 = PmAx*r_270_1 + 2*i2z*(r_170_1 + r_170_2*riz) + rPmQx*r_270_2;
   double r_703_1 = PmAz*r_702_1 + 2*i2z*(r_701_1 + r_701_2*riz) + rPmQz*r_702_2;
   double r_307_1 = PmAx*r_207_1 + 2*i2z*(r_107_1 + r_107_2*riz) + rPmQx*r_207_2;
   double r_073_1 = PmAz*r_072_1 + 2*i2z*(r_071_1 + r_071_2*riz) + rPmQz*r_072_2;
   double r_037_1 = PmAy*r_027_1 + 2*i2z*(r_017_1 + r_017_2*riz) + rPmQy*r_027_2;
   double r_640_1 = PmAy*r_630_1 + 3*i2z*(r_620_1 + r_620_2*riz) + rPmQy*r_630_2;
   double r_604_1 = PmAz*r_603_1 + 3*i2z*(r_602_1 + r_602_2*riz) + rPmQz*r_603_2;
   double r_460_1 = PmAx*r_360_1 + 3*i2z*(r_260_1 + r_260_2*riz) + rPmQx*r_360_2;
   double r_406_1 = PmAx*r_306_1 + 3*i2z*(r_206_1 + r_206_2*riz) + rPmQx*r_306_2;
   double r_064_1 = PmAz*r_063_1 + 3*i2z*(r_062_1 + r_062_2*riz) + rPmQz*r_063_2;
   double r_046_1 = PmAy*r_036_1 + 3*i2z*(r_026_1 + r_026_2*riz) + rPmQy*r_036_2;
   double r_550_1 = PmAy*r_540_1 + 4*i2z*(r_530_1 + r_530_2*riz) + rPmQy*r_540_2;
   double r_505_1 = PmAz*r_504_1 + 4*i2z*(r_503_1 + r_503_2*riz) + rPmQz*r_504_2;
   double r_055_1 = PmAz*r_054_1 + 4*i2z*(r_053_1 + r_053_2*riz) + rPmQz*r_054_2;
   double r_712_1 = PmAy*r_702_1 + rPmQy*r_702_2;
   double r_172_1 = PmAx*r_072_1 + rPmQx*r_072_2;
   double r_127_1 = PmAx*r_027_1 + rPmQx*r_027_2;
   double r_316_1 = PmAy*r_306_1 + rPmQy*r_306_2;
   double r_136_1 = PmAx*r_036_1 + rPmQx*r_036_2;
   double r_361_1 = PmAz*r_360_1 + rPmQz*r_360_2;
   double r_163_1 = PmAx*r_063_1 + rPmQx*r_063_2;
   double r_631_1 = PmAz*r_630_1 + rPmQz*r_630_2;
   double r_613_1 = PmAy*r_603_1 + rPmQy*r_603_2;
   double r_514_1 = PmAy*r_504_1 + rPmQy*r_504_2;
   double r_154_1 = PmAx*r_054_1 + rPmQx*r_054_2;
   double r_541_1 = PmAz*r_540_1 + rPmQz*r_540_2;
   double r_145_1 = PmAx*r_045_1 + rPmQx*r_045_2;
   double r_451_1 = PmAz*r_450_1 + rPmQz*r_450_2;
   double r_415_1 = PmAy*r_405_1 + rPmQy*r_405_2;
   double r_532_1 = PmAz*r_531_1 + i2z*(r_530_1 + r_530_2*riz) + rPmQz*r_531_2;
   double r_352_1 = PmAz*r_351_1 + i2z*(r_350_1 + r_350_2*riz) + rPmQz*r_351_2;
   double r_325_1 = PmAy*r_315_1 + i2z*(r_305_1 + r_305_2*riz) + rPmQy*r_315_2;
   double r_442_1 = PmAz*r_441_1 + i2z*(r_440_1 + r_440_2*riz) + rPmQz*r_441_2;
   double r_424_1 = PmAy*r_414_1 + i2z*(r_404_1 + r_404_2*riz) + rPmQy*r_414_2;
   double r_244_1 = PmAx*r_144_1 + i2z*(r_044_1 + r_044_2*riz) + rPmQx*r_144_2;
   pOut[286] = PmAx*pOut[220] + 10*i2z*(pOut[165] + r_900_1*riz) + rPmQx*r_a00_1;
   pOut[287] = PmAy*pOut[221] + 10*i2z*(pOut[166] + r_090_1*riz) + rPmQy*r_0a0_1;
   pOut[288] = PmAz*pOut[222] + 10*i2z*(pOut[167] + r_009_1*riz) + rPmQz*r_00a_1;
   pOut[289] = PmAx*pOut[221] + rPmQx*r_0a0_1;
   pOut[290] = PmAx*pOut[222] + rPmQx*r_00a_1;
   pOut[291] = PmAy*pOut[220] + rPmQy*r_a00_1;
   pOut[292] = PmAy*pOut[222] + rPmQy*r_00a_1;
   pOut[293] = PmAz*pOut[220] + rPmQz*r_a00_1;
   pOut[294] = PmAz*pOut[221] + rPmQz*r_0a0_1;
   pOut[295] = PmAy*pOut[223] + i2z*(pOut[165] + r_900_1*riz) + rPmQy*r_910_1;
   pOut[296] = PmAz*pOut[225] + i2z*(pOut[165] + r_900_1*riz) + rPmQz*r_901_1;
   pOut[297] = PmAx*pOut[224] + i2z*(pOut[166] + r_090_1*riz) + rPmQx*r_190_1;
   pOut[298] = PmAz*pOut[227] + i2z*(pOut[166] + r_090_1*riz) + rPmQz*r_091_1;
   pOut[299] = PmAx*pOut[226] + i2z*(pOut[167] + r_009_1*riz) + rPmQx*r_109_1;
   pOut[300] = PmAy*pOut[228] + i2z*(pOut[167] + r_009_1*riz) + rPmQy*r_019_1;
   pOut[301] = PmAx*pOut[231] + 2*i2z*(pOut[168] + r_180_1*riz) + rPmQx*r_280_1;
   pOut[302] = PmAx*pOut[232] + 2*i2z*(pOut[169] + r_108_1*riz) + rPmQx*r_208_1;
   pOut[303] = PmAy*pOut[229] + 2*i2z*(pOut[170] + r_810_1*riz) + rPmQy*r_820_1;
   pOut[304] = PmAy*pOut[234] + 2*i2z*(pOut[171] + r_018_1*riz) + rPmQy*r_028_1;
   pOut[305] = PmAz*pOut[230] + 2*i2z*(pOut[172] + r_801_1*riz) + rPmQz*r_802_1;
   pOut[306] = PmAz*pOut[233] + 2*i2z*(pOut[173] + r_081_1*riz) + rPmQz*r_082_1;
   pOut[307] = PmAy*pOut[235] + 3*i2z*(pOut[174] + r_720_1*riz) + rPmQy*r_730_1;
   pOut[308] = PmAz*pOut[237] + 3*i2z*(pOut[175] + r_702_1*riz) + rPmQz*r_703_1;
   pOut[309] = PmAx*pOut[236] + 3*i2z*(pOut[176] + r_270_1*riz) + rPmQx*r_370_1;
   pOut[310] = PmAz*pOut[239] + 3*i2z*(pOut[177] + r_072_1*riz) + rPmQz*r_073_1;
   pOut[311] = PmAx*pOut[238] + 3*i2z*(pOut[178] + r_207_1*riz) + rPmQx*r_307_1;
   pOut[312] = PmAy*pOut[240] + 3*i2z*(pOut[179] + r_027_1*riz) + rPmQy*r_037_1;
   pOut[313] = PmAx*pOut[243] + 4*i2z*(pOut[180] + r_360_1*riz) + rPmQx*r_460_1;
   pOut[314] = PmAx*pOut[244] + 4*i2z*(pOut[181] + r_306_1*riz) + rPmQx*r_406_1;
   pOut[315] = PmAy*pOut[241] + 4*i2z*(pOut[182] + r_630_1*riz) + rPmQy*r_640_1;
   pOut[316] = PmAy*pOut[246] + 4*i2z*(pOut[183] + r_036_1*riz) + rPmQy*r_046_1;
   pOut[317] = PmAz*pOut[242] + 4*i2z*(pOut[184] + r_603_1*riz) + rPmQz*r_604_1;
   pOut[318] = PmAz*pOut[245] + 4*i2z*(pOut[185] + r_063_1*riz) + rPmQz*r_064_1;
   pOut[319] = PmAz*pOut[223] + rPmQz*r_910_1;
   pOut[320] = PmAz*pOut[224] + rPmQz*r_190_1;
   pOut[321] = PmAy*pOut[226] + rPmQy*r_109_1;
   pOut[322] = PmAx*pOut[233] + rPmQx*r_082_1;
   pOut[323] = PmAx*pOut[234] + rPmQx*r_028_1;
   pOut[324] = PmAy*pOut[230] + rPmQy*r_802_1;
   pOut[325] = PmAy*pOut[232] + rPmQy*r_208_1;
   pOut[326] = PmAz*pOut[229] + rPmQz*r_820_1;
   pOut[327] = PmAz*pOut[231] + rPmQz*r_280_1;
   pOut[328] = PmAz*pOut[235] + rPmQz*r_730_1;
   pOut[329] = PmAy*pOut[237] + rPmQy*r_703_1;
   pOut[330] = PmAz*pOut[236] + rPmQz*r_370_1;
   pOut[331] = PmAy*pOut[238] + rPmQy*r_307_1;
   pOut[332] = PmAx*pOut[239] + rPmQx*r_073_1;
   pOut[333] = PmAx*pOut[240] + rPmQx*r_037_1;
   pOut[334] = PmAx*pOut[245] + rPmQx*r_064_1;
   pOut[335] = PmAx*pOut[246] + rPmQx*r_046_1;
   pOut[336] = PmAy*pOut[242] + rPmQy*r_604_1;
   pOut[337] = PmAy*pOut[244] + rPmQy*r_406_1;
   pOut[338] = PmAz*pOut[241] + rPmQz*r_640_1;
   pOut[339] = PmAz*pOut[243] + rPmQz*r_460_1;
   pOut[340] = PmAz*pOut[247] + rPmQz*r_550_1;
   pOut[341] = PmAy*pOut[248] + rPmQy*r_505_1;
   pOut[342] = PmAx*pOut[249] + rPmQx*r_055_1;
   pOut[343] = PmAy*pOut[253] + i2z*(pOut[175] + r_702_1*riz) + rPmQy*r_712_1;
   pOut[344] = PmAx*pOut[254] + i2z*(pOut[177] + r_072_1*riz) + rPmQx*r_172_1;
   pOut[345] = PmAx*pOut[256] + i2z*(pOut[179] + r_027_1*riz) + rPmQx*r_127_1;
   pOut[346] = PmAz*pOut[261] + i2z*(pOut[180] + r_360_1*riz) + rPmQz*r_361_1;
   pOut[347] = PmAy*pOut[259] + i2z*(pOut[181] + r_306_1*riz) + rPmQy*r_316_1;
   pOut[348] = PmAz*pOut[263] + i2z*(pOut[182] + r_630_1*riz) + rPmQz*r_631_1;
   pOut[349] = PmAx*pOut[260] + i2z*(pOut[183] + r_036_1*riz) + rPmQx*r_136_1;
   pOut[350] = PmAy*pOut[264] + i2z*(pOut[184] + r_603_1*riz) + rPmQy*r_613_1;
   pOut[351] = PmAx*pOut[262] + i2z*(pOut[185] + r_063_1*riz) + rPmQx*r_163_1;
   pOut[352] = PmAz*pOut[267] + i2z*(pOut[186] + r_540_1*riz) + rPmQz*r_541_1;
   pOut[353] = PmAy*pOut[265] + i2z*(pOut[187] + r_504_1*riz) + rPmQy*r_514_1;
   pOut[354] = PmAz*pOut[269] + i2z*(pOut[188] + r_450_1*riz) + rPmQz*r_451_1;
   pOut[355] = PmAx*pOut[266] + i2z*(pOut[189] + r_054_1*riz) + rPmQx*r_154_1;
   pOut[356] = PmAy*pOut[270] + i2z*(pOut[190] + r_405_1*riz) + rPmQy*r_415_1;
   pOut[357] = PmAx*pOut[268] + i2z*(pOut[191] + r_045_1*riz) + rPmQx*r_145_1;
   pOut[358] = PmAz*pOut[274] + 2*i2z*(pOut[201] + r_531_1*riz) + rPmQz*r_532_1;
   pOut[359] = PmAz*pOut[275] + 2*i2z*(pOut[203] + r_351_1*riz) + rPmQz*r_352_1;
   pOut[360] = PmAy*pOut[277] + 2*i2z*(pOut[204] + r_315_1*riz) + rPmQy*r_325_1;
   pOut[361] = PmAx*pOut[282] + 2*i2z*(pOut[207] + r_144_1*riz) + rPmQx*r_244_1;
   pOut[362] = PmAy*pOut[281] + 2*i2z*(pOut[208] + r_414_1*riz) + rPmQy*r_424_1;
   pOut[363] = PmAz*pOut[280] + 2*i2z*(pOut[209] + r_441_1*riz) + rPmQz*r_442_1;
   // 3.00 steps per component, 1.22 kb stack
   if (lab == 11) return;

   // m + lab == 12
   double r_100_b = PmAx*pIm[11] + pIm[12]*rPmQx;
   double r_010_b = PmAy*pIm[11] + pIm[12]*rPmQy;
   double r_001_b = PmAz*pIm[11] + pIm[12]*rPmQz;
   double r_200_a = PmAx*r_100_a + i2z*(pIm[10] + pIm[11]*riz) + rPmQx*r_100_b;
   double r_020_a = PmAy*r_010_a + i2z*(pIm[10] + pIm[11]*riz) + rPmQy*r_010_b;
   double r_002_a = PmAz*r_001_a + i2z*(pIm[10] + pIm[11]*riz) + rPmQz*r_001_b;
   double r_300_9 = PmAx*r_200_9 + 2*i2z*(r_100_9 + r_100_a*riz) + rPmQx*r_200_a;
   double r_030_9 = PmAy*r_020_9 + 2*i2z*(r_010_9 + r_010_a*riz) + rPmQy*r_020_a;
   double r_003_9 = PmAz*r_002_9 + 2*i2z*(r_001_9 + r_001_a*riz) + rPmQz*r_002_a;
   double r_400_8 = PmAx*r_300_8 + 3*i2z*(r_200_8 + r_200_9*riz) + rPmQx*r_300_9;
   double r_040_8 = PmAy*r_030_8 + 3*i2z*(r_020_8 + r_020_9*riz) + rPmQy*r_030_9;
   double r_004_8 = PmAz*r_003_8 + 3*i2z*(r_002_8 + r_002_9*riz) + rPmQz*r_003_9;
   double r_500_7 = PmAx*r_400_7 + 4*i2z*(r_300_7 + r_300_8*riz) + rPmQx*r_400_8;
   double r_050_7 = PmAy*r_040_7 + 4*i2z*(r_030_7 + r_030_8*riz) + rPmQy*r_040_8;
   double r_005_7 = PmAz*r_004_7 + 4*i2z*(r_003_7 + r_003_8*riz) + rPmQz*r_004_8;
   double r_014_7 = PmAy*r_004_7 + rPmQy*r_004_8;
   double r_600_6 = PmAx*r_500_6 + 5*i2z*(r_400_6 + r_400_7*riz) + rPmQx*r_500_7;
   double r_060_6 = PmAy*r_050_6 + 5*i2z*(r_040_6 + r_040_7*riz) + rPmQy*r_050_7;
   double r_006_6 = PmAz*r_005_6 + 5*i2z*(r_004_6 + r_004_7*riz) + rPmQz*r_005_7;
   double r_510_6 = PmAy*r_500_6 + rPmQy*r_500_7;
   double r_150_6 = PmAx*r_050_6 + rPmQx*r_050_7;
   double r_501_6 = PmAz*r_500_6 + rPmQz*r_500_7;
   double r_105_6 = PmAx*r_005_6 + rPmQx*r_005_7;
   double r_051_6 = PmAz*r_050_6 + rPmQz*r_050_7;
   double r_015_6 = PmAy*r_005_6 + rPmQy*r_005_7;
   double r_024_6 = PmAy*r_014_6 + i2z*(r_004_6 + r_004_7*riz) + rPmQy*r_014_7;
   double r_700_5 = PmAx*r_600_5 + 6*i2z*(r_500_5 + r_500_6*riz) + rPmQx*r_600_6;
   double r_070_5 = PmAy*r_060_5 + 6*i2z*(r_050_5 + r_050_6*riz) + rPmQy*r_060_6;
   double r_007_5 = PmAz*r_006_5 + 6*i2z*(r_005_5 + r_005_6*riz) + rPmQz*r_006_6;
   double r_160_5 = PmAx*r_060_5 + rPmQx*r_060_6;
   double r_106_5 = PmAx*r_006_5 + rPmQx*r_006_6;
   double r_610_5 = PmAy*r_600_5 + rPmQy*r_600_6;
   double r_016_5 = PmAy*r_006_5 + rPmQy*r_006_6;
   double r_601_5 = PmAz*r_600_5 + rPmQz*r_600_6;
   double r_061_5 = PmAz*r_060_5 + rPmQz*r_060_6;
   double r_520_5 = PmAy*r_510_5 + i2z*(r_500_5 + r_500_6*riz) + rPmQy*r_510_6;
   double r_502_5 = PmAz*r_501_5 + i2z*(r_500_5 + r_500_6*riz) + rPmQz*r_501_6;
   double r_250_5 = PmAx*r_150_5 + i2z*(r_050_5 + r_050_6*riz) + rPmQx*r_150_6;
   double r_052_5 = PmAz*r_051_5 + i2z*(r_050_5 + r_050_6*riz) + rPmQz*r_051_6;
   double r_205_5 = PmAx*r_105_5 + i2z*(r_005_5 + r_005_6*riz) + rPmQx*r_105_6;
   double r_025_5 = PmAy*r_015_5 + i2z*(r_005_5 + r_005_6*riz) + rPmQy*r_015_6;
   double r_034_5 = PmAy*r_024_5 + 2*i2z*(r_014_5 + r_014_6*riz) + rPmQy*r_024_6;
   double r_800_4 = PmAx*r_700_4 + 7*i2z*(r_600_4 + r_600_5*riz) + rPmQx*r_700_5;
   double r_080_4 = PmAy*r_070_4 + 7*i2z*(r_060_4 + r_060_5*riz) + rPmQy*r_070_5;
   double r_008_4 = PmAz*r_007_4 + 7*i2z*(r_006_4 + r_006_5*riz) + rPmQz*r_007_5;
   double r_710_4 = PmAy*r_700_4 + rPmQy*r_700_5;
   double r_170_4 = PmAx*r_070_4 + rPmQx*r_070_5;
   double r_701_4 = PmAz*r_700_4 + rPmQz*r_700_5;
   double r_107_4 = PmAx*r_007_4 + rPmQx*r_007_5;
   double r_071_4 = PmAz*r_070_4 + rPmQz*r_070_5;
   double r_017_4 = PmAy*r_007_4 + rPmQy*r_007_5;
   double r_620_4 = PmAy*r_610_4 + i2z*(r_600_4 + r_600_5*riz) + rPmQy*r_610_5;
   double r_602_4 = PmAz*r_601_4 + i2z*(r_600_4 + r_600_5*riz) + rPmQz*r_601_5;
   double r_260_4 = PmAx*r_160_4 + i2z*(r_060_4 + r_060_5*riz) + rPmQx*r_160_5;
   double r_206_4 = PmAx*r_106_4 + i2z*(r_006_4 + r_006_5*riz) + rPmQx*r_106_5;
   double r_062_4 = PmAz*r_061_4 + i2z*(r_060_4 + r_060_5*riz) + rPmQz*r_061_5;
   double r_026_4 = PmAy*r_016_4 + i2z*(r_006_4 + r_006_5*riz) + rPmQy*r_016_5;
   double r_530_4 = PmAy*r_520_4 + 2*i2z*(r_510_4 + r_510_5*riz) + rPmQy*r_520_5;
   double r_350_4 = PmAx*r_250_4 + 2*i2z*(r_150_4 + r_150_5*riz) + rPmQx*r_250_5;
   double r_503_4 = PmAz*r_502_4 + 2*i2z*(r_501_4 + r_501_5*riz) + rPmQz*r_502_5;
   double r_305_4 = PmAx*r_205_4 + 2*i2z*(r_105_4 + r_105_5*riz) + rPmQx*r_205_5;
   double r_053_4 = PmAz*r_052_4 + 2*i2z*(r_051_4 + r_051_5*riz) + rPmQz*r_052_5;
   double r_035_4 = PmAy*r_025_4 + 2*i2z*(r_015_4 + r_015_5*riz) + rPmQy*r_025_5;
   double r_044_4 = PmAy*r_034_4 + 3*i2z*(r_024_4 + r_024_5*riz) + rPmQy*r_034_5;
   double r_900_3 = PmAx*r_800_3 + 8*i2z*(r_700_3 + r_700_4*riz) + rPmQx*r_800_4;
   double r_090_3 = PmAy*r_080_3 + 8*i2z*(r_070_3 + r_070_4*riz) + rPmQy*r_080_4;
   double r_009_3 = PmAz*r_008_3 + 8*i2z*(r_007_3 + r_007_4*riz) + rPmQz*r_008_4;
   double r_180_3 = PmAx*r_080_3 + rPmQx*r_080_4;
   double r_108_3 = PmAx*r_008_3 + rPmQx*r_008_4;
   double r_810_3 = PmAy*r_800_3 + rPmQy*r_800_4;
   double r_018_3 = PmAy*r_008_3 + rPmQy*r_008_4;
   double r_801_3 = PmAz*r_800_3 + rPmQz*r_800_4;
   double r_081_3 = PmAz*r_080_3 + rPmQz*r_080_4;
   double r_720_3 = PmAy*r_710_3 + i2z*(r_700_3 + r_700_4*riz) + rPmQy*r_710_4;
   double r_702_3 = PmAz*r_701_3 + i2z*(r_700_3 + r_700_4*riz) + rPmQz*r_701_4;
   double r_270_3 = PmAx*r_170_3 + i2z*(r_070_3 + r_070_4*riz) + rPmQx*r_170_4;
   double r_072_3 = PmAz*r_071_3 + i2z*(r_070_3 + r_070_4*riz) + rPmQz*r_071_4;
   double r_207_3 = PmAx*r_107_3 + i2z*(r_007_3 + r_007_4*riz) + rPmQx*r_107_4;
   double r_027_3 = PmAy*r_017_3 + i2z*(r_007_3 + r_007_4*riz) + rPmQy*r_017_4;
   double r_360_3 = PmAx*r_260_3 + 2*i2z*(r_160_3 + r_160_4*riz) + rPmQx*r_260_4;
   double r_306_3 = PmAx*r_206_3 + 2*i2z*(r_106_3 + r_106_4*riz) + rPmQx*r_206_4;
   double r_630_3 = PmAy*r_620_3 + 2*i2z*(r_610_3 + r_610_4*riz) + rPmQy*r_620_4;
   double r_036_3 = PmAy*r_026_3 + 2*i2z*(r_016_3 + r_016_4*riz) + rPmQy*r_026_4;
   double r_603_3 = PmAz*r_602_3 + 2*i2z*(r_601_3 + r_601_4*riz) + rPmQz*r_602_4;
   double r_063_3 = PmAz*r_062_3 + 2*i2z*(r_061_3 + r_061_4*riz) + rPmQz*r_062_4;
   double r_540_3 = PmAy*r_530_3 + 3*i2z*(r_520_3 + r_520_4*riz) + rPmQy*r_530_4;
   double r_504_3 = PmAz*r_503_3 + 3*i2z*(r_502_3 + r_502_4*riz) + rPmQz*r_503_4;
   double r_450_3 = PmAx*r_350_3 + 3*i2z*(r_250_3 + r_250_4*riz) + rPmQx*r_350_4;
   double r_054_3 = PmAz*r_053_3 + 3*i2z*(r_052_3 + r_052_4*riz) + rPmQz*r_053_4;
   double r_405_3 = PmAx*r_305_3 + 3*i2z*(r_205_3 + r_205_4*riz) + rPmQx*r_305_4;
   double r_045_3 = PmAy*r_035_3 + 3*i2z*(r_025_3 + r_025_4*riz) + rPmQy*r_035_4;
   double r_144_3 = PmAx*r_044_3 + rPmQx*r_044_4;
   double r_a00_2 = PmAx*r_900_2 + 9*i2z*(r_800_2 + r_800_3*riz) + rPmQx*r_900_3;
   double r_0a0_2 = PmAy*r_090_2 + 9*i2z*(r_080_2 + r_080_3*riz) + rPmQy*r_090_3;
   double r_00a_2 = PmAz*r_009_2 + 9*i2z*(r_008_2 + r_008_3*riz) + rPmQz*r_009_3;
   double r_910_2 = PmAy*r_900_2 + rPmQy*r_900_3;
   double r_190_2 = PmAx*r_090_2 + rPmQx*r_090_3;
   double r_901_2 = PmAz*r_900_2 + rPmQz*r_900_3;
   double r_109_2 = PmAx*r_009_2 + rPmQx*r_009_3;
   double r_091_2 = PmAz*r_090_2 + rPmQz*r_090_3;
   double r_019_2 = PmAy*r_009_2 + rPmQy*r_009_3;
   double r_820_2 = PmAy*r_810_2 + i2z*(r_800_2 + r_800_3*riz) + rPmQy*r_810_3;
   double r_802_2 = PmAz*r_801_2 + i2z*(r_800_2 + r_800_3*riz) + rPmQz*r_801_3;
   double r_280_2 = PmAx*r_180_2 + i2z*(r_080_2 + r_080_3*riz) + rPmQx*r_180_3;
   double r_208_2 = PmAx*r_108_2 + i2z*(r_008_2 + r_008_3*riz) + rPmQx*r_108_3;
   double r_082_2 = PmAz*r_081_2 + i2z*(r_080_2 + r_080_3*riz) + rPmQz*r_081_3;
   double r_028_2 = PmAy*r_018_2 + i2z*(r_008_2 + r_008_3*riz) + rPmQy*r_018_3;
   double r_730_2 = PmAy*r_720_2 + 2*i2z*(r_710_2 + r_710_3*riz) + rPmQy*r_720_3;
   double r_370_2 = PmAx*r_270_2 + 2*i2z*(r_170_2 + r_170_3*riz) + rPmQx*r_270_3;
   double r_703_2 = PmAz*r_702_2 + 2*i2z*(r_701_2 + r_701_3*riz) + rPmQz*r_702_3;
   double r_307_2 = PmAx*r_207_2 + 2*i2z*(r_107_2 + r_107_3*riz) + rPmQx*r_207_3;
   double r_073_2 = PmAz*r_072_2 + 2*i2z*(r_071_2 + r_071_3*riz) + rPmQz*r_072_3;
   double r_037_2 = PmAy*r_027_2 + 2*i2z*(r_017_2 + r_017_3*riz) + rPmQy*r_027_3;
   double r_640_2 = PmAy*r_630_2 + 3*i2z*(r_620_2 + r_620_3*riz) + rPmQy*r_630_3;
   double r_604_2 = PmAz*r_603_2 + 3*i2z*(r_602_2 + r_602_3*riz) + rPmQz*r_603_3;
   double r_460_2 = PmAx*r_360_2 + 3*i2z*(r_260_2 + r_260_3*riz) + rPmQx*r_360_3;
   double r_406_2 = PmAx*r_306_2 + 3*i2z*(r_206_2 + r_206_3*riz) + rPmQx*r_306_3;
   double r_064_2 = PmAz*r_063_2 + 3*i2z*(r_062_2 + r_062_3*riz) + rPmQz*r_063_3;
   double r_046_2 = PmAy*r_036_2 + 3*i2z*(r_026_2 + r_026_3*riz) + rPmQy*r_036_3;
   double r_550_2 = PmAy*r_540_2 + 4*i2z*(r_530_2 + r_530_3*riz) + rPmQy*r_540_3;
   double r_505_2 = PmAz*r_504_2 + 4*i2z*(r_503_2 + r_503_3*riz) + rPmQz*r_504_3;
   double r_055_2 = PmAz*r_054_2 + 4*i2z*(r_053_2 + r_053_3*riz) + rPmQz*r_054_3;
   double r_316_2 = PmAy*r_306_2 + rPmQy*r_306_3;
   double r_361_2 = PmAz*r_360_2 + rPmQz*r_360_3;
   double r_631_2 = PmAz*r_630_2 + rPmQz*r_630_3;
   double r_514_2 = PmAy*r_504_2 + rPmQy*r_504_3;
   double r_154_2 = PmAx*r_054_2 + rPmQx*r_054_3;
   double r_541_2 = PmAz*r_540_2 + rPmQz*r_540_3;
   double r_145_2 = PmAx*r_045_2 + rPmQx*r_045_3;
   double r_451_2 = PmAz*r_450_2 + rPmQz*r_450_3;
   double r_415_2 = PmAy*r_405_2 + rPmQy*r_405_3;
   double r_244_2 = PmAx*r_144_2 + i2z*(r_044_2 + r_044_3*riz) + rPmQx*r_144_3;
   double r_b00_1 = PmAx*r_a00_1 + 10*i2z*(r_900_1 + r_900_2*riz) + rPmQx*r_a00_2;
   double r_0b0_1 = PmAy*r_0a0_1 + 10*i2z*(r_090_1 + r_090_2*riz) + rPmQy*r_0a0_2;
   double r_00b_1 = PmAz*r_00a_1 + 10*i2z*(r_009_1 + r_009_2*riz) + rPmQz*r_00a_2;
   double r_1a0_1 = PmAx*r_0a0_1 + rPmQx*r_0a0_2;
   double r_10a_1 = PmAx*r_00a_1 + rPmQx*r_00a_2;
   double r_a10_1 = PmAy*r_a00_1 + rPmQy*r_a00_2;
   double r_01a_1 = PmAy*r_00a_1 + rPmQy*r_00a_2;
   double r_a01_1 = PmAz*r_a00_1 + rPmQz*r_a00_2;
   double r_0a1_1 = PmAz*r_0a0_1 + rPmQz*r_0a0_2;
   double r_920_1 = PmAy*r_910_1 + i2z*(r_900_1 + r_900_2*riz) + rPmQy*r_910_2;
   double r_902_1 = PmAz*r_901_1 + i2z*(r_900_1 + r_900_2*riz) + rPmQz*r_901_2;
   double r_290_1 = PmAx*r_190_1 + i2z*(r_090_1 + r_090_2*riz) + rPmQx*r_190_2;
   double r_092_1 = PmAz*r_091_1 + i2z*(r_090_1 + r_090_2*riz) + rPmQz*r_091_2;
   double r_209_1 = PmAx*r_109_1 + i2z*(r_009_1 + r_009_2*riz) + rPmQx*r_109_2;
   double r_029_1 = PmAy*r_019_1 + i2z*(r_009_1 + r_009_2*riz) + rPmQy*r_019_2;
   double r_380_1 = PmAx*r_280_1 + 2*i2z*(r_180_1 + r_180_2*riz) + rPmQx*r_280_2;
   double r_308_1 = PmAx*r_208_1 + 2*i2z*(r_108_1 + r_108_2*riz) + rPmQx*r_208_2;
   double r_830_1 = PmAy*r_820_1 + 2*i2z*(r_810_1 + r_810_2*riz) + rPmQy*r_820_2;
   double r_038_1 = PmAy*r_028_1 + 2*i2z*(r_018_1 + r_018_2*riz) + rPmQy*r_028_2;
   double r_803_1 = PmAz*r_802_1 + 2*i2z*(r_801_1 + r_801_2*riz) + rPmQz*r_802_2;
   double r_083_1 = PmAz*r_082_1 + 2*i2z*(r_081_1 + r_081_2*riz) + rPmQz*r_082_2;
   double r_740_1 = PmAy*r_730_1 + 3*i2z*(r_720_1 + r_720_2*riz) + rPmQy*r_730_2;
   double r_704_1 = PmAz*r_703_1 + 3*i2z*(r_702_1 + r_702_2*riz) + rPmQz*r_703_2;
   double r_470_1 = PmAx*r_370_1 + 3*i2z*(r_270_1 + r_270_2*riz) + rPmQx*r_370_2;
   double r_074_1 = PmAz*r_073_1 + 3*i2z*(r_072_1 + r_072_2*riz) + rPmQz*r_073_2;
   double r_407_1 = PmAx*r_307_1 + 3*i2z*(r_207_1 + r_207_2*riz) + rPmQx*r_307_2;
   double r_047_1 = PmAy*r_037_1 + 3*i2z*(r_027_1 + r_027_2*riz) + rPmQy*r_037_2;
   double r_560_1 = PmAx*r_460_1 + 4*i2z*(r_360_1 + r_360_2*riz) + rPmQx*r_460_2;
   double r_506_1 = PmAx*r_406_1 + 4*i2z*(r_306_1 + r_306_2*riz) + rPmQx*r_406_2;
   double r_650_1 = PmAy*r_640_1 + 4*i2z*(r_630_1 + r_630_2*riz) + rPmQy*r_640_2;
   double r_056_1 = PmAy*r_046_1 + 4*i2z*(r_036_1 + r_036_2*riz) + rPmQy*r_046_2;
   double r_605_1 = PmAz*r_604_1 + 4*i2z*(r_603_1 + r_603_2*riz) + rPmQz*r_604_2;
   double r_065_1 = PmAz*r_064_1 + 4*i2z*(r_063_1 + r_063_2*riz) + rPmQz*r_064_2;
   double r_182_1 = PmAx*r_082_1 + rPmQx*r_082_2;
   double r_128_1 = PmAx*r_028_1 + rPmQx*r_028_2;
   double r_812_1 = PmAy*r_802_1 + rPmQy*r_802_2;
   double r_731_1 = PmAz*r_730_1 + rPmQz*r_730_2;
   double r_713_1 = PmAy*r_703_1 + rPmQy*r_703_2;
   double r_371_1 = PmAz*r_370_1 + rPmQz*r_370_2;
   double r_317_1 = PmAy*r_307_1 + rPmQy*r_307_2;
   double r_173_1 = PmAx*r_073_1 + rPmQx*r_073_2;
   double r_137_1 = PmAx*r_037_1 + rPmQx*r_037_2;
   double r_164_1 = PmAx*r_064_1 + rPmQx*r_064_2;
   double r_146_1 = PmAx*r_046_1 + rPmQx*r_046_2;
   double r_614_1 = PmAy*r_604_1 + rPmQy*r_604_2;
   double r_416_1 = PmAy*r_406_1 + rPmQy*r_406_2;
   double r_641_1 = PmAz*r_640_1 + rPmQz*r_640_2;
   double r_461_1 = PmAz*r_460_1 + rPmQz*r_460_2;
   double r_551_1 = PmAz*r_550_1 + rPmQz*r_550_2;
   double r_515_1 = PmAy*r_505_1 + rPmQy*r_505_2;
   double r_155_1 = PmAx*r_055_1 + rPmQx*r_055_2;
   double r_362_1 = PmAz*r_361_1 + i2z*(r_360_1 + r_360_2*riz) + rPmQz*r_361_2;
   double r_326_1 = PmAy*r_316_1 + i2z*(r_306_1 + r_306_2*riz) + rPmQy*r_316_2;
   double r_632_1 = PmAz*r_631_1 + i2z*(r_630_1 + r_630_2*riz) + rPmQz*r_631_2;
   double r_542_1 = PmAz*r_541_1 + i2z*(r_540_1 + r_540_2*riz) + rPmQz*r_541_2;
   double r_524_1 = PmAy*r_514_1 + i2z*(r_504_1 + r_504_2*riz) + rPmQy*r_514_2;
   double r_452_1 = PmAz*r_451_1 + i2z*(r_450_1 + r_450_2*riz) + rPmQz*r_451_2;
   double r_254_1 = PmAx*r_154_1 + i2z*(r_054_1 + r_054_2*riz) + rPmQx*r_154_2;
   double r_425_1 = PmAy*r_415_1 + i2z*(r_405_1 + r_405_2*riz) + rPmQy*r_415_2;
   double r_245_1 = PmAx*r_145_1 + i2z*(r_045_1 + r_045_2*riz) + rPmQx*r_145_2;
   double r_344_1 = PmAx*r_244_1 + 2*i2z*(r_144_1 + r_144_2*riz) + rPmQx*r_244_2;
   pOut[364] = PmAx*pOut[286] + 11*i2z*(pOut[220] + r_a00_1*riz) + rPmQx*r_b00_1;
   pOut[365] = PmAy*pOut[287] + 11*i2z*(pOut[221] + r_0a0_1*riz) + rPmQy*r_0b0_1;
   pOut[366] = PmAz*pOut[288] + 11*i2z*(pOut[222] + r_00a_1*riz) + rPmQz*r_00b_1;
   pOut[367] = PmAy*pOut[286] + rPmQy*r_b00_1;
   pOut[368] = PmAx*pOut[287] + rPmQx*r_0b0_1;
   pOut[369] = PmAz*pOut[286] + rPmQz*r_b00_1;
   pOut[370] = PmAx*pOut[288] + rPmQx*r_00b_1;
   pOut[371] = PmAz*pOut[287] + rPmQz*r_0b0_1;
   pOut[372] = PmAy*pOut[288] + rPmQy*r_00b_1;
   pOut[373] = PmAy*pOut[291] + i2z*(pOut[220] + r_a00_1*riz) + rPmQy*r_a10_1;
   pOut[374] = PmAz*pOut[293] + i2z*(pOut[220] + r_a00_1*riz) + rPmQz*r_a01_1;
   pOut[375] = PmAx*pOut[289] + i2z*(pOut[221] + r_0a0_1*riz) + rPmQx*r_1a0_1;
   pOut[376] = PmAx*pOut[290] + i2z*(pOut[222] + r_00a_1*riz) + rPmQx*r_10a_1;
   pOut[377] = PmAz*pOut[294] + i2z*(pOut[221] + r_0a0_1*riz) + rPmQz*r_0a1_1;
   pOut[378] = PmAy*pOut[292] + i2z*(pOut[222] + r_00a_1*riz) + rPmQy*r_01a_1;
   pOut[379] = PmAy*pOut[295] + 2*i2z*(pOut[223] + r_910_1*riz) + rPmQy*r_920_1;
   pOut[380] = PmAx*pOut[297] + 2*i2z*(pOut[224] + r_190_1*riz) + rPmQx*r_290_1;
   pOut[381] = PmAz*pOut[296] + 2*i2z*(pOut[225] + r_901_1*riz) + rPmQz*r_902_1;
   pOut[382] = PmAx*pOut[299] + 2*i2z*(pOut[226] + r_109_1*riz) + rPmQx*r_209_1;
   pOut[383] = PmAz*pOut[298] + 2*i2z*(pOut[227] + r_091_1*riz) + rPmQz*r_092_1;
   pOut[384] = PmAy*pOut[300] + 2*i2z*(pOut[228] + r_019_1*riz) + rPmQy*r_029_1;
   pOut[385] = PmAy*pOut[303] + 3*i2z*(pOut[229] + r_820_1*riz) + rPmQy*r_830_1;
   pOut[386] = PmAz*pOut[305] + 3*i2z*(pOut[230] + r_802_1*riz) + rPmQz*r_803_1;
   pOut[387] = PmAx*pOut[301] + 3*i2z*(pOut[231] + r_280_1*riz) + rPmQx*r_380_1;
   pOut[388] = PmAx*pOut[302] + 3*i2z*(pOut[232] + r_208_1*riz) + rPmQx*r_308_1;
   pOut[389] = PmAz*pOut[306] + 3*i2z*(pOut[233] + r_082_1*riz) + rPmQz*r_083_1;
   pOut[390] = PmAy*pOut[304] + 3*i2z*(pOut[234] + r_028_1*riz) + rPmQy*r_038_1;
   pOut[391] = PmAy*pOut[307] + 4*i2z*(pOut[235] + r_730_1*riz) + rPmQy*r_740_1;
   pOut[392] = PmAx*pOut[309] + 4*i2z*(pOut[236] + r_370_1*riz) + rPmQx*r_470_1;
   pOut[393] = PmAz*pOut[308] + 4*i2z*(pOut[237] + r_703_1*riz) + rPmQz*r_704_1;
   pOut[394] = PmAx*pOut[311] + 4*i2z*(pOut[238] + r_307_1*riz) + rPmQx*r_407_1;
   pOut[395] = PmAz*pOut[310] + 4*i2z*(pOut[239] + r_073_1*riz) + rPmQz*r_074_1;
   pOut[396] = PmAy*pOut[312] + 4*i2z*(pOut[240] + r_037_1*riz) + rPmQy*r_047_1;
   pOut[397] = PmAx*pOut[313] + 5*i2z*(pOut[243] + r_460_1*riz) + rPmQx*r_560_1;
   pOut[398] = PmAx*pOut[314] + 5*i2z*(pOut[244] + r_406_1*riz) + rPmQx*r_506_1;
   pOut[399] = PmAy*pOut[316] + 5*i2z*(pOut[246] + r_046_1*riz) + rPmQy*r_056_1;
   pOut[400] = PmAy*pOut[290] + rPmQy*r_10a_1;
   pOut[401] = PmAz*pOut[289] + rPmQz*r_1a0_1;
   pOut[402] = PmAz*pOut[291] + rPmQz*r_a10_1;
   pOut[403] = PmAy*pOut[296] + rPmQy*r_902_1;
   pOut[404] = PmAx*pOut[298] + rPmQx*r_092_1;
   pOut[405] = PmAz*pOut[295] + rPmQz*r_920_1;
   pOut[406] = PmAx*pOut[300] + rPmQx*r_029_1;
   pOut[407] = PmAz*pOut[297] + rPmQz*r_290_1;
   pOut[408] = PmAy*pOut[299] + rPmQy*r_209_1;
   pOut[409] = PmAy*pOut[302] + rPmQy*r_308_1;
   pOut[410] = PmAx*pOut[304] + rPmQx*r_038_1;
   pOut[411] = PmAz*pOut[301] + rPmQz*r_380_1;
   pOut[412] = PmAx*pOut[306] + rPmQx*r_083_1;
   pOut[413] = PmAz*pOut[303] + rPmQz*r_830_1;
   pOut[414] = PmAy*pOut[305] + rPmQy*r_803_1;
   pOut[415] = PmAy*pOut[308] + rPmQy*r_704_1;
   pOut[416] = PmAx*pOut[310] + rPmQx*r_074_1;
   pOut[417] = PmAz*pOut[307] + rPmQz*r_740_1;
   pOut[418] = PmAx*pOut[312] + rPmQx*r_047_1;
   pOut[419] = PmAz*pOut[309] + rPmQz*r_470_1;
   pOut[420] = PmAy*pOut[311] + rPmQy*r_407_1;
   pOut[421] = PmAy*pOut[314] + rPmQy*r_506_1;
   pOut[422] = PmAx*pOut[316] + rPmQx*r_056_1;
   pOut[423] = PmAz*pOut[313] + rPmQz*r_560_1;
   pOut[424] = PmAx*pOut[318] + rPmQx*r_065_1;
   pOut[425] = PmAz*pOut[315] + rPmQz*r_650_1;
   pOut[426] = PmAy*pOut[317] + rPmQy*r_605_1;
   pOut[427] = PmAy*pOut[324] + i2z*(pOut[230] + r_802_1*riz) + rPmQy*r_812_1;
   pOut[428] = PmAx*pOut[322] + i2z*(pOut[233] + r_082_1*riz) + rPmQx*r_182_1;
   pOut[429] = PmAx*pOut[323] + i2z*(pOut[234] + r_028_1*riz) + rPmQx*r_128_1;
   pOut[430] = PmAz*pOut[328] + i2z*(pOut[235] + r_730_1*riz) + rPmQz*r_731_1;
   pOut[431] = PmAz*pOut[330] + i2z*(pOut[236] + r_370_1*riz) + rPmQz*r_371_1;
   pOut[432] = PmAy*pOut[329] + i2z*(pOut[237] + r_703_1*riz) + rPmQy*r_713_1;
   pOut[433] = PmAy*pOut[331] + i2z*(pOut[238] + r_307_1*riz) + rPmQy*r_317_1;
   pOut[434] = PmAx*pOut[332] + i2z*(pOut[239] + r_073_1*riz) + rPmQx*r_173_1;
   pOut[435] = PmAx*pOut[333] + i2z*(pOut[240] + r_037_1*riz) + rPmQx*r_137_1;
   pOut[436] = PmAz*pOut[338] + i2z*(pOut[241] + r_640_1*riz) + rPmQz*r_641_1;
   pOut[437] = PmAy*pOut[336] + i2z*(pOut[242] + r_604_1*riz) + rPmQy*r_614_1;
   pOut[438] = PmAz*pOut[339] + i2z*(pOut[243] + r_460_1*riz) + rPmQz*r_461_1;
   pOut[439] = PmAy*pOut[337] + i2z*(pOut[244] + r_406_1*riz) + rPmQy*r_416_1;
   pOut[440] = PmAx*pOut[334] + i2z*(pOut[245] + r_064_1*riz) + rPmQx*r_164_1;
   pOut[441] = PmAx*pOut[335] + i2z*(pOut[246] + r_046_1*riz) + rPmQx*r_146_1;
   pOut[442] = PmAz*pOut[340] + i2z*(pOut[247] + r_550_1*riz) + rPmQz*r_551_1;
   pOut[443] = PmAy*pOut[341] + i2z*(pOut[248] + r_505_1*riz) + rPmQy*r_515_1;
   pOut[444] = PmAx*pOut[342] + i2z*(pOut[249] + r_055_1*riz) + rPmQx*r_155_1;
   pOut[445] = PmAy*pOut[347] + 2*i2z*(pOut[259] + r_316_1*riz) + rPmQy*r_326_1;
   pOut[446] = PmAz*pOut[346] + 2*i2z*(pOut[261] + r_361_1*riz) + rPmQz*r_362_1;
   pOut[447] = PmAz*pOut[348] + 2*i2z*(pOut[263] + r_631_1*riz) + rPmQz*r_632_1;
   pOut[448] = PmAy*pOut[353] + 2*i2z*(pOut[265] + r_514_1*riz) + rPmQy*r_524_1;
   pOut[449] = PmAx*pOut[355] + 2*i2z*(pOut[266] + r_154_1*riz) + rPmQx*r_254_1;
   pOut[450] = PmAz*pOut[352] + 2*i2z*(pOut[267] + r_541_1*riz) + rPmQz*r_542_1;
   pOut[451] = PmAx*pOut[357] + 2*i2z*(pOut[268] + r_145_1*riz) + rPmQx*r_245_1;
   pOut[452] = PmAz*pOut[354] + 2*i2z*(pOut[269] + r_451_1*riz) + rPmQz*r_452_1;
   pOut[453] = PmAy*pOut[356] + 2*i2z*(pOut[270] + r_415_1*riz) + rPmQy*r_425_1;
   pOut[454] = PmAx*pOut[361] + 3*i2z*(pOut[282] + r_244_1*riz) + rPmQx*r_344_1;
   // 3.12 steps per component, 1.51 kb stack
   if (lab == 12) return;

   // m + lab == 13
   double r_100_c = PmAx*pIm[12] + pIm[13]*rPmQx;
   double r_010_c = PmAy*pIm[12] + pIm[13]*rPmQy;
   double r_001_c = PmAz*pIm[12] + pIm[13]*rPmQz;
   double r_200_b = PmAx*r_100_b + i2z*(pIm[11] + pIm[12]*riz) + rPmQx*r_100_c;
   double r_020_b = PmAy*r_010_b + i2z*(pIm[11] + pIm[12]*riz) + rPmQy*r_010_c;
   double r_002_b = PmAz*r_001_b + i2z*(pIm[11] + pIm[12]*riz) + rPmQz*r_001_c;
   double r_300_a = PmAx*r_200_a + 2*i2z*(r_100_a + r_100_b*riz) + rPmQx*r_200_b;
   double r_030_a = PmAy*r_020_a + 2*i2z*(r_010_a + r_010_b*riz) + rPmQy*r_020_b;
   double r_003_a = PmAz*r_002_a + 2*i2z*(r_001_a + r_001_b*riz) + rPmQz*r_002_b;
   double r_400_9 = PmAx*r_300_9 + 3*i2z*(r_200_9 + r_200_a*riz) + rPmQx*r_300_a;
   double r_040_9 = PmAy*r_030_9 + 3*i2z*(r_020_9 + r_020_a*riz) + rPmQy*r_030_a;
   double r_004_9 = PmAz*r_003_9 + 3*i2z*(r_002_9 + r_002_a*riz) + rPmQz*r_003_a;
   double r_500_8 = PmAx*r_400_8 + 4*i2z*(r_300_8 + r_300_9*riz) + rPmQx*r_400_9;
   double r_050_8 = PmAy*r_040_8 + 4*i2z*(r_030_8 + r_030_9*riz) + rPmQy*r_040_9;
   double r_005_8 = PmAz*r_004_8 + 4*i2z*(r_003_8 + r_003_9*riz) + rPmQz*r_004_9;
   double r_600_7 = PmAx*r_500_7 + 5*i2z*(r_400_7 + r_400_8*riz) + rPmQx*r_500_8;
   double r_060_7 = PmAy*r_050_7 + 5*i2z*(r_040_7 + r_040_8*riz) + rPmQy*r_050_8;
   double r_006_7 = PmAz*r_005_7 + 5*i2z*(r_004_7 + r_004_8*riz) + rPmQz*r_005_8;
   double r_510_7 = PmAy*r_500_7 + rPmQy*r_500_8;
   double r_501_7 = PmAz*r_500_7 + rPmQz*r_500_8;
   double r_051_7 = PmAz*r_050_7 + rPmQz*r_050_8;
   double r_015_7 = PmAy*r_005_7 + rPmQy*r_005_8;
   double r_700_6 = PmAx*r_600_6 + 6*i2z*(r_500_6 + r_500_7*riz) + rPmQx*r_600_7;
   double r_070_6 = PmAy*r_060_6 + 6*i2z*(r_050_6 + r_050_7*riz) + rPmQy*r_060_7;
   double r_007_6 = PmAz*r_006_6 + 6*i2z*(r_005_6 + r_005_7*riz) + rPmQz*r_006_7;
   double r_160_6 = PmAx*r_060_6 + rPmQx*r_060_7;
   double r_106_6 = PmAx*r_006_6 + rPmQx*r_006_7;
   double r_610_6 = PmAy*r_600_6 + rPmQy*r_600_7;
   double r_016_6 = PmAy*r_006_6 + rPmQy*r_006_7;
   double r_601_6 = PmAz*r_600_6 + rPmQz*r_600_7;
   double r_061_6 = PmAz*r_060_6 + rPmQz*r_060_7;
   double r_520_6 = PmAy*r_510_6 + i2z*(r_500_6 + r_500_7*riz) + rPmQy*r_510_7;
   double r_502_6 = PmAz*r_501_6 + i2z*(r_500_6 + r_500_7*riz) + rPmQz*r_501_7;
   double r_052_6 = PmAz*r_051_6 + i2z*(r_050_6 + r_050_7*riz) + rPmQz*r_051_7;
   double r_025_6 = PmAy*r_015_6 + i2z*(r_005_6 + r_005_7*riz) + rPmQy*r_015_7;
   double r_800_5 = PmAx*r_700_5 + 7*i2z*(r_600_5 + r_600_6*riz) + rPmQx*r_700_6;
   double r_080_5 = PmAy*r_070_5 + 7*i2z*(r_060_5 + r_060_6*riz) + rPmQy*r_070_6;
   double r_008_5 = PmAz*r_007_5 + 7*i2z*(r_006_5 + r_006_6*riz) + rPmQz*r_007_6;
   double r_710_5 = PmAy*r_700_5 + rPmQy*r_700_6;
   double r_170_5 = PmAx*r_070_5 + rPmQx*r_070_6;
   double r_701_5 = PmAz*r_700_5 + rPmQz*r_700_6;
   double r_107_5 = PmAx*r_007_5 + rPmQx*r_007_6;
   double r_071_5 = PmAz*r_070_5 + rPmQz*r_070_6;
   double r_017_5 = PmAy*r_007_5 + rPmQy*r_007_6;
   double r_620_5 = PmAy*r_610_5 + i2z*(r_600_5 + r_600_6*riz) + rPmQy*r_610_6;
   double r_602_5 = PmAz*r_601_5 + i2z*(r_600_5 + r_600_6*riz) + rPmQz*r_601_6;
   double r_260_5 = PmAx*r_160_5 + i2z*(r_060_5 + r_060_6*riz) + rPmQx*r_160_6;
   double r_206_5 = PmAx*r_106_5 + i2z*(r_006_5 + r_006_6*riz) + rPmQx*r_106_6;
   double r_062_5 = PmAz*r_061_5 + i2z*(r_060_5 + r_060_6*riz) + rPmQz*r_061_6;
   double r_026_5 = PmAy*r_016_5 + i2z*(r_006_5 + r_006_6*riz) + rPmQy*r_016_6;
   double r_530_5 = PmAy*r_520_5 + 2*i2z*(r_510_5 + r_510_6*riz) + rPmQy*r_520_6;
   double r_503_5 = PmAz*r_502_5 + 2*i2z*(r_501_5 + r_501_6*riz) + rPmQz*r_502_6;
   double r_053_5 = PmAz*r_052_5 + 2*i2z*(r_051_5 + r_051_6*riz) + rPmQz*r_052_6;
   double r_035_5 = PmAy*r_025_5 + 2*i2z*(r_015_5 + r_015_6*riz) + rPmQy*r_025_6;
   double r_900_4 = PmAx*r_800_4 + 8*i2z*(r_700_4 + r_700_5*riz) + rPmQx*r_800_5;
   double r_090_4 = PmAy*r_080_4 + 8*i2z*(r_070_4 + r_070_5*riz) + rPmQy*r_080_5;
   double r_009_4 = PmAz*r_008_4 + 8*i2z*(r_007_4 + r_007_5*riz) + rPmQz*r_008_5;
   double r_180_4 = PmAx*r_080_4 + rPmQx*r_080_5;
   double r_108_4 = PmAx*r_008_4 + rPmQx*r_008_5;
   double r_810_4 = PmAy*r_800_4 + rPmQy*r_800_5;
   double r_018_4 = PmAy*r_008_4 + rPmQy*r_008_5;
   double r_801_4 = PmAz*r_800_4 + rPmQz*r_800_5;
   double r_081_4 = PmAz*r_080_4 + rPmQz*r_080_5;
   double r_720_4 = PmAy*r_710_4 + i2z*(r_700_4 + r_700_5*riz) + rPmQy*r_710_5;
   double r_702_4 = PmAz*r_701_4 + i2z*(r_700_4 + r_700_5*riz) + rPmQz*r_701_5;
   double r_270_4 = PmAx*r_170_4 + i2z*(r_070_4 + r_070_5*riz) + rPmQx*r_170_5;
   double r_072_4 = PmAz*r_071_4 + i2z*(r_070_4 + r_070_5*riz) + rPmQz*r_071_5;
   double r_207_4 = PmAx*r_107_4 + i2z*(r_007_4 + r_007_5*riz) + rPmQx*r_107_5;
   double r_027_4 = PmAy*r_017_4 + i2z*(r_007_4 + r_007_5*riz) + rPmQy*r_017_5;
   double r_360_4 = PmAx*r_260_4 + 2*i2z*(r_160_4 + r_160_5*riz) + rPmQx*r_260_5;
   double r_306_4 = PmAx*r_206_4 + 2*i2z*(r_106_4 + r_106_5*riz) + rPmQx*r_206_5;
   double r_630_4 = PmAy*r_620_4 + 2*i2z*(r_610_4 + r_610_5*riz) + rPmQy*r_620_5;
   double r_036_4 = PmAy*r_026_4 + 2*i2z*(r_016_4 + r_016_5*riz) + rPmQy*r_026_5;
   double r_603_4 = PmAz*r_602_4 + 2*i2z*(r_601_4 + r_601_5*riz) + rPmQz*r_602_5;
   double r_063_4 = PmAz*r_062_4 + 2*i2z*(r_061_4 + r_061_5*riz) + rPmQz*r_062_5;
   double r_540_4 = PmAy*r_530_4 + 3*i2z*(r_520_4 + r_520_5*riz) + rPmQy*r_530_5;
   double r_504_4 = PmAz*r_503_4 + 3*i2z*(r_502_4 + r_502_5*riz) + rPmQz*r_503_5;
   double r_054_4 = PmAz*r_053_4 + 3*i2z*(r_052_4 + r_052_5*riz) + rPmQz*r_053_5;
   double r_045_4 = PmAy*r_035_4 + 3*i2z*(r_025_4 + r_025_5*riz) + rPmQy*r_035_5;
   double r_a00_3 = PmAx*r_900_3 + 9*i2z*(r_800_3 + r_800_4*riz) + rPmQx*r_900_4;
   double r_0a0_3 = PmAy*r_090_3 + 9*i2z*(r_080_3 + r_080_4*riz) + rPmQy*r_090_4;
   double r_00a_3 = PmAz*r_009_3 + 9*i2z*(r_008_3 + r_008_4*riz) + rPmQz*r_009_4;
   double r_910_3 = PmAy*r_900_3 + rPmQy*r_900_4;
   double r_190_3 = PmAx*r_090_3 + rPmQx*r_090_4;
   double r_901_3 = PmAz*r_900_3 + rPmQz*r_900_4;
   double r_109_3 = PmAx*r_009_3 + rPmQx*r_009_4;
   double r_091_3 = PmAz*r_090_3 + rPmQz*r_090_4;
   double r_019_3 = PmAy*r_009_3 + rPmQy*r_009_4;
   double r_820_3 = PmAy*r_810_3 + i2z*(r_800_3 + r_800_4*riz) + rPmQy*r_810_4;
   double r_802_3 = PmAz*r_801_3 + i2z*(r_800_3 + r_800_4*riz) + rPmQz*r_801_4;
   double r_280_3 = PmAx*r_180_3 + i2z*(r_080_3 + r_080_4*riz) + rPmQx*r_180_4;
   double r_208_3 = PmAx*r_108_3 + i2z*(r_008_3 + r_008_4*riz) + rPmQx*r_108_4;
   double r_082_3 = PmAz*r_081_3 + i2z*(r_080_3 + r_080_4*riz) + rPmQz*r_081_4;
   double r_028_3 = PmAy*r_018_3 + i2z*(r_008_3 + r_008_4*riz) + rPmQy*r_018_4;
   double r_730_3 = PmAy*r_720_3 + 2*i2z*(r_710_3 + r_710_4*riz) + rPmQy*r_720_4;
   double r_370_3 = PmAx*r_270_3 + 2*i2z*(r_170_3 + r_170_4*riz) + rPmQx*r_270_4;
   double r_703_3 = PmAz*r_702_3 + 2*i2z*(r_701_3 + r_701_4*riz) + rPmQz*r_702_4;
   double r_307_3 = PmAx*r_207_3 + 2*i2z*(r_107_3 + r_107_4*riz) + rPmQx*r_207_4;
   double r_073_3 = PmAz*r_072_3 + 2*i2z*(r_071_3 + r_071_4*riz) + rPmQz*r_072_4;
   double r_037_3 = PmAy*r_027_3 + 2*i2z*(r_017_3 + r_017_4*riz) + rPmQy*r_027_4;
   double r_640_3 = PmAy*r_630_3 + 3*i2z*(r_620_3 + r_620_4*riz) + rPmQy*r_630_4;
   double r_604_3 = PmAz*r_603_3 + 3*i2z*(r_602_3 + r_602_4*riz) + rPmQz*r_603_4;
   double r_460_3 = PmAx*r_360_3 + 3*i2z*(r_260_3 + r_260_4*riz) + rPmQx*r_360_4;
   double r_406_3 = PmAx*r_306_3 + 3*i2z*(r_206_3 + r_206_4*riz) + rPmQx*r_306_4;
   double r_064_3 = PmAz*r_063_3 + 3*i2z*(r_062_3 + r_062_4*riz) + rPmQz*r_063_4;
   double r_046_3 = PmAy*r_036_3 + 3*i2z*(r_026_3 + r_026_4*riz) + rPmQy*r_036_4;
   double r_550_3 = PmAy*r_540_3 + 4*i2z*(r_530_3 + r_530_4*riz) + rPmQy*r_540_4;
   double r_505_3 = PmAz*r_504_3 + 4*i2z*(r_503_3 + r_503_4*riz) + rPmQz*r_504_4;
   double r_055_3 = PmAz*r_054_3 + 4*i2z*(r_053_3 + r_053_4*riz) + rPmQz*r_054_4;
   double r_514_3 = PmAy*r_504_3 + rPmQy*r_504_4;
   double r_154_3 = PmAx*r_054_3 + rPmQx*r_054_4;
   double r_145_3 = PmAx*r_045_3 + rPmQx*r_045_4;
   double r_b00_2 = PmAx*r_a00_2 + 10*i2z*(r_900_2 + r_900_3*riz) + rPmQx*r_a00_3;
   double r_0b0_2 = PmAy*r_0a0_2 + 10*i2z*(r_090_2 + r_090_3*riz) + rPmQy*r_0a0_3;
   double r_00b_2 = PmAz*r_00a_2 + 10*i2z*(r_009_2 + r_009_3*riz) + rPmQz*r_00a_3;
   double r_1a0_2 = PmAx*r_0a0_2 + rPmQx*r_0a0_3;
   double r_10a_2 = PmAx*r_00a_2 + rPmQx*r_00a_3;
   double r_a10_2 = PmAy*r_a00_2 + rPmQy*r_a00_3;
   double r_01a_2 = PmAy*r_00a_2 + rPmQy*r_00a_3;
   double r_a01_2 = PmAz*r_a00_2 + rPmQz*r_a00_3;
   double r_0a1_2 = PmAz*r_0a0_2 + rPmQz*r_0a0_3;
   double r_920_2 = PmAy*r_910_2 + i2z*(r_900_2 + r_900_3*riz) + rPmQy*r_910_3;
   double r_902_2 = PmAz*r_901_2 + i2z*(r_900_2 + r_900_3*riz) + rPmQz*r_901_3;
   double r_290_2 = PmAx*r_190_2 + i2z*(r_090_2 + r_090_3*riz) + rPmQx*r_190_3;
   double r_092_2 = PmAz*r_091_2 + i2z*(r_090_2 + r_090_3*riz) + rPmQz*r_091_3;
   double r_209_2 = PmAx*r_109_2 + i2z*(r_009_2 + r_009_3*riz) + rPmQx*r_109_3;
   double r_029_2 = PmAy*r_019_2 + i2z*(r_009_2 + r_009_3*riz) + rPmQy*r_019_3;
   double r_380_2 = PmAx*r_280_2 + 2*i2z*(r_180_2 + r_180_3*riz) + rPmQx*r_280_3;
   double r_308_2 = PmAx*r_208_2 + 2*i2z*(r_108_2 + r_108_3*riz) + rPmQx*r_208_3;
   double r_830_2 = PmAy*r_820_2 + 2*i2z*(r_810_2 + r_810_3*riz) + rPmQy*r_820_3;
   double r_038_2 = PmAy*r_028_2 + 2*i2z*(r_018_2 + r_018_3*riz) + rPmQy*r_028_3;
   double r_803_2 = PmAz*r_802_2 + 2*i2z*(r_801_2 + r_801_3*riz) + rPmQz*r_802_3;
   double r_083_2 = PmAz*r_082_2 + 2*i2z*(r_081_2 + r_081_3*riz) + rPmQz*r_082_3;
   double r_740_2 = PmAy*r_730_2 + 3*i2z*(r_720_2 + r_720_3*riz) + rPmQy*r_730_3;
   double r_704_2 = PmAz*r_703_2 + 3*i2z*(r_702_2 + r_702_3*riz) + rPmQz*r_703_3;
   double r_470_2 = PmAx*r_370_2 + 3*i2z*(r_270_2 + r_270_3*riz) + rPmQx*r_370_3;
   double r_074_2 = PmAz*r_073_2 + 3*i2z*(r_072_2 + r_072_3*riz) + rPmQz*r_073_3;
   double r_407_2 = PmAx*r_307_2 + 3*i2z*(r_207_2 + r_207_3*riz) + rPmQx*r_307_3;
   double r_047_2 = PmAy*r_037_2 + 3*i2z*(r_027_2 + r_027_3*riz) + rPmQy*r_037_3;
   double r_560_2 = PmAx*r_460_2 + 4*i2z*(r_360_2 + r_360_3*riz) + rPmQx*r_460_3;
   double r_506_2 = PmAx*r_406_2 + 4*i2z*(r_306_2 + r_306_3*riz) + rPmQx*r_406_3;
   double r_650_2 = PmAy*r_640_2 + 4*i2z*(r_630_2 + r_630_3*riz) + rPmQy*r_640_3;
   double r_056_2 = PmAy*r_046_2 + 4*i2z*(r_036_2 + r_036_3*riz) + rPmQy*r_046_3;
   double r_605_2 = PmAz*r_604_2 + 4*i2z*(r_603_2 + r_603_3*riz) + rPmQz*r_604_3;
   double r_065_2 = PmAz*r_064_2 + 4*i2z*(r_063_2 + r_063_3*riz) + rPmQz*r_064_3;
   double r_731_2 = PmAz*r_730_2 + rPmQz*r_730_3;
   double r_371_2 = PmAz*r_370_2 + rPmQz*r_370_3;
   double r_317_2 = PmAy*r_307_2 + rPmQy*r_307_3;
   double r_164_2 = PmAx*r_064_2 + rPmQx*r_064_3;
   double r_146_2 = PmAx*r_046_2 + rPmQx*r_046_3;
   double r_614_2 = PmAy*r_604_2 + rPmQy*r_604_3;
   double r_416_2 = PmAy*r_406_2 + rPmQy*r_406_3;
   double r_641_2 = PmAz*r_640_2 + rPmQz*r_640_3;
   double r_461_2 = PmAz*r_460_2 + rPmQz*r_460_3;
   double r_551_2 = PmAz*r_550_2 + rPmQz*r_550_3;
   double r_515_2 = PmAy*r_505_2 + rPmQy*r_505_3;
   double r_155_2 = PmAx*r_055_2 + rPmQx*r_055_3;
   double r_524_2 = PmAy*r_514_2 + i2z*(r_504_2 + r_504_3*riz) + rPmQy*r_514_3;
   double r_254_2 = PmAx*r_154_2 + i2z*(r_054_2 + r_054_3*riz) + rPmQx*r_154_3;
   double r_245_2 = PmAx*r_145_2 + i2z*(r_045_2 + r_045_3*riz) + rPmQx*r_145_3;
   double r_c00_1 = PmAx*r_b00_1 + 11*i2z*(r_a00_1 + r_a00_2*riz) + rPmQx*r_b00_2;
   double r_0c0_1 = PmAy*r_0b0_1 + 11*i2z*(r_0a0_1 + r_0a0_2*riz) + rPmQy*r_0b0_2;
   double r_00c_1 = PmAz*r_00b_1 + 11*i2z*(r_00a_1 + r_00a_2*riz) + rPmQz*r_00b_2;
   double r_b10_1 = PmAy*r_b00_1 + rPmQy*r_b00_2;
   double r_1b0_1 = PmAx*r_0b0_1 + rPmQx*r_0b0_2;
   double r_b01_1 = PmAz*r_b00_1 + rPmQz*r_b00_2;
   double r_10b_1 = PmAx*r_00b_1 + rPmQx*r_00b_2;
   double r_0b1_1 = PmAz*r_0b0_1 + rPmQz*r_0b0_2;
   double r_01b_1 = PmAy*r_00b_1 + rPmQy*r_00b_2;
   double r_a20_1 = PmAy*r_a10_1 + i2z*(r_a00_1 + r_a00_2*riz) + rPmQy*r_a10_2;
   double r_a02_1 = PmAz*r_a01_1 + i2z*(r_a00_1 + r_a00_2*riz) + rPmQz*r_a01_2;
   double r_2a0_1 = PmAx*r_1a0_1 + i2z*(r_0a0_1 + r_0a0_2*riz) + rPmQx*r_1a0_2;
   double r_20a_1 = PmAx*r_10a_1 + i2z*(r_00a_1 + r_00a_2*riz) + rPmQx*r_10a_2;
   double r_0a2_1 = PmAz*r_0a1_1 + i2z*(r_0a0_1 + r_0a0_2*riz) + rPmQz*r_0a1_2;
   double r_02a_1 = PmAy*r_01a_1 + i2z*(r_00a_1 + r_00a_2*riz) + rPmQy*r_01a_2;
   double r_930_1 = PmAy*r_920_1 + 2*i2z*(r_910_1 + r_910_2*riz) + rPmQy*r_920_2;
   double r_390_1 = PmAx*r_290_1 + 2*i2z*(r_190_1 + r_190_2*riz) + rPmQx*r_290_2;
   double r_903_1 = PmAz*r_902_1 + 2*i2z*(r_901_1 + r_901_2*riz) + rPmQz*r_902_2;
   double r_309_1 = PmAx*r_209_1 + 2*i2z*(r_109_1 + r_109_2*riz) + rPmQx*r_209_2;
   double r_093_1 = PmAz*r_092_1 + 2*i2z*(r_091_1 + r_091_2*riz) + rPmQz*r_092_2;
   double r_039_1 = PmAy*r_029_1 + 2*i2z*(r_019_1 + r_019_2*riz) + rPmQy*r_029_2;
   double r_840_1 = PmAy*r_830_1 + 3*i2z*(r_820_1 + r_820_2*riz) + rPmQy*r_830_2;
   double r_804_1 = PmAz*r_803_1 + 3*i2z*(r_802_1 + r_802_2*riz) + rPmQz*r_803_2;
   double r_480_1 = PmAx*r_380_1 + 3*i2z*(r_280_1 + r_280_2*riz) + rPmQx*r_380_2;
   double r_408_1 = PmAx*r_308_1 + 3*i2z*(r_208_1 + r_208_2*riz) + rPmQx*r_308_2;
   double r_084_1 = PmAz*r_083_1 + 3*i2z*(r_082_1 + r_082_2*riz) + rPmQz*r_083_2;
   double r_048_1 = PmAy*r_038_1 + 3*i2z*(r_028_1 + r_028_2*riz) + rPmQy*r_038_2;
   double r_750_1 = PmAy*r_740_1 + 4*i2z*(r_730_1 + r_730_2*riz) + rPmQy*r_740_2;
   double r_570_1 = PmAx*r_470_1 + 4*i2z*(r_370_1 + r_370_2*riz) + rPmQx*r_470_2;
   double r_705_1 = PmAz*r_704_1 + 4*i2z*(r_703_1 + r_703_2*riz) + rPmQz*r_704_2;
   double r_507_1 = PmAx*r_407_1 + 4*i2z*(r_307_1 + r_307_2*riz) + rPmQx*r_407_2;
   double r_075_1 = PmAz*r_074_1 + 4*i2z*(r_073_1 + r_073_2*riz) + rPmQz*r_074_2;
   double r_057_1 = PmAy*r_047_1 + 4*i2z*(r_037_1 + r_037_2*riz) + rPmQy*r_047_2;
   double r_660_1 = PmAx*r_560_1 + 5*i2z*(r_460_1 + r_460_2*riz) + rPmQx*r_560_2;
   double r_606_1 = PmAx*r_506_1 + 5*i2z*(r_406_1 + r_406_2*riz) + rPmQx*r_506_2;
   double r_066_1 = PmAy*r_056_1 + 5*i2z*(r_046_1 + r_046_2*riz) + rPmQy*r_056_2;
   double r_912_1 = PmAy*r_902_1 + rPmQy*r_902_2;
   double r_192_1 = PmAx*r_092_1 + rPmQx*r_092_2;
   double r_129_1 = PmAx*r_029_1 + rPmQx*r_029_2;
   double r_318_1 = PmAy*r_308_1 + rPmQy*r_308_2;
   double r_138_1 = PmAx*r_038_1 + rPmQx*r_038_2;
   double r_381_1 = PmAz*r_380_1 + rPmQz*r_380_2;
   double r_183_1 = PmAx*r_083_1 + rPmQx*r_083_2;
   double r_831_1 = PmAz*r_830_1 + rPmQz*r_830_2;
   double r_813_1 = PmAy*r_803_1 + rPmQy*r_803_2;
   double r_714_1 = PmAy*r_704_1 + rPmQy*r_704_2;
   double r_174_1 = PmAx*r_074_1 + rPmQx*r_074_2;
   double r_741_1 = PmAz*r_740_1 + rPmQz*r_740_2;
   double r_147_1 = PmAx*r_047_1 + rPmQx*r_047_2;
   double r_471_1 = PmAz*r_470_1 + rPmQz*r_470_2;
   double r_417_1 = PmAy*r_407_1 + rPmQy*r_407_2;
   double r_516_1 = PmAy*r_506_1 + rPmQy*r_506_2;
   double r_156_1 = PmAx*r_056_1 + rPmQx*r_056_2;
   double r_561_1 = PmAz*r_560_1 + rPmQz*r_560_2;
   double r_165_1 = PmAx*r_065_1 + rPmQx*r_065_2;
   double r_651_1 = PmAz*r_650_1 + rPmQz*r_650_2;
   double r_615_1 = PmAy*r_605_1 + rPmQy*r_605_2;
   double r_732_1 = PmAz*r_731_1 + i2z*(r_730_1 + r_730_2*riz) + rPmQz*r_731_2;
   double r_372_1 = PmAz*r_371_1 + i2z*(r_370_1 + r_370_2*riz) + rPmQz*r_371_2;
   double r_327_1 = PmAy*r_317_1 + i2z*(r_307_1 + r_307_2*riz) + rPmQy*r_317_2;
   double r_642_1 = PmAz*r_641_1 + i2z*(r_640_1 + r_640_2*riz) + rPmQz*r_641_2;
   double r_624_1 = PmAy*r_614_1 + i2z*(r_604_1 + r_604_2*riz) + rPmQy*r_614_2;
   double r_462_1 = PmAz*r_461_1 + i2z*(r_460_1 + r_460_2*riz) + rPmQz*r_461_2;
   double r_426_1 = PmAy*r_416_1 + i2z*(r_406_1 + r_406_2*riz) + rPmQy*r_416_2;
   double r_264_1 = PmAx*r_164_1 + i2z*(r_064_1 + r_064_2*riz) + rPmQx*r_164_2;
   double r_246_1 = PmAx*r_146_1 + i2z*(r_046_1 + r_046_2*riz) + rPmQx*r_146_2;
   double r_552_1 = PmAz*r_551_1 + i2z*(r_550_1 + r_550_2*riz) + rPmQz*r_551_2;
   double r_525_1 = PmAy*r_515_1 + i2z*(r_505_1 + r_505_2*riz) + rPmQy*r_515_2;
   double r_255_1 = PmAx*r_155_1 + i2z*(r_055_1 + r_055_2*riz) + rPmQx*r_155_2;
   double r_534_1 = PmAy*r_524_1 + 2*i2z*(r_514_1 + r_514_2*riz) + rPmQy*r_524_2;
   double r_354_1 = PmAx*r_254_1 + 2*i2z*(r_154_1 + r_154_2*riz) + rPmQx*r_254_2;
   double r_345_1 = PmAx*r_245_1 + 2*i2z*(r_145_1 + r_145_2*riz) + rPmQx*r_245_2;
   pOut[455] = PmAx*pOut[364] + 12*i2z*(pOut[286] + r_b00_1*riz) + rPmQx*r_c00_1;
   pOut[456] = PmAy*pOut[365] + 12*i2z*(pOut[287] + r_0b0_1*riz) + rPmQy*r_0c0_1;
   pOut[457] = PmAz*pOut[366] + 12*i2z*(pOut[288] + r_00b_1*riz) + rPmQz*r_00c_1;
   pOut[458] = PmAx*pOut[365] + rPmQx*r_0c0_1;
   pOut[459] = PmAx*pOut[366] + rPmQx*r_00c_1;
   pOut[460] = PmAy*pOut[364] + rPmQy*r_c00_1;
   pOut[461] = PmAy*pOut[366] + rPmQy*r_00c_1;
   pOut[462] = PmAz*pOut[364] + rPmQz*r_c00_1;
   pOut[463] = PmAz*pOut[365] + rPmQz*r_0c0_1;
   pOut[464] = PmAy*pOut[367] + i2z*(pOut[286] + r_b00_1*riz) + rPmQy*r_b10_1;
   pOut[465] = PmAz*pOut[369] + i2z*(pOut[286] + r_b00_1*riz) + rPmQz*r_b01_1;
   pOut[466] = PmAx*pOut[368] + i2z*(pOut[287] + r_0b0_1*riz) + rPmQx*r_1b0_1;
   pOut[467] = PmAz*pOut[371] + i2z*(pOut[287] + r_0b0_1*riz) + rPmQz*r_0b1_1;
   pOut[468] = PmAx*pOut[370] + i2z*(pOut[288] + r_00b_1*riz) + rPmQx*r_10b_1;
   pOut[469] = PmAy*pOut[372] + i2z*(pOut[288] + r_00b_1*riz) + rPmQy*r_01b_1;
   pOut[470] = PmAx*pOut[375] + 2*i2z*(pOut[289] + r_1a0_1*riz) + rPmQx*r_2a0_1;
   pOut[471] = PmAx*pOut[376] + 2*i2z*(pOut[290] + r_10a_1*riz) + rPmQx*r_20a_1;
   pOut[472] = PmAy*pOut[373] + 2*i2z*(pOut[291] + r_a10_1*riz) + rPmQy*r_a20_1;
   pOut[473] = PmAy*pOut[378] + 2*i2z*(pOut[292] + r_01a_1*riz) + rPmQy*r_02a_1;
   pOut[474] = PmAz*pOut[374] + 2*i2z*(pOut[293] + r_a01_1*riz) + rPmQz*r_a02_1;
   pOut[475] = PmAz*pOut[377] + 2*i2z*(pOut[294] + r_0a1_1*riz) + rPmQz*r_0a2_1;
   pOut[476] = PmAy*pOut[379] + 3*i2z*(pOut[295] + r_920_1*riz) + rPmQy*r_930_1;
   pOut[477] = PmAz*pOut[381] + 3*i2z*(pOut[296] + r_902_1*riz) + rPmQz*r_903_1;
   pOut[478] = PmAx*pOut[380] + 3*i2z*(pOut[297] + r_290_1*riz) + rPmQx*r_390_1;
   pOut[479] = PmAz*pOut[383] + 3*i2z*(pOut[298] + r_092_1*riz) + rPmQz*r_093_1;
   pOut[480] = PmAx*pOut[382] + 3*i2z*(pOut[299] + r_209_1*riz) + rPmQx*r_309_1;
   pOut[481] = PmAy*pOut[384] + 3*i2z*(pOut[300] + r_029_1*riz) + rPmQy*r_039_1;
   pOut[482] = PmAx*pOut[387] + 4*i2z*(pOut[301] + r_380_1*riz) + rPmQx*r_480_1;
   pOut[483] = PmAx*pOut[388] + 4*i2z*(pOut[302] + r_308_1*riz) + rPmQx*r_408_1;
   pOut[484] = PmAy*pOut[385] + 4*i2z*(pOut[303] + r_830_1*riz) + rPmQy*r_840_1;
   pOut[485] = PmAy*pOut[390] + 4*i2z*(pOut[304] + r_038_1*riz) + rPmQy*r_048_1;
   pOut[486] = PmAz*pOut[386] + 4*i2z*(pOut[305] + r_803_1*riz) + rPmQz*r_804_1;
   pOut[487] = PmAz*pOut[389] + 4*i2z*(pOut[306] + r_083_1*riz) + rPmQz*r_084_1;
   pOut[488] = PmAy*pOut[391] + 5*i2z*(pOut[307] + r_740_1*riz) + rPmQy*r_750_1;
   pOut[489] = PmAz*pOut[393] + 5*i2z*(pOut[308] + r_704_1*riz) + rPmQz*r_705_1;
   pOut[490] = PmAx*pOut[392] + 5*i2z*(pOut[309] + r_470_1*riz) + rPmQx*r_570_1;
   pOut[491] = PmAz*pOut[395] + 5*i2z*(pOut[310] + r_074_1*riz) + rPmQz*r_075_1;
   pOut[492] = PmAx*pOut[394] + 5*i2z*(pOut[311] + r_407_1*riz) + rPmQx*r_507_1;
   pOut[493] = PmAy*pOut[396] + 5*i2z*(pOut[312] + r_047_1*riz) + rPmQy*r_057_1;
   pOut[494] = PmAz*pOut[367] + rPmQz*r_b10_1;
   pOut[495] = PmAz*pOut[368] + rPmQz*r_1b0_1;
   pOut[496] = PmAy*pOut[370] + rPmQy*r_10b_1;
   pOut[497] = PmAx*pOut[377] + rPmQx*r_0a2_1;
   pOut[498] = PmAx*pOut[378] + rPmQx*r_02a_1;
   pOut[499] = PmAy*pOut[374] + rPmQy*r_a02_1;
   pOut[500] = PmAy*pOut[376] + rPmQy*r_20a_1;
   pOut[501] = PmAz*pOut[373] + rPmQz*r_a20_1;
   pOut[502] = PmAz*pOut[375] + rPmQz*r_2a0_1;
   pOut[503] = PmAz*pOut[379] + rPmQz*r_930_1;
   pOut[504] = PmAy*pOut[381] + rPmQy*r_903_1;
   pOut[505] = PmAz*pOut[380] + rPmQz*r_390_1;
   pOut[506] = PmAy*pOut[382] + rPmQy*r_309_1;
   pOut[507] = PmAx*pOut[383] + rPmQx*r_093_1;
   pOut[508] = PmAx*pOut[384] + rPmQx*r_039_1;
   pOut[509] = PmAx*pOut[389] + rPmQx*r_084_1;
   pOut[510] = PmAx*pOut[390] + rPmQx*r_048_1;
   pOut[511] = PmAy*pOut[386] + rPmQy*r_804_1;
   pOut[512] = PmAy*pOut[388] + rPmQy*r_408_1;
   pOut[513] = PmAz*pOut[385] + rPmQz*r_840_1;
   pOut[514] = PmAz*pOut[387] + rPmQz*r_480_1;
   pOut[515] = PmAz*pOut[391] + rPmQz*r_750_1;
   pOut[516] = PmAy*pOut[393] + rPmQy*r_705_1;
   pOut[517] = PmAz*pOut[392] + rPmQz*r_570_1;
   pOut[518] = PmAy*pOut[394] + rPmQy*r_507_1;
   pOut[519] = PmAx*pOut[395] + rPmQx*r_075_1;
   pOut[520] = PmAx*pOut[396] + rPmQx*r_057_1;
   pOut[521] = PmAx*pOut[399] + rPmQx*r_066_1;
   pOut[522] = PmAy*pOut[398] + rPmQy*r_606_1;
   pOut[523] = PmAz*pOut[397] + rPmQz*r_660_1;
   pOut[524] = PmAy*pOut[403] + i2z*(pOut[296] + r_902_1*riz) + rPmQy*r_912_1;
   pOut[525] = PmAx*pOut[404] + i2z*(pOut[298] + r_092_1*riz) + rPmQx*r_192_1;
   pOut[526] = PmAx*pOut[406] + i2z*(pOut[300] + r_029_1*riz) + rPmQx*r_129_1;
   pOut[527] = PmAz*pOut[411] + i2z*(pOut[301] + r_380_1*riz) + rPmQz*r_381_1;
   pOut[528] = PmAy*pOut[409] + i2z*(pOut[302] + r_308_1*riz) + rPmQy*r_318_1;
   pOut[529] = PmAz*pOut[413] + i2z*(pOut[303] + r_830_1*riz) + rPmQz*r_831_1;
   pOut[530] = PmAx*pOut[410] + i2z*(pOut[304] + r_038_1*riz) + rPmQx*r_138_1;
   pOut[531] = PmAy*pOut[414] + i2z*(pOut[305] + r_803_1*riz) + rPmQy*r_813_1;
   pOut[532] = PmAx*pOut[412] + i2z*(pOut[306] + r_083_1*riz) + rPmQx*r_183_1;
   pOut[533] = PmAz*pOut[417] + i2z*(pOut[307] + r_740_1*riz) + rPmQz*r_741_1;
   pOut[534] = PmAy*pOut[415] + i2z*(pOut[308] + r_704_1*riz) + rPmQy*r_714_1;
   pOut[535] = PmAz*pOut[419] + i2z*(pOut[309] + r_470_1*riz) + rPmQz*r_471_1;
   pOut[536] = PmAx*pOut[416] + i2z*(pOut[310] + r_074_1*riz) + rPmQx*r_174_1;
   pOut[537] = PmAy*pOut[420] + i2z*(pOut[311] + r_407_1*riz) + rPmQy*r_417_1;
   pOut[538] = PmAx*pOut[418] + i2z*(pOut[312] + r_047_1*riz) + rPmQx*r_147_1;
   pOut[539] = PmAz*pOut[423] + i2z*(pOut[313] + r_560_1*riz) + rPmQz*r_561_1;
   pOut[540] = PmAy*pOut[421] + i2z*(pOut[314] + r_506_1*riz) + rPmQy*r_516_1;
   pOut[541] = PmAz*pOut[425] + i2z*(pOut[315] + r_650_1*riz) + rPmQz*r_651_1;
   pOut[542] = PmAx*pOut[422] + i2z*(pOut[316] + r_056_1*riz) + rPmQx*r_156_1;
   pOut[543] = PmAy*pOut[426] + i2z*(pOut[317] + r_605_1*riz) + rPmQy*r_615_1;
   pOut[544] = PmAx*pOut[424] + i2z*(pOut[318] + r_065_1*riz) + rPmQx*r_165_1;
   pOut[545] = PmAz*pOut[430] + 2*i2z*(pOut[328] + r_731_1*riz) + rPmQz*r_732_1;
   pOut[546] = PmAz*pOut[431] + 2*i2z*(pOut[330] + r_371_1*riz) + rPmQz*r_372_1;
   pOut[547] = PmAy*pOut[433] + 2*i2z*(pOut[331] + r_317_1*riz) + rPmQy*r_327_1;
   pOut[548] = PmAx*pOut[440] + 2*i2z*(pOut[334] + r_164_1*riz) + rPmQx*r_264_1;
   pOut[549] = PmAx*pOut[441] + 2*i2z*(pOut[335] + r_146_1*riz) + rPmQx*r_246_1;
   pOut[550] = PmAy*pOut[437] + 2*i2z*(pOut[336] + r_614_1*riz) + rPmQy*r_624_1;
   pOut[551] = PmAy*pOut[439] + 2*i2z*(pOut[337] + r_416_1*riz) + rPmQy*r_426_1;
   pOut[552] = PmAz*pOut[436] + 2*i2z*(pOut[338] + r_641_1*riz) + rPmQz*r_642_1;
   pOut[553] = PmAz*pOut[438] + 2*i2z*(pOut[339] + r_461_1*riz) + rPmQz*r_462_1;
   pOut[554] = PmAz*pOut[442] + 2*i2z*(pOut[340] + r_551_1*riz) + rPmQz*r_552_1;
   pOut[555] = PmAy*pOut[443] + 2*i2z*(pOut[341] + r_515_1*riz) + rPmQy*r_525_1;
   pOut[556] = PmAx*pOut[444] + 2*i2z*(pOut[342] + r_155_1*riz) + rPmQx*r_255_1;
   pOut[557] = PmAy*pOut[448] + 3*i2z*(pOut[353] + r_524_1*riz) + rPmQy*r_534_1;
   pOut[558] = PmAx*pOut[449] + 3*i2z*(pOut[355] + r_254_1*riz) + rPmQx*r_354_1;
   pOut[559] = PmAx*pOut[451] + 3*i2z*(pOut[357] + r_245_1*riz) + rPmQx*r_345_1;
   // 3.21 steps per component, 1.81 kb stack
   if (lab == 13) return;

   // m + lab == 14
   double r_100_d = PmAx*pIm[13] + pIm[14]*rPmQx;
   double r_010_d = PmAy*pIm[13] + pIm[14]*rPmQy;
   double r_001_d = PmAz*pIm[13] + pIm[14]*rPmQz;
   double r_200_c = PmAx*r_100_c + i2z*(pIm[12] + pIm[13]*riz) + rPmQx*r_100_d;
   double r_020_c = PmAy*r_010_c + i2z*(pIm[12] + pIm[13]*riz) + rPmQy*r_010_d;
   double r_002_c = PmAz*r_001_c + i2z*(pIm[12] + pIm[13]*riz) + rPmQz*r_001_d;
   double r_300_b = PmAx*r_200_b + 2*i2z*(r_100_b + r_100_c*riz) + rPmQx*r_200_c;
   double r_030_b = PmAy*r_020_b + 2*i2z*(r_010_b + r_010_c*riz) + rPmQy*r_020_c;
   double r_003_b = PmAz*r_002_b + 2*i2z*(r_001_b + r_001_c*riz) + rPmQz*r_002_c;
   double r_400_a = PmAx*r_300_a + 3*i2z*(r_200_a + r_200_b*riz) + rPmQx*r_300_b;
   double r_040_a = PmAy*r_030_a + 3*i2z*(r_020_a + r_020_b*riz) + rPmQy*r_030_b;
   double r_004_a = PmAz*r_003_a + 3*i2z*(r_002_a + r_002_b*riz) + rPmQz*r_003_b;
   double r_500_9 = PmAx*r_400_9 + 4*i2z*(r_300_9 + r_300_a*riz) + rPmQx*r_400_a;
   double r_050_9 = PmAy*r_040_9 + 4*i2z*(r_030_9 + r_030_a*riz) + rPmQy*r_040_a;
   double r_005_9 = PmAz*r_004_9 + 4*i2z*(r_003_9 + r_003_a*riz) + rPmQz*r_004_a;
   double r_600_8 = PmAx*r_500_8 + 5*i2z*(r_400_8 + r_400_9*riz) + rPmQx*r_500_9;
   double r_060_8 = PmAy*r_050_8 + 5*i2z*(r_040_8 + r_040_9*riz) + rPmQy*r_050_9;
   double r_006_8 = PmAz*r_005_8 + 5*i2z*(r_004_8 + r_004_9*riz) + rPmQz*r_005_9;
   double r_510_8 = PmAy*r_500_8 + rPmQy*r_500_9;
   double r_501_8 = PmAz*r_500_8 + rPmQz*r_500_9;
   double r_051_8 = PmAz*r_050_8 + rPmQz*r_050_9;
   double r_700_7 = PmAx*r_600_7 + 6*i2z*(r_500_7 + r_500_8*riz) + rPmQx*r_600_8;
   double r_070_7 = PmAy*r_060_7 + 6*i2z*(r_050_7 + r_050_8*riz) + rPmQy*r_060_8;
   double r_007_7 = PmAz*r_006_7 + 6*i2z*(r_005_7 + r_005_8*riz) + rPmQz*r_006_8;
   double r_160_7 = PmAx*r_060_7 + rPmQx*r_060_8;
   double r_106_7 = PmAx*r_006_7 + rPmQx*r_006_8;
   double r_610_7 = PmAy*r_600_7 + rPmQy*r_600_8;
   double r_016_7 = PmAy*r_006_7 + rPmQy*r_006_8;
   double r_601_7 = PmAz*r_600_7 + rPmQz*r_600_8;
   double r_061_7 = PmAz*r_060_7 + rPmQz*r_060_8;
   double r_520_7 = PmAy*r_510_7 + i2z*(r_500_7 + r_500_8*riz) + rPmQy*r_510_8;
   double r_502_7 = PmAz*r_501_7 + i2z*(r_500_7 + r_500_8*riz) + rPmQz*r_501_8;
   double r_052_7 = PmAz*r_051_7 + i2z*(r_050_7 + r_050_8*riz) + rPmQz*r_051_8;
   double r_800_6 = PmAx*r_700_6 + 7*i2z*(r_600_6 + r_600_7*riz) + rPmQx*r_700_7;
   double r_080_6 = PmAy*r_070_6 + 7*i2z*(r_060_6 + r_060_7*riz) + rPmQy*r_070_7;
   double r_008_6 = PmAz*r_007_6 + 7*i2z*(r_006_6 + r_006_7*riz) + rPmQz*r_007_7;
   double r_710_6 = PmAy*r_700_6 + rPmQy*r_700_7;
   double r_170_6 = PmAx*r_070_6 + rPmQx*r_070_7;
   double r_701_6 = PmAz*r_700_6 + rPmQz*r_700_7;
   double r_107_6 = PmAx*r_007_6 + rPmQx*r_007_7;
   double r_071_6 = PmAz*r_070_6 + rPmQz*r_070_7;
   double r_017_6 = PmAy*r_007_6 + rPmQy*r_007_7;
   double r_620_6 = PmAy*r_610_6 + i2z*(r_600_6 + r_600_7*riz) + rPmQy*r_610_7;
   double r_602_6 = PmAz*r_601_6 + i2z*(r_600_6 + r_600_7*riz) + rPmQz*r_601_7;
   double r_260_6 = PmAx*r_160_6 + i2z*(r_060_6 + r_060_7*riz) + rPmQx*r_160_7;
   double r_206_6 = PmAx*r_106_6 + i2z*(r_006_6 + r_006_7*riz) + rPmQx*r_106_7;
   double r_062_6 = PmAz*r_061_6 + i2z*(r_060_6 + r_060_7*riz) + rPmQz*r_061_7;
   double r_026_6 = PmAy*r_016_6 + i2z*(r_006_6 + r_006_7*riz) + rPmQy*r_016_7;
   double r_530_6 = PmAy*r_520_6 + 2*i2z*(r_510_6 + r_510_7*riz) + rPmQy*r_520_7;
   double r_503_6 = PmAz*r_502_6 + 2*i2z*(r_501_6 + r_501_7*riz) + rPmQz*r_502_7;
   double r_053_6 = PmAz*r_052_6 + 2*i2z*(r_051_6 + r_051_7*riz) + rPmQz*r_052_7;
   double r_900_5 = PmAx*r_800_5 + 8*i2z*(r_700_5 + r_700_6*riz) + rPmQx*r_800_6;
   double r_090_5 = PmAy*r_080_5 + 8*i2z*(r_070_5 + r_070_6*riz) + rPmQy*r_080_6;
   double r_009_5 = PmAz*r_008_5 + 8*i2z*(r_007_5 + r_007_6*riz) + rPmQz*r_008_6;
   double r_180_5 = PmAx*r_080_5 + rPmQx*r_080_6;
   double r_108_5 = PmAx*r_008_5 + rPmQx*r_008_6;
   double r_810_5 = PmAy*r_800_5 + rPmQy*r_800_6;
   double r_018_5 = PmAy*r_008_5 + rPmQy*r_008_6;
   double r_801_5 = PmAz*r_800_5 + rPmQz*r_800_6;
   double r_081_5 = PmAz*r_080_5 + rPmQz*r_080_6;
   double r_720_5 = PmAy*r_710_5 + i2z*(r_700_5 + r_700_6*riz) + rPmQy*r_710_6;
   double r_702_5 = PmAz*r_701_5 + i2z*(r_700_5 + r_700_6*riz) + rPmQz*r_701_6;
   double r_270_5 = PmAx*r_170_5 + i2z*(r_070_5 + r_070_6*riz) + rPmQx*r_170_6;
   double r_072_5 = PmAz*r_071_5 + i2z*(r_070_5 + r_070_6*riz) + rPmQz*r_071_6;
   double r_207_5 = PmAx*r_107_5 + i2z*(r_007_5 + r_007_6*riz) + rPmQx*r_107_6;
   double r_027_5 = PmAy*r_017_5 + i2z*(r_007_5 + r_007_6*riz) + rPmQy*r_017_6;
   double r_360_5 = PmAx*r_260_5 + 2*i2z*(r_160_5 + r_160_6*riz) + rPmQx*r_260_6;
   double r_306_5 = PmAx*r_206_5 + 2*i2z*(r_106_5 + r_106_6*riz) + rPmQx*r_206_6;
   double r_630_5 = PmAy*r_620_5 + 2*i2z*(r_610_5 + r_610_6*riz) + rPmQy*r_620_6;
   double r_036_5 = PmAy*r_026_5 + 2*i2z*(r_016_5 + r_016_6*riz) + rPmQy*r_026_6;
   double r_603_5 = PmAz*r_602_5 + 2*i2z*(r_601_5 + r_601_6*riz) + rPmQz*r_602_6;
   double r_063_5 = PmAz*r_062_5 + 2*i2z*(r_061_5 + r_061_6*riz) + rPmQz*r_062_6;
   double r_540_5 = PmAy*r_530_5 + 3*i2z*(r_520_5 + r_520_6*riz) + rPmQy*r_530_6;
   double r_504_5 = PmAz*r_503_5 + 3*i2z*(r_502_5 + r_502_6*riz) + rPmQz*r_503_6;
   double r_054_5 = PmAz*r_053_5 + 3*i2z*(r_052_5 + r_052_6*riz) + rPmQz*r_053_6;
   double r_a00_4 = PmAx*r_900_4 + 9*i2z*(r_800_4 + r_800_5*riz) + rPmQx*r_900_5;
   double r_0a0_4 = PmAy*r_090_4 + 9*i2z*(r_080_4 + r_080_5*riz) + rPmQy*r_090_5;
   double r_00a_4 = PmAz*r_009_4 + 9*i2z*(r_008_4 + r_008_5*riz) + rPmQz*r_009_5;
   double r_910_4 = PmAy*r_900_4 + rPmQy*r_900_5;
   double r_190_4 = PmAx*r_090_4 + rPmQx*r_090_5;
   double r_901_4 = PmAz*r_900_4 + rPmQz*r_900_5;
   double r_109_4 = PmAx*r_009_4 + rPmQx*r_009_5;
   double r_091_4 = PmAz*r_090_4 + rPmQz*r_090_5;
   double r_019_4 = PmAy*r_009_4 + rPmQy*r_009_5;
   double r_820_4 = PmAy*r_810_4 + i2z*(r_800_4 + r_800_5*riz) + rPmQy*r_810_5;
   double r_802_4 = PmAz*r_801_4 + i2z*(r_800_4 + r_800_5*riz) + rPmQz*r_801_5;
   double r_280_4 = PmAx*r_180_4 + i2z*(r_080_4 + r_080_5*riz) + rPmQx*r_180_5;
   double r_208_4 = PmAx*r_108_4 + i2z*(r_008_4 + r_008_5*riz) + rPmQx*r_108_5;
   double r_082_4 = PmAz*r_081_4 + i2z*(r_080_4 + r_080_5*riz) + rPmQz*r_081_5;
   double r_028_4 = PmAy*r_018_4 + i2z*(r_008_4 + r_008_5*riz) + rPmQy*r_018_5;
   double r_730_4 = PmAy*r_720_4 + 2*i2z*(r_710_4 + r_710_5*riz) + rPmQy*r_720_5;
   double r_370_4 = PmAx*r_270_4 + 2*i2z*(r_170_4 + r_170_5*riz) + rPmQx*r_270_5;
   double r_703_4 = PmAz*r_702_4 + 2*i2z*(r_701_4 + r_701_5*riz) + rPmQz*r_702_5;
   double r_307_4 = PmAx*r_207_4 + 2*i2z*(r_107_4 + r_107_5*riz) + rPmQx*r_207_5;
   double r_073_4 = PmAz*r_072_4 + 2*i2z*(r_071_4 + r_071_5*riz) + rPmQz*r_072_5;
   double r_037_4 = PmAy*r_027_4 + 2*i2z*(r_017_4 + r_017_5*riz) + rPmQy*r_027_5;
   double r_640_4 = PmAy*r_630_4 + 3*i2z*(r_620_4 + r_620_5*riz) + rPmQy*r_630_5;
   double r_604_4 = PmAz*r_603_4 + 3*i2z*(r_602_4 + r_602_5*riz) + rPmQz*r_603_5;
   double r_460_4 = PmAx*r_360_4 + 3*i2z*(r_260_4 + r_260_5*riz) + rPmQx*r_360_5;
   double r_406_4 = PmAx*r_306_4 + 3*i2z*(r_206_4 + r_206_5*riz) + rPmQx*r_306_5;
   double r_064_4 = PmAz*r_063_4 + 3*i2z*(r_062_4 + r_062_5*riz) + rPmQz*r_063_5;
   double r_046_4 = PmAy*r_036_4 + 3*i2z*(r_026_4 + r_026_5*riz) + rPmQy*r_036_5;
   double r_550_4 = PmAy*r_540_4 + 4*i2z*(r_530_4 + r_530_5*riz) + rPmQy*r_540_5;
   double r_505_4 = PmAz*r_504_4 + 4*i2z*(r_503_4 + r_503_5*riz) + rPmQz*r_504_5;
   double r_055_4 = PmAz*r_054_4 + 4*i2z*(r_053_4 + r_053_5*riz) + rPmQz*r_054_5;
   double r_b00_3 = PmAx*r_a00_3 + 10*i2z*(r_900_3 + r_900_4*riz) + rPmQx*r_a00_4;
   double r_0b0_3 = PmAy*r_0a0_3 + 10*i2z*(r_090_3 + r_090_4*riz) + rPmQy*r_0a0_4;
   double r_00b_3 = PmAz*r_00a_3 + 10*i2z*(r_009_3 + r_009_4*riz) + rPmQz*r_00a_4;
   double r_1a0_3 = PmAx*r_0a0_3 + rPmQx*r_0a0_4;
   double r_10a_3 = PmAx*r_00a_3 + rPmQx*r_00a_4;
   double r_a10_3 = PmAy*r_a00_3 + rPmQy*r_a00_4;
   double r_01a_3 = PmAy*r_00a_3 + rPmQy*r_00a_4;
   double r_a01_3 = PmAz*r_a00_3 + rPmQz*r_a00_4;
   double r_0a1_3 = PmAz*r_0a0_3 + rPmQz*r_0a0_4;
   double r_920_3 = PmAy*r_910_3 + i2z*(r_900_3 + r_900_4*riz) + rPmQy*r_910_4;
   double r_902_3 = PmAz*r_901_3 + i2z*(r_900_3 + r_900_4*riz) + rPmQz*r_901_4;
   double r_290_3 = PmAx*r_190_3 + i2z*(r_090_3 + r_090_4*riz) + rPmQx*r_190_4;
   double r_092_3 = PmAz*r_091_3 + i2z*(r_090_3 + r_090_4*riz) + rPmQz*r_091_4;
   double r_209_3 = PmAx*r_109_3 + i2z*(r_009_3 + r_009_4*riz) + rPmQx*r_109_4;
   double r_029_3 = PmAy*r_019_3 + i2z*(r_009_3 + r_009_4*riz) + rPmQy*r_019_4;
   double r_380_3 = PmAx*r_280_3 + 2*i2z*(r_180_3 + r_180_4*riz) + rPmQx*r_280_4;
   double r_308_3 = PmAx*r_208_3 + 2*i2z*(r_108_3 + r_108_4*riz) + rPmQx*r_208_4;
   double r_830_3 = PmAy*r_820_3 + 2*i2z*(r_810_3 + r_810_4*riz) + rPmQy*r_820_4;
   double r_038_3 = PmAy*r_028_3 + 2*i2z*(r_018_3 + r_018_4*riz) + rPmQy*r_028_4;
   double r_803_3 = PmAz*r_802_3 + 2*i2z*(r_801_3 + r_801_4*riz) + rPmQz*r_802_4;
   double r_083_3 = PmAz*r_082_3 + 2*i2z*(r_081_3 + r_081_4*riz) + rPmQz*r_082_4;
   double r_740_3 = PmAy*r_730_3 + 3*i2z*(r_720_3 + r_720_4*riz) + rPmQy*r_730_4;
   double r_704_3 = PmAz*r_703_3 + 3*i2z*(r_702_3 + r_702_4*riz) + rPmQz*r_703_4;
   double r_470_3 = PmAx*r_370_3 + 3*i2z*(r_270_3 + r_270_4*riz) + rPmQx*r_370_4;
   double r_074_3 = PmAz*r_073_3 + 3*i2z*(r_072_3 + r_072_4*riz) + rPmQz*r_073_4;
   double r_407_3 = PmAx*r_307_3 + 3*i2z*(r_207_3 + r_207_4*riz) + rPmQx*r_307_4;
   double r_047_3 = PmAy*r_037_3 + 3*i2z*(r_027_3 + r_027_4*riz) + rPmQy*r_037_4;
   double r_560_3 = PmAx*r_460_3 + 4*i2z*(r_360_3 + r_360_4*riz) + rPmQx*r_460_4;
   double r_506_3 = PmAx*r_406_3 + 4*i2z*(r_306_3 + r_306_4*riz) + rPmQx*r_406_4;
   double r_650_3 = PmAy*r_640_3 + 4*i2z*(r_630_3 + r_630_4*riz) + rPmQy*r_640_4;
   double r_056_3 = PmAy*r_046_3 + 4*i2z*(r_036_3 + r_036_4*riz) + rPmQy*r_046_4;
   double r_605_3 = PmAz*r_604_3 + 4*i2z*(r_603_3 + r_603_4*riz) + rPmQz*r_604_4;
   double r_065_3 = PmAz*r_064_3 + 4*i2z*(r_063_3 + r_063_4*riz) + rPmQz*r_064_4;
   double r_164_3 = PmAx*r_064_3 + rPmQx*r_064_4;
   double r_146_3 = PmAx*r_046_3 + rPmQx*r_046_4;
   double r_614_3 = PmAy*r_604_3 + rPmQy*r_604_4;
   double r_551_3 = PmAz*r_550_3 + rPmQz*r_550_4;
   double r_515_3 = PmAy*r_505_3 + rPmQy*r_505_4;
   double r_155_3 = PmAx*r_055_3 + rPmQx*r_055_4;
   double r_c00_2 = PmAx*r_b00_2 + 11*i2z*(r_a00_2 + r_a00_3*riz) + rPmQx*r_b00_3;
   double r_0c0_2 = PmAy*r_0b0_2 + 11*i2z*(r_0a0_2 + r_0a0_3*riz) + rPmQy*r_0b0_3;
   double r_00c_2 = PmAz*r_00b_2 + 11*i2z*(r_00a_2 + r_00a_3*riz) + rPmQz*r_00b_3;
   double r_b10_2 = PmAy*r_b00_2 + rPmQy*r_b00_3;
   double r_1b0_2 = PmAx*r_0b0_2 + rPmQx*r_0b0_3;
   double r_b01_2 = PmAz*r_b00_2 + rPmQz*r_b00_3;
   double r_10b_2 = PmAx*r_00b_2 + rPmQx*r_00b_3;
   double r_0b1_2 = PmAz*r_0b0_2 + rPmQz*r_0b0_3;
   double r_01b_2 = PmAy*r_00b_2 + rPmQy*r_00b_3;
   double r_a20_2 = PmAy*r_a10_2 + i2z*(r_a00_2 + r_a00_3*riz) + rPmQy*r_a10_3;
   double r_a02_2 = PmAz*r_a01_2 + i2z*(r_a00_2 + r_a00_3*riz) + rPmQz*r_a01_3;
   double r_2a0_2 = PmAx*r_1a0_2 + i2z*(r_0a0_2 + r_0a0_3*riz) + rPmQx*r_1a0_3;
   double r_20a_2 = PmAx*r_10a_2 + i2z*(r_00a_2 + r_00a_3*riz) + rPmQx*r_10a_3;
   double r_0a2_2 = PmAz*r_0a1_2 + i2z*(r_0a0_2 + r_0a0_3*riz) + rPmQz*r_0a1_3;
   double r_02a_2 = PmAy*r_01a_2 + i2z*(r_00a_2 + r_00a_3*riz) + rPmQy*r_01a_3;
   double r_930_2 = PmAy*r_920_2 + 2*i2z*(r_910_2 + r_910_3*riz) + rPmQy*r_920_3;
   double r_390_2 = PmAx*r_290_2 + 2*i2z*(r_190_2 + r_190_3*riz) + rPmQx*r_290_3;
   double r_903_2 = PmAz*r_902_2 + 2*i2z*(r_901_2 + r_901_3*riz) + rPmQz*r_902_3;
   double r_309_2 = PmAx*r_209_2 + 2*i2z*(r_109_2 + r_109_3*riz) + rPmQx*r_209_3;
   double r_093_2 = PmAz*r_092_2 + 2*i2z*(r_091_2 + r_091_3*riz) + rPmQz*r_092_3;
   double r_039_2 = PmAy*r_029_2 + 2*i2z*(r_019_2 + r_019_3*riz) + rPmQy*r_029_3;
   double r_840_2 = PmAy*r_830_2 + 3*i2z*(r_820_2 + r_820_3*riz) + rPmQy*r_830_3;
   double r_804_2 = PmAz*r_803_2 + 3*i2z*(r_802_2 + r_802_3*riz) + rPmQz*r_803_3;
   double r_480_2 = PmAx*r_380_2 + 3*i2z*(r_280_2 + r_280_3*riz) + rPmQx*r_380_3;
   double r_408_2 = PmAx*r_308_2 + 3*i2z*(r_208_2 + r_208_3*riz) + rPmQx*r_308_3;
   double r_084_2 = PmAz*r_083_2 + 3*i2z*(r_082_2 + r_082_3*riz) + rPmQz*r_083_3;
   double r_048_2 = PmAy*r_038_2 + 3*i2z*(r_028_2 + r_028_3*riz) + rPmQy*r_038_3;
   double r_750_2 = PmAy*r_740_2 + 4*i2z*(r_730_2 + r_730_3*riz) + rPmQy*r_740_3;
   double r_570_2 = PmAx*r_470_2 + 4*i2z*(r_370_2 + r_370_3*riz) + rPmQx*r_470_3;
   double r_705_2 = PmAz*r_704_2 + 4*i2z*(r_703_2 + r_703_3*riz) + rPmQz*r_704_3;
   double r_507_2 = PmAx*r_407_2 + 4*i2z*(r_307_2 + r_307_3*riz) + rPmQx*r_407_3;
   double r_075_2 = PmAz*r_074_2 + 4*i2z*(r_073_2 + r_073_3*riz) + rPmQz*r_074_3;
   double r_057_2 = PmAy*r_047_2 + 4*i2z*(r_037_2 + r_037_3*riz) + rPmQy*r_047_3;
   double r_660_2 = PmAx*r_560_2 + 5*i2z*(r_460_2 + r_460_3*riz) + rPmQx*r_560_3;
   double r_606_2 = PmAx*r_506_2 + 5*i2z*(r_406_2 + r_406_3*riz) + rPmQx*r_506_3;
   double r_066_2 = PmAy*r_056_2 + 5*i2z*(r_046_2 + r_046_3*riz) + rPmQy*r_056_3;
   double r_318_2 = PmAy*r_308_2 + rPmQy*r_308_3;
   double r_381_2 = PmAz*r_380_2 + rPmQz*r_380_3;
   double r_831_2 = PmAz*r_830_2 + rPmQz*r_830_3;
   double r_714_2 = PmAy*r_704_2 + rPmQy*r_704_3;
   double r_174_2 = PmAx*r_074_2 + rPmQx*r_074_3;
   double r_741_2 = PmAz*r_740_2 + rPmQz*r_740_3;
   double r_147_2 = PmAx*r_047_2 + rPmQx*r_047_3;
   double r_471_2 = PmAz*r_470_2 + rPmQz*r_470_3;
   double r_417_2 = PmAy*r_407_2 + rPmQy*r_407_3;
   double r_516_2 = PmAy*r_506_2 + rPmQy*r_506_3;
   double r_156_2 = PmAx*r_056_2 + rPmQx*r_056_3;
   double r_561_2 = PmAz*r_560_2 + rPmQz*r_560_3;
   double r_165_2 = PmAx*r_065_2 + rPmQx*r_065_3;
   double r_651_2 = PmAz*r_650_2 + rPmQz*r_650_3;
   double r_615_2 = PmAy*r_605_2 + rPmQy*r_605_3;
   double r_624_2 = PmAy*r_614_2 + i2z*(r_604_2 + r_604_3*riz) + rPmQy*r_614_3;
   double r_264_2 = PmAx*r_164_2 + i2z*(r_064_2 + r_064_3*riz) + rPmQx*r_164_3;
   double r_246_2 = PmAx*r_146_2 + i2z*(r_046_2 + r_046_3*riz) + rPmQx*r_146_3;
   double r_552_2 = PmAz*r_551_2 + i2z*(r_550_2 + r_550_3*riz) + rPmQz*r_551_3;
   double r_525_2 = PmAy*r_515_2 + i2z*(r_505_2 + r_505_3*riz) + rPmQy*r_515_3;
   double r_255_2 = PmAx*r_155_2 + i2z*(r_055_2 + r_055_3*riz) + rPmQx*r_155_3;
   double r_d00_1 = PmAx*r_c00_1 + 12*i2z*(r_b00_1 + r_b00_2*riz) + rPmQx*r_c00_2;
   double r_0d0_1 = PmAy*r_0c0_1 + 12*i2z*(r_0b0_1 + r_0b0_2*riz) + rPmQy*r_0c0_2;
   double r_00d_1 = PmAz*r_00c_1 + 12*i2z*(r_00b_1 + r_00b_2*riz) + rPmQz*r_00c_2;
   double r_1c0_1 = PmAx*r_0c0_1 + rPmQx*r_0c0_2;
   double r_10c_1 = PmAx*r_00c_1 + rPmQx*r_00c_2;
   double r_c10_1 = PmAy*r_c00_1 + rPmQy*r_c00_2;
   double r_01c_1 = PmAy*r_00c_1 + rPmQy*r_00c_2;
   double r_c01_1 = PmAz*r_c00_1 + rPmQz*r_c00_2;
   double r_0c1_1 = PmAz*r_0c0_1 + rPmQz*r_0c0_2;
   double r_b20_1 = PmAy*r_b10_1 + i2z*(r_b00_1 + r_b00_2*riz) + rPmQy*r_b10_2;
   double r_b02_1 = PmAz*r_b01_1 + i2z*(r_b00_1 + r_b00_2*riz) + rPmQz*r_b01_2;
   double r_2b0_1 = PmAx*r_1b0_1 + i2z*(r_0b0_1 + r_0b0_2*riz) + rPmQx*r_1b0_2;
   double r_0b2_1 = PmAz*r_0b1_1 + i2z*(r_0b0_1 + r_0b0_2*riz) + rPmQz*r_0b1_2;
   double r_20b_1 = PmAx*r_10b_1 + i2z*(r_00b_1 + r_00b_2*riz) + rPmQx*r_10b_2;
   double r_02b_1 = PmAy*r_01b_1 + i2z*(r_00b_1 + r_00b_2*riz) + rPmQy*r_01b_2;
   double r_3a0_1 = PmAx*r_2a0_1 + 2*i2z*(r_1a0_1 + r_1a0_2*riz) + rPmQx*r_2a0_2;
   double r_30a_1 = PmAx*r_20a_1 + 2*i2z*(r_10a_1 + r_10a_2*riz) + rPmQx*r_20a_2;
   double r_a30_1 = PmAy*r_a20_1 + 2*i2z*(r_a10_1 + r_a10_2*riz) + rPmQy*r_a20_2;
   double r_03a_1 = PmAy*r_02a_1 + 2*i2z*(r_01a_1 + r_01a_2*riz) + rPmQy*r_02a_2;
   double r_a03_1 = PmAz*r_a02_1 + 2*i2z*(r_a01_1 + r_a01_2*riz) + rPmQz*r_a02_2;
   double r_0a3_1 = PmAz*r_0a2_1 + 2*i2z*(r_0a1_1 + r_0a1_2*riz) + rPmQz*r_0a2_2;
   double r_940_1 = PmAy*r_930_1 + 3*i2z*(r_920_1 + r_920_2*riz) + rPmQy*r_930_2;
   double r_904_1 = PmAz*r_903_1 + 3*i2z*(r_902_1 + r_902_2*riz) + rPmQz*r_903_2;
   double r_490_1 = PmAx*r_390_1 + 3*i2z*(r_290_1 + r_290_2*riz) + rPmQx*r_390_2;
   double r_094_1 = PmAz*r_093_1 + 3*i2z*(r_092_1 + r_092_2*riz) + rPmQz*r_093_2;
   double r_409_1 = PmAx*r_309_1 + 3*i2z*(r_209_1 + r_209_2*riz) + rPmQx*r_309_2;
   double r_049_1 = PmAy*r_039_1 + 3*i2z*(r_029_1 + r_029_2*riz) + rPmQy*r_039_2;
   double r_580_1 = PmAx*r_480_1 + 4*i2z*(r_380_1 + r_380_2*riz) + rPmQx*r_480_2;
   double r_508_1 = PmAx*r_408_1 + 4*i2z*(r_308_1 + r_308_2*riz) + rPmQx*r_408_2;
   double r_850_1 = PmAy*r_840_1 + 4*i2z*(r_830_1 + r_830_2*riz) + rPmQy*r_840_2;
   double r_058_1 = PmAy*r_048_1 + 4*i2z*(r_038_1 + r_038_2*riz) + rPmQy*r_048_2;
   double r_805_1 = PmAz*r_804_1 + 4*i2z*(r_803_1 + r_803_2*riz) + rPmQz*r_804_2;
   double r_085_1 = PmAz*r_084_1 + 4*i2z*(r_083_1 + r_083_2*riz) + rPmQz*r_084_2;
   double r_760_1 = PmAy*r_750_1 + 5*i2z*(r_740_1 + r_740_2*riz) + rPmQy*r_750_2;
   double r_706_1 = PmAz*r_705_1 + 5*i2z*(r_704_1 + r_704_2*riz) + rPmQz*r_705_2;
   double r_670_1 = PmAx*r_570_1 + 5*i2z*(r_470_1 + r_470_2*riz) + rPmQx*r_570_2;
   double r_076_1 = PmAz*r_075_1 + 5*i2z*(r_074_1 + r_074_2*riz) + rPmQz*r_075_2;
   double r_607_1 = PmAx*r_507_1 + 5*i2z*(r_407_1 + r_407_2*riz) + rPmQx*r_507_2;
   double r_067_1 = PmAy*r_057_1 + 5*i2z*(r_047_1 + r_047_2*riz) + rPmQy*r_057_2;
   double r_1a2_1 = PmAx*r_0a2_1 + rPmQx*r_0a2_2;
   double r_12a_1 = PmAx*r_02a_1 + rPmQx*r_02a_2;
   double r_a12_1 = PmAy*r_a02_1 + rPmQy*r_a02_2;
   double r_931_1 = PmAz*r_930_1 + rPmQz*r_930_2;
   double r_913_1 = PmAy*r_903_1 + rPmQy*r_903_2;
   double r_391_1 = PmAz*r_390_1 + rPmQz*r_390_2;
   double r_319_1 = PmAy*r_309_1 + rPmQy*r_309_2;
   double r_193_1 = PmAx*r_093_1 + rPmQx*r_093_2;
   double r_139_1 = PmAx*r_039_1 + rPmQx*r_039_2;
   double r_184_1 = PmAx*r_084_1 + rPmQx*r_084_2;
   double r_148_1 = PmAx*r_048_1 + rPmQx*r_048_2;
   double r_814_1 = PmAy*r_804_1 + rPmQy*r_804_2;
   double r_418_1 = PmAy*r_408_1 + rPmQy*r_408_2;
   double r_841_1 = PmAz*r_840_1 + rPmQz*r_840_2;
   double r_481_1 = PmAz*r_480_1 + rPmQz*r_480_2;
   double r_751_1 = PmAz*r_750_1 + rPmQz*r_750_2;
   double r_715_1 = PmAy*r_705_1 + rPmQy*r_705_2;
   double r_571_1 = PmAz*r_570_1 + rPmQz*r_570_2;
   double r_517_1 = PmAy*r_507_1 + rPmQy*r_507_2;
   double r_175_1 = PmAx*r_075_1 + rPmQx*r_075_2;
   double r_157_1 = PmAx*r_057_1 + rPmQx*r_057_2;
   double r_166_1 = PmAx*r_066_1 + rPmQx*r_066_2;
   double r_616_1 = PmAy*r_606_1 + rPmQy*r_606_2;
   double r_661_1 = PmAz*r_660_1 + rPmQz*r_660_2;
   double r_382_1 = PmAz*r_381_1 + i2z*(r_380_1 + r_380_2*riz) + rPmQz*r_381_2;
   double r_328_1 = PmAy*r_318_1 + i2z*(r_308_1 + r_308_2*riz) + rPmQy*r_318_2;
   double r_832_1 = PmAz*r_831_1 + i2z*(r_830_1 + r_830_2*riz) + rPmQz*r_831_2;
   double r_742_1 = PmAz*r_741_1 + i2z*(r_740_1 + r_740_2*riz) + rPmQz*r_741_2;
   double r_724_1 = PmAy*r_714_1 + i2z*(r_704_1 + r_704_2*riz) + rPmQy*r_714_2;
   double r_472_1 = PmAz*r_471_1 + i2z*(r_470_1 + r_470_2*riz) + rPmQz*r_471_2;
   double r_274_1 = PmAx*r_174_1 + i2z*(r_074_1 + r_074_2*riz) + rPmQx*r_174_2;
   double r_427_1 = PmAy*r_417_1 + i2z*(r_407_1 + r_407_2*riz) + rPmQy*r_417_2;
   double r_247_1 = PmAx*r_147_1 + i2z*(r_047_1 + r_047_2*riz) + rPmQx*r_147_2;
   double r_562_1 = PmAz*r_561_1 + i2z*(r_560_1 + r_560_2*riz) + rPmQz*r_561_2;
   double r_526_1 = PmAy*r_516_1 + i2z*(r_506_1 + r_506_2*riz) + rPmQy*r_516_2;
   double r_652_1 = PmAz*r_651_1 + i2z*(r_650_1 + r_650_2*riz) + rPmQz*r_651_2;
   double r_256_1 = PmAx*r_156_1 + i2z*(r_056_1 + r_056_2*riz) + rPmQx*r_156_2;
   double r_625_1 = PmAy*r_615_1 + i2z*(r_605_1 + r_605_2*riz) + rPmQy*r_615_2;
   double r_265_1 = PmAx*r_165_1 + i2z*(r_065_1 + r_065_2*riz) + rPmQx*r_165_2;
   double r_364_1 = PmAx*r_264_1 + 2*i2z*(r_164_1 + r_164_2*riz) + rPmQx*r_264_2;
   double r_346_1 = PmAx*r_246_1 + 2*i2z*(r_146_1 + r_146_2*riz) + rPmQx*r_246_2;
   double r_634_1 = PmAy*r_624_1 + 2*i2z*(r_614_1 + r_614_2*riz) + rPmQy*r_624_2;
   double r_553_1 = PmAz*r_552_1 + 2*i2z*(r_551_1 + r_551_2*riz) + rPmQz*r_552_2;
   double r_535_1 = PmAy*r_525_1 + 2*i2z*(r_515_1 + r_515_2*riz) + rPmQy*r_525_2;
   double r_355_1 = PmAx*r_255_1 + 2*i2z*(r_155_1 + r_155_2*riz) + rPmQx*r_255_2;
   pOut[560] = PmAx*pOut[455] + 13*i2z*(pOut[364] + r_c00_1*riz) + rPmQx*r_d00_1;
   pOut[561] = PmAy*pOut[456] + 13*i2z*(pOut[365] + r_0c0_1*riz) + rPmQy*r_0d0_1;
   pOut[562] = PmAz*pOut[457] + 13*i2z*(pOut[366] + r_00c_1*riz) + rPmQz*r_00d_1;
   pOut[563] = PmAy*pOut[455] + rPmQy*r_d00_1;
   pOut[564] = PmAx*pOut[456] + rPmQx*r_0d0_1;
   pOut[565] = PmAz*pOut[455] + rPmQz*r_d00_1;
   pOut[566] = PmAx*pOut[457] + rPmQx*r_00d_1;
   pOut[567] = PmAz*pOut[456] + rPmQz*r_0d0_1;
   pOut[568] = PmAy*pOut[457] + rPmQy*r_00d_1;
   pOut[569] = PmAy*pOut[460] + i2z*(pOut[364] + r_c00_1*riz) + rPmQy*r_c10_1;
   pOut[570] = PmAz*pOut[462] + i2z*(pOut[364] + r_c00_1*riz) + rPmQz*r_c01_1;
   pOut[571] = PmAx*pOut[458] + i2z*(pOut[365] + r_0c0_1*riz) + rPmQx*r_1c0_1;
   pOut[572] = PmAx*pOut[459] + i2z*(pOut[366] + r_00c_1*riz) + rPmQx*r_10c_1;
   pOut[573] = PmAz*pOut[463] + i2z*(pOut[365] + r_0c0_1*riz) + rPmQz*r_0c1_1;
   pOut[574] = PmAy*pOut[461] + i2z*(pOut[366] + r_00c_1*riz) + rPmQy*r_01c_1;
   pOut[575] = PmAy*pOut[464] + 2*i2z*(pOut[367] + r_b10_1*riz) + rPmQy*r_b20_1;
   pOut[576] = PmAx*pOut[466] + 2*i2z*(pOut[368] + r_1b0_1*riz) + rPmQx*r_2b0_1;
   pOut[577] = PmAz*pOut[465] + 2*i2z*(pOut[369] + r_b01_1*riz) + rPmQz*r_b02_1;
   pOut[578] = PmAx*pOut[468] + 2*i2z*(pOut[370] + r_10b_1*riz) + rPmQx*r_20b_1;
   pOut[579] = PmAz*pOut[467] + 2*i2z*(pOut[371] + r_0b1_1*riz) + rPmQz*r_0b2_1;
   pOut[580] = PmAy*pOut[469] + 2*i2z*(pOut[372] + r_01b_1*riz) + rPmQy*r_02b_1;
   pOut[581] = PmAy*pOut[472] + 3*i2z*(pOut[373] + r_a20_1*riz) + rPmQy*r_a30_1;
   pOut[582] = PmAz*pOut[474] + 3*i2z*(pOut[374] + r_a02_1*riz) + rPmQz*r_a03_1;
   pOut[583] = PmAx*pOut[470] + 3*i2z*(pOut[375] + r_2a0_1*riz) + rPmQx*r_3a0_1;
   pOut[584] = PmAx*pOut[471] + 3*i2z*(pOut[376] + r_20a_1*riz) + rPmQx*r_30a_1;
   pOut[585] = PmAz*pOut[475] + 3*i2z*(pOut[377] + r_0a2_1*riz) + rPmQz*r_0a3_1;
   pOut[586] = PmAy*pOut[473] + 3*i2z*(pOut[378] + r_02a_1*riz) + rPmQy*r_03a_1;
   pOut[587] = PmAy*pOut[476] + 4*i2z*(pOut[379] + r_930_1*riz) + rPmQy*r_940_1;
   pOut[588] = PmAx*pOut[478] + 4*i2z*(pOut[380] + r_390_1*riz) + rPmQx*r_490_1;
   pOut[589] = PmAz*pOut[477] + 4*i2z*(pOut[381] + r_903_1*riz) + rPmQz*r_904_1;
   pOut[590] = PmAx*pOut[480] + 4*i2z*(pOut[382] + r_309_1*riz) + rPmQx*r_409_1;
   pOut[591] = PmAz*pOut[479] + 4*i2z*(pOut[383] + r_093_1*riz) + rPmQz*r_094_1;
   pOut[592] = PmAy*pOut[481] + 4*i2z*(pOut[384] + r_039_1*riz) + rPmQy*r_049_1;
   pOut[593] = PmAy*pOut[484] + 5*i2z*(pOut[385] + r_840_1*riz) + rPmQy*r_850_1;
   pOut[594] = PmAz*pOut[486] + 5*i2z*(pOut[386] + r_804_1*riz) + rPmQz*r_805_1;
   pOut[595] = PmAx*pOut[482] + 5*i2z*(pOut[387] + r_480_1*riz) + rPmQx*r_580_1;
   pOut[596] = PmAx*pOut[483] + 5*i2z*(pOut[388] + r_408_1*riz) + rPmQx*r_508_1;
   pOut[597] = PmAz*pOut[487] + 5*i2z*(pOut[389] + r_084_1*riz) + rPmQz*r_085_1;
   pOut[598] = PmAy*pOut[485] + 5*i2z*(pOut[390] + r_048_1*riz) + rPmQy*r_058_1;
   pOut[599] = PmAy*pOut[488] + 6*i2z*(pOut[391] + r_750_1*riz) + rPmQy*r_760_1;
   pOut[600] = PmAz*pOut[489] + 6*i2z*(pOut[393] + r_705_1*riz) + rPmQz*r_706_1;
   pOut[601] = PmAz*pOut[491] + 6*i2z*(pOut[395] + r_075_1*riz) + rPmQz*r_076_1;
   pOut[602] = PmAy*pOut[459] + rPmQy*r_10c_1;
   pOut[603] = PmAz*pOut[458] + rPmQz*r_1c0_1;
   pOut[604] = PmAz*pOut[460] + rPmQz*r_c10_1;
   pOut[605] = PmAy*pOut[465] + rPmQy*r_b02_1;
   pOut[606] = PmAx*pOut[467] + rPmQx*r_0b2_1;
   pOut[607] = PmAz*pOut[464] + rPmQz*r_b20_1;
   pOut[608] = PmAx*pOut[469] + rPmQx*r_02b_1;
   pOut[609] = PmAz*pOut[466] + rPmQz*r_2b0_1;
   pOut[610] = PmAy*pOut[468] + rPmQy*r_20b_1;
   pOut[611] = PmAy*pOut[471] + rPmQy*r_30a_1;
   pOut[612] = PmAx*pOut[473] + rPmQx*r_03a_1;
   pOut[613] = PmAz*pOut[470] + rPmQz*r_3a0_1;
   pOut[614] = PmAx*pOut[475] + rPmQx*r_0a3_1;
   pOut[615] = PmAz*pOut[472] + rPmQz*r_a30_1;
   pOut[616] = PmAy*pOut[474] + rPmQy*r_a03_1;
   pOut[617] = PmAy*pOut[477] + rPmQy*r_904_1;
   pOut[618] = PmAx*pOut[479] + rPmQx*r_094_1;
   pOut[619] = PmAz*pOut[476] + rPmQz*r_940_1;
   pOut[620] = PmAx*pOut[481] + rPmQx*r_049_1;
   pOut[621] = PmAz*pOut[478] + rPmQz*r_490_1;
   pOut[622] = PmAy*pOut[480] + rPmQy*r_409_1;
   pOut[623] = PmAy*pOut[483] + rPmQy*r_508_1;
   pOut[624] = PmAx*pOut[485] + rPmQx*r_058_1;
   pOut[625] = PmAz*pOut[482] + rPmQz*r_580_1;
   pOut[626] = PmAx*pOut[487] + rPmQx*r_085_1;
   pOut[627] = PmAz*pOut[484] + rPmQz*r_850_1;
   pOut[628] = PmAy*pOut[486] + rPmQy*r_805_1;
   pOut[629] = PmAy*pOut[489] + rPmQy*r_706_1;
   pOut[630] = PmAx*pOut[491] + rPmQx*r_076_1;
   pOut[631] = PmAz*pOut[488] + rPmQz*r_760_1;
   pOut[632] = PmAx*pOut[493] + rPmQx*r_067_1;
   pOut[633] = PmAz*pOut[490] + rPmQz*r_670_1;
   pOut[634] = PmAy*pOut[492] + rPmQy*r_607_1;
   pOut[635] = PmAy*pOut[499] + i2z*(pOut[374] + r_a02_1*riz) + rPmQy*r_a12_1;
   pOut[636] = PmAx*pOut[497] + i2z*(pOut[377] + r_0a2_1*riz) + rPmQx*r_1a2_1;
   pOut[637] = PmAx*pOut[498] + i2z*(pOut[378] + r_02a_1*riz) + rPmQx*r_12a_1;
   pOut[638] = PmAz*pOut[503] + i2z*(pOut[379] + r_930_1*riz) + rPmQz*r_931_1;
   pOut[639] = PmAz*pOut[505] + i2z*(pOut[380] + r_390_1*riz) + rPmQz*r_391_1;
   pOut[640] = PmAy*pOut[504] + i2z*(pOut[381] + r_903_1*riz) + rPmQy*r_913_1;
   pOut[641] = PmAy*pOut[506] + i2z*(pOut[382] + r_309_1*riz) + rPmQy*r_319_1;
   pOut[642] = PmAx*pOut[507] + i2z*(pOut[383] + r_093_1*riz) + rPmQx*r_193_1;
   pOut[643] = PmAx*pOut[508] + i2z*(pOut[384] + r_039_1*riz) + rPmQx*r_139_1;
   pOut[644] = PmAz*pOut[513] + i2z*(pOut[385] + r_840_1*riz) + rPmQz*r_841_1;
   pOut[645] = PmAy*pOut[511] + i2z*(pOut[386] + r_804_1*riz) + rPmQy*r_814_1;
   pOut[646] = PmAz*pOut[514] + i2z*(pOut[387] + r_480_1*riz) + rPmQz*r_481_1;
   pOut[647] = PmAy*pOut[512] + i2z*(pOut[388] + r_408_1*riz) + rPmQy*r_418_1;
   pOut[648] = PmAx*pOut[509] + i2z*(pOut[389] + r_084_1*riz) + rPmQx*r_184_1;
   pOut[649] = PmAx*pOut[510] + i2z*(pOut[390] + r_048_1*riz) + rPmQx*r_148_1;
   pOut[650] = PmAz*pOut[515] + i2z*(pOut[391] + r_750_1*riz) + rPmQz*r_751_1;
   pOut[651] = PmAz*pOut[517] + i2z*(pOut[392] + r_570_1*riz) + rPmQz*r_571_1;
   pOut[652] = PmAy*pOut[516] + i2z*(pOut[393] + r_705_1*riz) + rPmQy*r_715_1;
   pOut[653] = PmAy*pOut[518] + i2z*(pOut[394] + r_507_1*riz) + rPmQy*r_517_1;
   pOut[654] = PmAx*pOut[519] + i2z*(pOut[395] + r_075_1*riz) + rPmQx*r_175_1;
   pOut[655] = PmAx*pOut[520] + i2z*(pOut[396] + r_057_1*riz) + rPmQx*r_157_1;
   pOut[656] = PmAz*pOut[523] + i2z*(pOut[397] + r_660_1*riz) + rPmQz*r_661_1;
   pOut[657] = PmAy*pOut[522] + i2z*(pOut[398] + r_606_1*riz) + rPmQy*r_616_1;
   pOut[658] = PmAx*pOut[521] + i2z*(pOut[399] + r_066_1*riz) + rPmQx*r_166_1;
   pOut[659] = PmAy*pOut[528] + 2*i2z*(pOut[409] + r_318_1*riz) + rPmQy*r_328_1;
   pOut[660] = PmAz*pOut[527] + 2*i2z*(pOut[411] + r_381_1*riz) + rPmQz*r_382_1;
   pOut[661] = PmAz*pOut[529] + 2*i2z*(pOut[413] + r_831_1*riz) + rPmQz*r_832_1;
   pOut[662] = PmAy*pOut[534] + 2*i2z*(pOut[415] + r_714_1*riz) + rPmQy*r_724_1;
   pOut[663] = PmAx*pOut[536] + 2*i2z*(pOut[416] + r_174_1*riz) + rPmQx*r_274_1;
   pOut[664] = PmAz*pOut[533] + 2*i2z*(pOut[417] + r_741_1*riz) + rPmQz*r_742_1;
   pOut[665] = PmAx*pOut[538] + 2*i2z*(pOut[418] + r_147_1*riz) + rPmQx*r_247_1;
   pOut[666] = PmAz*pOut[535] + 2*i2z*(pOut[419] + r_471_1*riz) + rPmQz*r_472_1;
   pOut[667] = PmAy*pOut[537] + 2*i2z*(pOut[420] + r_417_1*riz) + rPmQy*r_427_1;
   pOut[668] = PmAy*pOut[540] + 2*i2z*(pOut[421] + r_516_1*riz) + rPmQy*r_526_1;
   pOut[669] = PmAx*pOut[542] + 2*i2z*(pOut[422] + r_156_1*riz) + rPmQx*r_256_1;
   pOut[670] = PmAz*pOut[539] + 2*i2z*(pOut[423] + r_561_1*riz) + rPmQz*r_562_1;
   pOut[671] = PmAx*pOut[544] + 2*i2z*(pOut[424] + r_165_1*riz) + rPmQx*r_265_1;
   pOut[672] = PmAz*pOut[541] + 2*i2z*(pOut[425] + r_651_1*riz) + rPmQz*r_652_1;
   pOut[673] = PmAy*pOut[543] + 2*i2z*(pOut[426] + r_615_1*riz) + rPmQy*r_625_1;
   pOut[674] = PmAy*pOut[550] + 3*i2z*(pOut[437] + r_624_1*riz) + rPmQy*r_634_1;
   pOut[675] = PmAx*pOut[548] + 3*i2z*(pOut[440] + r_264_1*riz) + rPmQx*r_364_1;
   pOut[676] = PmAx*pOut[549] + 3*i2z*(pOut[441] + r_246_1*riz) + rPmQx*r_346_1;
   pOut[677] = PmAz*pOut[554] + 3*i2z*(pOut[442] + r_552_1*riz) + rPmQz*r_553_1;
   pOut[678] = PmAy*pOut[555] + 3*i2z*(pOut[443] + r_525_1*riz) + rPmQy*r_535_1;
   pOut[679] = PmAx*pOut[556] + 3*i2z*(pOut[444] + r_255_1*riz) + rPmQx*r_355_1;
   // 3.38 steps per component, 2.23 kb stack
   if (lab == 14) return;
   // If you arrive here we ran out of angular momenta. Regenerate with larger Lab.
   assert(0);
}

// Transform cartesians to Slc: map matrix [N] x [nCartY(l)] --> [N] x [(2*l+1)].
static void ShTrC6(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      double z0 = pIn[N*0], z1 = pIn[N*1], z3 = pIn[N*3], z4 = pIn[N*4], z5 = pIn[N*5], z7 = pIn[N*7], z9 = pIn[N*9], za = pIn[N*10], zb = pIn[N*11], zc = pIn[N*12], zd = pIn[N*13], ze = pIn[N*14], zf = pIn[N*15], z10 = pIn[N*16], z11 = pIn[N*17], z13 = pIn[N*19], z14 = pIn[N*20], z15 = pIn[N*21], z16 = pIn[N*22], z17 = pIn[N*23], z18 = pIn[N*24], z19 = pIn[N*25], z1a = pIn[N*26], z1b = pIn[N*27];
      pOut[N*0] = sd2b*z0 - sd2b*z1 - sd2c*z9 + sd2c*zb;
      pOut[N*1] = pIn[N*18]*sd2f + sd2d*z3 + sd2d*z4 + sd2e*zf - sd2f*z15 - sd2f*z16;
      pOut[N*2] = sd30*z5 - sd31*z17 + sd32*z13;
      pOut[N*3] = -sd33*z0 - sd33*z1 + sd34*z9 + sd34*zb + sd35*za + sd35*zd - sd36*z1b;
      pOut[N*4] = sd30*z7 - sd31*z19 + sd32*z14;
      pOut[N*5] = sd37*z0 - sd37*z1 + sd37*z9 - sd37*zb - sd38*za + sd38*zc + sd38*zd - sd38*ze;
      pOut[N*6] = sd39*z3 + sd39*z4 - sd3a*zf;
      pOut[N*7] = sd38*z10 - sd3b*z5 + sd3c*z17 + sd3d*z13 - sd3e*z18;
      pOut[N*8] = -sd3f*z3 + sd3f*z4 + sd40*z15 - sd40*z16;
      pOut[N*9] = pIn[N*2] - sd41*z0 - sd41*z1 - sd42*z9 - sd42*zb + sd43*za + sd43*zd + sd44*z1b - sd45*zc - sd45*ze;
      pOut[N*10] = -sd38*z11 + sd3b*z7 - sd3c*z19 - sd3d*z14 + sd3e*z1a;
      pOut[N*11] = pIn[N*8]*sd49 + sd46*z14 + sd46*z7 + sd47*z19 - sd48*z11 - sd48*z1a;
      pOut[N*12] = pIn[N*6]*sd49 + sd46*z13 + sd46*z5 + sd47*z17 - sd48*z10 - sd48*z18;
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC5(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      double z0 = pIn[N*0], z1 = pIn[N*1], z3 = pIn[N*3], z5 = pIn[N*5], z7 = pIn[N*7], z8 = pIn[N*8], z9 = pIn[N*9], za = pIn[N*10], zb = pIn[N*11], zc = pIn[N*12], zd = pIn[N*13], ze = pIn[N*14], zf = pIn[N*15], z10 = pIn[N*16], z12 = pIn[N*18], z13 = pIn[N*19], z14 = pIn[N*20];
      pOut[N*0] = pIn[N*4]*sd8 + sd17*z0 + sd17*z3 + sd18*z9 - sd19*z12 - sd19*za;
      pOut[N*1] = pIn[N*6]*sd8 + sd17*z1 + sd17*z5 + sd18*zb - sd19*z13 - sd19*zc;
      pOut[N*2] = -sd1a*z7 + sd1a*z8 + sd1b*zd - sd1b*ze;
      pOut[N*3] = -sd1c*z0 + sd1d*z9 + sd1e*za + sd1f*z3 - sd20*z12;
      pOut[N*4] = -sd21*z10 + sd21*zf;
      pOut[N*5] = sd1c*z1 - sd1d*zb - sd1e*zc - sd1f*z5 + sd20*z13;
      pOut[N*6] = sd22*z7 + sd22*z8 - sd23*z14;
      pOut[N*7] = sd24*z5 - sd25*zb + sd26*z1;
      pOut[N*8] = pIn[N*2] + sd27*z7 + sd27*z8 + sd28*z14 - sd29*zd - sd29*ze;
      pOut[N*9] = sd24*z3 - sd25*z9 + sd26*z0;
      pOut[N*10] = pIn[N*17]*sd2a - sd1b*z10 - sd1b*zf;
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC4(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      double z0 = pIn[N*0], z1 = pIn[N*1], z3 = pIn[N*3], z4 = pIn[N*4], z5 = pIn[N*5], z7 = pIn[N*7], z9 = pIn[N*9], za = pIn[N*10], zb = pIn[N*11], zd = pIn[N*13], ze = pIn[N*14];
      pOut[N*0] = pIn[N*2] + sda*z0 + sda*z1 + sdb*z9 - sdc*za - sdc*zb;
      pOut[N*1] = pIn[N*12]*sde - sdd*z3 - sdd*z4;
      pOut[N*2] = pIn[N*6]*sdf - sd7*z5 - sd7*zd;
      pOut[N*3] = sd10*z0 + sd10*z1 - sd11*z9;
      pOut[N*4] = pIn[N*8]*sdf - sd7*z7 - sd7*ze;
      pOut[N*5] = -sd12*z0 + sd12*z1 + sd13*za - sd13*zb;
      pOut[N*6] = sd14*z3 - sd14*z4;
      pOut[N*7] = sd15*z5 - sd16*zd;
      pOut[N*8] = -sd15*z7 + sd16*ze;
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC3(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      double z0 = pIn[N*0], z1 = pIn[N*1], z3 = pIn[N*3], z5 = pIn[N*5], z7 = pIn[N*7], z8 = pIn[N*8];
      pOut[N*0] = pIn[N*4]*sd4 - sd3*z0 - sd3*z3;
      pOut[N*1] = pIn[N*6]*sd4 - sd3*z1 - sd3*z5;
      pOut[N*2] = pIn[N*2] - sd5*z7 - sd5*z8;
      pOut[N*3] = sd6*z0 - sd7*z3;
      pOut[N*4] = pIn[N*9]*sd8;
      pOut[N*5] = -sd6*z1 + sd7*z5;
      pOut[N*6] = sd9*z7 - sd9*z8;
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC2(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      double z0 = pIn[N*0], z1 = pIn[N*1];
      pOut[N*0] = pIn[N*2] - sd0*z0 - sd0*z1;
      pOut[N*1] = pIn[N*3]*sd1;
      pOut[N*2] = pIn[N*4]*sd1;
      pOut[N*3] = sd2*z0 - sd2*z1;
      pOut[N*4] = pIn[N*5]*sd1;
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC1(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      pOut[N*0] = pIn[N*0];
      pOut[N*1] = pIn[N*1];
      pOut[N*2] = pIn[N*2];
      pOut += 1;
      pIn += 1;
   }
   return;
}

static void ShTrC0(double *IR_RP pOut, double const *IR_RP pIn, size_t N)
{
   for (size_t i = 0; i < N; ++ i) {
      pOut[N*0] = pIn[N*0];
      pOut += 1;
      pIn += 1;
   }
   return;
}

// Transform cartesians to Slc: map matrix [N] x [nCartY(l)] --> [N] x [(2*l+1)].
void ShTrN(double *IR_RP pOut, double const *IR_RP pIn, size_t N, unsigned l)
{
   switch(l) {
      case 0: return ShTrC0(pOut, pIn, N);
      case 1: return ShTrC1(pOut, pIn, N);
      case 2: return ShTrC2(pOut, pIn, N);
      case 3: return ShTrC3(pOut, pIn, N);
      case 4: return ShTrC4(pOut, pIn, N);
      case 5: return ShTrC5(pOut, pIn, N);
      case 6: return ShTrC6(pOut, pIn, N);
   }
   assert(0);
}

// Calculate (a0|c)^m from (a0|c-1x)^{m+1}, for a = la..lab
// i = _x/_y/_z is the reduction direction (x,y,z);
// fPmQi = PmQ[i] * (rho/gamma); i2ze = .5/eta_ABC = .5/(alpha + beta + gamma)
// note: the la is not for performance reasons (would likely make the actual recurrence
//       slower for small lab) but for preventing overwriting lower components.
static void OsrrB_3cKerM_x(double *IR_RP pOut, double const *IR_RP pIn, double fPmQx, unsigned la, unsigned lab, double i2ze)
{
   switch(la) {
   case 0:
      pOut[0] = fPmQx*pIn[0];
      if (lab == 0) return;
   case 1:
      pOut[1] = fPmQx*pIn[1] + i2ze*pIn[0];
      pOut[2] = fPmQx*pIn[2];
      pOut[3] = fPmQx*pIn[3];
      if (lab == 1) return;
   case 2:
      pOut[4] = fPmQx*pIn[4] + 2*i2ze*pIn[1];
      pOut[5] = fPmQx*pIn[5];
      pOut[6] = fPmQx*pIn[6];
      pOut[7] = fPmQx*pIn[7] + i2ze*pIn[2];
      pOut[8] = fPmQx*pIn[8] + i2ze*pIn[3];
      pOut[9] = fPmQx*pIn[9];
      if (lab == 2) return;
   case 3:
      pOut[10] = fPmQx*pIn[10] + 3*i2ze*pIn[4];
      pOut[11] = fPmQx*pIn[11];
      pOut[12] = fPmQx*pIn[12];
      pOut[13] = fPmQx*pIn[13] + i2ze*pIn[5];
      pOut[14] = fPmQx*pIn[14] + i2ze*pIn[6];
      pOut[15] = fPmQx*pIn[15] + 2*i2ze*pIn[7];
      pOut[16] = fPmQx*pIn[16];
      pOut[17] = fPmQx*pIn[17] + 2*i2ze*pIn[8];
      pOut[18] = fPmQx*pIn[18];
      pOut[19] = fPmQx*pIn[19] + i2ze*pIn[9];
      if (lab == 3) return;
   case 4:
      pOut[20] = fPmQx*pIn[20] + 4*i2ze*pIn[10];
      pOut[21] = fPmQx*pIn[21];
      pOut[22] = fPmQx*pIn[22];
      pOut[23] = fPmQx*pIn[23] + 3*i2ze*pIn[15];
      pOut[24] = fPmQx*pIn[24] + i2ze*pIn[11];
      pOut[25] = fPmQx*pIn[25] + 3*i2ze*pIn[17];
      pOut[26] = fPmQx*pIn[26] + i2ze*pIn[12];
      pOut[27] = fPmQx*pIn[27];
      pOut[28] = fPmQx*pIn[28];
      pOut[29] = fPmQx*pIn[29] + 2*i2ze*pIn[13];
      pOut[30] = fPmQx*pIn[30] + 2*i2ze*pIn[14];
      pOut[31] = fPmQx*pIn[31];
      pOut[32] = fPmQx*pIn[32] + i2ze*pIn[16];
      pOut[33] = fPmQx*pIn[33] + i2ze*pIn[18];
      pOut[34] = fPmQx*pIn[34] + 2*i2ze*pIn[19];
      if (lab == 4) return;
   case 5:
      pOut[35] = fPmQx*pIn[35] + 5*i2ze*pIn[20];
      pOut[36] = fPmQx*pIn[36];
      pOut[37] = fPmQx*pIn[37];
      pOut[38] = fPmQx*pIn[38] + i2ze*pIn[21];
      pOut[39] = fPmQx*pIn[39] + i2ze*pIn[22];
      pOut[40] = fPmQx*pIn[40] + 4*i2ze*pIn[23];
      pOut[41] = fPmQx*pIn[41];
      pOut[42] = fPmQx*pIn[42] + 4*i2ze*pIn[25];
      pOut[43] = fPmQx*pIn[43];
      pOut[44] = fPmQx*pIn[44] + 3*i2ze*pIn[29];
      pOut[45] = fPmQx*pIn[45] + 3*i2ze*pIn[30];
      pOut[46] = fPmQx*pIn[46] + 2*i2ze*pIn[24];
      pOut[47] = fPmQx*pIn[47];
      pOut[48] = fPmQx*pIn[48] + 2*i2ze*pIn[26];
      pOut[49] = fPmQx*pIn[49];
      pOut[50] = fPmQx*pIn[50] + 3*i2ze*pIn[34];
      pOut[51] = fPmQx*pIn[51] + i2ze*pIn[27];
      pOut[52] = fPmQx*pIn[52] + i2ze*pIn[28];
      pOut[53] = fPmQx*pIn[53] + i2ze*pIn[31];
      pOut[54] = fPmQx*pIn[54] + 2*i2ze*pIn[32];
      pOut[55] = fPmQx*pIn[55] + 2*i2ze*pIn[33];
      if (lab == 5) return;
   case 6:
      pOut[56] = fPmQx*pIn[56] + 6*i2ze*pIn[35];
      pOut[57] = fPmQx*pIn[57];
      pOut[58] = fPmQx*pIn[58];
      pOut[59] = fPmQx*pIn[59] + 5*i2ze*pIn[40];
      pOut[60] = fPmQx*pIn[60] + i2ze*pIn[36];
      pOut[61] = fPmQx*pIn[61] + 5*i2ze*pIn[42];
      pOut[62] = fPmQx*pIn[62] + i2ze*pIn[37];
      pOut[63] = fPmQx*pIn[63];
      pOut[64] = fPmQx*pIn[64];
      pOut[65] = fPmQx*pIn[65] + 4*i2ze*pIn[44];
      pOut[66] = fPmQx*pIn[66] + 4*i2ze*pIn[45];
      pOut[67] = fPmQx*pIn[67] + 2*i2ze*pIn[38];
      pOut[68] = fPmQx*pIn[68] + 2*i2ze*pIn[39];
      pOut[69] = fPmQx*pIn[69];
      pOut[70] = fPmQx*pIn[70];
      pOut[71] = fPmQx*pIn[71] + 3*i2ze*pIn[46];
      pOut[72] = fPmQx*pIn[72] + 3*i2ze*pIn[48];
      pOut[73] = fPmQx*pIn[73];
      pOut[74] = fPmQx*pIn[74] + i2ze*pIn[41];
      pOut[75] = fPmQx*pIn[75] + i2ze*pIn[43];
      pOut[76] = fPmQx*pIn[76] + 4*i2ze*pIn[50];
      pOut[77] = fPmQx*pIn[77] + 3*i2ze*pIn[54];
      pOut[78] = fPmQx*pIn[78] + i2ze*pIn[47];
      pOut[79] = fPmQx*pIn[79] + 3*i2ze*pIn[55];
      pOut[80] = fPmQx*pIn[80] + i2ze*pIn[49];
      pOut[81] = fPmQx*pIn[81] + 2*i2ze*pIn[51];
      pOut[82] = fPmQx*pIn[82] + 2*i2ze*pIn[52];
      pOut[83] = fPmQx*pIn[83] + 2*i2ze*pIn[53];
      if (lab == 6) return;
   case 7:
      pOut[84] = fPmQx*pIn[84] + 7*i2ze*pIn[56];
      pOut[85] = fPmQx*pIn[85];
      pOut[86] = fPmQx*pIn[86];
      pOut[87] = fPmQx*pIn[87] + i2ze*pIn[57];
      pOut[88] = fPmQx*pIn[88] + i2ze*pIn[58];
      pOut[89] = fPmQx*pIn[89] + 6*i2ze*pIn[59];
      pOut[90] = fPmQx*pIn[90];
      pOut[91] = fPmQx*pIn[91] + 6*i2ze*pIn[61];
      pOut[92] = fPmQx*pIn[92];
      pOut[93] = fPmQx*pIn[93] + 5*i2ze*pIn[65];
      pOut[94] = fPmQx*pIn[94] + 5*i2ze*pIn[66];
      pOut[95] = fPmQx*pIn[95] + 2*i2ze*pIn[60];
      pOut[96] = fPmQx*pIn[96];
      pOut[97] = fPmQx*pIn[97] + 2*i2ze*pIn[62];
      pOut[98] = fPmQx*pIn[98];
      pOut[99] = fPmQx*pIn[99] + 3*i2ze*pIn[67];
      pOut[100] = fPmQx*pIn[100] + 3*i2ze*pIn[68];
      pOut[101] = fPmQx*pIn[101] + 4*i2ze*pIn[71];
      pOut[102] = fPmQx*pIn[102];
      pOut[103] = fPmQx*pIn[103] + 4*i2ze*pIn[72];
      pOut[104] = fPmQx*pIn[104];
      pOut[105] = fPmQx*pIn[105] + 5*i2ze*pIn[76];
      pOut[106] = fPmQx*pIn[106] + i2ze*pIn[63];
      pOut[107] = fPmQx*pIn[107] + i2ze*pIn[64];
      pOut[108] = fPmQx*pIn[108] + i2ze*pIn[69];
      pOut[109] = fPmQx*pIn[109] + i2ze*pIn[70];
      pOut[110] = fPmQx*pIn[110] + 4*i2ze*pIn[77];
      pOut[111] = fPmQx*pIn[111] + 2*i2ze*pIn[74];
      pOut[112] = fPmQx*pIn[112] + 4*i2ze*pIn[79];
      pOut[113] = fPmQx*pIn[113] + 2*i2ze*pIn[75];
      pOut[114] = fPmQx*pIn[114] + 3*i2ze*pIn[81];
      pOut[115] = fPmQx*pIn[115] + 3*i2ze*pIn[82];
      pOut[116] = fPmQx*pIn[116] + i2ze*pIn[73];
      pOut[117] = fPmQx*pIn[117] + 3*i2ze*pIn[83];
      pOut[118] = fPmQx*pIn[118] + 2*i2ze*pIn[78];
      pOut[119] = fPmQx*pIn[119] + 2*i2ze*pIn[80];
      if (lab == 7) return;
   case 8:
      pOut[120] = fPmQx*pIn[120] + 8*i2ze*pIn[84];
      pOut[121] = fPmQx*pIn[121];
      pOut[122] = fPmQx*pIn[122];
      pOut[123] = fPmQx*pIn[123] + 7*i2ze*pIn[89];
      pOut[124] = fPmQx*pIn[124] + i2ze*pIn[85];
      pOut[125] = fPmQx*pIn[125] + 7*i2ze*pIn[91];
      pOut[126] = fPmQx*pIn[126] + i2ze*pIn[86];
      pOut[127] = fPmQx*pIn[127];
      pOut[128] = fPmQx*pIn[128];
      pOut[129] = fPmQx*pIn[129] + 6*i2ze*pIn[93];
      pOut[130] = fPmQx*pIn[130] + 6*i2ze*pIn[94];
      pOut[131] = fPmQx*pIn[131] + 2*i2ze*pIn[87];
      pOut[132] = fPmQx*pIn[132] + 2*i2ze*pIn[88];
      pOut[133] = fPmQx*pIn[133];
      pOut[134] = fPmQx*pIn[134];
      pOut[135] = fPmQx*pIn[135] + 5*i2ze*pIn[101];
      pOut[136] = fPmQx*pIn[136] + 3*i2ze*pIn[95];
      pOut[137] = fPmQx*pIn[137] + 5*i2ze*pIn[103];
      pOut[138] = fPmQx*pIn[138] + 3*i2ze*pIn[97];
      pOut[139] = fPmQx*pIn[139];
      pOut[140] = fPmQx*pIn[140];
      pOut[141] = fPmQx*pIn[141] + 4*i2ze*pIn[99];
      pOut[142] = fPmQx*pIn[142] + 4*i2ze*pIn[100];
      pOut[143] = fPmQx*pIn[143];
      pOut[144] = fPmQx*pIn[144] + i2ze*pIn[90];
      pOut[145] = fPmQx*pIn[145] + i2ze*pIn[92];
      pOut[146] = fPmQx*pIn[146] + 6*i2ze*pIn[105];
      pOut[147] = fPmQx*pIn[147] + 5*i2ze*pIn[110];
      pOut[148] = fPmQx*pIn[148] + i2ze*pIn[96];
      pOut[149] = fPmQx*pIn[149] + 5*i2ze*pIn[112];
      pOut[150] = fPmQx*pIn[150] + i2ze*pIn[98];
      pOut[151] = fPmQx*pIn[151] + 2*i2ze*pIn[106];
      pOut[152] = fPmQx*pIn[152] + 2*i2ze*pIn[107];
      pOut[153] = fPmQx*pIn[153] + 3*i2ze*pIn[111];
      pOut[154] = fPmQx*pIn[154] + i2ze*pIn[102];
      pOut[155] = fPmQx*pIn[155] + 3*i2ze*pIn[113];
      pOut[156] = fPmQx*pIn[156] + i2ze*pIn[104];
      pOut[157] = fPmQx*pIn[157] + 4*i2ze*pIn[114];
      pOut[158] = fPmQx*pIn[158] + 4*i2ze*pIn[115];
      pOut[159] = fPmQx*pIn[159] + 4*i2ze*pIn[117];
      pOut[160] = fPmQx*pIn[160] + 2*i2ze*pIn[108];
      pOut[161] = fPmQx*pIn[161] + 2*i2ze*pIn[109];
      pOut[162] = fPmQx*pIn[162] + 3*i2ze*pIn[118];
      pOut[163] = fPmQx*pIn[163] + 3*i2ze*pIn[119];
      pOut[164] = fPmQx*pIn[164] + 2*i2ze*pIn[116];
      if (lab == 8) return;
   case 9:
      pOut[165] = fPmQx*pIn[165] + 9*i2ze*pIn[120];
      pOut[166] = fPmQx*pIn[166];
      pOut[167] = fPmQx*pIn[167];
      pOut[168] = fPmQx*pIn[168] + i2ze*pIn[121];
      pOut[169] = fPmQx*pIn[169] + i2ze*pIn[122];
      pOut[170] = fPmQx*pIn[170] + 8*i2ze*pIn[123];
      pOut[171] = fPmQx*pIn[171];
      pOut[172] = fPmQx*pIn[172] + 8*i2ze*pIn[125];
      pOut[173] = fPmQx*pIn[173];
      pOut[174] = fPmQx*pIn[174] + 7*i2ze*pIn[129];
      pOut[175] = fPmQx*pIn[175] + 7*i2ze*pIn[130];
      pOut[176] = fPmQx*pIn[176] + 2*i2ze*pIn[124];
      pOut[177] = fPmQx*pIn[177];
      pOut[178] = fPmQx*pIn[178] + 2*i2ze*pIn[126];
      pOut[179] = fPmQx*pIn[179];
      pOut[180] = fPmQx*pIn[180] + 3*i2ze*pIn[131];
      pOut[181] = fPmQx*pIn[181] + 3*i2ze*pIn[132];
      pOut[182] = fPmQx*pIn[182] + 6*i2ze*pIn[135];
      pOut[183] = fPmQx*pIn[183];
      pOut[184] = fPmQx*pIn[184] + 6*i2ze*pIn[137];
      pOut[185] = fPmQx*pIn[185];
      pOut[186] = fPmQx*pIn[186] + 5*i2ze*pIn[141];
      pOut[187] = fPmQx*pIn[187] + 5*i2ze*pIn[142];
      pOut[188] = fPmQx*pIn[188] + 4*i2ze*pIn[136];
      pOut[189] = fPmQx*pIn[189];
      pOut[190] = fPmQx*pIn[190] + 4*i2ze*pIn[138];
      pOut[191] = fPmQx*pIn[191];
      pOut[192] = fPmQx*pIn[192] + 7*i2ze*pIn[146];
      pOut[193] = fPmQx*pIn[193] + i2ze*pIn[127];
      pOut[194] = fPmQx*pIn[194] + i2ze*pIn[128];
      pOut[195] = fPmQx*pIn[195] + i2ze*pIn[133];
      pOut[196] = fPmQx*pIn[196] + i2ze*pIn[134];
      pOut[197] = fPmQx*pIn[197] + 6*i2ze*pIn[147];
      pOut[198] = fPmQx*pIn[198] + 2*i2ze*pIn[144];
      pOut[199] = fPmQx*pIn[199] + 6*i2ze*pIn[149];
      pOut[200] = fPmQx*pIn[200] + 2*i2ze*pIn[145];
      pOut[201] = fPmQx*pIn[201] + 5*i2ze*pIn[157];
      pOut[202] = fPmQx*pIn[202] + 5*i2ze*pIn[158];
      pOut[203] = fPmQx*pIn[203] + 3*i2ze*pIn[151];
      pOut[204] = fPmQx*pIn[204] + 3*i2ze*pIn[152];
      pOut[205] = fPmQx*pIn[205] + i2ze*pIn[139];
      pOut[206] = fPmQx*pIn[206] + i2ze*pIn[140];
      pOut[207] = fPmQx*pIn[207] + i2ze*pIn[143];
      pOut[208] = fPmQx*pIn[208] + 4*i2ze*pIn[153];
      pOut[209] = fPmQx*pIn[209] + 4*i2ze*pIn[155];
      pOut[210] = fPmQx*pIn[210] + 5*i2ze*pIn[159];
      pOut[211] = fPmQx*pIn[211] + 2*i2ze*pIn[148];
      pOut[212] = fPmQx*pIn[212] + 2*i2ze*pIn[150];
      pOut[213] = fPmQx*pIn[213] + 3*i2ze*pIn[160];
      pOut[214] = fPmQx*pIn[214] + 3*i2ze*pIn[161];
      pOut[215] = fPmQx*pIn[215] + 4*i2ze*pIn[162];
      pOut[216] = fPmQx*pIn[216] + 2*i2ze*pIn[154];
      pOut[217] = fPmQx*pIn[217] + 4*i2ze*pIn[163];
      pOut[218] = fPmQx*pIn[218] + 2*i2ze*pIn[156];
      pOut[219] = fPmQx*pIn[219] + 3*i2ze*pIn[164];
      if (lab == 9) return;
   case 10:
      pOut[220] = fPmQx*pIn[220] + 10*i2ze*pIn[165];
      pOut[221] = fPmQx*pIn[221];
      pOut[222] = fPmQx*pIn[222];
      pOut[223] = fPmQx*pIn[223] + 9*i2ze*pIn[170];
      pOut[224] = fPmQx*pIn[224] + i2ze*pIn[166];
      pOut[225] = fPmQx*pIn[225] + 9*i2ze*pIn[172];
      pOut[226] = fPmQx*pIn[226] + i2ze*pIn[167];
      pOut[227] = fPmQx*pIn[227];
      pOut[228] = fPmQx*pIn[228];
      pOut[229] = fPmQx*pIn[229] + 8*i2ze*pIn[174];
      pOut[230] = fPmQx*pIn[230] + 8*i2ze*pIn[175];
      pOut[231] = fPmQx*pIn[231] + 2*i2ze*pIn[168];
      pOut[232] = fPmQx*pIn[232] + 2*i2ze*pIn[169];
      pOut[233] = fPmQx*pIn[233];
      pOut[234] = fPmQx*pIn[234];
      pOut[235] = fPmQx*pIn[235] + 7*i2ze*pIn[182];
      pOut[236] = fPmQx*pIn[236] + 3*i2ze*pIn[176];
      pOut[237] = fPmQx*pIn[237] + 7*i2ze*pIn[184];
      pOut[238] = fPmQx*pIn[238] + 3*i2ze*pIn[178];
      pOut[239] = fPmQx*pIn[239];
      pOut[240] = fPmQx*pIn[240];
      pOut[241] = fPmQx*pIn[241] + 6*i2ze*pIn[186];
      pOut[242] = fPmQx*pIn[242] + 6*i2ze*pIn[187];
      pOut[243] = fPmQx*pIn[243] + 4*i2ze*pIn[180];
      pOut[244] = fPmQx*pIn[244] + 4*i2ze*pIn[181];
      pOut[245] = fPmQx*pIn[245];
      pOut[246] = fPmQx*pIn[246];
      pOut[247] = fPmQx*pIn[247] + 5*i2ze*pIn[188];
      pOut[248] = fPmQx*pIn[248] + 5*i2ze*pIn[190];
      pOut[249] = fPmQx*pIn[249];
      pOut[250] = fPmQx*pIn[250] + i2ze*pIn[171];
      pOut[251] = fPmQx*pIn[251] + i2ze*pIn[173];
      pOut[252] = fPmQx*pIn[252] + 8*i2ze*pIn[192];
      pOut[253] = fPmQx*pIn[253] + 7*i2ze*pIn[197];
      pOut[254] = fPmQx*pIn[254] + i2ze*pIn[177];
      pOut[255] = fPmQx*pIn[255] + 7*i2ze*pIn[199];
      pOut[256] = fPmQx*pIn[256] + i2ze*pIn[179];
      pOut[257] = fPmQx*pIn[257] + 2*i2ze*pIn[193];
      pOut[258] = fPmQx*pIn[258] + 2*i2ze*pIn[194];
      pOut[259] = fPmQx*pIn[259] + 3*i2ze*pIn[198];
      pOut[260] = fPmQx*pIn[260] + i2ze*pIn[183];
      pOut[261] = fPmQx*pIn[261] + 3*i2ze*pIn[200];
      pOut[262] = fPmQx*pIn[262] + i2ze*pIn[185];
      pOut[263] = fPmQx*pIn[263] + 6*i2ze*pIn[201];
      pOut[264] = fPmQx*pIn[264] + 6*i2ze*pIn[202];
      pOut[265] = fPmQx*pIn[265] + 5*i2ze*pIn[208];
      pOut[266] = fPmQx*pIn[266] + i2ze*pIn[189];
      pOut[267] = fPmQx*pIn[267] + 5*i2ze*pIn[209];
      pOut[268] = fPmQx*pIn[268] + i2ze*pIn[191];
      pOut[269] = fPmQx*pIn[269] + 4*i2ze*pIn[203];
      pOut[270] = fPmQx*pIn[270] + 4*i2ze*pIn[204];
      pOut[271] = fPmQx*pIn[271] + 6*i2ze*pIn[210];
      pOut[272] = fPmQx*pIn[272] + 2*i2ze*pIn[195];
      pOut[273] = fPmQx*pIn[273] + 2*i2ze*pIn[196];
      pOut[274] = fPmQx*pIn[274] + 5*i2ze*pIn[215];
      pOut[275] = fPmQx*pIn[275] + 3*i2ze*pIn[211];
      pOut[276] = fPmQx*pIn[276] + 5*i2ze*pIn[217];
      pOut[277] = fPmQx*pIn[277] + 3*i2ze*pIn[212];
      pOut[278] = fPmQx*pIn[278] + 2*i2ze*pIn[205];
      pOut[279] = fPmQx*pIn[279] + 2*i2ze*pIn[206];
      pOut[280] = fPmQx*pIn[280] + 4*i2ze*pIn[213];
      pOut[281] = fPmQx*pIn[281] + 4*i2ze*pIn[214];
      pOut[282] = fPmQx*pIn[282] + 2*i2ze*pIn[207];
      pOut[283] = fPmQx*pIn[283] + 3*i2ze*pIn[216];
      pOut[284] = fPmQx*pIn[284] + 3*i2ze*pIn[218];
      pOut[285] = fPmQx*pIn[285] + 4*i2ze*pIn[219];
      if (lab == 10) return;
   case 11:
      pOut[286] = fPmQx*pIn[286] + 11*i2ze*pIn[220];
      pOut[287] = fPmQx*pIn[287];
      pOut[288] = fPmQx*pIn[288];
      pOut[289] = fPmQx*pIn[289] + i2ze*pIn[221];
      pOut[290] = fPmQx*pIn[290] + i2ze*pIn[222];
      pOut[291] = fPmQx*pIn[291] + 10*i2ze*pIn[223];
      pOut[292] = fPmQx*pIn[292];
      pOut[293] = fPmQx*pIn[293] + 10*i2ze*pIn[225];
      pOut[294] = fPmQx*pIn[294];
      pOut[295] = fPmQx*pIn[295] + 9*i2ze*pIn[229];
      pOut[296] = fPmQx*pIn[296] + 9*i2ze*pIn[230];
      pOut[297] = fPmQx*pIn[297] + 2*i2ze*pIn[224];
      pOut[298] = fPmQx*pIn[298];
      pOut[299] = fPmQx*pIn[299] + 2*i2ze*pIn[226];
      pOut[300] = fPmQx*pIn[300];
      pOut[301] = fPmQx*pIn[301] + 3*i2ze*pIn[231];
      pOut[302] = fPmQx*pIn[302] + 3*i2ze*pIn[232];
      pOut[303] = fPmQx*pIn[303] + 8*i2ze*pIn[235];
      pOut[304] = fPmQx*pIn[304];
      pOut[305] = fPmQx*pIn[305] + 8*i2ze*pIn[237];
      pOut[306] = fPmQx*pIn[306];
      pOut[307] = fPmQx*pIn[307] + 7*i2ze*pIn[241];
      pOut[308] = fPmQx*pIn[308] + 7*i2ze*pIn[242];
      pOut[309] = fPmQx*pIn[309] + 4*i2ze*pIn[236];
      pOut[310] = fPmQx*pIn[310];
      pOut[311] = fPmQx*pIn[311] + 4*i2ze*pIn[238];
      pOut[312] = fPmQx*pIn[312];
      pOut[313] = fPmQx*pIn[313] + 5*i2ze*pIn[243];
      pOut[314] = fPmQx*pIn[314] + 5*i2ze*pIn[244];
      pOut[315] = fPmQx*pIn[315] + 6*i2ze*pIn[247];
      pOut[316] = fPmQx*pIn[316];
      pOut[317] = fPmQx*pIn[317] + 6*i2ze*pIn[248];
      pOut[318] = fPmQx*pIn[318];
      pOut[319] = fPmQx*pIn[319] + 9*i2ze*pIn[252];
      pOut[320] = fPmQx*pIn[320] + i2ze*pIn[227];
      pOut[321] = fPmQx*pIn[321] + i2ze*pIn[228];
      pOut[322] = fPmQx*pIn[322] + i2ze*pIn[233];
      pOut[323] = fPmQx*pIn[323] + i2ze*pIn[234];
      pOut[324] = fPmQx*pIn[324] + 8*i2ze*pIn[253];
      pOut[325] = fPmQx*pIn[325] + 2*i2ze*pIn[250];
      pOut[326] = fPmQx*pIn[326] + 8*i2ze*pIn[255];
      pOut[327] = fPmQx*pIn[327] + 2*i2ze*pIn[251];
      pOut[328] = fPmQx*pIn[328] + 7*i2ze*pIn[263];
      pOut[329] = fPmQx*pIn[329] + 7*i2ze*pIn[264];
      pOut[330] = fPmQx*pIn[330] + 3*i2ze*pIn[257];
      pOut[331] = fPmQx*pIn[331] + 3*i2ze*pIn[258];
      pOut[332] = fPmQx*pIn[332] + i2ze*pIn[239];
      pOut[333] = fPmQx*pIn[333] + i2ze*pIn[240];
      pOut[334] = fPmQx*pIn[334] + i2ze*pIn[245];
      pOut[335] = fPmQx*pIn[335] + i2ze*pIn[246];
      pOut[336] = fPmQx*pIn[336] + 6*i2ze*pIn[265];
      pOut[337] = fPmQx*pIn[337] + 4*i2ze*pIn[259];
      pOut[338] = fPmQx*pIn[338] + 6*i2ze*pIn[267];
      pOut[339] = fPmQx*pIn[339] + 4*i2ze*pIn[261];
      pOut[340] = fPmQx*pIn[340] + 5*i2ze*pIn[269];
      pOut[341] = fPmQx*pIn[341] + 5*i2ze*pIn[270];
      pOut[342] = fPmQx*pIn[342] + i2ze*pIn[249];
      pOut[343] = fPmQx*pIn[343] + 7*i2ze*pIn[271];
      pOut[344] = fPmQx*pIn[344] + 2*i2ze*pIn[254];
      pOut[345] = fPmQx*pIn[345] + 2*i2ze*pIn[256];
      pOut[346] = fPmQx*pIn[346] + 3*i2ze*pIn[272];
      pOut[347] = fPmQx*pIn[347] + 3*i2ze*pIn[273];
      pOut[348] = fPmQx*pIn[348] + 6*i2ze*pIn[274];
      pOut[349] = fPmQx*pIn[349] + 2*i2ze*pIn[260];
      pOut[350] = fPmQx*pIn[350] + 6*i2ze*pIn[276];
      pOut[351] = fPmQx*pIn[351] + 2*i2ze*pIn[262];
      pOut[352] = fPmQx*pIn[352] + 5*i2ze*pIn[280];
      pOut[353] = fPmQx*pIn[353] + 5*i2ze*pIn[281];
      pOut[354] = fPmQx*pIn[354] + 4*i2ze*pIn[275];
      pOut[355] = fPmQx*pIn[355] + 2*i2ze*pIn[266];
      pOut[356] = fPmQx*pIn[356] + 4*i2ze*pIn[277];
      pOut[357] = fPmQx*pIn[357] + 2*i2ze*pIn[268];
      pOut[358] = fPmQx*pIn[358] + 5*i2ze*pIn[285];
      pOut[359] = fPmQx*pIn[359] + 3*i2ze*pIn[278];
      pOut[360] = fPmQx*pIn[360] + 3*i2ze*pIn[279];
      pOut[361] = fPmQx*pIn[361] + 3*i2ze*pIn[282];
      pOut[362] = fPmQx*pIn[362] + 4*i2ze*pIn[283];
      pOut[363] = fPmQx*pIn[363] + 4*i2ze*pIn[284];
      if (lab == 11) return;
   case 12:
      pOut[364] = fPmQx*pIn[364] + 12*i2ze*pIn[286];
      pOut[365] = fPmQx*pIn[365];
      pOut[366] = fPmQx*pIn[366];
      pOut[367] = fPmQx*pIn[367] + 11*i2ze*pIn[291];
      pOut[368] = fPmQx*pIn[368] + i2ze*pIn[287];
      pOut[369] = fPmQx*pIn[369] + 11*i2ze*pIn[293];
      pOut[370] = fPmQx*pIn[370] + i2ze*pIn[288];
      pOut[371] = fPmQx*pIn[371];
      pOut[372] = fPmQx*pIn[372];
      pOut[373] = fPmQx*pIn[373] + 10*i2ze*pIn[295];
      pOut[374] = fPmQx*pIn[374] + 10*i2ze*pIn[296];
      pOut[375] = fPmQx*pIn[375] + 2*i2ze*pIn[289];
      pOut[376] = fPmQx*pIn[376] + 2*i2ze*pIn[290];
      pOut[377] = fPmQx*pIn[377];
      pOut[378] = fPmQx*pIn[378];
      pOut[379] = fPmQx*pIn[379] + 9*i2ze*pIn[303];
      pOut[380] = fPmQx*pIn[380] + 3*i2ze*pIn[297];
      pOut[381] = fPmQx*pIn[381] + 9*i2ze*pIn[305];
      pOut[382] = fPmQx*pIn[382] + 3*i2ze*pIn[299];
      pOut[383] = fPmQx*pIn[383];
      pOut[384] = fPmQx*pIn[384];
      pOut[385] = fPmQx*pIn[385] + 8*i2ze*pIn[307];
      pOut[386] = fPmQx*pIn[386] + 8*i2ze*pIn[308];
      pOut[387] = fPmQx*pIn[387] + 4*i2ze*pIn[301];
      pOut[388] = fPmQx*pIn[388] + 4*i2ze*pIn[302];
      pOut[389] = fPmQx*pIn[389];
      pOut[390] = fPmQx*pIn[390];
      pOut[391] = fPmQx*pIn[391] + 7*i2ze*pIn[315];
      pOut[392] = fPmQx*pIn[392] + 5*i2ze*pIn[309];
      pOut[393] = fPmQx*pIn[393] + 7*i2ze*pIn[317];
      pOut[394] = fPmQx*pIn[394] + 5*i2ze*pIn[311];
      pOut[395] = fPmQx*pIn[395];
      pOut[396] = fPmQx*pIn[396];
      pOut[397] = fPmQx*pIn[397] + 6*i2ze*pIn[313];
      pOut[398] = fPmQx*pIn[398] + 6*i2ze*pIn[314];
      pOut[399] = fPmQx*pIn[399];
      pOut[400] = fPmQx*pIn[400] + i2ze*pIn[292];
      pOut[401] = fPmQx*pIn[401] + i2ze*pIn[294];
      pOut[402] = fPmQx*pIn[402] + 10*i2ze*pIn[319];
      pOut[403] = fPmQx*pIn[403] + 9*i2ze*pIn[324];
      pOut[404] = fPmQx*pIn[404] + i2ze*pIn[298];
      pOut[405] = fPmQx*pIn[405] + 9*i2ze*pIn[326];
      pOut[406] = fPmQx*pIn[406] + i2ze*pIn[300];
      pOut[407] = fPmQx*pIn[407] + 2*i2ze*pIn[320];
      pOut[408] = fPmQx*pIn[408] + 2*i2ze*pIn[321];
      pOut[409] = fPmQx*pIn[409] + 3*i2ze*pIn[325];
      pOut[410] = fPmQx*pIn[410] + i2ze*pIn[304];
      pOut[411] = fPmQx*pIn[411] + 3*i2ze*pIn[327];
      pOut[412] = fPmQx*pIn[412] + i2ze*pIn[306];
      pOut[413] = fPmQx*pIn[413] + 8*i2ze*pIn[328];
      pOut[414] = fPmQx*pIn[414] + 8*i2ze*pIn[329];
      pOut[415] = fPmQx*pIn[415] + 7*i2ze*pIn[336];
      pOut[416] = fPmQx*pIn[416] + i2ze*pIn[310];
      pOut[417] = fPmQx*pIn[417] + 7*i2ze*pIn[338];
      pOut[418] = fPmQx*pIn[418] + i2ze*pIn[312];
      pOut[419] = fPmQx*pIn[419] + 4*i2ze*pIn[330];
      pOut[420] = fPmQx*pIn[420] + 4*i2ze*pIn[331];
      pOut[421] = fPmQx*pIn[421] + 5*i2ze*pIn[337];
      pOut[422] = fPmQx*pIn[422] + i2ze*pIn[316];
      pOut[423] = fPmQx*pIn[423] + 5*i2ze*pIn[339];
      pOut[424] = fPmQx*pIn[424] + i2ze*pIn[318];
      pOut[425] = fPmQx*pIn[425] + 6*i2ze*pIn[340];
      pOut[426] = fPmQx*pIn[426] + 6*i2ze*pIn[341];
      pOut[427] = fPmQx*pIn[427] + 8*i2ze*pIn[343];
      pOut[428] = fPmQx*pIn[428] + 2*i2ze*pIn[322];
      pOut[429] = fPmQx*pIn[429] + 2*i2ze*pIn[323];
      pOut[430] = fPmQx*pIn[430] + 7*i2ze*pIn[348];
      pOut[431] = fPmQx*pIn[431] + 3*i2ze*pIn[344];
      pOut[432] = fPmQx*pIn[432] + 7*i2ze*pIn[350];
      pOut[433] = fPmQx*pIn[433] + 3*i2ze*pIn[345];
      pOut[434] = fPmQx*pIn[434] + 2*i2ze*pIn[332];
      pOut[435] = fPmQx*pIn[435] + 2*i2ze*pIn[333];
      pOut[436] = fPmQx*pIn[436] + 6*i2ze*pIn[352];
      pOut[437] = fPmQx*pIn[437] + 6*i2ze*pIn[353];
      pOut[438] = fPmQx*pIn[438] + 4*i2ze*pIn[346];
      pOut[439] = fPmQx*pIn[439] + 4*i2ze*pIn[347];
      pOut[440] = fPmQx*pIn[440] + 2*i2ze*pIn[334];
      pOut[441] = fPmQx*pIn[441] + 2*i2ze*pIn[335];
      pOut[442] = fPmQx*pIn[442] + 5*i2ze*pIn[354];
      pOut[443] = fPmQx*pIn[443] + 5*i2ze*pIn[356];
      pOut[444] = fPmQx*pIn[444] + 2*i2ze*pIn[342];
      pOut[445] = fPmQx*pIn[445] + 3*i2ze*pIn[349];
      pOut[446] = fPmQx*pIn[446] + 3*i2ze*pIn[351];
      pOut[447] = fPmQx*pIn[447] + 6*i2ze*pIn[358];
      pOut[448] = fPmQx*pIn[448] + 5*i2ze*pIn[362];
      pOut[449] = fPmQx*pIn[449] + 3*i2ze*pIn[355];
      pOut[450] = fPmQx*pIn[450] + 5*i2ze*pIn[363];
      pOut[451] = fPmQx*pIn[451] + 3*i2ze*pIn[357];
      pOut[452] = fPmQx*pIn[452] + 4*i2ze*pIn[359];
      pOut[453] = fPmQx*pIn[453] + 4*i2ze*pIn[360];
      pOut[454] = fPmQx*pIn[454] + 4*i2ze*pIn[361];
      if (lab == 12) return;
   case 13:
      pOut[455] = fPmQx*pIn[455] + 13*i2ze*pIn[364];
      pOut[456] = fPmQx*pIn[456];
      pOut[457] = fPmQx*pIn[457];
      pOut[458] = fPmQx*pIn[458] + i2ze*pIn[365];
      pOut[459] = fPmQx*pIn[459] + i2ze*pIn[366];
      pOut[460] = fPmQx*pIn[460] + 12*i2ze*pIn[367];
      pOut[461] = fPmQx*pIn[461];
      pOut[462] = fPmQx*pIn[462] + 12*i2ze*pIn[369];
      pOut[463] = fPmQx*pIn[463];
      pOut[464] = fPmQx*pIn[464] + 11*i2ze*pIn[373];
      pOut[465] = fPmQx*pIn[465] + 11*i2ze*pIn[374];
      pOut[466] = fPmQx*pIn[466] + 2*i2ze*pIn[368];
      pOut[467] = fPmQx*pIn[467];
      pOut[468] = fPmQx*pIn[468] + 2*i2ze*pIn[370];
      pOut[469] = fPmQx*pIn[469];
      pOut[470] = fPmQx*pIn[470] + 3*i2ze*pIn[375];
      pOut[471] = fPmQx*pIn[471] + 3*i2ze*pIn[376];
      pOut[472] = fPmQx*pIn[472] + 10*i2ze*pIn[379];
      pOut[473] = fPmQx*pIn[473];
      pOut[474] = fPmQx*pIn[474] + 10*i2ze*pIn[381];
      pOut[475] = fPmQx*pIn[475];
      pOut[476] = fPmQx*pIn[476] + 9*i2ze*pIn[385];
      pOut[477] = fPmQx*pIn[477] + 9*i2ze*pIn[386];
      pOut[478] = fPmQx*pIn[478] + 4*i2ze*pIn[380];
      pOut[479] = fPmQx*pIn[479];
      pOut[480] = fPmQx*pIn[480] + 4*i2ze*pIn[382];
      pOut[481] = fPmQx*pIn[481];
      pOut[482] = fPmQx*pIn[482] + 5*i2ze*pIn[387];
      pOut[483] = fPmQx*pIn[483] + 5*i2ze*pIn[388];
      pOut[484] = fPmQx*pIn[484] + 8*i2ze*pIn[391];
      pOut[485] = fPmQx*pIn[485];
      pOut[486] = fPmQx*pIn[486] + 8*i2ze*pIn[393];
      pOut[487] = fPmQx*pIn[487];
      pOut[488] = fPmQx*pIn[488] + 7*i2ze*pIn[397];
      pOut[489] = fPmQx*pIn[489] + 7*i2ze*pIn[398];
      pOut[490] = fPmQx*pIn[490] + 6*i2ze*pIn[392];
      pOut[491] = fPmQx*pIn[491];
      pOut[492] = fPmQx*pIn[492] + 6*i2ze*pIn[394];
      pOut[493] = fPmQx*pIn[493];
      pOut[494] = fPmQx*pIn[494] + 11*i2ze*pIn[402];
      pOut[495] = fPmQx*pIn[495] + i2ze*pIn[371];
      pOut[496] = fPmQx*pIn[496] + i2ze*pIn[372];
      pOut[497] = fPmQx*pIn[497] + i2ze*pIn[377];
      pOut[498] = fPmQx*pIn[498] + i2ze*pIn[378];
      pOut[499] = fPmQx*pIn[499] + 10*i2ze*pIn[403];
      pOut[500] = fPmQx*pIn[500] + 2*i2ze*pIn[400];
      pOut[501] = fPmQx*pIn[501] + 10*i2ze*pIn[405];
      pOut[502] = fPmQx*pIn[502] + 2*i2ze*pIn[401];
      pOut[503] = fPmQx*pIn[503] + 9*i2ze*pIn[413];
      pOut[504] = fPmQx*pIn[504] + 9*i2ze*pIn[414];
      pOut[505] = fPmQx*pIn[505] + 3*i2ze*pIn[407];
      pOut[506] = fPmQx*pIn[506] + 3*i2ze*pIn[408];
      pOut[507] = fPmQx*pIn[507] + i2ze*pIn[383];
      pOut[508] = fPmQx*pIn[508] + i2ze*pIn[384];
      pOut[509] = fPmQx*pIn[509] + i2ze*pIn[389];
      pOut[510] = fPmQx*pIn[510] + i2ze*pIn[390];
      pOut[511] = fPmQx*pIn[511] + 8*i2ze*pIn[415];
      pOut[512] = fPmQx*pIn[512] + 4*i2ze*pIn[409];
      pOut[513] = fPmQx*pIn[513] + 8*i2ze*pIn[417];
      pOut[514] = fPmQx*pIn[514] + 4*i2ze*pIn[411];
      pOut[515] = fPmQx*pIn[515] + 7*i2ze*pIn[425];
      pOut[516] = fPmQx*pIn[516] + 7*i2ze*pIn[426];
      pOut[517] = fPmQx*pIn[517] + 5*i2ze*pIn[419];
      pOut[518] = fPmQx*pIn[518] + 5*i2ze*pIn[420];
      pOut[519] = fPmQx*pIn[519] + i2ze*pIn[395];
      pOut[520] = fPmQx*pIn[520] + i2ze*pIn[396];
      pOut[521] = fPmQx*pIn[521] + i2ze*pIn[399];
      pOut[522] = fPmQx*pIn[522] + 6*i2ze*pIn[421];
      pOut[523] = fPmQx*pIn[523] + 6*i2ze*pIn[423];
      pOut[524] = fPmQx*pIn[524] + 9*i2ze*pIn[427];
      pOut[525] = fPmQx*pIn[525] + 2*i2ze*pIn[404];
      pOut[526] = fPmQx*pIn[526] + 2*i2ze*pIn[406];
      pOut[527] = fPmQx*pIn[527] + 3*i2ze*pIn[428];
      pOut[528] = fPmQx*pIn[528] + 3*i2ze*pIn[429];
      pOut[529] = fPmQx*pIn[529] + 8*i2ze*pIn[430];
      pOut[530] = fPmQx*pIn[530] + 2*i2ze*pIn[410];
      pOut[531] = fPmQx*pIn[531] + 8*i2ze*pIn[432];
      pOut[532] = fPmQx*pIn[532] + 2*i2ze*pIn[412];
      pOut[533] = fPmQx*pIn[533] + 7*i2ze*pIn[436];
      pOut[534] = fPmQx*pIn[534] + 7*i2ze*pIn[437];
      pOut[535] = fPmQx*pIn[535] + 4*i2ze*pIn[431];
      pOut[536] = fPmQx*pIn[536] + 2*i2ze*pIn[416];
      pOut[537] = fPmQx*pIn[537] + 4*i2ze*pIn[433];
      pOut[538] = fPmQx*pIn[538] + 2*i2ze*pIn[418];
      pOut[539] = fPmQx*pIn[539] + 5*i2ze*pIn[438];
      pOut[540] = fPmQx*pIn[540] + 5*i2ze*pIn[439];
      pOut[541] = fPmQx*pIn[541] + 6*i2ze*pIn[442];
      pOut[542] = fPmQx*pIn[542] + 2*i2ze*pIn[422];
      pOut[543] = fPmQx*pIn[543] + 6*i2ze*pIn[443];
      pOut[544] = fPmQx*pIn[544] + 2*i2ze*pIn[424];
      pOut[545] = fPmQx*pIn[545] + 7*i2ze*pIn[447];
      pOut[546] = fPmQx*pIn[546] + 3*i2ze*pIn[434];
      pOut[547] = fPmQx*pIn[547] + 3*i2ze*pIn[435];
      pOut[548] = fPmQx*pIn[548] + 3*i2ze*pIn[440];
      pOut[549] = fPmQx*pIn[549] + 3*i2ze*pIn[441];
      pOut[550] = fPmQx*pIn[550] + 6*i2ze*pIn[448];
      pOut[551] = fPmQx*pIn[551] + 4*i2ze*pIn[445];
      pOut[552] = fPmQx*pIn[552] + 6*i2ze*pIn[450];
      pOut[553] = fPmQx*pIn[553] + 4*i2ze*pIn[446];
      pOut[554] = fPmQx*pIn[554] + 5*i2ze*pIn[452];
      pOut[555] = fPmQx*pIn[555] + 5*i2ze*pIn[453];
      pOut[556] = fPmQx*pIn[556] + 3*i2ze*pIn[444];
      pOut[557] = fPmQx*pIn[557] + 5*i2ze*pIn[454];
      pOut[558] = fPmQx*pIn[558] + 4*i2ze*pIn[449];
      pOut[559] = fPmQx*pIn[559] + 4*i2ze*pIn[451];
      if (lab == 13) return;
   case 14:
      pOut[560] = fPmQx*pIn[560] + 14*i2ze*pIn[455];
      pOut[561] = fPmQx*pIn[561];
      pOut[562] = fPmQx*pIn[562];
      pOut[563] = fPmQx*pIn[563] + 13*i2ze*pIn[460];
      pOut[564] = fPmQx*pIn[564] + i2ze*pIn[456];
      pOut[565] = fPmQx*pIn[565] + 13*i2ze*pIn[462];
      pOut[566] = fPmQx*pIn[566] + i2ze*pIn[457];
      pOut[567] = fPmQx*pIn[567];
      pOut[568] = fPmQx*pIn[568];
      pOut[569] = fPmQx*pIn[569] + 12*i2ze*pIn[464];
      pOut[570] = fPmQx*pIn[570] + 12*i2ze*pIn[465];
      pOut[571] = fPmQx*pIn[571] + 2*i2ze*pIn[458];
      pOut[572] = fPmQx*pIn[572] + 2*i2ze*pIn[459];
      pOut[573] = fPmQx*pIn[573];
      pOut[574] = fPmQx*pIn[574];
      pOut[575] = fPmQx*pIn[575] + 11*i2ze*pIn[472];
      pOut[576] = fPmQx*pIn[576] + 3*i2ze*pIn[466];
      pOut[577] = fPmQx*pIn[577] + 11*i2ze*pIn[474];
      pOut[578] = fPmQx*pIn[578] + 3*i2ze*pIn[468];
      pOut[579] = fPmQx*pIn[579];
      pOut[580] = fPmQx*pIn[580];
      pOut[581] = fPmQx*pIn[581] + 10*i2ze*pIn[476];
      pOut[582] = fPmQx*pIn[582] + 10*i2ze*pIn[477];
      pOut[583] = fPmQx*pIn[583] + 4*i2ze*pIn[470];
      pOut[584] = fPmQx*pIn[584] + 4*i2ze*pIn[471];
      pOut[585] = fPmQx*pIn[585];
      pOut[586] = fPmQx*pIn[586];
      pOut[587] = fPmQx*pIn[587] + 9*i2ze*pIn[484];
      pOut[588] = fPmQx*pIn[588] + 5*i2ze*pIn[478];
      pOut[589] = fPmQx*pIn[589] + 9*i2ze*pIn[486];
      pOut[590] = fPmQx*pIn[590] + 5*i2ze*pIn[480];
      pOut[591] = fPmQx*pIn[591];
      pOut[592] = fPmQx*pIn[592];
      pOut[593] = fPmQx*pIn[593] + 8*i2ze*pIn[488];
      pOut[594] = fPmQx*pIn[594] + 8*i2ze*pIn[489];
      pOut[595] = fPmQx*pIn[595] + 6*i2ze*pIn[482];
      pOut[596] = fPmQx*pIn[596] + 6*i2ze*pIn[483];
      pOut[597] = fPmQx*pIn[597];
      pOut[598] = fPmQx*pIn[598];
      pOut[599] = fPmQx*pIn[599] + 7*i2ze*pIn[490];
      pOut[600] = fPmQx*pIn[600] + 7*i2ze*pIn[492];
      pOut[601] = fPmQx*pIn[601];
      pOut[602] = fPmQx*pIn[602] + i2ze*pIn[461];
      pOut[603] = fPmQx*pIn[603] + i2ze*pIn[463];
      pOut[604] = fPmQx*pIn[604] + 12*i2ze*pIn[494];
      pOut[605] = fPmQx*pIn[605] + 11*i2ze*pIn[499];
      pOut[606] = fPmQx*pIn[606] + i2ze*pIn[467];
      pOut[607] = fPmQx*pIn[607] + 11*i2ze*pIn[501];
      pOut[608] = fPmQx*pIn[608] + i2ze*pIn[469];
      pOut[609] = fPmQx*pIn[609] + 2*i2ze*pIn[495];
      pOut[610] = fPmQx*pIn[610] + 2*i2ze*pIn[496];
      pOut[611] = fPmQx*pIn[611] + 3*i2ze*pIn[500];
      pOut[612] = fPmQx*pIn[612] + i2ze*pIn[473];
      pOut[613] = fPmQx*pIn[613] + 3*i2ze*pIn[502];
      pOut[614] = fPmQx*pIn[614] + i2ze*pIn[475];
      pOut[615] = fPmQx*pIn[615] + 10*i2ze*pIn[503];
      pOut[616] = fPmQx*pIn[616] + 10*i2ze*pIn[504];
      pOut[617] = fPmQx*pIn[617] + 9*i2ze*pIn[511];
      pOut[618] = fPmQx*pIn[618] + i2ze*pIn[479];
      pOut[619] = fPmQx*pIn[619] + 9*i2ze*pIn[513];
      pOut[620] = fPmQx*pIn[620] + i2ze*pIn[481];
      pOut[621] = fPmQx*pIn[621] + 4*i2ze*pIn[505];
      pOut[622] = fPmQx*pIn[622] + 4*i2ze*pIn[506];
      pOut[623] = fPmQx*pIn[623] + 5*i2ze*pIn[512];
      pOut[624] = fPmQx*pIn[624] + i2ze*pIn[485];
      pOut[625] = fPmQx*pIn[625] + 5*i2ze*pIn[514];
      pOut[626] = fPmQx*pIn[626] + i2ze*pIn[487];
      pOut[627] = fPmQx*pIn[627] + 8*i2ze*pIn[515];
      pOut[628] = fPmQx*pIn[628] + 8*i2ze*pIn[516];
      pOut[629] = fPmQx*pIn[629] + 7*i2ze*pIn[522];
      pOut[630] = fPmQx*pIn[630] + i2ze*pIn[491];
      pOut[631] = fPmQx*pIn[631] + 7*i2ze*pIn[523];
      pOut[632] = fPmQx*pIn[632] + i2ze*pIn[493];
      pOut[633] = fPmQx*pIn[633] + 6*i2ze*pIn[517];
      pOut[634] = fPmQx*pIn[634] + 6*i2ze*pIn[518];
      pOut[635] = fPmQx*pIn[635] + 10*i2ze*pIn[524];
      pOut[636] = fPmQx*pIn[636] + 2*i2ze*pIn[497];
      pOut[637] = fPmQx*pIn[637] + 2*i2ze*pIn[498];
      pOut[638] = fPmQx*pIn[638] + 9*i2ze*pIn[529];
      pOut[639] = fPmQx*pIn[639] + 3*i2ze*pIn[525];
      pOut[640] = fPmQx*pIn[640] + 9*i2ze*pIn[531];
      pOut[641] = fPmQx*pIn[641] + 3*i2ze*pIn[526];
      pOut[642] = fPmQx*pIn[642] + 2*i2ze*pIn[507];
      pOut[643] = fPmQx*pIn[643] + 2*i2ze*pIn[508];
      pOut[644] = fPmQx*pIn[644] + 8*i2ze*pIn[533];
      pOut[645] = fPmQx*pIn[645] + 8*i2ze*pIn[534];
      pOut[646] = fPmQx*pIn[646] + 4*i2ze*pIn[527];
      pOut[647] = fPmQx*pIn[647] + 4*i2ze*pIn[528];
      pOut[648] = fPmQx*pIn[648] + 2*i2ze*pIn[509];
      pOut[649] = fPmQx*pIn[649] + 2*i2ze*pIn[510];
      pOut[650] = fPmQx*pIn[650] + 7*i2ze*pIn[541];
      pOut[651] = fPmQx*pIn[651] + 5*i2ze*pIn[535];
      pOut[652] = fPmQx*pIn[652] + 7*i2ze*pIn[543];
      pOut[653] = fPmQx*pIn[653] + 5*i2ze*pIn[537];
      pOut[654] = fPmQx*pIn[654] + 2*i2ze*pIn[519];
      pOut[655] = fPmQx*pIn[655] + 2*i2ze*pIn[520];
      pOut[656] = fPmQx*pIn[656] + 6*i2ze*pIn[539];
      pOut[657] = fPmQx*pIn[657] + 6*i2ze*pIn[540];
      pOut[658] = fPmQx*pIn[658] + 2*i2ze*pIn[521];
      pOut[659] = fPmQx*pIn[659] + 3*i2ze*pIn[530];
      pOut[660] = fPmQx*pIn[660] + 3*i2ze*pIn[532];
      pOut[661] = fPmQx*pIn[661] + 8*i2ze*pIn[545];
      pOut[662] = fPmQx*pIn[662] + 7*i2ze*pIn[550];
      pOut[663] = fPmQx*pIn[663] + 3*i2ze*pIn[536];
      pOut[664] = fPmQx*pIn[664] + 7*i2ze*pIn[552];
      pOut[665] = fPmQx*pIn[665] + 3*i2ze*pIn[538];
      pOut[666] = fPmQx*pIn[666] + 4*i2ze*pIn[546];
      pOut[667] = fPmQx*pIn[667] + 4*i2ze*pIn[547];
      pOut[668] = fPmQx*pIn[668] + 5*i2ze*pIn[551];
      pOut[669] = fPmQx*pIn[669] + 3*i2ze*pIn[542];
      pOut[670] = fPmQx*pIn[670] + 5*i2ze*pIn[553];
      pOut[671] = fPmQx*pIn[671] + 3*i2ze*pIn[544];
      pOut[672] = fPmQx*pIn[672] + 6*i2ze*pIn[554];
      pOut[673] = fPmQx*pIn[673] + 6*i2ze*pIn[555];
      pOut[674] = fPmQx*pIn[674] + 6*i2ze*pIn[557];
      pOut[675] = fPmQx*pIn[675] + 4*i2ze*pIn[548];
      pOut[676] = fPmQx*pIn[676] + 4*i2ze*pIn[549];
      pOut[677] = fPmQx*pIn[677] + 5*i2ze*pIn[558];
      pOut[678] = fPmQx*pIn[678] + 5*i2ze*pIn[559];
      pOut[679] = fPmQx*pIn[679] + 4*i2ze*pIn[556];
      if (lab == 14) return;
   }
   assert(0);
}

// Calculate (a0|c)^m from (a0|c-1y)^{m+1}, for a = la..lab
// i = _x/_y/_z is the reduction direction (x,y,z);
// fPmQi = PmQ[i] * (rho/gamma); i2ze = .5/eta_ABC = .5/(alpha + beta + gamma)
// note: the la is not for performance reasons (would likely make the actual recurrence
//       slower for small lab) but for preventing overwriting lower components.
static void OsrrB_3cKerM_y(double *IR_RP pOut, double const *IR_RP pIn, double fPmQy, unsigned la, unsigned lab, double i2ze)
{
   switch(la) {
   case 0:
      pOut[0] = fPmQy*pIn[0];
      if (lab == 0) return;
   case 1:
      pOut[1] = fPmQy*pIn[1];
      pOut[2] = fPmQy*pIn[2] + i2ze*pIn[0];
      pOut[3] = fPmQy*pIn[3];
      if (lab == 1) return;
   case 2:
      pOut[4] = fPmQy*pIn[4];
      pOut[5] = fPmQy*pIn[5] + 2*i2ze*pIn[2];
      pOut[6] = fPmQy*pIn[6];
      pOut[7] = fPmQy*pIn[7] + i2ze*pIn[1];
      pOut[8] = fPmQy*pIn[8];
      pOut[9] = fPmQy*pIn[9] + i2ze*pIn[3];
      if (lab == 2) return;
   case 3:
      pOut[10] = fPmQy*pIn[10];
      pOut[11] = fPmQy*pIn[11] + 3*i2ze*pIn[5];
      pOut[12] = fPmQy*pIn[12];
      pOut[13] = fPmQy*pIn[13] + 2*i2ze*pIn[7];
      pOut[14] = fPmQy*pIn[14];
      pOut[15] = fPmQy*pIn[15] + i2ze*pIn[4];
      pOut[16] = fPmQy*pIn[16] + i2ze*pIn[6];
      pOut[17] = fPmQy*pIn[17];
      pOut[18] = fPmQy*pIn[18] + 2*i2ze*pIn[9];
      pOut[19] = fPmQy*pIn[19] + i2ze*pIn[8];
      if (lab == 3) return;
   case 4:
      pOut[20] = fPmQy*pIn[20];
      pOut[21] = fPmQy*pIn[21] + 4*i2ze*pIn[11];
      pOut[22] = fPmQy*pIn[22];
      pOut[23] = fPmQy*pIn[23] + i2ze*pIn[10];
      pOut[24] = fPmQy*pIn[24] + 3*i2ze*pIn[13];
      pOut[25] = fPmQy*pIn[25];
      pOut[26] = fPmQy*pIn[26];
      pOut[27] = fPmQy*pIn[27] + 3*i2ze*pIn[18];
      pOut[28] = fPmQy*pIn[28] + i2ze*pIn[12];
      pOut[29] = fPmQy*pIn[29] + 2*i2ze*pIn[15];
      pOut[30] = fPmQy*pIn[30];
      pOut[31] = fPmQy*pIn[31] + 2*i2ze*pIn[16];
      pOut[32] = fPmQy*pIn[32] + i2ze*pIn[14];
      pOut[33] = fPmQy*pIn[33] + 2*i2ze*pIn[19];
      pOut[34] = fPmQy*pIn[34] + i2ze*pIn[17];
      if (lab == 4) return;
   case 5:
      pOut[35] = fPmQy*pIn[35];
      pOut[36] = fPmQy*pIn[36] + 5*i2ze*pIn[21];
      pOut[37] = fPmQy*pIn[37];
      pOut[38] = fPmQy*pIn[38] + 4*i2ze*pIn[24];
      pOut[39] = fPmQy*pIn[39];
      pOut[40] = fPmQy*pIn[40] + i2ze*pIn[20];
      pOut[41] = fPmQy*pIn[41] + i2ze*pIn[22];
      pOut[42] = fPmQy*pIn[42];
      pOut[43] = fPmQy*pIn[43] + 4*i2ze*pIn[27];
      pOut[44] = fPmQy*pIn[44] + 2*i2ze*pIn[23];
      pOut[45] = fPmQy*pIn[45];
      pOut[46] = fPmQy*pIn[46] + 3*i2ze*pIn[29];
      pOut[47] = fPmQy*pIn[47] + 3*i2ze*pIn[31];
      pOut[48] = fPmQy*pIn[48];
      pOut[49] = fPmQy*pIn[49] + 2*i2ze*pIn[28];
      pOut[50] = fPmQy*pIn[50] + i2ze*pIn[25];
      pOut[51] = fPmQy*pIn[51] + 3*i2ze*pIn[33];
      pOut[52] = fPmQy*pIn[52] + i2ze*pIn[26];
      pOut[53] = fPmQy*pIn[53] + 2*i2ze*pIn[32];
      pOut[54] = fPmQy*pIn[54] + i2ze*pIn[30];
      pOut[55] = fPmQy*pIn[55] + 2*i2ze*pIn[34];
      if (lab == 5) return;
   case 6:
      pOut[56] = fPmQy*pIn[56];
      pOut[57] = fPmQy*pIn[57] + 6*i2ze*pIn[36];
      pOut[58] = fPmQy*pIn[58];
      pOut[59] = fPmQy*pIn[59] + i2ze*pIn[35];
      pOut[60] = fPmQy*pIn[60] + 5*i2ze*pIn[38];
      pOut[61] = fPmQy*pIn[61];
      pOut[62] = fPmQy*pIn[62];
      pOut[63] = fPmQy*pIn[63] + 5*i2ze*pIn[43];
      pOut[64] = fPmQy*pIn[64] + i2ze*pIn[37];
      pOut[65] = fPmQy*pIn[65] + 2*i2ze*pIn[40];
      pOut[66] = fPmQy*pIn[66];
      pOut[67] = fPmQy*pIn[67] + 4*i2ze*pIn[46];
      pOut[68] = fPmQy*pIn[68];
      pOut[69] = fPmQy*pIn[69] + 4*i2ze*pIn[47];
      pOut[70] = fPmQy*pIn[70] + 2*i2ze*pIn[41];
      pOut[71] = fPmQy*pIn[71] + 3*i2ze*pIn[44];
      pOut[72] = fPmQy*pIn[72];
      pOut[73] = fPmQy*pIn[73] + 3*i2ze*pIn[49];
      pOut[74] = fPmQy*pIn[74] + i2ze*pIn[39];
      pOut[75] = fPmQy*pIn[75] + 4*i2ze*pIn[51];
      pOut[76] = fPmQy*pIn[76] + i2ze*pIn[42];
      pOut[77] = fPmQy*pIn[77] + i2ze*pIn[45];
      pOut[78] = fPmQy*pIn[78] + 3*i2ze*pIn[53];
      pOut[79] = fPmQy*pIn[79] + 2*i2ze*pIn[50];
      pOut[80] = fPmQy*pIn[80] + 2*i2ze*pIn[52];
      pOut[81] = fPmQy*pIn[81] + 3*i2ze*pIn[55];
      pOut[82] = fPmQy*pIn[82] + i2ze*pIn[48];
      pOut[83] = fPmQy*pIn[83] + 2*i2ze*pIn[54];
      if (lab == 6) return;
   case 7:
      pOut[84] = fPmQy*pIn[84];
      pOut[85] = fPmQy*pIn[85] + 7*i2ze*pIn[57];
      pOut[86] = fPmQy*pIn[86];
      pOut[87] = fPmQy*pIn[87] + 6*i2ze*pIn[60];
      pOut[88] = fPmQy*pIn[88];
      pOut[89] = fPmQy*pIn[89] + i2ze*pIn[56];
      pOut[90] = fPmQy*pIn[90] + i2ze*pIn[58];
      pOut[91] = fPmQy*pIn[91];
      pOut[92] = fPmQy*pIn[92] + 6*i2ze*pIn[63];
      pOut[93] = fPmQy*pIn[93] + 2*i2ze*pIn[59];
      pOut[94] = fPmQy*pIn[94];
      pOut[95] = fPmQy*pIn[95] + 5*i2ze*pIn[67];
      pOut[96] = fPmQy*pIn[96] + 5*i2ze*pIn[69];
      pOut[97] = fPmQy*pIn[97];
      pOut[98] = fPmQy*pIn[98] + 2*i2ze*pIn[64];
      pOut[99] = fPmQy*pIn[99] + 4*i2ze*pIn[71];
      pOut[100] = fPmQy*pIn[100];
      pOut[101] = fPmQy*pIn[101] + 3*i2ze*pIn[65];
      pOut[102] = fPmQy*pIn[102] + 3*i2ze*pIn[70];
      pOut[103] = fPmQy*pIn[103];
      pOut[104] = fPmQy*pIn[104] + 4*i2ze*pIn[73];
      pOut[105] = fPmQy*pIn[105] + i2ze*pIn[61];
      pOut[106] = fPmQy*pIn[106] + 5*i2ze*pIn[75];
      pOut[107] = fPmQy*pIn[107] + i2ze*pIn[62];
      pOut[108] = fPmQy*pIn[108] + 4*i2ze*pIn[78];
      pOut[109] = fPmQy*pIn[109] + 2*i2ze*pIn[74];
      pOut[110] = fPmQy*pIn[110] + i2ze*pIn[66];
      pOut[111] = fPmQy*pIn[111] + i2ze*pIn[68];
      pOut[112] = fPmQy*pIn[112] + 2*i2ze*pIn[76];
      pOut[113] = fPmQy*pIn[113] + 4*i2ze*pIn[81];
      pOut[114] = fPmQy*pIn[114] + 3*i2ze*pIn[79];
      pOut[115] = fPmQy*pIn[115] + i2ze*pIn[72];
      pOut[116] = fPmQy*pIn[116] + 3*i2ze*pIn[80];
      pOut[117] = fPmQy*pIn[117] + 2*i2ze*pIn[77];
      pOut[118] = fPmQy*pIn[118] + 3*i2ze*pIn[83];
      pOut[119] = fPmQy*pIn[119] + 2*i2ze*pIn[82];
      if (lab == 7) return;
   case 8:
      pOut[120] = fPmQy*pIn[120];
      pOut[121] = fPmQy*pIn[121] + 8*i2ze*pIn[85];
      pOut[122] = fPmQy*pIn[122];
      pOut[123] = fPmQy*pIn[123] + i2ze*pIn[84];
      pOut[124] = fPmQy*pIn[124] + 7*i2ze*pIn[87];
      pOut[125] = fPmQy*pIn[125];
      pOut[126] = fPmQy*pIn[126];
      pOut[127] = fPmQy*pIn[127] + 7*i2ze*pIn[92];
      pOut[128] = fPmQy*pIn[128] + i2ze*pIn[86];
      pOut[129] = fPmQy*pIn[129] + 2*i2ze*pIn[89];
      pOut[130] = fPmQy*pIn[130];
      pOut[131] = fPmQy*pIn[131] + 6*i2ze*pIn[95];
      pOut[132] = fPmQy*pIn[132];
      pOut[133] = fPmQy*pIn[133] + 6*i2ze*pIn[96];
      pOut[134] = fPmQy*pIn[134] + 2*i2ze*pIn[90];
      pOut[135] = fPmQy*pIn[135] + 3*i2ze*pIn[93];
      pOut[136] = fPmQy*pIn[136] + 5*i2ze*pIn[99];
      pOut[137] = fPmQy*pIn[137];
      pOut[138] = fPmQy*pIn[138];
      pOut[139] = fPmQy*pIn[139] + 5*i2ze*pIn[104];
      pOut[140] = fPmQy*pIn[140] + 3*i2ze*pIn[98];
      pOut[141] = fPmQy*pIn[141] + 4*i2ze*pIn[101];
      pOut[142] = fPmQy*pIn[142];
      pOut[143] = fPmQy*pIn[143] + 4*i2ze*pIn[102];
      pOut[144] = fPmQy*pIn[144] + i2ze*pIn[88];
      pOut[145] = fPmQy*pIn[145] + 6*i2ze*pIn[106];
      pOut[146] = fPmQy*pIn[146] + i2ze*pIn[91];
      pOut[147] = fPmQy*pIn[147] + i2ze*pIn[94];
      pOut[148] = fPmQy*pIn[148] + 5*i2ze*pIn[108];
      pOut[149] = fPmQy*pIn[149] + 2*i2ze*pIn[105];
      pOut[150] = fPmQy*pIn[150] + 2*i2ze*pIn[107];
      pOut[151] = fPmQy*pIn[151] + 5*i2ze*pIn[113];
      pOut[152] = fPmQy*pIn[152] + i2ze*pIn[97];
      pOut[153] = fPmQy*pIn[153] + i2ze*pIn[100];
      pOut[154] = fPmQy*pIn[154] + 3*i2ze*pIn[109];
      pOut[155] = fPmQy*pIn[155] + 4*i2ze*pIn[114];
      pOut[156] = fPmQy*pIn[156] + 4*i2ze*pIn[116];
      pOut[157] = fPmQy*pIn[157] + 3*i2ze*pIn[112];
      pOut[158] = fPmQy*pIn[158] + i2ze*pIn[103];
      pOut[159] = fPmQy*pIn[159] + 2*i2ze*pIn[110];
      pOut[160] = fPmQy*pIn[160] + 4*i2ze*pIn[118];
      pOut[161] = fPmQy*pIn[161] + 2*i2ze*pIn[111];
      pOut[162] = fPmQy*pIn[162] + 3*i2ze*pIn[117];
      pOut[163] = fPmQy*pIn[163] + 2*i2ze*pIn[115];
      pOut[164] = fPmQy*pIn[164] + 3*i2ze*pIn[119];
      if (lab == 8) return;
   case 9:
      pOut[165] = fPmQy*pIn[165];
      pOut[166] = fPmQy*pIn[166] + 9*i2ze*pIn[121];
      pOut[167] = fPmQy*pIn[167];
      pOut[168] = fPmQy*pIn[168] + 8*i2ze*pIn[124];
      pOut[169] = fPmQy*pIn[169];
      pOut[170] = fPmQy*pIn[170] + i2ze*pIn[120];
      pOut[171] = fPmQy*pIn[171] + i2ze*pIn[122];
      pOut[172] = fPmQy*pIn[172];
      pOut[173] = fPmQy*pIn[173] + 8*i2ze*pIn[127];
      pOut[174] = fPmQy*pIn[174] + 2*i2ze*pIn[123];
      pOut[175] = fPmQy*pIn[175];
      pOut[176] = fPmQy*pIn[176] + 7*i2ze*pIn[131];
      pOut[177] = fPmQy*pIn[177] + 7*i2ze*pIn[133];
      pOut[178] = fPmQy*pIn[178];
      pOut[179] = fPmQy*pIn[179] + 2*i2ze*pIn[128];
      pOut[180] = fPmQy*pIn[180] + 6*i2ze*pIn[136];
      pOut[181] = fPmQy*pIn[181];
      pOut[182] = fPmQy*pIn[182] + 3*i2ze*pIn[129];
      pOut[183] = fPmQy*pIn[183] + 3*i2ze*pIn[134];
      pOut[184] = fPmQy*pIn[184];
      pOut[185] = fPmQy*pIn[185] + 6*i2ze*pIn[139];
      pOut[186] = fPmQy*pIn[186] + 4*i2ze*pIn[135];
      pOut[187] = fPmQy*pIn[187];
      pOut[188] = fPmQy*pIn[188] + 5*i2ze*pIn[141];
      pOut[189] = fPmQy*pIn[189] + 5*i2ze*pIn[143];
      pOut[190] = fPmQy*pIn[190];
      pOut[191] = fPmQy*pIn[191] + 4*i2ze*pIn[140];
      pOut[192] = fPmQy*pIn[192] + i2ze*pIn[125];
      pOut[193] = fPmQy*pIn[193] + 7*i2ze*pIn[145];
      pOut[194] = fPmQy*pIn[194] + i2ze*pIn[126];
      pOut[195] = fPmQy*pIn[195] + 6*i2ze*pIn[148];
      pOut[196] = fPmQy*pIn[196] + 2*i2ze*pIn[144];
      pOut[197] = fPmQy*pIn[197] + i2ze*pIn[130];
      pOut[198] = fPmQy*pIn[198] + i2ze*pIn[132];
      pOut[199] = fPmQy*pIn[199] + 2*i2ze*pIn[146];
      pOut[200] = fPmQy*pIn[200] + 6*i2ze*pIn[151];
      pOut[201] = fPmQy*pIn[201] + 3*i2ze*pIn[149];
      pOut[202] = fPmQy*pIn[202] + i2ze*pIn[137];
      pOut[203] = fPmQy*pIn[203] + 5*i2ze*pIn[155];
      pOut[204] = fPmQy*pIn[204] + i2ze*pIn[138];
      pOut[205] = fPmQy*pIn[205] + 5*i2ze*pIn[156];
      pOut[206] = fPmQy*pIn[206] + 3*i2ze*pIn[150];
      pOut[207] = fPmQy*pIn[207] + 4*i2ze*pIn[154];
      pOut[208] = fPmQy*pIn[208] + i2ze*pIn[142];
      pOut[209] = fPmQy*pIn[209] + 4*i2ze*pIn[157];
      pOut[210] = fPmQy*pIn[210] + 2*i2ze*pIn[147];
      pOut[211] = fPmQy*pIn[211] + 5*i2ze*pIn[160];
      pOut[212] = fPmQy*pIn[212] + 2*i2ze*pIn[152];
      pOut[213] = fPmQy*pIn[213] + 4*i2ze*pIn[162];
      pOut[214] = fPmQy*pIn[214] + 2*i2ze*pIn[153];
      pOut[215] = fPmQy*pIn[215] + 3*i2ze*pIn[159];
      pOut[216] = fPmQy*pIn[216] + 3*i2ze*pIn[161];
      pOut[217] = fPmQy*pIn[217] + 2*i2ze*pIn[158];
      pOut[218] = fPmQy*pIn[218] + 4*i2ze*pIn[164];
      pOut[219] = fPmQy*pIn[219] + 3*i2ze*pIn[163];
      if (lab == 9) return;
   case 10:
      pOut[220] = fPmQy*pIn[220];
      pOut[221] = fPmQy*pIn[221] + 10*i2ze*pIn[166];
      pOut[222] = fPmQy*pIn[222];
      pOut[223] = fPmQy*pIn[223] + i2ze*pIn[165];
      pOut[224] = fPmQy*pIn[224] + 9*i2ze*pIn[168];
      pOut[225] = fPmQy*pIn[225];
      pOut[226] = fPmQy*pIn[226];
      pOut[227] = fPmQy*pIn[227] + 9*i2ze*pIn[173];
      pOut[228] = fPmQy*pIn[228] + i2ze*pIn[167];
      pOut[229] = fPmQy*pIn[229] + 2*i2ze*pIn[170];
      pOut[230] = fPmQy*pIn[230];
      pOut[231] = fPmQy*pIn[231] + 8*i2ze*pIn[176];
      pOut[232] = fPmQy*pIn[232];
      pOut[233] = fPmQy*pIn[233] + 8*i2ze*pIn[177];
      pOut[234] = fPmQy*pIn[234] + 2*i2ze*pIn[171];
      pOut[235] = fPmQy*pIn[235] + 3*i2ze*pIn[174];
      pOut[236] = fPmQy*pIn[236] + 7*i2ze*pIn[180];
      pOut[237] = fPmQy*pIn[237];
      pOut[238] = fPmQy*pIn[238];
      pOut[239] = fPmQy*pIn[239] + 7*i2ze*pIn[185];
      pOut[240] = fPmQy*pIn[240] + 3*i2ze*pIn[179];
      pOut[241] = fPmQy*pIn[241] + 4*i2ze*pIn[182];
      pOut[242] = fPmQy*pIn[242];
      pOut[243] = fPmQy*pIn[243] + 6*i2ze*pIn[188];
      pOut[244] = fPmQy*pIn[244];
      pOut[245] = fPmQy*pIn[245] + 6*i2ze*pIn[189];
      pOut[246] = fPmQy*pIn[246] + 4*i2ze*pIn[183];
      pOut[247] = fPmQy*pIn[247] + 5*i2ze*pIn[186];
      pOut[248] = fPmQy*pIn[248];
      pOut[249] = fPmQy*pIn[249] + 5*i2ze*pIn[191];
      pOut[250] = fPmQy*pIn[250] + i2ze*pIn[169];
      pOut[251] = fPmQy*pIn[251] + 8*i2ze*pIn[193];
      pOut[252] = fPmQy*pIn[252] + i2ze*pIn[172];
      pOut[253] = fPmQy*pIn[253] + i2ze*pIn[175];
      pOut[254] = fPmQy*pIn[254] + 7*i2ze*pIn[195];
      pOut[255] = fPmQy*pIn[255] + 2*i2ze*pIn[192];
      pOut[256] = fPmQy*pIn[256] + 2*i2ze*pIn[194];
      pOut[257] = fPmQy*pIn[257] + 7*i2ze*pIn[200];
      pOut[258] = fPmQy*pIn[258] + i2ze*pIn[178];
      pOut[259] = fPmQy*pIn[259] + i2ze*pIn[181];
      pOut[260] = fPmQy*pIn[260] + 3*i2ze*pIn[196];
      pOut[261] = fPmQy*pIn[261] + 6*i2ze*pIn[203];
      pOut[262] = fPmQy*pIn[262] + 6*i2ze*pIn[205];
      pOut[263] = fPmQy*pIn[263] + 3*i2ze*pIn[199];
      pOut[264] = fPmQy*pIn[264] + i2ze*pIn[184];
      pOut[265] = fPmQy*pIn[265] + i2ze*pIn[187];
      pOut[266] = fPmQy*pIn[266] + 5*i2ze*pIn[207];
      pOut[267] = fPmQy*pIn[267] + 4*i2ze*pIn[201];
      pOut[268] = fPmQy*pIn[268] + 4*i2ze*pIn[206];
      pOut[269] = fPmQy*pIn[269] + 5*i2ze*pIn[209];
      pOut[270] = fPmQy*pIn[270] + i2ze*pIn[190];
      pOut[271] = fPmQy*pIn[271] + 2*i2ze*pIn[197];
      pOut[272] = fPmQy*pIn[272] + 6*i2ze*pIn[211];
      pOut[273] = fPmQy*pIn[273] + 2*i2ze*pIn[198];
      pOut[274] = fPmQy*pIn[274] + 3*i2ze*pIn[210];
      pOut[275] = fPmQy*pIn[275] + 5*i2ze*pIn[213];
      pOut[276] = fPmQy*pIn[276] + 2*i2ze*pIn[202];
      pOut[277] = fPmQy*pIn[277] + 2*i2ze*pIn[204];
      pOut[278] = fPmQy*pIn[278] + 5*i2ze*pIn[218];
      pOut[279] = fPmQy*pIn[279] + 3*i2ze*pIn[212];
      pOut[280] = fPmQy*pIn[280] + 4*i2ze*pIn[215];
      pOut[281] = fPmQy*pIn[281] + 2*i2ze*pIn[208];
      pOut[282] = fPmQy*pIn[282] + 4*i2ze*pIn[216];
      pOut[283] = fPmQy*pIn[283] + 3*i2ze*pIn[214];
      pOut[284] = fPmQy*pIn[284] + 4*i2ze*pIn[219];
      pOut[285] = fPmQy*pIn[285] + 3*i2ze*pIn[217];
      if (lab == 10) return;
   case 11:
      pOut[286] = fPmQy*pIn[286];
      pOut[287] = fPmQy*pIn[287] + 11*i2ze*pIn[221];
      pOut[288] = fPmQy*pIn[288];
      pOut[289] = fPmQy*pIn[289] + 10*i2ze*pIn[224];
      pOut[290] = fPmQy*pIn[290];
      pOut[291] = fPmQy*pIn[291] + i2ze*pIn[220];
      pOut[292] = fPmQy*pIn[292] + i2ze*pIn[222];
      pOut[293] = fPmQy*pIn[293];
      pOut[294] = fPmQy*pIn[294] + 10*i2ze*pIn[227];
      pOut[295] = fPmQy*pIn[295] + 2*i2ze*pIn[223];
      pOut[296] = fPmQy*pIn[296];
      pOut[297] = fPmQy*pIn[297] + 9*i2ze*pIn[231];
      pOut[298] = fPmQy*pIn[298] + 9*i2ze*pIn[233];
      pOut[299] = fPmQy*pIn[299];
      pOut[300] = fPmQy*pIn[300] + 2*i2ze*pIn[228];
      pOut[301] = fPmQy*pIn[301] + 8*i2ze*pIn[236];
      pOut[302] = fPmQy*pIn[302];
      pOut[303] = fPmQy*pIn[303] + 3*i2ze*pIn[229];
      pOut[304] = fPmQy*pIn[304] + 3*i2ze*pIn[234];
      pOut[305] = fPmQy*pIn[305];
      pOut[306] = fPmQy*pIn[306] + 8*i2ze*pIn[239];
      pOut[307] = fPmQy*pIn[307] + 4*i2ze*pIn[235];
      pOut[308] = fPmQy*pIn[308];
      pOut[309] = fPmQy*pIn[309] + 7*i2ze*pIn[243];
      pOut[310] = fPmQy*pIn[310] + 7*i2ze*pIn[245];
      pOut[311] = fPmQy*pIn[311];
      pOut[312] = fPmQy*pIn[312] + 4*i2ze*pIn[240];
      pOut[313] = fPmQy*pIn[313] + 6*i2ze*pIn[247];
      pOut[314] = fPmQy*pIn[314];
      pOut[315] = fPmQy*pIn[315] + 5*i2ze*pIn[241];
      pOut[316] = fPmQy*pIn[316] + 5*i2ze*pIn[246];
      pOut[317] = fPmQy*pIn[317];
      pOut[318] = fPmQy*pIn[318] + 6*i2ze*pIn[249];
      pOut[319] = fPmQy*pIn[319] + i2ze*pIn[225];
      pOut[320] = fPmQy*pIn[320] + 9*i2ze*pIn[251];
      pOut[321] = fPmQy*pIn[321] + i2ze*pIn[226];
      pOut[322] = fPmQy*pIn[322] + 8*i2ze*pIn[254];
      pOut[323] = fPmQy*pIn[323] + 2*i2ze*pIn[250];
      pOut[324] = fPmQy*pIn[324] + i2ze*pIn[230];
      pOut[325] = fPmQy*pIn[325] + i2ze*pIn[232];
      pOut[326] = fPmQy*pIn[326] + 2*i2ze*pIn[252];
      pOut[327] = fPmQy*pIn[327] + 8*i2ze*pIn[257];
      pOut[328] = fPmQy*pIn[328] + 3*i2ze*pIn[255];
      pOut[329] = fPmQy*pIn[329] + i2ze*pIn[237];
      pOut[330] = fPmQy*pIn[330] + 7*i2ze*pIn[261];
      pOut[331] = fPmQy*pIn[331] + i2ze*pIn[238];
      pOut[332] = fPmQy*pIn[332] + 7*i2ze*pIn[262];
      pOut[333] = fPmQy*pIn[333] + 3*i2ze*pIn[256];
      pOut[334] = fPmQy*pIn[334] + 6*i2ze*pIn[266];
      pOut[335] = fPmQy*pIn[335] + 4*i2ze*pIn[260];
      pOut[336] = fPmQy*pIn[336] + i2ze*pIn[242];
      pOut[337] = fPmQy*pIn[337] + i2ze*pIn[244];
      pOut[338] = fPmQy*pIn[338] + 4*i2ze*pIn[263];
      pOut[339] = fPmQy*pIn[339] + 6*i2ze*pIn[269];
      pOut[340] = fPmQy*pIn[340] + 5*i2ze*pIn[267];
      pOut[341] = fPmQy*pIn[341] + i2ze*pIn[248];
      pOut[342] = fPmQy*pIn[342] + 5*i2ze*pIn[268];
      pOut[343] = fPmQy*pIn[343] + 2*i2ze*pIn[253];
      pOut[344] = fPmQy*pIn[344] + 7*i2ze*pIn[272];
      pOut[345] = fPmQy*pIn[345] + 2*i2ze*pIn[258];
      pOut[346] = fPmQy*pIn[346] + 6*i2ze*pIn[275];
      pOut[347] = fPmQy*pIn[347] + 2*i2ze*pIn[259];
      pOut[348] = fPmQy*pIn[348] + 3*i2ze*pIn[271];
      pOut[349] = fPmQy*pIn[349] + 3*i2ze*pIn[273];
      pOut[350] = fPmQy*pIn[350] + 2*i2ze*pIn[264];
      pOut[351] = fPmQy*pIn[351] + 6*i2ze*pIn[278];
      pOut[352] = fPmQy*pIn[352] + 4*i2ze*pIn[274];
      pOut[353] = fPmQy*pIn[353] + 2*i2ze*pIn[265];
      pOut[354] = fPmQy*pIn[354] + 5*i2ze*pIn[280];
      pOut[355] = fPmQy*pIn[355] + 5*i2ze*pIn[282];
      pOut[356] = fPmQy*pIn[356] + 2*i2ze*pIn[270];
      pOut[357] = fPmQy*pIn[357] + 4*i2ze*pIn[279];
      pOut[358] = fPmQy*pIn[358] + 3*i2ze*pIn[276];
      pOut[359] = fPmQy*pIn[359] + 5*i2ze*pIn[284];
      pOut[360] = fPmQy*pIn[360] + 3*i2ze*pIn[277];
      pOut[361] = fPmQy*pIn[361] + 4*i2ze*pIn[283];
      pOut[362] = fPmQy*pIn[362] + 3*i2ze*pIn[281];
      pOut[363] = fPmQy*pIn[363] + 4*i2ze*pIn[285];
      if (lab == 11) return;
   case 12:
      pOut[364] = fPmQy*pIn[364];
      pOut[365] = fPmQy*pIn[365] + 12*i2ze*pIn[287];
      pOut[366] = fPmQy*pIn[366];
      pOut[367] = fPmQy*pIn[367] + i2ze*pIn[286];
      pOut[368] = fPmQy*pIn[368] + 11*i2ze*pIn[289];
      pOut[369] = fPmQy*pIn[369];
      pOut[370] = fPmQy*pIn[370];
      pOut[371] = fPmQy*pIn[371] + 11*i2ze*pIn[294];
      pOut[372] = fPmQy*pIn[372] + i2ze*pIn[288];
      pOut[373] = fPmQy*pIn[373] + 2*i2ze*pIn[291];
      pOut[374] = fPmQy*pIn[374];
      pOut[375] = fPmQy*pIn[375] + 10*i2ze*pIn[297];
      pOut[376] = fPmQy*pIn[376];
      pOut[377] = fPmQy*pIn[377] + 10*i2ze*pIn[298];
      pOut[378] = fPmQy*pIn[378] + 2*i2ze*pIn[292];
      pOut[379] = fPmQy*pIn[379] + 3*i2ze*pIn[295];
      pOut[380] = fPmQy*pIn[380] + 9*i2ze*pIn[301];
      pOut[381] = fPmQy*pIn[381];
      pOut[382] = fPmQy*pIn[382];
      pOut[383] = fPmQy*pIn[383] + 9*i2ze*pIn[306];
      pOut[384] = fPmQy*pIn[384] + 3*i2ze*pIn[300];
      pOut[385] = fPmQy*pIn[385] + 4*i2ze*pIn[303];
      pOut[386] = fPmQy*pIn[386];
      pOut[387] = fPmQy*pIn[387] + 8*i2ze*pIn[309];
      pOut[388] = fPmQy*pIn[388];
      pOut[389] = fPmQy*pIn[389] + 8*i2ze*pIn[310];
      pOut[390] = fPmQy*pIn[390] + 4*i2ze*pIn[304];
      pOut[391] = fPmQy*pIn[391] + 5*i2ze*pIn[307];
      pOut[392] = fPmQy*pIn[392] + 7*i2ze*pIn[313];
      pOut[393] = fPmQy*pIn[393];
      pOut[394] = fPmQy*pIn[394];
      pOut[395] = fPmQy*pIn[395] + 7*i2ze*pIn[318];
      pOut[396] = fPmQy*pIn[396] + 5*i2ze*pIn[312];
      pOut[397] = fPmQy*pIn[397] + 6*i2ze*pIn[315];
      pOut[398] = fPmQy*pIn[398];
      pOut[399] = fPmQy*pIn[399] + 6*i2ze*pIn[316];
      pOut[400] = fPmQy*pIn[400] + i2ze*pIn[290];
      pOut[401] = fPmQy*pIn[401] + 10*i2ze*pIn[320];
      pOut[402] = fPmQy*pIn[402] + i2ze*pIn[293];
      pOut[403] = fPmQy*pIn[403] + i2ze*pIn[296];
      pOut[404] = fPmQy*pIn[404] + 9*i2ze*pIn[322];
      pOut[405] = fPmQy*pIn[405] + 2*i2ze*pIn[319];
      pOut[406] = fPmQy*pIn[406] + 2*i2ze*pIn[321];
      pOut[407] = fPmQy*pIn[407] + 9*i2ze*pIn[327];
      pOut[408] = fPmQy*pIn[408] + i2ze*pIn[299];
      pOut[409] = fPmQy*pIn[409] + i2ze*pIn[302];
      pOut[410] = fPmQy*pIn[410] + 3*i2ze*pIn[323];
      pOut[411] = fPmQy*pIn[411] + 8*i2ze*pIn[330];
      pOut[412] = fPmQy*pIn[412] + 8*i2ze*pIn[332];
      pOut[413] = fPmQy*pIn[413] + 3*i2ze*pIn[326];
      pOut[414] = fPmQy*pIn[414] + i2ze*pIn[305];
      pOut[415] = fPmQy*pIn[415] + i2ze*pIn[308];
      pOut[416] = fPmQy*pIn[416] + 7*i2ze*pIn[334];
      pOut[417] = fPmQy*pIn[417] + 4*i2ze*pIn[328];
      pOut[418] = fPmQy*pIn[418] + 4*i2ze*pIn[333];
      pOut[419] = fPmQy*pIn[419] + 7*i2ze*pIn[339];
      pOut[420] = fPmQy*pIn[420] + i2ze*pIn[311];
      pOut[421] = fPmQy*pIn[421] + i2ze*pIn[314];
      pOut[422] = fPmQy*pIn[422] + 5*i2ze*pIn[335];
      pOut[423] = fPmQy*pIn[423] + 6*i2ze*pIn[340];
      pOut[424] = fPmQy*pIn[424] + 6*i2ze*pIn[342];
      pOut[425] = fPmQy*pIn[425] + 5*i2ze*pIn[338];
      pOut[426] = fPmQy*pIn[426] + i2ze*pIn[317];
      pOut[427] = fPmQy*pIn[427] + 2*i2ze*pIn[324];
      pOut[428] = fPmQy*pIn[428] + 8*i2ze*pIn[344];
      pOut[429] = fPmQy*pIn[429] + 2*i2ze*pIn[325];
      pOut[430] = fPmQy*pIn[430] + 3*i2ze*pIn[343];
      pOut[431] = fPmQy*pIn[431] + 7*i2ze*pIn[346];
      pOut[432] = fPmQy*pIn[432] + 2*i2ze*pIn[329];
      pOut[433] = fPmQy*pIn[433] + 2*i2ze*pIn[331];
      pOut[434] = fPmQy*pIn[434] + 7*i2ze*pIn[351];
      pOut[435] = fPmQy*pIn[435] + 3*i2ze*pIn[345];
      pOut[436] = fPmQy*pIn[436] + 4*i2ze*pIn[348];
      pOut[437] = fPmQy*pIn[437] + 2*i2ze*pIn[336];
      pOut[438] = fPmQy*pIn[438] + 6*i2ze*pIn[354];
      pOut[439] = fPmQy*pIn[439] + 2*i2ze*pIn[337];
      pOut[440] = fPmQy*pIn[440] + 6*i2ze*pIn[355];
      pOut[441] = fPmQy*pIn[441] + 4*i2ze*pIn[349];
      pOut[442] = fPmQy*pIn[442] + 5*i2ze*pIn[352];
      pOut[443] = fPmQy*pIn[443] + 2*i2ze*pIn[341];
      pOut[444] = fPmQy*pIn[444] + 5*i2ze*pIn[357];
      pOut[445] = fPmQy*pIn[445] + 3*i2ze*pIn[347];
      pOut[446] = fPmQy*pIn[446] + 6*i2ze*pIn[359];
      pOut[447] = fPmQy*pIn[447] + 3*i2ze*pIn[350];
      pOut[448] = fPmQy*pIn[448] + 3*i2ze*pIn[353];
      pOut[449] = fPmQy*pIn[449] + 5*i2ze*pIn[361];
      pOut[450] = fPmQy*pIn[450] + 4*i2ze*pIn[358];
      pOut[451] = fPmQy*pIn[451] + 4*i2ze*pIn[360];
      pOut[452] = fPmQy*pIn[452] + 5*i2ze*pIn[363];
      pOut[453] = fPmQy*pIn[453] + 3*i2ze*pIn[356];
      pOut[454] = fPmQy*pIn[454] + 4*i2ze*pIn[362];
      if (lab == 12) return;
   case 13:
      pOut[455] = fPmQy*pIn[455];
      pOut[456] = fPmQy*pIn[456] + 13*i2ze*pIn[365];
      pOut[457] = fPmQy*pIn[457];
      pOut[458] = fPmQy*pIn[458] + 12*i2ze*pIn[368];
      pOut[459] = fPmQy*pIn[459];
      pOut[460] = fPmQy*pIn[460] + i2ze*pIn[364];
      pOut[461] = fPmQy*pIn[461] + i2ze*pIn[366];
      pOut[462] = fPmQy*pIn[462];
      pOut[463] = fPmQy*pIn[463] + 12*i2ze*pIn[371];
      pOut[464] = fPmQy*pIn[464] + 2*i2ze*pIn[367];
      pOut[465] = fPmQy*pIn[465];
      pOut[466] = fPmQy*pIn[466] + 11*i2ze*pIn[375];
      pOut[467] = fPmQy*pIn[467] + 11*i2ze*pIn[377];
      pOut[468] = fPmQy*pIn[468];
      pOut[469] = fPmQy*pIn[469] + 2*i2ze*pIn[372];
      pOut[470] = fPmQy*pIn[470] + 10*i2ze*pIn[380];
      pOut[471] = fPmQy*pIn[471];
      pOut[472] = fPmQy*pIn[472] + 3*i2ze*pIn[373];
      pOut[473] = fPmQy*pIn[473] + 3*i2ze*pIn[378];
      pOut[474] = fPmQy*pIn[474];
      pOut[475] = fPmQy*pIn[475] + 10*i2ze*pIn[383];
      pOut[476] = fPmQy*pIn[476] + 4*i2ze*pIn[379];
      pOut[477] = fPmQy*pIn[477];
      pOut[478] = fPmQy*pIn[478] + 9*i2ze*pIn[387];
      pOut[479] = fPmQy*pIn[479] + 9*i2ze*pIn[389];
      pOut[480] = fPmQy*pIn[480];
      pOut[481] = fPmQy*pIn[481] + 4*i2ze*pIn[384];
      pOut[482] = fPmQy*pIn[482] + 8*i2ze*pIn[392];
      pOut[483] = fPmQy*pIn[483];
      pOut[484] = fPmQy*pIn[484] + 5*i2ze*pIn[385];
      pOut[485] = fPmQy*pIn[485] + 5*i2ze*pIn[390];
      pOut[486] = fPmQy*pIn[486];
      pOut[487] = fPmQy*pIn[487] + 8*i2ze*pIn[395];
      pOut[488] = fPmQy*pIn[488] + 6*i2ze*pIn[391];
      pOut[489] = fPmQy*pIn[489];
      pOut[490] = fPmQy*pIn[490] + 7*i2ze*pIn[397];
      pOut[491] = fPmQy*pIn[491] + 7*i2ze*pIn[399];
      pOut[492] = fPmQy*pIn[492];
      pOut[493] = fPmQy*pIn[493] + 6*i2ze*pIn[396];
      pOut[494] = fPmQy*pIn[494] + i2ze*pIn[369];
      pOut[495] = fPmQy*pIn[495] + 11*i2ze*pIn[401];
      pOut[496] = fPmQy*pIn[496] + i2ze*pIn[370];
      pOut[497] = fPmQy*pIn[497] + 10*i2ze*pIn[404];
      pOut[498] = fPmQy*pIn[498] + 2*i2ze*pIn[400];
      pOut[499] = fPmQy*pIn[499] + i2ze*pIn[374];
      pOut[500] = fPmQy*pIn[500] + i2ze*pIn[376];
      pOut[501] = fPmQy*pIn[501] + 2*i2ze*pIn[402];
      pOut[502] = fPmQy*pIn[502] + 10*i2ze*pIn[407];
      pOut[503] = fPmQy*pIn[503] + 3*i2ze*pIn[405];
      pOut[504] = fPmQy*pIn[504] + i2ze*pIn[381];
      pOut[505] = fPmQy*pIn[505] + 9*i2ze*pIn[411];
      pOut[506] = fPmQy*pIn[506] + i2ze*pIn[382];
      pOut[507] = fPmQy*pIn[507] + 9*i2ze*pIn[412];
      pOut[508] = fPmQy*pIn[508] + 3*i2ze*pIn[406];
      pOut[509] = fPmQy*pIn[509] + 8*i2ze*pIn[416];
      pOut[510] = fPmQy*pIn[510] + 4*i2ze*pIn[410];
      pOut[511] = fPmQy*pIn[511] + i2ze*pIn[386];
      pOut[512] = fPmQy*pIn[512] + i2ze*pIn[388];
      pOut[513] = fPmQy*pIn[513] + 4*i2ze*pIn[413];
      pOut[514] = fPmQy*pIn[514] + 8*i2ze*pIn[419];
      pOut[515] = fPmQy*pIn[515] + 5*i2ze*pIn[417];
      pOut[516] = fPmQy*pIn[516] + i2ze*pIn[393];
      pOut[517] = fPmQy*pIn[517] + 7*i2ze*pIn[423];
      pOut[518] = fPmQy*pIn[518] + i2ze*pIn[394];
      pOut[519] = fPmQy*pIn[519] + 7*i2ze*pIn[424];
      pOut[520] = fPmQy*pIn[520] + 5*i2ze*pIn[418];
      pOut[521] = fPmQy*pIn[521] + 6*i2ze*pIn[422];
      pOut[522] = fPmQy*pIn[522] + i2ze*pIn[398];
      pOut[523] = fPmQy*pIn[523] + 6*i2ze*pIn[425];
      pOut[524] = fPmQy*pIn[524] + 2*i2ze*pIn[403];
      pOut[525] = fPmQy*pIn[525] + 9*i2ze*pIn[428];
      pOut[526] = fPmQy*pIn[526] + 2*i2ze*pIn[408];
      pOut[527] = fPmQy*pIn[527] + 8*i2ze*pIn[431];
      pOut[528] = fPmQy*pIn[528] + 2*i2ze*pIn[409];
      pOut[529] = fPmQy*pIn[529] + 3*i2ze*pIn[427];
      pOut[530] = fPmQy*pIn[530] + 3*i2ze*pIn[429];
      pOut[531] = fPmQy*pIn[531] + 2*i2ze*pIn[414];
      pOut[532] = fPmQy*pIn[532] + 8*i2ze*pIn[434];
      pOut[533] = fPmQy*pIn[533] + 4*i2ze*pIn[430];
      pOut[534] = fPmQy*pIn[534] + 2*i2ze*pIn[415];
      pOut[535] = fPmQy*pIn[535] + 7*i2ze*pIn[438];
      pOut[536] = fPmQy*pIn[536] + 7*i2ze*pIn[440];
      pOut[537] = fPmQy*pIn[537] + 2*i2ze*pIn[420];
      pOut[538] = fPmQy*pIn[538] + 4*i2ze*pIn[435];
      pOut[539] = fPmQy*pIn[539] + 6*i2ze*pIn[442];
      pOut[540] = fPmQy*pIn[540] + 2*i2ze*pIn[421];
      pOut[541] = fPmQy*pIn[541] + 5*i2ze*pIn[436];
      pOut[542] = fPmQy*pIn[542] + 5*i2ze*pIn[441];
      pOut[543] = fPmQy*pIn[543] + 2*i2ze*pIn[426];
      pOut[544] = fPmQy*pIn[544] + 6*i2ze*pIn[444];
      pOut[545] = fPmQy*pIn[545] + 3*i2ze*pIn[432];
      pOut[546] = fPmQy*pIn[546] + 7*i2ze*pIn[446];
      pOut[547] = fPmQy*pIn[547] + 3*i2ze*pIn[433];
      pOut[548] = fPmQy*pIn[548] + 6*i2ze*pIn[449];
      pOut[549] = fPmQy*pIn[549] + 4*i2ze*pIn[445];
      pOut[550] = fPmQy*pIn[550] + 3*i2ze*pIn[437];
      pOut[551] = fPmQy*pIn[551] + 3*i2ze*pIn[439];
      pOut[552] = fPmQy*pIn[552] + 4*i2ze*pIn[447];
      pOut[553] = fPmQy*pIn[553] + 6*i2ze*pIn[452];
      pOut[554] = fPmQy*pIn[554] + 5*i2ze*pIn[450];
      pOut[555] = fPmQy*pIn[555] + 3*i2ze*pIn[443];
      pOut[556] = fPmQy*pIn[556] + 5*i2ze*pIn[451];
      pOut[557] = fPmQy*pIn[557] + 4*i2ze*pIn[448];
      pOut[558] = fPmQy*pIn[558] + 5*i2ze*pIn[454];
      pOut[559] = fPmQy*pIn[559] + 4*i2ze*pIn[453];
      if (lab == 13) return;
   case 14:
      pOut[560] = fPmQy*pIn[560];
      pOut[561] = fPmQy*pIn[561] + 14*i2ze*pIn[456];
      pOut[562] = fPmQy*pIn[562];
      pOut[563] = fPmQy*pIn[563] + i2ze*pIn[455];
      pOut[564] = fPmQy*pIn[564] + 13*i2ze*pIn[458];
      pOut[565] = fPmQy*pIn[565];
      pOut[566] = fPmQy*pIn[566];
      pOut[567] = fPmQy*pIn[567] + 13*i2ze*pIn[463];
      pOut[568] = fPmQy*pIn[568] + i2ze*pIn[457];
      pOut[569] = fPmQy*pIn[569] + 2*i2ze*pIn[460];
      pOut[570] = fPmQy*pIn[570];
      pOut[571] = fPmQy*pIn[571] + 12*i2ze*pIn[466];
      pOut[572] = fPmQy*pIn[572];
      pOut[573] = fPmQy*pIn[573] + 12*i2ze*pIn[467];
      pOut[574] = fPmQy*pIn[574] + 2*i2ze*pIn[461];
      pOut[575] = fPmQy*pIn[575] + 3*i2ze*pIn[464];
      pOut[576] = fPmQy*pIn[576] + 11*i2ze*pIn[470];
      pOut[577] = fPmQy*pIn[577];
      pOut[578] = fPmQy*pIn[578];
      pOut[579] = fPmQy*pIn[579] + 11*i2ze*pIn[475];
      pOut[580] = fPmQy*pIn[580] + 3*i2ze*pIn[469];
      pOut[581] = fPmQy*pIn[581] + 4*i2ze*pIn[472];
      pOut[582] = fPmQy*pIn[582];
      pOut[583] = fPmQy*pIn[583] + 10*i2ze*pIn[478];
      pOut[584] = fPmQy*pIn[584];
      pOut[585] = fPmQy*pIn[585] + 10*i2ze*pIn[479];
      pOut[586] = fPmQy*pIn[586] + 4*i2ze*pIn[473];
      pOut[587] = fPmQy*pIn[587] + 5*i2ze*pIn[476];
      pOut[588] = fPmQy*pIn[588] + 9*i2ze*pIn[482];
      pOut[589] = fPmQy*pIn[589];
      pOut[590] = fPmQy*pIn[590];
      pOut[591] = fPmQy*pIn[591] + 9*i2ze*pIn[487];
      pOut[592] = fPmQy*pIn[592] + 5*i2ze*pIn[481];
      pOut[593] = fPmQy*pIn[593] + 6*i2ze*pIn[484];
      pOut[594] = fPmQy*pIn[594];
      pOut[595] = fPmQy*pIn[595] + 8*i2ze*pIn[490];
      pOut[596] = fPmQy*pIn[596];
      pOut[597] = fPmQy*pIn[597] + 8*i2ze*pIn[491];
      pOut[598] = fPmQy*pIn[598] + 6*i2ze*pIn[485];
      pOut[599] = fPmQy*pIn[599] + 7*i2ze*pIn[488];
      pOut[600] = fPmQy*pIn[600];
      pOut[601] = fPmQy*pIn[601] + 7*i2ze*pIn[493];
      pOut[602] = fPmQy*pIn[602] + i2ze*pIn[459];
      pOut[603] = fPmQy*pIn[603] + 12*i2ze*pIn[495];
      pOut[604] = fPmQy*pIn[604] + i2ze*pIn[462];
      pOut[605] = fPmQy*pIn[605] + i2ze*pIn[465];
      pOut[606] = fPmQy*pIn[606] + 11*i2ze*pIn[497];
      pOut[607] = fPmQy*pIn[607] + 2*i2ze*pIn[494];
      pOut[608] = fPmQy*pIn[608] + 2*i2ze*pIn[496];
      pOut[609] = fPmQy*pIn[609] + 11*i2ze*pIn[502];
      pOut[610] = fPmQy*pIn[610] + i2ze*pIn[468];
      pOut[611] = fPmQy*pIn[611] + i2ze*pIn[471];
      pOut[612] = fPmQy*pIn[612] + 3*i2ze*pIn[498];
      pOut[613] = fPmQy*pIn[613] + 10*i2ze*pIn[505];
      pOut[614] = fPmQy*pIn[614] + 10*i2ze*pIn[507];
      pOut[615] = fPmQy*pIn[615] + 3*i2ze*pIn[501];
      pOut[616] = fPmQy*pIn[616] + i2ze*pIn[474];
      pOut[617] = fPmQy*pIn[617] + i2ze*pIn[477];
      pOut[618] = fPmQy*pIn[618] + 9*i2ze*pIn[509];
      pOut[619] = fPmQy*pIn[619] + 4*i2ze*pIn[503];
      pOut[620] = fPmQy*pIn[620] + 4*i2ze*pIn[508];
      pOut[621] = fPmQy*pIn[621] + 9*i2ze*pIn[514];
      pOut[622] = fPmQy*pIn[622] + i2ze*pIn[480];
      pOut[623] = fPmQy*pIn[623] + i2ze*pIn[483];
      pOut[624] = fPmQy*pIn[624] + 5*i2ze*pIn[510];
      pOut[625] = fPmQy*pIn[625] + 8*i2ze*pIn[517];
      pOut[626] = fPmQy*pIn[626] + 8*i2ze*pIn[519];
      pOut[627] = fPmQy*pIn[627] + 5*i2ze*pIn[513];
      pOut[628] = fPmQy*pIn[628] + i2ze*pIn[486];
      pOut[629] = fPmQy*pIn[629] + i2ze*pIn[489];
      pOut[630] = fPmQy*pIn[630] + 7*i2ze*pIn[521];
      pOut[631] = fPmQy*pIn[631] + 6*i2ze*pIn[515];
      pOut[632] = fPmQy*pIn[632] + 6*i2ze*pIn[520];
      pOut[633] = fPmQy*pIn[633] + 7*i2ze*pIn[523];
      pOut[634] = fPmQy*pIn[634] + i2ze*pIn[492];
      pOut[635] = fPmQy*pIn[635] + 2*i2ze*pIn[499];
      pOut[636] = fPmQy*pIn[636] + 10*i2ze*pIn[525];
      pOut[637] = fPmQy*pIn[637] + 2*i2ze*pIn[500];
      pOut[638] = fPmQy*pIn[638] + 3*i2ze*pIn[524];
      pOut[639] = fPmQy*pIn[639] + 9*i2ze*pIn[527];
      pOut[640] = fPmQy*pIn[640] + 2*i2ze*pIn[504];
      pOut[641] = fPmQy*pIn[641] + 2*i2ze*pIn[506];
      pOut[642] = fPmQy*pIn[642] + 9*i2ze*pIn[532];
      pOut[643] = fPmQy*pIn[643] + 3*i2ze*pIn[526];
      pOut[644] = fPmQy*pIn[644] + 4*i2ze*pIn[529];
      pOut[645] = fPmQy*pIn[645] + 2*i2ze*pIn[511];
      pOut[646] = fPmQy*pIn[646] + 8*i2ze*pIn[535];
      pOut[647] = fPmQy*pIn[647] + 2*i2ze*pIn[512];
      pOut[648] = fPmQy*pIn[648] + 8*i2ze*pIn[536];
      pOut[649] = fPmQy*pIn[649] + 4*i2ze*pIn[530];
      pOut[650] = fPmQy*pIn[650] + 5*i2ze*pIn[533];
      pOut[651] = fPmQy*pIn[651] + 7*i2ze*pIn[539];
      pOut[652] = fPmQy*pIn[652] + 2*i2ze*pIn[516];
      pOut[653] = fPmQy*pIn[653] + 2*i2ze*pIn[518];
      pOut[654] = fPmQy*pIn[654] + 7*i2ze*pIn[544];
      pOut[655] = fPmQy*pIn[655] + 5*i2ze*pIn[538];
      pOut[656] = fPmQy*pIn[656] + 6*i2ze*pIn[541];
      pOut[657] = fPmQy*pIn[657] + 2*i2ze*pIn[522];
      pOut[658] = fPmQy*pIn[658] + 6*i2ze*pIn[542];
      pOut[659] = fPmQy*pIn[659] + 3*i2ze*pIn[528];
      pOut[660] = fPmQy*pIn[660] + 8*i2ze*pIn[546];
      pOut[661] = fPmQy*pIn[661] + 3*i2ze*pIn[531];
      pOut[662] = fPmQy*pIn[662] + 3*i2ze*pIn[534];
      pOut[663] = fPmQy*pIn[663] + 7*i2ze*pIn[548];
      pOut[664] = fPmQy*pIn[664] + 4*i2ze*pIn[545];
      pOut[665] = fPmQy*pIn[665] + 4*i2ze*pIn[547];
      pOut[666] = fPmQy*pIn[666] + 7*i2ze*pIn[553];
      pOut[667] = fPmQy*pIn[667] + 3*i2ze*pIn[537];
      pOut[668] = fPmQy*pIn[668] + 3*i2ze*pIn[540];
      pOut[669] = fPmQy*pIn[669] + 5*i2ze*pIn[549];
      pOut[670] = fPmQy*pIn[670] + 6*i2ze*pIn[554];
      pOut[671] = fPmQy*pIn[671] + 6*i2ze*pIn[556];
      pOut[672] = fPmQy*pIn[672] + 5*i2ze*pIn[552];
      pOut[673] = fPmQy*pIn[673] + 3*i2ze*pIn[543];
      pOut[674] = fPmQy*pIn[674] + 4*i2ze*pIn[550];
      pOut[675] = fPmQy*pIn[675] + 6*i2ze*pIn[558];
      pOut[676] = fPmQy*pIn[676] + 4*i2ze*pIn[551];
      pOut[677] = fPmQy*pIn[677] + 5*i2ze*pIn[557];
      pOut[678] = fPmQy*pIn[678] + 4*i2ze*pIn[555];
      pOut[679] = fPmQy*pIn[679] + 5*i2ze*pIn[559];
      if (lab == 14) return;
   }
   assert(0);
}

// Calculate (a0|c)^m from (a0|c-1z)^{m+1}, for a = la..lab
// i = _x/_y/_z is the reduction direction (x,y,z);
// fPmQi = PmQ[i] * (rho/gamma); i2ze = .5/eta_ABC = .5/(alpha + beta + gamma)
// note: the la is not for performance reasons (would likely make the actual recurrence
//       slower for small lab) but for preventing overwriting lower components.
static void OsrrB_3cKerM_z(double *IR_RP pOut, double const *IR_RP pIn, double fPmQz, unsigned la, unsigned lab, double i2ze)
{
   switch(la) {
   case 0:
      pOut[0] = fPmQz*pIn[0];
      if (lab == 0) return;
   case 1:
      pOut[1] = fPmQz*pIn[1];
      pOut[2] = fPmQz*pIn[2];
      pOut[3] = fPmQz*pIn[3] + i2ze*pIn[0];
      if (lab == 1) return;
   case 2:
      pOut[4] = fPmQz*pIn[4];
      pOut[5] = fPmQz*pIn[5];
      pOut[6] = fPmQz*pIn[6] + 2*i2ze*pIn[3];
      pOut[7] = fPmQz*pIn[7];
      pOut[8] = fPmQz*pIn[8] + i2ze*pIn[1];
      pOut[9] = fPmQz*pIn[9] + i2ze*pIn[2];
      if (lab == 2) return;
   case 3:
      pOut[10] = fPmQz*pIn[10];
      pOut[11] = fPmQz*pIn[11];
      pOut[12] = fPmQz*pIn[12] + 3*i2ze*pIn[6];
      pOut[13] = fPmQz*pIn[13];
      pOut[14] = fPmQz*pIn[14] + 2*i2ze*pIn[8];
      pOut[15] = fPmQz*pIn[15];
      pOut[16] = fPmQz*pIn[16] + 2*i2ze*pIn[9];
      pOut[17] = fPmQz*pIn[17] + i2ze*pIn[4];
      pOut[18] = fPmQz*pIn[18] + i2ze*pIn[5];
      pOut[19] = fPmQz*pIn[19] + i2ze*pIn[7];
      if (lab == 3) return;
   case 4:
      pOut[20] = fPmQz*pIn[20];
      pOut[21] = fPmQz*pIn[21];
      pOut[22] = fPmQz*pIn[22] + 4*i2ze*pIn[12];
      pOut[23] = fPmQz*pIn[23];
      pOut[24] = fPmQz*pIn[24];
      pOut[25] = fPmQz*pIn[25] + i2ze*pIn[10];
      pOut[26] = fPmQz*pIn[26] + 3*i2ze*pIn[14];
      pOut[27] = fPmQz*pIn[27] + i2ze*pIn[11];
      pOut[28] = fPmQz*pIn[28] + 3*i2ze*pIn[16];
      pOut[29] = fPmQz*pIn[29];
      pOut[30] = fPmQz*pIn[30] + 2*i2ze*pIn[17];
      pOut[31] = fPmQz*pIn[31] + 2*i2ze*pIn[18];
      pOut[32] = fPmQz*pIn[32] + 2*i2ze*pIn[19];
      pOut[33] = fPmQz*pIn[33] + i2ze*pIn[13];
      pOut[34] = fPmQz*pIn[34] + i2ze*pIn[15];
      if (lab == 4) return;
   case 5:
      pOut[35] = fPmQz*pIn[35];
      pOut[36] = fPmQz*pIn[36];
      pOut[37] = fPmQz*pIn[37] + 5*i2ze*pIn[22];
      pOut[38] = fPmQz*pIn[38];
      pOut[39] = fPmQz*pIn[39] + 4*i2ze*pIn[26];
      pOut[40] = fPmQz*pIn[40];
      pOut[41] = fPmQz*pIn[41] + 4*i2ze*pIn[28];
      pOut[42] = fPmQz*pIn[42] + i2ze*pIn[20];
      pOut[43] = fPmQz*pIn[43] + i2ze*pIn[21];
      pOut[44] = fPmQz*pIn[44];
      pOut[45] = fPmQz*pIn[45] + 2*i2ze*pIn[25];
      pOut[46] = fPmQz*pIn[46];
      pOut[47] = fPmQz*pIn[47] + 2*i2ze*pIn[27];
      pOut[48] = fPmQz*pIn[48] + 3*i2ze*pIn[30];
      pOut[49] = fPmQz*pIn[49] + 3*i2ze*pIn[31];
      pOut[50] = fPmQz*pIn[50] + i2ze*pIn[23];
      pOut[51] = fPmQz*pIn[51] + i2ze*pIn[24];
      pOut[52] = fPmQz*pIn[52] + 3*i2ze*pIn[32];
      pOut[53] = fPmQz*pIn[53] + 2*i2ze*pIn[33];
      pOut[54] = fPmQz*pIn[54] + 2*i2ze*pIn[34];
      pOut[55] = fPmQz*pIn[55] + i2ze*pIn[29];
      if (lab == 5) return;
   case 6:
      pOut[56] = fPmQz*pIn[56];
      pOut[57] = fPmQz*pIn[57];
      pOut[58] = fPmQz*pIn[58] + 6*i2ze*pIn[37];
      pOut[59] = fPmQz*pIn[59];
      pOut[60] = fPmQz*pIn[60];
      pOut[61] = fPmQz*pIn[61] + i2ze*pIn[35];
      pOut[62] = fPmQz*pIn[62] + 5*i2ze*pIn[39];
      pOut[63] = fPmQz*pIn[63] + i2ze*pIn[36];
      pOut[64] = fPmQz*pIn[64] + 5*i2ze*pIn[41];
      pOut[65] = fPmQz*pIn[65];
      pOut[66] = fPmQz*pIn[66] + 2*i2ze*pIn[42];
      pOut[67] = fPmQz*pIn[67];
      pOut[68] = fPmQz*pIn[68] + 4*i2ze*pIn[48];
      pOut[69] = fPmQz*pIn[69] + 2*i2ze*pIn[43];
      pOut[70] = fPmQz*pIn[70] + 4*i2ze*pIn[49];
      pOut[71] = fPmQz*pIn[71];
      pOut[72] = fPmQz*pIn[72] + 3*i2ze*pIn[45];
      pOut[73] = fPmQz*pIn[73] + 3*i2ze*pIn[47];
      pOut[74] = fPmQz*pIn[74] + 4*i2ze*pIn[52];
      pOut[75] = fPmQz*pIn[75] + i2ze*pIn[38];
      pOut[76] = fPmQz*pIn[76] + i2ze*pIn[40];
      pOut[77] = fPmQz*pIn[77] + 2*i2ze*pIn[50];
      pOut[78] = fPmQz*pIn[78] + 2*i2ze*pIn[51];
      pOut[79] = fPmQz*pIn[79] + i2ze*pIn[44];
      pOut[80] = fPmQz*pIn[80] + 3*i2ze*pIn[53];
      pOut[81] = fPmQz*pIn[81] + i2ze*pIn[46];
      pOut[82] = fPmQz*pIn[82] + 3*i2ze*pIn[54];
      pOut[83] = fPmQz*pIn[83] + 2*i2ze*pIn[55];
      if (lab == 6) return;
   case 7:
      pOut[84] = fPmQz*pIn[84];
      pOut[85] = fPmQz*pIn[85];
      pOut[86] = fPmQz*pIn[86] + 7*i2ze*pIn[58];
      pOut[87] = fPmQz*pIn[87];
      pOut[88] = fPmQz*pIn[88] + 6*i2ze*pIn[62];
      pOut[89] = fPmQz*pIn[89];
      pOut[90] = fPmQz*pIn[90] + 6*i2ze*pIn[64];
      pOut[91] = fPmQz*pIn[91] + i2ze*pIn[56];
      pOut[92] = fPmQz*pIn[92] + i2ze*pIn[57];
      pOut[93] = fPmQz*pIn[93];
      pOut[94] = fPmQz*pIn[94] + 2*i2ze*pIn[61];
      pOut[95] = fPmQz*pIn[95];
      pOut[96] = fPmQz*pIn[96] + 2*i2ze*pIn[63];
      pOut[97] = fPmQz*pIn[97] + 5*i2ze*pIn[68];
      pOut[98] = fPmQz*pIn[98] + 5*i2ze*pIn[70];
      pOut[99] = fPmQz*pIn[99];
      pOut[100] = fPmQz*pIn[100] + 4*i2ze*pIn[72];
      pOut[101] = fPmQz*pIn[101];
      pOut[102] = fPmQz*pIn[102] + 4*i2ze*pIn[73];
      pOut[103] = fPmQz*pIn[103] + 3*i2ze*pIn[66];
      pOut[104] = fPmQz*pIn[104] + 3*i2ze*pIn[69];
      pOut[105] = fPmQz*pIn[105] + i2ze*pIn[59];
      pOut[106] = fPmQz*pIn[106] + i2ze*pIn[60];
      pOut[107] = fPmQz*pIn[107] + 5*i2ze*pIn[74];
      pOut[108] = fPmQz*pIn[108] + 2*i2ze*pIn[75];
      pOut[109] = fPmQz*pIn[109] + 4*i2ze*pIn[80];
      pOut[110] = fPmQz*pIn[110] + 2*i2ze*pIn[76];
      pOut[111] = fPmQz*pIn[111] + 4*i2ze*pIn[82];
      pOut[112] = fPmQz*pIn[112] + i2ze*pIn[65];
      pOut[113] = fPmQz*pIn[113] + i2ze*pIn[67];
      pOut[114] = fPmQz*pIn[114] + i2ze*pIn[71];
      pOut[115] = fPmQz*pIn[115] + 3*i2ze*pIn[77];
      pOut[116] = fPmQz*pIn[116] + 3*i2ze*pIn[78];
      pOut[117] = fPmQz*pIn[117] + 2*i2ze*pIn[79];
      pOut[118] = fPmQz*pIn[118] + 2*i2ze*pIn[81];
      pOut[119] = fPmQz*pIn[119] + 3*i2ze*pIn[83];
      if (lab == 7) return;
   case 8:
      pOut[120] = fPmQz*pIn[120];
      pOut[121] = fPmQz*pIn[121];
      pOut[122] = fPmQz*pIn[122] + 8*i2ze*pIn[86];
      pOut[123] = fPmQz*pIn[123];
      pOut[124] = fPmQz*pIn[124];
      pOut[125] = fPmQz*pIn[125] + i2ze*pIn[84];
      pOut[126] = fPmQz*pIn[126] + 7*i2ze*pIn[88];
      pOut[127] = fPmQz*pIn[127] + i2ze*pIn[85];
      pOut[128] = fPmQz*pIn[128] + 7*i2ze*pIn[90];
      pOut[129] = fPmQz*pIn[129];
      pOut[130] = fPmQz*pIn[130] + 2*i2ze*pIn[91];
      pOut[131] = fPmQz*pIn[131];
      pOut[132] = fPmQz*pIn[132] + 6*i2ze*pIn[97];
      pOut[133] = fPmQz*pIn[133] + 2*i2ze*pIn[92];
      pOut[134] = fPmQz*pIn[134] + 6*i2ze*pIn[98];
      pOut[135] = fPmQz*pIn[135];
      pOut[136] = fPmQz*pIn[136];
      pOut[137] = fPmQz*pIn[137] + 3*i2ze*pIn[94];
      pOut[138] = fPmQz*pIn[138] + 5*i2ze*pIn[100];
      pOut[139] = fPmQz*pIn[139] + 3*i2ze*pIn[96];
      pOut[140] = fPmQz*pIn[140] + 5*i2ze*pIn[102];
      pOut[141] = fPmQz*pIn[141];
      pOut[142] = fPmQz*pIn[142] + 4*i2ze*pIn[103];
      pOut[143] = fPmQz*pIn[143] + 4*i2ze*pIn[104];
      pOut[144] = fPmQz*pIn[144] + 6*i2ze*pIn[107];
      pOut[145] = fPmQz*pIn[145] + i2ze*pIn[87];
      pOut[146] = fPmQz*pIn[146] + i2ze*pIn[89];
      pOut[147] = fPmQz*pIn[147] + 2*i2ze*pIn[105];
      pOut[148] = fPmQz*pIn[148] + 2*i2ze*pIn[106];
      pOut[149] = fPmQz*pIn[149] + i2ze*pIn[93];
      pOut[150] = fPmQz*pIn[150] + 5*i2ze*pIn[109];
      pOut[151] = fPmQz*pIn[151] + i2ze*pIn[95];
      pOut[152] = fPmQz*pIn[152] + 5*i2ze*pIn[111];
      pOut[153] = fPmQz*pIn[153] + 4*i2ze*pIn[115];
      pOut[154] = fPmQz*pIn[154] + 4*i2ze*pIn[116];
      pOut[155] = fPmQz*pIn[155] + i2ze*pIn[99];
      pOut[156] = fPmQz*pIn[156] + 3*i2ze*pIn[108];
      pOut[157] = fPmQz*pIn[157] + i2ze*pIn[101];
      pOut[158] = fPmQz*pIn[158] + 3*i2ze*pIn[110];
      pOut[159] = fPmQz*pIn[159] + 2*i2ze*pIn[112];
      pOut[160] = fPmQz*pIn[160] + 2*i2ze*pIn[113];
      pOut[161] = fPmQz*pIn[161] + 4*i2ze*pIn[119];
      pOut[162] = fPmQz*pIn[162] + 2*i2ze*pIn[114];
      pOut[163] = fPmQz*pIn[163] + 3*i2ze*pIn[117];
      pOut[164] = fPmQz*pIn[164] + 3*i2ze*pIn[118];
      if (lab == 8) return;
   case 9:
      pOut[165] = fPmQz*pIn[165];
      pOut[166] = fPmQz*pIn[166];
      pOut[167] = fPmQz*pIn[167] + 9*i2ze*pIn[122];
      pOut[168] = fPmQz*pIn[168];
      pOut[169] = fPmQz*pIn[169] + 8*i2ze*pIn[126];
      pOut[170] = fPmQz*pIn[170];
      pOut[171] = fPmQz*pIn[171] + 8*i2ze*pIn[128];
      pOut[172] = fPmQz*pIn[172] + i2ze*pIn[120];
      pOut[173] = fPmQz*pIn[173] + i2ze*pIn[121];
      pOut[174] = fPmQz*pIn[174];
      pOut[175] = fPmQz*pIn[175] + 2*i2ze*pIn[125];
      pOut[176] = fPmQz*pIn[176];
      pOut[177] = fPmQz*pIn[177] + 2*i2ze*pIn[127];
      pOut[178] = fPmQz*pIn[178] + 7*i2ze*pIn[132];
      pOut[179] = fPmQz*pIn[179] + 7*i2ze*pIn[134];
      pOut[180] = fPmQz*pIn[180];
      pOut[181] = fPmQz*pIn[181] + 6*i2ze*pIn[138];
      pOut[182] = fPmQz*pIn[182];
      pOut[183] = fPmQz*pIn[183] + 6*i2ze*pIn[140];
      pOut[184] = fPmQz*pIn[184] + 3*i2ze*pIn[130];
      pOut[185] = fPmQz*pIn[185] + 3*i2ze*pIn[133];
      pOut[186] = fPmQz*pIn[186];
      pOut[187] = fPmQz*pIn[187] + 4*i2ze*pIn[137];
      pOut[188] = fPmQz*pIn[188];
      pOut[189] = fPmQz*pIn[189] + 4*i2ze*pIn[139];
      pOut[190] = fPmQz*pIn[190] + 5*i2ze*pIn[142];
      pOut[191] = fPmQz*pIn[191] + 5*i2ze*pIn[143];
      pOut[192] = fPmQz*pIn[192] + i2ze*pIn[123];
      pOut[193] = fPmQz*pIn[193] + i2ze*pIn[124];
      pOut[194] = fPmQz*pIn[194] + 7*i2ze*pIn[144];
      pOut[195] = fPmQz*pIn[195] + 2*i2ze*pIn[145];
      pOut[196] = fPmQz*pIn[196] + 6*i2ze*pIn[150];
      pOut[197] = fPmQz*pIn[197] + 2*i2ze*pIn[146];
      pOut[198] = fPmQz*pIn[198] + 6*i2ze*pIn[152];
      pOut[199] = fPmQz*pIn[199] + i2ze*pIn[129];
      pOut[200] = fPmQz*pIn[200] + i2ze*pIn[131];
      pOut[201] = fPmQz*pIn[201] + i2ze*pIn[135];
      pOut[202] = fPmQz*pIn[202] + 3*i2ze*pIn[147];
      pOut[203] = fPmQz*pIn[203] + i2ze*pIn[136];
      pOut[204] = fPmQz*pIn[204] + 5*i2ze*pIn[153];
      pOut[205] = fPmQz*pIn[205] + 3*i2ze*pIn[148];
      pOut[206] = fPmQz*pIn[206] + 5*i2ze*pIn[154];
      pOut[207] = fPmQz*pIn[207] + 4*i2ze*pIn[156];
      pOut[208] = fPmQz*pIn[208] + 4*i2ze*pIn[158];
      pOut[209] = fPmQz*pIn[209] + i2ze*pIn[141];
      pOut[210] = fPmQz*pIn[210] + 2*i2ze*pIn[149];
      pOut[211] = fPmQz*pIn[211] + 2*i2ze*pIn[151];
      pOut[212] = fPmQz*pIn[212] + 5*i2ze*pIn[161];
      pOut[213] = fPmQz*pIn[213] + 2*i2ze*pIn[155];
      pOut[214] = fPmQz*pIn[214] + 4*i2ze*pIn[163];
      pOut[215] = fPmQz*pIn[215] + 2*i2ze*pIn[157];
      pOut[216] = fPmQz*pIn[216] + 4*i2ze*pIn[164];
      pOut[217] = fPmQz*pIn[217] + 3*i2ze*pIn[159];
      pOut[218] = fPmQz*pIn[218] + 3*i2ze*pIn[160];
      pOut[219] = fPmQz*pIn[219] + 3*i2ze*pIn[162];
      if (lab == 9) return;
   case 10:
      pOut[220] = fPmQz*pIn[220];
      pOut[221] = fPmQz*pIn[221];
      pOut[222] = fPmQz*pIn[222] + 10*i2ze*pIn[167];
      pOut[223] = fPmQz*pIn[223];
      pOut[224] = fPmQz*pIn[224];
      pOut[225] = fPmQz*pIn[225] + i2ze*pIn[165];
      pOut[226] = fPmQz*pIn[226] + 9*i2ze*pIn[169];
      pOut[227] = fPmQz*pIn[227] + i2ze*pIn[166];
      pOut[228] = fPmQz*pIn[228] + 9*i2ze*pIn[171];
      pOut[229] = fPmQz*pIn[229];
      pOut[230] = fPmQz*pIn[230] + 2*i2ze*pIn[172];
      pOut[231] = fPmQz*pIn[231];
      pOut[232] = fPmQz*pIn[232] + 8*i2ze*pIn[178];
      pOut[233] = fPmQz*pIn[233] + 2*i2ze*pIn[173];
      pOut[234] = fPmQz*pIn[234] + 8*i2ze*pIn[179];
      pOut[235] = fPmQz*pIn[235];
      pOut[236] = fPmQz*pIn[236];
      pOut[237] = fPmQz*pIn[237] + 3*i2ze*pIn[175];
      pOut[238] = fPmQz*pIn[238] + 7*i2ze*pIn[181];
      pOut[239] = fPmQz*pIn[239] + 3*i2ze*pIn[177];
      pOut[240] = fPmQz*pIn[240] + 7*i2ze*pIn[183];
      pOut[241] = fPmQz*pIn[241];
      pOut[242] = fPmQz*pIn[242] + 4*i2ze*pIn[184];
      pOut[243] = fPmQz*pIn[243];
      pOut[244] = fPmQz*pIn[244] + 6*i2ze*pIn[190];
      pOut[245] = fPmQz*pIn[245] + 4*i2ze*pIn[185];
      pOut[246] = fPmQz*pIn[246] + 6*i2ze*pIn[191];
      pOut[247] = fPmQz*pIn[247];
      pOut[248] = fPmQz*pIn[248] + 5*i2ze*pIn[187];
      pOut[249] = fPmQz*pIn[249] + 5*i2ze*pIn[189];
      pOut[250] = fPmQz*pIn[250] + 8*i2ze*pIn[194];
      pOut[251] = fPmQz*pIn[251] + i2ze*pIn[168];
      pOut[252] = fPmQz*pIn[252] + i2ze*pIn[170];
      pOut[253] = fPmQz*pIn[253] + 2*i2ze*pIn[192];
      pOut[254] = fPmQz*pIn[254] + 2*i2ze*pIn[193];
      pOut[255] = fPmQz*pIn[255] + i2ze*pIn[174];
      pOut[256] = fPmQz*pIn[256] + 7*i2ze*pIn[196];
      pOut[257] = fPmQz*pIn[257] + i2ze*pIn[176];
      pOut[258] = fPmQz*pIn[258] + 7*i2ze*pIn[198];
      pOut[259] = fPmQz*pIn[259] + 6*i2ze*pIn[204];
      pOut[260] = fPmQz*pIn[260] + 6*i2ze*pIn[206];
      pOut[261] = fPmQz*pIn[261] + i2ze*pIn[180];
      pOut[262] = fPmQz*pIn[262] + 3*i2ze*pIn[195];
      pOut[263] = fPmQz*pIn[263] + i2ze*pIn[182];
      pOut[264] = fPmQz*pIn[264] + 3*i2ze*pIn[197];
      pOut[265] = fPmQz*pIn[265] + 4*i2ze*pIn[202];
      pOut[266] = fPmQz*pIn[266] + 4*i2ze*pIn[205];
      pOut[267] = fPmQz*pIn[267] + i2ze*pIn[186];
      pOut[268] = fPmQz*pIn[268] + 5*i2ze*pIn[207];
      pOut[269] = fPmQz*pIn[269] + i2ze*pIn[188];
      pOut[270] = fPmQz*pIn[270] + 5*i2ze*pIn[208];
      pOut[271] = fPmQz*pIn[271] + 2*i2ze*pIn[199];
      pOut[272] = fPmQz*pIn[272] + 2*i2ze*pIn[200];
      pOut[273] = fPmQz*pIn[273] + 6*i2ze*pIn[212];
      pOut[274] = fPmQz*pIn[274] + 2*i2ze*pIn[201];
      pOut[275] = fPmQz*pIn[275] + 2*i2ze*pIn[203];
      pOut[276] = fPmQz*pIn[276] + 3*i2ze*pIn[210];
      pOut[277] = fPmQz*pIn[277] + 5*i2ze*pIn[214];
      pOut[278] = fPmQz*pIn[278] + 3*i2ze*pIn[211];
      pOut[279] = fPmQz*pIn[279] + 5*i2ze*pIn[216];
      pOut[280] = fPmQz*pIn[280] + 2*i2ze*pIn[209];
      pOut[281] = fPmQz*pIn[281] + 4*i2ze*pIn[217];
      pOut[282] = fPmQz*pIn[282] + 4*i2ze*pIn[218];
      pOut[283] = fPmQz*pIn[283] + 4*i2ze*pIn[219];
      pOut[284] = fPmQz*pIn[284] + 3*i2ze*pIn[213];
      pOut[285] = fPmQz*pIn[285] + 3*i2ze*pIn[215];
      if (lab == 10) return;
   case 11:
      pOut[286] = fPmQz*pIn[286];
      pOut[287] = fPmQz*pIn[287];
      pOut[288] = fPmQz*pIn[288] + 11*i2ze*pIn[222];
      pOut[289] = fPmQz*pIn[289];
      pOut[290] = fPmQz*pIn[290] + 10*i2ze*pIn[226];
      pOut[291] = fPmQz*pIn[291];
      pOut[292] = fPmQz*pIn[292] + 10*i2ze*pIn[228];
      pOut[293] = fPmQz*pIn[293] + i2ze*pIn[220];
      pOut[294] = fPmQz*pIn[294] + i2ze*pIn[221];
      pOut[295] = fPmQz*pIn[295];
      pOut[296] = fPmQz*pIn[296] + 2*i2ze*pIn[225];
      pOut[297] = fPmQz*pIn[297];
      pOut[298] = fPmQz*pIn[298] + 2*i2ze*pIn[227];
      pOut[299] = fPmQz*pIn[299] + 9*i2ze*pIn[232];
      pOut[300] = fPmQz*pIn[300] + 9*i2ze*pIn[234];
      pOut[301] = fPmQz*pIn[301];
      pOut[302] = fPmQz*pIn[302] + 8*i2ze*pIn[238];
      pOut[303] = fPmQz*pIn[303];
      pOut[304] = fPmQz*pIn[304] + 8*i2ze*pIn[240];
      pOut[305] = fPmQz*pIn[305] + 3*i2ze*pIn[230];
      pOut[306] = fPmQz*pIn[306] + 3*i2ze*pIn[233];
      pOut[307] = fPmQz*pIn[307];
      pOut[308] = fPmQz*pIn[308] + 4*i2ze*pIn[237];
      pOut[309] = fPmQz*pIn[309];
      pOut[310] = fPmQz*pIn[310] + 4*i2ze*pIn[239];
      pOut[311] = fPmQz*pIn[311] + 7*i2ze*pIn[244];
      pOut[312] = fPmQz*pIn[312] + 7*i2ze*pIn[246];
      pOut[313] = fPmQz*pIn[313];
      pOut[314] = fPmQz*pIn[314] + 6*i2ze*pIn[248];
      pOut[315] = fPmQz*pIn[315];
      pOut[316] = fPmQz*pIn[316] + 6*i2ze*pIn[249];
      pOut[317] = fPmQz*pIn[317] + 5*i2ze*pIn[242];
      pOut[318] = fPmQz*pIn[318] + 5*i2ze*pIn[245];
      pOut[319] = fPmQz*pIn[319] + i2ze*pIn[223];
      pOut[320] = fPmQz*pIn[320] + i2ze*pIn[224];
      pOut[321] = fPmQz*pIn[321] + 9*i2ze*pIn[250];
      pOut[322] = fPmQz*pIn[322] + 2*i2ze*pIn[251];
      pOut[323] = fPmQz*pIn[323] + 8*i2ze*pIn[256];
      pOut[324] = fPmQz*pIn[324] + 2*i2ze*pIn[252];
      pOut[325] = fPmQz*pIn[325] + 8*i2ze*pIn[258];
      pOut[326] = fPmQz*pIn[326] + i2ze*pIn[229];
      pOut[327] = fPmQz*pIn[327] + i2ze*pIn[231];
      pOut[328] = fPmQz*pIn[328] + i2ze*pIn[235];
      pOut[329] = fPmQz*pIn[329] + 3*i2ze*pIn[253];
      pOut[330] = fPmQz*pIn[330] + i2ze*pIn[236];
      pOut[331] = fPmQz*pIn[331] + 7*i2ze*pIn[259];
      pOut[332] = fPmQz*pIn[332] + 3*i2ze*pIn[254];
      pOut[333] = fPmQz*pIn[333] + 7*i2ze*pIn[260];
      pOut[334] = fPmQz*pIn[334] + 4*i2ze*pIn[262];
      pOut[335] = fPmQz*pIn[335] + 6*i2ze*pIn[268];
      pOut[336] = fPmQz*pIn[336] + 4*i2ze*pIn[264];
      pOut[337] = fPmQz*pIn[337] + 6*i2ze*pIn[270];
      pOut[338] = fPmQz*pIn[338] + i2ze*pIn[241];
      pOut[339] = fPmQz*pIn[339] + i2ze*pIn[243];
      pOut[340] = fPmQz*pIn[340] + i2ze*pIn[247];
      pOut[341] = fPmQz*pIn[341] + 5*i2ze*pIn[265];
      pOut[342] = fPmQz*pIn[342] + 5*i2ze*pIn[266];
      pOut[343] = fPmQz*pIn[343] + 2*i2ze*pIn[255];
      pOut[344] = fPmQz*pIn[344] + 2*i2ze*pIn[257];
      pOut[345] = fPmQz*pIn[345] + 7*i2ze*pIn[273];
      pOut[346] = fPmQz*pIn[346] + 2*i2ze*pIn[261];
      pOut[347] = fPmQz*pIn[347] + 6*i2ze*pIn[277];
      pOut[348] = fPmQz*pIn[348] + 2*i2ze*pIn[263];
      pOut[349] = fPmQz*pIn[349] + 6*i2ze*pIn[279];
      pOut[350] = fPmQz*pIn[350] + 3*i2ze*pIn[271];
      pOut[351] = fPmQz*pIn[351] + 3*i2ze*pIn[272];
      pOut[352] = fPmQz*pIn[352] + 2*i2ze*pIn[267];
      pOut[353] = fPmQz*pIn[353] + 4*i2ze*pIn[276];
      pOut[354] = fPmQz*pIn[354] + 2*i2ze*pIn[269];
      pOut[355] = fPmQz*pIn[355] + 4*i2ze*pIn[278];
      pOut[356] = fPmQz*pIn[356] + 5*i2ze*pIn[281];
      pOut[357] = fPmQz*pIn[357] + 5*i2ze*pIn[282];
      pOut[358] = fPmQz*pIn[358] + 3*i2ze*pIn[274];
      pOut[359] = fPmQz*pIn[359] + 3*i2ze*pIn[275];
      pOut[360] = fPmQz*pIn[360] + 5*i2ze*pIn[283];
      pOut[361] = fPmQz*pIn[361] + 4*i2ze*pIn[284];
      pOut[362] = fPmQz*pIn[362] + 4*i2ze*pIn[285];
      pOut[363] = fPmQz*pIn[363] + 3*i2ze*pIn[280];
      if (lab == 11) return;
   case 12:
      pOut[364] = fPmQz*pIn[364];
      pOut[365] = fPmQz*pIn[365];
      pOut[366] = fPmQz*pIn[366] + 12*i2ze*pIn[288];
      pOut[367] = fPmQz*pIn[367];
      pOut[368] = fPmQz*pIn[368];
      pOut[369] = fPmQz*pIn[369] + i2ze*pIn[286];
      pOut[370] = fPmQz*pIn[370] + 11*i2ze*pIn[290];
      pOut[371] = fPmQz*pIn[371] + i2ze*pIn[287];
      pOut[372] = fPmQz*pIn[372] + 11*i2ze*pIn[292];
      pOut[373] = fPmQz*pIn[373];
      pOut[374] = fPmQz*pIn[374] + 2*i2ze*pIn[293];
      pOut[375] = fPmQz*pIn[375];
      pOut[376] = fPmQz*pIn[376] + 10*i2ze*pIn[299];
      pOut[377] = fPmQz*pIn[377] + 2*i2ze*pIn[294];
      pOut[378] = fPmQz*pIn[378] + 10*i2ze*pIn[300];
      pOut[379] = fPmQz*pIn[379];
      pOut[380] = fPmQz*pIn[380];
      pOut[381] = fPmQz*pIn[381] + 3*i2ze*pIn[296];
      pOut[382] = fPmQz*pIn[382] + 9*i2ze*pIn[302];
      pOut[383] = fPmQz*pIn[383] + 3*i2ze*pIn[298];
      pOut[384] = fPmQz*pIn[384] + 9*i2ze*pIn[304];
      pOut[385] = fPmQz*pIn[385];
      pOut[386] = fPmQz*pIn[386] + 4*i2ze*pIn[305];
      pOut[387] = fPmQz*pIn[387];
      pOut[388] = fPmQz*pIn[388] + 8*i2ze*pIn[311];
      pOut[389] = fPmQz*pIn[389] + 4*i2ze*pIn[306];
      pOut[390] = fPmQz*pIn[390] + 8*i2ze*pIn[312];
      pOut[391] = fPmQz*pIn[391];
      pOut[392] = fPmQz*pIn[392];
      pOut[393] = fPmQz*pIn[393] + 5*i2ze*pIn[308];
      pOut[394] = fPmQz*pIn[394] + 7*i2ze*pIn[314];
      pOut[395] = fPmQz*pIn[395] + 5*i2ze*pIn[310];
      pOut[396] = fPmQz*pIn[396] + 7*i2ze*pIn[316];
      pOut[397] = fPmQz*pIn[397];
      pOut[398] = fPmQz*pIn[398] + 6*i2ze*pIn[317];
      pOut[399] = fPmQz*pIn[399] + 6*i2ze*pIn[318];
      pOut[400] = fPmQz*pIn[400] + 10*i2ze*pIn[321];
      pOut[401] = fPmQz*pIn[401] + i2ze*pIn[289];
      pOut[402] = fPmQz*pIn[402] + i2ze*pIn[291];
      pOut[403] = fPmQz*pIn[403] + 2*i2ze*pIn[319];
      pOut[404] = fPmQz*pIn[404] + 2*i2ze*pIn[320];
      pOut[405] = fPmQz*pIn[405] + i2ze*pIn[295];
      pOut[406] = fPmQz*pIn[406] + 9*i2ze*pIn[323];
      pOut[407] = fPmQz*pIn[407] + i2ze*pIn[297];
      pOut[408] = fPmQz*pIn[408] + 9*i2ze*pIn[325];
      pOut[409] = fPmQz*pIn[409] + 8*i2ze*pIn[331];
      pOut[410] = fPmQz*pIn[410] + 8*i2ze*pIn[333];
      pOut[411] = fPmQz*pIn[411] + i2ze*pIn[301];
      pOut[412] = fPmQz*pIn[412] + 3*i2ze*pIn[322];
      pOut[413] = fPmQz*pIn[413] + i2ze*pIn[303];
      pOut[414] = fPmQz*pIn[414] + 3*i2ze*pIn[324];
      pOut[415] = fPmQz*pIn[415] + 4*i2ze*pIn[329];
      pOut[416] = fPmQz*pIn[416] + 4*i2ze*pIn[332];
      pOut[417] = fPmQz*pIn[417] + i2ze*pIn[307];
      pOut[418] = fPmQz*pIn[418] + 7*i2ze*pIn[335];
      pOut[419] = fPmQz*pIn[419] + i2ze*pIn[309];
      pOut[420] = fPmQz*pIn[420] + 7*i2ze*pIn[337];
      pOut[421] = fPmQz*pIn[421] + 6*i2ze*pIn[341];
      pOut[422] = fPmQz*pIn[422] + 6*i2ze*pIn[342];
      pOut[423] = fPmQz*pIn[423] + i2ze*pIn[313];
      pOut[424] = fPmQz*pIn[424] + 5*i2ze*pIn[334];
      pOut[425] = fPmQz*pIn[425] + i2ze*pIn[315];
      pOut[426] = fPmQz*pIn[426] + 5*i2ze*pIn[336];
      pOut[427] = fPmQz*pIn[427] + 2*i2ze*pIn[326];
      pOut[428] = fPmQz*pIn[428] + 2*i2ze*pIn[327];
      pOut[429] = fPmQz*pIn[429] + 8*i2ze*pIn[345];
      pOut[430] = fPmQz*pIn[430] + 2*i2ze*pIn[328];
      pOut[431] = fPmQz*pIn[431] + 2*i2ze*pIn[330];
      pOut[432] = fPmQz*pIn[432] + 3*i2ze*pIn[343];
      pOut[433] = fPmQz*pIn[433] + 7*i2ze*pIn[347];
      pOut[434] = fPmQz*pIn[434] + 3*i2ze*pIn[344];
      pOut[435] = fPmQz*pIn[435] + 7*i2ze*pIn[349];
      pOut[436] = fPmQz*pIn[436] + 2*i2ze*pIn[338];
      pOut[437] = fPmQz*pIn[437] + 4*i2ze*pIn[350];
      pOut[438] = fPmQz*pIn[438] + 2*i2ze*pIn[339];
      pOut[439] = fPmQz*pIn[439] + 6*i2ze*pIn[356];
      pOut[440] = fPmQz*pIn[440] + 4*i2ze*pIn[351];
      pOut[441] = fPmQz*pIn[441] + 6*i2ze*pIn[357];
      pOut[442] = fPmQz*pIn[442] + 2*i2ze*pIn[340];
      pOut[443] = fPmQz*pIn[443] + 5*i2ze*pIn[353];
      pOut[444] = fPmQz*pIn[444] + 5*i2ze*pIn[355];
      pOut[445] = fPmQz*pIn[445] + 6*i2ze*pIn[360];
      pOut[446] = fPmQz*pIn[446] + 3*i2ze*pIn[346];
      pOut[447] = fPmQz*pIn[447] + 3*i2ze*pIn[348];
      pOut[448] = fPmQz*pIn[448] + 4*i2ze*pIn[358];
      pOut[449] = fPmQz*pIn[449] + 4*i2ze*pIn[359];
      pOut[450] = fPmQz*pIn[450] + 3*i2ze*pIn[352];
      pOut[451] = fPmQz*pIn[451] + 5*i2ze*pIn[361];
      pOut[452] = fPmQz*pIn[452] + 3*i2ze*pIn[354];
      pOut[453] = fPmQz*pIn[453] + 5*i2ze*pIn[362];
      pOut[454] = fPmQz*pIn[454] + 4*i2ze*pIn[363];
      if (lab == 12) return;
   case 13:
      pOut[455] = fPmQz*pIn[455];
      pOut[456] = fPmQz*pIn[456];
      pOut[457] = fPmQz*pIn[457] + 13*i2ze*pIn[366];
      pOut[458] = fPmQz*pIn[458];
      pOut[459] = fPmQz*pIn[459] + 12*i2ze*pIn[370];
      pOut[460] = fPmQz*pIn[460];
      pOut[461] = fPmQz*pIn[461] + 12*i2ze*pIn[372];
      pOut[462] = fPmQz*pIn[462] + i2ze*pIn[364];
      pOut[463] = fPmQz*pIn[463] + i2ze*pIn[365];
      pOut[464] = fPmQz*pIn[464];
      pOut[465] = fPmQz*pIn[465] + 2*i2ze*pIn[369];
      pOut[466] = fPmQz*pIn[466];
      pOut[467] = fPmQz*pIn[467] + 2*i2ze*pIn[371];
      pOut[468] = fPmQz*pIn[468] + 11*i2ze*pIn[376];
      pOut[469] = fPmQz*pIn[469] + 11*i2ze*pIn[378];
      pOut[470] = fPmQz*pIn[470];
      pOut[471] = fPmQz*pIn[471] + 10*i2ze*pIn[382];
      pOut[472] = fPmQz*pIn[472];
      pOut[473] = fPmQz*pIn[473] + 10*i2ze*pIn[384];
      pOut[474] = fPmQz*pIn[474] + 3*i2ze*pIn[374];
      pOut[475] = fPmQz*pIn[475] + 3*i2ze*pIn[377];
      pOut[476] = fPmQz*pIn[476];
      pOut[477] = fPmQz*pIn[477] + 4*i2ze*pIn[381];
      pOut[478] = fPmQz*pIn[478];
      pOut[479] = fPmQz*pIn[479] + 4*i2ze*pIn[383];
      pOut[480] = fPmQz*pIn[480] + 9*i2ze*pIn[388];
      pOut[481] = fPmQz*pIn[481] + 9*i2ze*pIn[390];
      pOut[482] = fPmQz*pIn[482];
      pOut[483] = fPmQz*pIn[483] + 8*i2ze*pIn[394];
      pOut[484] = fPmQz*pIn[484];
      pOut[485] = fPmQz*pIn[485] + 8*i2ze*pIn[396];
      pOut[486] = fPmQz*pIn[486] + 5*i2ze*pIn[386];
      pOut[487] = fPmQz*pIn[487] + 5*i2ze*pIn[389];
      pOut[488] = fPmQz*pIn[488];
      pOut[489] = fPmQz*pIn[489] + 6*i2ze*pIn[393];
      pOut[490] = fPmQz*pIn[490];
      pOut[491] = fPmQz*pIn[491] + 6*i2ze*pIn[395];
      pOut[492] = fPmQz*pIn[492] + 7*i2ze*pIn[398];
      pOut[493] = fPmQz*pIn[493] + 7*i2ze*pIn[399];
      pOut[494] = fPmQz*pIn[494] + i2ze*pIn[367];
      pOut[495] = fPmQz*pIn[495] + i2ze*pIn[368];
      pOut[496] = fPmQz*pIn[496] + 11*i2ze*pIn[400];
      pOut[497] = fPmQz*pIn[497] + 2*i2ze*pIn[401];
      pOut[498] = fPmQz*pIn[498] + 10*i2ze*pIn[406];
      pOut[499] = fPmQz*pIn[499] + 2*i2ze*pIn[402];
      pOut[500] = fPmQz*pIn[500] + 10*i2ze*pIn[408];
      pOut[501] = fPmQz*pIn[501] + i2ze*pIn[373];
      pOut[502] = fPmQz*pIn[502] + i2ze*pIn[375];
      pOut[503] = fPmQz*pIn[503] + i2ze*pIn[379];
      pOut[504] = fPmQz*pIn[504] + 3*i2ze*pIn[403];
      pOut[505] = fPmQz*pIn[505] + i2ze*pIn[380];
      pOut[506] = fPmQz*pIn[506] + 9*i2ze*pIn[409];
      pOut[507] = fPmQz*pIn[507] + 3*i2ze*pIn[404];
      pOut[508] = fPmQz*pIn[508] + 9*i2ze*pIn[410];
      pOut[509] = fPmQz*pIn[509] + 4*i2ze*pIn[412];
      pOut[510] = fPmQz*pIn[510] + 8*i2ze*pIn[418];
      pOut[511] = fPmQz*pIn[511] + 4*i2ze*pIn[414];
      pOut[512] = fPmQz*pIn[512] + 8*i2ze*pIn[420];
      pOut[513] = fPmQz*pIn[513] + i2ze*pIn[385];
      pOut[514] = fPmQz*pIn[514] + i2ze*pIn[387];
      pOut[515] = fPmQz*pIn[515] + i2ze*pIn[391];
      pOut[516] = fPmQz*pIn[516] + 5*i2ze*pIn[415];
      pOut[517] = fPmQz*pIn[517] + i2ze*pIn[392];
      pOut[518] = fPmQz*pIn[518] + 7*i2ze*pIn[421];
      pOut[519] = fPmQz*pIn[519] + 5*i2ze*pIn[416];
      pOut[520] = fPmQz*pIn[520] + 7*i2ze*pIn[422];
      pOut[521] = fPmQz*pIn[521] + 6*i2ze*pIn[424];
      pOut[522] = fPmQz*pIn[522] + 6*i2ze*pIn[426];
      pOut[523] = fPmQz*pIn[523] + i2ze*pIn[397];
      pOut[524] = fPmQz*pIn[524] + 2*i2ze*pIn[405];
      pOut[525] = fPmQz*pIn[525] + 2*i2ze*pIn[407];
      pOut[526] = fPmQz*pIn[526] + 9*i2ze*pIn[429];
      pOut[527] = fPmQz*pIn[527] + 2*i2ze*pIn[411];
      pOut[528] = fPmQz*pIn[528] + 8*i2ze*pIn[433];
      pOut[529] = fPmQz*pIn[529] + 2*i2ze*pIn[413];
      pOut[530] = fPmQz*pIn[530] + 8*i2ze*pIn[435];
      pOut[531] = fPmQz*pIn[531] + 3*i2ze*pIn[427];
      pOut[532] = fPmQz*pIn[532] + 3*i2ze*pIn[428];
      pOut[533] = fPmQz*pIn[533] + 2*i2ze*pIn[417];
      pOut[534] = fPmQz*pIn[534] + 4*i2ze*pIn[432];
      pOut[535] = fPmQz*pIn[535] + 2*i2ze*pIn[419];
      pOut[536] = fPmQz*pIn[536] + 4*i2ze*pIn[434];
      pOut[537] = fPmQz*pIn[537] + 7*i2ze*pIn[439];
      pOut[538] = fPmQz*pIn[538] + 7*i2ze*pIn[441];
      pOut[539] = fPmQz*pIn[539] + 2*i2ze*pIn[423];
      pOut[540] = fPmQz*pIn[540] + 6*i2ze*pIn[443];
      pOut[541] = fPmQz*pIn[541] + 2*i2ze*pIn[425];
      pOut[542] = fPmQz*pIn[542] + 6*i2ze*pIn[444];
      pOut[543] = fPmQz*pIn[543] + 5*i2ze*pIn[437];
      pOut[544] = fPmQz*pIn[544] + 5*i2ze*pIn[440];
      pOut[545] = fPmQz*pIn[545] + 3*i2ze*pIn[430];
      pOut[546] = fPmQz*pIn[546] + 3*i2ze*pIn[431];
      pOut[547] = fPmQz*pIn[547] + 7*i2ze*pIn[445];
      pOut[548] = fPmQz*pIn[548] + 4*i2ze*pIn[446];
      pOut[549] = fPmQz*pIn[549] + 6*i2ze*pIn[451];
      pOut[550] = fPmQz*pIn[550] + 4*i2ze*pIn[447];
      pOut[551] = fPmQz*pIn[551] + 6*i2ze*pIn[453];
      pOut[552] = fPmQz*pIn[552] + 3*i2ze*pIn[436];
      pOut[553] = fPmQz*pIn[553] + 3*i2ze*pIn[438];
      pOut[554] = fPmQz*pIn[554] + 3*i2ze*pIn[442];
      pOut[555] = fPmQz*pIn[555] + 5*i2ze*pIn[448];
      pOut[556] = fPmQz*pIn[556] + 5*i2ze*pIn[449];
      pOut[557] = fPmQz*pIn[557] + 4*i2ze*pIn[450];
      pOut[558] = fPmQz*pIn[558] + 4*i2ze*pIn[452];
      pOut[559] = fPmQz*pIn[559] + 5*i2ze*pIn[454];
      if (lab == 13) return;
   case 14:
      pOut[560] = fPmQz*pIn[560];
      pOut[561] = fPmQz*pIn[561];
      pOut[562] = fPmQz*pIn[562] + 14*i2ze*pIn[457];
      pOut[563] = fPmQz*pIn[563];
      pOut[564] = fPmQz*pIn[564];
      pOut[565] = fPmQz*pIn[565] + i2ze*pIn[455];
      pOut[566] = fPmQz*pIn[566] + 13*i2ze*pIn[459];
      pOut[567] = fPmQz*pIn[567] + i2ze*pIn[456];
      pOut[568] = fPmQz*pIn[568] + 13*i2ze*pIn[461];
      pOut[569] = fPmQz*pIn[569];
      pOut[570] = fPmQz*pIn[570] + 2*i2ze*pIn[462];
      pOut[571] = fPmQz*pIn[571];
      pOut[572] = fPmQz*pIn[572] + 12*i2ze*pIn[468];
      pOut[573] = fPmQz*pIn[573] + 2*i2ze*pIn[463];
      pOut[574] = fPmQz*pIn[574] + 12*i2ze*pIn[469];
      pOut[575] = fPmQz*pIn[575];
      pOut[576] = fPmQz*pIn[576];
      pOut[577] = fPmQz*pIn[577] + 3*i2ze*pIn[465];
      pOut[578] = fPmQz*pIn[578] + 11*i2ze*pIn[471];
      pOut[579] = fPmQz*pIn[579] + 3*i2ze*pIn[467];
      pOut[580] = fPmQz*pIn[580] + 11*i2ze*pIn[473];
      pOut[581] = fPmQz*pIn[581];
      pOut[582] = fPmQz*pIn[582] + 4*i2ze*pIn[474];
      pOut[583] = fPmQz*pIn[583];
      pOut[584] = fPmQz*pIn[584] + 10*i2ze*pIn[480];
      pOut[585] = fPmQz*pIn[585] + 4*i2ze*pIn[475];
      pOut[586] = fPmQz*pIn[586] + 10*i2ze*pIn[481];
      pOut[587] = fPmQz*pIn[587];
      pOut[588] = fPmQz*pIn[588];
      pOut[589] = fPmQz*pIn[589] + 5*i2ze*pIn[477];
      pOut[590] = fPmQz*pIn[590] + 9*i2ze*pIn[483];
      pOut[591] = fPmQz*pIn[591] + 5*i2ze*pIn[479];
      pOut[592] = fPmQz*pIn[592] + 9*i2ze*pIn[485];
      pOut[593] = fPmQz*pIn[593];
      pOut[594] = fPmQz*pIn[594] + 6*i2ze*pIn[486];
      pOut[595] = fPmQz*pIn[595];
      pOut[596] = fPmQz*pIn[596] + 8*i2ze*pIn[492];
      pOut[597] = fPmQz*pIn[597] + 6*i2ze*pIn[487];
      pOut[598] = fPmQz*pIn[598] + 8*i2ze*pIn[493];
      pOut[599] = fPmQz*pIn[599];
      pOut[600] = fPmQz*pIn[600] + 7*i2ze*pIn[489];
      pOut[601] = fPmQz*pIn[601] + 7*i2ze*pIn[491];
      pOut[602] = fPmQz*pIn[602] + 12*i2ze*pIn[496];
      pOut[603] = fPmQz*pIn[603] + i2ze*pIn[458];
      pOut[604] = fPmQz*pIn[604] + i2ze*pIn[460];
      pOut[605] = fPmQz*pIn[605] + 2*i2ze*pIn[494];
      pOut[606] = fPmQz*pIn[606] + 2*i2ze*pIn[495];
      pOut[607] = fPmQz*pIn[607] + i2ze*pIn[464];
      pOut[608] = fPmQz*pIn[608] + 11*i2ze*pIn[498];
      pOut[609] = fPmQz*pIn[609] + i2ze*pIn[466];
      pOut[610] = fPmQz*pIn[610] + 11*i2ze*pIn[500];
      pOut[611] = fPmQz*pIn[611] + 10*i2ze*pIn[506];
      pOut[612] = fPmQz*pIn[612] + 10*i2ze*pIn[508];
      pOut[613] = fPmQz*pIn[613] + i2ze*pIn[470];
      pOut[614] = fPmQz*pIn[614] + 3*i2ze*pIn[497];
      pOut[615] = fPmQz*pIn[615] + i2ze*pIn[472];
      pOut[616] = fPmQz*pIn[616] + 3*i2ze*pIn[499];
      pOut[617] = fPmQz*pIn[617] + 4*i2ze*pIn[504];
      pOut[618] = fPmQz*pIn[618] + 4*i2ze*pIn[507];
      pOut[619] = fPmQz*pIn[619] + i2ze*pIn[476];
      pOut[620] = fPmQz*pIn[620] + 9*i2ze*pIn[510];
      pOut[621] = fPmQz*pIn[621] + i2ze*pIn[478];
      pOut[622] = fPmQz*pIn[622] + 9*i2ze*pIn[512];
      pOut[623] = fPmQz*pIn[623] + 8*i2ze*pIn[518];
      pOut[624] = fPmQz*pIn[624] + 8*i2ze*pIn[520];
      pOut[625] = fPmQz*pIn[625] + i2ze*pIn[482];
      pOut[626] = fPmQz*pIn[626] + 5*i2ze*pIn[509];
      pOut[627] = fPmQz*pIn[627] + i2ze*pIn[484];
      pOut[628] = fPmQz*pIn[628] + 5*i2ze*pIn[511];
      pOut[629] = fPmQz*pIn[629] + 6*i2ze*pIn[516];
      pOut[630] = fPmQz*pIn[630] + 6*i2ze*pIn[519];
      pOut[631] = fPmQz*pIn[631] + i2ze*pIn[488];
      pOut[632] = fPmQz*pIn[632] + 7*i2ze*pIn[521];
      pOut[633] = fPmQz*pIn[633] + i2ze*pIn[490];
      pOut[634] = fPmQz*pIn[634] + 7*i2ze*pIn[522];
      pOut[635] = fPmQz*pIn[635] + 2*i2ze*pIn[501];
      pOut[636] = fPmQz*pIn[636] + 2*i2ze*pIn[502];
      pOut[637] = fPmQz*pIn[637] + 10*i2ze*pIn[526];
      pOut[638] = fPmQz*pIn[638] + 2*i2ze*pIn[503];
      pOut[639] = fPmQz*pIn[639] + 2*i2ze*pIn[505];
      pOut[640] = fPmQz*pIn[640] + 3*i2ze*pIn[524];
      pOut[641] = fPmQz*pIn[641] + 9*i2ze*pIn[528];
      pOut[642] = fPmQz*pIn[642] + 3*i2ze*pIn[525];
      pOut[643] = fPmQz*pIn[643] + 9*i2ze*pIn[530];
      pOut[644] = fPmQz*pIn[644] + 2*i2ze*pIn[513];
      pOut[645] = fPmQz*pIn[645] + 4*i2ze*pIn[531];
      pOut[646] = fPmQz*pIn[646] + 2*i2ze*pIn[514];
      pOut[647] = fPmQz*pIn[647] + 8*i2ze*pIn[537];
      pOut[648] = fPmQz*pIn[648] + 4*i2ze*pIn[532];
      pOut[649] = fPmQz*pIn[649] + 8*i2ze*pIn[538];
      pOut[650] = fPmQz*pIn[650] + 2*i2ze*pIn[515];
      pOut[651] = fPmQz*pIn[651] + 2*i2ze*pIn[517];
      pOut[652] = fPmQz*pIn[652] + 5*i2ze*pIn[534];
      pOut[653] = fPmQz*pIn[653] + 7*i2ze*pIn[540];
      pOut[654] = fPmQz*pIn[654] + 5*i2ze*pIn[536];
      pOut[655] = fPmQz*pIn[655] + 7*i2ze*pIn[542];
      pOut[656] = fPmQz*pIn[656] + 2*i2ze*pIn[523];
      pOut[657] = fPmQz*pIn[657] + 6*i2ze*pIn[543];
      pOut[658] = fPmQz*pIn[658] + 6*i2ze*pIn[544];
      pOut[659] = fPmQz*pIn[659] + 8*i2ze*pIn[547];
      pOut[660] = fPmQz*pIn[660] + 3*i2ze*pIn[527];
      pOut[661] = fPmQz*pIn[661] + 3*i2ze*pIn[529];
      pOut[662] = fPmQz*pIn[662] + 4*i2ze*pIn[545];
      pOut[663] = fPmQz*pIn[663] + 4*i2ze*pIn[546];
      pOut[664] = fPmQz*pIn[664] + 3*i2ze*pIn[533];
      pOut[665] = fPmQz*pIn[665] + 7*i2ze*pIn[549];
      pOut[666] = fPmQz*pIn[666] + 3*i2ze*pIn[535];
      pOut[667] = fPmQz*pIn[667] + 7*i2ze*pIn[551];
      pOut[668] = fPmQz*pIn[668] + 6*i2ze*pIn[555];
      pOut[669] = fPmQz*pIn[669] + 6*i2ze*pIn[556];
      pOut[670] = fPmQz*pIn[670] + 3*i2ze*pIn[539];
      pOut[671] = fPmQz*pIn[671] + 5*i2ze*pIn[548];
      pOut[672] = fPmQz*pIn[672] + 3*i2ze*pIn[541];
      pOut[673] = fPmQz*pIn[673] + 5*i2ze*pIn[550];
      pOut[674] = fPmQz*pIn[674] + 4*i2ze*pIn[552];
      pOut[675] = fPmQz*pIn[675] + 4*i2ze*pIn[553];
      pOut[676] = fPmQz*pIn[676] + 6*i2ze*pIn[559];
      pOut[677] = fPmQz*pIn[677] + 4*i2ze*pIn[554];
      pOut[678] = fPmQz*pIn[678] + 5*i2ze*pIn[557];
      pOut[679] = fPmQz*pIn[679] + 5*i2ze*pIn[558];
      if (lab == 14) return;
   }
   assert(0);
}

// Calculate (a0|c)^0 from (a0|0)^{m}, for a = la..lab, with solid harmonic 'c' functions
// Notes:
// - pMem must hold memory for nCartX(lab) x nCartX(lc) doubles.
// Inputs:
// - pIn: [nCartX(lab)]-array with elements (a0|0)^m for m=lc (only m=lc!)
// - InvEtaABC = 1/(alpha+beta+gamma)
// - PmQ = (P-Q) = (P-C) (since Q=C for 3c case here)
// - rie = rho/eta = rho/(gamma+delta) = rho/gamma (since delta=0 for 3c case here)
// Output:
// - pOut: [nCartX(lab) - nCartX(la-1)] x [(2*lc+1)] array.
void OsrrB_3c_shc(double *IR_RP pOut, double const *IR_RP pIn, double *IR_RP pMem, int la, unsigned lab, unsigned lc, double fPmQx, double fPmQy, double fPmQz, double InvEtaABC, double rie)
{
   size_t nCompA = nCartX(lab);
   size_t iCompA = nCartX(la-1);
   size_t nCompA_Out = nCompA - iCompA;
   double i2ze = .5 * InvEtaABC;
   fPmQx *= rie; fPmQy *= rie; fPmQz *= rie;
   switch(lc) {
      case 0: {
         for (size_t i = 0; i < nCompA_Out; ++ i)
            pOut[i] = pIn[i+iCompA];
         return;
      }
      case 1: {
         double *IR_RP pOut_ = pOut - iCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pIn[0], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pIn[0], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pIn[0], fPmQz, la, lab, i2ze);
         return;
      }
      case 2: {
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la1, lab, i2ze);
         double *IR_RP pOut_ = pMem + 3 * nCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 0*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 1*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 2*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[ 1*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 2*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[ 2*nCompA], fPmQy, la, lab, i2ze);
         ShTrC2(pOut, pOut_ + iCompA, nCompA_Out);
         return;
      }
      case 3: {
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 6*nCompA], &pMem[ 2*nCompA], fPmQy, la1, lab, i2ze);
         double *IR_RP pOut_ = pMem + 7 * nCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 3*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 4*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 5*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[ 4*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 5*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[ 3*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 6*nCompA_Out], &pMem[ 5*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[ 3*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 8*nCompA_Out], &pMem[ 4*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[ 6*nCompA], fPmQx, la, lab, i2ze);
         ShTrC3(pOut, pOut_ + iCompA, nCompA_Out);
         return;
      }
      case 4: {
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[ 9*nCompA], &pMem[ 4*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[10*nCompA], &pMem[ 5*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[11*nCompA], &pMem[ 3*nCompA], fPmQz, la1, lab, i2ze);
         double *IR_RP pOut_ = pMem + 12 * nCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 6*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 7*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 8*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 3*nCompA_Out], &pMem[ 6*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 7*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 5*nCompA_Out], &pMem[ 6*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 6*nCompA_Out], &pMem[ 8*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[ 7*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 8*nCompA_Out], &pMem[ 8*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[ 9*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[10*nCompA_Out], &pMem[11*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[11*nCompA_Out], &pMem[10*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[12*nCompA_Out], &pMem[10*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[13*nCompA_Out], &pMem[ 9*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[14*nCompA_Out], &pMem[11*nCompA], fPmQy, la, lab, i2ze);
         ShTrC4(pOut, pOut_ + iCompA, nCompA_Out);
         return;
      }
      case 5: {
         unsigned la4 = 0; if (la > 4) la4 = (unsigned)la - 4;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la4, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la4, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la4, lab, i2ze);
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[ 9*nCompA], &pMem[ 4*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[10*nCompA], &pMem[ 5*nCompA], fPmQy, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[11*nCompA], &pMem[ 6*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[12*nCompA], &pMem[ 7*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[13*nCompA], &pMem[ 8*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[14*nCompA], &pMem[ 7*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[15*nCompA], &pMem[ 6*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[16*nCompA], &pMem[ 8*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[17*nCompA], &pMem[ 9*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[18*nCompA], &pMem[10*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[19*nCompA], &pMem[10*nCompA], fPmQx, la1, lab, i2ze);
         double *IR_RP pOut_ = pMem + 20 * nCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[11*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[12*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[13*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[12*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[13*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[11*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 6*nCompA_Out], &pMem[13*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 7*nCompA_Out], &pMem[15*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 8*nCompA_Out], &pMem[12*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[17*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[10*nCompA_Out], &pMem[15*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[11*nCompA_Out], &pMem[14*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[12*nCompA_Out], &pMem[18*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[13*nCompA_Out], &pMem[16*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[14*nCompA_Out], &pMem[18*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[15*nCompA_Out], &pMem[15*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[16*nCompA_Out], &pMem[14*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[17*nCompA_Out], &pMem[19*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[18*nCompA_Out], &pMem[18*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[19*nCompA_Out], &pMem[19*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[20*nCompA_Out], &pMem[17*nCompA], fPmQz, la, lab, i2ze);
         ShTrC5(pOut, pOut_ + iCompA, nCompA_Out);
         return;
      }
      case 6: {
         unsigned la5 = 0; if (la > 5) la5 = (unsigned)la - 5;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la5, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la5, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la5, lab, i2ze);
         unsigned la4 = 0; if (la > 4) la4 = (unsigned)la - 4;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la4, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la4, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la4, lab, i2ze);
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 9*nCompA], &pMem[ 5*nCompA], fPmQy, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[10*nCompA], &pMem[ 6*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[11*nCompA], &pMem[ 7*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[12*nCompA], &pMem[ 8*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[13*nCompA], &pMem[ 7*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[14*nCompA], &pMem[ 6*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[15*nCompA], &pMem[ 9*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[16*nCompA], &pMem[ 9*nCompA], fPmQx, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[17*nCompA], &pMem[10*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[18*nCompA], &pMem[11*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[19*nCompA], &pMem[12*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[20*nCompA], &pMem[12*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[21*nCompA], &pMem[10*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[22*nCompA], &pMem[14*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[23*nCompA], &pMem[13*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[24*nCompA], &pMem[15*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[25*nCompA], &pMem[15*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[26*nCompA], &pMem[14*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[27*nCompA], &pMem[13*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[28*nCompA], &pMem[16*nCompA], fPmQx, la1, lab, i2ze);
         double *IR_RP pOut_ = pMem + 29 * nCompA;
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[17*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[18*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[19*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[21*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[18*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 5*nCompA_Out], &pMem[17*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 6*nCompA_Out], &pMem[19*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[18*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 8*nCompA_Out], &pMem[19*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 9*nCompA_Out], &pMem[21*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[10*nCompA_Out], &pMem[22*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[11*nCompA_Out], &pMem[23*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[12*nCompA_Out], &pMem[20*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[13*nCompA_Out], &pMem[24*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[14*nCompA_Out], &pMem[25*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[15*nCompA_Out], &pMem[23*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[16*nCompA_Out], &pMem[22*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[17*nCompA_Out], &pMem[25*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[18*nCompA_Out], &pMem[20*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[19*nCompA_Out], &pMem[27*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[20*nCompA_Out], &pMem[26*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[21*nCompA_Out], &pMem[28*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[22*nCompA_Out], &pMem[24*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[23*nCompA_Out], &pMem[26*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[24*nCompA_Out], &pMem[25*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[25*nCompA_Out], &pMem[27*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[26*nCompA_Out], &pMem[28*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[27*nCompA_Out], &pMem[28*nCompA], fPmQy, la, lab, i2ze);
         ShTrC6(pOut, pOut_ + iCompA, nCompA_Out);
         return;
      }
   }
   assert(0);
}

// Calculate (a0|c)^0 from (a0|0)^{m}, for a = la..lab, with Hermite-cartesian 'c' functions
// Notes:
// - pMem must hold memory for nCartX(lab) x nCartX(lc) doubles.
// Inputs:
// - pIn: [nCartX(lab)]-array with elements (a0|0)^m for m=lc (only m=lc!)
// - InvEtaABC = 1/(alpha+beta+gamma)
// - PmQ = (P-Q) = (P-C) (since Q=C for 3c case here)
// - rie = rho/eta = rho/(gamma+delta) = rho/gamma (since delta=0 for 3c case here)
// Output:
// - pOut: [nCartX(lab) - nCartX(la-1)] x [nCartY(lc)] array
void OsrrB_3c_cac(double *IR_RP pOut, double const *IR_RP pIn, double *IR_RP pMem, int la, unsigned lab, unsigned lc, double fPmQx, double fPmQy, double fPmQz, double InvEtaABC, double rie)
{
   size_t nCompA = nCartX(lab);
   size_t iCompA = nCartX(la-1);
   size_t nCompA_Out = nCompA - iCompA;
   double i2ze = .5 * InvEtaABC;
   double *IR_RP pOut_ = pOut - iCompA;
   fPmQx *= rie; fPmQy *= rie; fPmQz *= rie;
   switch(lc) {
      case 0: {
         for (size_t i = 0; i < nCompA_Out; ++ i)
            pOut[i] = pIn[i+iCompA];
         return;
      }
      case 1: {
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pIn[0], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pIn[0], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pIn[0], fPmQz, la, lab, i2ze);
         return;
      }
      case 2: {
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 0*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 1*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 2*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[ 1*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 2*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[ 2*nCompA], fPmQy, la, lab, i2ze);
         return;
      }
      case 3: {
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 6*nCompA], &pMem[ 2*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 3*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 4*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 5*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[ 4*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 5*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[ 3*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 6*nCompA_Out], &pMem[ 5*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[ 3*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 8*nCompA_Out], &pMem[ 4*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[ 6*nCompA], fPmQx, la, lab, i2ze);
         return;
      }
      case 4: {
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[ 9*nCompA], &pMem[ 4*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[10*nCompA], &pMem[ 5*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[11*nCompA], &pMem[ 3*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[ 6*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[ 7*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[ 8*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 3*nCompA_Out], &pMem[ 6*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[ 7*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 5*nCompA_Out], &pMem[ 6*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 6*nCompA_Out], &pMem[ 8*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[ 7*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 8*nCompA_Out], &pMem[ 8*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[ 9*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[10*nCompA_Out], &pMem[11*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[11*nCompA_Out], &pMem[10*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[12*nCompA_Out], &pMem[10*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[13*nCompA_Out], &pMem[ 9*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[14*nCompA_Out], &pMem[11*nCompA], fPmQy, la, lab, i2ze);
         return;
      }
      case 5: {
         unsigned la4 = 0; if (la > 4) la4 = (unsigned)la - 4;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la4, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la4, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la4, lab, i2ze);
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[ 9*nCompA], &pMem[ 4*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[10*nCompA], &pMem[ 5*nCompA], fPmQy, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[11*nCompA], &pMem[ 6*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[12*nCompA], &pMem[ 7*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[13*nCompA], &pMem[ 8*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[14*nCompA], &pMem[ 7*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[15*nCompA], &pMem[ 6*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[16*nCompA], &pMem[ 8*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[17*nCompA], &pMem[ 9*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[18*nCompA], &pMem[10*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[19*nCompA], &pMem[10*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[11*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[12*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[13*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[12*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[13*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 5*nCompA_Out], &pMem[11*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 6*nCompA_Out], &pMem[13*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 7*nCompA_Out], &pMem[15*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 8*nCompA_Out], &pMem[12*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 9*nCompA_Out], &pMem[17*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[10*nCompA_Out], &pMem[15*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[11*nCompA_Out], &pMem[14*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[12*nCompA_Out], &pMem[18*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[13*nCompA_Out], &pMem[16*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[14*nCompA_Out], &pMem[18*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[15*nCompA_Out], &pMem[15*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[16*nCompA_Out], &pMem[14*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[17*nCompA_Out], &pMem[19*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[18*nCompA_Out], &pMem[18*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[19*nCompA_Out], &pMem[19*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[20*nCompA_Out], &pMem[17*nCompA], fPmQz, la, lab, i2ze);
         return;
      }
      case 6: {
         unsigned la5 = 0; if (la > 5) la5 = (unsigned)la - 5;
         OsrrB_3cKerM_x(&pMem[ 0*nCompA], &pIn[0], fPmQx, la5, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 1*nCompA], &pIn[0], fPmQy, la5, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 2*nCompA], &pIn[0], fPmQz, la5, lab, i2ze);
         unsigned la4 = 0; if (la > 4) la4 = (unsigned)la - 4;
         OsrrB_3cKerM_x(&pMem[ 3*nCompA], &pMem[ 0*nCompA], fPmQx, la4, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 4*nCompA], &pMem[ 1*nCompA], fPmQy, la4, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 5*nCompA], &pMem[ 2*nCompA], fPmQz, la4, lab, i2ze);
         unsigned la3 = 0; if (la > 3) la3 = (unsigned)la - 3;
         OsrrB_3cKerM_x(&pMem[ 6*nCompA], &pMem[ 3*nCompA], fPmQx, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 7*nCompA], &pMem[ 4*nCompA], fPmQy, la3, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[ 8*nCompA], &pMem[ 5*nCompA], fPmQz, la3, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[ 9*nCompA], &pMem[ 5*nCompA], fPmQy, la3, lab, i2ze);
         unsigned la2 = 0; if (la > 2) la2 = (unsigned)la - 2;
         OsrrB_3cKerM_x(&pMem[10*nCompA], &pMem[ 6*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[11*nCompA], &pMem[ 7*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[12*nCompA], &pMem[ 8*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[13*nCompA], &pMem[ 7*nCompA], fPmQx, la2, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[14*nCompA], &pMem[ 6*nCompA], fPmQz, la2, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[15*nCompA], &pMem[ 9*nCompA], fPmQy, la2, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[16*nCompA], &pMem[ 9*nCompA], fPmQx, la2, lab, i2ze);
         unsigned la1 = 0; if (la > 1) la1 = (unsigned)la - 1;
         OsrrB_3cKerM_x(&pMem[17*nCompA], &pMem[10*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[18*nCompA], &pMem[11*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[19*nCompA], &pMem[12*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[20*nCompA], &pMem[12*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[21*nCompA], &pMem[10*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[22*nCompA], &pMem[14*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[23*nCompA], &pMem[13*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[24*nCompA], &pMem[15*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[25*nCompA], &pMem[15*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_y(&pMem[26*nCompA], &pMem[14*nCompA], fPmQy, la1, lab, i2ze);
         OsrrB_3cKerM_z(&pMem[27*nCompA], &pMem[13*nCompA], fPmQz, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pMem[28*nCompA], &pMem[16*nCompA], fPmQx, la1, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 0*nCompA_Out], &pMem[17*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 1*nCompA_Out], &pMem[18*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 2*nCompA_Out], &pMem[19*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 3*nCompA_Out], &pMem[21*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 4*nCompA_Out], &pMem[18*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 5*nCompA_Out], &pMem[17*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[ 6*nCompA_Out], &pMem[19*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[ 7*nCompA_Out], &pMem[18*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 8*nCompA_Out], &pMem[19*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[ 9*nCompA_Out], &pMem[21*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[10*nCompA_Out], &pMem[22*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[11*nCompA_Out], &pMem[23*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[12*nCompA_Out], &pMem[20*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[13*nCompA_Out], &pMem[24*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[14*nCompA_Out], &pMem[25*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[15*nCompA_Out], &pMem[23*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[16*nCompA_Out], &pMem[22*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[17*nCompA_Out], &pMem[25*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[18*nCompA_Out], &pMem[20*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[19*nCompA_Out], &pMem[27*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[20*nCompA_Out], &pMem[26*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[21*nCompA_Out], &pMem[28*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[22*nCompA_Out], &pMem[24*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[23*nCompA_Out], &pMem[26*nCompA], fPmQy, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[24*nCompA_Out], &pMem[25*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_x(&pOut_[25*nCompA_Out], &pMem[27*nCompA], fPmQx, la, lab, i2ze);
         OsrrB_3cKerM_z(&pOut_[26*nCompA_Out], &pMem[28*nCompA], fPmQz, la, lab, i2ze);
         OsrrB_3cKerM_y(&pOut_[27*nCompA_Out], &pMem[28*nCompA], fPmQy, la, lab, i2ze);
         return;
      }
   }
   assert(0);
}

// cart_vec_t representations of the components in CartX(14)
cart_vec_t const ix2v[680] = {
   0, 1, 15, 225, 2, 30, 450, 16, 226, 240, 3, 45, 675, 31, 451, 17, 465, 227, 255, 241, 4, 60, 900, 18, 46, 228,
   676, 270, 690, 32, 452, 480, 466, 256, 242, 5, 75, 1125, 61, 901, 19, 915, 229, 285, 33, 453, 47, 495, 677, 705, 243, 271,
   691, 481, 467, 257, 6, 90, 1350, 20, 76, 230, 1126, 300, 1140, 34, 454, 62, 902, 510, 930, 48, 678, 720, 916, 286, 244, 468,
   496, 258, 706, 272, 692, 482, 7, 105, 1575, 91, 1351, 21, 1365, 231, 315, 35, 455, 77, 525, 1127, 1155, 63, 903, 49, 945, 679,
   735, 245, 301, 1141, 511, 931, 469, 917, 259, 287, 273, 693, 721, 483, 497, 707, 8, 120, 1800, 22, 106, 232, 1576, 330, 1590, 36,
   456, 92, 1352, 540, 1380, 50, 78, 680, 1128, 750, 1170, 64, 904, 960, 1366, 316, 246, 470, 526, 260, 1156, 302, 1142, 918, 946, 288,
   736, 274, 694, 484, 512, 932, 498, 708, 722, 9, 135, 2025, 121, 1801, 23, 1815, 233, 345, 37, 457, 107, 555, 1577, 1605, 93, 1353,
   51, 1395, 681, 765, 65, 905, 79, 975, 1129, 1185, 247, 331, 1591, 541, 1381, 471, 1367, 261, 317, 275, 695, 303, 1143, 751, 1171, 961,
   919, 289, 485, 527, 1157, 513, 933, 499, 947, 709, 737, 723, 10, 150, 2250, 24, 136, 234, 2026, 360, 2040, 38, 458, 122, 1802, 570,
   1830, 52, 108, 682, 1578, 780, 1620, 66, 906, 94, 1354, 990, 1410, 80, 1130, 1200, 1816, 346, 248, 472, 556, 262, 1606, 332, 1592, 1368,
   1396, 318, 766, 276, 696, 920, 976, 290, 1186, 304, 1144, 486, 542, 1382, 500, 528, 710, 1158, 752, 1172, 514, 934, 962, 948, 738, 724,
   11, 165, 2475, 151, 2251, 25, 2265, 235, 375, 39, 459, 137, 585, 2027, 2055, 123, 1803, 53, 1845, 683, 795, 67, 907, 109, 1005, 1579,
   1635, 95, 1355, 81, 1425, 1131, 1215, 249, 361, 2041, 571, 1831, 473, 1817, 263, 347, 277, 697, 333, 1593, 781, 1621, 991, 1411, 921, 1369,
   291, 319, 305, 1145, 1201, 487, 557, 1607, 543, 1383, 501, 1397, 711, 767, 515, 935, 529, 977, 1159, 1187, 725, 753, 1173, 963, 949, 739,
   12, 180, 2700, 26, 166, 236, 2476, 390, 2490, 40, 460, 152, 2252, 600, 2280, 54, 138, 684, 2028, 810, 2070, 68, 908, 124, 1804, 1020,
   1860, 82, 110, 1132, 1580, 1230, 1650, 96, 1356, 1440, 2266, 376, 250, 474, 586, 264, 2056, 362, 2042, 1818, 1846, 348, 796, 278, 698, 922,
   1006, 292, 1636, 334, 1594, 1370, 1426, 320, 1216, 306, 1146, 488, 572, 1832, 502, 558, 712, 1608, 782, 1622, 516, 936, 544, 1384, 992, 1412,
   530, 1160, 1202, 1398, 768, 726, 950, 978, 740, 1188, 754, 1174, 964, 13, 195, 2925, 181, 2701, 27, 2715, 237, 405, 41, 461, 167, 615,
   2477, 2505, 153, 2253, 55, 2295, 685, 825, 69, 909, 139, 1035, 2029, 2085, 125, 1805, 83, 1875, 1133, 1245, 97, 1357, 111, 1455, 1581, 1665,
   251, 391, 2491, 601, 2281, 475, 2267, 265, 377, 279, 699, 363, 2043, 811, 2071, 1021, 1861, 923, 1819, 293, 349, 307, 1147, 335, 1595, 1231,
   1651, 1441, 1371, 321, 489, 587, 2057, 573, 1833, 503, 1847, 713, 797, 517, 937, 559, 1007, 1609, 1637, 545, 1385, 531, 1427, 1161, 1217, 727,
   783, 1623, 993, 1413, 951, 1399, 741, 769, 755, 1175, 1203, 965, 979, 1189, 14, 210, 3150, 28, 196, 238, 2926, 420, 2940, 42, 462, 182,
   2702, 630, 2730, 56, 168, 686, 2478, 840, 2520, 70, 910, 154, 2254, 1050, 2310, 84, 140, 1134, 2030, 1260, 2100, 98, 1358, 126, 1806, 1470,
   1890, 112, 1582, 1680, 2716, 406, 252, 476, 616, 266, 2506, 392, 2492, 2268, 2296, 378, 826, 280, 700, 924, 1036, 294, 2086, 364, 2044, 1820,
   1876, 350, 1246, 308, 1148, 1372, 1456, 322, 1666, 336, 1596, 490, 602, 2282, 504, 588, 714, 2058, 812, 2072, 518, 938, 574, 1834, 1022, 1862,
   532, 560, 1162, 1610, 1232, 1652, 546, 1386, 1442, 1848, 798, 728, 952, 1008, 742, 1638, 784, 1624, 1400, 1428, 770, 1218, 756, 1176, 966, 994,
   1414, 980, 1190, 1204
}; // 1.33 kb

// indices for mapping cart_vec_t objects into CartX. It's a 15^3 array.
cart_index_t const iv2x[3151] = {
   0, 1, 4, 10, 20, 35, 56, 84, 120, 165, 220, 286, 364, 455, 560, 2, 7, 15, 23, 40, 59, 89, 123, 170, 223, 291,
   367, 460, 563, 0, 5, 13, 29, 44, 65, 93, 129, 174, 229, 295, 373, 464, 569, 0, 0, 11, 24, 46, 71, 101, 135, 182,
   235, 303, 379, 472, 575, 0, 0, 0, 21, 38, 67, 99, 141, 186, 241, 307, 385, 476, 581, 0, 0, 0, 0, 36, 60, 95,
   136, 188, 247, 315, 391, 484, 587, 0, 0, 0, 0, 0, 57, 87, 131, 180, 243, 313, 397, 488, 593, 0, 0, 0, 0, 0,
   0, 85, 124, 176, 236, 309, 392, 490, 599, 0, 0, 0, 0, 0, 0, 0, 121, 168, 231, 301, 387, 482, 595, 0, 0, 0,
   0, 0, 0, 0, 0, 166, 224, 297, 380, 478, 588, 0, 0, 0, 0, 0, 0, 0, 0, 0, 221, 289, 375, 470, 583, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 287, 368, 466, 576, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 365, 458,
   571, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 456, 564, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 561, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 3, 8, 17, 25, 42, 61, 91, 125, 172,
   225, 293, 369, 462, 565, 0, 9, 19, 34, 50, 76, 105, 146, 192, 252, 319, 402, 494, 604, 0, 0, 18, 33, 55, 79, 112,
   149, 199, 255, 326, 405, 501, 607, 0, 0, 0, 27, 51, 81, 114, 157, 201, 263, 328, 413, 503, 615, 0, 0, 0, 0, 43,
   75, 113, 155, 209, 267, 338, 417, 513, 619, 0, 0, 0, 0, 0, 63, 106, 151, 203, 269, 340, 425, 515, 627, 0, 0, 0,
   0, 0, 0, 92, 145, 200, 261, 339, 423, 523, 631, 0, 0, 0, 0, 0, 0, 0, 127, 193, 257, 330, 419, 517, 633, 0,
   0, 0, 0, 0, 0, 0, 0, 173, 251, 327, 411, 514, 625, 0, 0, 0, 0, 0, 0, 0, 0, 0, 227, 320, 407, 505,
   621, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 294, 401, 502, 613, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   371, 495, 609, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 463, 603, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 567, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 6, 14, 30, 45, 66, 94, 130, 175, 230, 296, 374, 465, 570, 0, 0, 16, 32, 54,
   77, 110, 147, 197, 253, 324, 403, 499, 605, 0, 0, 0, 31, 53, 83, 117, 159, 210, 271, 343, 427, 524, 635, 0, 0, 0,
   0, 47, 78, 118, 162, 215, 274, 348, 430, 529, 638, 0, 0, 0, 0, 0, 69, 108, 160, 213, 280, 352, 436, 533, 644, 0,
   0, 0, 0, 0, 0, 96, 148, 211, 275, 354, 442, 541, 650, 0, 0, 0, 0, 0, 0, 0, 133, 195, 272, 346, 438, 539,
   656, 0, 0, 0, 0, 0, 0, 0, 0, 177, 254, 344, 431, 535, 651, 0, 0, 0, 0, 0, 0, 0, 0, 0, 233, 322,
   428, 527, 646, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 298, 404, 525, 639, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 377, 497, 636, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 467, 606, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 573, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 12,
   26, 48, 72, 103, 137, 184, 237, 305, 381, 474, 577, 0, 0, 0, 28, 52, 82, 115, 158, 202, 264, 329, 414, 504, 616, 0,
   0, 0, 0, 49, 80, 119, 163, 217, 276, 350, 432, 531, 640, 0, 0, 0, 0, 0, 73, 116, 164, 219, 285, 358, 447, 545,
   661, 0, 0, 0, 0, 0, 0, 104, 156, 218, 284, 363, 450, 552, 664, 0, 0, 0, 0, 0, 0, 0, 139, 205, 278, 359,
   452, 554, 672, 0, 0, 0, 0, 0, 0, 0, 0, 185, 262, 351, 446, 553, 670, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   239, 332, 434, 546, 666, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 306, 412, 532, 660, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 383, 507, 642, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 475, 614, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 579, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 22, 39, 68, 100, 142, 187, 242, 308, 386, 477,
   582, 0, 0, 0, 0, 41, 74, 111, 153, 208, 265, 336, 415, 511, 617, 0, 0, 0, 0, 0, 70, 109, 161, 214, 281, 353,
   437, 534, 645, 0, 0, 0, 0, 0, 0, 102, 154, 216, 283, 362, 448, 550, 662, 0, 0, 0, 0, 0, 0, 0, 143, 207,
   282, 361, 454, 557, 674, 0, 0, 0, 0, 0, 0, 0, 0, 189, 266, 355, 449, 558, 677, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 245, 334, 440, 548, 675, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 310, 416, 536, 663, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 389, 509, 648, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 479, 618, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 585, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 37, 62, 97, 138, 190, 248, 317, 393, 486, 589, 0, 0, 0, 0, 0, 64, 107, 152, 204,
   270, 341, 426, 516, 628, 0, 0, 0, 0, 0, 0, 98, 150, 212, 277, 356, 443, 543, 652, 0, 0, 0, 0, 0, 0, 0,
   140, 206, 279, 360, 453, 555, 673, 0, 0, 0, 0, 0, 0, 0, 0, 191, 268, 357, 451, 559, 678, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 249, 342, 444, 556, 679, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 318, 424, 544, 671, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 395, 519, 654, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 487, 626, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 591, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 58, 88,
   132, 181, 244, 314, 398, 489, 594, 0, 0, 0, 0, 0, 0, 90, 144, 198, 259, 337, 421, 522, 629, 0, 0, 0, 0, 0,
   0, 0, 134, 196, 273, 347, 439, 540, 657, 0, 0, 0, 0, 0, 0, 0, 0, 183, 260, 349, 445, 551, 668, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 246, 335, 441, 549, 676, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 316, 422, 542, 669, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 399, 521, 658, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 491,
   630, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 597, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 86, 126, 178, 238, 311, 394, 492, 600, 0, 0, 0,
   0, 0, 0, 0, 128, 194, 258, 331, 420, 518, 634, 0, 0, 0, 0, 0, 0, 0, 0, 179, 256, 345, 433, 537, 653, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 240, 333, 435, 547, 667, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 312, 418, 538,
   665, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 396, 520, 655, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 493, 632, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 601, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 122, 169, 232, 302, 388, 483, 596, 0, 0, 0, 0, 0, 0, 0, 0, 171, 250, 325, 409, 512,
   623, 0, 0, 0, 0, 0, 0, 0, 0, 0, 234, 323, 429, 528, 647, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 304,
   410, 530, 659, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 390, 510, 649, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 485, 624, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 598, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 167, 226, 299,
   382, 480, 590, 0, 0, 0, 0, 0, 0, 0, 0, 0, 228, 321, 408, 506, 622, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 300, 406, 526, 641, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 384, 508, 643, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 481, 620, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 592, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 222, 290, 376, 471, 584, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 292, 400, 500, 611, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 378, 498, 637, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 473, 612, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 586, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 288, 370, 468, 578, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 372, 496, 610, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 469, 608, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 580, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 366, 459, 572, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 461, 602, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   574, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 457, 566, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 568, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
   0, 0, 0, 0, 562
}; // 6.15 kb

// indices for factoring [CartX(la+lb) - nCartX(la)] into [CartX(lb)] x [CartY(la)]
// format: nCartY(la) x CartX(lb)
static const cart_index_t iCartXY_ab0_a0[1] = {
   0
}; // 0.00 kb
static const cart_index_t iCartXY_ab1_a0[4] = {
   0, 1, 2, 3
}; // 0.01 kb
static const cart_index_t iCartXY_ab1_a1[3] = {
   0, 1, 2
}; // 0.01 kb
static const cart_index_t iCartXY_ab2_a0[10] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9
}; // 0.02 kb
static const cart_index_t iCartXY_ab2_a1[12] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5
}; // 0.02 kb
static const cart_index_t iCartXY_ab2_a2[6] = {
   0, 1, 2, 3, 4, 5
}; // 0.01 kb
static const cart_index_t iCartXY_ab3_a0[20] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19
}; // 0.04 kb
static const cart_index_t iCartXY_ab3_a1[30] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15
}; // 0.06 kb
static const cart_index_t iCartXY_ab3_a2[24] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12
}; // 0.05 kb
static const cart_index_t iCartXY_ab3_a3[10] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9
}; // 0.02 kb
static const cart_index_t iCartXY_ab4_a0[35] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34
}; // 0.07 kb
static const cart_index_t iCartXY_ab4_a1[60] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31
}; // 0.12 kb
static const cart_index_t iCartXY_ab4_a2[60] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27
}; // 0.12 kb
static const cart_index_t iCartXY_ab4_a3[40] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22
}; // 0.08 kb
static const cart_index_t iCartXY_ab4_a4[15] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14
}; // 0.03 kb
static const cart_index_t iCartXY_ab5_a0[56] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55
}; // 0.11 kb
static const cart_index_t iCartXY_ab5_a1[105] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53
}; // 0.21 kb
static const cart_index_t iCartXY_ab5_a2[120] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49
}; // 0.23 kb
static const cart_index_t iCartXY_ab5_a3[100] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43
}; // 0.20 kb
static const cart_index_t iCartXY_ab5_a4[60] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34
}; // 0.12 kb
static const cart_index_t iCartXY_ab5_a5[21] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20
}; // 0.04 kb
static const cart_index_t iCartXY_ab6_a0[84] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83
}; // 0.16 kb
static const cart_index_t iCartXY_ab6_a1[168] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82
}; // 0.33 kb
static const cart_index_t iCartXY_ab6_a2[210] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79
}; // 0.41 kb
static const cart_index_t iCartXY_ab6_a3[200] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73
}; // 0.39 kb
static const cart_index_t iCartXY_ab6_a4[150] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63
}; // 0.29 kb
static const cart_index_t iCartXY_ab6_a5[84] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48
}; // 0.16 kb
static const cart_index_t iCartXY_ab6_a6[28] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27
}; // 0.05 kb
static const cart_index_t iCartXY_ab7_a0[120] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119
}; // 0.23 kb
static const cart_index_t iCartXY_ab7_a1[252] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118
}; // 0.49 kb
static const cart_index_t iCartXY_ab7_a2[336] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114
}; // 0.66 kb
static const cart_index_t iCartXY_ab7_a3[350] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107
}; // 0.68 kb
static const cart_index_t iCartXY_ab7_a4[300] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97
}; // 0.59 kb
static const cart_index_t iCartXY_ab7_a5[210] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83
}; // 0.41 kb
static const cart_index_t iCartXY_ab7_a6[112] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63
}; // 0.22 kb
static const cart_index_t iCartXY_ab8_a0[165] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164
}; // 0.32 kb
static const cart_index_t iCartXY_ab8_a1[360] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160
}; // 0.70 kb
static const cart_index_t iCartXY_ab8_a2[504] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160
}; // 0.98 kb
static const cart_index_t iCartXY_ab8_a3[560] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152
}; // 1.09 kb
static const cart_index_t iCartXY_ab8_a4[525] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139
}; // 1.03 kb
static const cart_index_t iCartXY_ab8_a5[420] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127
}; // 0.82 kb
static const cart_index_t iCartXY_ab8_a6[280] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108
}; // 0.55 kb
static const cart_index_t iCartXY_ab9_a0[220] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219
}; // 0.43 kb
static const cart_index_t iCartXY_ab9_a1[495] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215
}; // 0.97 kb
static const cart_index_t iCartXY_ab9_a2[720] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212
}; // 1.41 kb
static const cart_index_t iCartXY_ab9_a3[840] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209
}; // 1.64 kb
static const cart_index_t iCartXY_ab9_a4[840] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195
}; // 1.64 kb
static const cart_index_t iCartXY_ab9_a5[735] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180
}; // 1.44 kb
static const cart_index_t iCartXY_ab9_a6[560] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163
}; // 1.09 kb
static const cart_index_t iCartXY_ab10_a0[286] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233,
   234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259,
   260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285
}; // 0.56 kb
static const cart_index_t iCartXY_ab10_a1[660] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215, 219, 222, 224, 223, 220, 226, 225, 227, 221, 230, 223, 250, 231, 249, 225, 222, 228, 251, 249, 233, 227, 224, 251, 229, 250,
   226, 232, 228, 234, 254, 229, 252, 236, 235, 230, 256, 253, 232, 238, 237, 257, 231, 255, 239, 233, 242, 235, 260, 243, 258, 237,
   234, 240, 262, 259, 245, 239, 236, 263, 241, 261, 238, 244, 240, 246, 266, 241, 264, 247, 246, 242, 268, 265, 244, 248, 247, 269,
   243, 267, 248, 245, 251, 254, 252, 256, 250, 253, 257, 255, 249, 271, 253, 261, 272, 259, 255, 252, 270, 263, 258, 272, 257, 254,
   262, 270, 260, 256, 271, 262, 266, 273, 263, 275, 264, 268, 260, 274, 269, 276, 258, 277, 261, 265, 278, 267, 259, 281, 265, 267,
   264, 280, 269, 266, 268, 279, 270, 273, 275, 274, 271, 277, 276, 278, 272, 279, 274, 283, 280, 282, 276, 273, 279, 284, 282, 281,
   278, 275, 284, 280, 283, 277, 281, 284, 283, 282
}; // 1.29 kb
static const cart_index_t iCartXY_ab10_a2[990] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212, 216, 225, 226, 219, 221, 248, 227, 217,
   229, 220, 247, 223, 228, 230, 218, 246, 222, 224, 219, 231, 249, 225, 248, 251, 232, 220, 250, 227, 253, 247, 221, 251, 233, 248,
   226, 249, 234, 252, 222, 254, 228, 246, 253, 223, 235, 247, 250, 229, 254, 236, 224, 252, 246, 230, 225, 237, 267, 231, 251, 259,
   226, 267, 238, 249, 233, 260, 239, 227, 268, 232, 257, 253, 240, 269, 228, 255, 234, 254, 268, 229, 241, 250, 258, 235, 269, 242,
   230, 256, 252, 236, 231, 243, 270, 237, 259, 263, 243, 232, 271, 239, 265, 257, 233, 272, 244, 260, 238, 261, 244, 273, 234, 266,
   240, 255, 274, 235, 245, 258, 262, 241, 275, 245, 236, 264, 256, 242, 237, 239, 276, 243, 263, 265, 238, 277, 240, 261, 244, 266,
   278, 241, 242, 262, 264, 245, 255, 256, 246, 269, 254, 252, 257, 247, 258, 253, 268, 250, 248, 259, 260, 251, 249, 267, 249, 270,
   261, 267, 260, 272, 271, 250, 262, 268, 274, 258, 251, 263, 272, 259, 267, 270, 273, 264, 252, 275, 269, 256, 265, 253, 274, 257,
   271, 268, 266, 275, 254, 273, 255, 269, 261, 279, 255, 277, 266, 273, 279, 262, 256, 278, 275, 264, 263, 257, 280, 265, 276, 271,
   280, 258, 264, 274, 278, 262, 259, 265, 281, 263, 270, 276, 260, 281, 266, 272, 261, 277, 267, 276, 277, 270, 272, 281, 276, 268,
   278, 271, 280, 274, 277, 278, 269, 279, 273, 275, 270, 271, 279, 276, 281, 280, 272, 280, 273, 281, 277, 279, 281, 274, 275, 280,
   279, 278
}; // 1.93 kb
static const cart_index_t iCartXY_ab10_a3[1200] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209, 210, 225, 227, 219, 220, 213, 243, 215, 245, 242, 226, 211, 229, 214, 244, 221, 223, 247,
   217, 241, 228, 230, 212, 246, 216, 248, 218, 222, 224, 240, 233, 214, 252, 221, 262, 226, 244, 251, 241, 247, 234, 250, 216, 263,
   222, 249, 240, 228, 246, 248, 213, 231, 254, 225, 243, 219, 261, 242, 253, 245, 249, 236, 218, 250, 240, 263, 224, 248, 230, 246,
   215, 253, 232, 245, 227, 242, 254, 220, 261, 243, 251, 217, 235, 241, 252, 247, 229, 262, 223, 244, 219, 237, 266, 231, 261, 225,
   264, 245, 257, 253, 220, 264, 238, 261, 232, 243, 255, 227, 266, 254, 237, 221, 268, 226, 265, 233, 262, 259, 247, 251, 265, 223,
   239, 244, 256, 262, 235, 268, 229, 252, 238, 269, 222, 267, 228, 260, 248, 234, 263, 249, 267, 239, 224, 258, 246, 269, 230, 263,
   236, 250, 231, 226, 274, 233, 270, 237, 265, 257, 251, 259, 232, 273, 228, 271, 234, 255, 249, 238, 267, 260, 225, 233, 275, 237,
   264, 231, 270, 253, 259, 257, 273, 235, 230, 256, 250, 272, 236, 269, 239, 258, 227, 275, 234, 266, 238, 254, 260, 232, 271, 255,
   274, 229, 236, 252, 258, 268, 239, 272, 235, 256, 242, 257, 255, 253, 254, 245, 266, 243, 264, 261, 259, 241, 256, 247, 268, 251,
   252, 265, 244, 262, 260, 258, 240, 269, 248, 267, 246, 249, 250, 263, 270, 244, 258, 262, 272, 265, 256, 274, 252, 268, 271, 256,
   246, 272, 263, 273, 250, 267, 258, 269, 243, 270, 260, 264, 255, 261, 271, 254, 275, 266, 255, 272, 248, 273, 249, 271, 263, 260,
   269, 267, 245, 259, 271, 257, 266, 253, 275, 261, 270, 264, 257, 247, 272, 251, 274, 259, 268, 270, 262, 265, 253, 251, 273, 259,
   275, 257, 274, 264, 265, 270, 254, 274, 249, 275, 260, 266, 267, 255, 273, 271, 275, 252, 250, 268, 269, 274, 258, 273, 256, 272,
   261, 265, 267, 270, 271, 264, 273, 266, 274, 275, 264, 262, 269, 265, 273, 270, 272, 275, 268, 274, 266, 268, 263, 274, 267, 275,
   269, 271, 272, 273
}; // 2.34 kb
static const cart_index_t iCartXY_ab10_a4[1260] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195, 200, 221, 222, 203, 215, 205, 217, 243, 244, 209, 210, 251, 233, 235, 232, 223, 201, 225,
   216, 204, 241, 242, 207, 219, 211, 252, 213, 234, 231, 237, 224, 226, 202, 239, 240, 218, 206, 220, 208, 253, 212, 214, 230, 236,
   238, 203, 227, 245, 209, 221, 232, 244, 247, 256, 215, 233, 254, 251, 243, 235, 227, 204, 246, 223, 211, 249, 258, 231, 242, 216,
   255, 234, 252, 237, 241, 205, 247, 228, 232, 243, 210, 222, 254, 245, 235, 217, 256, 244, 251, 233, 228, 248, 206, 250, 259, 224,
   212, 240, 230, 257, 218, 236, 238, 253, 239, 249, 207, 229, 241, 231, 255, 246, 213, 225, 237, 258, 219, 242, 234, 252, 250, 229,
   208, 257, 248, 239, 230, 226, 214, 259, 238, 220, 236, 240, 253, 209, 223, 261, 215, 227, 235, 256, 249, 265, 221, 251, 260, 254,
   247, 243, 210, 260, 224, 233, 254, 217, 228, 265, 250, 251, 222, 261, 245, 256, 244, 221, 211, 262, 227, 216, 247, 264, 237, 258,
   223, 260, 252, 255, 241, 249, 222, 262, 212, 245, 263, 228, 218, 259, 238, 261, 224, 253, 239, 257, 250, 260, 213, 226, 255, 234,
   264, 248, 219, 229, 252, 262, 225, 246, 242, 258, 261, 225, 214, 263, 246, 257, 236, 229, 220, 262, 253, 226, 240, 248, 259, 215,
   216, 263, 221, 223, 243, 265, 241, 264, 227, 254, 255, 260, 249, 247, 217, 264, 218, 244, 265, 222, 224, 263, 239, 256, 228, 257,
   250, 261, 245, 265, 219, 220, 264, 242, 263, 240, 225, 226, 258, 259, 229, 248, 246, 262, 245, 246, 230, 261, 262, 250, 238, 248,
   236, 263, 239, 240, 253, 259, 257, 247, 231, 248, 249, 237, 260, 262, 234, 246, 241, 264, 242, 258, 252, 255, 232, 249, 250, 235,
   247, 233, 245, 260, 261, 243, 244, 265, 256, 254, 251, 233, 255, 239, 251, 260, 244, 250, 264, 257, 254, 245, 263, 261, 265, 256,
   254, 234, 240, 260, 252, 265, 259, 242, 248, 255, 263, 246, 262, 258, 264, 235, 241, 257, 243, 249, 251, 261, 255, 263, 247, 256,
   264, 265, 260, 254, 256, 242, 236, 265, 258, 261, 253, 246, 240, 264, 257, 248, 259, 262, 263, 243, 237, 259, 247, 241, 254, 263,
   252, 262, 249, 265, 258, 264, 255, 260, 244, 258, 238, 256, 264, 245, 239, 262, 253, 265, 250, 259, 257, 263, 261, 251, 252, 253,
   254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265
}; // 2.46 kb
static const cart_index_t iCartXY_ab10_a5[1176] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180, 185, 212, 213, 206, 207, 188, 230, 190, 232, 194, 195, 200, 239, 202, 241, 217, 228, 229, 236,
   218, 220, 212, 186, 214, 189, 231, 208, 210, 234, 192, 201, 240, 196, 198, 243, 204, 226, 216, 227, 219, 237, 222, 213, 214, 187,
   233, 191, 235, 193, 209, 211, 242, 203, 244, 205, 197, 199, 224, 225, 215, 221, 223, 238, 206, 189, 233, 196, 247, 212, 231, 232,
   216, 208, 245, 201, 219, 249, 227, 234, 222, 243, 237, 240, 226, 207, 231, 191, 247, 197, 230, 215, 213, 233, 246, 209, 248, 225,
   203, 221, 235, 244, 223, 238, 224, 242, 188, 208, 235, 212, 230, 194, 246, 217, 234, 200, 218, 206, 245, 229, 250, 220, 232, 241,
   239, 236, 228, 230, 210, 193, 231, 215, 246, 199, 235, 214, 248, 224, 247, 211, 223, 205, 242, 233, 221, 225, 238, 244, 190, 234,
   209, 232, 213, 217, 235, 195, 245, 220, 202, 228, 250, 207, 246, 218, 239, 230, 241, 229, 236, 232, 192, 211, 216, 233, 234, 214,
   245, 198, 226, 249, 222, 204, 247, 210, 240, 219, 231, 227, 243, 237, 194, 201, 242, 208, 246, 200, 248, 220, 226, 206, 236, 212,
   240, 241, 249, 228, 234, 250, 245, 239, 232, 195, 240, 203, 245, 209, 218, 224, 202, 249, 236, 207, 239, 248, 213, 242, 229, 250,
   235, 246, 230, 241, 200, 196, 244, 201, 248, 206, 247, 228, 222, 212, 239, 208, 237, 250, 243, 232, 226, 249, 240, 245, 234, 239,
   198, 205, 219, 225, 245, 211, 250, 204, 240, 248, 237, 210, 244, 214, 249, 227, 233, 231, 247, 243, 202, 243, 197, 249, 203, 229,
   223, 207, 247, 241, 213, 250, 244, 209, 238, 230, 248, 224, 242, 235, 246, 241, 204, 199, 227, 221, 250, 205, 246, 210, 249, 242,
   243, 214, 238, 211, 248, 231, 225, 233, 244, 247, 217, 226, 224, 234, 235, 220, 242, 218, 240, 228, 229, 232, 249, 230, 248, 236,
   245, 246, 250, 241, 239, 228, 216, 225, 222, 244, 232, 233, 239, 219, 234, 250, 226, 227, 248, 231, 245, 237, 247, 243, 249, 240,
   229, 227, 215, 243, 223, 241, 221, 230, 231, 250, 235, 249, 233, 224, 225, 246, 247, 238, 244, 242, 248, 236, 219, 221, 237, 238,
   239, 225, 241, 227, 245, 246, 240, 231, 242, 233, 250, 243, 244, 247, 248, 249, 218, 237, 223, 240, 224, 236, 238, 229, 243, 239,
   230, 245, 247, 235, 244, 241, 249, 242, 248, 246, 250, 220, 222, 238, 226, 242, 228, 244, 236, 237, 232, 241, 234, 243, 246, 247,
   239, 240, 248, 249, 250, 245
}; // 2.30 kb
static const cart_index_t iCartXY_ab10_a6[980] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163, 164, 187, 188, 167, 191, 169, 192, 213, 214, 173, 174, 185,
   186, 224, 225, 179, 181, 229, 209, 211, 196, 197, 218, 199, 220, 207, 208, 215, 185, 165, 190, 191, 168, 211, 212, 171, 193, 187,
   224, 175, 226, 177, 189, 180, 228, 183, 210, 195, 213, 219, 198, 205, 206, 201, 222, 216, 186, 189, 166, 209, 210, 192, 170, 193,
   172, 225, 188, 226, 176, 190, 178, 227, 182, 184, 194, 212, 214, 203, 204, 221, 200, 223, 202, 217, 167, 180, 203, 173, 187, 196,
   214, 205, 221, 179, 197, 191, 209, 219, 227, 185, 208, 228, 225, 213, 199, 215, 224, 207, 229, 211, 220, 218, 179, 168, 204, 185,
   175, 207, 223, 195, 212, 191, 218, 180, 227, 198, 210, 187, 229, 206, 226, 201, 211, 224, 216, 213, 222, 205, 228, 219, 169, 205,
   182, 196, 213, 174, 188, 219, 203, 199, 181, 211, 192, 228, 221, 207, 186, 227, 214, 224, 197, 208, 229, 215, 225, 218, 209, 220,
   181, 206, 170, 208, 222, 186, 176, 210, 194, 220, 192, 228, 182, 212, 200, 229, 188, 204, 202, 226, 209, 214, 223, 225, 217, 227,
   203, 221, 207, 171, 184, 211, 195, 218, 204, 177, 190, 213, 229, 201, 223, 183, 193, 205, 227, 189, 212, 198, 224, 228, 206, 219,
   210, 216, 226, 222, 208, 183, 172, 220, 206, 209, 194, 189, 178, 229, 214, 222, 202, 193, 184, 228, 203, 190, 200, 210, 225, 221,
   212, 227, 204, 226, 217, 223, 173, 175, 217, 179, 180, 199, 221, 201, 223, 185, 215, 187, 225, 216, 226, 191, 220, 222, 227, 205,
   207, 218, 219, 211, 228, 213, 229, 224, 174, 216, 176, 197, 219, 181, 182, 222, 202, 215, 186, 224, 188, 226, 217, 218, 192, 223,
   203, 228, 208, 209, 227, 220, 221, 229, 214, 225, 215, 177, 178, 218, 198, 220, 200, 183, 184, 224, 225, 216, 217, 189, 190, 219,
   221, 193, 204, 206, 229, 227, 210, 228, 212, 222, 223, 226, 197, 198, 194, 215, 216, 208, 202, 206, 200, 218, 209, 219, 203, 210,
   204, 224, 214, 212, 217, 222, 220, 225, 226, 229, 223, 228, 221, 227, 199, 195, 200, 207, 201, 215, 217, 198, 204, 211, 220, 205,
   221, 206, 212, 213, 225, 210, 223, 216, 218, 229, 222, 224, 226, 219, 227, 228, 196, 201, 202, 199, 205, 197, 203, 216, 217, 207,
   208, 213, 214, 222, 223, 211, 209, 226, 221, 219, 215, 220, 228, 218, 227, 224, 225, 229
}; // 1.91 kb
static const cart_index_t iCartXY_ab11_a0[364] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233,
   234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259,
   260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285,
   286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296, 297, 298, 299, 300, 301, 302, 303, 304, 305, 306, 307, 308, 309, 310, 311,
   312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 330, 331, 332, 333, 334, 335, 336, 337,
   338, 339, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 358, 359, 360, 361, 362, 363
}; // 0.71 kb
static const cart_index_t iCartXY_ab11_a1[858] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215, 219, 222, 224, 223, 220, 226, 225, 227, 221, 230, 223, 250, 231, 249, 225, 222, 228, 251, 249, 233, 227, 224, 251, 229, 250,
   226, 232, 228, 234, 254, 229, 252, 236, 235, 230, 256, 253, 232, 238, 237, 257, 231, 255, 239, 233, 242, 235, 260, 243, 258, 237,
   234, 240, 262, 259, 245, 239, 236, 263, 241, 261, 238, 244, 240, 246, 266, 241, 264, 247, 246, 242, 268, 265, 244, 248, 247, 269,
   243, 267, 248, 245, 251, 254, 252, 256, 250, 253, 257, 255, 249, 271, 253, 261, 272, 259, 255, 252, 270, 263, 258, 272, 257, 254,
   262, 270, 260, 256, 271, 262, 266, 273, 263, 275, 264, 268, 260, 274, 269, 276, 258, 277, 261, 265, 278, 267, 259, 281, 265, 267,
   264, 280, 269, 266, 268, 279, 270, 273, 275, 274, 271, 277, 276, 278, 272, 279, 274, 283, 280, 282, 276, 273, 279, 284, 282, 281,
   278, 275, 284, 280, 283, 277, 281, 284, 283, 282, 285, 290, 292, 288, 286, 293, 289, 291, 287, 290, 294, 318, 296, 288, 319, 292,
   318, 295, 298, 320, 289, 319, 293, 297, 320, 299, 291, 294, 302, 325, 295, 323, 304, 300, 296, 326, 301, 324, 298, 321, 297, 305,
   322, 303, 299, 302, 306, 327, 308, 300, 329, 304, 328, 307, 310, 330, 301, 331, 305, 309, 332, 311, 303, 306, 314, 337, 307, 335,
   316, 312, 308, 338, 313, 336, 310, 333, 309, 317, 334, 315, 311, 314, 312, 339, 316, 340, 313, 341, 317, 315, 324, 322, 320, 326,
   319, 321, 318, 325, 323, 323, 342, 328, 343, 321, 331, 325, 327, 342, 344, 332, 322, 329, 326, 343, 330, 344, 324, 336, 346, 330,
   348, 334, 332, 338, 329, 345, 350, 331, 333, 327, 337, 347, 328, 349, 335, 335, 352, 340, 354, 333, 341, 337, 339, 351, 356, 341,
   334, 339, 338, 353, 340, 355, 336, 342, 347, 349, 345, 343, 350, 346, 348, 344, 347, 351, 357, 353, 345, 358, 349, 357, 352, 355,
   359, 346, 358, 350, 354, 359, 356, 348, 351, 353, 362, 352, 361, 355, 360, 354, 356, 361, 360, 359, 362, 358, 360, 357, 362, 361
}; // 1.68 kb
static const cart_index_t iCartXY_ab11_a2[1320] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212, 216, 225, 226, 219, 221, 248, 227, 217,
   229, 220, 247, 223, 228, 230, 218, 246, 222, 224, 219, 231, 249, 225, 248, 251, 232, 220, 250, 227, 253, 247, 221, 251, 233, 248,
   226, 249, 234, 252, 222, 254, 228, 246, 253, 223, 235, 247, 250, 229, 254, 236, 224, 252, 246, 230, 225, 237, 267, 231, 251, 259,
   226, 267, 238, 249, 233, 260, 239, 227, 268, 232, 257, 253, 240, 269, 228, 255, 234, 254, 268, 229, 241, 250, 258, 235, 269, 242,
   230, 256, 252, 236, 231, 243, 270, 237, 259, 263, 243, 232, 271, 239, 265, 257, 233, 272, 244, 260, 238, 261, 244, 273, 234, 266,
   240, 255, 274, 235, 245, 258, 262, 241, 275, 245, 236, 264, 256, 242, 237, 239, 276, 243, 263, 265, 238, 277, 240, 261, 244, 266,
   278, 241, 242, 262, 264, 245, 255, 256, 246, 269, 254, 252, 257, 247, 258, 253, 268, 250, 248, 259, 260, 251, 249, 267, 249, 270,
   261, 267, 260, 272, 271, 250, 262, 268, 274, 258, 251, 263, 272, 259, 267, 270, 273, 264, 252, 275, 269, 256, 265, 253, 274, 257,
   271, 268, 266, 275, 254, 273, 255, 269, 261, 279, 255, 277, 266, 273, 279, 262, 256, 278, 275, 264, 263, 257, 280, 265, 276, 271,
   280, 258, 264, 274, 278, 262, 259, 265, 281, 263, 270, 276, 260, 281, 266, 272, 261, 277, 267, 276, 277, 270, 272, 281, 276, 268,
   278, 271, 280, 274, 277, 278, 269, 279, 273, 275, 270, 271, 279, 276, 281, 280, 272, 280, 273, 281, 277, 279, 281, 274, 275, 280,
   279, 278, 282, 291, 292, 287, 289, 315, 293, 283, 294, 285, 316, 290, 295, 296, 284, 317, 286, 288, 297, 285, 318, 293, 323, 316,
   298, 319, 286, 321, 295, 317, 287, 299, 320, 291, 315, 322, 321, 300, 288, 319, 317, 296, 289, 322, 301, 315, 292, 320, 323, 290,
   302, 316, 318, 294, 291, 303, 339, 299, 322, 324, 292, 339, 304, 320, 301, 325, 305, 293, 340, 297, 326, 323, 340, 294, 306, 318,
   328, 302, 307, 341, 295, 327, 298, 321, 341, 308, 296, 329, 319, 300, 309, 297, 342, 305, 335, 326, 310, 343, 298, 333, 307, 327,
   299, 311, 344, 303, 324, 334, 345, 312, 300, 331, 329, 308, 301, 346, 313, 325, 304, 332, 347, 302, 314, 328, 330, 306, 303, 309,
   348, 311, 334, 336, 304, 349, 310, 332, 313, 337, 311, 305, 350, 309, 336, 335, 351, 306, 312, 330, 338, 314, 313, 352, 307, 337,
   310, 333, 353, 314, 308, 338, 331, 312, 315, 324, 325, 322, 320, 339, 326, 316, 328, 323, 340, 318, 327, 329, 317, 341, 321, 319,
   342, 318, 330, 340, 347, 328, 343, 331, 319, 345, 341, 329, 320, 344, 332, 339, 325, 346, 333, 345, 321, 343, 327, 341, 322, 334,
   346, 324, 339, 344, 335, 323, 347, 326, 342, 340, 324, 336, 354, 334, 344, 348, 325, 354, 337, 346, 332, 349, 336, 326, 355, 335,
   350, 342, 337, 356, 327, 352, 333, 343, 355, 328, 338, 347, 351, 330, 356, 338, 329, 353, 345, 331, 357, 330, 331, 351, 353, 338,
   332, 358, 333, 349, 337, 352, 334, 335, 359, 336, 348, 350, 339, 348, 349, 344, 346, 354, 350, 340, 351, 342, 355, 347, 352, 353,
   341, 356, 343, 345, 348, 342, 357, 350, 359, 355, 349, 357, 343, 358, 352, 356, 344, 350, 358, 348, 354, 359, 358, 351, 345, 357,
   356, 353, 346, 359, 352, 354, 349, 358, 359, 347, 353, 355, 357, 351, 354, 355, 356, 359, 358, 357
}; // 2.58 kb
static const cart_index_t iCartXY_ab11_a3[1650] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209, 210, 225, 227, 219, 220, 213, 243, 215, 245, 242, 226, 211, 229, 214, 244, 221, 223, 247,
   217, 241, 228, 230, 212, 246, 216, 248, 218, 222, 224, 240, 233, 214, 252, 221, 262, 226, 244, 251, 241, 247, 234, 250, 216, 263,
   222, 249, 240, 228, 246, 248, 213, 231, 254, 225, 243, 219, 261, 242, 253, 245, 249, 236, 218, 250, 240, 263, 224, 248, 230, 246,
   215, 253, 232, 245, 227, 242, 254, 220, 261, 243, 251, 217, 235, 241, 252, 247, 229, 262, 223, 244, 219, 237, 266, 231, 261, 225,
   264, 245, 257, 253, 220, 264, 238, 261, 232, 243, 255, 227, 266, 254, 237, 221, 268, 226, 265, 233, 262, 259, 247, 251, 265, 223,
   239, 244, 256, 262, 235, 268, 229, 252, 238, 269, 222, 267, 228, 260, 248, 234, 263, 249, 267, 239, 224, 258, 246, 269, 230, 263,
   236, 250, 231, 226, 274, 233, 270, 237, 265, 257, 251, 259, 232, 273, 228, 271, 234, 255, 249, 238, 267, 260, 225, 233, 275, 237,
   264, 231, 270, 253, 259, 257, 273, 235, 230, 256, 250, 272, 236, 269, 239, 258, 227, 275, 234, 266, 238, 254, 260, 232, 271, 255,
   274, 229, 236, 252, 258, 268, 239, 272, 235, 256, 242, 257, 255, 253, 254, 245, 266, 243, 264, 261, 259, 241, 256, 247, 268, 251,
   252, 265, 244, 262, 260, 258, 240, 269, 248, 267, 246, 249, 250, 263, 270, 244, 258, 262, 272, 265, 256, 274, 252, 268, 271, 256,
   246, 272, 263, 273, 250, 267, 258, 269, 243, 270, 260, 264, 255, 261, 271, 254, 275, 266, 255, 272, 248, 273, 249, 271, 263, 260,
   269, 267, 245, 259, 271, 257, 266, 253, 275, 261, 270, 264, 257, 247, 272, 251, 274, 259, 268, 270, 262, 265, 253, 251, 273, 259,
   275, 257, 274, 264, 265, 270, 254, 274, 249, 275, 260, 266, 267, 255, 273, 271, 275, 252, 250, 268, 269, 274, 258, 273, 256, 272,
   261, 265, 267, 270, 271, 264, 273, 266, 274, 275, 264, 262, 269, 265, 273, 270, 272, 275, 268, 274, 266, 268, 263, 274, 267, 275,
   269, 271, 272, 273, 276, 293, 295, 285, 286, 281, 314, 283, 316, 309, 291, 277, 296, 279, 312, 287, 288, 317, 284, 310, 292, 294,
   278, 313, 280, 315, 282, 289, 290, 311, 281, 297, 319, 293, 314, 285, 333, 309, 318, 316, 299, 279, 322, 287, 334, 291, 312, 320,
   310, 317, 283, 318, 298, 316, 295, 309, 319, 286, 333, 314, 301, 323, 280, 335, 289, 321, 311, 292, 313, 315, 320, 284, 300, 310,
   322, 317, 296, 334, 288, 312, 321, 302, 282, 323, 311, 335, 290, 315, 294, 313, 285, 305, 340, 297, 333, 293, 338, 316, 328, 318,
   286, 338, 307, 333, 298, 314, 326, 295, 340, 319, 303, 287, 341, 291, 336, 299, 334, 329, 317, 320, 304, 339, 289, 337, 292, 327,
   315, 301, 335, 321, 336, 288, 308, 312, 324, 334, 300, 341, 296, 322, 337, 306, 290, 325, 313, 339, 294, 335, 302, 323, 293, 303,
   348, 305, 338, 297, 342, 318, 330, 328, 305, 291, 349, 299, 344, 303, 336, 330, 320, 329, 295, 348, 304, 340, 307, 319, 331, 298,
   343, 326, 307, 350, 292, 346, 301, 331, 321, 304, 337, 327, 349, 296, 306, 322, 332, 341, 308, 345, 300, 324, 350, 308, 294, 332,
   323, 347, 302, 339, 306, 325, 297, 299, 353, 303, 342, 305, 344, 328, 329, 330, 298, 352, 301, 343, 304, 326, 327, 307, 346, 331,
   351, 300, 302, 324, 325, 345, 306, 347, 308, 332, 327, 325, 311, 339, 315, 337, 313, 321, 323, 335, 329, 310, 324, 317, 341, 320,
   322, 336, 312, 334, 309, 328, 326, 318, 319, 316, 340, 314, 338, 333, 314, 342, 331, 338, 326, 333, 343, 319, 348, 340, 344, 312,
   332, 334, 345, 336, 324, 349, 322, 341, 316, 330, 343, 328, 340, 318, 348, 333, 342, 338, 346, 332, 313, 347, 335, 350, 323, 337,
   325, 339, 330, 317, 345, 320, 349, 329, 341, 344, 334, 336, 331, 347, 315, 350, 321, 346, 335, 327, 339, 337, 326, 351, 321, 352,
   327, 343, 337, 331, 350, 346, 352, 324, 323, 345, 339, 351, 325, 350, 332, 347, 328, 320, 351, 329, 353, 330, 349, 342, 336, 344,
   353, 322, 325, 341, 347, 349, 332, 351, 324, 345, 318, 329, 352, 330, 348, 328, 353, 338, 344, 342, 319, 353, 327, 348, 331, 340,
   346, 326, 352, 343, 333, 344, 346, 342, 343, 338, 352, 340, 353, 348, 342, 334, 347, 336, 351, 344, 345, 353, 341, 349, 343, 345,
   335, 351, 337, 352, 339, 346, 347, 350, 338, 336, 350, 344, 352, 342, 351, 348, 349, 353, 340, 349, 337, 353, 346, 348, 350, 343,
   351, 352, 348, 341, 339, 349, 350, 353, 347, 352, 345, 351
}; // 3.22 kb
static const cart_index_t iCartXY_ab11_a4[1800] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195, 200, 221, 222, 203, 215, 205, 217, 243, 244, 209, 210, 251, 233, 235, 232, 223, 201, 225,
   216, 204, 241, 242, 207, 219, 211, 252, 213, 234, 231, 237, 224, 226, 202, 239, 240, 218, 206, 220, 208, 253, 212, 214, 230, 236,
   238, 203, 227, 245, 209, 221, 232, 244, 247, 256, 215, 233, 254, 251, 243, 235, 227, 204, 246, 223, 211, 249, 258, 231, 242, 216,
   255, 234, 252, 237, 241, 205, 247, 228, 232, 243, 210, 222, 254, 245, 235, 217, 256, 244, 251, 233, 228, 248, 206, 250, 259, 224,
   212, 240, 230, 257, 218, 236, 238, 253, 239, 249, 207, 229, 241, 231, 255, 246, 213, 225, 237, 258, 219, 242, 234, 252, 250, 229,
   208, 257, 248, 239, 230, 226, 214, 259, 238, 220, 236, 240, 253, 209, 223, 261, 215, 227, 235, 256, 249, 265, 221, 251, 260, 254,
   247, 243, 210, 260, 224, 233, 254, 217, 228, 265, 250, 251, 222, 261, 245, 256, 244, 221, 211, 262, 227, 216, 247, 264, 237, 258,
   223, 260, 252, 255, 241, 249, 222, 262, 212, 245, 263, 228, 218, 259, 238, 261, 224, 253, 239, 257, 250, 260, 213, 226, 255, 234,
   264, 248, 219, 229, 252, 262, 225, 246, 242, 258, 261, 225, 214, 263, 246, 257, 236, 229, 220, 262, 253, 226, 240, 248, 259, 215,
   216, 263, 221, 223, 243, 265, 241, 264, 227, 254, 255, 260, 249, 247, 217, 264, 218, 244, 265, 222, 224, 263, 239, 256, 228, 257,
   250, 261, 245, 265, 219, 220, 264, 242, 263, 240, 225, 226, 258, 259, 229, 248, 246, 262, 245, 246, 230, 261, 262, 250, 238, 248,
   236, 263, 239, 240, 253, 259, 257, 247, 231, 248, 249, 237, 260, 262, 234, 246, 241, 264, 242, 258, 252, 255, 232, 249, 250, 235,
   247, 233, 245, 260, 261, 243, 244, 265, 256, 254, 251, 233, 255, 239, 251, 260, 244, 250, 264, 257, 254, 245, 263, 261, 265, 256,
   254, 234, 240, 260, 252, 265, 259, 242, 248, 255, 263, 246, 262, 258, 264, 235, 241, 257, 243, 249, 251, 261, 255, 263, 247, 256,
   264, 265, 260, 254, 256, 242, 236, 265, 258, 261, 253, 246, 240, 264, 257, 248, 259, 262, 263, 243, 237, 259, 247, 241, 254, 263,
   252, 262, 249, 265, 258, 264, 255, 260, 244, 258, 238, 256, 264, 245, 239, 262, 253, 265, 250, 259, 257, 263, 261, 251, 252, 253,
   254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 287, 288, 271, 283, 273, 285, 308, 309, 275, 276, 323, 304, 306,
   299, 289, 267, 290, 281, 269, 310, 312, 274, 286, 277, 324, 278, 302, 300, 307, 291, 292, 268, 311, 313, 282, 270, 284, 272, 325,
   279, 280, 301, 303, 305, 293, 269, 314, 289, 277, 319, 331, 300, 312, 281, 326, 302, 324, 307, 310, 294, 315, 270, 317, 329, 291,
   279, 313, 301, 327, 282, 303, 305, 325, 311, 271, 295, 316, 275, 287, 299, 309, 318, 330, 283, 304, 328, 323, 308, 306, 317, 296,
   272, 327, 315, 311, 301, 292, 280, 329, 305, 284, 303, 313, 325, 273, 318, 297, 299, 308, 276, 288, 328, 316, 306, 285, 330, 309,
   323, 304, 319, 274, 298, 310, 300, 326, 314, 278, 290, 307, 331, 286, 312, 302, 324, 275, 293, 333, 283, 295, 306, 330, 320, 338,
   287, 323, 332, 328, 318, 308, 276, 332, 294, 304, 328, 285, 297, 338, 321, 323, 288, 333, 316, 330, 309, 295, 277, 335, 293, 281,
   320, 339, 307, 331, 289, 334, 324, 326, 310, 319, 334, 278, 296, 326, 302, 339, 322, 286, 298, 324, 335, 290, 314, 312, 331, 297,
   337, 279, 321, 340, 294, 282, 329, 305, 336, 291, 325, 311, 327, 317, 336, 298, 280, 340, 322, 327, 303, 296, 284, 337, 325, 292,
   313, 315, 329, 287, 281, 341, 295, 289, 318, 343, 310, 339, 293, 332, 326, 334, 319, 320, 288, 341, 282, 316, 342, 297, 291, 340,
   311, 333, 294, 327, 317, 336, 321, 283, 289, 342, 287, 293, 308, 338, 319, 343, 295, 328, 334, 332, 320, 318, 342, 290, 284, 341,
   314, 340, 313, 298, 292, 335, 329, 296, 315, 322, 337, 285, 343, 291, 309, 338, 288, 294, 342, 317, 330, 297, 336, 321, 333, 316,
   343, 286, 292, 339, 312, 341, 315, 290, 296, 331, 337, 298, 322, 314, 335, 299, 320, 321, 306, 318, 304, 316, 332, 333, 308, 309,
   338, 330, 328, 323, 320, 300, 322, 319, 307, 334, 335, 302, 314, 310, 339, 312, 331, 324, 326, 321, 322, 301, 336, 337, 317, 305,
   315, 303, 340, 311, 313, 325, 329, 327, 332, 302, 315, 334, 324, 343, 337, 312, 322, 326, 341, 314, 335, 331, 339, 333, 314, 303,
   342, 335, 336, 325, 322, 313, 341, 327, 315, 329, 337, 340, 304, 334, 317, 323, 332, 309, 321, 343, 336, 328, 316, 342, 333, 338,
   330, 316, 335, 305, 333, 341, 321, 311, 337, 325, 342, 317, 329, 327, 340, 336, 306, 319, 336, 308, 320, 323, 333, 334, 342, 318,
   330, 343, 338, 332, 328, 318, 307, 337, 320, 310, 332, 341, 324, 335, 319, 343, 331, 339, 326, 334, 308, 310, 340, 318, 319, 328,
   342, 326, 341, 320, 338, 339, 343, 334, 332, 309, 339, 311, 330, 343, 316, 317, 341, 327, 338, 321, 340, 336, 342, 333, 338, 312,
   313, 343, 331, 342, 329, 314, 315, 339, 340, 322, 337, 335, 341, 323, 326, 327, 328, 334, 330, 336, 339, 340, 332, 333, 341, 342,
   343, 338, 328, 324, 329, 332, 326, 338, 340, 331, 337, 334, 342, 335, 341, 339, 343, 330, 331, 325, 338, 339, 333, 327, 335, 329,
   343, 336, 337, 340, 341, 342
}; // 3.52 kb
static const cart_index_t iCartXY_ab11_a5[1764] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180, 185, 212, 213, 206, 207, 188, 230, 190, 232, 194, 195, 200, 239, 202, 241, 217, 228, 229, 236,
   218, 220, 212, 186, 214, 189, 231, 208, 210, 234, 192, 201, 240, 196, 198, 243, 204, 226, 216, 227, 219, 237, 222, 213, 214, 187,
   233, 191, 235, 193, 209, 211, 242, 203, 244, 205, 197, 199, 224, 225, 215, 221, 223, 238, 206, 189, 233, 196, 247, 212, 231, 232,
   216, 208, 245, 201, 219, 249, 227, 234, 222, 243, 237, 240, 226, 207, 231, 191, 247, 197, 230, 215, 213, 233, 246, 209, 248, 225,
   203, 221, 235, 244, 223, 238, 224, 242, 188, 208, 235, 212, 230, 194, 246, 217, 234, 200, 218, 206, 245, 229, 250, 220, 232, 241,
   239, 236, 228, 230, 210, 193, 231, 215, 246, 199, 235, 214, 248, 224, 247, 211, 223, 205, 242, 233, 221, 225, 238, 244, 190, 234,
   209, 232, 213, 217, 235, 195, 245, 220, 202, 228, 250, 207, 246, 218, 239, 230, 241, 229, 236, 232, 192, 211, 216, 233, 234, 214,
   245, 198, 226, 249, 222, 204, 247, 210, 240, 219, 231, 227, 243, 237, 194, 201, 242, 208, 246, 200, 248, 220, 226, 206, 236, 212,
   240, 241, 249, 228, 234, 250, 245, 239, 232, 195, 240, 203, 245, 209, 218, 224, 202, 249, 236, 207, 239, 248, 213, 242, 229, 250,
   235, 246, 230, 241, 200, 196, 244, 201, 248, 206, 247, 228, 222, 212, 239, 208, 237, 250, 243, 232, 226, 249, 240, 245, 234, 239,
   198, 205, 219, 225, 245, 211, 250, 204, 240, 248, 237, 210, 244, 214, 249, 227, 233, 231, 247, 243, 202, 243, 197, 249, 203, 229,
   223, 207, 247, 241, 213, 250, 244, 209, 238, 230, 248, 224, 242, 235, 246, 241, 204, 199, 227, 221, 250, 205, 246, 210, 249, 242,
   243, 214, 238, 211, 248, 231, 225, 233, 244, 247, 217, 226, 224, 234, 235, 220, 242, 218, 240, 228, 229, 232, 249, 230, 248, 236,
   245, 246, 250, 241, 239, 228, 216, 225, 222, 244, 232, 233, 239, 219, 234, 250, 226, 227, 248, 231, 245, 237, 247, 243, 249, 240,
   229, 227, 215, 243, 223, 241, 221, 230, 231, 250, 235, 249, 233, 224, 225, 246, 247, 238, 244, 242, 248, 236, 219, 221, 237, 238,
   239, 225, 241, 227, 245, 246, 240, 231, 242, 233, 250, 243, 244, 247, 248, 249, 218, 237, 223, 240, 224, 236, 238, 229, 243, 239,
   230, 245, 247, 235, 244, 241, 249, 242, 248, 246, 250, 220, 222, 238, 226, 242, 228, 244, 236, 237, 232, 241, 234, 243, 246, 247,
   239, 240, 248, 249, 250, 245, 251, 280, 282, 272, 273, 256, 301, 258, 303, 260, 261, 268, 313, 270, 315, 284, 293, 294, 308, 289,
   291, 278, 252, 283, 254, 299, 274, 275, 304, 259, 266, 311, 262, 263, 316, 271, 295, 285, 297, 287, 309, 292, 279, 281, 253, 300,
   255, 302, 257, 276, 277, 312, 267, 314, 269, 264, 265, 296, 298, 286, 288, 290, 310, 256, 278, 306, 280, 301, 260, 318, 284, 305,
   268, 289, 272, 317, 294, 323, 291, 303, 315, 313, 308, 293, 280, 254, 307, 262, 320, 278, 299, 305, 285, 274, 319, 266, 287, 324,
   297, 304, 292, 316, 309, 311, 295, 258, 305, 279, 303, 282, 284, 306, 261, 317, 291, 270, 293, 323, 273, 318, 289, 313, 301, 315,
   294, 308, 282, 307, 255, 322, 264, 306, 286, 279, 300, 321, 276, 325, 298, 267, 288, 302, 314, 290, 310, 296, 312, 305, 259, 281,
   285, 307, 304, 283, 319, 263, 295, 324, 292, 271, 320, 275, 311, 287, 299, 297, 316, 309, 306, 283, 257, 307, 286, 321, 265, 302,
   281, 325, 296, 322, 277, 290, 269, 312, 300, 288, 298, 310, 314, 260, 274, 321, 278, 318, 268, 327, 291, 304, 272, 308, 280, 319,
   315, 328, 293, 305, 323, 317, 313, 303, 261, 319, 276, 317, 279, 289, 302, 270, 328, 308, 273, 313, 327, 282, 321, 294, 323, 306,
   318, 301, 315, 272, 262, 322, 266, 326, 280, 320, 303, 292, 278, 317, 274, 309, 328, 316, 305, 295, 324, 311, 319, 304, 273, 320,
   264, 326, 267, 301, 290, 282, 322, 318, 279, 327, 314, 276, 310, 306, 325, 296, 312, 302, 321, 317, 263, 277, 287, 300, 319, 281,
   328, 271, 311, 326, 309, 275, 322, 283, 324, 297, 307, 299, 320, 316, 318, 275, 265, 299, 288, 327, 269, 321, 283, 326, 312, 320,
   281, 310, 277, 325, 307, 298, 300, 314, 322, 268, 266, 325, 274, 327, 272, 326, 293, 295, 280, 313, 278, 311, 323, 324, 303, 304,
   328, 319, 317, 305, 270, 324, 267, 328, 276, 294, 296, 273, 326, 315, 282, 323, 325, 279, 312, 301, 327, 302, 321, 306, 318, 323,
   271, 269, 297, 298, 328, 277, 327, 275, 324, 325, 316, 283, 314, 281, 326, 299, 300, 307, 322, 320, 301, 299, 286, 320, 290, 318,
   288, 306, 307, 327, 302, 326, 300, 296, 298, 321, 322, 310, 314, 312, 325, 303, 285, 300, 292, 322, 305, 307, 317, 287, 304, 328,
   295, 297, 326, 299, 319, 309, 320, 316, 324, 311, 284, 304, 302, 305, 306, 291, 321, 289, 319, 293, 294, 303, 328, 301, 327, 308,
   317, 318, 323, 315, 313, 289, 311, 296, 319, 302, 308, 312, 294, 324, 313, 301, 317, 326, 306, 325, 315, 328, 321, 327, 318, 323,
   313, 287, 298, 309, 314, 317, 300, 323, 297, 319, 327, 311, 299, 325, 307, 328, 316, 322, 320, 326, 324, 291, 295, 312, 304, 321,
   293, 325, 308, 311, 303, 315, 305, 324, 318, 326, 313, 319, 327, 328, 323, 317, 315, 297, 288, 316, 310, 323, 298, 318, 299, 328,
   321, 324, 307, 312, 300, 327, 320, 314, 322, 325, 326, 293, 292, 314, 295, 325, 303, 322, 313, 309, 305, 323, 304, 316, 327, 320,
   317, 311, 326, 324, 328, 319, 294, 316, 290, 324, 296, 315, 310, 301, 320, 323, 306, 328, 322, 302, 314, 318, 326, 312, 325, 321,
   327, 308, 309, 310, 311, 312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328
}; // 3.45 kb
static const cart_index_t iCartXY_ab11_a6[1568] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163, 164, 187, 188, 167, 191, 169, 192, 213, 214, 173, 174, 185,
   186, 224, 225, 179, 181, 229, 209, 211, 196, 197, 218, 199, 220, 207, 208, 215, 185, 165, 190, 191, 168, 211, 212, 171, 193, 187,
   224, 175, 226, 177, 189, 180, 228, 183, 210, 195, 213, 219, 198, 205, 206, 201, 222, 216, 186, 189, 166, 209, 210, 192, 170, 193,
   172, 225, 188, 226, 176, 190, 178, 227, 182, 184, 194, 212, 214, 203, 204, 221, 200, 223, 202, 217, 167, 180, 203, 173, 187, 196,
   214, 205, 221, 179, 197, 191, 209, 219, 227, 185, 208, 228, 225, 213, 199, 215, 224, 207, 229, 211, 220, 218, 179, 168, 204, 185,
   175, 207, 223, 195, 212, 191, 218, 180, 227, 198, 210, 187, 229, 206, 226, 201, 211, 224, 216, 213, 222, 205, 228, 219, 169, 205,
   182, 196, 213, 174, 188, 219, 203, 199, 181, 211, 192, 228, 221, 207, 186, 227, 214, 224, 197, 208, 229, 215, 225, 218, 209, 220,
   181, 206, 170, 208, 222, 186, 176, 210, 194, 220, 192, 228, 182, 212, 200, 229, 188, 204, 202, 226, 209, 214, 223, 225, 217, 227,
   203, 221, 207, 171, 184, 211, 195, 218, 204, 177, 190, 213, 229, 201, 223, 183, 193, 205, 227, 189, 212, 198, 224, 228, 206, 219,
   210, 216, 226, 222, 208, 183, 172, 220, 206, 209, 194, 189, 178, 229, 214, 222, 202, 193, 184, 228, 203, 190, 200, 210, 225, 221,
   212, 227, 204, 226, 217, 223, 173, 175, 217, 179, 180, 199, 221, 201, 223, 185, 215, 187, 225, 216, 226, 191, 220, 222, 227, 205,
   207, 218, 219, 211, 228, 213, 229, 224, 174, 216, 176, 197, 219, 181, 182, 222, 202, 215, 186, 224, 188, 226, 217, 218, 192, 223,
   203, 228, 208, 209, 227, 220, 221, 229, 214, 225, 215, 177, 178, 218, 198, 220, 200, 183, 184, 224, 225, 216, 217, 189, 190, 219,
   221, 193, 204, 206, 229, 227, 210, 228, 212, 222, 223, 226, 197, 198, 194, 215, 216, 208, 202, 206, 200, 218, 209, 219, 203, 210,
   204, 224, 214, 212, 217, 222, 220, 225, 226, 229, 223, 228, 221, 227, 199, 195, 200, 207, 201, 215, 217, 198, 204, 211, 220, 205,
   221, 206, 212, 213, 225, 210, 223, 216, 218, 229, 222, 224, 226, 219, 227, 228, 196, 201, 202, 199, 205, 197, 203, 216, 217, 207,
   208, 213, 214, 222, 223, 211, 209, 226, 221, 219, 215, 220, 228, 218, 227, 224, 225, 229, 230, 257, 258, 235, 259, 237, 261, 284,
   285, 239, 240, 251, 252, 296, 297, 247, 249, 302, 280, 282, 263, 268, 292, 270, 294, 272, 273, 287, 259, 231, 260, 257, 233, 284,
   286, 238, 262, 253, 298, 241, 299, 242, 254, 245, 303, 250, 278, 264, 283, 290, 266, 274, 276, 271, 295, 288, 261, 262, 232, 285,
   286, 258, 234, 260, 236, 300, 255, 301, 243, 256, 244, 304, 246, 248, 265, 279, 281, 275, 277, 291, 267, 293, 269, 289, 251, 233,
   279, 259, 241, 282, 301, 264, 286, 257, 296, 245, 305, 266, 278, 253, 307, 276, 299, 271, 284, 298, 288, 283, 295, 274, 303, 290,
   252, 278, 234, 280, 299, 261, 243, 286, 265, 297, 258, 305, 246, 279, 267, 306, 255, 277, 269, 301, 285, 281, 293, 300, 289, 304,
   275, 291, 235, 253, 281, 239, 257, 263, 285, 283, 300, 247, 268, 259, 280, 298, 306, 251, 273, 307, 297, 284, 270, 287, 296, 272,
   302, 282, 294, 292, 280, 254, 236, 297, 278, 285, 265, 262, 244, 306, 281, 299, 269, 260, 248, 305, 275, 256, 267, 286, 300, 291,
   279, 304, 277, 301, 289, 293, 237, 283, 255, 263, 284, 240, 258, 298, 281, 270, 249, 282, 261, 307, 300, 272, 252, 306, 285, 296,
   268, 273, 302, 287, 297, 292, 280, 294, 282, 238, 256, 284, 264, 296, 279, 242, 260, 283, 307, 271, 301, 250, 262, 274, 305, 254,
   286, 266, 298, 303, 276, 290, 278, 288, 299, 295, 239, 245, 291, 247, 253, 270, 300, 274, 304, 251, 287, 257, 297, 290, 305, 259,
   294, 303, 306, 283, 272, 292, 298, 282, 307, 284, 302, 296, 240, 290, 246, 268, 298, 249, 255, 303, 275, 287, 252, 296, 258, 305,
   291, 292, 261, 304, 281, 307, 273, 280, 306, 294, 300, 302, 285, 297, 247, 241, 293, 251, 245, 272, 304, 271, 301, 259, 292, 253,
   306, 288, 299, 257, 302, 295, 305, 274, 282, 296, 290, 284, 303, 283, 307, 298, 292, 242, 248, 296, 266, 302, 277, 250, 256, 298,
   306, 288, 293, 254, 260, 290, 304, 262, 279, 276, 307, 305, 278, 303, 286, 295, 301, 299, 249, 295, 243, 273, 303, 252, 246, 299,
   269, 294, 261, 307, 255, 301, 289, 302, 258, 293, 275, 305, 280, 285, 304, 297, 291, 306, 281, 300, 294, 250, 244, 302, 276, 297,
   267, 254, 248, 307, 300, 295, 289, 262, 256, 303, 291, 260, 277, 278, 306, 304, 286, 305, 279, 299, 293, 301, 263, 274, 275, 270,
   283, 268, 281, 290, 291, 272, 273, 284, 285, 303, 304, 282, 280, 305, 300, 298, 287, 294, 307, 292, 306, 296, 297, 302, 272, 264,
   277, 282, 271, 292, 293, 266, 279, 284, 302, 274, 304, 276, 286, 283, 306, 278, 301, 288, 296, 307, 295, 298, 299, 290, 305, 303,
   273, 276, 265, 294, 295, 280, 269, 278, 267, 302, 285, 303, 275, 286, 277, 307, 281, 279, 289, 299, 297, 300, 301, 306, 293, 305,
   291, 304, 287, 266, 267, 292, 288, 294, 289, 276, 277, 296, 297, 290, 291, 278, 279, 298, 300, 286, 293, 295, 302, 306, 299, 307,
   301, 303, 304, 305, 268, 288, 269, 287, 290, 273, 275, 295, 289, 292, 280, 298, 281, 299, 293, 296, 285, 301, 291, 303, 294, 297,
   305, 302, 304, 307, 300, 306, 270, 271, 289, 272, 274, 287, 291, 288, 293, 282, 294, 283, 300, 295, 301, 284, 297, 299, 304, 290,
   292, 302, 303, 296, 305, 298, 306, 307
}; // 3.06 kb
static const cart_index_t iCartXY_ab12_a0[455] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233,
   234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259,
   260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285,
   286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296, 297, 298, 299, 300, 301, 302, 303, 304, 305, 306, 307, 308, 309, 310, 311,
   312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 330, 331, 332, 333, 334, 335, 336, 337,
   338, 339, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 358, 359, 360, 361, 362, 363,
   364, 365, 366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386, 387, 388, 389,
   390, 391, 392, 393, 394, 395, 396, 397, 398, 399, 400, 401, 402, 403, 404, 405, 406, 407, 408, 409, 410, 411, 412, 413, 414, 415,
   416, 417, 418, 419, 420, 421, 422, 423, 424, 425, 426, 427, 428, 429, 430, 431, 432, 433, 434, 435, 436, 437, 438, 439, 440, 441,
   442, 443, 444, 445, 446, 447, 448, 449, 450, 451, 452, 453, 454
}; // 0.89 kb
static const cart_index_t iCartXY_ab12_a1[1092] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215, 219, 222, 224, 223, 220, 226, 225, 227, 221, 230, 223, 250, 231, 249, 225, 222, 228, 251, 249, 233, 227, 224, 251, 229, 250,
   226, 232, 228, 234, 254, 229, 252, 236, 235, 230, 256, 253, 232, 238, 237, 257, 231, 255, 239, 233, 242, 235, 260, 243, 258, 237,
   234, 240, 262, 259, 245, 239, 236, 263, 241, 261, 238, 244, 240, 246, 266, 241, 264, 247, 246, 242, 268, 265, 244, 248, 247, 269,
   243, 267, 248, 245, 251, 254, 252, 256, 250, 253, 257, 255, 249, 271, 253, 261, 272, 259, 255, 252, 270, 263, 258, 272, 257, 254,
   262, 270, 260, 256, 271, 262, 266, 273, 263, 275, 264, 268, 260, 274, 269, 276, 258, 277, 261, 265, 278, 267, 259, 281, 265, 267,
   264, 280, 269, 266, 268, 279, 270, 273, 275, 274, 271, 277, 276, 278, 272, 279, 274, 283, 280, 282, 276, 273, 279, 284, 282, 281,
   278, 275, 284, 280, 283, 277, 281, 284, 283, 282, 285, 290, 292, 288, 286, 293, 289, 291, 287, 290, 294, 318, 296, 288, 319, 292,
   318, 295, 298, 320, 289, 319, 293, 297, 320, 299, 291, 294, 302, 325, 295, 323, 304, 300, 296, 326, 301, 324, 298, 321, 297, 305,
   322, 303, 299, 302, 306, 327, 308, 300, 329, 304, 328, 307, 310, 330, 301, 331, 305, 309, 332, 311, 303, 306, 314, 337, 307, 335,
   316, 312, 308, 338, 313, 336, 310, 333, 309, 317, 334, 315, 311, 314, 312, 339, 316, 340, 313, 341, 317, 315, 324, 322, 320, 326,
   319, 321, 318, 325, 323, 323, 342, 328, 343, 321, 331, 325, 327, 342, 344, 332, 322, 329, 326, 343, 330, 344, 324, 336, 346, 330,
   348, 334, 332, 338, 329, 345, 350, 331, 333, 327, 337, 347, 328, 349, 335, 335, 352, 340, 354, 333, 341, 337, 339, 351, 356, 341,
   334, 339, 338, 353, 340, 355, 336, 342, 347, 349, 345, 343, 350, 346, 348, 344, 347, 351, 357, 353, 345, 358, 349, 357, 352, 355,
   359, 346, 358, 350, 354, 359, 356, 348, 351, 353, 362, 352, 361, 355, 360, 354, 356, 361, 360, 359, 362, 358, 360, 357, 362, 361,
   363, 366, 368, 367, 364, 370, 369, 371, 365, 374, 367, 400, 375, 399, 369, 366, 372, 401, 399, 377, 371, 368, 401, 373, 400, 370,
   376, 372, 378, 404, 373, 402, 380, 379, 374, 406, 403, 376, 382, 381, 407, 375, 405, 383, 377, 386, 379, 410, 387, 408, 381, 378,
   384, 412, 409, 389, 383, 380, 413, 385, 411, 382, 388, 384, 390, 416, 385, 414, 392, 391, 386, 418, 415, 388, 394, 393, 419, 387,
   417, 395, 389, 396, 391, 422, 397, 420, 393, 390, 396, 424, 421, 398, 395, 392, 425, 397, 423, 394, 398, 401, 404, 402, 406, 400,
   403, 407, 405, 399, 427, 403, 411, 428, 409, 405, 402, 426, 413, 408, 428, 407, 404, 412, 426, 410, 406, 427, 412, 416, 429, 413,
   431, 414, 418, 410, 430, 419, 432, 408, 433, 411, 415, 434, 417, 409, 439, 415, 423, 440, 421, 417, 414, 436, 425, 420, 438, 419,
   416, 424, 435, 422, 418, 437, 424, 422, 441, 425, 442, 420, 443, 423, 421, 426, 429, 431, 430, 427, 433, 432, 434, 428, 437, 430,
   445, 438, 444, 432, 429, 435, 446, 444, 440, 434, 431, 446, 436, 445, 433, 439, 435, 441, 449, 436, 447, 442, 441, 437, 451, 448,
   439, 443, 442, 452, 438, 450, 443, 440, 446, 449, 447, 451, 445, 448, 452, 450, 444, 453, 448, 450, 447, 453, 452, 449, 451, 453
}; // 2.13 kb
static const cart_index_t iCartXY_ab12_a2[1716] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212, 216, 225, 226, 219, 221, 248, 227, 217,
   229, 220, 247, 223, 228, 230, 218, 246, 222, 224, 219, 231, 249, 225, 248, 251, 232, 220, 250, 227, 253, 247, 221, 251, 233, 248,
   226, 249, 234, 252, 222, 254, 228, 246, 253, 223, 235, 247, 250, 229, 254, 236, 224, 252, 246, 230, 225, 237, 267, 231, 251, 259,
   226, 267, 238, 249, 233, 260, 239, 227, 268, 232, 257, 253, 240, 269, 228, 255, 234, 254, 268, 229, 241, 250, 258, 235, 269, 242,
   230, 256, 252, 236, 231, 243, 270, 237, 259, 263, 243, 232, 271, 239, 265, 257, 233, 272, 244, 260, 238, 261, 244, 273, 234, 266,
   240, 255, 274, 235, 245, 258, 262, 241, 275, 245, 236, 264, 256, 242, 237, 239, 276, 243, 263, 265, 238, 277, 240, 261, 244, 266,
   278, 241, 242, 262, 264, 245, 255, 256, 246, 269, 254, 252, 257, 247, 258, 253, 268, 250, 248, 259, 260, 251, 249, 267, 249, 270,
   261, 267, 260, 272, 271, 250, 262, 268, 274, 258, 251, 263, 272, 259, 267, 270, 273, 264, 252, 275, 269, 256, 265, 253, 274, 257,
   271, 268, 266, 275, 254, 273, 255, 269, 261, 279, 255, 277, 266, 273, 279, 262, 256, 278, 275, 264, 263, 257, 280, 265, 276, 271,
   280, 258, 264, 274, 278, 262, 259, 265, 281, 263, 270, 276, 260, 281, 266, 272, 261, 277, 267, 276, 277, 270, 272, 281, 276, 268,
   278, 271, 280, 274, 277, 278, 269, 279, 273, 275, 270, 271, 279, 276, 281, 280, 272, 280, 273, 281, 277, 279, 281, 274, 275, 280,
   279, 278, 282, 291, 292, 287, 289, 315, 293, 283, 294, 285, 316, 290, 295, 296, 284, 317, 286, 288, 297, 285, 318, 293, 323, 316,
   298, 319, 286, 321, 295, 317, 287, 299, 320, 291, 315, 322, 321, 300, 288, 319, 317, 296, 289, 322, 301, 315, 292, 320, 323, 290,
   302, 316, 318, 294, 291, 303, 339, 299, 322, 324, 292, 339, 304, 320, 301, 325, 305, 293, 340, 297, 326, 323, 340, 294, 306, 318,
   328, 302, 307, 341, 295, 327, 298, 321, 341, 308, 296, 329, 319, 300, 309, 297, 342, 305, 335, 326, 310, 343, 298, 333, 307, 327,
   299, 311, 344, 303, 324, 334, 345, 312, 300, 331, 329, 308, 301, 346, 313, 325, 304, 332, 347, 302, 314, 328, 330, 306, 303, 309,
   348, 311, 334, 336, 304, 349, 310, 332, 313, 337, 311, 305, 350, 309, 336, 335, 351, 306, 312, 330, 338, 314, 313, 352, 307, 337,
   310, 333, 353, 314, 308, 338, 331, 312, 315, 324, 325, 322, 320, 339, 326, 316, 328, 323, 340, 318, 327, 329, 317, 341, 321, 319,
   342, 318, 330, 340, 347, 328, 343, 331, 319, 345, 341, 329, 320, 344, 332, 339, 325, 346, 333, 345, 321, 343, 327, 341, 322, 334,
   346, 324, 339, 344, 335, 323, 347, 326, 342, 340, 324, 336, 354, 334, 344, 348, 325, 354, 337, 346, 332, 349, 336, 326, 355, 335,
   350, 342, 337, 356, 327, 352, 333, 343, 355, 328, 338, 347, 351, 330, 356, 338, 329, 353, 345, 331, 357, 330, 331, 351, 353, 338,
   332, 358, 333, 349, 337, 352, 334, 335, 359, 336, 348, 350, 339, 348, 349, 344, 346, 354, 350, 340, 351, 342, 355, 347, 352, 353,
   341, 356, 343, 345, 348, 342, 357, 350, 359, 355, 349, 357, 343, 358, 352, 356, 344, 350, 358, 348, 354, 359, 358, 351, 345, 357,
   356, 353, 346, 359, 352, 354, 349, 358, 359, 347, 353, 355, 357, 351, 354, 355, 356, 359, 358, 357, 360, 369, 370, 363, 365, 398,
   371, 361, 373, 364, 397, 367, 372, 374, 362, 396, 366, 368, 363, 375, 399, 369, 398, 401, 376, 364, 400, 371, 403, 397, 365, 401,
   377, 398, 370, 399, 378, 402, 366, 404, 372, 396, 403, 367, 379, 397, 400, 373, 404, 380, 368, 402, 396, 374, 369, 381, 423, 375,
   401, 409, 370, 423, 382, 399, 377, 410, 383, 371, 424, 376, 407, 403, 384, 425, 372, 405, 378, 404, 424, 373, 385, 400, 408, 379,
   425, 386, 374, 406, 402, 380, 375, 387, 426, 381, 409, 413, 388, 376, 427, 383, 415, 407, 377, 428, 389, 410, 382, 411, 390, 429,
   378, 416, 384, 405, 430, 379, 391, 408, 412, 385, 431, 392, 380, 414, 406, 386, 381, 393, 432, 387, 413, 421, 382, 433, 394, 411,
   389, 422, 393, 383, 434, 388, 419, 415, 394, 435, 384, 417, 390, 416, 436, 385, 395, 412, 420, 391, 437, 395, 386, 418, 414, 392,
   387, 388, 438, 393, 421, 419, 389, 439, 390, 422, 394, 417, 440, 391, 392, 420, 418, 395, 405, 406, 396, 425, 404, 402, 407, 397,
   408, 403, 424, 400, 398, 409, 410, 401, 399, 423, 399, 426, 411, 423, 410, 428, 427, 400, 412, 424, 430, 408, 401, 413, 428, 409,
   423, 426, 429, 414, 402, 431, 425, 406, 415, 403, 430, 407, 427, 424, 416, 431, 404, 429, 405, 425, 417, 441, 405, 435, 416, 429,
   441, 418, 406, 437, 431, 414, 419, 407, 442, 415, 434, 427, 442, 408, 420, 430, 436, 412, 409, 421, 443, 413, 426, 432, 410, 443,
   422, 428, 411, 433, 411, 444, 417, 433, 422, 439, 445, 412, 418, 436, 440, 420, 413, 419, 446, 421, 432, 438, 447, 420, 414, 440,
   437, 418, 421, 415, 448, 419, 438, 434, 422, 449, 416, 439, 417, 435, 423, 432, 433, 426, 428, 443, 434, 424, 436, 427, 442, 430,
   435, 437, 425, 441, 429, 431, 426, 438, 444, 432, 443, 446, 438, 427, 445, 434, 448, 442, 428, 446, 439, 443, 433, 444, 439, 447,
   429, 449, 435, 441, 448, 430, 440, 442, 445, 436, 449, 440, 431, 447, 441, 437, 432, 434, 450, 438, 446, 448, 433, 450, 435, 444,
   439, 449, 450, 436, 437, 445, 447, 440, 444, 445, 441, 450, 449, 447, 446, 442, 447, 448, 450, 445, 443, 448, 449, 446, 444, 450
}; // 3.35 kb
static const cart_index_t iCartXY_ab12_a3[2200] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209, 210, 225, 227, 219, 220, 213, 243, 215, 245, 242, 226, 211, 229, 214, 244, 221, 223, 247,
   217, 241, 228, 230, 212, 246, 216, 248, 218, 222, 224, 240, 233, 214, 252, 221, 262, 226, 244, 251, 241, 247, 234, 250, 216, 263,
   222, 249, 240, 228, 246, 248, 213, 231, 254, 225, 243, 219, 261, 242, 253, 245, 249, 236, 218, 250, 240, 263, 224, 248, 230, 246,
   215, 253, 232, 245, 227, 242, 254, 220, 261, 243, 251, 217, 235, 241, 252, 247, 229, 262, 223, 244, 219, 237, 266, 231, 261, 225,
   264, 245, 257, 253, 220, 264, 238, 261, 232, 243, 255, 227, 266, 254, 237, 221, 268, 226, 265, 233, 262, 259, 247, 251, 265, 223,
   239, 244, 256, 262, 235, 268, 229, 252, 238, 269, 222, 267, 228, 260, 248, 234, 263, 249, 267, 239, 224, 258, 246, 269, 230, 263,
   236, 250, 231, 226, 274, 233, 270, 237, 265, 257, 251, 259, 232, 273, 228, 271, 234, 255, 249, 238, 267, 260, 225, 233, 275, 237,
   264, 231, 270, 253, 259, 257, 273, 235, 230, 256, 250, 272, 236, 269, 239, 258, 227, 275, 234, 266, 238, 254, 260, 232, 271, 255,
   274, 229, 236, 252, 258, 268, 239, 272, 235, 256, 242, 257, 255, 253, 254, 245, 266, 243, 264, 261, 259, 241, 256, 247, 268, 251,
   252, 265, 244, 262, 260, 258, 240, 269, 248, 267, 246, 249, 250, 263, 270, 244, 258, 262, 272, 265, 256, 274, 252, 268, 271, 256,
   246, 272, 263, 273, 250, 267, 258, 269, 243, 270, 260, 264, 255, 261, 271, 254, 275, 266, 255, 272, 248, 273, 249, 271, 263, 260,
   269, 267, 245, 259, 271, 257, 266, 253, 275, 261, 270, 264, 257, 247, 272, 251, 274, 259, 268, 270, 262, 265, 253, 251, 273, 259,
   275, 257, 274, 264, 265, 270, 254, 274, 249, 275, 260, 266, 267, 255, 273, 271, 275, 252, 250, 268, 269, 274, 258, 273, 256, 272,
   261, 265, 267, 270, 271, 264, 273, 266, 274, 275, 264, 262, 269, 265, 273, 270, 272, 275, 268, 274, 266, 268, 263, 274, 267, 275,
   269, 271, 272, 273, 276, 293, 295, 285, 286, 281, 314, 283, 316, 309, 291, 277, 296, 279, 312, 287, 288, 317, 284, 310, 292, 294,
   278, 313, 280, 315, 282, 289, 290, 311, 281, 297, 319, 293, 314, 285, 333, 309, 318, 316, 299, 279, 322, 287, 334, 291, 312, 320,
   310, 317, 283, 318, 298, 316, 295, 309, 319, 286, 333, 314, 301, 323, 280, 335, 289, 321, 311, 292, 313, 315, 320, 284, 300, 310,
   322, 317, 296, 334, 288, 312, 321, 302, 282, 323, 311, 335, 290, 315, 294, 313, 285, 305, 340, 297, 333, 293, 338, 316, 328, 318,
   286, 338, 307, 333, 298, 314, 326, 295, 340, 319, 303, 287, 341, 291, 336, 299, 334, 329, 317, 320, 304, 339, 289, 337, 292, 327,
   315, 301, 335, 321, 336, 288, 308, 312, 324, 334, 300, 341, 296, 322, 337, 306, 290, 325, 313, 339, 294, 335, 302, 323, 293, 303,
   348, 305, 338, 297, 342, 318, 330, 328, 305, 291, 349, 299, 344, 303, 336, 330, 320, 329, 295, 348, 304, 340, 307, 319, 331, 298,
   343, 326, 307, 350, 292, 346, 301, 331, 321, 304, 337, 327, 349, 296, 306, 322, 332, 341, 308, 345, 300, 324, 350, 308, 294, 332,
   323, 347, 302, 339, 306, 325, 297, 299, 353, 303, 342, 305, 344, 328, 329, 330, 298, 352, 301, 343, 304, 326, 327, 307, 346, 331,
   351, 300, 302, 324, 325, 345, 306, 347, 308, 332, 327, 325, 311, 339, 315, 337, 313, 321, 323, 335, 329, 310, 324, 317, 341, 320,
   322, 336, 312, 334, 309, 328, 326, 318, 319, 316, 340, 314, 338, 333, 314, 342, 331, 338, 326, 333, 343, 319, 348, 340, 344, 312,
   332, 334, 345, 336, 324, 349, 322, 341, 316, 330, 343, 328, 340, 318, 348, 333, 342, 338, 346, 332, 313, 347, 335, 350, 323, 337,
   325, 339, 330, 317, 345, 320, 349, 329, 341, 344, 334, 336, 331, 347, 315, 350, 321, 346, 335, 327, 339, 337, 326, 351, 321, 352,
   327, 343, 337, 331, 350, 346, 352, 324, 323, 345, 339, 351, 325, 350, 332, 347, 328, 320, 351, 329, 353, 330, 349, 342, 336, 344,
   353, 322, 325, 341, 347, 349, 332, 351, 324, 345, 318, 329, 352, 330, 348, 328, 353, 338, 344, 342, 319, 353, 327, 348, 331, 340,
   346, 326, 352, 343, 333, 344, 346, 342, 343, 338, 352, 340, 353, 348, 342, 334, 347, 336, 351, 344, 345, 353, 341, 349, 343, 345,
   335, 351, 337, 352, 339, 346, 347, 350, 338, 336, 350, 344, 352, 342, 351, 348, 349, 353, 340, 349, 337, 353, 346, 348, 350, 343,
   351, 352, 348, 341, 339, 349, 350, 353, 347, 352, 345, 351, 354, 369, 371, 363, 364, 357, 393, 359, 395, 392, 370, 355, 373, 358,
   394, 365, 367, 397, 361, 391, 372, 374, 356, 396, 360, 398, 362, 366, 368, 390, 377, 358, 402, 365, 418, 370, 394, 401, 391, 397,
   378, 400, 360, 419, 366, 399, 390, 372, 396, 398, 357, 375, 404, 369, 393, 363, 417, 392, 403, 395, 399, 380, 362, 400, 390, 419,
   368, 398, 374, 396, 359, 403, 376, 395, 371, 392, 404, 364, 417, 393, 401, 361, 379, 391, 402, 397, 373, 418, 367, 394, 363, 381,
   422, 375, 417, 369, 420, 395, 407, 403, 364, 420, 383, 417, 376, 393, 405, 371, 422, 404, 382, 365, 424, 370, 421, 377, 418, 409,
   397, 401, 421, 367, 385, 394, 406, 418, 379, 424, 373, 402, 384, 425, 366, 423, 372, 410, 398, 378, 419, 399, 423, 386, 368, 408,
   396, 425, 374, 419, 380, 400, 387, 370, 436, 377, 428, 382, 421, 413, 401, 409, 388, 435, 372, 429, 378, 411, 399, 384, 423, 410,
   369, 387, 437, 381, 420, 375, 426, 403, 415, 407, 435, 389, 374, 412, 400, 431, 380, 425, 386, 408, 371, 437, 388, 422, 383, 404,
   416, 376, 427, 405, 436, 373, 389, 402, 414, 424, 385, 430, 379, 406, 375, 382, 440, 387, 426, 381, 432, 407, 413, 415, 376, 438,
   384, 427, 388, 405, 411, 383, 433, 416, 381, 377, 442, 382, 432, 387, 428, 415, 409, 413, 439, 379, 386, 406, 412, 430, 389, 434,
   385, 414, 383, 443, 378, 433, 384, 416, 410, 388, 429, 411, 441, 385, 380, 414, 408, 434, 386, 431, 389, 412, 392, 407, 405, 403,
   404, 395, 422, 393, 420, 417, 409, 391, 406, 397, 424, 401, 402, 421, 394, 418, 410, 408, 390, 425, 398, 423, 396, 399, 400, 419,
   428, 394, 414, 418, 430, 421, 406, 436, 402, 424, 429, 412, 396, 431, 419, 435, 400, 423, 408, 425, 393, 426, 416, 420, 405, 417,
   427, 404, 437, 422, 411, 431, 398, 435, 399, 429, 419, 410, 425, 423, 395, 415, 427, 407, 422, 403, 437, 417, 426, 420, 413, 397,
   430, 401, 436, 409, 424, 428, 418, 421, 403, 413, 438, 415, 437, 407, 440, 420, 432, 426, 404, 440, 411, 437, 416, 422, 433, 405,
   438, 427, 415, 401, 439, 409, 442, 413, 436, 432, 421, 428, 416, 441, 399, 443, 410, 433, 423, 411, 435, 429, 442, 402, 412, 424,
   434, 436, 414, 439, 406, 430, 443, 414, 400, 434, 425, 441, 408, 435, 412, 431, 444, 406, 408, 430, 431, 439, 412, 441, 414, 434,
   405, 444, 410, 438, 411, 427, 429, 416, 443, 433, 407, 409, 444, 413, 440, 415, 442, 426, 428, 432, 417, 432, 433, 426, 427, 420,
   438, 422, 440, 437, 432, 418, 434, 421, 439, 428, 430, 442, 424, 436, 433, 434, 419, 441, 423, 443, 425, 429, 431, 435, 426, 421,
   441, 428, 444, 432, 439, 440, 436, 442, 427, 439, 423, 444, 429, 438, 435, 433, 441, 443, 420, 428, 443, 432, 438, 426, 444, 437,
   442, 440, 438, 430, 425, 439, 435, 444, 431, 443, 434, 441, 422, 442, 429, 440, 433, 437, 443, 427, 444, 438, 440, 424, 431, 436,
   441, 442, 434, 444, 430, 439, 437, 436, 435, 442, 443, 440, 441, 438, 439, 444
}; // 4.30 kb
static const cart_index_t iCartXY_ab12_a4[2475] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195, 200, 221, 222, 203, 215, 205, 217, 243, 244, 209, 210, 251, 233, 235, 232, 223, 201, 225,
   216, 204, 241, 242, 207, 219, 211, 252, 213, 234, 231, 237, 224, 226, 202, 239, 240, 218, 206, 220, 208, 253, 212, 214, 230, 236,
   238, 203, 227, 245, 209, 221, 232, 244, 247, 256, 215, 233, 254, 251, 243, 235, 227, 204, 246, 223, 211, 249, 258, 231, 242, 216,
   255, 234, 252, 237, 241, 205, 247, 228, 232, 243, 210, 222, 254, 245, 235, 217, 256, 244, 251, 233, 228, 248, 206, 250, 259, 224,
   212, 240, 230, 257, 218, 236, 238, 253, 239, 249, 207, 229, 241, 231, 255, 246, 213, 225, 237, 258, 219, 242, 234, 252, 250, 229,
   208, 257, 248, 239, 230, 226, 214, 259, 238, 220, 236, 240, 253, 209, 223, 261, 215, 227, 235, 256, 249, 265, 221, 251, 260, 254,
   247, 243, 210, 260, 224, 233, 254, 217, 228, 265, 250, 251, 222, 261, 245, 256, 244, 221, 211, 262, 227, 216, 247, 264, 237, 258,
   223, 260, 252, 255, 241, 249, 222, 262, 212, 245, 263, 228, 218, 259, 238, 261, 224, 253, 239, 257, 250, 260, 213, 226, 255, 234,
   264, 248, 219, 229, 252, 262, 225, 246, 242, 258, 261, 225, 214, 263, 246, 257, 236, 229, 220, 262, 253, 226, 240, 248, 259, 215,
   216, 263, 221, 223, 243, 265, 241, 264, 227, 254, 255, 260, 249, 247, 217, 264, 218, 244, 265, 222, 224, 263, 239, 256, 228, 257,
   250, 261, 245, 265, 219, 220, 264, 242, 263, 240, 225, 226, 258, 259, 229, 248, 246, 262, 245, 246, 230, 261, 262, 250, 238, 248,
   236, 263, 239, 240, 253, 259, 257, 247, 231, 248, 249, 237, 260, 262, 234, 246, 241, 264, 242, 258, 252, 255, 232, 249, 250, 235,
   247, 233, 245, 260, 261, 243, 244, 265, 256, 254, 251, 233, 255, 239, 251, 260, 244, 250, 264, 257, 254, 245, 263, 261, 265, 256,
   254, 234, 240, 260, 252, 265, 259, 242, 248, 255, 263, 246, 262, 258, 264, 235, 241, 257, 243, 249, 251, 261, 255, 263, 247, 256,
   264, 265, 260, 254, 256, 242, 236, 265, 258, 261, 253, 246, 240, 264, 257, 248, 259, 262, 263, 243, 237, 259, 247, 241, 254, 263,
   252, 262, 249, 265, 258, 264, 255, 260, 244, 258, 238, 256, 264, 245, 239, 262, 253, 265, 250, 259, 257, 263, 261, 251, 252, 253,
   254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 287, 288, 271, 283, 273, 285, 308, 309, 275, 276, 323, 304, 306,
   299, 289, 267, 290, 281, 269, 310, 312, 274, 286, 277, 324, 278, 302, 300, 307, 291, 292, 268, 311, 313, 282, 270, 284, 272, 325,
   279, 280, 301, 303, 305, 293, 269, 314, 289, 277, 319, 331, 300, 312, 281, 326, 302, 324, 307, 310, 294, 315, 270, 317, 329, 291,
   279, 313, 301, 327, 282, 303, 305, 325, 311, 271, 295, 316, 275, 287, 299, 309, 318, 330, 283, 304, 328, 323, 308, 306, 317, 296,
   272, 327, 315, 311, 301, 292, 280, 329, 305, 284, 303, 313, 325, 273, 318, 297, 299, 308, 276, 288, 328, 316, 306, 285, 330, 309,
   323, 304, 319, 274, 298, 310, 300, 326, 314, 278, 290, 307, 331, 286, 312, 302, 324, 275, 293, 333, 283, 295, 306, 330, 320, 338,
   287, 323, 332, 328, 318, 308, 276, 332, 294, 304, 328, 285, 297, 338, 321, 323, 288, 333, 316, 330, 309, 295, 277, 335, 293, 281,
   320, 339, 307, 331, 289, 334, 324, 326, 310, 319, 334, 278, 296, 326, 302, 339, 322, 286, 298, 324, 335, 290, 314, 312, 331, 297,
   337, 279, 321, 340, 294, 282, 329, 305, 336, 291, 325, 311, 327, 317, 336, 298, 280, 340, 322, 327, 303, 296, 284, 337, 325, 292,
   313, 315, 329, 287, 281, 341, 295, 289, 318, 343, 310, 339, 293, 332, 326, 334, 319, 320, 288, 341, 282, 316, 342, 297, 291, 340,
   311, 333, 294, 327, 317, 336, 321, 283, 289, 342, 287, 293, 308, 338, 319, 343, 295, 328, 334, 332, 320, 318, 342, 290, 284, 341,
   314, 340, 313, 298, 292, 335, 329, 296, 315, 322, 337, 285, 343, 291, 309, 338, 288, 294, 342, 317, 330, 297, 336, 321, 333, 316,
   343, 286, 292, 339, 312, 341, 315, 290, 296, 331, 337, 298, 322, 314, 335, 299, 320, 321, 306, 318, 304, 316, 332, 333, 308, 309,
   338, 330, 328, 323, 320, 300, 322, 319, 307, 334, 335, 302, 314, 310, 339, 312, 331, 324, 326, 321, 322, 301, 336, 337, 317, 305,
   315, 303, 340, 311, 313, 325, 329, 327, 332, 302, 315, 334, 324, 343, 337, 312, 322, 326, 341, 314, 335, 331, 339, 333, 314, 303,
   342, 335, 336, 325, 322, 313, 341, 327, 315, 329, 337, 340, 304, 334, 317, 323, 332, 309, 321, 343, 336, 328, 316, 342, 333, 338,
   330, 316, 335, 305, 333, 341, 321, 311, 337, 325, 342, 317, 329, 327, 340, 336, 306, 319, 336, 308, 320, 323, 333, 334, 342, 318,
   330, 343, 338, 332, 328, 318, 307, 337, 320, 310, 332, 341, 324, 335, 319, 343, 331, 339, 326, 334, 308, 310, 340, 318, 319, 328,
   342, 326, 341, 320, 338, 339, 343, 334, 332, 309, 339, 311, 330, 343, 316, 317, 341, 327, 338, 321, 340, 336, 342, 333, 338, 312,
   313, 343, 331, 342, 329, 314, 315, 339, 340, 322, 337, 335, 341, 323, 326, 327, 328, 334, 330, 336, 339, 340, 332, 333, 341, 342,
   343, 338, 328, 324, 329, 332, 326, 338, 340, 331, 337, 334, 342, 335, 341, 339, 343, 330, 331, 325, 338, 339, 333, 327, 335, 329,
   343, 336, 337, 340, 341, 342, 344, 365, 366, 347, 359, 349, 361, 393, 394, 353, 354, 407, 383, 385, 382, 367, 345, 369, 360, 348,
   391, 392, 351, 363, 355, 408, 357, 384, 381, 387, 368, 370, 346, 389, 390, 362, 350, 364, 352, 409, 356, 358, 380, 386, 388, 347,
   371, 395, 353, 365, 382, 394, 397, 412, 359, 383, 410, 407, 393, 385, 372, 348, 396, 367, 355, 399, 414, 381, 392, 360, 411, 384,
   408, 387, 391, 349, 397, 373, 382, 393, 354, 366, 410, 395, 385, 361, 412, 394, 407, 383, 374, 398, 350, 400, 415, 368, 356, 390,
   380, 413, 362, 386, 388, 409, 389, 399, 351, 375, 391, 381, 411, 396, 357, 369, 387, 414, 363, 392, 384, 408, 400, 376, 352, 413,
   398, 389, 380, 370, 358, 415, 388, 364, 386, 390, 409, 353, 377, 417, 359, 371, 385, 412, 405, 427, 365, 407, 416, 410, 397, 393,
   354, 416, 378, 383, 410, 361, 373, 427, 406, 407, 366, 417, 395, 412, 394, 377, 355, 420, 372, 360, 403, 426, 387, 414, 367, 418,
   408, 411, 391, 399, 378, 421, 356, 401, 425, 374, 362, 415, 388, 419, 368, 409, 389, 413, 400, 418, 357, 379, 411, 384, 426, 404,
   363, 375, 408, 420, 369, 396, 392, 414, 419, 379, 358, 425, 402, 413, 386, 376, 364, 421, 409, 370, 390, 398, 415, 359, 372, 428,
   365, 377, 393, 427, 403, 430, 371, 410, 422, 416, 405, 397, 371, 360, 429, 377, 367, 405, 432, 391, 426, 372, 422, 411, 418, 399,
   403, 361, 430, 374, 394, 427, 366, 378, 428, 401, 412, 373, 423, 406, 417, 395, 373, 431, 362, 406, 433, 378, 368, 425, 389, 423,
   374, 413, 400, 419, 401, 432, 363, 376, 426, 392, 429, 402, 369, 379, 414, 424, 375, 404, 396, 420, 433, 375, 364, 431, 404, 425,
   390, 379, 370, 424, 415, 376, 398, 402, 421, 365, 367, 434, 371, 372, 397, 430, 399, 432, 377, 416, 418, 422, 403, 405, 366, 434,
   368, 395, 428, 373, 374, 433, 400, 417, 378, 419, 401, 423, 406, 434, 369, 370, 429, 396, 431, 398, 375, 376, 420, 421, 379, 402,
   404, 424, 401, 402, 380, 419, 421, 400, 388, 398, 386, 425, 389, 390, 409, 415, 413, 403, 381, 404, 399, 387, 418, 420, 384, 396,
   391, 426, 392, 414, 408, 411, 382, 405, 406, 385, 397, 383, 395, 416, 417, 393, 394, 427, 412, 410, 407, 383, 422, 401, 407, 416,
   394, 406, 430, 423, 410, 395, 428, 417, 427, 412, 422, 384, 402, 418, 408, 432, 424, 392, 404, 411, 429, 396, 420, 414, 426, 385,
   403, 423, 393, 405, 407, 417, 422, 428, 397, 412, 430, 427, 416, 410, 423, 404, 386, 433, 424, 419, 409, 402, 390, 431, 413, 398,
   415, 421, 425, 405, 387, 424, 403, 391, 422, 429, 408, 420, 399, 432, 414, 426, 411, 418, 406, 424, 388, 423, 431, 401, 389, 421,
   409, 433, 400, 415, 413, 425, 419, 395, 429, 389, 417, 434, 406, 400, 431, 413, 428, 401, 425, 419, 433, 423, 428, 396, 390, 434,
   420, 433, 415, 404, 398, 429, 425, 402, 421, 424, 431, 397, 391, 431, 405, 399, 416, 434, 411, 429, 403, 430, 426, 432, 418, 422,
   430, 392, 398, 432, 414, 434, 421, 396, 402, 426, 431, 404, 424, 420, 429, 393, 399, 433, 397, 403, 410, 428, 418, 434, 405, 427,
   432, 430, 422, 416, 394, 432, 400, 412, 430, 395, 401, 434, 419, 427, 406, 433, 423, 428, 417, 407, 418, 419, 410, 422, 412, 423,
   432, 433, 416, 417, 434, 428, 430, 427, 416, 408, 421, 422, 411, 430, 431, 414, 424, 418, 434, 420, 429, 426, 432, 417, 420, 409,
   428, 429, 423, 413, 424, 415, 434, 419, 421, 425, 431, 433, 410, 411, 425, 416, 418, 427, 433, 426, 431, 422, 428, 429, 434, 432,
   430, 412, 426, 413, 427, 432, 417, 419, 429, 425, 430, 423, 431, 433, 434, 428, 427, 414, 415, 430, 426, 428, 425, 420, 421, 432,
   433, 424, 431, 429, 434
}; // 4.83 kb
static const cart_index_t iCartXY_ab12_a5[2520] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180, 185, 212, 213, 206, 207, 188, 230, 190, 232, 194, 195, 200, 239, 202, 241, 217, 228, 229, 236,
   218, 220, 212, 186, 214, 189, 231, 208, 210, 234, 192, 201, 240, 196, 198, 243, 204, 226, 216, 227, 219, 237, 222, 213, 214, 187,
   233, 191, 235, 193, 209, 211, 242, 203, 244, 205, 197, 199, 224, 225, 215, 221, 223, 238, 206, 189, 233, 196, 247, 212, 231, 232,
   216, 208, 245, 201, 219, 249, 227, 234, 222, 243, 237, 240, 226, 207, 231, 191, 247, 197, 230, 215, 213, 233, 246, 209, 248, 225,
   203, 221, 235, 244, 223, 238, 224, 242, 188, 208, 235, 212, 230, 194, 246, 217, 234, 200, 218, 206, 245, 229, 250, 220, 232, 241,
   239, 236, 228, 230, 210, 193, 231, 215, 246, 199, 235, 214, 248, 224, 247, 211, 223, 205, 242, 233, 221, 225, 238, 244, 190, 234,
   209, 232, 213, 217, 235, 195, 245, 220, 202, 228, 250, 207, 246, 218, 239, 230, 241, 229, 236, 232, 192, 211, 216, 233, 234, 214,
   245, 198, 226, 249, 222, 204, 247, 210, 240, 219, 231, 227, 243, 237, 194, 201, 242, 208, 246, 200, 248, 220, 226, 206, 236, 212,
   240, 241, 249, 228, 234, 250, 245, 239, 232, 195, 240, 203, 245, 209, 218, 224, 202, 249, 236, 207, 239, 248, 213, 242, 229, 250,
   235, 246, 230, 241, 200, 196, 244, 201, 248, 206, 247, 228, 222, 212, 239, 208, 237, 250, 243, 232, 226, 249, 240, 245, 234, 239,
   198, 205, 219, 225, 245, 211, 250, 204, 240, 248, 237, 210, 244, 214, 249, 227, 233, 231, 247, 243, 202, 243, 197, 249, 203, 229,
   223, 207, 247, 241, 213, 250, 244, 209, 238, 230, 248, 224, 242, 235, 246, 241, 204, 199, 227, 221, 250, 205, 246, 210, 249, 242,
   243, 214, 238, 211, 248, 231, 225, 233, 244, 247, 217, 226, 224, 234, 235, 220, 242, 218, 240, 228, 229, 232, 249, 230, 248, 236,
   245, 246, 250, 241, 239, 228, 216, 225, 222, 244, 232, 233, 239, 219, 234, 250, 226, 227, 248, 231, 245, 237, 247, 243, 249, 240,
   229, 227, 215, 243, 223, 241, 221, 230, 231, 250, 235, 249, 233, 224, 225, 246, 247, 238, 244, 242, 248, 236, 219, 221, 237, 238,
   239, 225, 241, 227, 245, 246, 240, 231, 242, 233, 250, 243, 244, 247, 248, 249, 218, 237, 223, 240, 224, 236, 238, 229, 243, 239,
   230, 245, 247, 235, 244, 241, 249, 242, 248, 246, 250, 220, 222, 238, 226, 242, 228, 244, 236, 237, 232, 241, 234, 243, 246, 247,
   239, 240, 248, 249, 250, 245, 251, 280, 282, 272, 273, 256, 301, 258, 303, 260, 261, 268, 313, 270, 315, 284, 293, 294, 308, 289,
   291, 278, 252, 283, 254, 299, 274, 275, 304, 259, 266, 311, 262, 263, 316, 271, 295, 285, 297, 287, 309, 292, 279, 281, 253, 300,
   255, 302, 257, 276, 277, 312, 267, 314, 269, 264, 265, 296, 298, 286, 288, 290, 310, 256, 278, 306, 280, 301, 260, 318, 284, 305,
   268, 289, 272, 317, 294, 323, 291, 303, 315, 313, 308, 293, 280, 254, 307, 262, 320, 278, 299, 305, 285, 274, 319, 266, 287, 324,
   297, 304, 292, 316, 309, 311, 295, 258, 305, 279, 303, 282, 284, 306, 261, 317, 291, 270, 293, 323, 273, 318, 289, 313, 301, 315,
   294, 308, 282, 307, 255, 322, 264, 306, 286, 279, 300, 321, 276, 325, 298, 267, 288, 302, 314, 290, 310, 296, 312, 305, 259, 281,
   285, 307, 304, 283, 319, 263, 295, 324, 292, 271, 320, 275, 311, 287, 299, 297, 316, 309, 306, 283, 257, 307, 286, 321, 265, 302,
   281, 325, 296, 322, 277, 290, 269, 312, 300, 288, 298, 310, 314, 260, 274, 321, 278, 318, 268, 327, 291, 304, 272, 308, 280, 319,
   315, 328, 293, 305, 323, 317, 313, 303, 261, 319, 276, 317, 279, 289, 302, 270, 328, 308, 273, 313, 327, 282, 321, 294, 323, 306,
   318, 301, 315, 272, 262, 322, 266, 326, 280, 320, 303, 292, 278, 317, 274, 309, 328, 316, 305, 295, 324, 311, 319, 304, 273, 320,
   264, 326, 267, 301, 290, 282, 322, 318, 279, 327, 314, 276, 310, 306, 325, 296, 312, 302, 321, 317, 263, 277, 287, 300, 319, 281,
   328, 271, 311, 326, 309, 275, 322, 283, 324, 297, 307, 299, 320, 316, 318, 275, 265, 299, 288, 327, 269, 321, 283, 326, 312, 320,
   281, 310, 277, 325, 307, 298, 300, 314, 322, 268, 266, 325, 274, 327, 272, 326, 293, 295, 280, 313, 278, 311, 323, 324, 303, 304,
   328, 319, 317, 305, 270, 324, 267, 328, 276, 294, 296, 273, 326, 315, 282, 323, 325, 279, 312, 301, 327, 302, 321, 306, 318, 323,
   271, 269, 297, 298, 328, 277, 327, 275, 324, 325, 316, 283, 314, 281, 326, 299, 300, 307, 322, 320, 301, 299, 286, 320, 290, 318,
   288, 306, 307, 327, 302, 326, 300, 296, 298, 321, 322, 310, 314, 312, 325, 303, 285, 300, 292, 322, 305, 307, 317, 287, 304, 328,
   295, 297, 326, 299, 319, 309, 320, 316, 324, 311, 284, 304, 302, 305, 306, 291, 321, 289, 319, 293, 294, 303, 328, 301, 327, 308,
   317, 318, 323, 315, 313, 289, 311, 296, 319, 302, 308, 312, 294, 324, 313, 301, 317, 326, 306, 325, 315, 328, 321, 327, 318, 323,
   313, 287, 298, 309, 314, 317, 300, 323, 297, 319, 327, 311, 299, 325, 307, 328, 316, 322, 320, 326, 324, 291, 295, 312, 304, 321,
   293, 325, 308, 311, 303, 315, 305, 324, 318, 326, 313, 319, 327, 328, 323, 317, 315, 297, 288, 316, 310, 323, 298, 318, 299, 328,
   321, 324, 307, 312, 300, 327, 320, 314, 322, 325, 326, 293, 292, 314, 295, 325, 303, 322, 313, 309, 305, 323, 304, 316, 327, 320,
   317, 311, 326, 324, 328, 319, 294, 316, 290, 324, 296, 315, 310, 301, 320, 323, 306, 328, 322, 302, 314, 318, 326, 312, 325, 321,
   327, 308, 309, 310, 311, 312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 356, 358, 350,
   351, 332, 380, 334, 382, 338, 339, 344, 395, 346, 397, 367, 378, 379, 392, 368, 370, 357, 330, 360, 333, 381, 352, 354, 384, 336,
   345, 396, 340, 342, 399, 348, 376, 366, 377, 369, 393, 372, 359, 361, 331, 383, 335, 385, 337, 353, 355, 398, 347, 400, 349, 341,
   343, 374, 375, 365, 371, 373, 394, 362, 333, 389, 340, 405, 357, 381, 388, 366, 352, 403, 345, 369, 411, 377, 384, 372, 399, 393,
   396, 376, 363, 387, 335, 406, 341, 386, 365, 359, 383, 404, 353, 410, 375, 347, 371, 385, 400, 373, 394, 374, 398, 332, 362, 391,
   356, 380, 338, 402, 367, 390, 344, 368, 350, 401, 379, 412, 370, 382, 397, 395, 392, 378, 386, 364, 337, 387, 365, 404, 343, 385,
   361, 410, 374, 406, 355, 373, 349, 398, 383, 371, 375, 394, 400, 334, 390, 363, 382, 358, 367, 391, 339, 401, 370, 346, 378, 412,
   351, 402, 368, 395, 380, 397, 379, 392, 388, 336, 364, 366, 389, 384, 360, 403, 342, 376, 411, 372, 348, 405, 354, 396, 369, 381,
   377, 399, 393, 338, 357, 408, 362, 402, 344, 413, 370, 388, 350, 392, 356, 407, 397, 415, 378, 390, 412, 401, 395, 382, 339, 407,
   359, 401, 363, 368, 386, 346, 415, 392, 351, 395, 413, 358, 408, 379, 412, 391, 402, 380, 397, 356, 340, 409, 345, 414, 362, 405,
   390, 372, 357, 407, 352, 393, 417, 399, 388, 376, 411, 396, 403, 384, 407, 342, 361, 369, 387, 403, 364, 417, 348, 396, 414, 393,
   354, 409, 360, 411, 377, 389, 381, 405, 399, 358, 409, 341, 416, 347, 391, 373, 363, 406, 408, 359, 418, 400, 353, 394, 386, 410,
   374, 398, 385, 404, 408, 360, 343, 389, 371, 418, 349, 404, 364, 416, 398, 409, 361, 394, 355, 410, 387, 375, 383, 400, 406, 350,
   345, 416, 352, 419, 356, 414, 382, 376, 362, 401, 357, 396, 415, 411, 390, 384, 417, 403, 407, 388, 351, 414, 347, 419, 353, 380,
   374, 358, 416, 402, 363, 413, 410, 359, 398, 391, 418, 385, 404, 386, 408, 344, 352, 418, 357, 413, 350, 419, 378, 384, 356, 395,
   362, 403, 412, 417, 382, 388, 415, 407, 401, 390, 413, 354, 349, 381, 375, 419, 355, 418, 360, 414, 410, 405, 364, 400, 361, 416,
   389, 383, 387, 406, 409, 346, 417, 353, 415, 359, 379, 385, 351, 419, 397, 358, 412, 418, 363, 404, 380, 413, 386, 408, 391, 402,
   415, 348, 355, 377, 383, 417, 361, 419, 354, 411, 416, 399, 360, 406, 364, 414, 381, 387, 389, 409, 405, 367, 388, 386, 390, 391,
   370, 408, 368, 407, 378, 379, 382, 415, 380, 413, 392, 401, 402, 412, 397, 395, 390, 366, 387, 372, 409, 388, 389, 407, 369, 384,
   417, 376, 377, 414, 381, 403, 393, 405, 399, 411, 396, 391, 389, 365, 409, 373, 408, 371, 386, 387, 418, 385, 416, 383, 374, 375,
   404, 406, 394, 400, 398, 410, 401, 369, 383, 393, 406, 407, 387, 415, 377, 403, 419, 396, 381, 416, 389, 417, 399, 409, 405, 414,
   411, 402, 381, 371, 405, 394, 413, 375, 408, 389, 419, 404, 414, 387, 398, 383, 418, 409, 400, 406, 410, 416, 368, 403, 385, 407,
   386, 392, 404, 379, 417, 395, 380, 401, 419, 391, 418, 397, 415, 408, 413, 402, 412, 380, 405, 373, 414, 374, 402, 394, 391, 409,
   413, 386, 419, 406, 385, 400, 408, 416, 398, 410, 404, 418, 370, 384, 404, 388, 408, 378, 418, 392, 403, 382, 397, 390, 417, 402,
   419, 395, 407, 413, 415, 412, 401, 382, 372, 406, 376, 416, 390, 409, 401, 393, 388, 415, 384, 399, 419, 405, 407, 396, 414, 411,
   417, 403, 378, 376, 410, 384, 418, 382, 416, 395, 396, 390, 412, 388, 411, 413, 414, 401, 403, 419, 417, 415, 407, 379, 411, 374,
   417, 385, 397, 398, 380, 414, 412, 391, 415, 416, 386, 410, 402, 419, 404, 418, 408, 413, 412, 377, 375, 399, 400, 415, 383, 413,
   381, 417, 418, 411, 389, 410, 387, 419, 405, 406, 409, 416, 414, 392, 396, 398, 403, 404, 395, 410, 397, 411, 401, 402, 407, 414,
   408, 416, 412, 417, 418, 419, 413, 415, 395, 393, 400, 396, 410, 401, 406, 412, 399, 407, 413, 403, 405, 418, 409, 415, 411, 416,
   414, 419, 417, 397, 399, 394, 411, 398, 412, 400, 402, 405, 415, 408, 417, 409, 404, 406, 413, 414, 410, 416, 418, 419
}; // 4.92 kb
static const cart_index_t iCartXY_ab12_a6[2352] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163, 164, 187, 188, 167, 191, 169, 192, 213, 214, 173, 174, 185,
   186, 224, 225, 179, 181, 229, 209, 211, 196, 197, 218, 199, 220, 207, 208, 215, 185, 165, 190, 191, 168, 211, 212, 171, 193, 187,
   224, 175, 226, 177, 189, 180, 228, 183, 210, 195, 213, 219, 198, 205, 206, 201, 222, 216, 186, 189, 166, 209, 210, 192, 170, 193,
   172, 225, 188, 226, 176, 190, 178, 227, 182, 184, 194, 212, 214, 203, 204, 221, 200, 223, 202, 217, 167, 180, 203, 173, 187, 196,
   214, 205, 221, 179, 197, 191, 209, 219, 227, 185, 208, 228, 225, 213, 199, 215, 224, 207, 229, 211, 220, 218, 179, 168, 204, 185,
   175, 207, 223, 195, 212, 191, 218, 180, 227, 198, 210, 187, 229, 206, 226, 201, 211, 224, 216, 213, 222, 205, 228, 219, 169, 205,
   182, 196, 213, 174, 188, 219, 203, 199, 181, 211, 192, 228, 221, 207, 186, 227, 214, 224, 197, 208, 229, 215, 225, 218, 209, 220,
   181, 206, 170, 208, 222, 186, 176, 210, 194, 220, 192, 228, 182, 212, 200, 229, 188, 204, 202, 226, 209, 214, 223, 225, 217, 227,
   203, 221, 207, 171, 184, 211, 195, 218, 204, 177, 190, 213, 229, 201, 223, 183, 193, 205, 227, 189, 212, 198, 224, 228, 206, 219,
   210, 216, 226, 222, 208, 183, 172, 220, 206, 209, 194, 189, 178, 229, 214, 222, 202, 193, 184, 228, 203, 190, 200, 210, 225, 221,
   212, 227, 204, 226, 217, 223, 173, 175, 217, 179, 180, 199, 221, 201, 223, 185, 215, 187, 225, 216, 226, 191, 220, 222, 227, 205,
   207, 218, 219, 211, 228, 213, 229, 224, 174, 216, 176, 197, 219, 181, 182, 222, 202, 215, 186, 224, 188, 226, 217, 218, 192, 223,
   203, 228, 208, 209, 227, 220, 221, 229, 214, 225, 215, 177, 178, 218, 198, 220, 200, 183, 184, 224, 225, 216, 217, 189, 190, 219,
   221, 193, 204, 206, 229, 227, 210, 228, 212, 222, 223, 226, 197, 198, 194, 215, 216, 208, 202, 206, 200, 218, 209, 219, 203, 210,
   204, 224, 214, 212, 217, 222, 220, 225, 226, 229, 223, 228, 221, 227, 199, 195, 200, 207, 201, 215, 217, 198, 204, 211, 220, 205,
   221, 206, 212, 213, 225, 210, 223, 216, 218, 229, 222, 224, 226, 219, 227, 228, 196, 201, 202, 199, 205, 197, 203, 216, 217, 207,
   208, 213, 214, 222, 223, 211, 209, 226, 221, 219, 215, 220, 228, 218, 227, 224, 225, 229, 230, 257, 258, 235, 259, 237, 261, 284,
   285, 239, 240, 251, 252, 296, 297, 247, 249, 302, 280, 282, 263, 268, 292, 270, 294, 272, 273, 287, 259, 231, 260, 257, 233, 284,
   286, 238, 262, 253, 298, 241, 299, 242, 254, 245, 303, 250, 278, 264, 283, 290, 266, 274, 276, 271, 295, 288, 261, 262, 232, 285,
   286, 258, 234, 260, 236, 300, 255, 301, 243, 256, 244, 304, 246, 248, 265, 279, 281, 275, 277, 291, 267, 293, 269, 289, 251, 233,
   279, 259, 241, 282, 301, 264, 286, 257, 296, 245, 305, 266, 278, 253, 307, 276, 299, 271, 284, 298, 288, 283, 295, 274, 303, 290,
   252, 278, 234, 280, 299, 261, 243, 286, 265, 297, 258, 305, 246, 279, 267, 306, 255, 277, 269, 301, 285, 281, 293, 300, 289, 304,
   275, 291, 235, 253, 281, 239, 257, 263, 285, 283, 300, 247, 268, 259, 280, 298, 306, 251, 273, 307, 297, 284, 270, 287, 296, 272,
   302, 282, 294, 292, 280, 254, 236, 297, 278, 285, 265, 262, 244, 306, 281, 299, 269, 260, 248, 305, 275, 256, 267, 286, 300, 291,
   279, 304, 277, 301, 289, 293, 237, 283, 255, 263, 284, 240, 258, 298, 281, 270, 249, 282, 261, 307, 300, 272, 252, 306, 285, 296,
   268, 273, 302, 287, 297, 292, 280, 294, 282, 238, 256, 284, 264, 296, 279, 242, 260, 283, 307, 271, 301, 250, 262, 274, 305, 254,
   286, 266, 298, 303, 276, 290, 278, 288, 299, 295, 239, 245, 291, 247, 253, 270, 300, 274, 304, 251, 287, 257, 297, 290, 305, 259,
   294, 303, 306, 283, 272, 292, 298, 282, 307, 284, 302, 296, 240, 290, 246, 268, 298, 249, 255, 303, 275, 287, 252, 296, 258, 305,
   291, 292, 261, 304, 281, 307, 273, 280, 306, 294, 300, 302, 285, 297, 247, 241, 293, 251, 245, 272, 304, 271, 301, 259, 292, 253,
   306, 288, 299, 257, 302, 295, 305, 274, 282, 296, 290, 284, 303, 283, 307, 298, 292, 242, 248, 296, 266, 302, 277, 250, 256, 298,
   306, 288, 293, 254, 260, 290, 304, 262, 279, 276, 307, 305, 278, 303, 286, 295, 301, 299, 249, 295, 243, 273, 303, 252, 246, 299,
   269, 294, 261, 307, 255, 301, 289, 302, 258, 293, 275, 305, 280, 285, 304, 297, 291, 306, 281, 300, 294, 250, 244, 302, 276, 297,
   267, 254, 248, 307, 300, 295, 289, 262, 256, 303, 291, 260, 277, 278, 306, 304, 286, 305, 279, 299, 293, 301, 263, 274, 275, 270,
   283, 268, 281, 290, 291, 272, 273, 284, 285, 303, 304, 282, 280, 305, 300, 298, 287, 294, 307, 292, 306, 296, 297, 302, 272, 264,
   277, 282, 271, 292, 293, 266, 279, 284, 302, 274, 304, 276, 286, 283, 306, 278, 301, 288, 296, 307, 295, 298, 299, 290, 305, 303,
   273, 276, 265, 294, 295, 280, 269, 278, 267, 302, 285, 303, 275, 286, 277, 307, 281, 279, 289, 299, 297, 300, 301, 306, 293, 305,
   291, 304, 287, 266, 267, 292, 288, 294, 289, 276, 277, 296, 297, 290, 291, 278, 279, 298, 300, 286, 293, 295, 302, 306, 299, 307,
   301, 303, 304, 305, 268, 288, 269, 287, 290, 273, 275, 295, 289, 292, 280, 298, 281, 299, 293, 296, 285, 301, 291, 303, 294, 297,
   305, 302, 304, 307, 300, 306, 270, 271, 289, 272, 274, 287, 291, 288, 293, 282, 294, 283, 300, 295, 301, 284, 297, 299, 304, 290,
   292, 302, 303, 296, 305, 298, 306, 307, 308, 341, 342, 311, 335, 313, 337, 369, 370, 317, 318, 329, 330, 380, 381, 323, 325, 391,
   359, 361, 346, 347, 374, 349, 376, 357, 358, 371, 341, 309, 343, 336, 312, 367, 368, 315, 339, 331, 382, 319, 384, 321, 333, 324,
   390, 327, 360, 345, 363, 375, 348, 355, 356, 351, 378, 372, 342, 343, 310, 365, 366, 338, 314, 340, 316, 383, 332, 385, 320, 334,
   322, 389, 326, 328, 344, 362, 364, 353, 354, 377, 350, 379, 352, 373, 311, 336, 365, 317, 341, 346, 370, 367, 387, 323, 347, 335,
   359, 386, 392, 329, 358, 394, 381, 369, 349, 371, 380, 357, 391, 361, 376, 374, 335, 312, 366, 341, 319, 369, 388, 345, 368, 336,
   386, 324, 393, 348, 360, 331, 396, 356, 384, 351, 367, 382, 372, 363, 378, 355, 390, 375, 313, 367, 338, 346, 369, 318, 342, 386,
   365, 349, 325, 361, 337, 394, 387, 357, 330, 392, 370, 380, 347, 358, 391, 371, 381, 374, 359, 376, 337, 368, 314, 370, 388, 342,
   320, 366, 344, 387, 338, 395, 326, 362, 350, 397, 332, 354, 352, 385, 365, 364, 379, 383, 373, 389, 353, 377, 369, 315, 340, 367,
   345, 386, 366, 321, 343, 363, 396, 351, 388, 327, 339, 355, 393, 333, 368, 348, 382, 390, 356, 375, 360, 372, 384, 378, 370, 339,
   316, 387, 368, 365, 344, 343, 322, 397, 364, 388, 352, 340, 328, 395, 353, 334, 350, 366, 383, 377, 362, 389, 354, 385, 373, 379,
   317, 331, 383, 323, 336, 349, 387, 363, 397, 329, 371, 341, 381, 382, 398, 335, 376, 396, 392, 367, 357, 374, 386, 361, 394, 369,
   391, 380, 318, 382, 332, 347, 386, 325, 338, 396, 364, 371, 330, 380, 342, 398, 383, 374, 337, 397, 365, 394, 358, 359, 392, 376,
   387, 391, 370, 381, 329, 319, 385, 335, 324, 361, 395, 351, 388, 341, 380, 331, 398, 372, 384, 336, 394, 378, 393, 355, 369, 386,
   375, 367, 390, 363, 396, 382, 330, 384, 320, 359, 393, 337, 326, 388, 352, 381, 342, 398, 332, 385, 373, 392, 338, 379, 353, 395,
   370, 365, 389, 387, 377, 397, 364, 383, 380, 321, 334, 386, 348, 394, 362, 327, 340, 382, 398, 372, 385, 333, 343, 375, 395, 339,
   366, 356, 396, 393, 360, 390, 368, 378, 388, 384, 381, 333, 322, 392, 360, 387, 350, 339, 328, 398, 383, 384, 373, 343, 334, 393,
   377, 340, 354, 368, 397, 389, 366, 395, 362, 388, 379, 385, 323, 324, 389, 329, 331, 357, 397, 355, 395, 335, 374, 336, 392, 375,
   393, 341, 391, 390, 398, 363, 361, 380, 382, 369, 396, 367, 394, 386, 325, 390, 326, 358, 396, 330, 332, 393, 353, 376, 337, 394,
   338, 395, 377, 391, 342, 389, 364, 398, 359, 370, 397, 381, 383, 392, 365, 387, 391, 327, 328, 394, 356, 392, 354, 333, 334, 396,
   397, 378, 379, 339, 340, 390, 389, 343, 362, 360, 398, 395, 368, 393, 366, 384, 385, 388, 359, 360, 344, 381, 384, 370, 352, 368,
   350, 392, 365, 393, 353, 366, 354, 398, 364, 362, 373, 388, 387, 383, 385, 397, 379, 395, 377, 389, 361, 345, 362, 369, 351, 380,
   385, 348, 366, 367, 394, 355, 395, 356, 368, 363, 398, 360, 388, 372, 386, 396, 378, 382, 384, 375, 393, 390, 346, 363, 364, 349,
   367, 347, 365, 382, 383, 357, 358, 369, 370, 396, 397, 361, 359, 398, 387, 386, 371, 376, 394, 374, 392, 380, 381, 391, 347, 375,
   353, 371, 382, 358, 364, 390, 377, 374, 359, 386, 365, 393, 389, 380, 370, 395, 383, 396, 376, 381, 398, 391, 397, 394, 387, 392,
   374, 348, 354, 380, 372, 391, 379, 356, 362, 386, 392, 375, 389, 360, 366, 382, 397, 368, 385, 378, 394, 398, 384, 396, 388, 390,
   395, 393, 349, 355, 377, 357, 363, 371, 383, 375, 389, 361, 376, 367, 387, 390, 395, 369, 381, 393, 397, 382, 374, 391, 396, 380,
   398, 386, 392, 394, 376, 356, 350, 391, 378, 381, 373, 360, 354, 394, 387, 390, 377, 368, 362, 396, 383, 366, 379, 384, 392, 397,
   388, 398, 385, 393, 389, 395, 357, 351, 379, 361, 355, 374, 389, 372, 385, 369, 391, 363, 397, 378, 388, 367, 392, 384, 395, 375,
   380, 394, 390, 386, 393, 382, 398, 396, 358, 378, 352, 376, 390, 359, 353, 384, 373, 391, 370, 396, 364, 388, 379, 394, 365, 385,
   377, 393, 381, 387, 395, 392, 389, 398, 383, 397, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386,
   387, 388, 389, 390, 391, 392, 393, 394, 395, 396, 397, 398
}; // 4.59 kb
static const cart_index_t iCartXY_ab13_a0[560] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233,
   234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259,
   260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285,
   286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296, 297, 298, 299, 300, 301, 302, 303, 304, 305, 306, 307, 308, 309, 310, 311,
   312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 330, 331, 332, 333, 334, 335, 336, 337,
   338, 339, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 358, 359, 360, 361, 362, 363,
   364, 365, 366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386, 387, 388, 389,
   390, 391, 392, 393, 394, 395, 396, 397, 398, 399, 400, 401, 402, 403, 404, 405, 406, 407, 408, 409, 410, 411, 412, 413, 414, 415,
   416, 417, 418, 419, 420, 421, 422, 423, 424, 425, 426, 427, 428, 429, 430, 431, 432, 433, 434, 435, 436, 437, 438, 439, 440, 441,
   442, 443, 444, 445, 446, 447, 448, 449, 450, 451, 452, 453, 454, 455, 456, 457, 458, 459, 460, 461, 462, 463, 464, 465, 466, 467,
   468, 469, 470, 471, 472, 473, 474, 475, 476, 477, 478, 479, 480, 481, 482, 483, 484, 485, 486, 487, 488, 489, 490, 491, 492, 493,
   494, 495, 496, 497, 498, 499, 500, 501, 502, 503, 504, 505, 506, 507, 508, 509, 510, 511, 512, 513, 514, 515, 516, 517, 518, 519,
   520, 521, 522, 523, 524, 525, 526, 527, 528, 529, 530, 531, 532, 533, 534, 535, 536, 537, 538, 539, 540, 541, 542, 543, 544, 545,
   546, 547, 548, 549, 550, 551, 552, 553, 554, 555, 556, 557, 558, 559
}; // 1.09 kb
static const cart_index_t iCartXY_ab13_a1[1365] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215, 219, 222, 224, 223, 220, 226, 225, 227, 221, 230, 223, 250, 231, 249, 225, 222, 228, 251, 249, 233, 227, 224, 251, 229, 250,
   226, 232, 228, 234, 254, 229, 252, 236, 235, 230, 256, 253, 232, 238, 237, 257, 231, 255, 239, 233, 242, 235, 260, 243, 258, 237,
   234, 240, 262, 259, 245, 239, 236, 263, 241, 261, 238, 244, 240, 246, 266, 241, 264, 247, 246, 242, 268, 265, 244, 248, 247, 269,
   243, 267, 248, 245, 251, 254, 252, 256, 250, 253, 257, 255, 249, 271, 253, 261, 272, 259, 255, 252, 270, 263, 258, 272, 257, 254,
   262, 270, 260, 256, 271, 262, 266, 273, 263, 275, 264, 268, 260, 274, 269, 276, 258, 277, 261, 265, 278, 267, 259, 281, 265, 267,
   264, 280, 269, 266, 268, 279, 270, 273, 275, 274, 271, 277, 276, 278, 272, 279, 274, 283, 280, 282, 276, 273, 279, 284, 282, 281,
   278, 275, 284, 280, 283, 277, 281, 284, 283, 282, 285, 290, 292, 288, 286, 293, 289, 291, 287, 290, 294, 318, 296, 288, 319, 292,
   318, 295, 298, 320, 289, 319, 293, 297, 320, 299, 291, 294, 302, 325, 295, 323, 304, 300, 296, 326, 301, 324, 298, 321, 297, 305,
   322, 303, 299, 302, 306, 327, 308, 300, 329, 304, 328, 307, 310, 330, 301, 331, 305, 309, 332, 311, 303, 306, 314, 337, 307, 335,
   316, 312, 308, 338, 313, 336, 310, 333, 309, 317, 334, 315, 311, 314, 312, 339, 316, 340, 313, 341, 317, 315, 324, 322, 320, 326,
   319, 321, 318, 325, 323, 323, 342, 328, 343, 321, 331, 325, 327, 342, 344, 332, 322, 329, 326, 343, 330, 344, 324, 336, 346, 330,
   348, 334, 332, 338, 329, 345, 350, 331, 333, 327, 337, 347, 328, 349, 335, 335, 352, 340, 354, 333, 341, 337, 339, 351, 356, 341,
   334, 339, 338, 353, 340, 355, 336, 342, 347, 349, 345, 343, 350, 346, 348, 344, 347, 351, 357, 353, 345, 358, 349, 357, 352, 355,
   359, 346, 358, 350, 354, 359, 356, 348, 351, 353, 362, 352, 361, 355, 360, 354, 356, 361, 360, 359, 362, 358, 360, 357, 362, 361,
   363, 366, 368, 367, 364, 370, 369, 371, 365, 374, 367, 400, 375, 399, 369, 366, 372, 401, 399, 377, 371, 368, 401, 373, 400, 370,
   376, 372, 378, 404, 373, 402, 380, 379, 374, 406, 403, 376, 382, 381, 407, 375, 405, 383, 377, 386, 379, 410, 387, 408, 381, 378,
   384, 412, 409, 389, 383, 380, 413, 385, 411, 382, 388, 384, 390, 416, 385, 414, 392, 391, 386, 418, 415, 388, 394, 393, 419, 387,
   417, 395, 389, 396, 391, 422, 397, 420, 393, 390, 396, 424, 421, 398, 395, 392, 425, 397, 423, 394, 398, 401, 404, 402, 406, 400,
   403, 407, 405, 399, 427, 403, 411, 428, 409, 405, 402, 426, 413, 408, 428, 407, 404, 412, 426, 410, 406, 427, 412, 416, 429, 413,
   431, 414, 418, 410, 430, 419, 432, 408, 433, 411, 415, 434, 417, 409, 439, 415, 423, 440, 421, 417, 414, 436, 425, 420, 438, 419,
   416, 424, 435, 422, 418, 437, 424, 422, 441, 425, 442, 420, 443, 423, 421, 426, 429, 431, 430, 427, 433, 432, 434, 428, 437, 430,
   445, 438, 444, 432, 429, 435, 446, 444, 440, 434, 431, 446, 436, 445, 433, 439, 435, 441, 449, 436, 447, 442, 441, 437, 451, 448,
   439, 443, 442, 452, 438, 450, 443, 440, 446, 449, 447, 451, 445, 448, 452, 450, 444, 453, 448, 450, 447, 453, 452, 449, 451, 453,
   454, 459, 461, 457, 455, 462, 458, 460, 456, 459, 463, 493, 465, 457, 494, 461, 493, 464, 467, 495, 458, 494, 462, 466, 495, 468,
   460, 463, 471, 500, 464, 498, 473, 469, 465, 501, 470, 499, 467, 496, 466, 474, 497, 472, 468, 471, 475, 502, 477, 469, 504, 473,
   503, 476, 479, 505, 470, 506, 474, 478, 507, 480, 472, 475, 483, 512, 476, 510, 485, 481, 477, 513, 482, 511, 479, 508, 478, 486,
   509, 484, 480, 483, 487, 514, 489, 481, 516, 485, 515, 488, 491, 517, 482, 518, 486, 490, 519, 492, 484, 487, 489, 522, 488, 521,
   491, 520, 490, 492, 499, 497, 495, 501, 494, 496, 493, 500, 498, 498, 523, 503, 524, 496, 506, 500, 502, 523, 525, 507, 497, 504,
   501, 524, 505, 525, 499, 511, 527, 505, 529, 509, 507, 513, 504, 526, 531, 506, 508, 502, 512, 528, 503, 530, 510, 510, 533, 515,
   535, 508, 518, 512, 514, 532, 537, 519, 509, 516, 513, 534, 517, 536, 511, 521, 539, 517, 541, 520, 519, 522, 516, 538, 543, 518,
   520, 514, 522, 540, 515, 542, 521, 523, 528, 530, 526, 524, 531, 527, 529, 525, 528, 532, 544, 534, 526, 545, 530, 544, 533, 536,
   546, 527, 545, 531, 535, 546, 537, 529, 532, 540, 551, 533, 549, 542, 538, 534, 552, 539, 550, 536, 547, 535, 543, 548, 541, 537,
   540, 538, 553, 542, 554, 539, 555, 543, 541, 550, 548, 546, 552, 545, 547, 544, 551, 549, 549, 556, 554, 557, 547, 555, 551, 553,
   556, 558, 555, 548, 553, 552, 557, 554, 558, 550, 556, 557, 558
}; // 2.67 kb
static const cart_index_t iCartXY_ab13_a2[2184] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212, 216, 225, 226, 219, 221, 248, 227, 217,
   229, 220, 247, 223, 228, 230, 218, 246, 222, 224, 219, 231, 249, 225, 248, 251, 232, 220, 250, 227, 253, 247, 221, 251, 233, 248,
   226, 249, 234, 252, 222, 254, 228, 246, 253, 223, 235, 247, 250, 229, 254, 236, 224, 252, 246, 230, 225, 237, 267, 231, 251, 259,
   226, 267, 238, 249, 233, 260, 239, 227, 268, 232, 257, 253, 240, 269, 228, 255, 234, 254, 268, 229, 241, 250, 258, 235, 269, 242,
   230, 256, 252, 236, 231, 243, 270, 237, 259, 263, 243, 232, 271, 239, 265, 257, 233, 272, 244, 260, 238, 261, 244, 273, 234, 266,
   240, 255, 274, 235, 245, 258, 262, 241, 275, 245, 236, 264, 256, 242, 237, 239, 276, 243, 263, 265, 238, 277, 240, 261, 244, 266,
   278, 241, 242, 262, 264, 245, 255, 256, 246, 269, 254, 252, 257, 247, 258, 253, 268, 250, 248, 259, 260, 251, 249, 267, 249, 270,
   261, 267, 260, 272, 271, 250, 262, 268, 274, 258, 251, 263, 272, 259, 267, 270, 273, 264, 252, 275, 269, 256, 265, 253, 274, 257,
   271, 268, 266, 275, 254, 273, 255, 269, 261, 279, 255, 277, 266, 273, 279, 262, 256, 278, 275, 264, 263, 257, 280, 265, 276, 271,
   280, 258, 264, 274, 278, 262, 259, 265, 281, 263, 270, 276, 260, 281, 266, 272, 261, 277, 267, 276, 277, 270, 272, 281, 276, 268,
   278, 271, 280, 274, 277, 278, 269, 279, 273, 275, 270, 271, 279, 276, 281, 280, 272, 280, 273, 281, 277, 279, 281, 274, 275, 280,
   279, 278, 282, 291, 292, 287, 289, 315, 293, 283, 294, 285, 316, 290, 295, 296, 284, 317, 286, 288, 297, 285, 318, 293, 323, 316,
   298, 319, 286, 321, 295, 317, 287, 299, 320, 291, 315, 322, 321, 300, 288, 319, 317, 296, 289, 322, 301, 315, 292, 320, 323, 290,
   302, 316, 318, 294, 291, 303, 339, 299, 322, 324, 292, 339, 304, 320, 301, 325, 305, 293, 340, 297, 326, 323, 340, 294, 306, 318,
   328, 302, 307, 341, 295, 327, 298, 321, 341, 308, 296, 329, 319, 300, 309, 297, 342, 305, 335, 326, 310, 343, 298, 333, 307, 327,
   299, 311, 344, 303, 324, 334, 345, 312, 300, 331, 329, 308, 301, 346, 313, 325, 304, 332, 347, 302, 314, 328, 330, 306, 303, 309,
   348, 311, 334, 336, 304, 349, 310, 332, 313, 337, 311, 305, 350, 309, 336, 335, 351, 306, 312, 330, 338, 314, 313, 352, 307, 337,
   310, 333, 353, 314, 308, 338, 331, 312, 315, 324, 325, 322, 320, 339, 326, 316, 328, 323, 340, 318, 327, 329, 317, 341, 321, 319,
   342, 318, 330, 340, 347, 328, 343, 331, 319, 345, 341, 329, 320, 344, 332, 339, 325, 346, 333, 345, 321, 343, 327, 341, 322, 334,
   346, 324, 339, 344, 335, 323, 347, 326, 342, 340, 324, 336, 354, 334, 344, 348, 325, 354, 337, 346, 332, 349, 336, 326, 355, 335,
   350, 342, 337, 356, 327, 352, 333, 343, 355, 328, 338, 347, 351, 330, 356, 338, 329, 353, 345, 331, 357, 330, 331, 351, 353, 338,
   332, 358, 333, 349, 337, 352, 334, 335, 359, 336, 348, 350, 339, 348, 349, 344, 346, 354, 350, 340, 351, 342, 355, 347, 352, 353,
   341, 356, 343, 345, 348, 342, 357, 350, 359, 355, 349, 357, 343, 358, 352, 356, 344, 350, 358, 348, 354, 359, 358, 351, 345, 357,
   356, 353, 346, 359, 352, 354, 349, 358, 359, 347, 353, 355, 357, 351, 354, 355, 356, 359, 358, 357, 360, 369, 370, 363, 365, 398,
   371, 361, 373, 364, 397, 367, 372, 374, 362, 396, 366, 368, 363, 375, 399, 369, 398, 401, 376, 364, 400, 371, 403, 397, 365, 401,
   377, 398, 370, 399, 378, 402, 366, 404, 372, 396, 403, 367, 379, 397, 400, 373, 404, 380, 368, 402, 396, 374, 369, 381, 423, 375,
   401, 409, 370, 423, 382, 399, 377, 410, 383, 371, 424, 376, 407, 403, 384, 425, 372, 405, 378, 404, 424, 373, 385, 400, 408, 379,
   425, 386, 374, 406, 402, 380, 375, 387, 426, 381, 409, 413, 388, 376, 427, 383, 415, 407, 377, 428, 389, 410, 382, 411, 390, 429,
   378, 416, 384, 405, 430, 379, 391, 408, 412, 385, 431, 392, 380, 414, 406, 386, 381, 393, 432, 387, 413, 421, 382, 433, 394, 411,
   389, 422, 393, 383, 434, 388, 419, 415, 394, 435, 384, 417, 390, 416, 436, 385, 395, 412, 420, 391, 437, 395, 386, 418, 414, 392,
   387, 388, 438, 393, 421, 419, 389, 439, 390, 422, 394, 417, 440, 391, 392, 420, 418, 395, 405, 406, 396, 425, 404, 402, 407, 397,
   408, 403, 424, 400, 398, 409, 410, 401, 399, 423, 399, 426, 411, 423, 410, 428, 427, 400, 412, 424, 430, 408, 401, 413, 428, 409,
   423, 426, 429, 414, 402, 431, 425, 406, 415, 403, 430, 407, 427, 424, 416, 431, 404, 429, 405, 425, 417, 441, 405, 435, 416, 429,
   441, 418, 406, 437, 431, 414, 419, 407, 442, 415, 434, 427, 442, 408, 420, 430, 436, 412, 409, 421, 443, 413, 426, 432, 410, 443,
   422, 428, 411, 433, 411, 444, 417, 433, 422, 439, 445, 412, 418, 436, 440, 420, 413, 419, 446, 421, 432, 438, 447, 420, 414, 440,
   437, 418, 421, 415, 448, 419, 438, 434, 422, 449, 416, 439, 417, 435, 423, 432, 433, 426, 428, 443, 434, 424, 436, 427, 442, 430,
   435, 437, 425, 441, 429, 431, 426, 438, 444, 432, 443, 446, 438, 427, 445, 434, 448, 442, 428, 446, 439, 443, 433, 444, 439, 447,
   429, 449, 435, 441, 448, 430, 440, 442, 445, 436, 449, 440, 431, 447, 441, 437, 432, 434, 450, 438, 446, 448, 433, 450, 435, 444,
   439, 449, 450, 436, 437, 445, 447, 440, 444, 445, 441, 450, 449, 447, 446, 442, 447, 448, 450, 445, 443, 448, 449, 446, 444, 450,
   451, 460, 461, 456, 458, 490, 462, 452, 463, 454, 491, 459, 464, 465, 453, 492, 455, 457, 466, 454, 493, 462, 498, 491, 467, 494,
   455, 496, 464, 492, 456, 468, 495, 460, 490, 497, 496, 469, 457, 494, 492, 465, 458, 497, 470, 490, 461, 495, 498, 459, 471, 491,
   493, 463, 460, 472, 520, 468, 497, 499, 461, 520, 473, 495, 470, 500, 474, 462, 521, 466, 501, 498, 521, 463, 475, 493, 503, 471,
   476, 522, 464, 502, 467, 496, 522, 477, 465, 504, 494, 469, 478, 466, 523, 474, 510, 501, 479, 524, 467, 508, 476, 502, 468, 480,
   525, 472, 499, 509, 526, 481, 469, 506, 504, 477, 470, 527, 482, 500, 473, 507, 528, 471, 483, 503, 505, 475, 472, 484, 529, 480,
   509, 511, 473, 530, 485, 507, 482, 512, 486, 474, 531, 478, 513, 510, 532, 475, 487, 505, 515, 483, 488, 533, 476, 514, 479, 508,
   534, 489, 477, 516, 506, 481, 484, 478, 535, 486, 519, 513, 485, 536, 479, 518, 488, 514, 480, 486, 537, 484, 511, 519, 538, 487,
   481, 517, 516, 489, 482, 539, 488, 512, 485, 518, 540, 483, 489, 515, 517, 487, 490, 499, 500, 497, 495, 520, 501, 491, 503, 498,
   521, 493, 502, 504, 492, 522, 496, 494, 523, 493, 505, 521, 528, 503, 524, 506, 494, 526, 522, 504, 495, 525, 507, 520, 500, 527,
   508, 526, 496, 524, 502, 522, 497, 509, 527, 499, 520, 525, 510, 498, 528, 501, 523, 521, 499, 511, 541, 509, 525, 529, 500, 541,
   512, 527, 507, 530, 513, 501, 542, 510, 531, 523, 514, 543, 502, 533, 508, 524, 542, 503, 515, 528, 532, 505, 543, 516, 504, 534,
   526, 506, 544, 505, 517, 532, 540, 515, 545, 517, 506, 538, 534, 516, 507, 546, 518, 530, 512, 539, 518, 547, 508, 536, 514, 533,
   509, 519, 548, 511, 529, 537, 519, 510, 549, 513, 535, 531, 511, 513, 550, 519, 537, 535, 512, 551, 514, 539, 518, 536, 552, 515,
   516, 540, 538, 517, 520, 529, 530, 525, 527, 541, 531, 521, 532, 523, 542, 528, 533, 534, 522, 543, 524, 526, 535, 523, 544, 531,
   549, 542, 536, 545, 524, 547, 533, 543, 525, 537, 546, 529, 541, 548, 547, 538, 526, 545, 543, 534, 527, 548, 539, 541, 530, 546,
   549, 528, 540, 542, 544, 532, 529, 535, 553, 537, 548, 550, 530, 553, 536, 546, 539, 551, 537, 531, 554, 535, 550, 549, 554, 532,
   538, 544, 552, 540, 539, 555, 533, 551, 536, 547, 555, 540, 534, 552, 545, 538, 541, 550, 551, 548, 546, 553, 550, 542, 552, 549,
   554, 544, 551, 552, 543, 555, 547, 545, 553, 544, 545, 554, 555, 552, 546, 554, 547, 553, 551, 555, 548, 549, 555, 550, 553, 554
}; // 4.27 kb
static const cart_index_t iCartXY_ab13_a3[2860] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209, 210, 225, 227, 219, 220, 213, 243, 215, 245, 242, 226, 211, 229, 214, 244, 221, 223, 247,
   217, 241, 228, 230, 212, 246, 216, 248, 218, 222, 224, 240, 233, 214, 252, 221, 262, 226, 244, 251, 241, 247, 234, 250, 216, 263,
   222, 249, 240, 228, 246, 248, 213, 231, 254, 225, 243, 219, 261, 242, 253, 245, 249, 236, 218, 250, 240, 263, 224, 248, 230, 246,
   215, 253, 232, 245, 227, 242, 254, 220, 261, 243, 251, 217, 235, 241, 252, 247, 229, 262, 223, 244, 219, 237, 266, 231, 261, 225,
   264, 245, 257, 253, 220, 264, 238, 261, 232, 243, 255, 227, 266, 254, 237, 221, 268, 226, 265, 233, 262, 259, 247, 251, 265, 223,
   239, 244, 256, 262, 235, 268, 229, 252, 238, 269, 222, 267, 228, 260, 248, 234, 263, 249, 267, 239, 224, 258, 246, 269, 230, 263,
   236, 250, 231, 226, 274, 233, 270, 237, 265, 257, 251, 259, 232, 273, 228, 271, 234, 255, 249, 238, 267, 260, 225, 233, 275, 237,
   264, 231, 270, 253, 259, 257, 273, 235, 230, 256, 250, 272, 236, 269, 239, 258, 227, 275, 234, 266, 238, 254, 260, 232, 271, 255,
   274, 229, 236, 252, 258, 268, 239, 272, 235, 256, 242, 257, 255, 253, 254, 245, 266, 243, 264, 261, 259, 241, 256, 247, 268, 251,
   252, 265, 244, 262, 260, 258, 240, 269, 248, 267, 246, 249, 250, 263, 270, 244, 258, 262, 272, 265, 256, 274, 252, 268, 271, 256,
   246, 272, 263, 273, 250, 267, 258, 269, 243, 270, 260, 264, 255, 261, 271, 254, 275, 266, 255, 272, 248, 273, 249, 271, 263, 260,
   269, 267, 245, 259, 271, 257, 266, 253, 275, 261, 270, 264, 257, 247, 272, 251, 274, 259, 268, 270, 262, 265, 253, 251, 273, 259,
   275, 257, 274, 264, 265, 270, 254, 274, 249, 275, 260, 266, 267, 255, 273, 271, 275, 252, 250, 268, 269, 274, 258, 273, 256, 272,
   261, 265, 267, 270, 271, 264, 273, 266, 274, 275, 264, 262, 269, 265, 273, 270, 272, 275, 268, 274, 266, 268, 263, 274, 267, 275,
   269, 271, 272, 273, 276, 293, 295, 285, 286, 281, 314, 283, 316, 309, 291, 277, 296, 279, 312, 287, 288, 317, 284, 310, 292, 294,
   278, 313, 280, 315, 282, 289, 290, 311, 281, 297, 319, 293, 314, 285, 333, 309, 318, 316, 299, 279, 322, 287, 334, 291, 312, 320,
   310, 317, 283, 318, 298, 316, 295, 309, 319, 286, 333, 314, 301, 323, 280, 335, 289, 321, 311, 292, 313, 315, 320, 284, 300, 310,
   322, 317, 296, 334, 288, 312, 321, 302, 282, 323, 311, 335, 290, 315, 294, 313, 285, 305, 340, 297, 333, 293, 338, 316, 328, 318,
   286, 338, 307, 333, 298, 314, 326, 295, 340, 319, 303, 287, 341, 291, 336, 299, 334, 329, 317, 320, 304, 339, 289, 337, 292, 327,
   315, 301, 335, 321, 336, 288, 308, 312, 324, 334, 300, 341, 296, 322, 337, 306, 290, 325, 313, 339, 294, 335, 302, 323, 293, 303,
   348, 305, 338, 297, 342, 318, 330, 328, 305, 291, 349, 299, 344, 303, 336, 330, 320, 329, 295, 348, 304, 340, 307, 319, 331, 298,
   343, 326, 307, 350, 292, 346, 301, 331, 321, 304, 337, 327, 349, 296, 306, 322, 332, 341, 308, 345, 300, 324, 350, 308, 294, 332,
   323, 347, 302, 339, 306, 325, 297, 299, 353, 303, 342, 305, 344, 328, 329, 330, 298, 352, 301, 343, 304, 326, 327, 307, 346, 331,
   351, 300, 302, 324, 325, 345, 306, 347, 308, 332, 327, 325, 311, 339, 315, 337, 313, 321, 323, 335, 329, 310, 324, 317, 341, 320,
   322, 336, 312, 334, 309, 328, 326, 318, 319, 316, 340, 314, 338, 333, 314, 342, 331, 338, 326, 333, 343, 319, 348, 340, 344, 312,
   332, 334, 345, 336, 324, 349, 322, 341, 316, 330, 343, 328, 340, 318, 348, 333, 342, 338, 346, 332, 313, 347, 335, 350, 323, 337,
   325, 339, 330, 317, 345, 320, 349, 329, 341, 344, 334, 336, 331, 347, 315, 350, 321, 346, 335, 327, 339, 337, 326, 351, 321, 352,
   327, 343, 337, 331, 350, 346, 352, 324, 323, 345, 339, 351, 325, 350, 332, 347, 328, 320, 351, 329, 353, 330, 349, 342, 336, 344,
   353, 322, 325, 341, 347, 349, 332, 351, 324, 345, 318, 329, 352, 330, 348, 328, 353, 338, 344, 342, 319, 353, 327, 348, 331, 340,
   346, 326, 352, 343, 333, 344, 346, 342, 343, 338, 352, 340, 353, 348, 342, 334, 347, 336, 351, 344, 345, 353, 341, 349, 343, 345,
   335, 351, 337, 352, 339, 346, 347, 350, 338, 336, 350, 344, 352, 342, 351, 348, 349, 353, 340, 349, 337, 353, 346, 348, 350, 343,
   351, 352, 348, 341, 339, 349, 350, 353, 347, 352, 345, 351, 354, 369, 371, 363, 364, 357, 393, 359, 395, 392, 370, 355, 373, 358,
   394, 365, 367, 397, 361, 391, 372, 374, 356, 396, 360, 398, 362, 366, 368, 390, 377, 358, 402, 365, 418, 370, 394, 401, 391, 397,
   378, 400, 360, 419, 366, 399, 390, 372, 396, 398, 357, 375, 404, 369, 393, 363, 417, 392, 403, 395, 399, 380, 362, 400, 390, 419,
   368, 398, 374, 396, 359, 403, 376, 395, 371, 392, 404, 364, 417, 393, 401, 361, 379, 391, 402, 397, 373, 418, 367, 394, 363, 381,
   422, 375, 417, 369, 420, 395, 407, 403, 364, 420, 383, 417, 376, 393, 405, 371, 422, 404, 382, 365, 424, 370, 421, 377, 418, 409,
   397, 401, 421, 367, 385, 394, 406, 418, 379, 424, 373, 402, 384, 425, 366, 423, 372, 410, 398, 378, 419, 399, 423, 386, 368, 408,
   396, 425, 374, 419, 380, 400, 387, 370, 436, 377, 428, 382, 421, 413, 401, 409, 388, 435, 372, 429, 378, 411, 399, 384, 423, 410,
   369, 387, 437, 381, 420, 375, 426, 403, 415, 407, 435, 389, 374, 412, 400, 431, 380, 425, 386, 408, 371, 437, 388, 422, 383, 404,
   416, 376, 427, 405, 436, 373, 389, 402, 414, 424, 385, 430, 379, 406, 375, 382, 440, 387, 426, 381, 432, 407, 413, 415, 376, 438,
   384, 427, 388, 405, 411, 383, 433, 416, 381, 377, 442, 382, 432, 387, 428, 415, 409, 413, 439, 379, 386, 406, 412, 430, 389, 434,
   385, 414, 383, 443, 378, 433, 384, 416, 410, 388, 429, 411, 441, 385, 380, 414, 408, 434, 386, 431, 389, 412, 392, 407, 405, 403,
   404, 395, 422, 393, 420, 417, 409, 391, 406, 397, 424, 401, 402, 421, 394, 418, 410, 408, 390, 425, 398, 423, 396, 399, 400, 419,
   428, 394, 414, 418, 430, 421, 406, 436, 402, 424, 429, 412, 396, 431, 419, 435, 400, 423, 408, 425, 393, 426, 416, 420, 405, 417,
   427, 404, 437, 422, 411, 431, 398, 435, 399, 429, 419, 410, 425, 423, 395, 415, 427, 407, 422, 403, 437, 417, 426, 420, 413, 397,
   430, 401, 436, 409, 424, 428, 418, 421, 403, 413, 438, 415, 437, 407, 440, 420, 432, 426, 404, 440, 411, 437, 416, 422, 433, 405,
   438, 427, 415, 401, 439, 409, 442, 413, 436, 432, 421, 428, 416, 441, 399, 443, 410, 433, 423, 411, 435, 429, 442, 402, 412, 424,
   434, 436, 414, 439, 406, 430, 443, 414, 400, 434, 425, 441, 408, 435, 412, 431, 444, 406, 408, 430, 431, 439, 412, 441, 414, 434,
   405, 444, 410, 438, 411, 427, 429, 416, 443, 433, 407, 409, 444, 413, 440, 415, 442, 426, 428, 432, 417, 432, 433, 426, 427, 420,
   438, 422, 440, 437, 432, 418, 434, 421, 439, 428, 430, 442, 424, 436, 433, 434, 419, 441, 423, 443, 425, 429, 431, 435, 426, 421,
   441, 428, 444, 432, 439, 440, 436, 442, 427, 439, 423, 444, 429, 438, 435, 433, 441, 443, 420, 428, 443, 432, 438, 426, 444, 437,
   442, 440, 438, 430, 425, 439, 435, 444, 431, 443, 434, 441, 422, 442, 429, 440, 433, 437, 443, 427, 444, 438, 440, 424, 431, 436,
   441, 442, 434, 444, 430, 439, 437, 436, 435, 442, 443, 440, 441, 438, 439, 444, 445, 462, 464, 454, 455, 450, 489, 452, 491, 484,
   460, 446, 465, 448, 487, 456, 457, 492, 453, 485, 461, 463, 447, 488, 449, 490, 451, 458, 459, 486, 450, 466, 494, 462, 489, 454,
   514, 484, 493, 491, 468, 448, 497, 456, 515, 460, 487, 495, 485, 492, 452, 493, 467, 491, 464, 484, 494, 455, 514, 489, 470, 498,
   449, 516, 458, 496, 486, 461, 488, 490, 495, 453, 469, 485, 497, 492, 465, 515, 457, 487, 496, 471, 451, 498, 486, 516, 459, 490,
   463, 488, 454, 474, 521, 466, 514, 462, 519, 491, 503, 493, 455, 519, 476, 514, 467, 489, 501, 464, 521, 494, 472, 456, 522, 460,
   517, 468, 515, 504, 492, 495, 473, 520, 458, 518, 461, 502, 490, 470, 516, 496, 517, 457, 477, 487, 499, 515, 469, 522, 465, 497,
   518, 475, 459, 500, 488, 520, 463, 516, 471, 498, 462, 478, 535, 474, 519, 466, 523, 493, 505, 503, 480, 460, 536, 468, 525, 472,
   517, 507, 495, 504, 464, 535, 479, 521, 476, 494, 506, 467, 524, 501, 482, 537, 461, 527, 470, 508, 496, 473, 518, 502, 536, 465,
   481, 497, 509, 522, 477, 526, 469, 499, 537, 483, 463, 510, 498, 528, 471, 520, 475, 500, 466, 480, 542, 478, 523, 474, 531, 503,
   513, 505, 467, 540, 482, 524, 479, 501, 512, 476, 533, 506, 478, 468, 543, 472, 529, 480, 525, 513, 504, 507, 479, 541, 470, 530,
   473, 512, 502, 482, 527, 508, 538, 469, 483, 499, 511, 526, 481, 534, 477, 509, 539, 481, 471, 511, 500, 532, 475, 528, 483, 510,
   474, 472, 544, 480, 531, 478, 529, 505, 507, 513, 476, 545, 473, 533, 482, 506, 508, 479, 530, 512, 546, 477, 475, 509, 510, 534,
   483, 532, 481, 511, 502, 500, 486, 520, 490, 518, 488, 496, 498, 516, 504, 485, 499, 492, 522, 495, 497, 517, 487, 515, 484, 503,
   501, 493, 494, 491, 521, 489, 519, 514, 489, 523, 506, 519, 501, 514, 524, 494, 535, 521, 525, 487, 509, 515, 526, 517, 499, 536,
   497, 522, 491, 505, 524, 503, 521, 493, 535, 514, 523, 519, 527, 510, 488, 528, 516, 537, 498, 518, 500, 520, 507, 492, 526, 495,
   536, 504, 522, 525, 515, 517, 508, 528, 490, 537, 496, 527, 516, 502, 520, 518, 512, 539, 496, 541, 502, 530, 518, 508, 537, 527,
   541, 511, 498, 532, 520, 539, 500, 537, 510, 528, 513, 495, 538, 504, 543, 507, 536, 529, 517, 525, 543, 497, 511, 522, 534, 536,
   509, 538, 499, 526, 493, 513, 540, 505, 535, 503, 542, 519, 531, 523, 494, 542, 512, 535, 506, 521, 533, 501, 540, 524, 501, 547,
   508, 540, 512, 524, 530, 506, 545, 533, 548, 499, 510, 526, 532, 538, 511, 546, 509, 534, 503, 507, 547, 513, 542, 505, 544, 523,
   529, 531, 549, 509, 500, 534, 528, 546, 510, 539, 511, 532, 505, 504, 548, 507, 544, 513, 543, 531, 525, 529, 506, 549, 502, 545,
   508, 533, 527, 512, 541, 530, 514, 531, 533, 523, 524, 519, 540, 521, 542, 535, 529, 515, 534, 517, 538, 525, 526, 543, 522, 536,
   530, 532, 516, 539, 518, 541, 520, 527, 528, 537, 519, 529, 545, 531, 540, 523, 547, 535, 544, 542, 531, 517, 546, 525, 548, 529,
   538, 544, 536, 543, 521, 544, 530, 542, 533, 535, 545, 524, 547, 540, 533, 546, 518, 549, 527, 545, 537, 530, 539, 541, 544, 522,
   532, 536, 546, 543, 534, 548, 526, 538, 545, 534, 520, 546, 537, 549, 528, 541, 532, 539, 523, 525, 549, 529, 547, 531, 548, 542,
   543, 544, 524, 548, 527, 547, 530, 540, 541, 533, 549, 545, 547, 526, 528, 538, 539, 548, 532, 549, 534, 546, 540, 538, 537, 548,
   541, 547, 539, 545, 546, 549, 542, 536, 539, 543, 549, 544, 546, 547, 538, 548, 535, 543, 541, 544, 545, 542, 549, 540, 548, 547
}; // 5.59 kb
static const cart_index_t iCartXY_ab13_a4[3300] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195, 200, 221, 222, 203, 215, 205, 217, 243, 244, 209, 210, 251, 233, 235, 232, 223, 201, 225,
   216, 204, 241, 242, 207, 219, 211, 252, 213, 234, 231, 237, 224, 226, 202, 239, 240, 218, 206, 220, 208, 253, 212, 214, 230, 236,
   238, 203, 227, 245, 209, 221, 232, 244, 247, 256, 215, 233, 254, 251, 243, 235, 227, 204, 246, 223, 211, 249, 258, 231, 242, 216,
   255, 234, 252, 237, 241, 205, 247, 228, 232, 243, 210, 222, 254, 245, 235, 217, 256, 244, 251, 233, 228, 248, 206, 250, 259, 224,
   212, 240, 230, 257, 218, 236, 238, 253, 239, 249, 207, 229, 241, 231, 255, 246, 213, 225, 237, 258, 219, 242, 234, 252, 250, 229,
   208, 257, 248, 239, 230, 226, 214, 259, 238, 220, 236, 240, 253, 209, 223, 261, 215, 227, 235, 256, 249, 265, 221, 251, 260, 254,
   247, 243, 210, 260, 224, 233, 254, 217, 228, 265, 250, 251, 222, 261, 245, 256, 244, 221, 211, 262, 227, 216, 247, 264, 237, 258,
   223, 260, 252, 255, 241, 249, 222, 262, 212, 245, 263, 228, 218, 259, 238, 261, 224, 253, 239, 257, 250, 260, 213, 226, 255, 234,
   264, 248, 219, 229, 252, 262, 225, 246, 242, 258, 261, 225, 214, 263, 246, 257, 236, 229, 220, 262, 253, 226, 240, 248, 259, 215,
   216, 263, 221, 223, 243, 265, 241, 264, 227, 254, 255, 260, 249, 247, 217, 264, 218, 244, 265, 222, 224, 263, 239, 256, 228, 257,
   250, 261, 245, 265, 219, 220, 264, 242, 263, 240, 225, 226, 258, 259, 229, 248, 246, 262, 245, 246, 230, 261, 262, 250, 238, 248,
   236, 263, 239, 240, 253, 259, 257, 247, 231, 248, 249, 237, 260, 262, 234, 246, 241, 264, 242, 258, 252, 255, 232, 249, 250, 235,
   247, 233, 245, 260, 261, 243, 244, 265, 256, 254, 251, 233, 255, 239, 251, 260, 244, 250, 264, 257, 254, 245, 263, 261, 265, 256,
   254, 234, 240, 260, 252, 265, 259, 242, 248, 255, 263, 246, 262, 258, 264, 235, 241, 257, 243, 249, 251, 261, 255, 263, 247, 256,
   264, 265, 260, 254, 256, 242, 236, 265, 258, 261, 253, 246, 240, 264, 257, 248, 259, 262, 263, 243, 237, 259, 247, 241, 254, 263,
   252, 262, 249, 265, 258, 264, 255, 260, 244, 258, 238, 256, 264, 245, 239, 262, 253, 265, 250, 259, 257, 263, 261, 251, 252, 253,
   254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 287, 288, 271, 283, 273, 285, 308, 309, 275, 276, 323, 304, 306,
   299, 289, 267, 290, 281, 269, 310, 312, 274, 286, 277, 324, 278, 302, 300, 307, 291, 292, 268, 311, 313, 282, 270, 284, 272, 325,
   279, 280, 301, 303, 305, 293, 269, 314, 289, 277, 319, 331, 300, 312, 281, 326, 302, 324, 307, 310, 294, 315, 270, 317, 329, 291,
   279, 313, 301, 327, 282, 303, 305, 325, 311, 271, 295, 316, 275, 287, 299, 309, 318, 330, 283, 304, 328, 323, 308, 306, 317, 296,
   272, 327, 315, 311, 301, 292, 280, 329, 305, 284, 303, 313, 325, 273, 318, 297, 299, 308, 276, 288, 328, 316, 306, 285, 330, 309,
   323, 304, 319, 274, 298, 310, 300, 326, 314, 278, 290, 307, 331, 286, 312, 302, 324, 275, 293, 333, 283, 295, 306, 330, 320, 338,
   287, 323, 332, 328, 318, 308, 276, 332, 294, 304, 328, 285, 297, 338, 321, 323, 288, 333, 316, 330, 309, 295, 277, 335, 293, 281,
   320, 339, 307, 331, 289, 334, 324, 326, 310, 319, 334, 278, 296, 326, 302, 339, 322, 286, 298, 324, 335, 290, 314, 312, 331, 297,
   337, 279, 321, 340, 294, 282, 329, 305, 336, 291, 325, 311, 327, 317, 336, 298, 280, 340, 322, 327, 303, 296, 284, 337, 325, 292,
   313, 315, 329, 287, 281, 341, 295, 289, 318, 343, 310, 339, 293, 332, 326, 334, 319, 320, 288, 341, 282, 316, 342, 297, 291, 340,
   311, 333, 294, 327, 317, 336, 321, 283, 289, 342, 287, 293, 308, 338, 319, 343, 295, 328, 334, 332, 320, 318, 342, 290, 284, 341,
   314, 340, 313, 298, 292, 335, 329, 296, 315, 322, 337, 285, 343, 291, 309, 338, 288, 294, 342, 317, 330, 297, 336, 321, 333, 316,
   343, 286, 292, 339, 312, 341, 315, 290, 296, 331, 337, 298, 322, 314, 335, 299, 320, 321, 306, 318, 304, 316, 332, 333, 308, 309,
   338, 330, 328, 323, 320, 300, 322, 319, 307, 334, 335, 302, 314, 310, 339, 312, 331, 324, 326, 321, 322, 301, 336, 337, 317, 305,
   315, 303, 340, 311, 313, 325, 329, 327, 332, 302, 315, 334, 324, 343, 337, 312, 322, 326, 341, 314, 335, 331, 339, 333, 314, 303,
   342, 335, 336, 325, 322, 313, 341, 327, 315, 329, 337, 340, 304, 334, 317, 323, 332, 309, 321, 343, 336, 328, 316, 342, 333, 338,
   330, 316, 335, 305, 333, 341, 321, 311, 337, 325, 342, 317, 329, 327, 340, 336, 306, 319, 336, 308, 320, 323, 333, 334, 342, 318,
   330, 343, 338, 332, 328, 318, 307, 337, 320, 310, 332, 341, 324, 335, 319, 343, 331, 339, 326, 334, 308, 310, 340, 318, 319, 328,
   342, 326, 341, 320, 338, 339, 343, 334, 332, 309, 339, 311, 330, 343, 316, 317, 341, 327, 338, 321, 340, 336, 342, 333, 338, 312,
   313, 343, 331, 342, 329, 314, 315, 339, 340, 322, 337, 335, 341, 323, 326, 327, 328, 334, 330, 336, 339, 340, 332, 333, 341, 342,
   343, 338, 328, 324, 329, 332, 326, 338, 340, 331, 337, 334, 342, 335, 341, 339, 343, 330, 331, 325, 338, 339, 333, 327, 335, 329,
   343, 336, 337, 340, 341, 342, 344, 365, 366, 347, 359, 349, 361, 393, 394, 353, 354, 407, 383, 385, 382, 367, 345, 369, 360, 348,
   391, 392, 351, 363, 355, 408, 357, 384, 381, 387, 368, 370, 346, 389, 390, 362, 350, 364, 352, 409, 356, 358, 380, 386, 388, 347,
   371, 395, 353, 365, 382, 394, 397, 412, 359, 383, 410, 407, 393, 385, 372, 348, 396, 367, 355, 399, 414, 381, 392, 360, 411, 384,
   408, 387, 391, 349, 397, 373, 382, 393, 354, 366, 410, 395, 385, 361, 412, 394, 407, 383, 374, 398, 350, 400, 415, 368, 356, 390,
   380, 413, 362, 386, 388, 409, 389, 399, 351, 375, 391, 381, 411, 396, 357, 369, 387, 414, 363, 392, 384, 408, 400, 376, 352, 413,
   398, 389, 380, 370, 358, 415, 388, 364, 386, 390, 409, 353, 377, 417, 359, 371, 385, 412, 405, 427, 365, 407, 416, 410, 397, 393,
   354, 416, 378, 383, 410, 361, 373, 427, 406, 407, 366, 417, 395, 412, 394, 377, 355, 420, 372, 360, 403, 426, 387, 414, 367, 418,
   408, 411, 391, 399, 378, 421, 356, 401, 425, 374, 362, 415, 388, 419, 368, 409, 389, 413, 400, 418, 357, 379, 411, 384, 426, 404,
   363, 375, 408, 420, 369, 396, 392, 414, 419, 379, 358, 425, 402, 413, 386, 376, 364, 421, 409, 370, 390, 398, 415, 359, 372, 428,
   365, 377, 393, 427, 403, 430, 371, 410, 422, 416, 405, 397, 371, 360, 429, 377, 367, 405, 432, 391, 426, 372, 422, 411, 418, 399,
   403, 361, 430, 374, 394, 427, 366, 378, 428, 401, 412, 373, 423, 406, 417, 395, 373, 431, 362, 406, 433, 378, 368, 425, 389, 423,
   374, 413, 400, 419, 401, 432, 363, 376, 426, 392, 429, 402, 369, 379, 414, 424, 375, 404, 396, 420, 433, 375, 364, 431, 404, 425,
   390, 379, 370, 424, 415, 376, 398, 402, 421, 365, 367, 434, 371, 372, 397, 430, 399, 432, 377, 416, 418, 422, 403, 405, 366, 434,
   368, 395, 428, 373, 374, 433, 400, 417, 378, 419, 401, 423, 406, 434, 369, 370, 429, 396, 431, 398, 375, 376, 420, 421, 379, 402,
   404, 424, 401, 402, 380, 419, 421, 400, 388, 398, 386, 425, 389, 390, 409, 415, 413, 403, 381, 404, 399, 387, 418, 420, 384, 396,
   391, 426, 392, 414, 408, 411, 382, 405, 406, 385, 397, 383, 395, 416, 417, 393, 394, 427, 412, 410, 407, 383, 422, 401, 407, 416,
   394, 406, 430, 423, 410, 395, 428, 417, 427, 412, 422, 384, 402, 418, 408, 432, 424, 392, 404, 411, 429, 396, 420, 414, 426, 385,
   403, 423, 393, 405, 407, 417, 422, 428, 397, 412, 430, 427, 416, 410, 423, 404, 386, 433, 424, 419, 409, 402, 390, 431, 413, 398,
   415, 421, 425, 405, 387, 424, 403, 391, 422, 429, 408, 420, 399, 432, 414, 426, 411, 418, 406, 424, 388, 423, 431, 401, 389, 421,
   409, 433, 400, 415, 413, 425, 419, 395, 429, 389, 417, 434, 406, 400, 431, 413, 428, 401, 425, 419, 433, 423, 428, 396, 390, 434,
   420, 433, 415, 404, 398, 429, 425, 402, 421, 424, 431, 397, 391, 431, 405, 399, 416, 434, 411, 429, 403, 430, 426, 432, 418, 422,
   430, 392, 398, 432, 414, 434, 421, 396, 402, 426, 431, 404, 424, 420, 429, 393, 399, 433, 397, 403, 410, 428, 418, 434, 405, 427,
   432, 430, 422, 416, 394, 432, 400, 412, 430, 395, 401, 434, 419, 427, 406, 433, 423, 428, 417, 407, 418, 419, 410, 422, 412, 423,
   432, 433, 416, 417, 434, 428, 430, 427, 416, 408, 421, 422, 411, 430, 431, 414, 424, 418, 434, 420, 429, 426, 432, 417, 420, 409,
   428, 429, 423, 413, 424, 415, 434, 419, 421, 425, 431, 433, 410, 411, 425, 416, 418, 427, 433, 426, 431, 422, 428, 429, 434, 432,
   430, 412, 426, 413, 427, 432, 417, 419, 429, 425, 430, 423, 431, 433, 434, 428, 427, 414, 415, 430, 426, 428, 425, 420, 421, 432,
   433, 424, 431, 429, 434, 435, 456, 457, 440, 452, 442, 454, 483, 484, 444, 445, 504, 479, 481, 474, 458, 436, 459, 450, 438, 485,
   487, 443, 455, 446, 505, 447, 477, 475, 482, 460, 461, 437, 486, 488, 451, 439, 453, 441, 506, 448, 449, 476, 478, 480, 462, 438,
   489, 458, 446, 494, 512, 475, 487, 450, 507, 477, 505, 482, 485, 463, 490, 439, 492, 510, 460, 448, 488, 476, 508, 451, 478, 480,
   506, 486, 440, 464, 491, 444, 456, 474, 484, 493, 511, 452, 479, 509, 504, 483, 481, 492, 465, 441, 508, 490, 486, 476, 461, 449,
   510, 480, 453, 478, 488, 506, 442, 493, 466, 474, 483, 445, 457, 509, 491, 481, 454, 511, 484, 504, 479, 494, 443, 467, 485, 475,
   507, 489, 447, 459, 482, 512, 455, 487, 477, 505, 444, 468, 514, 452, 464, 481, 511, 495, 525, 456, 504, 513, 509, 493, 483, 445,
   513, 469, 479, 509, 454, 466, 525, 496, 504, 457, 514, 491, 511, 484, 470, 446, 516, 462, 450, 497, 526, 482, 512, 458, 515, 505,
   507, 485, 494, 515, 447, 471, 507, 477, 526, 499, 455, 467, 505, 516, 459, 489, 487, 512, 472, 518, 448, 498, 527, 463, 451, 510,
   480, 517, 460, 506, 486, 508, 492, 517, 473, 449, 527, 500, 508, 478, 465, 453, 518, 506, 461, 488, 490, 510, 468, 450, 528, 470,
   458, 503, 533, 485, 526, 462, 519, 507, 515, 494, 497, 469, 529, 451, 502, 531, 472, 460, 527, 486, 520, 463, 508, 492, 517, 498,
   452, 470, 530, 456, 468, 483, 525, 503, 532, 464, 509, 521, 513, 495, 493, 531, 471, 453, 529, 501, 527, 488, 473, 461, 522, 510,
   465, 490, 500, 518, 454, 532, 472, 484, 525, 457, 469, 530, 502, 511, 466, 523, 496, 514, 491, 533, 455, 473, 526, 487, 528, 501,
   459, 471, 512, 524, 467, 499, 489, 516, 456, 462, 537, 464, 470, 493, 532, 497, 534, 468, 513, 519, 521, 503, 495, 457, 537, 463,
   491, 530, 466, 472, 535, 498, 514, 469, 520, 502, 523, 496, 464, 458, 538, 468, 462, 495, 534, 494, 533, 470, 521, 515, 519, 497,
   503, 538, 459, 465, 528, 489, 536, 500, 467, 473, 516, 522, 471, 501, 499, 524, 466, 539, 460, 496, 535, 469, 463, 531, 492, 523,
   472, 517, 498, 520, 502, 539, 467, 461, 536, 499, 529, 490, 471, 465, 524, 518, 473, 500, 501, 522, 474, 495, 496, 481, 493, 479,
   491, 513, 514, 483, 484, 525, 511, 509, 504, 497, 475, 499, 494, 482, 515, 516, 477, 489, 485, 526, 487, 512, 505, 507, 498, 500,
   476, 517, 518, 492, 480, 490, 478, 527, 486, 488, 506, 510, 508, 519, 477, 501, 515, 505, 533, 524, 487, 499, 507, 528, 489, 516,
   512, 526, 520, 501, 478, 531, 522, 517, 506, 500, 488, 529, 508, 490, 510, 518, 527, 479, 521, 502, 504, 513, 484, 496, 532, 523,
   509, 491, 530, 514, 525, 511, 502, 522, 480, 520, 529, 498, 486, 518, 506, 531, 492, 510, 508, 527, 517, 481, 503, 523, 483, 495,
   504, 514, 521, 530, 493, 511, 532, 525, 513, 509, 503, 482, 524, 497, 485, 519, 528, 505, 516, 494, 533, 512, 526, 507, 515, 483,
   497, 535, 493, 503, 509, 530, 519, 537, 495, 525, 534, 532, 521, 513, 484, 534, 498, 511, 532, 491, 502, 537, 520, 525, 496, 535,
   523, 530, 514, 495, 485, 536, 503, 494, 521, 538, 507, 528, 497, 534, 526, 533, 515, 519, 496, 536, 486, 523, 539, 502, 492, 529,
   508, 535, 498, 527, 517, 531, 520, 534, 487, 500, 533, 512, 538, 522, 489, 501, 526, 536, 499, 524, 516, 528, 535, 499, 488, 539,
   524, 531, 510, 501, 490, 536, 527, 500, 518, 522, 529, 537, 489, 490, 538, 516, 539, 518, 499, 500, 528, 529, 501, 522, 524, 536,
   491, 538, 492, 514, 537, 496, 498, 539, 517, 530, 502, 531, 520, 535, 523, 493, 494, 539, 495, 497, 513, 537, 515, 538, 503, 532,
   533, 534, 519, 521, 504, 519, 520, 509, 521, 511, 523, 534, 535, 513, 514, 537, 530, 532, 525, 521, 505, 522, 519, 507, 534, 536,
   512, 524, 515, 538, 516, 528, 526, 533, 523, 524, 506, 535, 536, 520, 508, 522, 510, 539, 517, 518, 527, 529, 531, 513, 507, 529,
   521, 515, 532, 539, 526, 536, 519, 537, 528, 538, 533, 534, 514, 528, 508, 530, 538, 523, 517, 536, 527, 537, 520, 529, 531, 539,
   535, 509, 515, 531, 513, 519, 525, 535, 533, 539, 521, 530, 538, 537, 534, 532, 530, 516, 510, 537, 528, 535, 527, 524, 518, 538,
   531, 522, 529, 536, 539, 511, 533, 517, 525, 534, 514, 520, 538, 531, 532, 523, 539, 535, 537, 530, 532, 512, 518, 534, 526, 537,
   529, 516, 522, 533, 539, 524, 536, 528, 538, 525, 526, 527, 532, 533, 530, 531, 528, 529, 534, 535, 536, 539, 538, 537
}; // 6.45 kb
static const cart_index_t iCartXY_ab13_a5[3465] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180, 185, 212, 213, 206, 207, 188, 230, 190, 232, 194, 195, 200, 239, 202, 241, 217, 228, 229, 236,
   218, 220, 212, 186, 214, 189, 231, 208, 210, 234, 192, 201, 240, 196, 198, 243, 204, 226, 216, 227, 219, 237, 222, 213, 214, 187,
   233, 191, 235, 193, 209, 211, 242, 203, 244, 205, 197, 199, 224, 225, 215, 221, 223, 238, 206, 189, 233, 196, 247, 212, 231, 232,
   216, 208, 245, 201, 219, 249, 227, 234, 222, 243, 237, 240, 226, 207, 231, 191, 247, 197, 230, 215, 213, 233, 246, 209, 248, 225,
   203, 221, 235, 244, 223, 238, 224, 242, 188, 208, 235, 212, 230, 194, 246, 217, 234, 200, 218, 206, 245, 229, 250, 220, 232, 241,
   239, 236, 228, 230, 210, 193, 231, 215, 246, 199, 235, 214, 248, 224, 247, 211, 223, 205, 242, 233, 221, 225, 238, 244, 190, 234,
   209, 232, 213, 217, 235, 195, 245, 220, 202, 228, 250, 207, 246, 218, 239, 230, 241, 229, 236, 232, 192, 211, 216, 233, 234, 214,
   245, 198, 226, 249, 222, 204, 247, 210, 240, 219, 231, 227, 243, 237, 194, 201, 242, 208, 246, 200, 248, 220, 226, 206, 236, 212,
   240, 241, 249, 228, 234, 250, 245, 239, 232, 195, 240, 203, 245, 209, 218, 224, 202, 249, 236, 207, 239, 248, 213, 242, 229, 250,
   235, 246, 230, 241, 200, 196, 244, 201, 248, 206, 247, 228, 222, 212, 239, 208, 237, 250, 243, 232, 226, 249, 240, 245, 234, 239,
   198, 205, 219, 225, 245, 211, 250, 204, 240, 248, 237, 210, 244, 214, 249, 227, 233, 231, 247, 243, 202, 243, 197, 249, 203, 229,
   223, 207, 247, 241, 213, 250, 244, 209, 238, 230, 248, 224, 242, 235, 246, 241, 204, 199, 227, 221, 250, 205, 246, 210, 249, 242,
   243, 214, 238, 211, 248, 231, 225, 233, 244, 247, 217, 226, 224, 234, 235, 220, 242, 218, 240, 228, 229, 232, 249, 230, 248, 236,
   245, 246, 250, 241, 239, 228, 216, 225, 222, 244, 232, 233, 239, 219, 234, 250, 226, 227, 248, 231, 245, 237, 247, 243, 249, 240,
   229, 227, 215, 243, 223, 241, 221, 230, 231, 250, 235, 249, 233, 224, 225, 246, 247, 238, 244, 242, 248, 236, 219, 221, 237, 238,
   239, 225, 241, 227, 245, 246, 240, 231, 242, 233, 250, 243, 244, 247, 248, 249, 218, 237, 223, 240, 224, 236, 238, 229, 243, 239,
   230, 245, 247, 235, 244, 241, 249, 242, 248, 246, 250, 220, 222, 238, 226, 242, 228, 244, 236, 237, 232, 241, 234, 243, 246, 247,
   239, 240, 248, 249, 250, 245, 251, 280, 282, 272, 273, 256, 301, 258, 303, 260, 261, 268, 313, 270, 315, 284, 293, 294, 308, 289,
   291, 278, 252, 283, 254, 299, 274, 275, 304, 259, 266, 311, 262, 263, 316, 271, 295, 285, 297, 287, 309, 292, 279, 281, 253, 300,
   255, 302, 257, 276, 277, 312, 267, 314, 269, 264, 265, 296, 298, 286, 288, 290, 310, 256, 278, 306, 280, 301, 260, 318, 284, 305,
   268, 289, 272, 317, 294, 323, 291, 303, 315, 313, 308, 293, 280, 254, 307, 262, 320, 278, 299, 305, 285, 274, 319, 266, 287, 324,
   297, 304, 292, 316, 309, 311, 295, 258, 305, 279, 303, 282, 284, 306, 261, 317, 291, 270, 293, 323, 273, 318, 289, 313, 301, 315,
   294, 308, 282, 307, 255, 322, 264, 306, 286, 279, 300, 321, 276, 325, 298, 267, 288, 302, 314, 290, 310, 296, 312, 305, 259, 281,
   285, 307, 304, 283, 319, 263, 295, 324, 292, 271, 320, 275, 311, 287, 299, 297, 316, 309, 306, 283, 257, 307, 286, 321, 265, 302,
   281, 325, 296, 322, 277, 290, 269, 312, 300, 288, 298, 310, 314, 260, 274, 321, 278, 318, 268, 327, 291, 304, 272, 308, 280, 319,
   315, 328, 293, 305, 323, 317, 313, 303, 261, 319, 276, 317, 279, 289, 302, 270, 328, 308, 273, 313, 327, 282, 321, 294, 323, 306,
   318, 301, 315, 272, 262, 322, 266, 326, 280, 320, 303, 292, 278, 317, 274, 309, 328, 316, 305, 295, 324, 311, 319, 304, 273, 320,
   264, 326, 267, 301, 290, 282, 322, 318, 279, 327, 314, 276, 310, 306, 325, 296, 312, 302, 321, 317, 263, 277, 287, 300, 319, 281,
   328, 271, 311, 326, 309, 275, 322, 283, 324, 297, 307, 299, 320, 316, 318, 275, 265, 299, 288, 327, 269, 321, 283, 326, 312, 320,
   281, 310, 277, 325, 307, 298, 300, 314, 322, 268, 266, 325, 274, 327, 272, 326, 293, 295, 280, 313, 278, 311, 323, 324, 303, 304,
   328, 319, 317, 305, 270, 324, 267, 328, 276, 294, 296, 273, 326, 315, 282, 323, 325, 279, 312, 301, 327, 302, 321, 306, 318, 323,
   271, 269, 297, 298, 328, 277, 327, 275, 324, 325, 316, 283, 314, 281, 326, 299, 300, 307, 322, 320, 301, 299, 286, 320, 290, 318,
   288, 306, 307, 327, 302, 326, 300, 296, 298, 321, 322, 310, 314, 312, 325, 303, 285, 300, 292, 322, 305, 307, 317, 287, 304, 328,
   295, 297, 326, 299, 319, 309, 320, 316, 324, 311, 284, 304, 302, 305, 306, 291, 321, 289, 319, 293, 294, 303, 328, 301, 327, 308,
   317, 318, 323, 315, 313, 289, 311, 296, 319, 302, 308, 312, 294, 324, 313, 301, 317, 326, 306, 325, 315, 328, 321, 327, 318, 323,
   313, 287, 298, 309, 314, 317, 300, 323, 297, 319, 327, 311, 299, 325, 307, 328, 316, 322, 320, 326, 324, 291, 295, 312, 304, 321,
   293, 325, 308, 311, 303, 315, 305, 324, 318, 326, 313, 319, 327, 328, 323, 317, 315, 297, 288, 316, 310, 323, 298, 318, 299, 328,
   321, 324, 307, 312, 300, 327, 320, 314, 322, 325, 326, 293, 292, 314, 295, 325, 303, 322, 313, 309, 305, 323, 304, 316, 327, 320,
   317, 311, 326, 324, 328, 319, 294, 316, 290, 324, 296, 315, 310, 301, 320, 323, 306, 328, 322, 302, 314, 318, 326, 312, 325, 321,
   327, 308, 309, 310, 311, 312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 356, 358, 350,
   351, 332, 380, 334, 382, 338, 339, 344, 395, 346, 397, 367, 378, 379, 392, 368, 370, 357, 330, 360, 333, 381, 352, 354, 384, 336,
   345, 396, 340, 342, 399, 348, 376, 366, 377, 369, 393, 372, 359, 361, 331, 383, 335, 385, 337, 353, 355, 398, 347, 400, 349, 341,
   343, 374, 375, 365, 371, 373, 394, 362, 333, 389, 340, 405, 357, 381, 388, 366, 352, 403, 345, 369, 411, 377, 384, 372, 399, 393,
   396, 376, 363, 387, 335, 406, 341, 386, 365, 359, 383, 404, 353, 410, 375, 347, 371, 385, 400, 373, 394, 374, 398, 332, 362, 391,
   356, 380, 338, 402, 367, 390, 344, 368, 350, 401, 379, 412, 370, 382, 397, 395, 392, 378, 386, 364, 337, 387, 365, 404, 343, 385,
   361, 410, 374, 406, 355, 373, 349, 398, 383, 371, 375, 394, 400, 334, 390, 363, 382, 358, 367, 391, 339, 401, 370, 346, 378, 412,
   351, 402, 368, 395, 380, 397, 379, 392, 388, 336, 364, 366, 389, 384, 360, 403, 342, 376, 411, 372, 348, 405, 354, 396, 369, 381,
   377, 399, 393, 338, 357, 408, 362, 402, 344, 413, 370, 388, 350, 392, 356, 407, 397, 415, 378, 390, 412, 401, 395, 382, 339, 407,
   359, 401, 363, 368, 386, 346, 415, 392, 351, 395, 413, 358, 408, 379, 412, 391, 402, 380, 397, 356, 340, 409, 345, 414, 362, 405,
   390, 372, 357, 407, 352, 393, 417, 399, 388, 376, 411, 396, 403, 384, 407, 342, 361, 369, 387, 403, 364, 417, 348, 396, 414, 393,
   354, 409, 360, 411, 377, 389, 381, 405, 399, 358, 409, 341, 416, 347, 391, 373, 363, 406, 408, 359, 418, 400, 353, 394, 386, 410,
   374, 398, 385, 404, 408, 360, 343, 389, 371, 418, 349, 404, 364, 416, 398, 409, 361, 394, 355, 410, 387, 375, 383, 400, 406, 350,
   345, 416, 352, 419, 356, 414, 382, 376, 362, 401, 357, 396, 415, 411, 390, 384, 417, 403, 407, 388, 351, 414, 347, 419, 353, 380,
   374, 358, 416, 402, 363, 413, 410, 359, 398, 391, 418, 385, 404, 386, 408, 344, 352, 418, 357, 413, 350, 419, 378, 384, 356, 395,
   362, 403, 412, 417, 382, 388, 415, 407, 401, 390, 413, 354, 349, 381, 375, 419, 355, 418, 360, 414, 410, 405, 364, 400, 361, 416,
   389, 383, 387, 406, 409, 346, 417, 353, 415, 359, 379, 385, 351, 419, 397, 358, 412, 418, 363, 404, 380, 413, 386, 408, 391, 402,
   415, 348, 355, 377, 383, 417, 361, 419, 354, 411, 416, 399, 360, 406, 364, 414, 381, 387, 389, 409, 405, 367, 388, 386, 390, 391,
   370, 408, 368, 407, 378, 379, 382, 415, 380, 413, 392, 401, 402, 412, 397, 395, 390, 366, 387, 372, 409, 388, 389, 407, 369, 384,
   417, 376, 377, 414, 381, 403, 393, 405, 399, 411, 396, 391, 389, 365, 409, 373, 408, 371, 386, 387, 418, 385, 416, 383, 374, 375,
   404, 406, 394, 400, 398, 410, 401, 369, 383, 393, 406, 407, 387, 415, 377, 403, 419, 396, 381, 416, 389, 417, 399, 409, 405, 414,
   411, 402, 381, 371, 405, 394, 413, 375, 408, 389, 419, 404, 414, 387, 398, 383, 418, 409, 400, 406, 410, 416, 368, 403, 385, 407,
   386, 392, 404, 379, 417, 395, 380, 401, 419, 391, 418, 397, 415, 408, 413, 402, 412, 380, 405, 373, 414, 374, 402, 394, 391, 409,
   413, 386, 419, 406, 385, 400, 408, 416, 398, 410, 404, 418, 370, 384, 404, 388, 408, 378, 418, 392, 403, 382, 397, 390, 417, 402,
   419, 395, 407, 413, 415, 412, 401, 382, 372, 406, 376, 416, 390, 409, 401, 393, 388, 415, 384, 399, 419, 405, 407, 396, 414, 411,
   417, 403, 378, 376, 410, 384, 418, 382, 416, 395, 396, 390, 412, 388, 411, 413, 414, 401, 403, 419, 417, 415, 407, 379, 411, 374,
   417, 385, 397, 398, 380, 414, 412, 391, 415, 416, 386, 410, 402, 419, 404, 418, 408, 413, 412, 377, 375, 399, 400, 415, 383, 413,
   381, 417, 418, 411, 389, 410, 387, 419, 405, 406, 409, 416, 414, 392, 396, 398, 403, 404, 395, 410, 397, 411, 401, 402, 407, 414,
   408, 416, 412, 417, 418, 419, 413, 415, 395, 393, 400, 396, 410, 401, 406, 412, 399, 407, 413, 403, 405, 418, 409, 415, 411, 416,
   414, 419, 417, 397, 399, 394, 411, 398, 412, 400, 402, 405, 415, 408, 417, 409, 404, 406, 413, 414, 410, 416, 418, 419, 420, 449,
   451, 441, 442, 425, 476, 427, 478, 429, 430, 437, 494, 439, 496, 459, 468, 469, 489, 464, 466, 447, 421, 452, 423, 474, 443, 444,
   479, 428, 435, 492, 431, 432, 497, 440, 470, 460, 472, 462, 490, 467, 448, 450, 422, 475, 424, 477, 426, 445, 446, 493, 436, 495,
   438, 433, 434, 471, 473, 461, 463, 465, 491, 425, 453, 481, 449, 476, 429, 499, 459, 480, 437, 464, 441, 498, 469, 510, 466, 478,
   496, 494, 489, 468, 455, 423, 484, 431, 501, 447, 474, 482, 460, 443, 500, 435, 462, 511, 472, 479, 467, 497, 490, 492, 470, 427,
   480, 454, 478, 451, 459, 481, 430, 498, 466, 439, 468, 510, 442, 499, 464, 494, 476, 496, 469, 489, 457, 485, 424, 503, 433, 483,
   461, 448, 475, 502, 445, 512, 473, 436, 463, 477, 495, 465, 491, 471, 493, 482, 428, 456, 460, 484, 479, 452, 500, 432, 470, 511,
   467, 440, 501, 444, 492, 462, 474, 472, 497, 490, 483, 458, 426, 485, 461, 502, 434, 477, 450, 512, 471, 503, 446, 465, 438, 493,
   475, 463, 473, 491, 495, 429, 455, 508, 453, 499, 437, 515, 466, 488, 441, 489, 449, 506, 496, 517, 468, 480, 510, 498, 494, 478,
   430, 506, 457, 498, 454, 464, 487, 439, 517, 489, 442, 494, 515, 451, 508, 469, 510, 481, 499, 476, 496, 453, 431, 509, 435, 513,
   455, 501, 488, 467, 447, 504, 443, 490, 518, 497, 482, 470, 511, 492, 500, 479, 454, 507, 433, 514, 436, 487, 465, 457, 503, 505,
   448, 516, 495, 445, 491, 483, 512, 471, 493, 477, 502, 504, 432, 458, 462, 486, 500, 456, 518, 440, 492, 513, 490, 444, 509, 452,
   511, 472, 484, 474, 501, 497, 505, 456, 434, 486, 463, 516, 438, 502, 458, 514, 493, 507, 450, 491, 446, 512, 485, 473, 475, 495,
   503, 437, 447, 520, 455, 515, 441, 522, 468, 482, 449, 494, 453, 504, 510, 519, 478, 488, 517, 506, 498, 480, 449, 435, 521, 443,
   523, 453, 513, 480, 470, 455, 506, 447, 492, 519, 511, 488, 479, 518, 500, 504, 482, 439, 519, 448, 517, 457, 469, 483, 442, 522,
   496, 451, 510, 520, 454, 505, 476, 515, 487, 508, 481, 499, 451, 521, 436, 524, 445, 481, 471, 454, 514, 508, 457, 520, 512, 448,
   493, 487, 516, 477, 502, 483, 505, 519, 440, 450, 472, 485, 518, 458, 523, 444, 511, 521, 497, 452, 507, 456, 513, 474, 486, 484,
   509, 501, 520, 452, 438, 484, 473, 524, 446, 516, 456, 521, 512, 509, 458, 495, 450, 514, 486, 475, 485, 503, 507, 441, 443, 524,
   447, 522, 449, 523, 478, 479, 453, 498, 455, 500, 517, 518, 480, 482, 519, 504, 506, 488, 442, 523, 445, 522, 448, 476, 477, 451,
   524, 499, 454, 515, 516, 457, 502, 481, 520, 483, 505, 487, 508, 522, 444, 446, 474, 475, 523, 450, 524, 452, 513, 514, 501, 456,
   503, 458, 521, 484, 485, 486, 507, 509, 487, 486, 461, 507, 465, 505, 463, 483, 485, 516, 477, 514, 475, 471, 473, 502, 503, 491,
   495, 493, 512, 488, 460, 486, 467, 509, 482, 484, 504, 462, 479, 518, 470, 472, 513, 474, 500, 490, 501, 497, 511, 492, 459, 488,
   487, 480, 481, 466, 508, 464, 506, 468, 469, 478, 517, 476, 515, 489, 498, 499, 510, 496, 494, 464, 504, 483, 506, 487, 489, 505,
   469, 519, 494, 476, 498, 522, 481, 520, 496, 517, 508, 515, 499, 510, 506, 462, 485, 490, 507, 504, 486, 519, 472, 500, 523, 492,
   474, 521, 484, 518, 497, 509, 501, 513, 511, 466, 482, 505, 488, 508, 468, 520, 489, 504, 478, 496, 480, 519, 499, 522, 494, 506,
   515, 517, 510, 498, 508, 484, 463, 509, 491, 520, 473, 505, 486, 524, 502, 521, 485, 493, 475, 516, 507, 495, 503, 512, 514, 480,
   467, 507, 470, 521, 488, 509, 506, 490, 482, 519, 479, 497, 523, 501, 504, 492, 513, 511, 518, 500, 481, 509, 465, 521, 471, 508,
   491, 487, 507, 520, 483, 524, 503, 477, 495, 505, 514, 493, 512, 502, 516, 476, 513, 471, 523, 477, 499, 493, 481, 521, 515, 487,
   522, 514, 483, 512, 508, 524, 502, 516, 505, 520, 515, 474, 473, 501, 495, 522, 475, 520, 484, 523, 516, 513, 486, 512, 485, 524,
   509, 503, 507, 514, 521, 478, 470, 514, 479, 524, 480, 521, 498, 492, 488, 517, 482, 511, 522, 513, 506, 500, 523, 518, 519, 504,
   517, 472, 475, 497, 503, 519, 485, 522, 474, 518, 524, 511, 484, 514, 486, 523, 501, 507, 509, 521, 513, 468, 479, 516, 482, 520,
   478, 524, 494, 500, 480, 510, 488, 518, 515, 523, 498, 504, 522, 519, 517, 506, 469, 518, 477, 519, 483, 496, 502, 476, 523, 510,
   481, 517, 524, 487, 516, 499, 522, 505, 520, 508, 515, 489, 500, 502, 504, 505, 494, 516, 496, 518, 498, 499, 506, 523, 508, 524,
   510, 519, 520, 522, 515, 517, 498, 490, 503, 492, 514, 506, 507, 517, 497, 504, 522, 500, 501, 524, 509, 519, 511, 521, 513, 523,
   518, 499, 501, 491, 513, 493, 515, 495, 508, 509, 522, 505, 523, 507, 502, 503, 520, 521, 512, 514, 516, 524, 494, 492, 512, 500,
   516, 498, 514, 510, 511, 506, 515, 504, 513, 520, 521, 517, 518, 524, 523, 522, 519, 496, 511, 493, 518, 502, 510, 512, 499, 513,
   517, 508, 519, 521, 505, 514, 515, 523, 516, 524, 520, 522, 510, 497, 495, 511, 512, 517, 503, 515, 501, 519, 520, 518, 509, 516,
   507, 522, 513, 514, 521, 524, 523
}; // 6.77 kb
static const cart_index_t iCartXY_ab13_a6[3360] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163, 164, 187, 188, 167, 191, 169, 192, 213, 214, 173, 174, 185,
   186, 224, 225, 179, 181, 229, 209, 211, 196, 197, 218, 199, 220, 207, 208, 215, 185, 165, 190, 191, 168, 211, 212, 171, 193, 187,
   224, 175, 226, 177, 189, 180, 228, 183, 210, 195, 213, 219, 198, 205, 206, 201, 222, 216, 186, 189, 166, 209, 210, 192, 170, 193,
   172, 225, 188, 226, 176, 190, 178, 227, 182, 184, 194, 212, 214, 203, 204, 221, 200, 223, 202, 217, 167, 180, 203, 173, 187, 196,
   214, 205, 221, 179, 197, 191, 209, 219, 227, 185, 208, 228, 225, 213, 199, 215, 224, 207, 229, 211, 220, 218, 179, 168, 204, 185,
   175, 207, 223, 195, 212, 191, 218, 180, 227, 198, 210, 187, 229, 206, 226, 201, 211, 224, 216, 213, 222, 205, 228, 219, 169, 205,
   182, 196, 213, 174, 188, 219, 203, 199, 181, 211, 192, 228, 221, 207, 186, 227, 214, 224, 197, 208, 229, 215, 225, 218, 209, 220,
   181, 206, 170, 208, 222, 186, 176, 210, 194, 220, 192, 228, 182, 212, 200, 229, 188, 204, 202, 226, 209, 214, 223, 225, 217, 227,
   203, 221, 207, 171, 184, 211, 195, 218, 204, 177, 190, 213, 229, 201, 223, 183, 193, 205, 227, 189, 212, 198, 224, 228, 206, 219,
   210, 216, 226, 222, 208, 183, 172, 220, 206, 209, 194, 189, 178, 229, 214, 222, 202, 193, 184, 228, 203, 190, 200, 210, 225, 221,
   212, 227, 204, 226, 217, 223, 173, 175, 217, 179, 180, 199, 221, 201, 223, 185, 215, 187, 225, 216, 226, 191, 220, 222, 227, 205,
   207, 218, 219, 211, 228, 213, 229, 224, 174, 216, 176, 197, 219, 181, 182, 222, 202, 215, 186, 224, 188, 226, 217, 218, 192, 223,
   203, 228, 208, 209, 227, 220, 221, 229, 214, 225, 215, 177, 178, 218, 198, 220, 200, 183, 184, 224, 225, 216, 217, 189, 190, 219,
   221, 193, 204, 206, 229, 227, 210, 228, 212, 222, 223, 226, 197, 198, 194, 215, 216, 208, 202, 206, 200, 218, 209, 219, 203, 210,
   204, 224, 214, 212, 217, 222, 220, 225, 226, 229, 223, 228, 221, 227, 199, 195, 200, 207, 201, 215, 217, 198, 204, 211, 220, 205,
   221, 206, 212, 213, 225, 210, 223, 216, 218, 229, 222, 224, 226, 219, 227, 228, 196, 201, 202, 199, 205, 197, 203, 216, 217, 207,
   208, 213, 214, 222, 223, 211, 209, 226, 221, 219, 215, 220, 228, 218, 227, 224, 225, 229, 230, 257, 258, 235, 259, 237, 261, 284,
   285, 239, 240, 251, 252, 296, 297, 247, 249, 302, 280, 282, 263, 268, 292, 270, 294, 272, 273, 287, 259, 231, 260, 257, 233, 284,
   286, 238, 262, 253, 298, 241, 299, 242, 254, 245, 303, 250, 278, 264, 283, 290, 266, 274, 276, 271, 295, 288, 261, 262, 232, 285,
   286, 258, 234, 260, 236, 300, 255, 301, 243, 256, 244, 304, 246, 248, 265, 279, 281, 275, 277, 291, 267, 293, 269, 289, 251, 233,
   279, 259, 241, 282, 301, 264, 286, 257, 296, 245, 305, 266, 278, 253, 307, 276, 299, 271, 284, 298, 288, 283, 295, 274, 303, 290,
   252, 278, 234, 280, 299, 261, 243, 286, 265, 297, 258, 305, 246, 279, 267, 306, 255, 277, 269, 301, 285, 281, 293, 300, 289, 304,
   275, 291, 235, 253, 281, 239, 257, 263, 285, 283, 300, 247, 268, 259, 280, 298, 306, 251, 273, 307, 297, 284, 270, 287, 296, 272,
   302, 282, 294, 292, 280, 254, 236, 297, 278, 285, 265, 262, 244, 306, 281, 299, 269, 260, 248, 305, 275, 256, 267, 286, 300, 291,
   279, 304, 277, 301, 289, 293, 237, 283, 255, 263, 284, 240, 258, 298, 281, 270, 249, 282, 261, 307, 300, 272, 252, 306, 285, 296,
   268, 273, 302, 287, 297, 292, 280, 294, 282, 238, 256, 284, 264, 296, 279, 242, 260, 283, 307, 271, 301, 250, 262, 274, 305, 254,
   286, 266, 298, 303, 276, 290, 278, 288, 299, 295, 239, 245, 291, 247, 253, 270, 300, 274, 304, 251, 287, 257, 297, 290, 305, 259,
   294, 303, 306, 283, 272, 292, 298, 282, 307, 284, 302, 296, 240, 290, 246, 268, 298, 249, 255, 303, 275, 287, 252, 296, 258, 305,
   291, 292, 261, 304, 281, 307, 273, 280, 306, 294, 300, 302, 285, 297, 247, 241, 293, 251, 245, 272, 304, 271, 301, 259, 292, 253,
   306, 288, 299, 257, 302, 295, 305, 274, 282, 296, 290, 284, 303, 283, 307, 298, 292, 242, 248, 296, 266, 302, 277, 250, 256, 298,
   306, 288, 293, 254, 260, 290, 304, 262, 279, 276, 307, 305, 278, 303, 286, 295, 301, 299, 249, 295, 243, 273, 303, 252, 246, 299,
   269, 294, 261, 307, 255, 301, 289, 302, 258, 293, 275, 305, 280, 285, 304, 297, 291, 306, 281, 300, 294, 250, 244, 302, 276, 297,
   267, 254, 248, 307, 300, 295, 289, 262, 256, 303, 291, 260, 277, 278, 306, 304, 286, 305, 279, 299, 293, 301, 263, 274, 275, 270,
   283, 268, 281, 290, 291, 272, 273, 284, 285, 303, 304, 282, 280, 305, 300, 298, 287, 294, 307, 292, 306, 296, 297, 302, 272, 264,
   277, 282, 271, 292, 293, 266, 279, 284, 302, 274, 304, 276, 286, 283, 306, 278, 301, 288, 296, 307, 295, 298, 299, 290, 305, 303,
   273, 276, 265, 294, 295, 280, 269, 278, 267, 302, 285, 303, 275, 286, 277, 307, 281, 279, 289, 299, 297, 300, 301, 306, 293, 305,
   291, 304, 287, 266, 267, 292, 288, 294, 289, 276, 277, 296, 297, 290, 291, 278, 279, 298, 300, 286, 293, 295, 302, 306, 299, 307,
   301, 303, 304, 305, 268, 288, 269, 287, 290, 273, 275, 295, 289, 292, 280, 298, 281, 299, 293, 296, 285, 301, 291, 303, 294, 297,
   305, 302, 304, 307, 300, 306, 270, 271, 289, 272, 274, 287, 291, 288, 293, 282, 294, 283, 300, 295, 301, 284, 297, 299, 304, 290,
   292, 302, 303, 296, 305, 298, 306, 307, 308, 341, 342, 311, 335, 313, 337, 369, 370, 317, 318, 329, 330, 380, 381, 323, 325, 391,
   359, 361, 346, 347, 374, 349, 376, 357, 358, 371, 341, 309, 343, 336, 312, 367, 368, 315, 339, 331, 382, 319, 384, 321, 333, 324,
   390, 327, 360, 345, 363, 375, 348, 355, 356, 351, 378, 372, 342, 343, 310, 365, 366, 338, 314, 340, 316, 383, 332, 385, 320, 334,
   322, 389, 326, 328, 344, 362, 364, 353, 354, 377, 350, 379, 352, 373, 311, 336, 365, 317, 341, 346, 370, 367, 387, 323, 347, 335,
   359, 386, 392, 329, 358, 394, 381, 369, 349, 371, 380, 357, 391, 361, 376, 374, 335, 312, 366, 341, 319, 369, 388, 345, 368, 336,
   386, 324, 393, 348, 360, 331, 396, 356, 384, 351, 367, 382, 372, 363, 378, 355, 390, 375, 313, 367, 338, 346, 369, 318, 342, 386,
   365, 349, 325, 361, 337, 394, 387, 357, 330, 392, 370, 380, 347, 358, 391, 371, 381, 374, 359, 376, 337, 368, 314, 370, 388, 342,
   320, 366, 344, 387, 338, 395, 326, 362, 350, 397, 332, 354, 352, 385, 365, 364, 379, 383, 373, 389, 353, 377, 369, 315, 340, 367,
   345, 386, 366, 321, 343, 363, 396, 351, 388, 327, 339, 355, 393, 333, 368, 348, 382, 390, 356, 375, 360, 372, 384, 378, 370, 339,
   316, 387, 368, 365, 344, 343, 322, 397, 364, 388, 352, 340, 328, 395, 353, 334, 350, 366, 383, 377, 362, 389, 354, 385, 373, 379,
   317, 331, 383, 323, 336, 349, 387, 363, 397, 329, 371, 341, 381, 382, 398, 335, 376, 396, 392, 367, 357, 374, 386, 361, 394, 369,
   391, 380, 318, 382, 332, 347, 386, 325, 338, 396, 364, 371, 330, 380, 342, 398, 383, 374, 337, 397, 365, 394, 358, 359, 392, 376,
   387, 391, 370, 381, 329, 319, 385, 335, 324, 361, 395, 351, 388, 341, 380, 331, 398, 372, 384, 336, 394, 378, 393, 355, 369, 386,
   375, 367, 390, 363, 396, 382, 330, 384, 320, 359, 393, 337, 326, 388, 352, 381, 342, 398, 332, 385, 373, 392, 338, 379, 353, 395,
   370, 365, 389, 387, 377, 397, 364, 383, 380, 321, 334, 386, 348, 394, 362, 327, 340, 382, 398, 372, 385, 333, 343, 375, 395, 339,
   366, 356, 396, 393, 360, 390, 368, 378, 388, 384, 381, 333, 322, 392, 360, 387, 350, 339, 328, 398, 383, 384, 373, 343, 334, 393,
   377, 340, 354, 368, 397, 389, 366, 395, 362, 388, 379, 385, 323, 324, 389, 329, 331, 357, 397, 355, 395, 335, 374, 336, 392, 375,
   393, 341, 391, 390, 398, 363, 361, 380, 382, 369, 396, 367, 394, 386, 325, 390, 326, 358, 396, 330, 332, 393, 353, 376, 337, 394,
   338, 395, 377, 391, 342, 389, 364, 398, 359, 370, 397, 381, 383, 392, 365, 387, 391, 327, 328, 394, 356, 392, 354, 333, 334, 396,
   397, 378, 379, 339, 340, 390, 389, 343, 362, 360, 398, 395, 368, 393, 366, 384, 385, 388, 359, 360, 344, 381, 384, 370, 352, 368,
   350, 392, 365, 393, 353, 366, 354, 398, 364, 362, 373, 388, 387, 383, 385, 397, 379, 395, 377, 389, 361, 345, 362, 369, 351, 380,
   385, 348, 366, 367, 394, 355, 395, 356, 368, 363, 398, 360, 388, 372, 386, 396, 378, 382, 384, 375, 393, 390, 346, 363, 364, 349,
   367, 347, 365, 382, 383, 357, 358, 369, 370, 396, 397, 361, 359, 398, 387, 386, 371, 376, 394, 374, 392, 380, 381, 391, 347, 375,
   353, 371, 382, 358, 364, 390, 377, 374, 359, 386, 365, 393, 389, 380, 370, 395, 383, 396, 376, 381, 398, 391, 397, 394, 387, 392,
   374, 348, 354, 380, 372, 391, 379, 356, 362, 386, 392, 375, 389, 360, 366, 382, 397, 368, 385, 378, 394, 398, 384, 396, 388, 390,
   395, 393, 349, 355, 377, 357, 363, 371, 383, 375, 389, 361, 376, 367, 387, 390, 395, 369, 381, 393, 397, 382, 374, 391, 396, 380,
   398, 386, 392, 394, 376, 356, 350, 391, 378, 381, 373, 360, 354, 394, 387, 390, 377, 368, 362, 396, 383, 366, 379, 384, 392, 397,
   388, 398, 385, 393, 389, 395, 357, 351, 379, 361, 355, 374, 389, 372, 385, 369, 391, 363, 397, 378, 388, 367, 392, 384, 395, 375,
   380, 394, 390, 386, 393, 382, 398, 396, 358, 378, 352, 376, 390, 359, 353, 384, 373, 391, 370, 396, 364, 388, 379, 394, 365, 385,
   377, 393, 381, 387, 395, 392, 389, 398, 383, 397, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386,
   387, 388, 389, 390, 391, 392, 393, 394, 395, 396, 397, 398, 399, 432, 433, 404, 428, 406, 430, 459, 460, 408, 409, 420, 421, 477,
   478, 416, 418, 489, 455, 457, 438, 443, 473, 445, 475, 447, 448, 468, 434, 400, 435, 426, 402, 461, 463, 407, 431, 422, 479, 410,
   480, 411, 423, 414, 490, 419, 453, 439, 458, 471, 441, 449, 451, 446, 476, 469, 436, 437, 401, 462, 464, 427, 403, 429, 405, 481,
   424, 482, 412, 425, 413, 491, 415, 417, 440, 454, 456, 450, 452, 472, 442, 474, 444, 470, 432, 402, 465, 434, 410, 467, 488, 439,
   463, 426, 483, 414, 492, 441, 453, 422, 497, 451, 480, 446, 461, 479, 469, 458, 476, 449, 490, 471, 433, 465, 403, 466, 486, 436,
   412, 464, 440, 484, 427, 493, 415, 454, 442, 495, 424, 452, 444, 482, 462, 456, 474, 481, 470, 491, 450, 472, 404, 434, 466, 408,
   432, 438, 460, 467, 487, 416, 443, 428, 455, 485, 494, 420, 448, 496, 478, 459, 445, 468, 477, 447, 489, 457, 475, 473, 466, 435,
   405, 484, 465, 462, 440, 437, 413, 495, 456, 486, 444, 429, 417, 493, 450, 425, 442, 464, 481, 472, 454, 491, 452, 482, 470, 474,
   406, 467, 436, 438, 459, 409, 433, 485, 466, 445, 418, 457, 430, 496, 487, 447, 421, 494, 460, 477, 443, 448, 489, 468, 478, 473,
   455, 475, 467, 407, 437, 461, 439, 483, 465, 411, 435, 458, 497, 446, 488, 419, 431, 449, 492, 423, 463, 441, 479, 490, 451, 471,
   453, 469, 480, 476, 408, 426, 484, 416, 434, 445, 487, 461, 499, 420, 468, 432, 478, 483, 501, 428, 475, 498, 494, 467, 447, 473,
   485, 457, 496, 459, 489, 477, 409, 483, 427, 443, 485, 418, 436, 498, 462, 468, 421, 477, 433, 501, 484, 473, 430, 499, 466, 496,
   448, 455, 494, 475, 487, 489, 460, 478, 428, 410, 486, 432, 414, 459, 500, 446, 488, 434, 485, 422, 502, 469, 480, 426, 498, 476,
   492, 449, 467, 483, 471, 461, 490, 458, 497, 479, 485, 411, 429, 483, 441, 498, 464, 419, 437, 479, 502, 469, 486, 423, 435, 471,
   500, 431, 465, 451, 497, 492, 453, 490, 463, 476, 488, 480, 430, 488, 412, 460, 500, 433, 415, 486, 444, 487, 436, 503, 424, 482,
   470, 499, 427, 474, 450, 493, 466, 462, 491, 484, 472, 495, 456, 481, 487, 431, 413, 499, 463, 484, 442, 435, 417, 503, 481, 488,
   470, 437, 425, 500, 472, 429, 452, 465, 495, 491, 464, 493, 454, 486, 474, 482, 420, 414, 493, 428, 422, 457, 503, 449, 500, 432,
   477, 426, 501, 471, 492, 434, 496, 490, 502, 458, 459, 485, 479, 467, 497, 461, 498, 483, 421, 492, 415, 455, 502, 430, 424, 500,
   450, 478, 433, 501, 427, 493, 472, 494, 436, 491, 456, 503, 460, 466, 495, 487, 481, 499, 462, 484, 416, 422, 495, 420, 426, 447,
   499, 458, 503, 428, 473, 434, 494, 479, 502, 432, 489, 497, 501, 461, 457, 477, 483, 459, 498, 467, 496, 485, 494, 423, 417, 501,
   453, 499, 452, 431, 425, 502, 495, 480, 474, 435, 429, 492, 491, 437, 454, 463, 503, 493, 465, 500, 464, 488, 482, 486, 418, 497,
   424, 448, 498, 421, 427, 502, 456, 475, 430, 496, 436, 503, 481, 489, 433, 495, 462, 501, 455, 460, 499, 478, 484, 494, 466, 487,
   496, 419, 425, 498, 451, 501, 454, 423, 429, 497, 503, 476, 482, 431, 437, 490, 493, 435, 464, 453, 502, 500, 463, 492, 465, 480,
   486, 488, 438, 461, 462, 445, 467, 443, 466, 483, 484, 447, 448, 459, 460, 498, 499, 457, 455, 501, 487, 485, 468, 475, 496, 473,
   494, 477, 478, 489, 459, 439, 464, 467, 446, 485, 486, 441, 465, 461, 498, 449, 500, 451, 463, 458, 502, 453, 488, 469, 483, 497,
   476, 479, 480, 471, 492, 490, 460, 463, 440, 487, 488, 466, 444, 465, 442, 499, 462, 500, 450, 464, 452, 503, 456, 454, 470, 486,
   484, 481, 482, 495, 474, 493, 472, 491, 477, 441, 454, 485, 469, 496, 482, 451, 464, 483, 501, 471, 493, 453, 465, 479, 503, 463,
   486, 476, 498, 502, 480, 497, 488, 490, 500, 492, 478, 453, 442, 494, 480, 487, 470, 463, 452, 501, 484, 492, 472, 465, 454, 502,
   481, 464, 474, 488, 499, 495, 486, 503, 482, 500, 491, 493, 443, 479, 456, 468, 483, 448, 462, 497, 481, 473, 455, 485, 466, 502,
   495, 477, 460, 503, 484, 498, 475, 478, 501, 489, 499, 496, 487, 494, 455, 480, 444, 478, 492, 460, 450, 488, 470, 494, 466, 502,
   456, 486, 474, 501, 462, 482, 472, 500, 487, 484, 493, 499, 491, 503, 481, 495, 445, 458, 481, 447, 461, 468, 484, 479, 495, 457,
   475, 467, 487, 497, 503, 459, 478, 502, 499, 483, 473, 489, 498, 477, 501, 485, 494, 496, 457, 446, 482, 459, 449, 477, 493, 469,
   486, 467, 496, 458, 503, 476, 488, 461, 501, 480, 500, 471, 485, 498, 490, 483, 492, 479, 502, 497, 447, 449, 491, 457, 458, 473,
   495, 471, 493, 459, 489, 461, 499, 490, 500, 467, 494, 492, 503, 479, 477, 496, 497, 485, 502, 483, 501, 498, 448, 490, 450, 475,
   497, 455, 456, 492, 472, 489, 460, 498, 462, 500, 491, 496, 466, 493, 481, 502, 478, 487, 503, 494, 495, 501, 484, 499, 489, 451,
   452, 496, 476, 494, 474, 453, 454, 498, 499, 490, 491, 463, 464, 497, 495, 465, 482, 480, 501, 503, 488, 502, 486, 492, 493, 500,
   468, 471, 472, 473, 479, 475, 481, 490, 491, 477, 478, 483, 484, 492, 493, 485, 487, 500, 495, 497, 489, 494, 502, 496, 503, 498,
   499, 501, 473, 469, 474, 477, 471, 489, 491, 476, 482, 485, 494, 479, 495, 480, 486, 483, 499, 488, 493, 490, 496, 501, 492, 498,
   500, 497, 503, 502, 475, 476, 470, 489, 490, 478, 472, 480, 474, 496, 487, 497, 481, 488, 482, 498, 484, 486, 491, 492, 494, 499,
   500, 501, 493, 502, 495, 503
}; // 6.56 kb
static const cart_index_t iCartXY_ab14_a0[680] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119, 120, 121, 122, 123, 124, 125, 126, 127, 128, 129,
   130, 131, 132, 133, 134, 135, 136, 137, 138, 139, 140, 141, 142, 143, 144, 145, 146, 147, 148, 149, 150, 151, 152, 153, 154, 155,
   156, 157, 158, 159, 160, 161, 162, 163, 164, 165, 166, 167, 168, 169, 170, 171, 172, 173, 174, 175, 176, 177, 178, 179, 180, 181,
   182, 183, 184, 185, 186, 187, 188, 189, 190, 191, 192, 193, 194, 195, 196, 197, 198, 199, 200, 201, 202, 203, 204, 205, 206, 207,
   208, 209, 210, 211, 212, 213, 214, 215, 216, 217, 218, 219, 220, 221, 222, 223, 224, 225, 226, 227, 228, 229, 230, 231, 232, 233,
   234, 235, 236, 237, 238, 239, 240, 241, 242, 243, 244, 245, 246, 247, 248, 249, 250, 251, 252, 253, 254, 255, 256, 257, 258, 259,
   260, 261, 262, 263, 264, 265, 266, 267, 268, 269, 270, 271, 272, 273, 274, 275, 276, 277, 278, 279, 280, 281, 282, 283, 284, 285,
   286, 287, 288, 289, 290, 291, 292, 293, 294, 295, 296, 297, 298, 299, 300, 301, 302, 303, 304, 305, 306, 307, 308, 309, 310, 311,
   312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 330, 331, 332, 333, 334, 335, 336, 337,
   338, 339, 340, 341, 342, 343, 344, 345, 346, 347, 348, 349, 350, 351, 352, 353, 354, 355, 356, 357, 358, 359, 360, 361, 362, 363,
   364, 365, 366, 367, 368, 369, 370, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386, 387, 388, 389,
   390, 391, 392, 393, 394, 395, 396, 397, 398, 399, 400, 401, 402, 403, 404, 405, 406, 407, 408, 409, 410, 411, 412, 413, 414, 415,
   416, 417, 418, 419, 420, 421, 422, 423, 424, 425, 426, 427, 428, 429, 430, 431, 432, 433, 434, 435, 436, 437, 438, 439, 440, 441,
   442, 443, 444, 445, 446, 447, 448, 449, 450, 451, 452, 453, 454, 455, 456, 457, 458, 459, 460, 461, 462, 463, 464, 465, 466, 467,
   468, 469, 470, 471, 472, 473, 474, 475, 476, 477, 478, 479, 480, 481, 482, 483, 484, 485, 486, 487, 488, 489, 490, 491, 492, 493,
   494, 495, 496, 497, 498, 499, 500, 501, 502, 503, 504, 505, 506, 507, 508, 509, 510, 511, 512, 513, 514, 515, 516, 517, 518, 519,
   520, 521, 522, 523, 524, 525, 526, 527, 528, 529, 530, 531, 532, 533, 534, 535, 536, 537, 538, 539, 540, 541, 542, 543, 544, 545,
   546, 547, 548, 549, 550, 551, 552, 553, 554, 555, 556, 557, 558, 559, 560, 561, 562, 563, 564, 565, 566, 567, 568, 569, 570, 571,
   572, 573, 574, 575, 576, 577, 578, 579, 580, 581, 582, 583, 584, 585, 586, 587, 588, 589, 590, 591, 592, 593, 594, 595, 596, 597,
   598, 599, 600, 601, 602, 603, 604, 605, 606, 607, 608, 609, 610, 611, 612, 613, 614, 615, 616, 617, 618, 619, 620, 621, 622, 623,
   624, 625, 626, 627, 628, 629, 630, 631, 632, 633, 634, 635, 636, 637, 638, 639, 640, 641, 642, 643, 644, 645, 646, 647, 648, 649,
   650, 651, 652, 653, 654, 655, 656, 657, 658, 659, 660, 661, 662, 663, 664, 665, 666, 667, 668, 669, 670, 671, 672, 673, 674, 675,
   676, 677, 678, 679
}; // 1.33 kb
static const cart_index_t iCartXY_ab14_a1[1680] = {
   0, 1, 2, 3, 6, 7, 6, 4, 8, 7, 8, 5, 9, 14, 16, 12, 10, 17, 13, 15, 11, 14, 12, 18, 16, 18,
   13, 18, 17, 15, 19, 22, 24, 23, 20, 26, 25, 27, 21, 28, 23, 32, 29, 31, 25, 22, 28, 33, 31, 30, 27, 24,
   33, 29, 32, 26, 30, 33, 32, 31, 34, 39, 41, 37, 35, 42, 38, 40, 36, 39, 43, 49, 45, 37, 50, 41, 49, 44,
   47, 51, 38, 50, 42, 46, 51, 48, 40, 43, 45, 54, 44, 53, 47, 52, 46, 48, 53, 52, 51, 54, 50, 52, 49, 54,
   53, 55, 58, 60, 59, 56, 62, 61, 63, 57, 66, 59, 74, 67, 73, 61, 58, 64, 75, 73, 69, 63, 60, 75, 65, 74,
   62, 68, 64, 70, 78, 65, 76, 71, 70, 66, 80, 77, 68, 72, 71, 81, 67, 79, 72, 69, 75, 78, 76, 80, 74, 77,
   81, 79, 73, 82, 77, 79, 76, 82, 81, 78, 80, 82, 83, 88, 90, 86, 84, 91, 87, 89, 85, 88, 92, 104, 94, 86,
   105, 90, 104, 93, 96, 106, 87, 105, 91, 95, 106, 97, 89, 92, 100, 111, 93, 109, 102, 98, 94, 112, 99, 110, 96, 107,
   95, 103, 108, 101, 97, 100, 98, 113, 102, 114, 99, 115, 103, 101, 110, 108, 106, 112, 105, 107, 104, 111, 109, 109, 116, 114,
   117, 107, 115, 111, 113, 116, 118, 115, 108, 113, 112, 117, 114, 118, 110, 116, 117, 118, 119, 122, 124, 123, 120, 126, 125, 127,
   121, 130, 123, 144, 131, 143, 125, 122, 128, 145, 143, 133, 127, 124, 145, 129, 144, 126, 132, 128, 134, 148, 129, 146, 136, 135,
   130, 150, 147, 132, 138, 137, 151, 131, 149, 139, 133, 140, 135, 154, 141, 152, 137, 134, 140, 156, 153, 142, 139, 136, 157, 141,
   155, 138, 142, 145, 148, 146, 150, 144, 147, 151, 149, 143, 159, 147, 155, 160, 153, 149, 146, 158, 157, 152, 160, 151, 148, 156,
   158, 154, 150, 159, 156, 154, 161, 157, 162, 152, 163, 155, 153, 158, 161, 162, 161, 159, 163, 162, 163, 160, 164, 169, 171, 167,
   165, 172, 168, 170, 166, 169, 173, 191, 175, 167, 192, 171, 191, 174, 177, 193, 168, 192, 172, 176, 193, 178, 170, 173, 181, 198,
   174, 196, 183, 179, 175, 199, 180, 197, 177, 194, 176, 184, 195, 182, 178, 181, 185, 200, 187, 179, 202, 183, 201, 186, 189, 203,
   180, 204, 184, 188, 205, 190, 182, 185, 187, 208, 186, 207, 189, 206, 188, 190, 197, 195, 193, 199, 192, 194, 191, 198, 196, 196,
   209, 201, 210, 194, 204, 198, 200, 209, 211, 205, 195, 202, 199, 210, 203, 211, 197, 207, 213, 203, 215, 206, 205, 208, 202, 212,
   217, 204, 206, 200, 208, 214, 201, 216, 207, 209, 214, 216, 212, 210, 217, 213, 215, 211, 214, 212, 218, 216, 218, 213, 218, 217,
   215, 219, 222, 224, 223, 220, 226, 225, 227, 221, 230, 223, 250, 231, 249, 225, 222, 228, 251, 249, 233, 227, 224, 251, 229, 250,
   226, 232, 228, 234, 254, 229, 252, 236, 235, 230, 256, 253, 232, 238, 237, 257, 231, 255, 239, 233, 242, 235, 260, 243, 258, 237,
   234, 240, 262, 259, 245, 239, 236, 263, 241, 261, 238, 244, 240, 246, 266, 241, 264, 247, 246, 242, 268, 265, 244, 248, 247, 269,
   243, 267, 248, 245, 251, 254, 252, 256, 250, 253, 257, 255, 249, 271, 253, 261, 272, 259, 255, 252, 270, 263, 258, 272, 257, 254,
   262, 270, 260, 256, 271, 262, 266, 273, 263, 275, 264, 268, 260, 274, 269, 276, 258, 277, 261, 265, 278, 267, 259, 281, 265, 267,
   264, 280, 269, 266, 268, 279, 270, 273, 275, 274, 271, 277, 276, 278, 272, 279, 274, 283, 280, 282, 276, 273, 279, 284, 282, 281,
   278, 275, 284, 280, 283, 277, 281, 284, 283, 282, 285, 290, 292, 288, 286, 293, 289, 291, 287, 290, 294, 318, 296, 288, 319, 292,
   318, 295, 298, 320, 289, 319, 293, 297, 320, 299, 291, 294, 302, 325, 295, 323, 304, 300, 296, 326, 301, 324, 298, 321, 297, 305,
   322, 303, 299, 302, 306, 327, 308, 300, 329, 304, 328, 307, 310, 330, 301, 331, 305, 309, 332, 311, 303, 306, 314, 337, 307, 335,
   316, 312, 308, 338, 313, 336, 310, 333, 309, 317, 334, 315, 311, 314, 312, 339, 316, 340, 313, 341, 317, 315, 324, 322, 320, 326,
   319, 321, 318, 325, 323, 323, 342, 328, 343, 321, 331, 325, 327, 342, 344, 332, 322, 329, 326, 343, 330, 344, 324, 336, 346, 330,
   348, 334, 332, 338, 329, 345, 350, 331, 333, 327, 337, 347, 328, 349, 335, 335, 352, 340, 354, 333, 341, 337, 339, 351, 356, 341,
   334, 339, 338, 353, 340, 355, 336, 342, 347, 349, 345, 343, 350, 346, 348, 344, 347, 351, 357, 353, 345, 358, 349, 357, 352, 355,
   359, 346, 358, 350, 354, 359, 356, 348, 351, 353, 362, 352, 361, 355, 360, 354, 356, 361, 360, 359, 362, 358, 360, 357, 362, 361,
   363, 366, 368, 367, 364, 370, 369, 371, 365, 374, 367, 400, 375, 399, 369, 366, 372, 401, 399, 377, 371, 368, 401, 373, 400, 370,
   376, 372, 378, 404, 373, 402, 380, 379, 374, 406, 403, 376, 382, 381, 407, 375, 405, 383, 377, 386, 379, 410, 387, 408, 381, 378,
   384, 412, 409, 389, 383, 380, 413, 385, 411, 382, 388, 384, 390, 416, 385, 414, 392, 391, 386, 418, 415, 388, 394, 393, 419, 387,
   417, 395, 389, 396, 391, 422, 397, 420, 393, 390, 396, 424, 421, 398, 395, 392, 425, 397, 423, 394, 398, 401, 404, 402, 406, 400,
   403, 407, 405, 399, 427, 403, 411, 428, 409, 405, 402, 426, 413, 408, 428, 407, 404, 412, 426, 410, 406, 427, 412, 416, 429, 413,
   431, 414, 418, 410, 430, 419, 432, 408, 433, 411, 415, 434, 417, 409, 439, 415, 423, 440, 421, 417, 414, 436, 425, 420, 438, 419,
   416, 424, 435, 422, 418, 437, 424, 422, 441, 425, 442, 420, 443, 423, 421, 426, 429, 431, 430, 427, 433, 432, 434, 428, 437, 430,
   445, 438, 444, 432, 429, 435, 446, 444, 440, 434, 431, 446, 436, 445, 433, 439, 435, 441, 449, 436, 447, 442, 441, 437, 451, 448,
   439, 443, 442, 452, 438, 450, 443, 440, 446, 449, 447, 451, 445, 448, 452, 450, 444, 453, 448, 450, 447, 453, 452, 449, 451, 453,
   454, 459, 461, 457, 455, 462, 458, 460, 456, 459, 463, 493, 465, 457, 494, 461, 493, 464, 467, 495, 458, 494, 462, 466, 495, 468,
   460, 463, 471, 500, 464, 498, 473, 469, 465, 501, 470, 499, 467, 496, 466, 474, 497, 472, 468, 471, 475, 502, 477, 469, 504, 473,
   503, 476, 479, 505, 470, 506, 474, 478, 507, 480, 472, 475, 483, 512, 476, 510, 485, 481, 477, 513, 482, 511, 479, 508, 478, 486,
   509, 484, 480, 483, 487, 514, 489, 481, 516, 485, 515, 488, 491, 517, 482, 518, 486, 490, 519, 492, 484, 487, 489, 522, 488, 521,
   491, 520, 490, 492, 499, 497, 495, 501, 494, 496, 493, 500, 498, 498, 523, 503, 524, 496, 506, 500, 502, 523, 525, 507, 497, 504,
   501, 524, 505, 525, 499, 511, 527, 505, 529, 509, 507, 513, 504, 526, 531, 506, 508, 502, 512, 528, 503, 530, 510, 510, 533, 515,
   535, 508, 518, 512, 514, 532, 537, 519, 509, 516, 513, 534, 517, 536, 511, 521, 539, 517, 541, 520, 519, 522, 516, 538, 543, 518,
   520, 514, 522, 540, 515, 542, 521, 523, 528, 530, 526, 524, 531, 527, 529, 525, 528, 532, 544, 534, 526, 545, 530, 544, 533, 536,
   546, 527, 545, 531, 535, 546, 537, 529, 532, 540, 551, 533, 549, 542, 538, 534, 552, 539, 550, 536, 547, 535, 543, 548, 541, 537,
   540, 538, 553, 542, 554, 539, 555, 543, 541, 550, 548, 546, 552, 545, 547, 544, 551, 549, 549, 556, 554, 557, 547, 555, 551, 553,
   556, 558, 555, 548, 553, 552, 557, 554, 558, 550, 556, 557, 558, 559, 562, 564, 563, 560, 566, 565, 567, 561, 570, 563, 602, 571,
   601, 565, 562, 568, 603, 601, 573, 567, 564, 603, 569, 602, 566, 572, 568, 574, 606, 569, 604, 576, 575, 570, 608, 605, 572, 578,
   577, 609, 571, 607, 579, 573, 582, 575, 612, 583, 610, 577, 574, 580, 614, 611, 585, 579, 576, 615, 581, 613, 578, 584, 580, 586,
   618, 581, 616, 588, 587, 582, 620, 617, 584, 590, 589, 621, 583, 619, 591, 585, 594, 587, 624, 595, 622, 589, 586, 592, 626, 623,
   597, 591, 588, 627, 593, 625, 590, 596, 592, 598, 630, 593, 628, 599, 598, 594, 632, 629, 596, 600, 599, 633, 595, 631, 600, 597,
   603, 606, 604, 608, 602, 605, 609, 607, 601, 635, 605, 613, 636, 611, 607, 604, 634, 615, 610, 636, 609, 606, 614, 634, 612, 608,
   635, 614, 618, 637, 615, 639, 616, 620, 612, 638, 621, 640, 610, 641, 613, 617, 642, 619, 611, 647, 617, 625, 648, 623, 619, 616,
   644, 627, 622, 646, 621, 618, 626, 643, 624, 620, 645, 626, 630, 649, 627, 651, 628, 632, 624, 650, 633, 652, 622, 653, 625, 629,
   654, 631, 623, 657, 629, 631, 628, 656, 633, 630, 632, 655, 634, 637, 639, 638, 635, 641, 640, 642, 636, 645, 638, 659, 646, 658,
   640, 637, 643, 660, 658, 648, 642, 639, 660, 644, 659, 641, 647, 643, 649, 663, 644, 661, 651, 650, 645, 665, 662, 647, 653, 652,
   666, 646, 664, 654, 648, 655, 650, 669, 656, 667, 652, 649, 655, 671, 668, 657, 654, 651, 672, 656, 670, 653, 657, 660, 663, 661,
   665, 659, 662, 666, 664, 658, 674, 662, 670, 675, 668, 664, 661, 673, 672, 667, 675, 666, 663, 671, 673, 669, 665, 674, 671, 669,
   676, 672, 677, 667, 678, 670, 668, 673, 676, 677, 676, 674, 678, 677, 678, 675
}; // 3.28 kb
static const cart_index_t iCartXY_ab14_a2[2730] = {
   0, 1, 2, 3, 4, 5, 6, 9, 10, 11, 13, 15, 11, 7, 12, 9, 15, 14, 13, 14, 8, 15, 10, 12, 16, 25,
   26, 19, 21, 30, 25, 17, 27, 20, 29, 23, 26, 27, 18, 28, 22, 24, 19, 20, 28, 25, 30, 29, 21, 29, 22, 30,
   26, 28, 30, 23, 24, 29, 28, 27, 31, 40, 41, 36, 38, 46, 42, 32, 43, 34, 47, 39, 44, 45, 33, 48, 35, 37,
   40, 34, 49, 42, 51, 47, 41, 49, 35, 50, 44, 48, 36, 42, 50, 40, 46, 51, 50, 43, 37, 49, 48, 45, 38, 51,
   44, 46, 41, 50, 51, 39, 45, 47, 49, 43, 46, 47, 48, 51, 50, 49, 52, 61, 62, 55, 57, 72, 63, 53, 65, 56,
   71, 59, 64, 66, 54, 70, 58, 60, 55, 67, 73, 61, 72, 75, 67, 56, 74, 63, 77, 71, 57, 75, 68, 72, 62, 73,
   68, 76, 58, 78, 64, 70, 77, 59, 69, 71, 74, 65, 78, 69, 60, 76, 70, 66, 61, 63, 79, 67, 75, 77, 62, 79,
   64, 73, 68, 78, 79, 65, 66, 74, 76, 69, 73, 74, 70, 79, 78, 76, 75, 71, 76, 77, 79, 74, 72, 77, 78, 75,
   73, 79, 80, 89, 90, 85, 87, 101, 91, 81, 92, 83, 102, 88, 93, 94, 82, 103, 84, 86, 95, 83, 104, 91, 109, 102,
   96, 105, 84, 107, 93, 103, 85, 97, 106, 89, 101, 108, 107, 98, 86, 105, 103, 94, 87, 108, 99, 101, 90, 106, 109, 88,
   100, 102, 104, 92, 89, 95, 113, 97, 108, 110, 90, 113, 96, 106, 99, 111, 97, 91, 114, 95, 110, 109, 114, 92, 98, 104,
   112, 100, 99, 115, 93, 111, 96, 107, 115, 100, 94, 112, 105, 98, 101, 110, 111, 108, 106, 113, 110, 102, 112, 109, 114, 104,
   111, 112, 103, 115, 107, 105, 113, 104, 105, 114, 115, 112, 106, 114, 107, 113, 111, 115, 108, 109, 115, 110, 113, 114, 116, 125,
   126, 119, 121, 142, 127, 117, 129, 120, 141, 123, 128, 130, 118, 140, 122, 124, 119, 131, 143, 125, 142, 145, 132, 120, 144, 127,
   147, 141, 121, 145, 133, 142, 126, 143, 134, 146, 122, 148, 128, 140, 147, 123, 135, 141, 144, 129, 148, 136, 124, 146, 140, 130,
   125, 137, 155, 131, 145, 153, 126, 155, 138, 143, 133, 154, 137, 127, 156, 132, 151, 147, 138, 157, 128, 149, 134, 148, 156, 129,
   139, 144, 152, 135, 157, 139, 130, 150, 146, 136, 131, 132, 158, 137, 153, 151, 133, 159, 134, 154, 138, 149, 160, 135, 136, 152,
   150, 139, 149, 150, 140, 157, 148, 146, 151, 141, 152, 147, 156, 144, 142, 153, 154, 145, 143, 155, 143, 158, 149, 155, 154, 159,
   158, 144, 150, 156, 160, 152, 145, 151, 159, 153, 155, 158, 159, 152, 146, 160, 157, 150, 153, 147, 160, 151, 158, 156, 154, 160,
   148, 159, 149, 157, 155, 156, 157, 158, 159, 160, 161, 170, 171, 166, 168, 188, 172, 162, 173, 164, 189, 169, 174, 175, 163, 190,
   165, 167, 176, 164, 191, 172, 196, 189, 177, 192, 165, 194, 174, 190, 166, 178, 193, 170, 188, 195, 194, 179, 167, 192, 190, 175,
   168, 195, 180, 188, 171, 193, 196, 169, 181, 189, 191, 173, 170, 182, 206, 178, 195, 197, 171, 206, 183, 193, 180, 198, 184, 172,
   207, 176, 199, 196, 207, 173, 185, 191, 201, 181, 186, 208, 174, 200, 177, 194, 208, 187, 175, 202, 192, 179, 182, 176, 209, 184,
   205, 199, 183, 210, 177, 204, 186, 200, 178, 184, 211, 182, 197, 205, 212, 185, 179, 203, 202, 187, 180, 213, 186, 198, 183, 204,
   214, 181, 187, 201, 203, 185, 188, 197, 198, 195, 193, 206, 199, 189, 201, 196, 207, 191, 200, 202, 190, 208, 194, 192, 209, 191,
   203, 207, 214, 201, 210, 203, 192, 212, 208, 202, 193, 211, 204, 206, 198, 213, 204, 212, 194, 210, 200, 208, 195, 205, 213, 197,
   206, 211, 205, 196, 214, 199, 209, 207, 197, 199, 215, 205, 211, 209, 198, 215, 200, 213, 204, 210, 215, 201, 202, 214, 212, 203,
   206, 209, 210, 211, 213, 215, 211, 207, 212, 209, 215, 214, 213, 214, 208, 215, 210, 212, 216, 225, 226, 219, 221, 248, 227, 217,
   229, 220, 247, 223, 228, 230, 218, 246, 222, 224, 219, 231, 249, 225, 248, 251, 232, 220, 250, 227, 253, 247, 221, 251, 233, 248,
   226, 249, 234, 252, 222, 254, 228, 246, 253, 223, 235, 247, 250, 229, 254, 236, 224, 252, 246, 230, 225, 237, 267, 231, 251, 259,
   226, 267, 238, 249, 233, 260, 239, 227, 268, 232, 257, 253, 240, 269, 228, 255, 234, 254, 268, 229, 241, 250, 258, 235, 269, 242,
   230, 256, 252, 236, 231, 243, 270, 237, 259, 263, 243, 232, 271, 239, 265, 257, 233, 272, 244, 260, 238, 261, 244, 273, 234, 266,
   240, 255, 274, 235, 245, 258, 262, 241, 275, 245, 236, 264, 256, 242, 237, 239, 276, 243, 263, 265, 238, 277, 240, 261, 244, 266,
   278, 241, 242, 262, 264, 245, 255, 256, 246, 269, 254, 252, 257, 247, 258, 253, 268, 250, 248, 259, 260, 251, 249, 267, 249, 270,
   261, 267, 260, 272, 271, 250, 262, 268, 274, 258, 251, 263, 272, 259, 267, 270, 273, 264, 252, 275, 269, 256, 265, 253, 274, 257,
   271, 268, 266, 275, 254, 273, 255, 269, 261, 279, 255, 277, 266, 273, 279, 262, 256, 278, 275, 264, 263, 257, 280, 265, 276, 271,
   280, 258, 264, 274, 278, 262, 259, 265, 281, 263, 270, 276, 260, 281, 266, 272, 261, 277, 267, 276, 277, 270, 272, 281, 276, 268,
   278, 271, 280, 274, 277, 278, 269, 279, 273, 275, 270, 271, 279, 276, 281, 280, 272, 280, 273, 281, 277, 279, 281, 274, 275, 280,
   279, 278, 282, 291, 292, 287, 289, 315, 293, 283, 294, 285, 316, 290, 295, 296, 284, 317, 286, 288, 297, 285, 318, 293, 323, 316,
   298, 319, 286, 321, 295, 317, 287, 299, 320, 291, 315, 322, 321, 300, 288, 319, 317, 296, 289, 322, 301, 315, 292, 320, 323, 290,
   302, 316, 318, 294, 291, 303, 339, 299, 322, 324, 292, 339, 304, 320, 301, 325, 305, 293, 340, 297, 326, 323, 340, 294, 306, 318,
   328, 302, 307, 341, 295, 327, 298, 321, 341, 308, 296, 329, 319, 300, 309, 297, 342, 305, 335, 326, 310, 343, 298, 333, 307, 327,
   299, 311, 344, 303, 324, 334, 345, 312, 300, 331, 329, 308, 301, 346, 313, 325, 304, 332, 347, 302, 314, 328, 330, 306, 303, 309,
   348, 311, 334, 336, 304, 349, 310, 332, 313, 337, 311, 305, 350, 309, 336, 335, 351, 306, 312, 330, 338, 314, 313, 352, 307, 337,
   310, 333, 353, 314, 308, 338, 331, 312, 315, 324, 325, 322, 320, 339, 326, 316, 328, 323, 340, 318, 327, 329, 317, 341, 321, 319,
   342, 318, 330, 340, 347, 328, 343, 331, 319, 345, 341, 329, 320, 344, 332, 339, 325, 346, 333, 345, 321, 343, 327, 341, 322, 334,
   346, 324, 339, 344, 335, 323, 347, 326, 342, 340, 324, 336, 354, 334, 344, 348, 325, 354, 337, 346, 332, 349, 336, 326, 355, 335,
   350, 342, 337, 356, 327, 352, 333, 343, 355, 328, 338, 347, 351, 330, 356, 338, 329, 353, 345, 331, 357, 330, 331, 351, 353, 338,
   332, 358, 333, 349, 337, 352, 334, 335, 359, 336, 348, 350, 339, 348, 349, 344, 346, 354, 350, 340, 351, 342, 355, 347, 352, 353,
   341, 356, 343, 345, 348, 342, 357, 350, 359, 355, 349, 357, 343, 358, 352, 356, 344, 350, 358, 348, 354, 359, 358, 351, 345, 357,
   356, 353, 346, 359, 352, 354, 349, 358, 359, 347, 353, 355, 357, 351, 354, 355, 356, 359, 358, 357, 360, 369, 370, 363, 365, 398,
   371, 361, 373, 364, 397, 367, 372, 374, 362, 396, 366, 368, 363, 375, 399, 369, 398, 401, 376, 364, 400, 371, 403, 397, 365, 401,
   377, 398, 370, 399, 378, 402, 366, 404, 372, 396, 403, 367, 379, 397, 400, 373, 404, 380, 368, 402, 396, 374, 369, 381, 423, 375,
   401, 409, 370, 423, 382, 399, 377, 410, 383, 371, 424, 376, 407, 403, 384, 425, 372, 405, 378, 404, 424, 373, 385, 400, 408, 379,
   425, 386, 374, 406, 402, 380, 375, 387, 426, 381, 409, 413, 388, 376, 427, 383, 415, 407, 377, 428, 389, 410, 382, 411, 390, 429,
   378, 416, 384, 405, 430, 379, 391, 408, 412, 385, 431, 392, 380, 414, 406, 386, 381, 393, 432, 387, 413, 421, 382, 433, 394, 411,
   389, 422, 393, 383, 434, 388, 419, 415, 394, 435, 384, 417, 390, 416, 436, 385, 395, 412, 420, 391, 437, 395, 386, 418, 414, 392,
   387, 388, 438, 393, 421, 419, 389, 439, 390, 422, 394, 417, 440, 391, 392, 420, 418, 395, 405, 406, 396, 425, 404, 402, 407, 397,
   408, 403, 424, 400, 398, 409, 410, 401, 399, 423, 399, 426, 411, 423, 410, 428, 427, 400, 412, 424, 430, 408, 401, 413, 428, 409,
   423, 426, 429, 414, 402, 431, 425, 406, 415, 403, 430, 407, 427, 424, 416, 431, 404, 429, 405, 425, 417, 441, 405, 435, 416, 429,
   441, 418, 406, 437, 431, 414, 419, 407, 442, 415, 434, 427, 442, 408, 420, 430, 436, 412, 409, 421, 443, 413, 426, 432, 410, 443,
   422, 428, 411, 433, 411, 444, 417, 433, 422, 439, 445, 412, 418, 436, 440, 420, 413, 419, 446, 421, 432, 438, 447, 420, 414, 440,
   437, 418, 421, 415, 448, 419, 438, 434, 422, 449, 416, 439, 417, 435, 423, 432, 433, 426, 428, 443, 434, 424, 436, 427, 442, 430,
   435, 437, 425, 441, 429, 431, 426, 438, 444, 432, 443, 446, 438, 427, 445, 434, 448, 442, 428, 446, 439, 443, 433, 444, 439, 447,
   429, 449, 435, 441, 448, 430, 440, 442, 445, 436, 449, 440, 431, 447, 441, 437, 432, 434, 450, 438, 446, 448, 433, 450, 435, 444,
   439, 449, 450, 436, 437, 445, 447, 440, 444, 445, 441, 450, 449, 447, 446, 442, 447, 448, 450, 445, 443, 448, 449, 446, 444, 450,
   451, 460, 461, 456, 458, 490, 462, 452, 463, 454, 491, 459, 464, 465, 453, 492, 455, 457, 466, 454, 493, 462, 498, 491, 467, 494,
   455, 496, 464, 492, 456, 468, 495, 460, 490, 497, 496, 469, 457, 494, 492, 465, 458, 497, 470, 490, 461, 495, 498, 459, 471, 491,
   493, 463, 460, 472, 520, 468, 497, 499, 461, 520, 473, 495, 470, 500, 474, 462, 521, 466, 501, 498, 521, 463, 475, 493, 503, 471,
   476, 522, 464, 502, 467, 496, 522, 477, 465, 504, 494, 469, 478, 466, 523, 474, 510, 501, 479, 524, 467, 508, 476, 502, 468, 480,
   525, 472, 499, 509, 526, 481, 469, 506, 504, 477, 470, 527, 482, 500, 473, 507, 528, 471, 483, 503, 505, 475, 472, 484, 529, 480,
   509, 511, 473, 530, 485, 507, 482, 512, 486, 474, 531, 478, 513, 510, 532, 475, 487, 505, 515, 483, 488, 533, 476, 514, 479, 508,
   534, 489, 477, 516, 506, 481, 484, 478, 535, 486, 519, 513, 485, 536, 479, 518, 488, 514, 480, 486, 537, 484, 511, 519, 538, 487,
   481, 517, 516, 489, 482, 539, 488, 512, 485, 518, 540, 483, 489, 515, 517, 487, 490, 499, 500, 497, 495, 520, 501, 491, 503, 498,
   521, 493, 502, 504, 492, 522, 496, 494, 523, 493, 505, 521, 528, 503, 524, 506, 494, 526, 522, 504, 495, 525, 507, 520, 500, 527,
   508, 526, 496, 524, 502, 522, 497, 509, 527, 499, 520, 525, 510, 498, 528, 501, 523, 521, 499, 511, 541, 509, 525, 529, 500, 541,
   512, 527, 507, 530, 513, 501, 542, 510, 531, 523, 514, 543, 502, 533, 508, 524, 542, 503, 515, 528, 532, 505, 543, 516, 504, 534,
   526, 506, 544, 505, 517, 532, 540, 515, 545, 517, 506, 538, 534, 516, 507, 546, 518, 530, 512, 539, 518, 547, 508, 536, 514, 533,
   509, 519, 548, 511, 529, 537, 519, 510, 549, 513, 535, 531, 511, 513, 550, 519, 537, 535, 512, 551, 514, 539, 518, 536, 552, 515,
   516, 540, 538, 517, 520, 529, 530, 525, 527, 541, 531, 521, 532, 523, 542, 528, 533, 534, 522, 543, 524, 526, 535, 523, 544, 531,
   549, 542, 536, 545, 524, 547, 533, 543, 525, 537, 546, 529, 541, 548, 547, 538, 526, 545, 543, 534, 527, 548, 539, 541, 530, 546,
   549, 528, 540, 542, 544, 532, 529, 535, 553, 537, 548, 550, 530, 553, 536, 546, 539, 551, 537, 531, 554, 535, 550, 549, 554, 532,
   538, 544, 552, 540, 539, 555, 533, 551, 536, 547, 555, 540, 534, 552, 545, 538, 541, 550, 551, 548, 546, 553, 550, 542, 552, 549,
   554, 544, 551, 552, 543, 555, 547, 545, 553, 544, 545, 554, 555, 552, 546, 554, 547, 553, 551, 555, 548, 549, 555, 550, 553, 554,
   556, 565, 566, 559, 561, 600, 567, 557, 569, 560, 599, 563, 568, 570, 558, 598, 562, 564, 559, 571, 601, 565, 600, 603, 572, 560,
   602, 567, 605, 599, 561, 603, 573, 600, 566, 601, 574, 604, 562, 606, 568, 598, 605, 563, 575, 599, 602, 569, 606, 576, 564, 604,
   598, 570, 565, 577, 631, 571, 603, 611, 566, 631, 578, 601, 573, 612, 579, 567, 632, 572, 609, 605, 580, 633, 568, 607, 574, 606,
   632, 569, 581, 602, 610, 575, 633, 582, 570, 608, 604, 576, 571, 583, 634, 577, 611, 615, 584, 572, 635, 579, 617, 609, 573, 636,
   585, 612, 578, 613, 586, 637, 574, 618, 580, 607, 638, 575, 587, 610, 614, 581, 639, 588, 576, 616, 608, 582, 577, 589, 640, 583,
   615, 623, 578, 641, 590, 613, 585, 624, 591, 579, 642, 584, 621, 617, 592, 643, 580, 619, 586, 618, 644, 581, 593, 614, 622, 587,
   645, 594, 582, 620, 616, 588, 583, 595, 646, 589, 623, 627, 595, 584, 647, 591, 629, 621, 585, 648, 596, 624, 590, 625, 596, 649,
   586, 630, 592, 619, 650, 587, 597, 622, 626, 593, 651, 597, 588, 628, 620, 594, 589, 591, 652, 595, 627, 629, 590, 653, 592, 625,
   596, 630, 654, 593, 594, 626, 628, 597, 607, 608, 598, 633, 606, 604, 609, 599, 610, 605, 632, 602, 600, 611, 612, 603, 601, 631,
   601, 634, 613, 631, 612, 636, 635, 602, 614, 632, 638, 610, 603, 615, 636, 611, 631, 634, 637, 616, 604, 639, 633, 608, 617, 605,
   638, 609, 635, 632, 618, 639, 606, 637, 607, 633, 619, 655, 607, 643, 618, 637, 655, 620, 608, 645, 639, 616, 621, 609, 656, 617,
   642, 635, 656, 610, 622, 638, 644, 614, 611, 623, 657, 615, 634, 640, 612, 657, 624, 636, 613, 641, 613, 658, 625, 641, 624, 648,
   659, 614, 626, 644, 650, 622, 615, 627, 660, 623, 640, 646, 661, 628, 616, 651, 645, 620, 629, 617, 662, 621, 647, 642, 630, 663,
   618, 649, 619, 643, 625, 664, 619, 653, 630, 649, 665, 626, 620, 654, 651, 628, 627, 621, 666, 629, 652, 647, 667, 622, 628, 650,
   654, 626, 623, 629, 668, 627, 646, 652, 624, 669, 630, 648, 625, 653, 631, 640, 641, 634, 636, 657, 642, 632, 644, 635, 656, 638,
   643, 645, 633, 655, 637, 639, 634, 646, 658, 640, 657, 660, 647, 635, 659, 642, 662, 656, 636, 660, 648, 657, 641, 658, 649, 661,
   637, 663, 643, 655, 662, 638, 650, 656, 659, 644, 663, 651, 639, 661, 655, 645, 640, 652, 670, 646, 660, 668, 641, 670, 653, 658,
   648, 669, 652, 642, 671, 647, 666, 662, 653, 672, 643, 664, 649, 663, 671, 644, 654, 659, 667, 650, 672, 654, 645, 665, 661, 651,
   646, 647, 673, 652, 668, 666, 648, 674, 649, 669, 653, 664, 675, 650, 651, 667, 665, 654, 664, 665, 655, 672, 663, 661, 666, 656,
   667, 662, 671, 659, 657, 668, 669, 660, 658, 670, 658, 673, 664, 670, 669, 674, 673, 659, 665, 671, 675, 667, 660, 666, 674, 668,
   670, 673, 674, 667, 661, 675, 672, 665, 668, 662, 675, 666, 673, 671, 669, 675, 663, 674, 664, 672, 670, 671, 672, 673, 674, 675
}; // 5.33 kb
static const cart_index_t iCartXY_ab14_a3[3640] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 14, 16, 19, 20, 13, 22, 15, 23, 24, 13, 11, 18, 14, 22, 19,
   21, 24, 17, 23, 15, 17, 12, 23, 16, 24, 18, 20, 21, 22, 25, 36, 38, 34, 35, 30, 44, 32, 45, 40, 34, 26,
   39, 28, 43, 36, 37, 45, 33, 41, 35, 37, 27, 43, 29, 44, 31, 38, 39, 42, 30, 28, 42, 36, 44, 34, 43, 40,
   41, 45, 32, 41, 29, 45, 38, 40, 42, 35, 43, 44, 40, 33, 31, 41, 42, 45, 39, 44, 37, 43, 46, 61, 62, 55,
   56, 49, 67, 51, 69, 66, 61, 47, 63, 50, 68, 57, 59, 71, 53, 65, 62, 63, 48, 70, 52, 72, 54, 58, 60, 64,
   55, 50, 70, 57, 73, 61, 68, 69, 65, 71, 56, 68, 52, 73, 58, 67, 64, 62, 70, 72, 49, 57, 72, 61, 67, 55,
   73, 66, 71, 69, 67, 59, 54, 68, 64, 73, 60, 72, 63, 70, 51, 71, 58, 69, 62, 66, 72, 56, 73, 67, 69, 53,
   60, 65, 70, 71, 63, 73, 59, 68, 66, 65, 64, 71, 72, 69, 70, 67, 68, 73, 74, 91, 93, 83, 84, 79, 100, 81,
   102, 95, 89, 75, 94, 77, 98, 85, 86, 103, 82, 96, 90, 92, 76, 99, 78, 101, 80, 87, 88, 97, 79, 89, 105, 91,
   100, 83, 107, 95, 104, 102, 91, 77, 106, 85, 108, 89, 98, 104, 96, 103, 81, 104, 90, 102, 93, 95, 105, 84, 107, 100,
   93, 106, 78, 109, 87, 105, 97, 90, 99, 101, 104, 82, 92, 96, 106, 103, 94, 108, 86, 98, 105, 94, 80, 106, 97, 109,
   88, 101, 92, 99, 83, 85, 109, 89, 107, 91, 108, 102, 103, 104, 84, 108, 87, 107, 90, 100, 101, 93, 109, 105, 107, 86,
   88, 98, 99, 108, 92, 109, 94, 106, 100, 98, 97, 108, 101, 107, 99, 105, 106, 109, 102, 96, 99, 103, 109, 104, 106, 107,
   98, 108, 95, 103, 101, 104, 105, 102, 109, 100, 108, 107, 110, 125, 127, 119, 120, 113, 137, 115, 139, 136, 126, 111, 129, 114,
   138, 121, 123, 141, 117, 135, 128, 130, 112, 140, 116, 142, 118, 122, 124, 134, 131, 114, 146, 121, 150, 126, 138, 145, 135, 141,
   132, 144, 116, 151, 122, 143, 134, 128, 140, 142, 113, 131, 148, 125, 137, 119, 149, 136, 147, 139, 143, 133, 118, 144, 134, 151,
   124, 142, 130, 140, 115, 147, 132, 139, 127, 136, 148, 120, 149, 137, 145, 117, 133, 135, 146, 141, 129, 150, 123, 138, 119, 126,
   153, 131, 149, 125, 152, 139, 145, 147, 120, 152, 128, 149, 132, 137, 143, 127, 153, 148, 125, 121, 154, 126, 152, 131, 150, 147,
   141, 145, 152, 123, 130, 138, 144, 150, 133, 154, 129, 146, 127, 154, 122, 153, 128, 148, 142, 132, 151, 143, 153, 129, 124, 146,
   140, 154, 130, 151, 133, 144, 136, 145, 143, 147, 148, 139, 153, 137, 152, 149, 147, 135, 144, 141, 154, 145, 146, 152, 138, 150,
   148, 146, 134, 154, 142, 153, 140, 143, 144, 151, 149, 138, 140, 150, 151, 152, 144, 153, 146, 154, 137, 150, 142, 152, 143, 149,
   151, 148, 154, 153, 139, 141, 151, 145, 153, 147, 154, 149, 150, 152, 155, 172, 174, 164, 165, 160, 187, 162, 189, 182, 170, 156,
   175, 158, 185, 166, 167, 190, 163, 183, 171, 173, 157, 186, 159, 188, 161, 168, 169, 184, 160, 176, 192, 172, 187, 164, 200, 182,
   191, 189, 178, 158, 195, 166, 201, 170, 185, 193, 183, 190, 162, 191, 177, 189, 174, 182, 192, 165, 200, 187, 180, 196, 159, 202,
   168, 194, 184, 171, 186, 188, 193, 163, 179, 183, 195, 190, 175, 201, 167, 185, 194, 181, 161, 196, 184, 202, 169, 188, 173, 186,
   164, 178, 207, 176, 200, 172, 205, 189, 199, 191, 165, 205, 180, 200, 177, 187, 198, 174, 207, 192, 176, 166, 208, 170, 203, 178,
   201, 199, 190, 193, 177, 206, 168, 204, 171, 198, 188, 180, 202, 194, 203, 167, 181, 185, 197, 201, 179, 208, 175, 195, 204, 179,
   169, 197, 186, 206, 173, 202, 181, 196, 172, 170, 209, 178, 205, 176, 203, 191, 193, 199, 174, 209, 171, 207, 180, 192, 194, 177,
   204, 198, 209, 175, 173, 195, 196, 208, 181, 206, 179, 197, 198, 197, 184, 206, 188, 204, 186, 194, 196, 202, 199, 183, 197, 190,
   208, 193, 195, 203, 185, 201, 182, 199, 198, 191, 192, 189, 207, 187, 205, 200, 187, 203, 194, 205, 198, 200, 204, 192, 209, 207,
   205, 185, 196, 201, 206, 203, 197, 209, 195, 208, 189, 193, 204, 199, 207, 191, 209, 200, 203, 205, 207, 195, 186, 208, 202, 209,
   196, 204, 197, 206, 191, 190, 206, 193, 209, 199, 208, 205, 201, 203, 192, 208, 188, 209, 194, 207, 202, 198, 206, 204, 200, 201,
   202, 203, 204, 205, 206, 207, 208, 209, 210, 225, 227, 219, 220, 213, 243, 215, 245, 242, 226, 211, 229, 214, 244, 221, 223, 247,
   217, 241, 228, 230, 212, 246, 216, 248, 218, 222, 224, 240, 233, 214, 252, 221, 262, 226, 244, 251, 241, 247, 234, 250, 216, 263,
   222, 249, 240, 228, 246, 248, 213, 231, 254, 225, 243, 219, 261, 242, 253, 245, 249, 236, 218, 250, 240, 263, 224, 248, 230, 246,
   215, 253, 232, 245, 227, 242, 254, 220, 261, 243, 251, 217, 235, 241, 252, 247, 229, 262, 223, 244, 219, 237, 266, 231, 261, 225,
   264, 245, 257, 253, 220, 264, 238, 261, 232, 243, 255, 227, 266, 254, 237, 221, 268, 226, 265, 233, 262, 259, 247, 251, 265, 223,
   239, 244, 256, 262, 235, 268, 229, 252, 238, 269, 222, 267, 228, 260, 248, 234, 263, 249, 267, 239, 224, 258, 246, 269, 230, 263,
   236, 250, 231, 226, 274, 233, 270, 237, 265, 257, 251, 259, 232, 273, 228, 271, 234, 255, 249, 238, 267, 260, 225, 233, 275, 237,
   264, 231, 270, 253, 259, 257, 273, 235, 230, 256, 250, 272, 236, 269, 239, 258, 227, 275, 234, 266, 238, 254, 260, 232, 271, 255,
   274, 229, 236, 252, 258, 268, 239, 272, 235, 256, 242, 257, 255, 253, 254, 245, 266, 243, 264, 261, 259, 241, 256, 247, 268, 251,
   252, 265, 244, 262, 260, 258, 240, 269, 248, 267, 246, 249, 250, 263, 270, 244, 258, 262, 272, 265, 256, 274, 252, 268, 271, 256,
   246, 272, 263, 273, 250, 267, 258, 269, 243, 270, 260, 264, 255, 261, 271, 254, 275, 266, 255, 272, 248, 273, 249, 271, 263, 260,
   269, 267, 245, 259, 271, 257, 266, 253, 275, 261, 270, 264, 257, 247, 272, 251, 274, 259, 268, 270, 262, 265, 253, 251, 273, 259,
   275, 257, 274, 264, 265, 270, 254, 274, 249, 275, 260, 266, 267, 255, 273, 271, 275, 252, 250, 268, 269, 274, 258, 273, 256, 272,
   261, 265, 267, 270, 271, 264, 273, 266, 274, 275, 264, 262, 269, 265, 273, 270, 272, 275, 268, 274, 266, 268, 263, 274, 267, 275,
   269, 271, 272, 273, 276, 293, 295, 285, 286, 281, 314, 283, 316, 309, 291, 277, 296, 279, 312, 287, 288, 317, 284, 310, 292, 294,
   278, 313, 280, 315, 282, 289, 290, 311, 281, 297, 319, 293, 314, 285, 333, 309, 318, 316, 299, 279, 322, 287, 334, 291, 312, 320,
   310, 317, 283, 318, 298, 316, 295, 309, 319, 286, 333, 314, 301, 323, 280, 335, 289, 321, 311, 292, 313, 315, 320, 284, 300, 310,
   322, 317, 296, 334, 288, 312, 321, 302, 282, 323, 311, 335, 290, 315, 294, 313, 285, 305, 340, 297, 333, 293, 338, 316, 328, 318,
   286, 338, 307, 333, 298, 314, 326, 295, 340, 319, 303, 287, 341, 291, 336, 299, 334, 329, 317, 320, 304, 339, 289, 337, 292, 327,
   315, 301, 335, 321, 336, 288, 308, 312, 324, 334, 300, 341, 296, 322, 337, 306, 290, 325, 313, 339, 294, 335, 302, 323, 293, 303,
   348, 305, 338, 297, 342, 318, 330, 328, 305, 291, 349, 299, 344, 303, 336, 330, 320, 329, 295, 348, 304, 340, 307, 319, 331, 298,
   343, 326, 307, 350, 292, 346, 301, 331, 321, 304, 337, 327, 349, 296, 306, 322, 332, 341, 308, 345, 300, 324, 350, 308, 294, 332,
   323, 347, 302, 339, 306, 325, 297, 299, 353, 303, 342, 305, 344, 328, 329, 330, 298, 352, 301, 343, 304, 326, 327, 307, 346, 331,
   351, 300, 302, 324, 325, 345, 306, 347, 308, 332, 327, 325, 311, 339, 315, 337, 313, 321, 323, 335, 329, 310, 324, 317, 341, 320,
   322, 336, 312, 334, 309, 328, 326, 318, 319, 316, 340, 314, 338, 333, 314, 342, 331, 338, 326, 333, 343, 319, 348, 340, 344, 312,
   332, 334, 345, 336, 324, 349, 322, 341, 316, 330, 343, 328, 340, 318, 348, 333, 342, 338, 346, 332, 313, 347, 335, 350, 323, 337,
   325, 339, 330, 317, 345, 320, 349, 329, 341, 344, 334, 336, 331, 347, 315, 350, 321, 346, 335, 327, 339, 337, 326, 351, 321, 352,
   327, 343, 337, 331, 350, 346, 352, 324, 323, 345, 339, 351, 325, 350, 332, 347, 328, 320, 351, 329, 353, 330, 349, 342, 336, 344,
   353, 322, 325, 341, 347, 349, 332, 351, 324, 345, 318, 329, 352, 330, 348, 328, 353, 338, 344, 342, 319, 353, 327, 348, 331, 340,
   346, 326, 352, 343, 333, 344, 346, 342, 343, 338, 352, 340, 353, 348, 342, 334, 347, 336, 351, 344, 345, 353, 341, 349, 343, 345,
   335, 351, 337, 352, 339, 346, 347, 350, 338, 336, 350, 344, 352, 342, 351, 348, 349, 353, 340, 349, 337, 353, 346, 348, 350, 343,
   351, 352, 348, 341, 339, 349, 350, 353, 347, 352, 345, 351, 354, 369, 371, 363, 364, 357, 393, 359, 395, 392, 370, 355, 373, 358,
   394, 365, 367, 397, 361, 391, 372, 374, 356, 396, 360, 398, 362, 366, 368, 390, 377, 358, 402, 365, 418, 370, 394, 401, 391, 397,
   378, 400, 360, 419, 366, 399, 390, 372, 396, 398, 357, 375, 404, 369, 393, 363, 417, 392, 403, 395, 399, 380, 362, 400, 390, 419,
   368, 398, 374, 396, 359, 403, 376, 395, 371, 392, 404, 364, 417, 393, 401, 361, 379, 391, 402, 397, 373, 418, 367, 394, 363, 381,
   422, 375, 417, 369, 420, 395, 407, 403, 364, 420, 383, 417, 376, 393, 405, 371, 422, 404, 382, 365, 424, 370, 421, 377, 418, 409,
   397, 401, 421, 367, 385, 394, 406, 418, 379, 424, 373, 402, 384, 425, 366, 423, 372, 410, 398, 378, 419, 399, 423, 386, 368, 408,
   396, 425, 374, 419, 380, 400, 387, 370, 436, 377, 428, 382, 421, 413, 401, 409, 388, 435, 372, 429, 378, 411, 399, 384, 423, 410,
   369, 387, 437, 381, 420, 375, 426, 403, 415, 407, 435, 389, 374, 412, 400, 431, 380, 425, 386, 408, 371, 437, 388, 422, 383, 404,
   416, 376, 427, 405, 436, 373, 389, 402, 414, 424, 385, 430, 379, 406, 375, 382, 440, 387, 426, 381, 432, 407, 413, 415, 376, 438,
   384, 427, 388, 405, 411, 383, 433, 416, 381, 377, 442, 382, 432, 387, 428, 415, 409, 413, 439, 379, 386, 406, 412, 430, 389, 434,
   385, 414, 383, 443, 378, 433, 384, 416, 410, 388, 429, 411, 441, 385, 380, 414, 408, 434, 386, 431, 389, 412, 392, 407, 405, 403,
   404, 395, 422, 393, 420, 417, 409, 391, 406, 397, 424, 401, 402, 421, 394, 418, 410, 408, 390, 425, 398, 423, 396, 399, 400, 419,
   428, 394, 414, 418, 430, 421, 406, 436, 402, 424, 429, 412, 396, 431, 419, 435, 400, 423, 408, 425, 393, 426, 416, 420, 405, 417,
   427, 404, 437, 422, 411, 431, 398, 435, 399, 429, 419, 410, 425, 423, 395, 415, 427, 407, 422, 403, 437, 417, 426, 420, 413, 397,
   430, 401, 436, 409, 424, 428, 418, 421, 403, 413, 438, 415, 437, 407, 440, 420, 432, 426, 404, 440, 411, 437, 416, 422, 433, 405,
   438, 427, 415, 401, 439, 409, 442, 413, 436, 432, 421, 428, 416, 441, 399, 443, 410, 433, 423, 411, 435, 429, 442, 402, 412, 424,
   434, 436, 414, 439, 406, 430, 443, 414, 400, 434, 425, 441, 408, 435, 412, 431, 444, 406, 408, 430, 431, 439, 412, 441, 414, 434,
   405, 444, 410, 438, 411, 427, 429, 416, 443, 433, 407, 409, 444, 413, 440, 415, 442, 426, 428, 432, 417, 432, 433, 426, 427, 420,
   438, 422, 440, 437, 432, 418, 434, 421, 439, 428, 430, 442, 424, 436, 433, 434, 419, 441, 423, 443, 425, 429, 431, 435, 426, 421,
   441, 428, 444, 432, 439, 440, 436, 442, 427, 439, 423, 444, 429, 438, 435, 433, 441, 443, 420, 428, 443, 432, 438, 426, 444, 437,
   442, 440, 438, 430, 425, 439, 435, 444, 431, 443, 434, 441, 422, 442, 429, 440, 433, 437, 443, 427, 444, 438, 440, 424, 431, 436,
   441, 442, 434, 444, 430, 439, 437, 436, 435, 442, 443, 440, 441, 438, 439, 444, 445, 462, 464, 454, 455, 450, 489, 452, 491, 484,
   460, 446, 465, 448, 487, 456, 457, 492, 453, 485, 461, 463, 447, 488, 449, 490, 451, 458, 459, 486, 450, 466, 494, 462, 489, 454,
   514, 484, 493, 491, 468, 448, 497, 456, 515, 460, 487, 495, 485, 492, 452, 493, 467, 491, 464, 484, 494, 455, 514, 489, 470, 498,
   449, 516, 458, 496, 486, 461, 488, 490, 495, 453, 469, 485, 497, 492, 465, 515, 457, 487, 496, 471, 451, 498, 486, 516, 459, 490,
   463, 488, 454, 474, 521, 466, 514, 462, 519, 491, 503, 493, 455, 519, 476, 514, 467, 489, 501, 464, 521, 494, 472, 456, 522, 460,
   517, 468, 515, 504, 492, 495, 473, 520, 458, 518, 461, 502, 490, 470, 516, 496, 517, 457, 477, 487, 499, 515, 469, 522, 465, 497,
   518, 475, 459, 500, 488, 520, 463, 516, 471, 498, 462, 478, 535, 474, 519, 466, 523, 493, 505, 503, 480, 460, 536, 468, 525, 472,
   517, 507, 495, 504, 464, 535, 479, 521, 476, 494, 506, 467, 524, 501, 482, 537, 461, 527, 470, 508, 496, 473, 518, 502, 536, 465,
   481, 497, 509, 522, 477, 526, 469, 499, 537, 483, 463, 510, 498, 528, 471, 520, 475, 500, 466, 480, 542, 478, 523, 474, 531, 503,
   513, 505, 467, 540, 482, 524, 479, 501, 512, 476, 533, 506, 478, 468, 543, 472, 529, 480, 525, 513, 504, 507, 479, 541, 470, 530,
   473, 512, 502, 482, 527, 508, 538, 469, 483, 499, 511, 526, 481, 534, 477, 509, 539, 481, 471, 511, 500, 532, 475, 528, 483, 510,
   474, 472, 544, 480, 531, 478, 529, 505, 507, 513, 476, 545, 473, 533, 482, 506, 508, 479, 530, 512, 546, 477, 475, 509, 510, 534,
   483, 532, 481, 511, 502, 500, 486, 520, 490, 518, 488, 496, 498, 516, 504, 485, 499, 492, 522, 495, 497, 517, 487, 515, 484, 503,
   501, 493, 494, 491, 521, 489, 519, 514, 489, 523, 506, 519, 501, 514, 524, 494, 535, 521, 525, 487, 509, 515, 526, 517, 499, 536,
   497, 522, 491, 505, 524, 503, 521, 493, 535, 514, 523, 519, 527, 510, 488, 528, 516, 537, 498, 518, 500, 520, 507, 492, 526, 495,
   536, 504, 522, 525, 515, 517, 508, 528, 490, 537, 496, 527, 516, 502, 520, 518, 512, 539, 496, 541, 502, 530, 518, 508, 537, 527,
   541, 511, 498, 532, 520, 539, 500, 537, 510, 528, 513, 495, 538, 504, 543, 507, 536, 529, 517, 525, 543, 497, 511, 522, 534, 536,
   509, 538, 499, 526, 493, 513, 540, 505, 535, 503, 542, 519, 531, 523, 494, 542, 512, 535, 506, 521, 533, 501, 540, 524, 501, 547,
   508, 540, 512, 524, 530, 506, 545, 533, 548, 499, 510, 526, 532, 538, 511, 546, 509, 534, 503, 507, 547, 513, 542, 505, 544, 523,
   529, 531, 549, 509, 500, 534, 528, 546, 510, 539, 511, 532, 505, 504, 548, 507, 544, 513, 543, 531, 525, 529, 506, 549, 502, 545,
   508, 533, 527, 512, 541, 530, 514, 531, 533, 523, 524, 519, 540, 521, 542, 535, 529, 515, 534, 517, 538, 525, 526, 543, 522, 536,
   530, 532, 516, 539, 518, 541, 520, 527, 528, 537, 519, 529, 545, 531, 540, 523, 547, 535, 544, 542, 531, 517, 546, 525, 548, 529,
   538, 544, 536, 543, 521, 544, 530, 542, 533, 535, 545, 524, 547, 540, 533, 546, 518, 549, 527, 545, 537, 530, 539, 541, 544, 522,
   532, 536, 546, 543, 534, 548, 526, 538, 545, 534, 520, 546, 537, 549, 528, 541, 532, 539, 523, 525, 549, 529, 547, 531, 548, 542,
   543, 544, 524, 548, 527, 547, 530, 540, 541, 533, 549, 545, 547, 526, 528, 538, 539, 548, 532, 549, 534, 546, 540, 538, 537, 548,
   541, 547, 539, 545, 546, 549, 542, 536, 539, 543, 549, 544, 546, 547, 538, 548, 535, 543, 541, 544, 545, 542, 549, 540, 548, 547,
   550, 565, 567, 559, 560, 553, 595, 555, 597, 594, 566, 551, 569, 554, 596, 561, 563, 599, 557, 593, 568, 570, 552, 598, 556, 600,
   558, 562, 564, 592, 573, 554, 604, 561, 626, 566, 596, 603, 593, 599, 574, 602, 556, 627, 562, 601, 592, 568, 598, 600, 553, 571,
   606, 565, 595, 559, 625, 594, 605, 597, 601, 576, 558, 602, 592, 627, 564, 600, 570, 598, 555, 605, 572, 597, 567, 594, 606, 560,
   625, 595, 603, 557, 575, 593, 604, 599, 569, 626, 563, 596, 559, 577, 630, 571, 625, 565, 628, 597, 609, 605, 560, 628, 579, 625,
   572, 595, 607, 567, 630, 606, 578, 561, 632, 566, 629, 573, 626, 611, 599, 603, 629, 563, 581, 596, 608, 626, 575, 632, 569, 604,
   580, 633, 562, 631, 568, 612, 600, 574, 627, 601, 631, 582, 564, 610, 598, 633, 570, 627, 576, 602, 585, 566, 650, 573, 636, 578,
   629, 615, 603, 611, 586, 649, 568, 637, 574, 613, 601, 580, 631, 612, 565, 583, 651, 577, 628, 571, 634, 605, 617, 609, 649, 588,
   570, 614, 602, 639, 576, 633, 582, 610, 567, 651, 584, 630, 579, 606, 618, 572, 635, 607, 650, 569, 587, 604, 616, 632, 581, 638,
   575, 608, 571, 589, 654, 583, 634, 577, 640, 609, 621, 617, 572, 652, 590, 635, 584, 607, 619, 579, 642, 618, 589, 573, 656, 578,
   641, 585, 636, 623, 611, 615, 653, 575, 591, 608, 620, 638, 587, 644, 581, 616, 590, 657, 574, 643, 580, 624, 612, 586, 637, 613,
   655, 591, 576, 622, 610, 645, 582, 639, 588, 614, 583, 578, 660, 585, 646, 589, 641, 621, 615, 623, 584, 658, 580, 647, 586, 619,
   613, 590, 643, 624, 577, 585, 662, 589, 640, 583, 646, 617, 623, 621, 659, 587, 582, 620, 614, 648, 588, 645, 591, 622, 579, 663,
   586, 642, 590, 618, 624, 584, 647, 619, 661, 581, 588, 616, 622, 644, 591, 648, 587, 620, 594, 609, 607, 605, 606, 597, 630, 595,
   628, 625, 611, 593, 608, 599, 632, 603, 604, 629, 596, 626, 612, 610, 592, 633, 600, 631, 598, 601, 602, 627, 636, 596, 616, 626,
   638, 629, 608, 650, 604, 632, 637, 614, 598, 639, 627, 649, 602, 631, 610, 633, 595, 634, 618, 628, 607, 625, 635, 606, 651, 630,
   613, 639, 600, 649, 601, 637, 627, 612, 633, 631, 597, 617, 635, 609, 630, 605, 651, 625, 634, 628, 615, 599, 638, 603, 650, 611,
   632, 636, 626, 629, 605, 621, 652, 617, 651, 609, 654, 628, 640, 634, 606, 654, 619, 651, 618, 630, 642, 607, 652, 635, 623, 603,
   653, 611, 656, 615, 650, 641, 629, 636, 624, 655, 601, 657, 612, 643, 631, 613, 649, 637, 656, 604, 620, 632, 644, 650, 616, 653,
   608, 638, 657, 622, 602, 645, 633, 655, 610, 649, 614, 639, 665, 608, 622, 638, 648, 653, 620, 661, 616, 644, 666, 620, 610, 648,
   639, 659, 614, 655, 622, 645, 607, 664, 624, 652, 619, 635, 647, 618, 663, 642, 619, 666, 612, 658, 613, 647, 637, 624, 657, 643,
   609, 623, 664, 621, 654, 617, 662, 634, 646, 640, 621, 611, 665, 615, 660, 623, 656, 646, 636, 641, 617, 615, 667, 623, 662, 621,
   660, 640, 641, 646, 618, 668, 613, 663, 624, 642, 643, 619, 658, 647, 669, 616, 614, 644, 645, 661, 622, 659, 620, 648, 625, 640,
   642, 634, 635, 628, 652, 630, 654, 651, 641, 626, 644, 629, 653, 636, 638, 656, 632, 650, 643, 645, 627, 655, 631, 657, 633, 637,
   639, 649, 646, 629, 661, 636, 665, 641, 653, 660, 650, 656, 647, 659, 631, 666, 637, 658, 649, 643, 655, 657, 628, 646, 663, 640,
   652, 634, 664, 651, 662, 654, 658, 648, 633, 659, 649, 666, 639, 657, 645, 655, 630, 662, 647, 654, 642, 651, 663, 635, 664, 652,
   660, 632, 648, 650, 661, 656, 644, 665, 638, 653, 634, 641, 668, 646, 664, 640, 667, 654, 660, 662, 635, 667, 643, 664, 647, 652,
   658, 642, 668, 663, 640, 636, 669, 641, 667, 646, 665, 662, 656, 660, 667, 638, 645, 653, 659, 665, 648, 669, 644, 661, 642, 669,
   637, 668, 643, 663, 657, 647, 666, 658, 668, 644, 639, 661, 655, 669, 645, 666, 648, 659, 651, 660, 658, 662, 663, 654, 668, 652,
   667, 664, 662, 650, 659, 656, 669, 660, 661, 667, 653, 665, 663, 661, 649, 669, 657, 668, 655, 658, 659, 666, 664, 653, 655, 665,
   666, 667, 659, 668, 661, 669, 652, 665, 657, 667, 658, 664, 666, 663, 669, 668, 654, 656, 666, 660, 668, 662, 669, 664, 665, 667
}; // 7.11 kb
static const cart_index_t iCartXY_ab14_a4[4290] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 18, 19, 20, 26, 22, 28, 31, 32, 24, 25,
   33, 34, 35, 30, 20, 16, 21, 24, 18, 30, 32, 23, 29, 26, 34, 27, 33, 31, 35, 22, 23, 17, 30, 31, 25, 19,
   27, 21, 35, 28, 29, 32, 33, 34, 36, 47, 48, 39, 51, 41, 52, 61, 62, 45, 46, 63, 57, 59, 56, 45, 37, 50,
   51, 40, 59, 60, 43, 53, 47, 63, 49, 58, 55, 61, 46, 49, 38, 57, 58, 52, 42, 53, 44, 63, 48, 50, 54, 60,
   62, 39, 40, 54, 45, 47, 56, 62, 55, 60, 51, 57, 58, 63, 61, 59, 41, 55, 42, 56, 61, 46, 48, 58, 54, 59,
   52, 60, 62, 63, 57, 56, 43, 44, 59, 55, 57, 54, 49, 50, 61, 62, 53, 60, 58, 63, 64, 79, 80, 69, 81, 71,
   83, 94, 95, 73, 74, 97, 90, 92, 85, 81, 65, 82, 79, 67, 94, 96, 72, 84, 75, 98, 76, 88, 86, 93, 83, 84,
   66, 95, 96, 80, 68, 82, 70, 99, 77, 78, 87, 89, 91, 73, 67, 89, 81, 75, 92, 99, 86, 96, 79, 97, 88, 98,
   93, 94, 74, 88, 68, 90, 98, 83, 77, 96, 87, 97, 80, 89, 91, 99, 95, 69, 75, 91, 73, 79, 85, 95, 93, 99,
   81, 90, 98, 97, 94, 92, 90, 76, 70, 97, 88, 95, 87, 84, 78, 98, 91, 82, 89, 96, 99, 71, 93, 77, 85, 94,
   74, 80, 98, 91, 92, 83, 99, 95, 97, 90, 92, 72, 78, 94, 86, 97, 89, 76, 82, 93, 99, 84, 96, 88, 98, 85,
   86, 87, 92, 93, 90, 91, 88, 89, 94, 95, 96, 99, 98, 97, 100, 121, 122, 103, 115, 105, 117, 137, 138, 109, 110, 139,
   127, 129, 126, 121, 101, 123, 116, 104, 135, 136, 107, 119, 111, 140, 113, 128, 125, 131, 122, 123, 102, 133, 134, 118, 106, 120,
   108, 141, 112, 114, 124, 130, 132, 103, 116, 133, 109, 121, 126, 138, 135, 143, 115, 127, 142, 139, 137, 129, 115, 104, 134, 121,
   111, 137, 144, 125, 136, 116, 142, 128, 140, 131, 135, 105, 135, 118, 126, 137, 110, 122, 142, 133, 129, 117, 143, 138, 139, 127,
   117, 136, 106, 138, 144, 122, 112, 134, 124, 143, 118, 130, 132, 141, 133, 137, 107, 120, 135, 125, 142, 134, 113, 123, 131, 144,
   119, 136, 128, 140, 138, 119, 108, 143, 136, 133, 124, 123, 114, 144, 132, 120, 130, 134, 141, 109, 111, 141, 115, 116, 129, 143,
   131, 144, 121, 139, 140, 142, 135, 137, 110, 140, 112, 127, 142, 117, 118, 144, 132, 139, 122, 141, 133, 143, 138, 139, 113, 114,
   142, 128, 143, 130, 119, 120, 140, 141, 123, 134, 136, 144, 127, 128, 124, 139, 140, 138, 132, 136, 130, 142, 133, 134, 141, 144,
   143, 129, 125, 130, 137, 131, 139, 141, 128, 134, 135, 143, 136, 144, 140, 142, 126, 131, 132, 129, 135, 127, 133, 140, 141, 137,
   138, 144, 143, 142, 139, 145, 166, 167, 150, 162, 152, 164, 181, 182, 154, 155, 190, 177, 179, 172, 168, 146, 169, 160, 148, 183,
   185, 153, 165, 156, 191, 157, 175, 173, 180, 170, 171, 147, 184, 186, 161, 149, 163, 151, 192, 158, 159, 174, 176, 178, 166, 148,
   187, 168, 156, 189, 198, 173, 185, 160, 193, 175, 191, 180, 183, 167, 187, 149, 188, 196, 170, 158, 186, 174, 194, 161, 176, 178,
   192, 184, 150, 168, 188, 154, 166, 172, 182, 189, 197, 162, 177, 195, 190, 181, 179, 188, 169, 151, 194, 187, 184, 174, 171, 159,
   196, 178, 163, 176, 186, 192, 152, 189, 170, 172, 181, 155, 167, 195, 188, 179, 164, 197, 182, 190, 177, 189, 153, 171, 183, 173,
   193, 187, 157, 169, 180, 198, 165, 185, 175, 191, 154, 160, 194, 162, 168, 179, 197, 183, 199, 166, 190, 193, 195, 189, 181, 155,
   193, 161, 177, 195, 164, 170, 199, 184, 190, 167, 194, 188, 197, 182, 162, 156, 196, 166, 160, 181, 199, 180, 198, 168, 195, 191,
   193, 183, 189, 195, 157, 163, 193, 175, 199, 186, 165, 171, 191, 196, 169, 187, 185, 198, 164, 198, 158, 182, 199, 167, 161, 196,
   178, 197, 170, 192, 184, 194, 188, 197, 165, 159, 199, 185, 194, 176, 169, 163, 198, 192, 171, 186, 187, 196, 172, 183, 184, 179,
   189, 177, 188, 193, 194, 181, 182, 199, 197, 195, 190, 181, 173, 186, 189, 180, 195, 196, 175, 187, 183, 199, 185, 198, 191, 193,
   182, 185, 174, 197, 198, 188, 178, 187, 176, 199, 184, 186, 192, 196, 194, 190, 175, 176, 195, 191, 197, 192, 185, 186, 193, 194,
   187, 196, 198, 199, 177, 191, 178, 190, 193, 182, 184, 198, 192, 195, 188, 196, 194, 199, 197, 179, 180, 192, 181, 183, 190, 194,
   191, 196, 189, 197, 198, 199, 193, 195, 200, 221, 222, 203, 215, 205, 217, 243, 244, 209, 210, 251, 233, 235, 232, 223, 201, 225,
   216, 204, 241, 242, 207, 219, 211, 252, 213, 234, 231, 237, 224, 226, 202, 239, 240, 218, 206, 220, 208, 253, 212, 214, 230, 236,
   238, 203, 227, 245, 209, 221, 232, 244, 247, 256, 215, 233, 254, 251, 243, 235, 227, 204, 246, 223, 211, 249, 258, 231, 242, 216,
   255, 234, 252, 237, 241, 205, 247, 228, 232, 243, 210, 222, 254, 245, 235, 217, 256, 244, 251, 233, 228, 248, 206, 250, 259, 224,
   212, 240, 230, 257, 218, 236, 238, 253, 239, 249, 207, 229, 241, 231, 255, 246, 213, 225, 237, 258, 219, 242, 234, 252, 250, 229,
   208, 257, 248, 239, 230, 226, 214, 259, 238, 220, 236, 240, 253, 209, 223, 261, 215, 227, 235, 256, 249, 265, 221, 251, 260, 254,
   247, 243, 210, 260, 224, 233, 254, 217, 228, 265, 250, 251, 222, 261, 245, 256, 244, 221, 211, 262, 227, 216, 247, 264, 237, 258,
   223, 260, 252, 255, 241, 249, 222, 262, 212, 245, 263, 228, 218, 259, 238, 261, 224, 253, 239, 257, 250, 260, 213, 226, 255, 234,
   264, 248, 219, 229, 252, 262, 225, 246, 242, 258, 261, 225, 214, 263, 246, 257, 236, 229, 220, 262, 253, 226, 240, 248, 259, 215,
   216, 263, 221, 223, 243, 265, 241, 264, 227, 254, 255, 260, 249, 247, 217, 264, 218, 244, 265, 222, 224, 263, 239, 256, 228, 257,
   250, 261, 245, 265, 219, 220, 264, 242, 263, 240, 225, 226, 258, 259, 229, 248, 246, 262, 245, 246, 230, 261, 262, 250, 238, 248,
   236, 263, 239, 240, 253, 259, 257, 247, 231, 248, 249, 237, 260, 262, 234, 246, 241, 264, 242, 258, 252, 255, 232, 249, 250, 235,
   247, 233, 245, 260, 261, 243, 244, 265, 256, 254, 251, 233, 255, 239, 251, 260, 244, 250, 264, 257, 254, 245, 263, 261, 265, 256,
   254, 234, 240, 260, 252, 265, 259, 242, 248, 255, 263, 246, 262, 258, 264, 235, 241, 257, 243, 249, 251, 261, 255, 263, 247, 256,
   264, 265, 260, 254, 256, 242, 236, 265, 258, 261, 253, 246, 240, 264, 257, 248, 259, 262, 263, 243, 237, 259, 247, 241, 254, 263,
   252, 262, 249, 265, 258, 264, 255, 260, 244, 258, 238, 256, 264, 245, 239, 262, 253, 265, 250, 259, 257, 263, 261, 251, 252, 253,
   254, 255, 256, 257, 258, 259, 260, 261, 262, 263, 264, 265, 266, 287, 288, 271, 283, 273, 285, 308, 309, 275, 276, 323, 304, 306,
   299, 289, 267, 290, 281, 269, 310, 312, 274, 286, 277, 324, 278, 302, 300, 307, 291, 292, 268, 311, 313, 282, 270, 284, 272, 325,
   279, 280, 301, 303, 305, 293, 269, 314, 289, 277, 319, 331, 300, 312, 281, 326, 302, 324, 307, 310, 294, 315, 270, 317, 329, 291,
   279, 313, 301, 327, 282, 303, 305, 325, 311, 271, 295, 316, 275, 287, 299, 309, 318, 330, 283, 304, 328, 323, 308, 306, 317, 296,
   272, 327, 315, 311, 301, 292, 280, 329, 305, 284, 303, 313, 325, 273, 318, 297, 299, 308, 276, 288, 328, 316, 306, 285, 330, 309,
   323, 304, 319, 274, 298, 310, 300, 326, 314, 278, 290, 307, 331, 286, 312, 302, 324, 275, 293, 333, 283, 295, 306, 330, 320, 338,
   287, 323, 332, 328, 318, 308, 276, 332, 294, 304, 328, 285, 297, 338, 321, 323, 288, 333, 316, 330, 309, 295, 277, 335, 293, 281,
   320, 339, 307, 331, 289, 334, 324, 326, 310, 319, 334, 278, 296, 326, 302, 339, 322, 286, 298, 324, 335, 290, 314, 312, 331, 297,
   337, 279, 321, 340, 294, 282, 329, 305, 336, 291, 325, 311, 327, 317, 336, 298, 280, 340, 322, 327, 303, 296, 284, 337, 325, 292,
   313, 315, 329, 287, 281, 341, 295, 289, 318, 343, 310, 339, 293, 332, 326, 334, 319, 320, 288, 341, 282, 316, 342, 297, 291, 340,
   311, 333, 294, 327, 317, 336, 321, 283, 289, 342, 287, 293, 308, 338, 319, 343, 295, 328, 334, 332, 320, 318, 342, 290, 284, 341,
   314, 340, 313, 298, 292, 335, 329, 296, 315, 322, 337, 285, 343, 291, 309, 338, 288, 294, 342, 317, 330, 297, 336, 321, 333, 316,
   343, 286, 292, 339, 312, 341, 315, 290, 296, 331, 337, 298, 322, 314, 335, 299, 320, 321, 306, 318, 304, 316, 332, 333, 308, 309,
   338, 330, 328, 323, 320, 300, 322, 319, 307, 334, 335, 302, 314, 310, 339, 312, 331, 324, 326, 321, 322, 301, 336, 337, 317, 305,
   315, 303, 340, 311, 313, 325, 329, 327, 332, 302, 315, 334, 324, 343, 337, 312, 322, 326, 341, 314, 335, 331, 339, 333, 314, 303,
   342, 335, 336, 325, 322, 313, 341, 327, 315, 329, 337, 340, 304, 334, 317, 323, 332, 309, 321, 343, 336, 328, 316, 342, 333, 338,
   330, 316, 335, 305, 333, 341, 321, 311, 337, 325, 342, 317, 329, 327, 340, 336, 306, 319, 336, 308, 320, 323, 333, 334, 342, 318,
   330, 343, 338, 332, 328, 318, 307, 337, 320, 310, 332, 341, 324, 335, 319, 343, 331, 339, 326, 334, 308, 310, 340, 318, 319, 328,
   342, 326, 341, 320, 338, 339, 343, 334, 332, 309, 339, 311, 330, 343, 316, 317, 341, 327, 338, 321, 340, 336, 342, 333, 338, 312,
   313, 343, 331, 342, 329, 314, 315, 339, 340, 322, 337, 335, 341, 323, 326, 327, 328, 334, 330, 336, 339, 340, 332, 333, 341, 342,
   343, 338, 328, 324, 329, 332, 326, 338, 340, 331, 337, 334, 342, 335, 341, 339, 343, 330, 331, 325, 338, 339, 333, 327, 335, 329,
   343, 336, 337, 340, 341, 342, 344, 365, 366, 347, 359, 349, 361, 393, 394, 353, 354, 407, 383, 385, 382, 367, 345, 369, 360, 348,
   391, 392, 351, 363, 355, 408, 357, 384, 381, 387, 368, 370, 346, 389, 390, 362, 350, 364, 352, 409, 356, 358, 380, 386, 388, 347,
   371, 395, 353, 365, 382, 394, 397, 412, 359, 383, 410, 407, 393, 385, 372, 348, 396, 367, 355, 399, 414, 381, 392, 360, 411, 384,
   408, 387, 391, 349, 397, 373, 382, 393, 354, 366, 410, 395, 385, 361, 412, 394, 407, 383, 374, 398, 350, 400, 415, 368, 356, 390,
   380, 413, 362, 386, 388, 409, 389, 399, 351, 375, 391, 381, 411, 396, 357, 369, 387, 414, 363, 392, 384, 408, 400, 376, 352, 413,
   398, 389, 380, 370, 358, 415, 388, 364, 386, 390, 409, 353, 377, 417, 359, 371, 385, 412, 405, 427, 365, 407, 416, 410, 397, 393,
   354, 416, 378, 383, 410, 361, 373, 427, 406, 407, 366, 417, 395, 412, 394, 377, 355, 420, 372, 360, 403, 426, 387, 414, 367, 418,
   408, 411, 391, 399, 378, 421, 356, 401, 425, 374, 362, 415, 388, 419, 368, 409, 389, 413, 400, 418, 357, 379, 411, 384, 426, 404,
   363, 375, 408, 420, 369, 396, 392, 414, 419, 379, 358, 425, 402, 413, 386, 376, 364, 421, 409, 370, 390, 398, 415, 359, 372, 428,
   365, 377, 393, 427, 403, 430, 371, 410, 422, 416, 405, 397, 371, 360, 429, 377, 367, 405, 432, 391, 426, 372, 422, 411, 418, 399,
   403, 361, 430, 374, 394, 427, 366, 378, 428, 401, 412, 373, 423, 406, 417, 395, 373, 431, 362, 406, 433, 378, 368, 425, 389, 423,
   374, 413, 400, 419, 401, 432, 363, 376, 426, 392, 429, 402, 369, 379, 414, 424, 375, 404, 396, 420, 433, 375, 364, 431, 404, 425,
   390, 379, 370, 424, 415, 376, 398, 402, 421, 365, 367, 434, 371, 372, 397, 430, 399, 432, 377, 416, 418, 422, 403, 405, 366, 434,
   368, 395, 428, 373, 374, 433, 400, 417, 378, 419, 401, 423, 406, 434, 369, 370, 429, 396, 431, 398, 375, 376, 420, 421, 379, 402,
   404, 424, 401, 402, 380, 419, 421, 400, 388, 398, 386, 425, 389, 390, 409, 415, 413, 403, 381, 404, 399, 387, 418, 420, 384, 396,
   391, 426, 392, 414, 408, 411, 382, 405, 406, 385, 397, 383, 395, 416, 417, 393, 394, 427, 412, 410, 407, 383, 422, 401, 407, 416,
   394, 406, 430, 423, 410, 395, 428, 417, 427, 412, 422, 384, 402, 418, 408, 432, 424, 392, 404, 411, 429, 396, 420, 414, 426, 385,
   403, 423, 393, 405, 407, 417, 422, 428, 397, 412, 430, 427, 416, 410, 423, 404, 386, 433, 424, 419, 409, 402, 390, 431, 413, 398,
   415, 421, 425, 405, 387, 424, 403, 391, 422, 429, 408, 420, 399, 432, 414, 426, 411, 418, 406, 424, 388, 423, 431, 401, 389, 421,
   409, 433, 400, 415, 413, 425, 419, 395, 429, 389, 417, 434, 406, 400, 431, 413, 428, 401, 425, 419, 433, 423, 428, 396, 390, 434,
   420, 433, 415, 404, 398, 429, 425, 402, 421, 424, 431, 397, 391, 431, 405, 399, 416, 434, 411, 429, 403, 430, 426, 432, 418, 422,
   430, 392, 398, 432, 414, 434, 421, 396, 402, 426, 431, 404, 424, 420, 429, 393, 399, 433, 397, 403, 410, 428, 418, 434, 405, 427,
   432, 430, 422, 416, 394, 432, 400, 412, 430, 395, 401, 434, 419, 427, 406, 433, 423, 428, 417, 407, 418, 419, 410, 422, 412, 423,
   432, 433, 416, 417, 434, 428, 430, 427, 416, 408, 421, 422, 411, 430, 431, 414, 424, 418, 434, 420, 429, 426, 432, 417, 420, 409,
   428, 429, 423, 413, 424, 415, 434, 419, 421, 425, 431, 433, 410, 411, 425, 416, 418, 427, 433, 426, 431, 422, 428, 429, 434, 432,
   430, 412, 426, 413, 427, 432, 417, 419, 429, 425, 430, 423, 431, 433, 434, 428, 427, 414, 415, 430, 426, 428, 425, 420, 421, 432,
   433, 424, 431, 429, 434, 435, 456, 457, 440, 452, 442, 454, 483, 484, 444, 445, 504, 479, 481, 474, 458, 436, 459, 450, 438, 485,
   487, 443, 455, 446, 505, 447, 477, 475, 482, 460, 461, 437, 486, 488, 451, 439, 453, 441, 506, 448, 449, 476, 478, 480, 462, 438,
   489, 458, 446, 494, 512, 475, 487, 450, 507, 477, 505, 482, 485, 463, 490, 439, 492, 510, 460, 448, 488, 476, 508, 451, 478, 480,
   506, 486, 440, 464, 491, 444, 456, 474, 484, 493, 511, 452, 479, 509, 504, 483, 481, 492, 465, 441, 508, 490, 486, 476, 461, 449,
   510, 480, 453, 478, 488, 506, 442, 493, 466, 474, 483, 445, 457, 509, 491, 481, 454, 511, 484, 504, 479, 494, 443, 467, 485, 475,
   507, 489, 447, 459, 482, 512, 455, 487, 477, 505, 444, 468, 514, 452, 464, 481, 511, 495, 525, 456, 504, 513, 509, 493, 483, 445,
   513, 469, 479, 509, 454, 466, 525, 496, 504, 457, 514, 491, 511, 484, 470, 446, 516, 462, 450, 497, 526, 482, 512, 458, 515, 505,
   507, 485, 494, 515, 447, 471, 507, 477, 526, 499, 455, 467, 505, 516, 459, 489, 487, 512, 472, 518, 448, 498, 527, 463, 451, 510,
   480, 517, 460, 506, 486, 508, 492, 517, 473, 449, 527, 500, 508, 478, 465, 453, 518, 506, 461, 488, 490, 510, 468, 450, 528, 470,
   458, 503, 533, 485, 526, 462, 519, 507, 515, 494, 497, 469, 529, 451, 502, 531, 472, 460, 527, 486, 520, 463, 508, 492, 517, 498,
   452, 470, 530, 456, 468, 483, 525, 503, 532, 464, 509, 521, 513, 495, 493, 531, 471, 453, 529, 501, 527, 488, 473, 461, 522, 510,
   465, 490, 500, 518, 454, 532, 472, 484, 525, 457, 469, 530, 502, 511, 466, 523, 496, 514, 491, 533, 455, 473, 526, 487, 528, 501,
   459, 471, 512, 524, 467, 499, 489, 516, 456, 462, 537, 464, 470, 493, 532, 497, 534, 468, 513, 519, 521, 503, 495, 457, 537, 463,
   491, 530, 466, 472, 535, 498, 514, 469, 520, 502, 523, 496, 464, 458, 538, 468, 462, 495, 534, 494, 533, 470, 521, 515, 519, 497,
   503, 538, 459, 465, 528, 489, 536, 500, 467, 473, 516, 522, 471, 501, 499, 524, 466, 539, 460, 496, 535, 469, 463, 531, 492, 523,
   472, 517, 498, 520, 502, 539, 467, 461, 536, 499, 529, 490, 471, 465, 524, 518, 473, 500, 501, 522, 474, 495, 496, 481, 493, 479,
   491, 513, 514, 483, 484, 525, 511, 509, 504, 497, 475, 499, 494, 482, 515, 516, 477, 489, 485, 526, 487, 512, 505, 507, 498, 500,
   476, 517, 518, 492, 480, 490, 478, 527, 486, 488, 506, 510, 508, 519, 477, 501, 515, 505, 533, 524, 487, 499, 507, 528, 489, 516,
   512, 526, 520, 501, 478, 531, 522, 517, 506, 500, 488, 529, 508, 490, 510, 518, 527, 479, 521, 502, 504, 513, 484, 496, 532, 523,
   509, 491, 530, 514, 525, 511, 502, 522, 480, 520, 529, 498, 486, 518, 506, 531, 492, 510, 508, 527, 517, 481, 503, 523, 483, 495,
   504, 514, 521, 530, 493, 511, 532, 525, 513, 509, 503, 482, 524, 497, 485, 519, 528, 505, 516, 494, 533, 512, 526, 507, 515, 483,
   497, 535, 493, 503, 509, 530, 519, 537, 495, 525, 534, 532, 521, 513, 484, 534, 498, 511, 532, 491, 502, 537, 520, 525, 496, 535,
   523, 530, 514, 495, 485, 536, 503, 494, 521, 538, 507, 528, 497, 534, 526, 533, 515, 519, 496, 536, 486, 523, 539, 502, 492, 529,
   508, 535, 498, 527, 517, 531, 520, 534, 487, 500, 533, 512, 538, 522, 489, 501, 526, 536, 499, 524, 516, 528, 535, 499, 488, 539,
   524, 531, 510, 501, 490, 536, 527, 500, 518, 522, 529, 537, 489, 490, 538, 516, 539, 518, 499, 500, 528, 529, 501, 522, 524, 536,
   491, 538, 492, 514, 537, 496, 498, 539, 517, 530, 502, 531, 520, 535, 523, 493, 494, 539, 495, 497, 513, 537, 515, 538, 503, 532,
   533, 534, 519, 521, 504, 519, 520, 509, 521, 511, 523, 534, 535, 513, 514, 537, 530, 532, 525, 521, 505, 522, 519, 507, 534, 536,
   512, 524, 515, 538, 516, 528, 526, 533, 523, 524, 506, 535, 536, 520, 508, 522, 510, 539, 517, 518, 527, 529, 531, 513, 507, 529,
   521, 515, 532, 539, 526, 536, 519, 537, 528, 538, 533, 534, 514, 528, 508, 530, 538, 523, 517, 536, 527, 537, 520, 529, 531, 539,
   535, 509, 515, 531, 513, 519, 525, 535, 533, 539, 521, 530, 538, 537, 534, 532, 530, 516, 510, 537, 528, 535, 527, 524, 518, 538,
   531, 522, 529, 536, 539, 511, 533, 517, 525, 534, 514, 520, 538, 531, 532, 523, 539, 535, 537, 530, 532, 512, 518, 534, 526, 537,
   529, 516, 522, 533, 539, 524, 536, 528, 538, 525, 526, 527, 532, 533, 530, 531, 528, 529, 534, 535, 536, 539, 538, 537, 540, 561,
   562, 543, 555, 545, 557, 595, 596, 549, 550, 615, 585, 587, 584, 563, 541, 565, 556, 544, 593, 594, 547, 559, 551, 616, 553, 586,
   583, 589, 564, 566, 542, 591, 592, 558, 546, 560, 548, 617, 552, 554, 582, 588, 590, 543, 567, 597, 549, 561, 584, 596, 599, 620,
   555, 585, 618, 615, 595, 587, 568, 544, 598, 563, 551, 601, 622, 583, 594, 556, 619, 586, 616, 589, 593, 545, 599, 569, 584, 595,
   550, 562, 618, 597, 587, 557, 620, 596, 615, 585, 570, 600, 546, 602, 623, 564, 552, 592, 582, 621, 558, 588, 590, 617, 591, 601,
   547, 571, 593, 583, 619, 598, 553, 565, 589, 622, 559, 594, 586, 616, 602, 572, 548, 621, 600, 591, 582, 566, 554, 623, 590, 560,
   588, 592, 617, 549, 573, 625, 555, 567, 587, 620, 607, 641, 561, 615, 624, 618, 599, 595, 550, 624, 574, 585, 618, 557, 569, 641,
   608, 615, 562, 625, 597, 620, 596, 575, 551, 628, 568, 556, 605, 640, 589, 622, 563, 626, 616, 619, 593, 601, 576, 629, 552, 603,
   639, 570, 558, 623, 590, 627, 564, 617, 591, 621, 602, 626, 553, 577, 619, 586, 640, 606, 559, 571, 616, 628, 565, 598, 594, 622,
   627, 578, 554, 639, 604, 621, 588, 572, 560, 629, 617, 566, 592, 600, 623, 555, 579, 642, 561, 573, 595, 641, 611, 644, 567, 618,
   630, 624, 607, 599, 579, 556, 643, 575, 563, 613, 646, 593, 640, 568, 631, 619, 626, 601, 605, 557, 644, 580, 596, 641, 562, 574,
   642, 609, 620, 569, 632, 608, 625, 597, 580, 645, 558, 614, 647, 576, 564, 639, 591, 633, 570, 621, 602, 627, 603, 646, 559, 581,
   640, 594, 643, 610, 565, 577, 622, 634, 571, 606, 598, 628, 647, 581, 560, 645, 612, 639, 592, 578, 566, 635, 623, 572, 600, 604,
   629, 561, 575, 654, 567, 579, 599, 644, 613, 652, 573, 624, 636, 630, 611, 607, 562, 654, 576, 597, 642, 569, 580, 653, 614, 625,
   574, 637, 609, 632, 608, 573, 563, 655, 579, 568, 611, 650, 601, 646, 575, 636, 626, 631, 605, 613, 574, 656, 564, 609, 648, 580,
   570, 647, 602, 637, 576, 627, 603, 633, 614, 655, 565, 578, 643, 598, 651, 612, 571, 581, 628, 638, 577, 610, 606, 634, 656, 577,
   566, 649, 610, 645, 600, 581, 572, 638, 629, 578, 604, 612, 635, 567, 568, 657, 573, 575, 607, 652, 605, 650, 579, 630, 631, 636,
   613, 611, 569, 658, 570, 608, 653, 574, 576, 648, 603, 632, 580, 633, 614, 637, 609, 659, 571, 572, 651, 606, 649, 604, 577, 578,
   634, 635, 581, 612, 610, 638, 603, 604, 582, 627, 629, 602, 590, 600, 588, 639, 591, 592, 617, 623, 621, 605, 583, 606, 601, 589,
   626, 628, 586, 598, 593, 640, 594, 622, 616, 619, 584, 607, 608, 587, 599, 585, 597, 624, 625, 595, 596, 641, 620, 618, 615, 585,
   630, 609, 615, 624, 596, 608, 644, 632, 618, 597, 642, 625, 641, 620, 631, 586, 610, 626, 616, 646, 634, 594, 606, 619, 643, 598,
   628, 622, 640, 587, 611, 632, 595, 607, 615, 625, 630, 642, 599, 620, 644, 641, 624, 618, 633, 612, 588, 647, 635, 627, 617, 604,
   592, 645, 621, 600, 623, 629, 639, 613, 589, 634, 605, 593, 631, 643, 616, 628, 601, 646, 622, 640, 619, 626, 614, 635, 590, 633,
   645, 603, 591, 629, 617, 647, 602, 623, 621, 639, 627, 609, 649, 591, 637, 656, 614, 602, 645, 621, 648, 603, 639, 627, 647, 633,
   648, 610, 592, 656, 638, 647, 623, 612, 600, 649, 639, 604, 629, 635, 645, 611, 593, 651, 613, 601, 636, 655, 619, 643, 605, 650,
   640, 646, 626, 631, 650, 594, 612, 646, 622, 655, 638, 598, 610, 640, 651, 606, 634, 628, 643, 595, 613, 653, 599, 611, 618, 642,
   636, 654, 607, 641, 652, 644, 630, 624, 596, 652, 614, 620, 644, 597, 609, 654, 637, 641, 608, 653, 632, 642, 625, 597, 657, 603,
   625, 654, 608, 614, 658, 633, 642, 609, 648, 637, 653, 632, 657, 598, 604, 655, 628, 659, 635, 606, 612, 643, 649, 610, 638, 634,
   651, 599, 605, 658, 607, 613, 624, 654, 631, 657, 611, 644, 650, 652, 636, 630, 658, 606, 600, 659, 634, 656, 629, 610, 604, 651,
   645, 612, 635, 638, 649, 607, 601, 659, 611, 605, 630, 657, 626, 655, 613, 652, 646, 650, 631, 636, 608, 659, 602, 632, 658, 609,
   603, 656, 627, 653, 614, 647, 633, 648, 637, 615, 636, 637, 618, 630, 620, 632, 652, 653, 624, 625, 654, 642, 644, 641, 636, 616,
   638, 631, 619, 650, 651, 622, 634, 626, 655, 628, 643, 640, 646, 637, 638, 617, 648, 649, 633, 621, 635, 623, 656, 627, 629, 639,
   645, 647, 618, 631, 648, 624, 636, 641, 653, 650, 658, 630, 642, 657, 654, 652, 644, 630, 619, 649, 636, 626, 652, 659, 640, 651,
   631, 657, 643, 655, 646, 650, 620, 650, 633, 641, 652, 625, 637, 657, 648, 644, 632, 658, 653, 654, 642, 632, 651, 621, 653, 659,
   637, 627, 649, 639, 658, 633, 645, 647, 656, 648, 652, 622, 635, 650, 640, 657, 649, 628, 638, 646, 659, 634, 651, 643, 655, 653,
   634, 623, 658, 651, 648, 639, 638, 629, 659, 647, 635, 645, 649, 656, 624, 626, 656, 630, 631, 644, 658, 646, 659, 636, 654, 655,
   657, 650, 652, 625, 655, 627, 642, 657, 632, 633, 659, 647, 654, 637, 656, 648, 658, 653, 654, 628, 629, 657, 643, 658, 645, 634,
   635, 655, 656, 638, 649, 651, 659, 642, 643, 639, 654, 655, 653, 647, 651, 645, 657, 648, 649, 656, 659, 658, 644, 640, 645, 652,
   646, 654, 656, 643, 649, 650, 658, 651, 659, 655, 657, 641, 646, 647, 644, 650, 642, 648, 655, 656, 652, 653, 659, 658, 657, 654
}; // 8.38 kb
static const cart_index_t iCartXY_ab14_a5[4620] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 25, 27, 32, 33,
   24, 39, 26, 40, 30, 31, 36, 43, 37, 45, 41, 46, 47, 48, 42, 44, 24, 22, 29, 25, 39, 30, 35, 41, 28, 36,
   42, 32, 34, 47, 38, 44, 40, 45, 43, 48, 46, 26, 28, 23, 40, 27, 41, 29, 31, 34, 44, 37, 46, 38, 33, 35,
   42, 43, 39, 45, 47, 48, 49, 60, 62, 64, 65, 54, 76, 56, 78, 58, 59, 66, 83, 68, 84, 70, 79, 80, 82, 75,
   77, 58, 50, 63, 52, 74, 66, 67, 77, 57, 64, 82, 60, 61, 84, 69, 79, 71, 81, 73, 83, 78, 59, 61, 51, 73,
   53, 75, 55, 68, 69, 82, 65, 83, 67, 62, 63, 80, 81, 72, 74, 76, 84, 54, 52, 72, 60, 76, 58, 74, 70, 71,
   66, 75, 64, 73, 80, 81, 77, 78, 84, 83, 82, 79, 56, 71, 53, 78, 62, 70, 72, 59, 73, 77, 68, 79, 81, 65,
   74, 75, 83, 76, 84, 80, 82, 70, 57, 55, 71, 72, 77, 63, 75, 61, 79, 80, 78, 69, 76, 67, 82, 73, 74, 81,
   84, 83, 85, 101, 103, 106, 107, 88, 118, 90, 120, 94, 95, 100, 127, 102, 128, 111, 122, 123, 124, 112, 114, 100, 86, 105,
   89, 119, 106, 108, 122, 92, 101, 127, 96, 98, 129, 104, 120, 110, 121, 113, 125, 116, 102, 104, 87, 121, 91, 123, 93, 107,
   108, 128, 103, 129, 105, 97, 99, 118, 119, 109, 115, 117, 126, 94, 89, 115, 96, 126, 100, 119, 114, 110, 106, 124, 101, 113,
   128, 121, 122, 116, 129, 125, 127, 120, 95, 113, 91, 125, 97, 112, 109, 102, 121, 124, 107, 127, 119, 103, 115, 123, 129, 117,
   126, 118, 128, 88, 96, 117, 101, 118, 94, 126, 111, 116, 100, 112, 106, 125, 123, 129, 114, 120, 128, 127, 124, 122, 112, 98,
   93, 113, 109, 124, 99, 123, 104, 127, 118, 125, 108, 117, 105, 128, 121, 115, 119, 126, 129, 90, 116, 97, 120, 103, 111, 117,
   95, 125, 114, 102, 122, 129, 107, 126, 112, 127, 118, 128, 123, 124, 114, 92, 99, 110, 115, 122, 105, 124, 98, 120, 128, 116,
   104, 126, 108, 127, 113, 119, 121, 129, 125, 111, 110, 109, 116, 117, 114, 115, 112, 113, 122, 123, 120, 121, 118, 119, 124, 125,
   126, 129, 128, 127, 130, 153, 155, 151, 152, 135, 173, 137, 174, 139, 140, 147, 180, 149, 182, 157, 166, 167, 175, 162, 164, 151,
   131, 156, 133, 172, 153, 154, 174, 138, 145, 178, 141, 142, 183, 150, 168, 158, 170, 160, 176, 165, 152, 154, 132, 172, 134, 173,
   136, 155, 156, 179, 146, 181, 148, 143, 144, 169, 171, 159, 161, 163, 177, 135, 145, 169, 153, 173, 139, 179, 157, 168, 147, 162,
   151, 178, 167, 184, 164, 174, 182, 180, 175, 166, 147, 133, 171, 141, 181, 151, 172, 166, 158, 153, 180, 145, 160, 184, 170, 174,
   165, 183, 176, 178, 168, 137, 168, 146, 174, 155, 157, 169, 140, 178, 164, 149, 166, 184, 152, 179, 162, 180, 173, 182, 167, 175,
   149, 170, 134, 183, 143, 167, 159, 152, 172, 182, 155, 184, 171, 146, 161, 173, 181, 163, 177, 169, 179, 166, 138, 148, 158, 171,
   174, 156, 180, 142, 168, 184, 165, 150, 181, 154, 178, 160, 172, 170, 183, 176, 167, 150, 136, 170, 159, 182, 144, 173, 154, 184,
   169, 183, 156, 163, 148, 179, 172, 161, 171, 177, 181, 139, 141, 177, 145, 179, 147, 181, 164, 165, 151, 175, 153, 176, 182, 183,
   166, 168, 184, 178, 180, 174, 140, 176, 143, 178, 146, 162, 163, 149, 183, 175, 152, 180, 181, 155, 177, 167, 184, 169, 179, 173,
   182, 175, 142, 144, 160, 161, 180, 148, 182, 150, 178, 179, 176, 154, 177, 156, 184, 170, 171, 172, 181, 183, 162, 160, 159, 176,
   163, 175, 161, 167, 170, 180, 173, 178, 172, 169, 171, 182, 183, 177, 181, 179, 184, 164, 158, 161, 165, 177, 166, 171, 175, 160,
   174, 182, 168, 170, 179, 172, 180, 176, 181, 183, 184, 178, 157, 165, 163, 168, 169, 164, 177, 162, 176, 166, 167, 174, 183, 173,
   181, 175, 178, 179, 184, 182, 180, 185, 212, 213, 206, 207, 188, 230, 190, 232, 194, 195, 200, 239, 202, 241, 217, 228, 229, 236,
   218, 220, 212, 186, 214, 189, 231, 208, 210, 234, 192, 201, 240, 196, 198, 243, 204, 226, 216, 227, 219, 237, 222, 213, 214, 187,
   233, 191, 235, 193, 209, 211, 242, 203, 244, 205, 197, 199, 224, 225, 215, 221, 223, 238, 206, 189, 233, 196, 247, 212, 231, 232,
   216, 208, 245, 201, 219, 249, 227, 234, 222, 243, 237, 240, 226, 207, 231, 191, 247, 197, 230, 215, 213, 233, 246, 209, 248, 225,
   203, 221, 235, 244, 223, 238, 224, 242, 188, 208, 235, 212, 230, 194, 246, 217, 234, 200, 218, 206, 245, 229, 250, 220, 232, 241,
   239, 236, 228, 230, 210, 193, 231, 215, 246, 199, 235, 214, 248, 224, 247, 211, 223, 205, 242, 233, 221, 225, 238, 244, 190, 234,
   209, 232, 213, 217, 235, 195, 245, 220, 202, 228, 250, 207, 246, 218, 239, 230, 241, 229, 236, 232, 192, 211, 216, 233, 234, 214,
   245, 198, 226, 249, 222, 204, 247, 210, 240, 219, 231, 227, 243, 237, 194, 201, 242, 208, 246, 200, 248, 220, 226, 206, 236, 212,
   240, 241, 249, 228, 234, 250, 245, 239, 232, 195, 240, 203, 245, 209, 218, 224, 202, 249, 236, 207, 239, 248, 213, 242, 229, 250,
   235, 246, 230, 241, 200, 196, 244, 201, 248, 206, 247, 228, 222, 212, 239, 208, 237, 250, 243, 232, 226, 249, 240, 245, 234, 239,
   198, 205, 219, 225, 245, 211, 250, 204, 240, 248, 237, 210, 244, 214, 249, 227, 233, 231, 247, 243, 202, 243, 197, 249, 203, 229,
   223, 207, 247, 241, 213, 250, 244, 209, 238, 230, 248, 224, 242, 235, 246, 241, 204, 199, 227, 221, 250, 205, 246, 210, 249, 242,
   243, 214, 238, 211, 248, 231, 225, 233, 244, 247, 217, 226, 224, 234, 235, 220, 242, 218, 240, 228, 229, 232, 249, 230, 248, 236,
   245, 246, 250, 241, 239, 228, 216, 225, 222, 244, 232, 233, 239, 219, 234, 250, 226, 227, 248, 231, 245, 237, 247, 243, 249, 240,
   229, 227, 215, 243, 223, 241, 221, 230, 231, 250, 235, 249, 233, 224, 225, 246, 247, 238, 244, 242, 248, 236, 219, 221, 237, 238,
   239, 225, 241, 227, 245, 246, 240, 231, 242, 233, 250, 243, 244, 247, 248, 249, 218, 237, 223, 240, 224, 236, 238, 229, 243, 239,
   230, 245, 247, 235, 244, 241, 249, 242, 248, 246, 250, 220, 222, 238, 226, 242, 228, 244, 236, 237, 232, 241, 234, 243, 246, 247,
   239, 240, 248, 249, 250, 245, 251, 280, 282, 272, 273, 256, 301, 258, 303, 260, 261, 268, 313, 270, 315, 284, 293, 294, 308, 289,
   291, 278, 252, 283, 254, 299, 274, 275, 304, 259, 266, 311, 262, 263, 316, 271, 295, 285, 297, 287, 309, 292, 279, 281, 253, 300,
   255, 302, 257, 276, 277, 312, 267, 314, 269, 264, 265, 296, 298, 286, 288, 290, 310, 256, 278, 306, 280, 301, 260, 318, 284, 305,
   268, 289, 272, 317, 294, 323, 291, 303, 315, 313, 308, 293, 280, 254, 307, 262, 320, 278, 299, 305, 285, 274, 319, 266, 287, 324,
   297, 304, 292, 316, 309, 311, 295, 258, 305, 279, 303, 282, 284, 306, 261, 317, 291, 270, 293, 323, 273, 318, 289, 313, 301, 315,
   294, 308, 282, 307, 255, 322, 264, 306, 286, 279, 300, 321, 276, 325, 298, 267, 288, 302, 314, 290, 310, 296, 312, 305, 259, 281,
   285, 307, 304, 283, 319, 263, 295, 324, 292, 271, 320, 275, 311, 287, 299, 297, 316, 309, 306, 283, 257, 307, 286, 321, 265, 302,
   281, 325, 296, 322, 277, 290, 269, 312, 300, 288, 298, 310, 314, 260, 274, 321, 278, 318, 268, 327, 291, 304, 272, 308, 280, 319,
   315, 328, 293, 305, 323, 317, 313, 303, 261, 319, 276, 317, 279, 289, 302, 270, 328, 308, 273, 313, 327, 282, 321, 294, 323, 306,
   318, 301, 315, 272, 262, 322, 266, 326, 280, 320, 303, 292, 278, 317, 274, 309, 328, 316, 305, 295, 324, 311, 319, 304, 273, 320,
   264, 326, 267, 301, 290, 282, 322, 318, 279, 327, 314, 276, 310, 306, 325, 296, 312, 302, 321, 317, 263, 277, 287, 300, 319, 281,
   328, 271, 311, 326, 309, 275, 322, 283, 324, 297, 307, 299, 320, 316, 318, 275, 265, 299, 288, 327, 269, 321, 283, 326, 312, 320,
   281, 310, 277, 325, 307, 298, 300, 314, 322, 268, 266, 325, 274, 327, 272, 326, 293, 295, 280, 313, 278, 311, 323, 324, 303, 304,
   328, 319, 317, 305, 270, 324, 267, 328, 276, 294, 296, 273, 326, 315, 282, 323, 325, 279, 312, 301, 327, 302, 321, 306, 318, 323,
   271, 269, 297, 298, 328, 277, 327, 275, 324, 325, 316, 283, 314, 281, 326, 299, 300, 307, 322, 320, 301, 299, 286, 320, 290, 318,
   288, 306, 307, 327, 302, 326, 300, 296, 298, 321, 322, 310, 314, 312, 325, 303, 285, 300, 292, 322, 305, 307, 317, 287, 304, 328,
   295, 297, 326, 299, 319, 309, 320, 316, 324, 311, 284, 304, 302, 305, 306, 291, 321, 289, 319, 293, 294, 303, 328, 301, 327, 308,
   317, 318, 323, 315, 313, 289, 311, 296, 319, 302, 308, 312, 294, 324, 313, 301, 317, 326, 306, 325, 315, 328, 321, 327, 318, 323,
   313, 287, 298, 309, 314, 317, 300, 323, 297, 319, 327, 311, 299, 325, 307, 328, 316, 322, 320, 326, 324, 291, 295, 312, 304, 321,
   293, 325, 308, 311, 303, 315, 305, 324, 318, 326, 313, 319, 327, 328, 323, 317, 315, 297, 288, 316, 310, 323, 298, 318, 299, 328,
   321, 324, 307, 312, 300, 327, 320, 314, 322, 325, 326, 293, 292, 314, 295, 325, 303, 322, 313, 309, 305, 323, 304, 316, 327, 320,
   317, 311, 326, 324, 328, 319, 294, 316, 290, 324, 296, 315, 310, 301, 320, 323, 306, 328, 322, 302, 314, 318, 326, 312, 325, 321,
   327, 308, 309, 310, 311, 312, 313, 314, 315, 316, 317, 318, 319, 320, 321, 322, 323, 324, 325, 326, 327, 328, 329, 356, 358, 350,
   351, 332, 380, 334, 382, 338, 339, 344, 395, 346, 397, 367, 378, 379, 392, 368, 370, 357, 330, 360, 333, 381, 352, 354, 384, 336,
   345, 396, 340, 342, 399, 348, 376, 366, 377, 369, 393, 372, 359, 361, 331, 383, 335, 385, 337, 353, 355, 398, 347, 400, 349, 341,
   343, 374, 375, 365, 371, 373, 394, 362, 333, 389, 340, 405, 357, 381, 388, 366, 352, 403, 345, 369, 411, 377, 384, 372, 399, 393,
   396, 376, 363, 387, 335, 406, 341, 386, 365, 359, 383, 404, 353, 410, 375, 347, 371, 385, 400, 373, 394, 374, 398, 332, 362, 391,
   356, 380, 338, 402, 367, 390, 344, 368, 350, 401, 379, 412, 370, 382, 397, 395, 392, 378, 386, 364, 337, 387, 365, 404, 343, 385,
   361, 410, 374, 406, 355, 373, 349, 398, 383, 371, 375, 394, 400, 334, 390, 363, 382, 358, 367, 391, 339, 401, 370, 346, 378, 412,
   351, 402, 368, 395, 380, 397, 379, 392, 388, 336, 364, 366, 389, 384, 360, 403, 342, 376, 411, 372, 348, 405, 354, 396, 369, 381,
   377, 399, 393, 338, 357, 408, 362, 402, 344, 413, 370, 388, 350, 392, 356, 407, 397, 415, 378, 390, 412, 401, 395, 382, 339, 407,
   359, 401, 363, 368, 386, 346, 415, 392, 351, 395, 413, 358, 408, 379, 412, 391, 402, 380, 397, 356, 340, 409, 345, 414, 362, 405,
   390, 372, 357, 407, 352, 393, 417, 399, 388, 376, 411, 396, 403, 384, 407, 342, 361, 369, 387, 403, 364, 417, 348, 396, 414, 393,
   354, 409, 360, 411, 377, 389, 381, 405, 399, 358, 409, 341, 416, 347, 391, 373, 363, 406, 408, 359, 418, 400, 353, 394, 386, 410,
   374, 398, 385, 404, 408, 360, 343, 389, 371, 418, 349, 404, 364, 416, 398, 409, 361, 394, 355, 410, 387, 375, 383, 400, 406, 350,
   345, 416, 352, 419, 356, 414, 382, 376, 362, 401, 357, 396, 415, 411, 390, 384, 417, 403, 407, 388, 351, 414, 347, 419, 353, 380,
   374, 358, 416, 402, 363, 413, 410, 359, 398, 391, 418, 385, 404, 386, 408, 344, 352, 418, 357, 413, 350, 419, 378, 384, 356, 395,
   362, 403, 412, 417, 382, 388, 415, 407, 401, 390, 413, 354, 349, 381, 375, 419, 355, 418, 360, 414, 410, 405, 364, 400, 361, 416,
   389, 383, 387, 406, 409, 346, 417, 353, 415, 359, 379, 385, 351, 419, 397, 358, 412, 418, 363, 404, 380, 413, 386, 408, 391, 402,
   415, 348, 355, 377, 383, 417, 361, 419, 354, 411, 416, 399, 360, 406, 364, 414, 381, 387, 389, 409, 405, 367, 388, 386, 390, 391,
   370, 408, 368, 407, 378, 379, 382, 415, 380, 413, 392, 401, 402, 412, 397, 395, 390, 366, 387, 372, 409, 388, 389, 407, 369, 384,
   417, 376, 377, 414, 381, 403, 393, 405, 399, 411, 396, 391, 389, 365, 409, 373, 408, 371, 386, 387, 418, 385, 416, 383, 374, 375,
   404, 406, 394, 400, 398, 410, 401, 369, 383, 393, 406, 407, 387, 415, 377, 403, 419, 396, 381, 416, 389, 417, 399, 409, 405, 414,
   411, 402, 381, 371, 405, 394, 413, 375, 408, 389, 419, 404, 414, 387, 398, 383, 418, 409, 400, 406, 410, 416, 368, 403, 385, 407,
   386, 392, 404, 379, 417, 395, 380, 401, 419, 391, 418, 397, 415, 408, 413, 402, 412, 380, 405, 373, 414, 374, 402, 394, 391, 409,
   413, 386, 419, 406, 385, 400, 408, 416, 398, 410, 404, 418, 370, 384, 404, 388, 408, 378, 418, 392, 403, 382, 397, 390, 417, 402,
   419, 395, 407, 413, 415, 412, 401, 382, 372, 406, 376, 416, 390, 409, 401, 393, 388, 415, 384, 399, 419, 405, 407, 396, 414, 411,
   417, 403, 378, 376, 410, 384, 418, 382, 416, 395, 396, 390, 412, 388, 411, 413, 414, 401, 403, 419, 417, 415, 407, 379, 411, 374,
   417, 385, 397, 398, 380, 414, 412, 391, 415, 416, 386, 410, 402, 419, 404, 418, 408, 413, 412, 377, 375, 399, 400, 415, 383, 413,
   381, 417, 418, 411, 389, 410, 387, 419, 405, 406, 409, 416, 414, 392, 396, 398, 403, 404, 395, 410, 397, 411, 401, 402, 407, 414,
   408, 416, 412, 417, 418, 419, 413, 415, 395, 393, 400, 396, 410, 401, 406, 412, 399, 407, 413, 403, 405, 418, 409, 415, 411, 416,
   414, 419, 417, 397, 399, 394, 411, 398, 412, 400, 402, 405, 415, 408, 417, 409, 404, 406, 413, 414, 410, 416, 418, 419, 420, 449,
   451, 441, 442, 425, 476, 427, 478, 429, 430, 437, 494, 439, 496, 459, 468, 469, 489, 464, 466, 447, 421, 452, 423, 474, 443, 444,
   479, 428, 435, 492, 431, 432, 497, 440, 470, 460, 472, 462, 490, 467, 448, 450, 422, 475, 424, 477, 426, 445, 446, 493, 436, 495,
   438, 433, 434, 471, 473, 461, 463, 465, 491, 425, 453, 481, 449, 476, 429, 499, 459, 480, 437, 464, 441, 498, 469, 510, 466, 478,
   496, 494, 489, 468, 455, 423, 484, 431, 501, 447, 474, 482, 460, 443, 500, 435, 462, 511, 472, 479, 467, 497, 490, 492, 470, 427,
   480, 454, 478, 451, 459, 481, 430, 498, 466, 439, 468, 510, 442, 499, 464, 494, 476, 496, 469, 489, 457, 485, 424, 503, 433, 483,
   461, 448, 475, 502, 445, 512, 473, 436, 463, 477, 495, 465, 491, 471, 493, 482, 428, 456, 460, 484, 479, 452, 500, 432, 470, 511,
   467, 440, 501, 444, 492, 462, 474, 472, 497, 490, 483, 458, 426, 485, 461, 502, 434, 477, 450, 512, 471, 503, 446, 465, 438, 493,
   475, 463, 473, 491, 495, 429, 455, 508, 453, 499, 437, 515, 466, 488, 441, 489, 449, 506, 496, 517, 468, 480, 510, 498, 494, 478,
   430, 506, 457, 498, 454, 464, 487, 439, 517, 489, 442, 494, 515, 451, 508, 469, 510, 481, 499, 476, 496, 453, 431, 509, 435, 513,
   455, 501, 488, 467, 447, 504, 443, 490, 518, 497, 482, 470, 511, 492, 500, 479, 454, 507, 433, 514, 436, 487, 465, 457, 503, 505,
   448, 516, 495, 445, 491, 483, 512, 471, 493, 477, 502, 504, 432, 458, 462, 486, 500, 456, 518, 440, 492, 513, 490, 444, 509, 452,
   511, 472, 484, 474, 501, 497, 505, 456, 434, 486, 463, 516, 438, 502, 458, 514, 493, 507, 450, 491, 446, 512, 485, 473, 475, 495,
   503, 437, 447, 520, 455, 515, 441, 522, 468, 482, 449, 494, 453, 504, 510, 519, 478, 488, 517, 506, 498, 480, 449, 435, 521, 443,
   523, 453, 513, 480, 470, 455, 506, 447, 492, 519, 511, 488, 479, 518, 500, 504, 482, 439, 519, 448, 517, 457, 469, 483, 442, 522,
   496, 451, 510, 520, 454, 505, 476, 515, 487, 508, 481, 499, 451, 521, 436, 524, 445, 481, 471, 454, 514, 508, 457, 520, 512, 448,
   493, 487, 516, 477, 502, 483, 505, 519, 440, 450, 472, 485, 518, 458, 523, 444, 511, 521, 497, 452, 507, 456, 513, 474, 486, 484,
   509, 501, 520, 452, 438, 484, 473, 524, 446, 516, 456, 521, 512, 509, 458, 495, 450, 514, 486, 475, 485, 503, 507, 441, 443, 524,
   447, 522, 449, 523, 478, 479, 453, 498, 455, 500, 517, 518, 480, 482, 519, 504, 506, 488, 442, 523, 445, 522, 448, 476, 477, 451,
   524, 499, 454, 515, 516, 457, 502, 481, 520, 483, 505, 487, 508, 522, 444, 446, 474, 475, 523, 450, 524, 452, 513, 514, 501, 456,
   503, 458, 521, 484, 485, 486, 507, 509, 487, 486, 461, 507, 465, 505, 463, 483, 485, 516, 477, 514, 475, 471, 473, 502, 503, 491,
   495, 493, 512, 488, 460, 486, 467, 509, 482, 484, 504, 462, 479, 518, 470, 472, 513, 474, 500, 490, 501, 497, 511, 492, 459, 488,
   487, 480, 481, 466, 508, 464, 506, 468, 469, 478, 517, 476, 515, 489, 498, 499, 510, 496, 494, 464, 504, 483, 506, 487, 489, 505,
   469, 519, 494, 476, 498, 522, 481, 520, 496, 517, 508, 515, 499, 510, 506, 462, 485, 490, 507, 504, 486, 519, 472, 500, 523, 492,
   474, 521, 484, 518, 497, 509, 501, 513, 511, 466, 482, 505, 488, 508, 468, 520, 489, 504, 478, 496, 480, 519, 499, 522, 494, 506,
   515, 517, 510, 498, 508, 484, 463, 509, 491, 520, 473, 505, 486, 524, 502, 521, 485, 493, 475, 516, 507, 495, 503, 512, 514, 480,
   467, 507, 470, 521, 488, 509, 506, 490, 482, 519, 479, 497, 523, 501, 504, 492, 513, 511, 518, 500, 481, 509, 465, 521, 471, 508,
   491, 487, 507, 520, 483, 524, 503, 477, 495, 505, 514, 493, 512, 502, 516, 476, 513, 471, 523, 477, 499, 493, 481, 521, 515, 487,
   522, 514, 483, 512, 508, 524, 502, 516, 505, 520, 515, 474, 473, 501, 495, 522, 475, 520, 484, 523, 516, 513, 486, 512, 485, 524,
   509, 503, 507, 514, 521, 478, 470, 514, 479, 524, 480, 521, 498, 492, 488, 517, 482, 511, 522, 513, 506, 500, 523, 518, 519, 504,
   517, 472, 475, 497, 503, 519, 485, 522, 474, 518, 524, 511, 484, 514, 486, 523, 501, 507, 509, 521, 513, 468, 479, 516, 482, 520,
   478, 524, 494, 500, 480, 510, 488, 518, 515, 523, 498, 504, 522, 519, 517, 506, 469, 518, 477, 519, 483, 496, 502, 476, 523, 510,
   481, 517, 524, 487, 516, 499, 522, 505, 520, 508, 515, 489, 500, 502, 504, 505, 494, 516, 496, 518, 498, 499, 506, 523, 508, 524,
   510, 519, 520, 522, 515, 517, 498, 490, 503, 492, 514, 506, 507, 517, 497, 504, 522, 500, 501, 524, 509, 519, 511, 521, 513, 523,
   518, 499, 501, 491, 513, 493, 515, 495, 508, 509, 522, 505, 523, 507, 502, 503, 520, 521, 512, 514, 516, 524, 494, 492, 512, 500,
   516, 498, 514, 510, 511, 506, 515, 504, 513, 520, 521, 517, 518, 524, 523, 522, 519, 496, 511, 493, 518, 502, 510, 512, 499, 513,
   517, 508, 519, 521, 505, 514, 515, 523, 516, 524, 520, 522, 510, 497, 495, 511, 512, 517, 503, 515, 501, 519, 520, 518, 509, 516,
   507, 522, 513, 514, 521, 524, 523, 525, 552, 554, 546, 547, 528, 582, 530, 584, 534, 535, 540, 603, 542, 605, 569, 580, 581, 600,
   570, 572, 553, 526, 556, 529, 583, 548, 550, 586, 532, 541, 604, 536, 538, 607, 544, 578, 568, 579, 571, 601, 574, 555, 557, 527,
   585, 531, 587, 533, 549, 551, 606, 543, 608, 545, 537, 539, 576, 577, 567, 573, 575, 602, 560, 529, 591, 536, 613, 553, 583, 590,
   568, 548, 611, 541, 571, 625, 579, 586, 574, 607, 601, 604, 578, 561, 589, 531, 614, 537, 588, 567, 555, 585, 612, 549, 624, 577,
   543, 573, 587, 608, 575, 602, 576, 606, 528, 558, 593, 552, 582, 534, 610, 569, 592, 540, 570, 546, 609, 581, 626, 572, 584, 605,
   603, 600, 580, 588, 563, 533, 589, 567, 612, 539, 587, 557, 624, 576, 614, 551, 575, 545, 606, 585, 573, 577, 602, 608, 530, 592,
   559, 584, 554, 569, 593, 535, 609, 572, 542, 580, 626, 547, 610, 570, 603, 582, 605, 581, 600, 590, 532, 562, 568, 591, 586, 556,
   611, 538, 578, 625, 574, 544, 613, 550, 604, 571, 583, 579, 607, 601, 534, 564, 617, 558, 610, 540, 627, 572, 596, 546, 600, 552,
   615, 605, 629, 580, 592, 626, 609, 603, 584, 535, 615, 565, 609, 559, 570, 594, 542, 629, 600, 547, 603, 627, 554, 617, 581, 626,
   593, 610, 582, 605, 564, 536, 619, 541, 628, 560, 613, 598, 574, 553, 616, 548, 601, 631, 607, 590, 578, 625, 604, 611, 586, 616,
   538, 566, 571, 595, 611, 562, 631, 544, 604, 628, 601, 550, 619, 556, 625, 579, 591, 583, 613, 607, 565, 620, 537, 630, 543, 599,
   575, 561, 614, 618, 555, 632, 608, 549, 602, 588, 624, 576, 606, 587, 612, 618, 566, 539, 597, 573, 632, 545, 612, 563, 630, 606,
   620, 557, 602, 551, 624, 589, 577, 585, 608, 614, 558, 541, 636, 548, 640, 564, 628, 596, 578, 560, 621, 553, 604, 635, 625, 598,
   586, 631, 611, 616, 590, 559, 634, 543, 641, 549, 594, 576, 565, 630, 622, 561, 633, 624, 555, 606, 599, 632, 587, 612, 588, 618,
   540, 560, 638, 564, 627, 546, 639, 580, 598, 552, 603, 558, 621, 626, 637, 584, 596, 629, 615, 609, 592, 633, 562, 545, 595, 577,
   641, 551, 632, 566, 634, 624, 623, 563, 608, 557, 630, 597, 585, 589, 614, 620, 542, 637, 561, 629, 565, 581, 599, 547, 639, 605,
   554, 626, 638, 559, 622, 582, 627, 594, 617, 593, 610, 635, 544, 563, 579, 597, 631, 566, 640, 550, 625, 636, 607, 556, 623, 562,
   628, 583, 595, 591, 619, 613, 546, 553, 643, 560, 639, 552, 642, 584, 590, 558, 609, 564, 616, 629, 635, 592, 598, 637, 621, 615,
   596, 547, 642, 555, 639, 561, 582, 588, 554, 643, 610, 559, 627, 633, 565, 618, 593, 638, 599, 622, 594, 617, 552, 548, 644, 553,
   642, 558, 640, 592, 586, 564, 615, 560, 611, 637, 631, 596, 590, 635, 616, 621, 598, 642, 550, 557, 583, 589, 640, 563, 644, 556,
   628, 634, 613, 562, 620, 566, 636, 591, 597, 595, 623, 619, 554, 644, 549, 643, 555, 593, 587, 559, 641, 617, 565, 638, 632, 561,
   612, 594, 633, 588, 618, 599, 622, 643, 556, 551, 591, 585, 644, 557, 641, 562, 636, 630, 619, 566, 614, 563, 634, 595, 589, 597,
   620, 623, 569, 596, 594, 592, 593, 572, 617, 570, 615, 580, 581, 584, 629, 582, 627, 600, 609, 610, 626, 605, 603, 598, 568, 595,
   574, 619, 590, 591, 616, 571, 586, 631, 578, 579, 628, 583, 611, 601, 613, 607, 625, 604, 599, 597, 567, 620, 575, 618, 573, 588,
   589, 632, 587, 630, 585, 576, 577, 612, 614, 602, 608, 606, 624, 621, 571, 597, 601, 623, 616, 595, 635, 579, 611, 640, 604, 583,
   636, 591, 631, 607, 619, 613, 628, 625, 622, 595, 573, 623, 602, 633, 577, 618, 597, 641, 612, 634, 589, 606, 585, 632, 620, 608,
   614, 624, 630, 570, 621, 599, 615, 594, 600, 622, 581, 637, 603, 582, 609, 639, 593, 638, 605, 629, 617, 627, 610, 626, 594, 623,
   575, 634, 576, 622, 602, 599, 620, 633, 588, 641, 614, 587, 608, 618, 630, 606, 624, 612, 632, 572, 598, 622, 596, 617, 580, 638,
   600, 621, 584, 605, 592, 637, 610, 639, 603, 615, 627, 629, 626, 609, 596, 574, 623, 578, 636, 598, 619, 621, 601, 590, 635, 586,
   607, 640, 613, 616, 604, 628, 625, 631, 611, 580, 590, 633, 598, 638, 584, 643, 603, 616, 592, 626, 596, 635, 627, 642, 609, 621,
   639, 637, 629, 615, 581, 635, 588, 637, 599, 605, 618, 582, 642, 626, 593, 629, 643, 594, 633, 610, 639, 622, 638, 617, 627, 592,
   578, 634, 586, 644, 596, 636, 615, 604, 598, 637, 590, 625, 642, 628, 621, 611, 640, 631, 635, 616, 593, 636, 576, 644, 587, 617,
   606, 594, 634, 638, 599, 643, 630, 588, 624, 622, 641, 612, 632, 618, 633, 637, 579, 589, 607, 620, 635, 597, 642, 583, 631, 644,
   625, 591, 634, 595, 640, 613, 623, 619, 636, 628, 638, 591, 577, 619, 608, 643, 585, 633, 595, 644, 632, 636, 597, 624, 589, 641,
   623, 614, 620, 630, 634, 639, 583, 585, 613, 614, 642, 589, 643, 591, 640, 641, 628, 595, 630, 597, 644, 619, 620, 623, 634, 636,
   582, 640, 587, 642, 588, 610, 612, 593, 644, 627, 594, 639, 641, 599, 632, 617, 643, 618, 633, 622, 638, 584, 586, 641, 590, 643,
   592, 644, 609, 611, 596, 629, 598, 631, 639, 640, 615, 616, 642, 635, 637, 621, 600, 616, 618, 621, 622, 603, 633, 605, 635, 609,
   610, 615, 642, 617, 643, 626, 637, 638, 639, 627, 629, 615, 601, 620, 604, 634, 621, 623, 637, 607, 616, 642, 611, 613, 644, 619,
   635, 625, 636, 628, 640, 631, 617, 619, 602, 636, 606, 638, 608, 622, 623, 643, 618, 644, 620, 612, 614, 633, 634, 624, 630, 632,
   641, 609, 604, 630, 611, 641, 615, 634, 629, 625, 621, 639, 616, 628, 643, 636, 637, 631, 644, 640, 642, 635, 610, 628, 606, 640,
   612, 627, 624, 617, 636, 639, 622, 642, 634, 618, 630, 638, 644, 632, 641, 633, 643, 603, 611, 632, 616, 633, 609, 641, 626, 631,
   615, 627, 621, 640, 638, 644, 629, 635, 643, 642, 639, 637, 627, 613, 608, 628, 624, 639, 614, 638, 619, 642, 633, 640, 623, 632,
   620, 643, 636, 630, 634, 641, 644, 605, 631, 612, 635, 618, 626, 632, 610, 640, 629, 617, 637, 644, 622, 641, 627, 642, 633, 643,
   638, 639, 629, 607, 614, 625, 630, 637, 620, 639, 613, 635, 643, 631, 619, 641, 623, 642, 628, 634, 636, 644, 640, 626, 625, 624,
   631, 632, 629, 630, 627, 628, 637, 638, 635, 636, 633, 634, 639, 640, 641, 644, 643, 642
}; // 9.02 kb
static const cart_index_t iCartXY_ab14_a6[4620] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 31, 32, 33, 39, 35, 41, 50, 51, 37, 38, 43, 44, 52, 53, 45, 47, 60, 55, 57, 49, 54, 62, 56,
   63, 58, 59, 61, 33, 29, 34, 37, 31, 49, 51, 36, 42, 45, 54, 39, 55, 40, 46, 43, 59, 48, 53, 50, 56, 61,
   52, 58, 60, 57, 63, 62, 35, 36, 30, 49, 50, 38, 32, 40, 34, 56, 47, 57, 41, 48, 42, 58, 44, 46, 51, 52,
   54, 59, 60, 61, 53, 62, 55, 63, 64, 75, 76, 67, 80, 69, 82, 95, 96, 73, 74, 85, 86, 104, 105, 79, 81, 108,
   97, 99, 90, 91, 106, 93, 107, 101, 102, 103, 73, 65, 78, 79, 68, 93, 94, 71, 84, 85, 103, 75, 105, 77, 87, 80,
   107, 83, 98, 89, 101, 106, 92, 99, 100, 95, 108, 104, 74, 77, 66, 91, 92, 81, 70, 83, 72, 103, 86, 104, 76, 87,
   78, 106, 82, 84, 88, 100, 102, 97, 98, 107, 94, 108, 96, 105, 67, 68, 88, 73, 75, 90, 96, 89, 94, 79, 91, 80,
   97, 92, 98, 85, 102, 100, 105, 95, 93, 103, 104, 101, 108, 99, 107, 106, 69, 89, 70, 90, 95, 74, 76, 92, 88, 93,
   81, 99, 82, 100, 94, 101, 86, 98, 96, 104, 91, 102, 108, 103, 105, 106, 97, 107, 90, 71, 72, 93, 89, 91, 88, 77,
   78, 101, 102, 95, 96, 83, 84, 99, 97, 87, 94, 92, 103, 107, 100, 106, 98, 104, 105, 108, 109, 124, 125, 114, 132, 116,
   134, 147, 148, 118, 119, 130, 131, 157, 158, 126, 128, 163, 152, 153, 136, 141, 159, 143, 161, 145, 146, 154, 126, 110, 127, 130,
   112, 145, 150, 117, 135, 132, 159, 120, 160, 121, 133, 124, 163, 129, 151, 137, 153, 157, 139, 147, 149, 144, 162, 155, 128, 129,
   111, 146, 149, 131, 113, 133, 115, 161, 134, 162, 122, 135, 123, 163, 125, 127, 138, 151, 152, 148, 150, 158, 140, 160, 142, 156,
   118, 112, 140, 126, 120, 143, 156, 137, 150, 130, 154, 124, 158, 139, 151, 132, 161, 149, 160, 144, 145, 159, 155, 153, 162, 147,
   163, 157, 119, 139, 113, 141, 155, 128, 122, 149, 138, 154, 131, 157, 125, 151, 140, 159, 134, 150, 142, 162, 146, 152, 160, 161,
   156, 163, 148, 158, 114, 120, 142, 118, 124, 136, 148, 144, 156, 126, 141, 132, 152, 155, 160, 130, 146, 162, 158, 147, 143, 154,
   157, 145, 163, 153, 161, 159, 141, 121, 115, 154, 139, 146, 138, 129, 123, 159, 152, 155, 142, 133, 127, 157, 148, 135, 140, 149,
   161, 158, 151, 163, 150, 162, 156, 160, 116, 144, 122, 136, 147, 119, 125, 155, 142, 143, 128, 153, 134, 162, 156, 145, 131, 160,
   148, 157, 141, 146, 163, 154, 158, 159, 152, 161, 143, 117, 123, 145, 137, 154, 140, 121, 127, 153, 161, 144, 156, 129, 135, 147,
   158, 133, 150, 139, 159, 163, 149, 157, 151, 155, 160, 162, 136, 137, 138, 143, 144, 141, 142, 139, 140, 145, 146, 147, 148, 149,
   150, 153, 152, 151, 156, 155, 154, 161, 162, 159, 160, 157, 158, 163, 164, 187, 188, 167, 191, 169, 192, 213, 214, 173, 174, 185,
   186, 224, 225, 179, 181, 229, 209, 211, 196, 197, 218, 199, 220, 207, 208, 215, 185, 165, 190, 191, 168, 211, 212, 171, 193, 187,
   224, 175, 226, 177, 189, 180, 228, 183, 210, 195, 213, 219, 198, 205, 206, 201, 222, 216, 186, 189, 166, 209, 210, 192, 170, 193,
   172, 225, 188, 226, 176, 190, 178, 227, 182, 184, 194, 212, 214, 203, 204, 221, 200, 223, 202, 217, 167, 180, 203, 173, 187, 196,
   214, 205, 221, 179, 197, 191, 209, 219, 227, 185, 208, 228, 225, 213, 199, 215, 224, 207, 229, 211, 220, 218, 179, 168, 204, 185,
   175, 207, 223, 195, 212, 191, 218, 180, 227, 198, 210, 187, 229, 206, 226, 201, 211, 224, 216, 213, 222, 205, 228, 219, 169, 205,
   182, 196, 213, 174, 188, 219, 203, 199, 181, 211, 192, 228, 221, 207, 186, 227, 214, 224, 197, 208, 229, 215, 225, 218, 209, 220,
   181, 206, 170, 208, 222, 186, 176, 210, 194, 220, 192, 228, 182, 212, 200, 229, 188, 204, 202, 226, 209, 214, 223, 225, 217, 227,
   203, 221, 207, 171, 184, 211, 195, 218, 204, 177, 190, 213, 229, 201, 223, 183, 193, 205, 227, 189, 212, 198, 224, 228, 206, 219,
   210, 216, 226, 222, 208, 183, 172, 220, 206, 209, 194, 189, 178, 229, 214, 222, 202, 193, 184, 228, 203, 190, 200, 210, 225, 221,
   212, 227, 204, 226, 217, 223, 173, 175, 217, 179, 180, 199, 221, 201, 223, 185, 215, 187, 225, 216, 226, 191, 220, 222, 227, 205,
   207, 218, 219, 211, 228, 213, 229, 224, 174, 216, 176, 197, 219, 181, 182, 222, 202, 215, 186, 224, 188, 226, 217, 218, 192, 223,
   203, 228, 208, 209, 227, 220, 221, 229, 214, 225, 215, 177, 178, 218, 198, 220, 200, 183, 184, 224, 225, 216, 217, 189, 190, 219,
   221, 193, 204, 206, 229, 227, 210, 228, 212, 222, 223, 226, 197, 198, 194, 215, 216, 208, 202, 206, 200, 218, 209, 219, 203, 210,
   204, 224, 214, 212, 217, 222, 220, 225, 226, 229, 223, 228, 221, 227, 199, 195, 200, 207, 201, 215, 217, 198, 204, 211, 220, 205,
   221, 206, 212, 213, 225, 210, 223, 216, 218, 229, 222, 224, 226, 219, 227, 228, 196, 201, 202, 199, 205, 197, 203, 216, 217, 207,
   208, 213, 214, 222, 223, 211, 209, 226, 221, 219, 215, 220, 228, 218, 227, 224, 225, 229, 230, 257, 258, 235, 259, 237, 261, 284,
   285, 239, 240, 251, 252, 296, 297, 247, 249, 302, 280, 282, 263, 268, 292, 270, 294, 272, 273, 287, 259, 231, 260, 257, 233, 284,
   286, 238, 262, 253, 298, 241, 299, 242, 254, 245, 303, 250, 278, 264, 283, 290, 266, 274, 276, 271, 295, 288, 261, 262, 232, 285,
   286, 258, 234, 260, 236, 300, 255, 301, 243, 256, 244, 304, 246, 248, 265, 279, 281, 275, 277, 291, 267, 293, 269, 289, 251, 233,
   279, 259, 241, 282, 301, 264, 286, 257, 296, 245, 305, 266, 278, 253, 307, 276, 299, 271, 284, 298, 288, 283, 295, 274, 303, 290,
   252, 278, 234, 280, 299, 261, 243, 286, 265, 297, 258, 305, 246, 279, 267, 306, 255, 277, 269, 301, 285, 281, 293, 300, 289, 304,
   275, 291, 235, 253, 281, 239, 257, 263, 285, 283, 300, 247, 268, 259, 280, 298, 306, 251, 273, 307, 297, 284, 270, 287, 296, 272,
   302, 282, 294, 292, 280, 254, 236, 297, 278, 285, 265, 262, 244, 306, 281, 299, 269, 260, 248, 305, 275, 256, 267, 286, 300, 291,
   279, 304, 277, 301, 289, 293, 237, 283, 255, 263, 284, 240, 258, 298, 281, 270, 249, 282, 261, 307, 300, 272, 252, 306, 285, 296,
   268, 273, 302, 287, 297, 292, 280, 294, 282, 238, 256, 284, 264, 296, 279, 242, 260, 283, 307, 271, 301, 250, 262, 274, 305, 254,
   286, 266, 298, 303, 276, 290, 278, 288, 299, 295, 239, 245, 291, 247, 253, 270, 300, 274, 304, 251, 287, 257, 297, 290, 305, 259,
   294, 303, 306, 283, 272, 292, 298, 282, 307, 284, 302, 296, 240, 290, 246, 268, 298, 249, 255, 303, 275, 287, 252, 296, 258, 305,
   291, 292, 261, 304, 281, 307, 273, 280, 306, 294, 300, 302, 285, 297, 247, 241, 293, 251, 245, 272, 304, 271, 301, 259, 292, 253,
   306, 288, 299, 257, 302, 295, 305, 274, 282, 296, 290, 284, 303, 283, 307, 298, 292, 242, 248, 296, 266, 302, 277, 250, 256, 298,
   306, 288, 293, 254, 260, 290, 304, 262, 279, 276, 307, 305, 278, 303, 286, 295, 301, 299, 249, 295, 243, 273, 303, 252, 246, 299,
   269, 294, 261, 307, 255, 301, 289, 302, 258, 293, 275, 305, 280, 285, 304, 297, 291, 306, 281, 300, 294, 250, 244, 302, 276, 297,
   267, 254, 248, 307, 300, 295, 289, 262, 256, 303, 291, 260, 277, 278, 306, 304, 286, 305, 279, 299, 293, 301, 263, 274, 275, 270,
   283, 268, 281, 290, 291, 272, 273, 284, 285, 303, 304, 282, 280, 305, 300, 298, 287, 294, 307, 292, 306, 296, 297, 302, 272, 264,
   277, 282, 271, 292, 293, 266, 279, 284, 302, 274, 304, 276, 286, 283, 306, 278, 301, 288, 296, 307, 295, 298, 299, 290, 305, 303,
   273, 276, 265, 294, 295, 280, 269, 278, 267, 302, 285, 303, 275, 286, 277, 307, 281, 279, 289, 299, 297, 300, 301, 306, 293, 305,
   291, 304, 287, 266, 267, 292, 288, 294, 289, 276, 277, 296, 297, 290, 291, 278, 279, 298, 300, 286, 293, 295, 302, 306, 299, 307,
   301, 303, 304, 305, 268, 288, 269, 287, 290, 273, 275, 295, 289, 292, 280, 298, 281, 299, 293, 296, 285, 301, 291, 303, 294, 297,
   305, 302, 304, 307, 300, 306, 270, 271, 289, 272, 274, 287, 291, 288, 293, 282, 294, 283, 300, 295, 301, 284, 297, 299, 304, 290,
   292, 302, 303, 296, 305, 298, 306, 307, 308, 341, 342, 311, 335, 313, 337, 369, 370, 317, 318, 329, 330, 380, 381, 323, 325, 391,
   359, 361, 346, 347, 374, 349, 376, 357, 358, 371, 341, 309, 343, 336, 312, 367, 368, 315, 339, 331, 382, 319, 384, 321, 333, 324,
   390, 327, 360, 345, 363, 375, 348, 355, 356, 351, 378, 372, 342, 343, 310, 365, 366, 338, 314, 340, 316, 383, 332, 385, 320, 334,
   322, 389, 326, 328, 344, 362, 364, 353, 354, 377, 350, 379, 352, 373, 311, 336, 365, 317, 341, 346, 370, 367, 387, 323, 347, 335,
   359, 386, 392, 329, 358, 394, 381, 369, 349, 371, 380, 357, 391, 361, 376, 374, 335, 312, 366, 341, 319, 369, 388, 345, 368, 336,
   386, 324, 393, 348, 360, 331, 396, 356, 384, 351, 367, 382, 372, 363, 378, 355, 390, 375, 313, 367, 338, 346, 369, 318, 342, 386,
   365, 349, 325, 361, 337, 394, 387, 357, 330, 392, 370, 380, 347, 358, 391, 371, 381, 374, 359, 376, 337, 368, 314, 370, 388, 342,
   320, 366, 344, 387, 338, 395, 326, 362, 350, 397, 332, 354, 352, 385, 365, 364, 379, 383, 373, 389, 353, 377, 369, 315, 340, 367,
   345, 386, 366, 321, 343, 363, 396, 351, 388, 327, 339, 355, 393, 333, 368, 348, 382, 390, 356, 375, 360, 372, 384, 378, 370, 339,
   316, 387, 368, 365, 344, 343, 322, 397, 364, 388, 352, 340, 328, 395, 353, 334, 350, 366, 383, 377, 362, 389, 354, 385, 373, 379,
   317, 331, 383, 323, 336, 349, 387, 363, 397, 329, 371, 341, 381, 382, 398, 335, 376, 396, 392, 367, 357, 374, 386, 361, 394, 369,
   391, 380, 318, 382, 332, 347, 386, 325, 338, 396, 364, 371, 330, 380, 342, 398, 383, 374, 337, 397, 365, 394, 358, 359, 392, 376,
   387, 391, 370, 381, 329, 319, 385, 335, 324, 361, 395, 351, 388, 341, 380, 331, 398, 372, 384, 336, 394, 378, 393, 355, 369, 386,
   375, 367, 390, 363, 396, 382, 330, 384, 320, 359, 393, 337, 326, 388, 352, 381, 342, 398, 332, 385, 373, 392, 338, 379, 353, 395,
   370, 365, 389, 387, 377, 397, 364, 383, 380, 321, 334, 386, 348, 394, 362, 327, 340, 382, 398, 372, 385, 333, 343, 375, 395, 339,
   366, 356, 396, 393, 360, 390, 368, 378, 388, 384, 381, 333, 322, 392, 360, 387, 350, 339, 328, 398, 383, 384, 373, 343, 334, 393,
   377, 340, 354, 368, 397, 389, 366, 395, 362, 388, 379, 385, 323, 324, 389, 329, 331, 357, 397, 355, 395, 335, 374, 336, 392, 375,
   393, 341, 391, 390, 398, 363, 361, 380, 382, 369, 396, 367, 394, 386, 325, 390, 326, 358, 396, 330, 332, 393, 353, 376, 337, 394,
   338, 395, 377, 391, 342, 389, 364, 398, 359, 370, 397, 381, 383, 392, 365, 387, 391, 327, 328, 394, 356, 392, 354, 333, 334, 396,
   397, 378, 379, 339, 340, 390, 389, 343, 362, 360, 398, 395, 368, 393, 366, 384, 385, 388, 359, 360, 344, 381, 384, 370, 352, 368,
   350, 392, 365, 393, 353, 366, 354, 398, 364, 362, 373, 388, 387, 383, 385, 397, 379, 395, 377, 389, 361, 345, 362, 369, 351, 380,
   385, 348, 366, 367, 394, 355, 395, 356, 368, 363, 398, 360, 388, 372, 386, 396, 378, 382, 384, 375, 393, 390, 346, 363, 364, 349,
   367, 347, 365, 382, 383, 357, 358, 369, 370, 396, 397, 361, 359, 398, 387, 386, 371, 376, 394, 374, 392, 380, 381, 391, 347, 375,
   353, 371, 382, 358, 364, 390, 377, 374, 359, 386, 365, 393, 389, 380, 370, 395, 383, 396, 376, 381, 398, 391, 397, 394, 387, 392,
   374, 348, 354, 380, 372, 391, 379, 356, 362, 386, 392, 375, 389, 360, 366, 382, 397, 368, 385, 378, 394, 398, 384, 396, 388, 390,
   395, 393, 349, 355, 377, 357, 363, 371, 383, 375, 389, 361, 376, 367, 387, 390, 395, 369, 381, 393, 397, 382, 374, 391, 396, 380,
   398, 386, 392, 394, 376, 356, 350, 391, 378, 381, 373, 360, 354, 394, 387, 390, 377, 368, 362, 396, 383, 366, 379, 384, 392, 397,
   388, 398, 385, 393, 389, 395, 357, 351, 379, 361, 355, 374, 389, 372, 385, 369, 391, 363, 397, 378, 388, 367, 392, 384, 395, 375,
   380, 394, 390, 386, 393, 382, 398, 396, 358, 378, 352, 376, 390, 359, 353, 384, 373, 391, 370, 396, 364, 388, 379, 394, 365, 385,
   377, 393, 381, 387, 395, 392, 389, 398, 383, 397, 371, 372, 373, 374, 375, 376, 377, 378, 379, 380, 381, 382, 383, 384, 385, 386,
   387, 388, 389, 390, 391, 392, 393, 394, 395, 396, 397, 398, 399, 432, 433, 404, 428, 406, 430, 459, 460, 408, 409, 420, 421, 477,
   478, 416, 418, 489, 455, 457, 438, 443, 473, 445, 475, 447, 448, 468, 434, 400, 435, 426, 402, 461, 463, 407, 431, 422, 479, 410,
   480, 411, 423, 414, 490, 419, 453, 439, 458, 471, 441, 449, 451, 446, 476, 469, 436, 437, 401, 462, 464, 427, 403, 429, 405, 481,
   424, 482, 412, 425, 413, 491, 415, 417, 440, 454, 456, 450, 452, 472, 442, 474, 444, 470, 432, 402, 465, 434, 410, 467, 488, 439,
   463, 426, 483, 414, 492, 441, 453, 422, 497, 451, 480, 446, 461, 479, 469, 458, 476, 449, 490, 471, 433, 465, 403, 466, 486, 436,
   412, 464, 440, 484, 427, 493, 415, 454, 442, 495, 424, 452, 444, 482, 462, 456, 474, 481, 470, 491, 450, 472, 404, 434, 466, 408,
   432, 438, 460, 467, 487, 416, 443, 428, 455, 485, 494, 420, 448, 496, 478, 459, 445, 468, 477, 447, 489, 457, 475, 473, 466, 435,
   405, 484, 465, 462, 440, 437, 413, 495, 456, 486, 444, 429, 417, 493, 450, 425, 442, 464, 481, 472, 454, 491, 452, 482, 470, 474,
   406, 467, 436, 438, 459, 409, 433, 485, 466, 445, 418, 457, 430, 496, 487, 447, 421, 494, 460, 477, 443, 448, 489, 468, 478, 473,
   455, 475, 467, 407, 437, 461, 439, 483, 465, 411, 435, 458, 497, 446, 488, 419, 431, 449, 492, 423, 463, 441, 479, 490, 451, 471,
   453, 469, 480, 476, 408, 426, 484, 416, 434, 445, 487, 461, 499, 420, 468, 432, 478, 483, 501, 428, 475, 498, 494, 467, 447, 473,
   485, 457, 496, 459, 489, 477, 409, 483, 427, 443, 485, 418, 436, 498, 462, 468, 421, 477, 433, 501, 484, 473, 430, 499, 466, 496,
   448, 455, 494, 475, 487, 489, 460, 478, 428, 410, 486, 432, 414, 459, 500, 446, 488, 434, 485, 422, 502, 469, 480, 426, 498, 476,
   492, 449, 467, 483, 471, 461, 490, 458, 497, 479, 485, 411, 429, 483, 441, 498, 464, 419, 437, 479, 502, 469, 486, 423, 435, 471,
   500, 431, 465, 451, 497, 492, 453, 490, 463, 476, 488, 480, 430, 488, 412, 460, 500, 433, 415, 486, 444, 487, 436, 503, 424, 482,
   470, 499, 427, 474, 450, 493, 466, 462, 491, 484, 472, 495, 456, 481, 487, 431, 413, 499, 463, 484, 442, 435, 417, 503, 481, 488,
   470, 437, 425, 500, 472, 429, 452, 465, 495, 491, 464, 493, 454, 486, 474, 482, 420, 414, 493, 428, 422, 457, 503, 449, 500, 432,
   477, 426, 501, 471, 492, 434, 496, 490, 502, 458, 459, 485, 479, 467, 497, 461, 498, 483, 421, 492, 415, 455, 502, 430, 424, 500,
   450, 478, 433, 501, 427, 493, 472, 494, 436, 491, 456, 503, 460, 466, 495, 487, 481, 499, 462, 484, 416, 422, 495, 420, 426, 447,
   499, 458, 503, 428, 473, 434, 494, 479, 502, 432, 489, 497, 501, 461, 457, 477, 483, 459, 498, 467, 496, 485, 494, 423, 417, 501,
   453, 499, 452, 431, 425, 502, 495, 480, 474, 435, 429, 492, 491, 437, 454, 463, 503, 493, 465, 500, 464, 488, 482, 486, 418, 497,
   424, 448, 498, 421, 427, 502, 456, 475, 430, 496, 436, 503, 481, 489, 433, 495, 462, 501, 455, 460, 499, 478, 484, 494, 466, 487,
   496, 419, 425, 498, 451, 501, 454, 423, 429, 497, 503, 476, 482, 431, 437, 490, 493, 435, 464, 453, 502, 500, 463, 492, 465, 480,
   486, 488, 438, 461, 462, 445, 467, 443, 466, 483, 484, 447, 448, 459, 460, 498, 499, 457, 455, 501, 487, 485, 468, 475, 496, 473,
   494, 477, 478, 489, 459, 439, 464, 467, 446, 485, 486, 441, 465, 461, 498, 449, 500, 451, 463, 458, 502, 453, 488, 469, 483, 497,
   476, 479, 480, 471, 492, 490, 460, 463, 440, 487, 488, 466, 444, 465, 442, 499, 462, 500, 450, 464, 452, 503, 456, 454, 470, 486,
   484, 481, 482, 495, 474, 493, 472, 491, 477, 441, 454, 485, 469, 496, 482, 451, 464, 483, 501, 471, 493, 453, 465, 479, 503, 463,
   486, 476, 498, 502, 480, 497, 488, 490, 500, 492, 478, 453, 442, 494, 480, 487, 470, 463, 452, 501, 484, 492, 472, 465, 454, 502,
   481, 464, 474, 488, 499, 495, 486, 503, 482, 500, 491, 493, 443, 479, 456, 468, 483, 448, 462, 497, 481, 473, 455, 485, 466, 502,
   495, 477, 460, 503, 484, 498, 475, 478, 501, 489, 499, 496, 487, 494, 455, 480, 444, 478, 492, 460, 450, 488, 470, 494, 466, 502,
   456, 486, 474, 501, 462, 482, 472, 500, 487, 484, 493, 499, 491, 503, 481, 495, 445, 458, 481, 447, 461, 468, 484, 479, 495, 457,
   475, 467, 487, 497, 503, 459, 478, 502, 499, 483, 473, 489, 498, 477, 501, 485, 494, 496, 457, 446, 482, 459, 449, 477, 493, 469,
   486, 467, 496, 458, 503, 476, 488, 461, 501, 480, 500, 471, 485, 498, 490, 483, 492, 479, 502, 497, 447, 449, 491, 457, 458, 473,
   495, 471, 493, 459, 489, 461, 499, 490, 500, 467, 494, 492, 503, 479, 477, 496, 497, 485, 502, 483, 501, 498, 448, 490, 450, 475,
   497, 455, 456, 492, 472, 489, 460, 498, 462, 500, 491, 496, 466, 493, 481, 502, 478, 487, 503, 494, 495, 501, 484, 499, 489, 451,
   452, 496, 476, 494, 474, 453, 454, 498, 499, 490, 491, 463, 464, 497, 495, 465, 482, 480, 501, 503, 488, 502, 486, 492, 493, 500,
   468, 471, 472, 473, 479, 475, 481, 490, 491, 477, 478, 483, 484, 492, 493, 485, 487, 500, 495, 497, 489, 494, 502, 496, 503, 498,
   499, 501, 473, 469, 474, 477, 471, 489, 491, 476, 482, 485, 494, 479, 495, 480, 486, 483, 499, 488, 493, 490, 496, 501, 492, 498,
   500, 497, 503, 502, 475, 476, 470, 489, 490, 478, 472, 480, 474, 496, 487, 497, 481, 488, 482, 498, 484, 486, 491, 492, 494, 499,
   500, 501, 493, 502, 495, 503, 504, 537, 538, 507, 531, 509, 533, 571, 572, 513, 514, 525, 526, 588, 589, 519, 521, 605, 561, 563,
   548, 549, 582, 551, 584, 559, 560, 579, 539, 505, 541, 532, 508, 569, 570, 511, 535, 527, 590, 515, 592, 517, 529, 520, 604, 523,
   562, 547, 565, 583, 550, 557, 558, 553, 586, 580, 540, 542, 506, 567, 568, 534, 510, 536, 512, 591, 528, 593, 516, 530, 518, 603,
   522, 524, 546, 564, 566, 555, 556, 585, 552, 587, 554, 581, 507, 543, 573, 513, 537, 548, 572, 575, 596, 519, 549, 531, 561, 594,
   606, 525, 560, 608, 589, 571, 551, 579, 588, 559, 605, 563, 584, 582, 543, 508, 574, 539, 515, 577, 598, 547, 570, 532, 595, 520,
   607, 550, 562, 527, 610, 558, 592, 553, 569, 590, 580, 565, 586, 557, 604, 583, 509, 575, 544, 548, 571, 514, 538, 594, 573, 551,
   521, 563, 533, 608, 596, 559, 526, 606, 572, 588, 549, 560, 605, 579, 589, 582, 561, 584, 544, 576, 510, 578, 599, 540, 516, 568,
   546, 597, 534, 609, 522, 564, 552, 611, 528, 556, 554, 593, 567, 566, 587, 591, 581, 603, 555, 585, 577, 511, 545, 569, 547, 595,
   574, 517, 541, 565, 610, 553, 598, 523, 535, 557, 607, 529, 570, 550, 590, 604, 558, 583, 562, 580, 592, 586, 578, 545, 512, 597,
   576, 567, 546, 542, 518, 611, 566, 599, 554, 536, 524, 609, 555, 530, 552, 568, 591, 585, 564, 603, 556, 593, 581, 587, 513, 539,
   601, 519, 543, 551, 596, 577, 617, 525, 579, 537, 589, 600, 618, 531, 584, 616, 606, 575, 559, 582, 594, 563, 608, 571, 605, 588,
   514, 600, 540, 549, 594, 521, 544, 616, 578, 579, 526, 588, 538, 618, 601, 582, 533, 617, 573, 608, 560, 561, 606, 584, 596, 605,
   572, 589, 537, 515, 602, 543, 520, 575, 615, 553, 598, 539, 600, 527, 619, 580, 592, 532, 614, 586, 607, 557, 577, 595, 583, 569,
   604, 565, 610, 590, 538, 602, 516, 573, 613, 544, 522, 599, 554, 601, 540, 620, 528, 593, 581, 612, 534, 587, 555, 609, 578, 567,
   603, 597, 585, 611, 566, 591, 600, 517, 542, 595, 550, 614, 576, 523, 545, 590, 619, 580, 602, 529, 541, 583, 615, 535, 574, 558,
   610, 607, 562, 604, 570, 586, 598, 592, 601, 541, 518, 612, 574, 597, 552, 545, 524, 620, 591, 602, 581, 542, 530, 613, 585, 536,
   556, 576, 611, 603, 568, 609, 564, 599, 587, 593, 519, 532, 612, 525, 539, 559, 617, 569, 622, 531, 582, 543, 606, 595, 621, 537,
   605, 614, 618, 577, 563, 588, 600, 571, 616, 575, 608, 594, 531, 520, 613, 537, 527, 571, 623, 557, 615, 543, 594, 532, 621, 583,
   607, 539, 616, 604, 619, 565, 575, 600, 590, 577, 610, 569, 614, 595, 521, 614, 534, 560, 616, 526, 540, 621, 567, 584, 533, 608,
   544, 622, 597, 605, 538, 612, 578, 618, 561, 572, 617, 589, 601, 606, 573, 596, 533, 615, 522, 572, 623, 538, 528, 613, 555, 596,
   544, 622, 534, 609, 585, 617, 540, 603, 566, 620, 573, 578, 611, 601, 591, 612, 567, 597, 616, 523, 536, 614, 558, 621, 568, 529,
   542, 610, 623, 586, 599, 535, 545, 604, 613, 541, 576, 562, 619, 615, 570, 607, 574, 592, 602, 598, 617, 535, 524, 622, 570, 612,
   556, 541, 530, 623, 611, 598, 587, 545, 536, 615, 603, 542, 564, 574, 620, 609, 576, 613, 568, 602, 593, 599, 525, 527, 620, 531,
   532, 563, 622, 565, 623, 537, 588, 539, 618, 590, 619, 543, 608, 610, 621, 569, 571, 594, 595, 575, 614, 577, 616, 600, 526, 619,
   528, 561, 621, 533, 534, 623, 566, 589, 538, 618, 540, 620, 591, 606, 544, 611, 567, 622, 572, 573, 612, 596, 597, 617, 578, 601,
   618, 529, 530, 621, 562, 622, 564, 535, 536, 619, 620, 592, 593, 541, 542, 607, 609, 545, 568, 570, 623, 613, 574, 615, 576, 598,
   599, 602, 573, 574, 546, 601, 602, 578, 554, 576, 552, 612, 567, 613, 555, 568, 556, 620, 566, 564, 581, 599, 597, 591, 593, 611,
   587, 609, 585, 603, 575, 547, 576, 577, 553, 600, 602, 550, 574, 569, 614, 557, 615, 558, 570, 565, 619, 562, 598, 580, 595, 610,
   586, 590, 592, 583, 607, 604, 548, 577, 578, 551, 575, 549, 573, 600, 601, 559, 560, 571, 572, 616, 617, 563, 561, 618, 596, 594,
   579, 584, 608, 582, 606, 588, 589, 605, 549, 595, 567, 579, 600, 560, 578, 614, 597, 582, 561, 594, 573, 621, 612, 588, 572, 622,
   601, 616, 584, 589, 618, 605, 617, 608, 596, 606, 594, 550, 568, 600, 580, 616, 599, 558, 576, 595, 621, 583, 613, 562, 574, 590,
   623, 570, 602, 586, 614, 619, 592, 610, 598, 604, 615, 607, 551, 569, 597, 559, 577, 579, 601, 595, 612, 563, 584, 575, 596, 614,
   622, 571, 589, 621, 617, 600, 582, 605, 616, 588, 618, 594, 606, 608, 596, 570, 552, 617, 598, 601, 581, 574, 556, 622, 597, 615,
   585, 576, 564, 623, 591, 568, 587, 602, 612, 611, 599, 620, 593, 613, 603, 609, 571, 553, 599, 575, 557, 594, 613, 580, 602, 577,
   616, 565, 623, 586, 598, 569, 621, 592, 615, 583, 600, 614, 604, 595, 607, 590, 619, 610, 572, 598, 554, 596, 615, 573, 555, 602,
   581, 617, 578, 623, 566, 599, 587, 622, 567, 593, 585, 613, 601, 597, 609, 612, 603, 620, 591, 611, 561, 607, 555, 589, 619, 572,
   566, 615, 585, 606, 573, 621, 567, 613, 603, 618, 578, 609, 591, 623, 596, 601, 620, 617, 611, 622, 597, 612, 606, 562, 556, 618,
   592, 617, 587, 570, 564, 621, 612, 607, 603, 574, 568, 619, 611, 576, 593, 598, 622, 620, 602, 623, 599, 615, 609, 613, 563, 557,
   609, 571, 565, 588, 620, 583, 613, 575, 608, 569, 622, 604, 615, 577, 618, 607, 623, 590, 594, 616, 610, 600, 619, 595, 621, 614,
   608, 558, 564, 616, 586, 618, 593, 562, 568, 614, 622, 604, 609, 570, 576, 610, 620, 574, 599, 592, 621, 623, 598, 619, 602, 607,
   613, 615, 559, 565, 611, 563, 569, 582, 612, 590, 620, 571, 605, 577, 617, 610, 623, 575, 606, 619, 622, 595, 588, 608, 614, 594,
   621, 600, 618, 616, 560, 610, 566, 584, 614, 561, 567, 619, 591, 605, 572, 616, 578, 623, 611, 608, 573, 620, 597, 621, 589, 596,
   622, 606, 612, 618, 601, 617, 579, 590, 591, 582, 595, 584, 597, 610, 611, 588, 589, 600, 601, 619, 620, 594, 596, 623, 612, 614,
   605, 606, 621, 608, 622, 616, 617, 618, 588, 580, 593, 594, 583, 608, 609, 586, 599, 600, 618, 590, 620, 592, 602, 595, 622, 598,
   613, 604, 616, 621, 607, 614, 615, 610, 623, 619, 589, 592, 581, 606, 607, 596, 585, 598, 587, 618, 601, 619, 591, 602, 593, 621,
   597, 599, 603, 615, 617, 612, 613, 622, 609, 623, 611, 620, 582, 583, 603, 588, 590, 605, 611, 604, 609, 594, 606, 595, 612, 607,
   613, 600, 617, 615, 620, 610, 608, 618, 619, 616, 623, 614, 622, 621, 584, 604, 585, 605, 610, 589, 591, 607, 603, 608, 596, 614,
   597, 615, 609, 616, 601, 613, 611, 619, 606, 617, 623, 618, 620, 621, 612, 622, 605, 586, 587, 608, 604, 606, 603, 592, 593, 616,
   617, 610, 611, 598, 599, 614, 612, 602, 609, 607, 618, 622, 615, 621, 613, 619, 620, 623
}; // 9.02 kb

// indices for factoring [CartY(la+lb)] into [CartY(lb)] x [CartY(la)]
// format: nCartY(la) x CartY(lb)
static const cart_index_t iCartYY_ab0_a0[1] = {
   0
}; // 0.00 kb
static const cart_index_t iCartYY_ab1_a0[3] = {
   0, 1, 2
}; // 0.01 kb
static const cart_index_t iCartYY_ab1_a1[3] = {
   0, 1, 2
}; // 0.01 kb
static const cart_index_t iCartYY_ab2_a0[6] = {
   0, 1, 2, 3, 4, 5
}; // 0.01 kb
static const cart_index_t iCartYY_ab2_a1[9] = {
   0, 3, 4, 3, 1, 5, 4, 5, 2
}; // 0.02 kb
static const cart_index_t iCartYY_ab2_a2[6] = {
   0, 1, 2, 3, 4, 5
}; // 0.01 kb
static const cart_index_t iCartYY_ab3_a0[10] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9
}; // 0.02 kb
static const cart_index_t iCartYY_ab3_a1[18] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 3, 9, 7, 9, 4, 9, 8, 6
}; // 0.04 kb
static const cart_index_t iCartYY_ab3_a2[18] = {
   0, 3, 4, 5, 7, 9, 5, 1, 6, 3, 9, 8, 7, 8, 2, 9, 4, 6
}; // 0.04 kb
static const cart_index_t iCartYY_ab3_a3[10] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9
}; // 0.02 kb
static const cart_index_t iCartYY_ab4_a0[15] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14
}; // 0.03 kb
static const cart_index_t iCartYY_ab4_a1[30] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 9, 4, 13, 10, 12, 6, 3, 9, 14, 12, 11, 8, 5, 14, 10, 13, 7,
   11, 14, 13, 12
}; // 0.06 kb
static const cart_index_t iCartYY_ab4_a2[36] = {
   0, 9, 10, 3, 5, 14, 9, 1, 11, 4, 13, 7, 10, 11, 2, 12, 6, 8, 3, 4, 12, 9, 14, 13, 5, 13,
   6, 14, 10, 12, 14, 7, 8, 13, 12, 11
}; // 0.07 kb
static const cart_index_t iCartYY_ab4_a3[30] = {
   0, 4, 6, 9, 10, 3, 12, 5, 13, 14, 3, 1, 8, 4, 12, 9, 11, 14, 7, 13, 5, 7, 2, 13, 6, 14,
   8, 10, 11, 12
}; // 0.06 kb
static const cart_index_t iCartYY_ab4_a4[15] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14
}; // 0.03 kb
static const cart_index_t iCartYY_ab5_a0[21] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20
}; // 0.04 kb
static const cart_index_t iCartYY_ab5_a1[45] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 9, 15, 11, 3, 16, 7, 15, 10, 13, 17, 4, 16, 8, 12, 17, 14,
   6, 9, 11, 20, 10, 19, 13, 18, 12, 14, 19, 18, 17, 20, 16, 18, 15, 20, 19
}; // 0.09 kb
static const cart_index_t iCartYY_ab5_a2[60] = {
   0, 9, 10, 5, 7, 15, 11, 1, 12, 3, 16, 8, 13, 14, 2, 17, 4, 6, 9, 3, 18, 11, 20, 16, 10, 18,
   4, 19, 13, 17, 5, 11, 19, 9, 15, 20, 19, 12, 6, 18, 17, 14, 7, 20, 13, 15, 10, 19, 20, 8, 14, 16,
   18, 12, 15, 16, 17, 20, 19, 18
}; // 0.12 kb
static const cart_index_t iCartYY_ab5_a3[60] = {
   0, 11, 13, 9, 10, 5, 19, 7, 20, 15, 9, 1, 14, 3, 18, 11, 12, 20, 8, 16, 10, 12, 2, 18, 4, 19,
   6, 13, 14, 17, 5, 3, 17, 11, 19, 9, 18, 15, 16, 20, 7, 16, 4, 20, 13, 15, 17, 10, 18, 19, 15, 8,
   6, 16, 17, 20, 14, 19, 12, 18
}; // 0.12 kb
static const cart_index_t iCartYY_ab5_a4[45] = {
   0, 3, 4, 5, 11, 7, 13, 16, 17, 9, 10, 18, 19, 20, 15, 5, 1, 6, 9, 3, 15, 17, 8, 14, 11, 19,
   12, 18, 16, 20, 7, 8, 2, 15, 16, 10, 4, 12, 6, 20, 13, 14, 17, 18, 19
}; // 0.09 kb
static const cart_index_t iCartYY_ab5_a5[21] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20
}; // 0.04 kb
static const cart_index_t iCartYY_ab6_a0[28] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27
}; // 0.05 kb
static const cart_index_t iCartYY_ab6_a1[63] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 11, 4, 19, 12, 18, 6, 3, 9, 20, 18, 14, 8, 5, 20, 10, 19, 7,
   13, 9, 15, 23, 10, 21, 16, 15, 11, 25, 22, 13, 17, 16, 26, 12, 24, 17, 14, 20, 23, 21, 25, 19, 22, 26,
   24, 18, 27, 22, 24, 21, 27, 26, 23, 25, 27
}; // 0.12 kb
static const cart_index_t iCartYY_ab6_a2[90] = {
   0, 9, 10, 3, 5, 20, 11, 1, 13, 4, 19, 7, 12, 14, 2, 18, 6, 8, 3, 15, 21, 9, 20, 23, 15, 4,
   22, 11, 25, 19, 5, 23, 16, 20, 10, 21, 16, 24, 6, 26, 12, 18, 25, 7, 17, 19, 22, 13, 26, 17, 8, 24,
   18, 14, 9, 11, 27, 15, 23, 25, 10, 27, 12, 21, 16, 26, 27, 13, 14, 22, 24, 17, 21, 22, 18, 27, 26, 24,
   23, 19, 24, 25, 27, 22, 20, 25, 26, 23, 21, 27
}; // 0.18 kb
static const cart_index_t iCartYY_ab6_a3[100] = {
   0, 15, 16, 9, 10, 3, 21, 5, 23, 20, 15, 1, 17, 4, 22, 11, 13, 25, 7, 19, 16, 17, 2, 24, 6, 26,
   8, 12, 14, 18, 9, 4, 24, 11, 27, 15, 22, 23, 19, 25, 10, 22, 6, 27, 12, 21, 18, 16, 24, 26, 3, 11,
   26, 15, 21, 9, 27, 20, 25, 23, 21, 13, 8, 22, 18, 27, 14, 26, 17, 24, 5, 25, 12, 23, 16, 20, 26, 10,
   27, 21, 23, 7, 14, 19, 24, 25, 17, 27, 13, 22, 20, 19, 18, 25, 26, 23, 24, 21, 22, 27
}; // 0.20 kb
static const cart_index_t iCartYY_ab6_a4[90] = {
   0, 11, 12, 3, 15, 5, 16, 25, 26, 9, 10, 27, 21, 23, 20, 9, 1, 14, 15, 4, 23, 24, 7, 17, 11, 27,
   13, 22, 19, 25, 10, 13, 2, 21, 22, 16, 6, 17, 8, 27, 12, 14, 18, 24, 26, 3, 4, 18, 9, 11, 20, 26,
   19, 24, 15, 21, 22, 27, 25, 23, 5, 19, 6, 20, 25, 10, 12, 22, 18, 23, 16, 24, 26, 27, 21, 20, 7, 8,
   23, 19, 21, 18, 13, 14, 25, 26, 17, 24, 22, 27
}; // 0.18 kb
static const cart_index_t iCartYY_ab6_a5[63] = {
   0, 4, 6, 11, 12, 3, 18, 5, 19, 9, 10, 15, 22, 16, 24, 20, 25, 26, 27, 21, 23, 3, 1, 8, 4, 18,
   9, 14, 20, 7, 15, 21, 11, 13, 26, 17, 23, 19, 24, 22, 27, 25, 5, 7, 2, 19, 6, 20, 8, 10, 13, 23,
   16, 25, 17, 12, 14, 21, 22, 18, 24, 26, 27
}; // 0.12 kb
static const cart_index_t iCartYY_ab6_a6[28] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27
}; // 0.05 kb
static const cart_index_t iCartYY_ab7_a0[36] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35
}; // 0.07 kb
static const cart_index_t iCartYY_ab7_a1[84] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 9, 21, 11, 3, 22, 7, 21, 10, 13, 23, 4, 22, 8, 12, 23, 14,
   6, 9, 17, 28, 10, 26, 19, 15, 11, 29, 16, 27, 13, 24, 12, 20, 25, 18, 14, 17, 15, 30, 19, 31, 16, 32,
   20, 18, 27, 25, 23, 29, 22, 24, 21, 28, 26, 26, 33, 31, 34, 24, 32, 28, 30, 33, 35, 32, 25, 30, 29, 34,
   31, 35, 27, 33, 34, 35
}; // 0.16 kb
static const cart_index_t iCartYY_ab7_a2[126] = {
   0, 9, 10, 5, 7, 21, 11, 1, 12, 3, 22, 8, 13, 14, 2, 23, 4, 6, 15, 3, 24, 11, 29, 22, 16, 25,
   4, 27, 13, 23, 5, 17, 26, 9, 21, 28, 27, 18, 6, 25, 23, 14, 7, 28, 19, 21, 10, 26, 29, 8, 20, 22,
   24, 12, 9, 15, 33, 17, 28, 30, 10, 33, 16, 26, 19, 31, 17, 11, 34, 15, 30, 29, 34, 12, 18, 24, 32, 20,
   19, 35, 13, 31, 16, 27, 35, 20, 14, 32, 25, 18, 21, 30, 31, 28, 26, 33, 30, 22, 32, 29, 34, 24, 31, 32,
   23, 35, 27, 25, 33, 24, 25, 34, 35, 32, 26, 34, 27, 33, 31, 35, 28, 29, 35, 30, 33, 34
}; // 0.25 kb
static const cart_index_t iCartYY_ab7_a3[150] = {
   0, 17, 19, 9, 10, 5, 26, 7, 28, 21, 15, 1, 20, 3, 24, 11, 12, 29, 8, 22, 16, 18, 2, 25, 4, 27,
   6, 13, 14, 23, 5, 15, 31, 17, 26, 9, 33, 21, 30, 28, 17, 3, 32, 11, 34, 15, 24, 30, 22, 29, 7, 30,
   16, 28, 19, 21, 31, 10, 33, 26, 19, 32, 4, 35, 13, 31, 23, 16, 25, 27, 30, 8, 18, 22, 32, 29, 20, 34,
   12, 24, 31, 20, 6, 32, 23, 35, 14, 27, 18, 25, 9, 11, 35, 15, 33, 17, 34, 28, 29, 30, 10, 34, 13, 33,
   16, 26, 27, 19, 35, 31, 33, 12, 14, 24, 25, 34, 18, 35, 20, 32, 26, 24, 23, 34, 27, 33, 25, 31, 32, 35,
   28, 22, 25, 29, 35, 30, 32, 33, 24, 34, 21, 29, 27, 30, 31, 28, 35, 26, 34, 33
}; // 0.29 kb
static const cart_index_t iCartYY_ab7_a4[150] = {
   0, 15, 16, 5, 17, 7, 19, 30, 31, 9, 10, 33, 26, 28, 21, 17, 1, 18, 15, 3, 30, 32, 8, 20, 11, 34,
   12, 24, 22, 29, 19, 20, 2, 31, 32, 16, 4, 18, 6, 35, 13, 14, 23, 25, 27, 9, 3, 25, 17, 11, 28, 35,
   22, 32, 15, 33, 24, 34, 29, 30, 10, 24, 4, 26, 34, 19, 13, 32, 23, 33, 16, 25, 27, 35, 31, 5, 11, 27,
   9, 15, 21, 31, 29, 35, 17, 26, 34, 33, 30, 28, 26, 12, 6, 33, 24, 31, 23, 20, 14, 34, 27, 18, 25, 32,
   35, 7, 29, 13, 21, 30, 10, 16, 34, 27, 28, 19, 35, 31, 33, 26, 28, 8, 14, 30, 22, 33, 25, 12, 18, 29,
   35, 20, 32, 24, 34, 21, 22, 23, 28, 29, 26, 27, 24, 25, 30, 31, 32, 35, 34, 33
}; // 0.29 kb
static const cart_index_t iCartYY_ab7_a5[126] = {
   0, 11, 13, 15, 16, 5, 27, 7, 29, 9, 10, 17, 34, 19, 35, 21, 30, 31, 33, 26, 28, 9, 1, 14, 3, 25,
   17, 18, 28, 8, 15, 33, 11, 12, 35, 20, 30, 22, 32, 24, 34, 29, 10, 12, 2, 24, 4, 26, 6, 19, 20, 33,
   16, 34, 18, 13, 14, 31, 32, 23, 25, 27, 35, 5, 3, 23, 11, 27, 9, 25, 21, 22, 17, 26, 15, 24, 31, 32,
   28, 29, 35, 34, 33, 30, 7, 22, 4, 29, 13, 21, 23, 10, 24, 28, 19, 30, 32, 16, 25, 26, 34, 27, 35, 31,
   33, 21, 8, 6, 22, 23, 28, 14, 26, 12, 30, 31, 29, 20, 27, 18, 33, 24, 25, 32, 35, 34
}; // 0.25 kb
static const cart_index_t iCartYY_ab7_a6[84] = {
   0, 3, 4, 5, 11, 7, 13, 22, 23, 9, 10, 15, 16, 24, 25, 17, 19, 32, 27, 29, 21, 26, 34, 28, 35, 30,
   31, 33, 5, 1, 6, 9, 3, 21, 23, 8, 14, 17, 26, 11, 27, 12, 18, 15, 31, 20, 25, 22, 28, 33, 24, 30,
   32, 29, 35, 34, 7, 8, 2, 21, 22, 10, 4, 12, 6, 28, 19, 29, 13, 20, 14, 30, 16, 18, 23, 24, 26, 31,
   32, 33, 25, 34, 27, 35
}; // 0.16 kb
static const cart_index_t iCartYY_ab7_a7[36] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35
}; // 0.07 kb
static const cart_index_t iCartYY_ab8_a0[45] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44
}; // 0.09 kb
static const cart_index_t iCartYY_ab8_a1[108] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 11, 4, 25, 12, 24, 6, 3, 9, 26, 24, 14, 8, 5, 26, 10, 25, 7,
   13, 9, 15, 29, 10, 27, 17, 16, 11, 31, 28, 13, 19, 18, 32, 12, 30, 20, 14, 21, 16, 35, 22, 33, 18, 15,
   21, 37, 34, 23, 20, 17, 38, 22, 36, 19, 23, 26, 29, 27, 31, 25, 28, 32, 30, 24, 40, 28, 36, 41, 34, 30,
   27, 39, 38, 33, 41, 32, 29, 37, 39, 35, 31, 40, 37, 35, 42, 38, 43, 33, 44, 36, 34, 39, 42, 43, 42, 40,
   44, 43, 44, 41
}; // 0.21 kb
static const cart_index_t iCartYY_ab8_a2[168] = {
   0, 9, 10, 3, 5, 26, 11, 1, 13, 4, 25, 7, 12, 14, 2, 24, 6, 8, 3, 15, 27, 9, 26, 29, 16, 4,
   28, 11, 31, 25, 5, 29, 17, 26, 10, 27, 18, 30, 6, 32, 12, 24, 31, 7, 19, 25, 28, 13, 32, 20, 8, 30,
   24, 14, 9, 21, 39, 15, 29, 37, 10, 39, 22, 27, 17, 38, 21, 11, 40, 16, 35, 31, 22, 41, 12, 33, 18, 32,
   40, 13, 23, 28, 36, 19, 41, 23, 14, 34, 30, 20, 15, 16, 42, 21, 37, 35, 17, 43, 18, 38, 22, 33, 44, 19,
   20, 36, 34, 23, 33, 34, 24, 41, 32, 30, 35, 25, 36, 31, 40, 28, 26, 37, 38, 29, 27, 39, 27, 42, 33, 39,
   38, 43, 42, 28, 34, 40, 44, 36, 29, 35, 43, 37, 39, 42, 43, 36, 30, 44, 41, 34, 37, 31, 44, 35, 42, 40,
   38, 44, 32, 43, 33, 41, 39, 40, 41, 42, 43, 44
}; // 0.33 kb
static const cart_index_t iCartYY_ab8_a3[210] = {
   0, 15, 17, 9, 10, 3, 27, 5, 29, 26, 16, 1, 19, 4, 28, 11, 13, 31, 7, 25, 18, 20, 2, 30, 6, 32,
   8, 12, 14, 24, 21, 4, 36, 11, 40, 16, 28, 35, 25, 31, 22, 34, 6, 41, 12, 33, 24, 18, 30, 32, 3, 21,
   38, 15, 27, 9, 39, 26, 37, 29, 33, 23, 8, 34, 24, 41, 14, 32, 20, 30, 5, 37, 22, 29, 17, 26, 38, 10,
   39, 27, 35, 7, 23, 25, 36, 31, 19, 40, 13, 28, 9, 16, 43, 21, 39, 15, 42, 29, 35, 37, 10, 42, 18, 39,
   22, 27, 33, 17, 43, 38, 15, 11, 44, 16, 42, 21, 40, 37, 31, 35, 42, 13, 20, 28, 34, 40, 23, 44, 19, 36,
   17, 44, 12, 43, 18, 38, 32, 22, 41, 33, 43, 19, 14, 36, 30, 44, 20, 41, 23, 34, 26, 35, 33, 37, 38, 29,
   43, 27, 42, 39, 37, 25, 34, 31, 44, 35, 36, 42, 28, 40, 38, 36, 24, 44, 32, 43, 30, 33, 34, 41, 39, 28,
   30, 40, 41, 42, 34, 43, 36, 44, 27, 40, 32, 42, 33, 39, 41, 38, 44, 43, 29, 31, 41, 35, 43, 37, 44, 39,
   40, 42
}; // 0.41 kb
static const cart_index_t iCartYY_ab8_a4[225] = {
   0, 21, 22, 3, 15, 5, 17, 37, 38, 9, 10, 39, 27, 29, 26, 21, 1, 23, 16, 4, 35, 36, 7, 19, 11, 40,
   13, 28, 25, 31, 22, 23, 2, 33, 34, 18, 6, 20, 8, 41, 12, 14, 24, 30, 32, 3, 16, 33, 9, 21, 26, 38,
   35, 43, 15, 27, 42, 39, 37, 29, 15, 4, 34, 21, 11, 37, 44, 25, 36, 16, 42, 28, 40, 31, 35, 5, 35, 18,
   26, 37, 10, 22, 42, 33, 29, 17, 43, 38, 39, 27, 17, 36, 6, 38, 44, 22, 12, 34, 24, 43, 18, 30, 32, 41,
   33, 37, 7, 20, 35, 25, 42, 34, 13, 23, 31, 44, 19, 36, 28, 40, 38, 19, 8, 43, 36, 33, 24, 23, 14, 44,
   32, 20, 30, 34, 41, 9, 11, 41, 15, 16, 29, 43, 31, 44, 21, 39, 40, 42, 35, 37, 10, 40, 12, 27, 42, 17,
   18, 44, 32, 39, 22, 41, 33, 43, 38, 39, 13, 14, 42, 28, 43, 30, 19, 20, 40, 41, 23, 34, 36, 44, 27, 28,
   24, 39, 40, 38, 32, 36, 30, 42, 33, 34, 41, 44, 43, 29, 25, 30, 37, 31, 39, 41, 28, 34, 35, 43, 36, 44,
   40, 42, 26, 31, 32, 29, 35, 27, 33, 40, 41, 37, 38, 44, 43, 42, 39
}; // 0.44 kb
static const cart_index_t iCartYY_ab8_a5[210] = {
   0, 16, 18, 21, 22, 3, 33, 5, 35, 9, 10, 15, 42, 17, 43, 26, 37, 38, 39, 27, 29, 15, 1, 20, 4, 34,
   21, 23, 37, 7, 16, 42, 11, 13, 44, 19, 35, 25, 36, 28, 40, 31, 17, 19, 2, 36, 6, 38, 8, 22, 23, 43,
   18, 44, 20, 12, 14, 33, 34, 24, 30, 32, 41, 9, 4, 30, 11, 41, 15, 34, 29, 25, 21, 39, 16, 28, 43, 36,
   37, 31, 44, 40, 42, 35, 10, 28, 6, 40, 12, 27, 24, 17, 36, 39, 22, 42, 34, 18, 30, 38, 44, 32, 41, 33,
   43, 3, 11, 32, 16, 33, 9, 41, 26, 31, 15, 27, 21, 40, 38, 44, 29, 35, 43, 42, 39, 37, 27, 13, 8, 28,
   24, 39, 14, 38, 19, 42, 33, 40, 23, 32, 20, 43, 36, 30, 34, 41, 44, 5, 31, 12, 35, 18, 26, 32, 10, 40,
   29, 17, 37, 44, 22, 41, 27, 42, 33, 43, 38, 39, 29, 7, 14, 25, 30, 37, 20, 39, 13, 35, 43, 31, 19, 41,
   23, 42, 28, 34, 36, 44, 40, 26, 25, 24, 31, 32, 29, 30, 27, 28, 37, 38, 35, 36, 33, 34, 39, 40, 41, 44,
   43, 42
}; // 0.41 kb
static const cart_index_t iCartYY_ab8_a6[168] = {
   0, 11, 12, 3, 16, 5, 18, 31, 32, 9, 10, 21, 22, 40, 41, 15, 17, 44, 33, 35, 26, 27, 42, 29, 43, 37,
   38, 39, 9, 1, 14, 15, 4, 29, 30, 7, 20, 21, 39, 11, 41, 13, 23, 16, 43, 19, 34, 25, 37, 42, 28, 35,
   36, 31, 44, 40, 10, 13, 2, 27, 28, 17, 6, 19, 8, 39, 22, 40, 12, 23, 14, 42, 18, 20, 24, 36, 38, 33,
   34, 43, 30, 44, 32, 41, 3, 4, 24, 9, 11, 26, 32, 25, 30, 15, 27, 16, 33, 28, 34, 21, 38, 36, 41, 31,
   29, 39, 40, 37, 44, 35, 43, 42, 5, 25, 6, 26, 31, 10, 12, 28, 24, 29, 17, 35, 18, 36, 30, 37, 22, 34,
   32, 40, 27, 38, 44, 39, 41, 42, 33, 43, 26, 7, 8, 29, 25, 27, 24, 13, 14, 37, 38, 31, 32, 19, 20, 35,
   33, 23, 30, 28, 39, 43, 36, 42, 34, 40, 41, 44
}; // 0.33 kb
static const cart_index_t iCartYY_ab8_a7[108] = {
   0, 4, 6, 11, 12, 3, 24, 5, 25, 9, 10, 16, 28, 18, 30, 21, 22, 15, 34, 17, 36, 26, 31, 32, 40, 41,
   27, 33, 29, 35, 37, 38, 44, 39, 42, 43, 3, 1, 8, 4, 24, 9, 14, 26, 7, 15, 27, 11, 13, 32, 20, 16,
   33, 21, 23, 38, 19, 29, 25, 30, 28, 34, 39, 41, 37, 31, 35, 43, 36, 42, 40, 44, 5, 7, 2, 25, 6, 26,
   8, 10, 13, 29, 17, 31, 19, 12, 14, 35, 18, 37, 20, 22, 23, 27, 28, 24, 36, 30, 38, 32, 39, 40, 42, 33,
   34, 43, 44, 41
}; // 0.21 kb
static const cart_index_t iCartYY_ab8_a8[45] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44
}; // 0.09 kb
static const cart_index_t iCartYY_ab9_a0[55] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54
}; // 0.11 kb
static const cart_index_t iCartYY_ab9_a1[135] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 9, 27, 11, 3, 28, 7, 27, 10, 13, 29, 4, 28, 8, 12, 29, 14,
   6, 9, 17, 34, 10, 32, 19, 15, 11, 35, 16, 33, 13, 30, 12, 20, 31, 18, 14, 17, 21, 36, 23, 15, 38, 19,
   37, 22, 25, 39, 16, 40, 20, 24, 41, 26, 18, 21, 23, 44, 22, 43, 25, 42, 24, 26, 33, 31, 29, 35, 28, 30,
   27, 34, 32, 32, 45, 37, 46, 30, 40, 34, 36, 45, 47, 41, 31, 38, 35, 46, 39, 47, 33, 43, 49, 39, 51, 42,
   41, 44, 38, 48, 53, 40, 42, 36, 44, 50, 37, 52, 43, 45, 50, 52, 48, 46, 53, 49, 51, 47, 50, 48, 54, 52,
   54, 49, 54, 53, 51
}; // 0.26 kb
static const cart_index_t iCartYY_ab9_a2[216] = {
   0, 9, 10, 5, 7, 27, 11, 1, 12, 3, 28, 8, 13, 14, 2, 29, 4, 6, 15, 3, 30, 11, 35, 28, 16, 31,
   4, 33, 13, 29, 5, 17, 32, 9, 27, 34, 33, 18, 6, 31, 29, 14, 7, 34, 19, 27, 10, 32, 35, 8, 20, 28,
   30, 12, 9, 21, 45, 17, 34, 36, 10, 45, 22, 32, 19, 37, 23, 11, 46, 15, 38, 35, 46, 12, 24, 30, 40, 20,
   25, 47, 13, 39, 16, 33, 47, 26, 14, 41, 31, 18, 21, 15, 48, 23, 44, 38, 22, 49, 16, 43, 25, 39, 17, 23,
   50, 21, 36, 44, 51, 24, 18, 42, 41, 26, 19, 52, 25, 37, 22, 43, 53, 20, 26, 40, 42, 24, 27, 36, 37, 34,
   32, 45, 38, 28, 40, 35, 46, 30, 39, 41, 29, 47, 33, 31, 48, 30, 42, 46, 53, 40, 49, 42, 31, 51, 47, 41,
   32, 50, 43, 45, 37, 52, 43, 51, 33, 49, 39, 47, 34, 44, 52, 36, 45, 50, 44, 35, 53, 38, 48, 46, 36, 38,
   54, 44, 50, 48, 37, 54, 39, 52, 43, 49, 54, 40, 41, 53, 51, 42, 45, 48, 49, 50, 52, 54, 50, 46, 51, 48,
   54, 53, 52, 53, 47, 54, 49, 51
}; // 0.42 kb
static const cart_index_t iCartYY_ab9_a3[280] = {
   0, 17, 19, 9, 10, 5, 32, 7, 34, 27, 15, 1, 20, 3, 30, 11, 12, 35, 8, 28, 16, 18, 2, 31, 4, 33,
   6, 13, 14, 29, 5, 21, 37, 17, 32, 9, 45, 27, 36, 34, 23, 3, 40, 11, 46, 15, 30, 38, 28, 35, 7, 36,
   22, 34, 19, 27, 37, 10, 45, 32, 25, 41, 4, 47, 13, 39, 29, 16, 31, 33, 38, 8, 24, 28, 40, 35, 20, 46,
   12, 30, 39, 26, 6, 41, 29, 47, 14, 33, 18, 31, 9, 23, 52, 21, 45, 17, 50, 34, 44, 36, 10, 50, 25, 45,
   22, 32, 43, 19, 52, 37, 21, 11, 53, 15, 48, 23, 46, 44, 35, 38, 22, 51, 13, 49, 16, 43, 33, 25, 47, 39,
   48, 12, 26, 30, 42, 46, 24, 53, 20, 40, 49, 24, 14, 42, 31, 51, 18, 47, 26, 41, 17, 15, 54, 23, 50, 21,
   48, 36, 38, 44, 19, 54, 16, 52, 25, 37, 39, 22, 49, 43, 54, 20, 18, 40, 41, 53, 26, 51, 24, 42, 43, 42,
   29, 51, 33, 49, 31, 39, 41, 47, 44, 28, 42, 35, 53, 38, 40, 48, 30, 46, 27, 44, 43, 36, 37, 34, 52, 32,
   50, 45, 32, 48, 39, 50, 43, 45, 49, 37, 54, 52, 50, 30, 41, 46, 51, 48, 42, 54, 40, 53, 34, 38, 49, 44,
   52, 36, 54, 45, 48, 50, 52, 40, 31, 53, 47, 54, 41, 49, 42, 51, 36, 35, 51, 38, 54, 44, 53, 50, 46, 48,
   37, 53, 33, 54, 39, 52, 47, 43, 51, 49, 45, 46, 47, 48, 49, 50, 51, 52, 53, 54
}; // 0.55 kb
static const cart_index_t iCartYY_ab9_a4[315] = {
   0, 21, 22, 5, 17, 7, 19, 36, 37, 9, 10, 45, 32, 34, 27, 23, 1, 24, 15, 3, 38, 40, 8, 20, 11, 46,
   12, 30, 28, 35, 25, 26, 2, 39, 41, 16, 4, 18, 6, 47, 13, 14, 29, 31, 33, 21, 3, 42, 23, 11, 44, 53,
   28, 40, 15, 48, 30, 46, 35, 38, 22, 42, 4, 43, 51, 25, 13, 41, 29, 49, 16, 31, 33, 47, 39, 5, 23, 43,
   9, 21, 27, 37, 44, 52, 17, 32, 50, 45, 36, 34, 43, 24, 6, 49, 42, 39, 29, 26, 14, 51, 33, 18, 31, 41,
   47, 7, 44, 25, 27, 36, 10, 22, 50, 43, 34, 19, 52, 37, 45, 32, 44, 8, 26, 38, 28, 48, 42, 12, 24, 35,
   53, 20, 40, 30, 46, 9, 15, 49, 17, 23, 34, 52, 38, 54, 21, 45, 48, 50, 44, 36, 10, 48, 16, 32, 50, 19,
   25, 54, 39, 45, 22, 49, 43, 52, 37, 17, 11, 51, 21, 15, 36, 54, 35, 53, 23, 50, 46, 48, 38, 44, 50, 12,
   18, 48, 30, 54, 41, 20, 26, 46, 51, 24, 42, 40, 53, 19, 53, 13, 37, 54, 22, 16, 51, 33, 52, 25, 47, 39,
   49, 43, 52, 20, 14, 54, 40, 49, 31, 24, 18, 53, 47, 26, 41, 42, 51, 27, 38, 39, 34, 44, 32, 43, 48, 49,
   36, 37, 54, 52, 50, 45, 36, 28, 41, 44, 35, 50, 51, 30, 42, 38, 54, 40, 53, 46, 48, 37, 40, 29, 52, 53,
   43, 33, 42, 31, 54, 39, 41, 47, 51, 49, 45, 30, 31, 50, 46, 52, 47, 40, 41, 48, 49, 42, 51, 53, 54, 32,
   46, 33, 45, 48, 37, 39, 53, 47, 50, 43, 51, 49, 54, 52, 34, 35, 47, 36, 38, 45, 49, 46, 51, 44, 52, 53,
   54, 48, 50
}; // 0.62 kb
static const cart_index_t iCartYY_ab9_a5[315] = {
   0, 23, 25, 21, 22, 5, 43, 7, 44, 9, 10, 17, 50, 19, 52, 27, 36, 37, 45, 32, 34, 21, 1, 26, 3, 42,
   23, 24, 44, 8, 15, 48, 11, 12, 53, 20, 38, 28, 40, 30, 46, 35, 22, 24, 2, 42, 4, 43, 6, 25, 26, 49,
   16, 51, 18, 13, 14, 39, 41, 29, 31, 33, 47, 5, 15, 39, 23, 43, 9, 49, 27, 38, 17, 32, 21, 48, 37, 54,
   34, 44, 52, 50, 45, 36, 17, 3, 41, 11, 51, 21, 42, 36, 28, 23, 50, 15, 30, 54, 40, 44, 35, 53, 46, 48,
   38, 7, 38, 16, 44, 25, 27, 39, 10, 48, 34, 19, 36, 54, 22, 49, 32, 50, 43, 52, 37, 45, 19, 40, 4, 53,
   13, 37, 29, 22, 42, 52, 25, 54, 41, 16, 31, 43, 51, 33, 47, 39, 49, 36, 8, 18, 28, 41, 44, 26, 50, 12,
   38, 54, 35, 20, 51, 24, 48, 30, 42, 40, 53, 46, 37, 20, 6, 40, 29, 52, 14, 43, 24, 54, 39, 53, 26, 33,
   18, 49, 42, 31, 41, 47, 51, 9, 11, 47, 15, 49, 17, 51, 34, 35, 21, 45, 23, 46, 52, 53, 36, 38, 54, 48,
   50, 44, 10, 46, 13, 48, 16, 32, 33, 19, 53, 45, 22, 50, 51, 25, 47, 37, 54, 39, 49, 43, 52, 45, 12, 14,
   30, 31, 50, 18, 52, 20, 48, 49, 46, 24, 47, 26, 54, 40, 41, 42, 51, 53, 32, 30, 29, 46, 33, 45, 31, 37,
   40, 50, 43, 48, 42, 39, 41, 52, 53, 47, 51, 49, 54, 34, 28, 31, 35, 47, 36, 41, 45, 30, 44, 52, 38, 40,
   49, 42, 50, 46, 51, 53, 54, 48, 27, 35, 33, 38, 39, 34, 47, 32, 46, 36, 37, 44, 53, 43, 51, 45, 48, 49,
   54, 52, 50
}; // 0.62 kb
static const cart_index_t iCartYY_ab9_a6[280] = {
   0, 15, 16, 5, 23, 7, 25, 38, 39, 9, 10, 21, 22, 48, 49, 17, 19, 54, 43, 44, 27, 32, 50, 34, 52, 36,
   37, 45, 17, 1, 18, 21, 3, 36, 41, 8, 26, 23, 50, 11, 51, 12, 24, 15, 54, 20, 42, 28, 44, 48, 30, 38,
   40, 35, 53, 46, 19, 20, 2, 37, 40, 22, 4, 24, 6, 52, 25, 53, 13, 26, 14, 54, 16, 18, 29, 42, 43, 39,
   41, 49, 31, 51, 33, 47, 9, 3, 31, 17, 11, 34, 47, 28, 41, 21, 45, 15, 49, 30, 42, 23, 52, 40, 51, 35,
   36, 50, 46, 44, 53, 38, 54, 48, 10, 30, 4, 32, 46, 19, 13, 40, 29, 45, 22, 48, 16, 42, 31, 50, 25, 41,
   33, 53, 37, 43, 51, 52, 47, 54, 39, 49, 5, 11, 33, 9, 15, 27, 39, 35, 47, 17, 32, 23, 43, 46, 51, 21,
   37, 53, 49, 38, 34, 45, 48, 36, 54, 44, 52, 50, 32, 12, 6, 45, 30, 37, 29, 20, 14, 50, 43, 46, 33, 24,
   18, 48, 39, 26, 31, 40, 52, 49, 42, 54, 41, 53, 47, 51, 7, 35, 13, 27, 38, 10, 16, 46, 33, 34, 19, 44,
   25, 53, 47, 36, 22, 51, 39, 48, 32, 37, 54, 45, 49, 50, 43, 52, 34, 8, 14, 36, 28, 45, 31, 12, 18, 44,
   52, 35, 47, 20, 26, 38, 49, 24, 41, 30, 50, 54, 40, 48, 42, 46, 51, 53, 27, 28, 29, 34, 35, 32, 33, 30,
   31, 36, 37, 38, 39, 40, 41, 44, 43, 42, 47, 46, 45, 52, 53, 50, 51, 48, 49, 54
}; // 0.55 kb
static const cart_index_t iCartYY_ab9_a7[216] = {
   0, 11, 13, 15, 16, 5, 33, 7, 35, 9, 10, 23, 46, 25, 47, 21, 22, 17, 51, 19, 53, 27, 38, 39, 48, 49,
   32, 43, 34, 44, 36, 37, 54, 45, 50, 52, 9, 1, 14, 3, 31, 17, 18, 34, 8, 21, 45, 11, 12, 47, 26, 15,
   49, 23, 24, 52, 20, 36, 28, 41, 30, 42, 50, 51, 44, 35, 38, 54, 40, 48, 46, 53, 10, 12, 2, 30, 4, 32,
   6, 19, 20, 45, 22, 46, 24, 13, 14, 48, 16, 50, 18, 25, 26, 37, 40, 29, 42, 31, 43, 33, 52, 53, 54, 39,
   41, 49, 51, 47, 5, 3, 29, 11, 33, 9, 31, 27, 28, 17, 32, 15, 30, 39, 41, 23, 43, 21, 42, 37, 40, 34,
   35, 47, 46, 51, 45, 49, 36, 38, 44, 52, 53, 50, 48, 54, 7, 28, 4, 35, 13, 27, 29, 10, 30, 34, 19, 38,
   40, 16, 31, 44, 25, 36, 41, 22, 42, 32, 46, 33, 53, 47, 37, 39, 45, 48, 50, 43, 51, 52, 54, 49, 27, 8,
   6, 28, 29, 34, 14, 32, 12, 36, 37, 35, 20, 33, 18, 38, 39, 44, 26, 43, 24, 45, 30, 31, 40, 41, 52, 47,
   50, 46, 48, 49, 42, 54, 53, 51
}; // 0.42 kb
static const cart_index_t iCartYY_ab9_a8[135] = {
   0, 3, 4, 5, 11, 7, 13, 28, 29, 9, 10, 15, 16, 30, 31, 17, 23, 19, 25, 40, 41, 21, 22, 42, 33, 35,
   27, 32, 46, 34, 47, 38, 39, 43, 51, 44, 53, 36, 37, 45, 48, 49, 50, 52, 54, 5, 1, 6, 9, 3, 27, 29,
   8, 14, 17, 32, 11, 33, 12, 18, 21, 15, 37, 39, 20, 26, 23, 43, 24, 31, 28, 34, 45, 30, 36, 41, 35, 47,
   49, 42, 38, 40, 44, 52, 50, 46, 51, 48, 54, 53, 7, 8, 2, 27, 28, 10, 4, 12, 6, 34, 19, 35, 13, 20,
   14, 36, 38, 22, 16, 24, 18, 44, 25, 26, 29, 30, 32, 37, 40, 45, 31, 46, 33, 39, 41, 48, 42, 50, 43, 52,
   53, 47, 54, 49, 51
}; // 0.26 kb
static const cart_index_t iCartYY_ab9_a9[55] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54
}; // 0.11 kb
static const cart_index_t iCartYY_ab10_a0[66] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65
}; // 0.13 kb
static const cart_index_t iCartYY_ab10_a1[165] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 11, 4, 31, 12, 30, 6, 3, 9, 32, 30, 14, 8, 5, 32, 10, 31, 7,
   13, 9, 15, 35, 10, 33, 17, 16, 11, 37, 34, 13, 19, 18, 38, 12, 36, 20, 14, 23, 16, 41, 24, 39, 18, 15,
   21, 43, 40, 26, 20, 17, 44, 22, 42, 19, 25, 21, 27, 47, 22, 45, 28, 27, 23, 49, 46, 25, 29, 28, 50, 24,
   48, 29, 26, 32, 35, 33, 37, 31, 34, 38, 36, 30, 52, 34, 42, 53, 40, 36, 33, 51, 44, 39, 53, 38, 35, 43,
   51, 41, 37, 52, 43, 47, 54, 44, 56, 45, 49, 41, 55, 50, 57, 39, 58, 42, 46, 59, 48, 40, 62, 46, 48, 45,
   61, 50, 47, 49, 60, 51, 54, 56, 55, 52, 58, 57, 59, 53, 60, 55, 64, 61, 63, 57, 54, 60, 65, 63, 62, 59,
   56, 65, 61, 64, 58, 62, 65, 64, 63
}; // 0.32 kb
static const cart_index_t iCartYY_ab10_a2[270] = {
   0, 9, 10, 3, 5, 32, 11, 1, 13, 4, 31, 7, 12, 14, 2, 30, 6, 8, 3, 15, 33, 9, 32, 35, 16, 4,
   34, 11, 37, 31, 5, 35, 17, 32, 10, 33, 18, 36, 6, 38, 12, 30, 37, 7, 19, 31, 34, 13, 38, 20, 8, 36,
   30, 14, 9, 21, 51, 15, 35, 43, 10, 51, 22, 33, 17, 44, 23, 11, 52, 16, 41, 37, 24, 53, 12, 39, 18, 38,
   52, 13, 25, 34, 42, 19, 53, 26, 14, 40, 36, 20, 15, 27, 54, 21, 43, 47, 27, 16, 55, 23, 49, 41, 17, 56,
   28, 44, 22, 45, 28, 57, 18, 50, 24, 39, 58, 19, 29, 42, 46, 25, 59, 29, 20, 48, 40, 26, 21, 23, 60, 27,
   47, 49, 22, 61, 24, 45, 28, 50, 62, 25, 26, 46, 48, 29, 39, 40, 30, 53, 38, 36, 41, 31, 42, 37, 52, 34,
   32, 43, 44, 35, 33, 51, 33, 54, 45, 51, 44, 56, 55, 34, 46, 52, 58, 42, 35, 47, 56, 43, 51, 54, 57, 48,
   36, 59, 53, 40, 49, 37, 58, 41, 55, 52, 50, 59, 38, 57, 39, 53, 45, 63, 39, 61, 50, 57, 63, 46, 40, 62,
   59, 48, 47, 41, 64, 49, 60, 55, 64, 42, 48, 58, 62, 46, 43, 49, 65, 47, 54, 60, 44, 65, 50, 56, 45, 61,
   51, 60, 61, 54, 56, 65, 60, 52, 62, 55, 64, 58, 61, 62, 53, 63, 57, 59, 54, 55, 63, 60, 65, 64, 56, 64,
   57, 65, 61, 63, 65, 58, 59, 64, 63, 62
}; // 0.53 kb
static const cart_index_t iCartYY_ab10_a3[360] = {
   0, 15, 17, 9, 10, 3, 33, 5, 35, 32, 16, 1, 19, 4, 34, 11, 13, 37, 7, 31, 18, 20, 2, 36, 6, 38,
   8, 12, 14, 30, 23, 4, 42, 11, 52, 16, 34, 41, 31, 37, 24, 40, 6, 53, 12, 39, 30, 18, 36, 38, 3, 21,
   44, 15, 33, 9, 51, 32, 43, 35, 39, 26, 8, 40, 30, 53, 14, 38, 20, 36, 5, 43, 22, 35, 17, 32, 44, 10,
   51, 33, 41, 7, 25, 31, 42, 37, 19, 52, 13, 34, 9, 27, 56, 21, 51, 15, 54, 35, 47, 43, 10, 54, 28, 51,
   22, 33, 45, 17, 56, 44, 27, 11, 58, 16, 55, 23, 52, 49, 37, 41, 55, 13, 29, 34, 46, 52, 25, 58, 19, 42,
   28, 59, 12, 57, 18, 50, 38, 24, 53, 39, 57, 29, 14, 48, 36, 59, 20, 53, 26, 40, 21, 16, 64, 23, 60, 27,
   55, 47, 41, 49, 22, 63, 18, 61, 24, 45, 39, 28, 57, 50, 15, 23, 65, 27, 54, 21, 60, 43, 49, 47, 63, 25,
   20, 46, 40, 62, 26, 59, 29, 48, 17, 65, 24, 56, 28, 44, 50, 22, 61, 45, 64, 19, 26, 42, 48, 58, 29, 62,
   25, 46, 32, 47, 45, 43, 44, 35, 56, 33, 54, 51, 49, 31, 46, 37, 58, 41, 42, 55, 34, 52, 50, 48, 30, 59,
   38, 57, 36, 39, 40, 53, 60, 34, 48, 52, 62, 55, 46, 64, 42, 58, 61, 46, 36, 62, 53, 63, 40, 57, 48, 59,
   33, 60, 50, 54, 45, 51, 61, 44, 65, 56, 45, 62, 38, 63, 39, 61, 53, 50, 59, 57, 35, 49, 61, 47, 56, 43,
   65, 51, 60, 54, 47, 37, 62, 41, 64, 49, 58, 60, 52, 55, 43, 41, 63, 49, 65, 47, 64, 54, 55, 60, 44, 64,
   39, 65, 50, 56, 57, 45, 63, 61, 65, 42, 40, 58, 59, 64, 48, 63, 46, 62, 51, 55, 57, 60, 61, 54, 63, 56,
   64, 65, 54, 52, 59, 55, 63, 60, 62, 65, 58, 64, 56, 58, 53, 64, 57, 65, 59, 61, 62, 63
}; // 0.70 kb
static const cart_index_t iCartYY_ab10_a4[420] = {
   0, 21, 22, 3, 15, 5, 17, 43, 44, 9, 10, 51, 33, 35, 32, 23, 1, 25, 16, 4, 41, 42, 7, 19, 11, 52,
   13, 34, 31, 37, 24, 26, 2, 39, 40, 18, 6, 20, 8, 53, 12, 14, 30, 36, 38, 3, 27, 45, 9, 21, 32, 44,
   47, 56, 15, 33, 54, 51, 43, 35, 27, 4, 46, 23, 11, 49, 58, 31, 42, 16, 55, 34, 52, 37, 41, 5, 47, 28,
   32, 43, 10, 22, 54, 45, 35, 17, 56, 44, 51, 33, 28, 48, 6, 50, 59, 24, 12, 40, 30, 57, 18, 36, 38, 53,
   39, 49, 7, 29, 41, 31, 55, 46, 13, 25, 37, 58, 19, 42, 34, 52, 50, 29, 8, 57, 48, 39, 30, 26, 14, 59,
   38, 20, 36, 40, 53, 9, 23, 61, 15, 27, 35, 56, 49, 65, 21, 51, 60, 54, 47, 43, 10, 60, 24, 33, 54, 17,
   28, 65, 50, 51, 22, 61, 45, 56, 44, 21, 11, 62, 27, 16, 47, 64, 37, 58, 23, 60, 52, 55, 41, 49, 22, 62,
   12, 45, 63, 28, 18, 59, 38, 61, 24, 53, 39, 57, 50, 60, 13, 26, 55, 34, 64, 48, 19, 29, 52, 62, 25, 46,
   42, 58, 61, 25, 14, 63, 46, 57, 36, 29, 20, 62, 53, 26, 40, 48, 59, 15, 16, 63, 21, 23, 43, 65, 41, 64,
   27, 54, 55, 60, 49, 47, 17, 64, 18, 44, 65, 22, 24, 63, 39, 56, 28, 57, 50, 61, 45, 65, 19, 20, 64, 42,
   63, 40, 25, 26, 58, 59, 29, 48, 46, 62, 45, 46, 30, 61, 62, 50, 38, 48, 36, 63, 39, 40, 53, 59, 57, 47,
   31, 48, 49, 37, 60, 62, 34, 46, 41, 64, 42, 58, 52, 55, 32, 49, 50, 35, 47, 33, 45, 60, 61, 43, 44, 65,
   56, 54, 51, 33, 55, 39, 51, 60, 44, 50, 64, 57, 54, 45, 63, 61, 65, 56, 54, 34, 40, 60, 52, 65, 59, 42,
   48, 55, 63, 46, 62, 58, 64, 35, 41, 57, 43, 49, 51, 61, 55, 63, 47, 56, 64, 65, 60, 54, 56, 42, 36, 65,
   58, 61, 53, 46, 40, 64, 57, 48, 59, 62, 63, 43, 37, 59, 47, 41, 54, 63, 52, 62, 49, 65, 58, 64, 55, 60,
   44, 58, 38, 56, 64, 45, 39, 62, 53, 65, 50, 59, 57, 63, 61, 51, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61,
   62, 63, 64, 65
}; // 0.82 kb
static const cart_index_t iCartYY_ab10_a5[441] = {
   0, 27, 28, 21, 22, 3, 45, 5, 47, 9, 10, 15, 54, 17, 56, 32, 43, 44, 51, 33, 35, 27, 1, 29, 4, 46,
   23, 25, 49, 7, 16, 55, 11, 13, 58, 19, 41, 31, 42, 34, 52, 37, 28, 29, 2, 48, 6, 50, 8, 24, 26, 57,
   18, 59, 20, 12, 14, 39, 40, 30, 36, 38, 53, 21, 4, 48, 11, 62, 27, 46, 47, 31, 23, 60, 16, 34, 64, 42,
   49, 37, 58, 52, 55, 41, 22, 46, 6, 62, 12, 45, 30, 28, 48, 61, 24, 63, 40, 18, 36, 50, 59, 38, 53, 39,
   57, 3, 23, 50, 27, 45, 9, 61, 32, 49, 15, 33, 21, 60, 44, 65, 35, 47, 56, 54, 51, 43, 45, 25, 8, 46,
   30, 61, 14, 50, 29, 63, 39, 62, 26, 38, 20, 57, 48, 36, 40, 53, 59, 5, 49, 24, 47, 28, 32, 50, 10, 60,
   35, 17, 43, 65, 22, 61, 33, 54, 45, 56, 44, 51, 47, 7, 26, 31, 48, 49, 29, 60, 13, 41, 64, 37, 19, 62,
   25, 55, 34, 46, 42, 58, 52, 9, 16, 57, 23, 61, 15, 63, 35, 41, 21, 51, 27, 55, 56, 64, 43, 49, 65, 60,
   54, 47, 10, 55, 18, 60, 24, 33, 39, 17, 64, 51, 22, 54, 63, 28, 57, 44, 65, 50, 61, 45, 56, 15, 11, 59,
   16, 63, 21, 62, 43, 37, 27, 54, 23, 52, 65, 58, 47, 41, 64, 55, 60, 49, 54, 13, 20, 34, 40, 60, 26, 65,
   19, 55, 63, 52, 25, 59, 29, 64, 42, 48, 46, 62, 58, 17, 58, 12, 64, 18, 44, 38, 22, 62, 56, 28, 65, 59,
   24, 53, 45, 63, 39, 57, 50, 61, 56, 19, 14, 42, 36, 65, 20, 61, 25, 64, 57, 58, 29, 53, 26, 63, 46, 40,
   48, 59, 62, 32, 41, 39, 49, 50, 35, 57, 33, 55, 43, 44, 47, 64, 45, 63, 51, 60, 61, 65, 56, 54, 43, 31,
   40, 37, 59, 47, 48, 54, 34, 49, 65, 41, 42, 63, 46, 60, 52, 62, 58, 64, 55, 44, 42, 30, 58, 38, 56, 36,
   45, 46, 65, 50, 64, 48, 39, 40, 61, 62, 53, 59, 57, 63, 51, 34, 36, 52, 53, 54, 40, 56, 42, 60, 61, 55,
   46, 57, 48, 65, 58, 59, 62, 63, 64, 33, 52, 38, 55, 39, 51, 53, 44, 58, 54, 45, 60, 62, 50, 59, 56, 64,
   57, 63, 61, 65, 35, 37, 53, 41, 57, 43, 59, 51, 52, 47, 56, 49, 58, 61, 62, 54, 55, 63, 64, 65, 60
}; // 0.86 kb
static const cart_index_t iCartYY_ab10_a6[420] = {
   0, 23, 24, 3, 27, 5, 28, 49, 50, 9, 10, 21, 22, 60, 61, 15, 17, 65, 45, 47, 32, 33, 54, 35, 56, 43,
   44, 51, 21, 1, 26, 27, 4, 47, 48, 7, 29, 23, 60, 11, 62, 13, 25, 16, 64, 19, 46, 31, 49, 55, 34, 41,
   42, 37, 58, 52, 22, 25, 2, 45, 46, 28, 6, 29, 8, 61, 24, 62, 12, 26, 14, 63, 18, 20, 30, 48, 50, 39,
   40, 57, 36, 59, 38, 53, 3, 16, 39, 9, 23, 32, 50, 41, 57, 15, 33, 27, 45, 55, 63, 21, 44, 64, 61, 49,
   35, 51, 60, 43, 65, 47, 56, 54, 15, 4, 40, 21, 11, 43, 59, 31, 48, 27, 54, 16, 63, 34, 46, 23, 65, 42,
   62, 37, 47, 60, 52, 49, 58, 41, 64, 55, 5, 41, 18, 32, 49, 10, 24, 55, 39, 35, 17, 47, 28, 64, 57, 43,
   22, 63, 50, 60, 33, 44, 65, 51, 61, 54, 45, 56, 17, 42, 6, 44, 58, 22, 12, 46, 30, 56, 28, 64, 18, 48,
   36, 65, 24, 40, 38, 62, 45, 50, 59, 61, 53, 63, 39, 57, 43, 7, 20, 47, 31, 54, 40, 13, 26, 49, 65, 37,
   59, 19, 29, 41, 63, 25, 48, 34, 60, 64, 42, 55, 46, 52, 62, 58, 44, 19, 8, 56, 42, 45, 30, 25, 14, 65,
   50, 58, 38, 29, 20, 64, 39, 26, 36, 46, 61, 57, 48, 63, 40, 62, 53, 59, 9, 11, 53, 15, 16, 35, 57, 37,
   59, 21, 51, 23, 61, 52, 62, 27, 56, 58, 63, 41, 43, 54, 55, 47, 64, 49, 65, 60, 10, 52, 12, 33, 55, 17,
   18, 58, 38, 51, 22, 60, 24, 62, 53, 54, 28, 59, 39, 64, 44, 45, 63, 56, 57, 65, 50, 61, 51, 13, 14, 54,
   34, 56, 36, 19, 20, 60, 61, 52, 53, 25, 26, 55, 57, 29, 40, 42, 65, 63, 46, 64, 48, 58, 59, 62, 33, 34,
   30, 51, 52, 44, 38, 42, 36, 54, 45, 55, 39, 46, 40, 60, 50, 48, 53, 58, 56, 61, 62, 65, 59, 64, 57, 63,
   35, 31, 36, 43, 37, 51, 53, 34, 40, 47, 56, 41, 57, 42, 48, 49, 61, 46, 59, 52, 54, 65, 58, 60, 62, 55,
   63, 64, 32, 37, 38, 35, 41, 33, 39, 52, 53, 43, 44, 49, 50, 58, 59, 47, 45, 62, 57, 55, 51, 56, 64, 54,
   63, 60, 61, 65
}; // 0.82 kb
static const cart_index_t iCartYY_ab10_a7[360] = {
   0, 16, 18, 23, 24, 3, 39, 5, 41, 9, 10, 27, 55, 28, 57, 21, 22, 15, 63, 17, 64, 32, 49, 50, 60, 61,
   33, 45, 35, 47, 43, 44, 65, 51, 54, 56, 15, 1, 20, 4, 40, 21, 26, 43, 7, 27, 54, 11, 13, 59, 29, 16,
   63, 23, 25, 65, 19, 47, 31, 48, 34, 46, 60, 62, 49, 37, 41, 64, 42, 55, 52, 58, 17, 19, 2, 42, 6, 44,
   8, 22, 25, 56, 28, 58, 29, 12, 14, 64, 18, 65, 20, 24, 26, 45, 46, 30, 48, 36, 50, 38, 61, 62, 63, 39,
   40, 57, 59, 53, 9, 4, 36, 11, 53, 15, 40, 35, 31, 21, 51, 16, 34, 57, 48, 23, 61, 27, 46, 56, 42, 43,
   37, 59, 52, 62, 54, 63, 47, 41, 49, 65, 58, 60, 55, 64, 10, 34, 6, 52, 12, 33, 30, 17, 42, 51, 22, 55,
   46, 18, 36, 60, 24, 54, 40, 28, 48, 44, 58, 38, 62, 53, 45, 39, 56, 64, 65, 50, 59, 61, 63, 57, 3, 11,
   38, 16, 39, 9, 53, 32, 37, 15, 33, 23, 52, 50, 59, 27, 45, 21, 62, 44, 58, 35, 41, 57, 55, 63, 51, 61,
   43, 49, 47, 56, 64, 54, 60, 65, 33, 13, 8, 34, 30, 51, 14, 44, 19, 54, 45, 52, 25, 38, 20, 55, 39, 60,
   26, 50, 29, 56, 42, 36, 46, 40, 61, 53, 65, 58, 64, 57, 48, 63, 62, 59, 5, 37, 12, 41, 18, 32, 38, 10,
   52, 35, 17, 49, 58, 24, 53, 47, 28, 43, 59, 22, 62, 33, 55, 39, 64, 57, 44, 50, 51, 60, 54, 45, 63, 56,
   65, 61, 35, 7, 14, 31, 36, 43, 20, 51, 13, 47, 56, 37, 19, 53, 26, 41, 57, 49, 29, 61, 25, 54, 34, 40,
   42, 48, 65, 59, 60, 52, 55, 63, 46, 64, 58, 62, 32, 31, 30, 37, 38, 35, 36, 33, 34, 43, 44, 41, 42, 39,
   40, 49, 50, 47, 48, 45, 46, 51, 52, 53, 58, 59, 56, 57, 54, 55, 60, 61, 62, 65, 64, 63
}; // 0.70 kb
static const cart_index_t iCartYY_ab10_a8[270] = {
   0, 11, 12, 3, 16, 5, 18, 37, 38, 9, 10, 23, 24, 52, 53, 15, 27, 17, 28, 58, 59, 21, 22, 62, 39, 41,
   32, 33, 55, 35, 57, 49, 50, 45, 63, 47, 64, 43, 44, 51, 60, 61, 54, 56, 65, 9, 1, 14, 15, 4, 35, 36,
   7, 20, 21, 51, 11, 53, 13, 26, 27, 16, 56, 57, 19, 29, 23, 61, 25, 40, 31, 43, 54, 34, 47, 48, 37, 59,
   63, 46, 41, 42, 49, 65, 60, 52, 62, 55, 64, 58, 10, 13, 2, 33, 34, 17, 6, 19, 8, 51, 22, 52, 12, 25,
   14, 54, 55, 28, 18, 29, 20, 60, 24, 26, 30, 42, 44, 45, 46, 56, 36, 58, 38, 39, 40, 64, 48, 65, 50, 61,
   62, 53, 63, 57, 59, 3, 4, 30, 9, 11, 32, 38, 31, 36, 15, 33, 16, 39, 34, 40, 21, 23, 44, 50, 42, 48,
   27, 45, 46, 53, 37, 35, 51, 52, 43, 59, 41, 57, 61, 62, 49, 58, 47, 56, 54, 55, 63, 60, 65, 64, 5, 31,
   6, 32, 37, 10, 12, 34, 30, 35, 17, 41, 18, 42, 36, 43, 49, 22, 24, 46, 40, 47, 28, 48, 38, 52, 33, 44,
   58, 51, 53, 55, 39, 50, 59, 60, 62, 54, 45, 56, 64, 57, 65, 61, 63, 32, 7, 8, 35, 31, 33, 30, 13, 14,
   43, 44, 37, 38, 19, 20, 47, 41, 45, 39, 25, 26, 49, 50, 29, 36, 34, 51, 56, 42, 54, 40, 52, 53, 57, 48,
   55, 46, 60, 61, 65, 58, 59, 64, 63, 62
}; // 0.53 kb
static const cart_index_t iCartYY_ab10_a9[165] = {
   0, 4, 6, 11, 12, 3, 30, 5, 31, 9, 10, 16, 34, 18, 36, 23, 24, 15, 40, 17, 42, 21, 22, 27, 46, 28,
   48, 32, 37, 38, 52, 53, 33, 39, 35, 41, 43, 44, 49, 50, 58, 59, 62, 45, 47, 51, 55, 57, 60, 61, 54, 63,
   56, 64, 65, 3, 1, 8, 4, 30, 9, 14, 32, 7, 15, 33, 11, 13, 38, 20, 16, 39, 21, 26, 44, 19, 27, 45,
   23, 25, 50, 29, 35, 31, 36, 34, 40, 51, 53, 43, 37, 47, 56, 41, 57, 42, 48, 46, 61, 49, 54, 52, 59, 55,
   63, 60, 62, 65, 58, 64, 5, 7, 2, 31, 6, 32, 8, 10, 13, 35, 17, 37, 19, 12, 14, 41, 18, 43, 20, 22,
   25, 47, 28, 49, 29, 24, 26, 33, 34, 30, 42, 36, 44, 38, 51, 52, 54, 45, 55, 39, 46, 40, 48, 50, 60, 56,
   58, 53, 64, 57, 65, 59, 61, 62, 63
}; // 0.32 kb
static const cart_index_t iCartYY_ab10_a10[66] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65
}; // 0.13 kb
static const cart_index_t iCartYY_ab11_a0[78] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77
}; // 0.15 kb
static const cart_index_t iCartYY_ab11_a1[198] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 9, 33, 11, 3, 34, 7, 33, 10, 13, 35, 4, 34, 8, 12, 35, 14,
   6, 9, 17, 40, 10, 38, 19, 15, 11, 41, 16, 39, 13, 36, 12, 20, 37, 18, 14, 17, 21, 42, 23, 15, 44, 19,
   43, 22, 25, 45, 16, 46, 20, 24, 47, 26, 18, 21, 29, 52, 22, 50, 31, 27, 23, 53, 28, 51, 25, 48, 24, 32,
   49, 30, 26, 29, 27, 54, 31, 55, 28, 56, 32, 30, 39, 37, 35, 41, 34, 36, 33, 40, 38, 38, 57, 43, 58, 36,
   46, 40, 42, 57, 59, 47, 37, 44, 41, 58, 45, 59, 39, 51, 61, 45, 63, 49, 47, 53, 44, 60, 65, 46, 48, 42,
   52, 62, 43, 64, 50, 50, 67, 55, 69, 48, 56, 52, 54, 66, 71, 56, 49, 54, 53, 68, 55, 70, 51, 57, 62, 64,
   60, 58, 65, 61, 63, 59, 62, 66, 72, 68, 60, 73, 64, 72, 67, 70, 74, 61, 73, 65, 69, 74, 71, 63, 66, 68,
   77, 67, 76, 70, 75, 69, 71, 76, 75, 74, 77, 73, 75, 72, 77, 76
}; // 0.39 kb
static const cart_index_t iCartYY_ab11_a2[330] = {
   0, 9, 10, 5, 7, 33, 11, 1, 12, 3, 34, 8, 13, 14, 2, 35, 4, 6, 15, 3, 36, 11, 41, 34, 16, 37,
   4, 39, 13, 35, 5, 17, 38, 9, 33, 40, 39, 18, 6, 37, 35, 14, 7, 40, 19, 33, 10, 38, 41, 8, 20, 34,
   36, 12, 9, 21, 57, 17, 40, 42, 10, 57, 22, 38, 19, 43, 23, 11, 58, 15, 44, 41, 58, 12, 24, 36, 46, 20,
   25, 59, 13, 45, 16, 39, 59, 26, 14, 47, 37, 18, 27, 15, 60, 23, 53, 44, 28, 61, 16, 51, 25, 45, 17, 29,
   62, 21, 42, 52, 63, 30, 18, 49, 47, 26, 19, 64, 31, 43, 22, 50, 65, 20, 32, 46, 48, 24, 21, 27, 66, 29,
   52, 54, 22, 67, 28, 50, 31, 55, 29, 23, 68, 27, 54, 53, 69, 24, 30, 48, 56, 32, 31, 70, 25, 55, 28, 51,
   71, 32, 26, 56, 49, 30, 33, 42, 43, 40, 38, 57, 44, 34, 46, 41, 58, 36, 45, 47, 35, 59, 39, 37, 60, 36,
   48, 58, 65, 46, 61, 49, 37, 63, 59, 47, 38, 62, 50, 57, 43, 64, 51, 63, 39, 61, 45, 59, 40, 52, 64, 42,
   57, 62, 53, 41, 65, 44, 60, 58, 42, 54, 72, 52, 62, 66, 43, 72, 55, 64, 50, 67, 54, 44, 73, 53, 68, 60,
   55, 74, 45, 70, 51, 61, 73, 46, 56, 65, 69, 48, 74, 56, 47, 71, 63, 49, 75, 48, 49, 69, 71, 56, 50, 76,
   51, 67, 55, 70, 52, 53, 77, 54, 66, 68, 57, 66, 67, 62, 64, 72, 68, 58, 69, 60, 73, 65, 70, 71, 59, 74,
   61, 63, 66, 60, 75, 68, 77, 73, 67, 75, 61, 76, 70, 74, 62, 68, 76, 66, 72, 77, 76, 69, 63, 75, 74, 71,
   64, 77, 70, 72, 67, 76, 77, 65, 71, 73, 75, 69, 72, 73, 74, 77, 76, 75
}; // 0.64 kb
static const cart_index_t iCartYY_ab11_a3[450] = {
   0, 17, 19, 9, 10, 5, 38, 7, 40, 33, 15, 1, 20, 3, 36, 11, 12, 41, 8, 34, 16, 18, 2, 37, 4, 39,
   6, 13, 14, 35, 5, 21, 43, 17, 38, 9, 57, 33, 42, 40, 23, 3, 46, 11, 58, 15, 36, 44, 34, 41, 7, 42,
   22, 40, 19, 33, 43, 10, 57, 38, 25, 47, 4, 59, 13, 45, 35, 16, 37, 39, 44, 8, 24, 34, 46, 41, 20, 58,
   12, 36, 45, 26, 6, 47, 35, 59, 14, 39, 18, 37, 9, 29, 64, 21, 57, 17, 62, 40, 52, 42, 10, 62, 31, 57,
   22, 38, 50, 19, 64, 43, 27, 11, 65, 15, 60, 23, 58, 53, 41, 44, 28, 63, 13, 61, 16, 51, 39, 25, 59, 45,
   60, 12, 32, 36, 48, 58, 24, 65, 20, 46, 61, 30, 14, 49, 37, 63, 18, 59, 26, 47, 17, 27, 72, 29, 62, 21,
   66, 42, 54, 52, 29, 15, 73, 23, 68, 27, 60, 54, 44, 53, 19, 72, 28, 64, 31, 43, 55, 22, 67, 50, 31, 74,
   16, 70, 25, 55, 45, 28, 61, 51, 73, 20, 30, 46, 56, 65, 32, 69, 24, 48, 74, 32, 18, 56, 47, 71, 26, 63,
   30, 49, 21, 23, 77, 27, 66, 29, 68, 52, 53, 54, 22, 76, 25, 67, 28, 50, 51, 31, 70, 55, 75, 24, 26, 48,
   49, 69, 30, 71, 32, 56, 51, 49, 35, 63, 39, 61, 37, 45, 47, 59, 53, 34, 48, 41, 65, 44, 46, 60, 36, 58,
   33, 52, 50, 42, 43, 40, 64, 38, 62, 57, 38, 66, 55, 62, 50, 57, 67, 43, 72, 64, 68, 36, 56, 58, 69, 60,
   48, 73, 46, 65, 40, 54, 67, 52, 64, 42, 72, 57, 66, 62, 70, 56, 37, 71, 59, 74, 47, 61, 49, 63, 54, 41,
   69, 44, 73, 53, 65, 68, 58, 60, 55, 71, 39, 74, 45, 70, 59, 51, 63, 61, 50, 75, 45, 76, 51, 67, 61, 55,
   74, 70, 76, 48, 47, 69, 63, 75, 49, 74, 56, 71, 52, 44, 75, 53, 77, 54, 73, 66, 60, 68, 77, 46, 49, 65,
   71, 73, 56, 75, 48, 69, 42, 53, 76, 54, 72, 52, 77, 62, 68, 66, 43, 77, 51, 72, 55, 64, 70, 50, 76, 67,
   57, 68, 70, 66, 67, 62, 76, 64, 77, 72, 66, 58, 71, 60, 75, 68, 69, 77, 65, 73, 67, 69, 59, 75, 61, 76,
   63, 70, 71, 74, 62, 60, 74, 68, 76, 66, 75, 72, 73, 77, 64, 73, 61, 77, 70, 72, 74, 67, 75, 76, 72, 65,
   63, 73, 74, 77, 71, 76, 69, 75
}; // 0.88 kb
static const cart_index_t iCartYY_ab11_a4[540] = {
   0, 21, 22, 5, 17, 7, 19, 42, 43, 9, 10, 57, 38, 40, 33, 23, 1, 24, 15, 3, 44, 46, 8, 20, 11, 58,
   12, 36, 34, 41, 25, 26, 2, 45, 47, 16, 4, 18, 6, 59, 13, 14, 35, 37, 39, 27, 3, 48, 23, 11, 53, 65,
   34, 46, 15, 60, 36, 58, 41, 44, 28, 49, 4, 51, 63, 25, 13, 47, 35, 61, 16, 37, 39, 59, 45, 5, 29, 50,
   9, 21, 33, 43, 52, 64, 17, 38, 62, 57, 42, 40, 51, 30, 6, 61, 49, 45, 35, 26, 14, 63, 39, 18, 37, 47,
   59, 7, 52, 31, 33, 42, 10, 22, 62, 50, 40, 19, 64, 43, 57, 38, 53, 8, 32, 44, 34, 60, 48, 12, 24, 41,
   65, 20, 46, 36, 58, 9, 27, 67, 17, 29, 40, 64, 54, 72, 21, 57, 66, 62, 52, 42, 10, 66, 28, 38, 62, 19,
   31, 72, 55, 57, 22, 67, 50, 64, 43, 29, 11, 69, 27, 15, 54, 73, 41, 65, 23, 68, 58, 60, 44, 53, 68, 12,
   30, 60, 36, 73, 56, 20, 32, 58, 69, 24, 48, 46, 65, 31, 71, 13, 55, 74, 28, 16, 63, 39, 70, 25, 59, 45,
   61, 51, 70, 32, 14, 74, 56, 61, 37, 30, 18, 71, 59, 26, 47, 49, 63, 21, 15, 75, 29, 23, 52, 77, 44, 73,
   27, 66, 60, 68, 53, 54, 22, 75, 16, 50, 76, 31, 25, 74, 45, 67, 28, 61, 51, 70, 55, 17, 23, 76, 21, 27,
   42, 72, 53, 77, 29, 62, 68, 66, 54, 52, 76, 24, 18, 75, 48, 74, 47, 32, 26, 69, 63, 30, 49, 56, 71, 19,
   77, 25, 43, 72, 22, 28, 76, 51, 64, 31, 70, 55, 67, 50, 77, 20, 26, 73, 46, 75, 49, 24, 30, 65, 71, 32,
   56, 48, 69, 33, 54, 55, 40, 52, 38, 50, 66, 67, 42, 43, 72, 64, 62, 57, 54, 34, 56, 53, 41, 68, 69, 36,
   48, 44, 73, 46, 65, 58, 60, 55, 56, 35, 70, 71, 51, 39, 49, 37, 74, 45, 47, 59, 63, 61, 66, 36, 49, 68,
   58, 77, 71, 46, 56, 60, 75, 48, 69, 65, 73, 67, 48, 37, 76, 69, 70, 59, 56, 47, 75, 61, 49, 63, 71, 74,
   38, 68, 51, 57, 66, 43, 55, 77, 70, 62, 50, 76, 67, 72, 64, 50, 69, 39, 67, 75, 55, 45, 71, 59, 76, 51,
   63, 61, 74, 70, 40, 53, 70, 42, 54, 57, 67, 68, 76, 52, 64, 77, 72, 66, 62, 52, 41, 71, 54, 44, 66, 75,
   58, 69, 53, 77, 65, 73, 60, 68, 42, 44, 74, 52, 53, 62, 76, 60, 75, 54, 72, 73, 77, 68, 66, 43, 73, 45,
   64, 77, 50, 51, 75, 61, 72, 55, 74, 70, 76, 67, 72, 46, 47, 77, 65, 76, 63, 48, 49, 73, 74, 56, 71, 69,
   75, 57, 60, 61, 62, 68, 64, 70, 73, 74, 66, 67, 75, 76, 77, 72, 62, 58, 63, 66, 60, 72, 74, 65, 71, 68,
   76, 69, 75, 73, 77, 64, 65, 59, 72, 73, 67, 61, 69, 63, 77, 70, 71, 74, 75, 76
}; // 1.05 kb
static const cart_index_t iCartYY_ab11_a5[588] = {
   0, 29, 31, 21, 22, 5, 50, 7, 52, 9, 10, 17, 62, 19, 64, 33, 42, 43, 57, 38, 40, 27, 1, 32, 3, 48,
   23, 24, 53, 8, 15, 60, 11, 12, 65, 20, 44, 34, 46, 36, 58, 41, 28, 30, 2, 49, 4, 51, 6, 25, 26, 61,
   16, 63, 18, 13, 14, 45, 47, 35, 37, 39, 59, 5, 27, 55, 29, 50, 9, 67, 33, 54, 17, 38, 21, 66, 43, 72,
   40, 52, 64, 62, 57, 42, 29, 3, 56, 11, 69, 27, 48, 54, 34, 23, 68, 15, 36, 73, 46, 53, 41, 65, 58, 60,
   44, 7, 54, 28, 52, 31, 33, 55, 10, 66, 40, 19, 42, 72, 22, 67, 38, 62, 50, 64, 43, 57, 31, 56, 4, 71,
   13, 55, 35, 28, 49, 70, 25, 74, 47, 16, 37, 51, 63, 39, 59, 45, 61, 54, 8, 30, 34, 56, 53, 32, 68, 12,
   44, 73, 41, 20, 69, 24, 60, 36, 48, 46, 65, 58, 55, 32, 6, 56, 35, 70, 14, 51, 30, 74, 45, 71, 26, 39,
   18, 61, 49, 37, 47, 59, 63, 9, 23, 70, 27, 67, 17, 76, 40, 53, 21, 57, 29, 68, 64, 77, 42, 54, 72, 66,
   62, 52, 10, 68, 25, 66, 28, 38, 51, 19, 77, 57, 22, 62, 76, 31, 70, 43, 72, 55, 67, 50, 64, 21, 11, 71,
   15, 75, 29, 69, 52, 41, 27, 66, 23, 58, 77, 65, 54, 44, 73, 60, 68, 53, 22, 69, 13, 75, 16, 50, 39, 31,
   71, 67, 28, 76, 63, 25, 59, 55, 74, 45, 61, 51, 70, 66, 12, 26, 36, 49, 68, 30, 77, 20, 60, 75, 58, 24,
   71, 32, 73, 46, 56, 48, 69, 65, 67, 24, 14, 48, 37, 76, 18, 70, 32, 75, 61, 69, 30, 59, 26, 74, 56, 47,
   49, 63, 71, 17, 15, 74, 23, 76, 21, 75, 42, 44, 29, 62, 27, 60, 72, 73, 52, 53, 77, 68, 66, 54, 19, 73,
   16, 77, 25, 43, 45, 22, 75, 64, 31, 72, 74, 28, 61, 50, 76, 51, 70, 55, 67, 72, 20, 18, 46, 47, 77, 26,
   76, 24, 73, 74, 65, 32, 63, 30, 75, 48, 49, 56, 71, 69, 50, 48, 35, 69, 39, 67, 37, 55, 56, 76, 51, 75,
   49, 45, 47, 70, 71, 59, 63, 61, 74, 52, 34, 49, 41, 71, 54, 56, 66, 36, 53, 77, 44, 46, 75, 48, 68, 58,
   69, 65, 73, 60, 33, 53, 51, 54, 55, 40, 70, 38, 68, 42, 43, 52, 77, 50, 76, 57, 66, 67, 72, 64, 62, 38,
   60, 45, 68, 51, 57, 61, 43, 73, 62, 50, 66, 75, 55, 74, 64, 77, 70, 76, 67, 72, 62, 36, 47, 58, 63, 66,
   49, 72, 46, 68, 76, 60, 48, 74, 56, 77, 65, 71, 69, 75, 73, 40, 44, 61, 53, 70, 42, 74, 57, 60, 52, 64,
   54, 73, 67, 75, 62, 68, 76, 77, 72, 66, 64, 46, 37, 65, 59, 72, 47, 67, 48, 77, 70, 73, 56, 61, 49, 76,
   69, 63, 71, 74, 75, 42, 41, 63, 44, 74, 52, 71, 62, 58, 54, 72, 53, 65, 76, 69, 66, 60, 75, 73, 77, 68,
   43, 65, 39, 73, 45, 64, 59, 50, 69, 72, 55, 77, 71, 51, 63, 67, 75, 61, 74, 70, 76, 57, 58, 59, 60, 61,
   62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77
}; // 1.15 kb
static const cart_index_t iCartYY_ab11_a6[588] = {
   0, 27, 28, 5, 29, 7, 31, 54, 55, 9, 10, 21, 22, 66, 67, 17, 19, 72, 50, 52, 33, 38, 62, 40, 64, 42,
   43, 57, 29, 1, 30, 27, 3, 54, 56, 8, 32, 23, 68, 11, 69, 12, 24, 15, 73, 20, 48, 34, 53, 60, 36, 44,
   46, 41, 65, 58, 31, 32, 2, 55, 56, 28, 4, 30, 6, 70, 25, 71, 13, 26, 14, 74, 16, 18, 35, 49, 51, 45,
   47, 61, 37, 63, 39, 59, 21, 3, 49, 29, 11, 52, 71, 34, 56, 27, 66, 15, 75, 36, 48, 23, 77, 46, 69, 41,
   54, 68, 58, 53, 65, 44, 73, 60, 22, 48, 4, 50, 69, 31, 13, 56, 35, 67, 28, 75, 16, 49, 37, 76, 25, 47,
   39, 71, 55, 51, 63, 70, 59, 74, 45, 61, 5, 23, 51, 9, 27, 33, 55, 53, 70, 17, 38, 29, 50, 68, 76, 21,
   43, 77, 67, 54, 40, 57, 66, 42, 72, 52, 64, 62, 50, 24, 6, 67, 48, 55, 35, 32, 14, 76, 51, 69, 39, 30,
   18, 75, 45, 26, 37, 56, 70, 61, 49, 74, 47, 71, 59, 63, 7, 53, 25, 33, 54, 10, 28, 68, 51, 40, 19, 52,
   31, 77, 70, 42, 22, 76, 55, 66, 38, 43, 72, 57, 67, 62, 50, 64, 52, 8, 26, 54, 34, 66, 49, 12, 30, 53,
   77, 41, 71, 20, 32, 44, 75, 24, 56, 36, 68, 73, 46, 60, 48, 58, 69, 65, 9, 15, 61, 17, 23, 40, 70, 44,
   74, 21, 57, 27, 67, 60, 75, 29, 64, 73, 76, 53, 42, 62, 68, 52, 77, 54, 72, 66, 10, 60, 16, 38, 68, 19,
   25, 73, 45, 57, 22, 66, 28, 75, 61, 62, 31, 74, 51, 77, 43, 50, 76, 64, 70, 72, 55, 67, 17, 11, 63, 21,
   15, 42, 74, 41, 71, 29, 62, 23, 76, 58, 69, 27, 72, 65, 75, 44, 52, 66, 60, 54, 73, 53, 77, 68, 62, 12,
   18, 66, 36, 72, 47, 20, 26, 68, 76, 58, 63, 24, 30, 60, 74, 32, 49, 46, 77, 75, 48, 73, 56, 65, 71, 69,
   19, 65, 13, 43, 73, 22, 16, 69, 39, 64, 31, 77, 25, 71, 59, 72, 28, 63, 45, 75, 50, 55, 74, 67, 61, 76,
   51, 70, 64, 20, 14, 72, 46, 67, 37, 24, 18, 77, 70, 65, 59, 32, 26, 73, 61, 30, 47, 48, 76, 74, 56, 75,
   49, 69, 63, 71, 33, 44, 45, 40, 53, 38, 51, 60, 61, 42, 43, 54, 55, 73, 74, 52, 50, 75, 70, 68, 57, 64,
   77, 62, 76, 66, 67, 72, 42, 34, 47, 52, 41, 62, 63, 36, 49, 54, 72, 44, 74, 46, 56, 53, 76, 48, 71, 58,
   66, 77, 65, 68, 69, 60, 75, 73, 43, 46, 35, 64, 65, 50, 39, 48, 37, 72, 55, 73, 45, 56, 47, 77, 51, 49,
   59, 69, 67, 70, 71, 76, 63, 75, 61, 74, 57, 36, 37, 62, 58, 64, 59, 46, 47, 66, 67, 60, 61, 48, 49, 68,
   70, 56, 63, 65, 72, 76, 69, 77, 71, 73, 74, 75, 38, 58, 39, 57, 60, 43, 45, 65, 59, 62, 50, 68, 51, 69,
   63, 66, 55, 71, 61, 73, 64, 67, 75, 72, 74, 77, 70, 76, 40, 41, 59, 42, 44, 57, 61, 58, 63, 52, 64, 53,
   70, 65, 71, 54, 67, 69, 74, 60, 62, 72, 73, 66, 75, 68, 76, 77
}; // 1.15 kb
static const cart_index_t iCartYY_ab11_a7[540] = {
   0, 23, 25, 27, 28, 5, 51, 7, 53, 9, 10, 29, 68, 31, 70, 21, 22, 17, 76, 19, 77, 33, 54, 55, 66, 67,
   38, 50, 40, 52, 42, 43, 72, 57, 62, 64, 21, 1, 26, 3, 49, 29, 30, 52, 8, 27, 66, 11, 12, 71, 32, 15,
   75, 23, 24, 77, 20, 54, 34, 56, 36, 48, 68, 69, 53, 41, 44, 73, 46, 60, 58, 65, 22, 24, 2, 48, 4, 50,
   6, 31, 32, 67, 28, 69, 30, 13, 14, 75, 16, 76, 18, 25, 26, 55, 56, 35, 49, 37, 51, 39, 70, 71, 74, 45,
   47, 61, 63, 59, 5, 15, 45, 23, 51, 9, 61, 33, 44, 17, 38, 27, 60, 55, 74, 29, 50, 21, 75, 43, 73, 40,
   53, 70, 68, 76, 57, 67, 42, 54, 52, 64, 77, 62, 66, 72, 17, 3, 47, 11, 63, 21, 49, 42, 34, 29, 62, 15,
   36, 74, 56, 23, 76, 27, 48, 72, 46, 52, 41, 71, 58, 69, 66, 75, 54, 44, 53, 77, 65, 68, 60, 73, 7, 44,
   16, 53, 25, 33, 45, 10, 60, 40, 19, 54, 73, 28, 61, 52, 31, 42, 74, 22, 75, 38, 68, 51, 77, 70, 43, 55,
   57, 66, 62, 50, 76, 64, 72, 67, 19, 46, 4, 65, 13, 43, 35, 22, 48, 64, 31, 73, 56, 16, 37, 77, 25, 72,
   47, 28, 49, 50, 69, 39, 71, 59, 55, 45, 67, 75, 76, 51, 63, 70, 74, 61, 42, 8, 18, 34, 47, 52, 26, 62,
   12, 54, 72, 41, 20, 63, 30, 44, 74, 53, 32, 76, 24, 66, 36, 49, 46, 56, 77, 71, 68, 58, 60, 75, 48, 73,
   65, 69, 43, 20, 6, 46, 35, 64, 14, 50, 24, 72, 55, 65, 32, 39, 18, 73, 45, 77, 26, 51, 30, 67, 48, 37,
   56, 47, 70, 59, 76, 69, 75, 61, 49, 74, 71, 63, 9, 11, 59, 15, 61, 17, 63, 40, 41, 21, 57, 23, 58, 70,
   71, 27, 67, 29, 69, 64, 65, 42, 44, 74, 60, 75, 62, 76, 52, 53, 54, 72, 73, 66, 68, 77, 10, 58, 13, 60,
   16, 38, 39, 19, 65, 57, 22, 68, 69, 25, 59, 66, 28, 62, 63, 31, 71, 43, 73, 45, 75, 61, 50, 51, 64, 77,
   72, 55, 74, 67, 76, 70, 57, 12, 14, 36, 37, 62, 18, 64, 20, 66, 67, 58, 24, 59, 26, 60, 61, 68, 30, 70,
   32, 72, 46, 47, 48, 49, 76, 63, 77, 65, 73, 74, 56, 75, 69, 71, 38, 36, 35, 58, 39, 57, 37, 43, 46, 62,
   50, 60, 48, 45, 47, 68, 51, 66, 49, 55, 56, 64, 65, 59, 69, 63, 67, 61, 72, 73, 77, 70, 71, 76, 75, 74,
   40, 34, 37, 41, 59, 42, 47, 57, 36, 52, 64, 44, 46, 61, 49, 53, 70, 54, 56, 67, 48, 62, 58, 63, 65, 71,
   72, 74, 66, 60, 68, 76, 69, 77, 73, 75, 33, 41, 39, 44, 45, 40, 59, 38, 58, 42, 43, 53, 65, 51, 63, 54,
   55, 52, 71, 50, 69, 57, 60, 61, 73, 74, 64, 70, 62, 68, 66, 67, 75, 72, 77, 76
}; // 1.05 kb
static const cart_index_t iCartYY_ab11_a8[450] = {
   0, 15, 16, 5, 23, 7, 25, 44, 45, 9, 10, 27, 28, 60, 61, 17, 29, 19, 31, 73, 74, 21, 22, 75, 51, 53,
   33, 38, 68, 40, 70, 54, 55, 50, 76, 52, 77, 42, 43, 57, 66, 67, 62, 64, 72, 17, 1, 18, 21, 3, 42, 47,
   8, 26, 29, 62, 11, 63, 12, 30, 27, 15, 72, 74, 20, 32, 23, 76, 24, 49, 34, 52, 66, 36, 54, 56, 41, 71,
   75, 48, 44, 46, 53, 77, 68, 58, 69, 60, 73, 65, 19, 20, 2, 43, 46, 22, 4, 24, 6, 64, 31, 65, 13, 32,
   14, 72, 73, 28, 16, 30, 18, 77, 25, 26, 35, 48, 50, 55, 56, 67, 37, 69, 39, 45, 47, 75, 49, 76, 51, 70,
   71, 59, 74, 61, 63, 9, 3, 37, 17, 11, 40, 59, 34, 47, 21, 57, 15, 61, 36, 49, 29, 23, 64, 70, 46, 56,
   27, 67, 48, 63, 41, 42, 62, 58, 52, 71, 44, 74, 76, 69, 53, 65, 54, 72, 66, 60, 75, 68, 77, 73, 10, 36,
   4, 38, 58, 19, 13, 46, 35, 57, 22, 60, 16, 48, 37, 62, 68, 31, 25, 56, 47, 66, 28, 49, 39, 65, 43, 50,
   69, 64, 59, 73, 45, 51, 63, 77, 71, 72, 55, 67, 75, 61, 76, 70, 74, 5, 11, 39, 9, 15, 33, 45, 41, 59,
   17, 38, 23, 51, 58, 63, 21, 27, 43, 55, 65, 71, 29, 50, 69, 61, 44, 40, 57, 60, 42, 74, 53, 70, 67, 75,
   54, 73, 52, 64, 62, 68, 76, 66, 72, 77, 38, 12, 6, 57, 36, 43, 35, 20, 14, 62, 50, 58, 39, 24, 18, 66,
   60, 55, 45, 32, 26, 68, 51, 30, 37, 46, 64, 67, 48, 72, 47, 65, 59, 61, 49, 73, 56, 77, 70, 76, 69, 63,
   75, 74, 71, 7, 41, 13, 33, 44, 10, 16, 58, 39, 40, 19, 53, 25, 65, 59, 42, 54, 22, 28, 69, 63, 52, 31,
   71, 45, 60, 38, 43, 73, 57, 61, 68, 51, 55, 74, 66, 75, 62, 50, 64, 77, 70, 72, 67, 76, 40, 8, 14, 42,
   34, 57, 37, 12, 18, 52, 64, 41, 59, 20, 26, 54, 44, 67, 61, 24, 30, 53, 70, 32, 47, 36, 62, 72, 46, 66,
   49, 58, 63, 74, 56, 60, 48, 68, 76, 77, 65, 71, 73, 75, 69, 33, 34, 35, 40, 41, 38, 39, 36, 37, 42, 43,
   44, 45, 46, 47, 52, 53, 50, 51, 48, 49, 54, 55, 56, 59, 58, 57, 64, 65, 62, 63, 60, 61, 70, 71, 68, 69,
   66, 67, 72, 73, 74, 77, 76, 75
}; // 0.88 kb
static const cart_index_t iCartYY_ab11_a9[330] = {
   0, 11, 13, 15, 16, 5, 39, 7, 41, 9, 10, 23, 58, 25, 59, 27, 28, 17, 63, 19, 65, 21, 22, 29, 69, 31,
   71, 33, 44, 45, 60, 61, 38, 51, 40, 53, 42, 43, 54, 55, 73, 74, 75, 50, 52, 57, 68, 70, 66, 67, 62, 76,
   64, 77, 72, 9, 1, 14, 3, 37, 17, 18, 40, 8, 21, 57, 11, 12, 59, 26, 15, 61, 29, 30, 64, 20, 27, 67,
   23, 24, 70, 32, 42, 34, 47, 36, 49, 62, 63, 52, 41, 54, 72, 44, 74, 46, 56, 48, 76, 53, 66, 58, 71, 60,
   75, 68, 69, 77, 65, 73, 10, 12, 2, 36, 4, 38, 6, 19, 20, 57, 22, 58, 24, 13, 14, 60, 16, 62, 18, 31,
   32, 66, 28, 68, 30, 25, 26, 43, 46, 35, 48, 37, 50, 39, 64, 65, 72, 55, 73, 45, 56, 47, 49, 51, 77, 67,
   69, 59, 75, 61, 76, 63, 70, 71, 74, 5, 3, 35, 11, 39, 9, 37, 33, 34, 17, 38, 15, 36, 45, 47, 23, 51,
   21, 49, 43, 46, 29, 50, 27, 48, 55, 56, 40, 41, 59, 58, 63, 57, 61, 42, 44, 52, 64, 53, 70, 65, 71, 69,
   67, 54, 62, 60, 74, 68, 76, 66, 75, 72, 73, 77, 7, 34, 4, 41, 13, 33, 35, 10, 36, 40, 19, 44, 46, 16,
   37, 53, 25, 42, 47, 22, 48, 52, 31, 54, 56, 28, 49, 38, 58, 39, 65, 59, 43, 45, 57, 60, 62, 50, 68, 51,
   69, 63, 71, 55, 66, 64, 73, 61, 77, 70, 72, 74, 67, 75, 76, 33, 8, 6, 34, 35, 40, 14, 38, 12, 42, 43,
   41, 20, 39, 18, 44, 45, 52, 26, 50, 24, 54, 55, 53, 32, 51, 30, 57, 36, 37, 46, 47, 64, 59, 62, 58, 66,
   67, 60, 61, 48, 49, 56, 70, 68, 72, 65, 63, 73, 74, 77, 71, 76, 69, 75
}; // 0.64 kb
static const cart_index_t iCartYY_ab11_a10[198] = {
   0, 3, 4, 5, 11, 7, 13, 34, 35, 9, 10, 15, 16, 36, 37, 17, 23, 19, 25, 46, 47, 21, 22, 27, 28, 48,
   49, 29, 31, 56, 39, 41, 33, 38, 58, 40, 59, 44, 45, 51, 63, 53, 65, 42, 43, 50, 69, 52, 71, 54, 55, 57,
   60, 61, 62, 68, 64, 70, 73, 74, 66, 67, 75, 76, 77, 72, 5, 1, 6, 9, 3, 33, 35, 8, 14, 17, 38, 11,
   39, 12, 18, 21, 15, 43, 45, 20, 26, 29, 50, 23, 51, 24, 30, 27, 55, 32, 37, 34, 40, 57, 36, 42, 47, 41,
   59, 61, 49, 44, 46, 52, 64, 67, 48, 54, 56, 53, 70, 62, 58, 63, 66, 60, 72, 74, 65, 71, 68, 76, 69, 75,
   73, 77, 7, 8, 2, 33, 34, 10, 4, 12, 6, 40, 19, 41, 13, 20, 14, 42, 44, 22, 16, 24, 18, 52, 31, 53,
   25, 32, 26, 54, 28, 30, 35, 36, 38, 43, 46, 57, 37, 58, 39, 45, 47, 60, 48, 62, 50, 55, 56, 66, 49, 68,
   51, 64, 65, 59, 72, 73, 67, 61, 69, 63, 77, 70, 71, 74, 75, 76
}; // 0.39 kb
static const cart_index_t iCartYY_ab11_a11[78] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77
}; // 0.15 kb
static const cart_index_t iCartYY_ab12_a0[91] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90
}; // 0.18 kb
static const cart_index_t iCartYY_ab12_a1[234] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 11, 4, 37, 12, 36, 6, 3, 9, 38, 36, 14, 8, 5, 38, 10, 37, 7,
   13, 9, 15, 41, 10, 39, 17, 16, 11, 43, 40, 13, 19, 18, 44, 12, 42, 20, 14, 23, 16, 47, 24, 45, 18, 15,
   21, 49, 46, 26, 20, 17, 50, 22, 48, 19, 25, 21, 27, 53, 22, 51, 29, 28, 23, 55, 52, 25, 31, 30, 56, 24,
   54, 32, 26, 33, 28, 59, 34, 57, 30, 27, 33, 61, 58, 35, 32, 29, 62, 34, 60, 31, 35, 38, 41, 39, 43, 37,
   40, 44, 42, 36, 64, 40, 48, 65, 46, 42, 39, 63, 50, 45, 65, 44, 41, 49, 63, 47, 43, 64, 49, 53, 66, 50,
   68, 51, 55, 47, 67, 56, 69, 45, 70, 48, 52, 71, 54, 46, 76, 52, 60, 77, 58, 54, 51, 73, 62, 57, 75, 56,
   53, 61, 72, 59, 55, 74, 61, 59, 78, 62, 79, 57, 80, 60, 58, 63, 66, 68, 67, 64, 70, 69, 71, 65, 74, 67,
   82, 75, 81, 69, 66, 72, 83, 81, 77, 71, 68, 83, 73, 82, 70, 76, 72, 78, 86, 73, 84, 79, 78, 74, 88, 85,
   76, 80, 79, 89, 75, 87, 80, 77, 83, 86, 84, 88, 82, 85, 89, 87, 81, 90, 85, 87, 84, 90, 89, 86, 88, 90
}; // 0.46 kb
static const cart_index_t iCartYY_ab12_a2[396] = {
   0, 9, 10, 3, 5, 38, 11, 1, 13, 4, 37, 7, 12, 14, 2, 36, 6, 8, 3, 15, 39, 9, 38, 41, 16, 4,
   40, 11, 43, 37, 5, 41, 17, 38, 10, 39, 18, 42, 6, 44, 12, 36, 43, 7, 19, 37, 40, 13, 44, 20, 8, 42,
   36, 14, 9, 21, 63, 15, 41, 49, 10, 63, 22, 39, 17, 50, 23, 11, 64, 16, 47, 43, 24, 65, 12, 45, 18, 44,
   64, 13, 25, 40, 48, 19, 65, 26, 14, 46, 42, 20, 15, 27, 66, 21, 49, 53, 28, 16, 67, 23, 55, 47, 17, 68,
   29, 50, 22, 51, 30, 69, 18, 56, 24, 45, 70, 19, 31, 48, 52, 25, 71, 32, 20, 54, 46, 26, 21, 33, 72, 27,
   53, 61, 22, 73, 34, 51, 29, 62, 33, 23, 74, 28, 59, 55, 34, 75, 24, 57, 30, 56, 76, 25, 35, 52, 60, 31,
   77, 35, 26, 58, 54, 32, 27, 28, 78, 33, 61, 59, 29, 79, 30, 62, 34, 57, 80, 31, 32, 60, 58, 35, 45, 46,
   36, 65, 44, 42, 47, 37, 48, 43, 64, 40, 38, 49, 50, 41, 39, 63, 39, 66, 51, 63, 50, 68, 67, 40, 52, 64,
   70, 48, 41, 53, 68, 49, 63, 66, 69, 54, 42, 71, 65, 46, 55, 43, 70, 47, 67, 64, 56, 71, 44, 69, 45, 65,
   57, 81, 45, 75, 56, 69, 81, 58, 46, 77, 71, 54, 59, 47, 82, 55, 74, 67, 82, 48, 60, 70, 76, 52, 49, 61,
   83, 53, 66, 72, 50, 83, 62, 68, 51, 73, 51, 84, 57, 73, 62, 79, 85, 52, 58, 76, 80, 60, 53, 59, 86, 61,
   72, 78, 87, 60, 54, 80, 77, 58, 61, 55, 88, 59, 78, 74, 62, 89, 56, 79, 57, 75, 63, 72, 73, 66, 68, 83,
   74, 64, 76, 67, 82, 70, 75, 77, 65, 81, 69, 71, 66, 78, 84, 72, 83, 86, 78, 67, 85, 74, 88, 82, 68, 86,
   79, 83, 73, 84, 79, 87, 69, 89, 75, 81, 88, 70, 80, 82, 85, 76, 89, 80, 71, 87, 81, 77, 72, 74, 90, 78,
   86, 88, 73, 90, 75, 84, 79, 89, 90, 76, 77, 85, 87, 80, 84, 85, 81, 90, 89, 87, 86, 82, 87, 88, 90, 85,
   83, 88, 89, 86, 84, 90
}; // 0.77 kb
static const cart_index_t iCartYY_ab12_a3[550] = {
   0, 15, 17, 9, 10, 3, 39, 5, 41, 38, 16, 1, 19, 4, 40, 11, 13, 43, 7, 37, 18, 20, 2, 42, 6, 44,
   8, 12, 14, 36, 23, 4, 48, 11, 64, 16, 40, 47, 37, 43, 24, 46, 6, 65, 12, 45, 36, 18, 42, 44, 3, 21,
   50, 15, 39, 9, 63, 38, 49, 41, 45, 26, 8, 46, 36, 65, 14, 44, 20, 42, 5, 49, 22, 41, 17, 38, 50, 10,
   63, 39, 47, 7, 25, 37, 48, 43, 19, 64, 13, 40, 9, 27, 68, 21, 63, 15, 66, 41, 53, 49, 10, 66, 29, 63,
   22, 39, 51, 17, 68, 50, 28, 11, 70, 16, 67, 23, 64, 55, 43, 47, 67, 13, 31, 40, 52, 64, 25, 70, 19, 48,
   30, 71, 12, 69, 18, 56, 44, 24, 65, 45, 69, 32, 14, 54, 42, 71, 20, 65, 26, 46, 33, 16, 82, 23, 74, 28,
   67, 59, 47, 55, 34, 81, 18, 75, 24, 57, 45, 30, 69, 56, 15, 33, 83, 27, 66, 21, 72, 49, 61, 53, 81, 35,
   20, 58, 46, 77, 26, 71, 32, 54, 17, 83, 34, 68, 29, 50, 62, 22, 73, 51, 82, 19, 35, 48, 60, 70, 31, 76,
   25, 52, 21, 28, 86, 33, 72, 27, 78, 53, 59, 61, 22, 84, 30, 73, 34, 51, 57, 29, 79, 62, 27, 23, 88, 28,
   78, 33, 74, 61, 55, 59, 85, 25, 32, 52, 58, 76, 35, 80, 31, 60, 29, 89, 24, 79, 30, 62, 56, 34, 75, 57,
   87, 31, 26, 60, 54, 80, 32, 77, 35, 58, 38, 53, 51, 49, 50, 41, 68, 39, 66, 63, 55, 37, 52, 43, 70, 47,
   48, 67, 40, 64, 56, 54, 36, 71, 44, 69, 42, 45, 46, 65, 74, 40, 60, 64, 76, 67, 52, 82, 48, 70, 75, 58,
   42, 77, 65, 81, 46, 69, 54, 71, 39, 72, 62, 66, 51, 63, 73, 50, 83, 68, 57, 77, 44, 81, 45, 75, 65, 56,
   71, 69, 41, 61, 73, 53, 68, 49, 83, 63, 72, 66, 59, 43, 76, 47, 82, 55, 70, 74, 64, 67, 49, 59, 84, 61,
   83, 53, 86, 66, 78, 72, 50, 86, 57, 83, 62, 68, 79, 51, 84, 73, 61, 47, 85, 55, 88, 59, 82, 78, 67, 74,
   62, 87, 45, 89, 56, 79, 69, 57, 81, 75, 88, 48, 58, 70, 80, 82, 60, 85, 52, 76, 89, 60, 46, 80, 71, 87,
   54, 81, 58, 77, 90, 52, 54, 76, 77, 85, 58, 87, 60, 80, 51, 90, 56, 84, 57, 73, 75, 62, 89, 79, 53, 55,
   90, 59, 86, 61, 88, 72, 74, 78, 63, 78, 79, 72, 73, 66, 84, 68, 86, 83, 78, 64, 80, 67, 85, 74, 76, 88,
   70, 82, 79, 80, 65, 87, 69, 89, 71, 75, 77, 81, 72, 67, 87, 74, 90, 78, 85, 86, 82, 88, 73, 85, 69, 90,
   75, 84, 81, 79, 87, 89, 66, 74, 89, 78, 84, 72, 90, 83, 88, 86, 84, 76, 71, 85, 81, 90, 77, 89, 80, 87,
   68, 88, 75, 86, 79, 83, 89, 73, 90, 84, 86, 70, 77, 82, 87, 88, 80, 90, 76, 85, 83, 82, 81, 88, 89, 86,
   87, 84, 85, 90
}; // 1.07 kb
static const cart_index_t iCartYY_ab12_a4[675] = {
   0, 21, 22, 3, 15, 5, 17, 49, 50, 9, 10, 63, 39, 41, 38, 23, 1, 25, 16, 4, 47, 48, 7, 19, 11, 64,
   13, 40, 37, 43, 24, 26, 2, 45, 46, 18, 6, 20, 8, 65, 12, 14, 36, 42, 44, 3, 27, 51, 9, 21, 38, 50,
   53, 68, 15, 39, 66, 63, 49, 41, 28, 4, 52, 23, 11, 55, 70, 37, 48, 16, 67, 40, 64, 43, 47, 5, 53, 29,
   38, 49, 10, 22, 66, 51, 41, 17, 68, 50, 63, 39, 30, 54, 6, 56, 71, 24, 12, 46, 36, 69, 18, 42, 44, 65,
   45, 55, 7, 31, 47, 37, 67, 52, 13, 25, 43, 70, 19, 48, 40, 64, 56, 32, 8, 69, 54, 45, 36, 26, 14, 71,
   44, 20, 42, 46, 65, 9, 33, 73, 15, 27, 41, 68, 61, 83, 21, 63, 72, 66, 53, 49, 10, 72, 34, 39, 66, 17,
   29, 83, 62, 63, 22, 73, 51, 68, 50, 33, 11, 76, 28, 16, 59, 82, 43, 70, 23, 74, 64, 67, 47, 55, 34, 77,
   12, 57, 81, 30, 18, 71, 44, 75, 24, 65, 45, 69, 56, 74, 13, 35, 67, 40, 82, 60, 19, 31, 64, 76, 25, 52,
   48, 70, 75, 35, 14, 81, 58, 69, 42, 32, 20, 77, 65, 26, 46, 54, 71, 15, 28, 84, 21, 33, 49, 83, 59, 86,
   27, 66, 78, 72, 61, 53, 27, 16, 85, 33, 23, 61, 88, 47, 82, 28, 78, 67, 74, 55, 59, 17, 86, 30, 50, 83,
   22, 34, 84, 57, 68, 29, 79, 62, 73, 51, 29, 87, 18, 62, 89, 34, 24, 81, 45, 79, 30, 69, 56, 75, 57, 88,
   19, 32, 82, 48, 85, 58, 25, 35, 70, 80, 31, 60, 52, 76, 89, 31, 20, 87, 60, 81, 46, 35, 26, 80, 71, 32,
   54, 58, 77, 21, 23, 90, 27, 28, 53, 86, 55, 88, 33, 72, 74, 78, 59, 61, 22, 90, 24, 51, 84, 29, 30, 89,
   56, 73, 34, 75, 57, 79, 62, 90, 25, 26, 85, 52, 87, 54, 31, 32, 76, 77, 35, 58, 60, 80, 57, 58, 36, 75,
   77, 56, 44, 54, 42, 81, 45, 46, 65, 71, 69, 59, 37, 60, 55, 43, 74, 76, 40, 52, 47, 82, 48, 70, 64, 67,
   38, 61, 62, 41, 53, 39, 51, 72, 73, 49, 50, 83, 68, 66, 63, 39, 78, 57, 63, 72, 50, 62, 86, 79, 66, 51,
   84, 73, 83, 68, 78, 40, 58, 74, 64, 88, 80, 48, 60, 67, 85, 52, 76, 70, 82, 41, 59, 79, 49, 61, 63, 73,
   78, 84, 53, 68, 86, 83, 72, 66, 79, 60, 42, 89, 80, 75, 65, 58, 46, 87, 69, 54, 71, 77, 81, 61, 43, 80,
   59, 47, 78, 85, 64, 76, 55, 88, 70, 82, 67, 74, 62, 80, 44, 79, 87, 57, 45, 77, 65, 89, 56, 71, 69, 81,
   75, 51, 85, 45, 73, 90, 62, 56, 87, 69, 84, 57, 81, 75, 89, 79, 84, 52, 46, 90, 76, 89, 71, 60, 54, 85,
   81, 58, 77, 80, 87, 53, 47, 87, 61, 55, 72, 90, 67, 85, 59, 86, 82, 88, 74, 78, 86, 48, 54, 88, 70, 90,
   77, 52, 58, 82, 87, 60, 80, 76, 85, 49, 55, 89, 53, 59, 66, 84, 74, 90, 61, 83, 88, 86, 78, 72, 50, 88,
   56, 68, 86, 51, 57, 90, 75, 83, 62, 89, 79, 84, 73, 63, 74, 75, 66, 78, 68, 79, 88, 89, 72, 73, 90, 84,
   86, 83, 72, 64, 77, 78, 67, 86, 87, 70, 80, 74, 90, 76, 85, 82, 88, 73, 76, 65, 84, 85, 79, 69, 80, 71,
   90, 75, 77, 81, 87, 89, 66, 67, 81, 72, 74, 83, 89, 82, 87, 78, 84, 85, 90, 88, 86, 68, 82, 69, 83, 88,
   73, 75, 85, 81, 86, 79, 87, 89, 90, 84, 83, 70, 71, 86, 82, 84, 81, 76, 77, 88, 89, 80, 87, 85, 90
}; // 1.32 kb
static const cart_index_t iCartYY_ab12_a5[756] = {
   0, 27, 29, 21, 22, 3, 51, 5, 53, 9, 10, 15, 66, 17, 68, 38, 49, 50, 63, 39, 41, 28, 1, 31, 4, 52,
   23, 25, 55, 7, 16, 67, 11, 13, 70, 19, 47, 37, 48, 40, 64, 43, 30, 32, 2, 54, 6, 56, 8, 24, 26, 69,
   18, 71, 20, 12, 14, 45, 46, 36, 42, 44, 65, 33, 4, 60, 11, 76, 28, 52, 59, 37, 23, 74, 16, 40, 82, 48,
   55, 43, 70, 64, 67, 47, 34, 58, 6, 77, 12, 57, 36, 30, 54, 75, 24, 81, 46, 18, 42, 56, 71, 44, 65, 45,
   69, 3, 33, 62, 27, 51, 9, 73, 38, 61, 15, 39, 21, 72, 50, 83, 41, 53, 68, 66, 63, 49, 57, 35, 8, 58,
   36, 75, 14, 56, 32, 81, 45, 77, 26, 44, 20, 69, 54, 42, 46, 65, 71, 5, 61, 34, 53, 29, 38, 62, 10, 72,
   41, 17, 49, 83, 22, 73, 39, 66, 51, 68, 50, 63, 59, 7, 35, 37, 60, 55, 31, 74, 13, 47, 82, 43, 19, 76,
   25, 67, 40, 52, 48, 70, 64, 9, 28, 79, 33, 73, 15, 84, 41, 59, 21, 63, 27, 78, 68, 86, 49, 61, 83, 72,
   66, 53, 10, 78, 30, 72, 34, 39, 57, 17, 86, 63, 22, 66, 84, 29, 79, 50, 83, 62, 73, 51, 68, 27, 11, 80,
   16, 85, 33, 76, 61, 43, 28, 78, 23, 64, 88, 70, 59, 47, 82, 67, 74, 55, 78, 13, 32, 40, 58, 74, 35, 88,
   19, 67, 85, 64, 25, 80, 31, 82, 48, 60, 52, 76, 70, 29, 80, 12, 87, 18, 62, 44, 34, 77, 79, 30, 89, 71,
   24, 65, 57, 81, 45, 69, 56, 75, 79, 31, 14, 60, 42, 89, 20, 75, 35, 87, 69, 80, 32, 65, 26, 81, 58, 46,
   54, 71, 77, 21, 16, 87, 23, 90, 27, 85, 53, 47, 33, 72, 28, 67, 86, 82, 61, 55, 88, 74, 78, 59, 22, 85,
   18, 90, 24, 51, 45, 29, 87, 73, 34, 84, 81, 30, 69, 62, 89, 56, 75, 57, 79, 15, 23, 89, 28, 84, 21, 90,
   49, 55, 27, 66, 33, 74, 83, 88, 53, 59, 86, 78, 72, 61, 84, 25, 20, 52, 46, 90, 26, 89, 31, 85, 81, 76,
   35, 71, 32, 87, 60, 54, 58, 77, 80, 17, 88, 24, 86, 30, 50, 56, 22, 90, 68, 29, 83, 89, 34, 75, 51, 84,
   57, 79, 62, 73, 86, 19, 26, 48, 54, 88, 32, 90, 25, 82, 87, 70, 31, 77, 35, 85, 52, 58, 60, 80, 76, 38,
   59, 57, 61, 62, 41, 79, 39, 78, 49, 50, 53, 86, 51, 84, 63, 72, 73, 83, 68, 66, 61, 37, 58, 43, 80, 59,
   60, 78, 40, 55, 88, 47, 48, 85, 52, 74, 64, 76, 70, 82, 67, 62, 60, 36, 80, 44, 79, 42, 57, 58, 89, 56,
   87, 54, 45, 46, 75, 77, 65, 71, 69, 81, 72, 40, 54, 64, 77, 78, 58, 86, 48, 74, 90, 67, 52, 87, 60, 88,
   70, 80, 76, 85, 82, 73, 52, 42, 76, 65, 84, 46, 79, 60, 90, 75, 85, 58, 69, 54, 89, 80, 71, 77, 81, 87,
   39, 74, 56, 78, 57, 63, 75, 50, 88, 66, 51, 72, 90, 62, 89, 68, 86, 79, 84, 73, 83, 51, 76, 44, 85, 45,
   73, 65, 62, 80, 84, 57, 90, 77, 56, 71, 79, 87, 69, 81, 75, 89, 41, 55, 75, 59, 79, 49, 89, 63, 74, 53,
   68, 61, 88, 73, 90, 66, 78, 84, 86, 83, 72, 53, 43, 77, 47, 87, 61, 80, 72, 64, 59, 86, 55, 70, 90, 76,
   78, 67, 85, 82, 88, 74, 49, 47, 81, 55, 89, 53, 87, 66, 67, 61, 83, 59, 82, 84, 85, 72, 74, 90, 88, 86,
   78, 50, 82, 45, 88, 56, 68, 69, 51, 85, 83, 62, 86, 87, 57, 81, 73, 90, 75, 89, 79, 84, 83, 48, 46, 70,
   71, 86, 54, 84, 52, 88, 89, 82, 60, 81, 58, 90, 76, 77, 80, 87, 85, 63, 67, 69, 74, 75, 66, 81, 68, 82,
   72, 73, 78, 85, 79, 87, 83, 88, 89, 90, 84, 86, 66, 64, 71, 67, 81, 72, 77, 83, 70, 78, 84, 74, 76, 89,
   80, 86, 82, 87, 85, 90, 88, 68, 70, 65, 82, 69, 83, 71, 73, 76, 86, 79, 88, 80, 75, 77, 84, 85, 81, 87,
   89, 90
}; // 1.48 kb
static const cart_index_t iCartYY_ab12_a6[784] = {
   0, 33, 34, 3, 27, 5, 29, 61, 62, 9, 10, 21, 22, 72, 73, 15, 17, 83, 51, 53, 38, 39, 66, 41, 68, 49,
   50, 63, 33, 1, 35, 28, 4, 59, 60, 7, 31, 23, 74, 11, 76, 13, 25, 16, 82, 19, 52, 37, 55, 67, 40, 47,
   48, 43, 70, 64, 34, 35, 2, 57, 58, 30, 6, 32, 8, 75, 24, 77, 12, 26, 14, 81, 18, 20, 36, 54, 56, 45,
   46, 69, 42, 71, 44, 65, 3, 28, 57, 9, 33, 38, 62, 59, 79, 15, 39, 27, 51, 78, 84, 21, 50, 86, 73, 61,
   41, 63, 72, 49, 83, 53, 68, 66, 27, 4, 58, 33, 11, 61, 80, 37, 60, 28, 78, 16, 85, 40, 52, 23, 88, 48,
   76, 43, 59, 74, 64, 55, 70, 47, 82, 67, 5, 59, 30, 38, 61, 10, 34, 78, 57, 41, 17, 53, 29, 86, 79, 49,
   22, 84, 62, 72, 39, 50, 83, 63, 73, 66, 51, 68, 29, 60, 6, 62, 80, 34, 12, 58, 36, 79, 30, 87, 18, 54,
   42, 89, 24, 46, 44, 77, 57, 56, 71, 75, 65, 81, 45, 69, 61, 7, 32, 59, 37, 78, 58, 13, 35, 55, 88, 43,
   80, 19, 31, 47, 85, 25, 60, 40, 74, 82, 48, 67, 52, 64, 76, 70, 62, 31, 8, 79, 60, 57, 36, 35, 14, 89,
   56, 80, 44, 32, 20, 87, 45, 26, 42, 58, 75, 69, 54, 81, 46, 77, 65, 71, 9, 23, 75, 15, 28, 41, 79, 55,
   89, 21, 63, 33, 73, 74, 90, 27, 68, 88, 84, 59, 49, 66, 78, 53, 86, 61, 83, 72, 10, 74, 24, 39, 78, 17,
   30, 88, 56, 63, 22, 72, 34, 90, 75, 66, 29, 89, 57, 86, 50, 51, 84, 68, 79, 83, 62, 73, 21, 11, 77, 27,
   16, 53, 87, 43, 80, 33, 72, 23, 90, 64, 76, 28, 86, 70, 85, 47, 61, 78, 67, 59, 82, 55, 88, 74, 22, 76,
   12, 51, 85, 29, 18, 80, 44, 73, 34, 90, 24, 77, 65, 84, 30, 71, 45, 87, 62, 57, 81, 79, 69, 89, 56, 75,
   72, 13, 26, 78, 40, 86, 54, 19, 32, 74, 90, 64, 77, 25, 35, 67, 87, 31, 58, 48, 88, 85, 52, 82, 60, 70,
   80, 76, 73, 25, 14, 84, 52, 79, 42, 31, 20, 90, 75, 76, 65, 35, 26, 85, 69, 32, 46, 60, 89, 81, 58, 87,
   54, 80, 71, 77, 15, 16, 81, 21, 23, 49, 89, 47, 87, 27, 66, 28, 84, 67, 85, 33, 83, 82, 90, 55, 53, 72,
   74, 61, 88, 59, 86, 78, 17, 82, 18, 50, 88, 22, 24, 85, 45, 68, 29, 86, 30, 87, 69, 83, 34, 81, 56, 90,
   51, 62, 89, 73, 75, 84, 57, 79, 83, 19, 20, 86, 48, 84, 46, 25, 26, 88, 89, 70, 71, 31, 32, 82, 81, 35,
   54, 52, 90, 87, 60, 85, 58, 76, 77, 80, 51, 52, 36, 73, 76, 62, 44, 60, 42, 84, 57, 85, 45, 58, 46, 90,
   56, 54, 65, 80, 79, 75, 77, 89, 71, 87, 69, 81, 53, 37, 54, 61, 43, 72, 77, 40, 58, 59, 86, 47, 87, 48,
   60, 55, 90, 52, 80, 64, 78, 88, 70, 74, 76, 67, 85, 82, 38, 55, 56, 41, 59, 39, 57, 74, 75, 49, 50, 61,
   62, 88, 89, 53, 51, 90, 79, 78, 63, 68, 86, 66, 84, 72, 73, 83, 39, 67, 45, 63, 74, 50, 56, 82, 69, 66,
   51, 78, 57, 85, 81, 72, 62, 87, 75, 88, 68, 73, 90, 83, 89, 86, 79, 84, 66, 40, 46, 72, 64, 83, 71, 48,
   54, 78, 84, 67, 81, 52, 58, 74, 89, 60, 77, 70, 86, 90, 76, 88, 80, 82, 87, 85, 41, 47, 69, 49, 55, 63,
   75, 67, 81, 53, 68, 59, 79, 82, 87, 61, 73, 85, 89, 74, 66, 83, 88, 72, 90, 78, 84, 86, 68, 48, 42, 83,
   70, 73, 65, 52, 46, 86, 79, 82, 69, 60, 54, 88, 75, 58, 71, 76, 84, 89, 80, 90, 77, 85, 81, 87, 49, 43,
   71, 53, 47, 66, 81, 64, 77, 61, 83, 55, 89, 70, 80, 59, 84, 76, 87, 67, 72, 86, 82, 78, 85, 74, 90, 88,
   50, 70, 44, 68, 82, 51, 45, 76, 65, 83, 62, 88, 56, 80, 71, 86, 57, 77, 69, 85, 73, 79, 87, 84, 81, 90,
   75, 89, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84, 85, 86,
   87, 88, 89, 90
}; // 1.53 kb
static const cart_index_t iCartYY_ab12_a7[756] = {
   0, 28, 30, 33, 34, 3, 57, 5, 59, 9, 10, 27, 78, 29, 79, 21, 22, 15, 84, 17, 86, 38, 61, 62, 72, 73,
   39, 51, 41, 53, 49, 50, 83, 63, 66, 68, 27, 1, 32, 4, 58, 33, 35, 61, 7, 28, 78, 11, 13, 80, 31, 16,
   85, 23, 25, 88, 19, 59, 37, 60, 40, 52, 74, 76, 55, 43, 47, 82, 48, 67, 64, 70, 29, 31, 2, 60, 6, 62,
   8, 34, 35, 79, 30, 80, 32, 12, 14, 87, 18, 89, 20, 24, 26, 57, 58, 36, 54, 42, 56, 44, 75, 77, 81, 45,
   46, 69, 71, 65, 21, 4, 54, 11, 77, 27, 58, 53, 37, 33, 72, 16, 40, 87, 60, 23, 90, 28, 52, 86, 48, 61,
   43, 80, 64, 76, 78, 85, 59, 47, 55, 88, 70, 74, 67, 82, 22, 52, 6, 76, 12, 51, 36, 29, 60, 73, 34, 85,
   58, 18, 42, 90, 24, 84, 46, 30, 54, 62, 80, 44, 77, 65, 57, 45, 79, 87, 89, 56, 71, 75, 81, 69, 3, 23,
   56, 28, 57, 9, 75, 38, 55, 15, 39, 33, 74, 62, 89, 27, 51, 21, 90, 50, 88, 41, 59, 79, 78, 84, 63, 73,
   49, 61, 53, 68, 86, 66, 72, 83, 51, 25, 8, 52, 36, 73, 14, 62, 31, 84, 57, 76, 35, 44, 20, 85, 45, 90,
   26, 56, 32, 79, 60, 42, 58, 46, 75, 65, 89, 80, 87, 69, 54, 81, 77, 71, 5, 55, 24, 59, 30, 38, 56, 10,
   74, 41, 17, 61, 88, 34, 75, 53, 29, 49, 89, 22, 90, 39, 78, 57, 86, 79, 50, 62, 63, 72, 66, 51, 84, 68,
   83, 73, 53, 7, 26, 37, 54, 61, 32, 72, 13, 59, 86, 43, 19, 77, 35, 47, 87, 55, 31, 90, 25, 78, 40, 58,
   48, 60, 88, 80, 74, 64, 67, 85, 52, 82, 70, 76, 9, 16, 69, 23, 75, 15, 81, 41, 47, 21, 63, 28, 67, 79,
   87, 33, 73, 27, 85, 68, 82, 49, 55, 89, 74, 90, 66, 84, 53, 59, 61, 83, 88, 72, 78, 86, 10, 67, 18, 74,
   24, 39, 45, 17, 82, 63, 22, 78, 85, 30, 69, 72, 34, 66, 81, 29, 87, 50, 88, 56, 90, 75, 51, 57, 68, 86,
   83, 62, 89, 73, 84, 79, 15, 11, 71, 16, 81, 21, 77, 49, 43, 27, 66, 23, 64, 89, 80, 28, 84, 33, 76, 83,
   70, 53, 47, 87, 67, 85, 72, 90, 61, 55, 59, 86, 82, 78, 74, 88, 66, 13, 20, 40, 46, 72, 26, 83, 19, 78,
   84, 64, 25, 71, 32, 67, 81, 74, 35, 89, 31, 86, 48, 54, 52, 58, 90, 77, 88, 70, 82, 87, 60, 85, 76, 80,
   17, 70, 12, 82, 18, 50, 44, 22, 76, 68, 29, 88, 80, 24, 65, 86, 30, 83, 71, 34, 77, 51, 85, 45, 87, 69,
   62, 56, 73, 90, 84, 57, 81, 79, 89, 75, 68, 19, 14, 48, 42, 83, 20, 73, 25, 86, 79, 70, 31, 65, 26, 82,
   69, 88, 32, 75, 35, 84, 52, 46, 60, 54, 89, 71, 90, 76, 85, 81, 58, 87, 80, 77, 38, 47, 45, 55, 56, 41,
   69, 39, 67, 49, 50, 59, 82, 57, 81, 61, 62, 53, 87, 51, 85, 63, 74, 75, 88, 89, 68, 79, 66, 78, 72, 73,
   90, 83, 86, 84, 49, 37, 46, 43, 71, 53, 54, 66, 40, 61, 83, 47, 48, 81, 58, 55, 89, 59, 60, 84, 52, 72,
   64, 77, 70, 80, 86, 87, 78, 67, 74, 90, 76, 88, 82, 85, 50, 48, 36, 70, 44, 68, 42, 51, 52, 83, 62, 82,
   60, 45, 46, 88, 56, 86, 54, 57, 58, 73, 76, 65, 80, 71, 79, 69, 84, 85, 90, 75, 77, 89, 87, 81, 63, 40,
   42, 64, 65, 66, 46, 68, 48, 72, 73, 67, 52, 69, 54, 74, 75, 78, 58, 79, 60, 83, 70, 71, 76, 77, 84, 81,
   86, 82, 88, 89, 80, 90, 85, 87, 39, 64, 44, 67, 45, 63, 65, 50, 70, 66, 51, 74, 76, 56, 71, 78, 57, 72,
   77, 62, 80, 68, 82, 69, 85, 81, 73, 75, 83, 88, 86, 79, 87, 84, 90, 89, 41, 43, 65, 47, 69, 49, 71, 63,
   64, 53, 68, 55, 70, 75, 77, 59, 79, 61, 80, 73, 76, 66, 67, 81, 82, 87, 83, 89, 72, 74, 78, 84, 85, 86,
   88, 90
}; // 1.48 kb
static const cart_index_t iCartYY_ab12_a8[675] = {
   0, 23, 24, 3, 28, 5, 30, 55, 56, 9, 10, 33, 34, 74, 75, 15, 27, 17, 29, 88, 89, 21, 22, 90, 57, 59,
   38, 39, 78, 41, 79, 61, 62, 51, 84, 53, 86, 49, 50, 63, 72, 73, 66, 68, 83, 21, 1, 26, 27, 4, 53, 54,
   7, 32, 33, 72, 11, 77, 13, 35, 28, 16, 86, 87, 19, 31, 23, 90, 25, 58, 37, 61, 78, 40, 59, 60, 43, 80,
   85, 52, 47, 48, 55, 88, 74, 64, 76, 67, 82, 70, 22, 25, 2, 51, 52, 29, 6, 31, 8, 73, 34, 76, 12, 35,
   14, 84, 85, 30, 18, 32, 20, 90, 24, 26, 36, 60, 62, 57, 58, 79, 42, 80, 44, 45, 46, 87, 54, 89, 56, 75,
   77, 65, 81, 69, 71, 3, 16, 45, 9, 23, 38, 56, 47, 69, 15, 39, 28, 57, 67, 81, 21, 33, 50, 62, 82, 87,
   27, 51, 85, 75, 55, 41, 63, 74, 49, 89, 59, 79, 73, 90, 61, 88, 53, 68, 66, 78, 84, 72, 83, 86, 15, 4,
   46, 21, 11, 49, 71, 37, 54, 27, 66, 16, 81, 40, 58, 33, 23, 83, 89, 48, 60, 28, 84, 52, 77, 43, 53, 72,
   64, 61, 80, 47, 87, 90, 76, 55, 70, 59, 86, 78, 67, 85, 74, 88, 82, 5, 47, 18, 38, 55, 10, 24, 67, 45,
   41, 17, 59, 30, 82, 69, 49, 61, 22, 34, 85, 81, 53, 29, 87, 56, 74, 39, 50, 88, 63, 75, 78, 57, 62, 89,
   72, 90, 66, 51, 68, 86, 79, 83, 73, 84, 17, 48, 6, 50, 70, 22, 12, 52, 36, 68, 29, 82, 18, 60, 42, 83,
   88, 34, 24, 58, 46, 86, 30, 54, 44, 76, 51, 62, 80, 73, 65, 85, 45, 56, 71, 90, 77, 84, 57, 79, 87, 69,
   89, 75, 81, 49, 7, 20, 53, 37, 66, 46, 13, 26, 61, 83, 43, 71, 19, 32, 59, 47, 84, 81, 25, 35, 55, 89,
   31, 54, 40, 72, 86, 48, 78, 58, 64, 77, 87, 60, 67, 52, 74, 90, 88, 70, 80, 82, 85, 76, 50, 19, 8, 68,
   48, 51, 36, 25, 14, 83, 62, 70, 44, 31, 20, 86, 82, 57, 45, 35, 26, 88, 56, 32, 42, 52, 73, 79, 60, 84,
   46, 76, 65, 69, 54, 85, 58, 90, 75, 89, 80, 71, 87, 81, 77, 9, 11, 65, 15, 16, 41, 69, 43, 71, 21, 63,
   23, 75, 64, 77, 27, 28, 68, 79, 70, 80, 33, 73, 76, 81, 47, 49, 66, 67, 53, 87, 55, 89, 84, 85, 59, 82,
   61, 83, 72, 74, 90, 78, 86, 88, 10, 64, 12, 39, 67, 17, 18, 70, 44, 63, 22, 74, 24, 76, 65, 66, 78, 29,
   30, 80, 71, 72, 34, 77, 45, 82, 50, 51, 85, 68, 69, 88, 56, 57, 81, 86, 87, 83, 62, 73, 90, 75, 84, 79,
   89, 63, 13, 14, 66, 40, 68, 42, 19, 20, 72, 73, 64, 65, 25, 26, 78, 67, 79, 69, 31, 32, 74, 75, 35, 46,
   48, 83, 84, 52, 86, 54, 70, 71, 81, 58, 82, 60, 88, 89, 90, 76, 77, 85, 87, 80, 39, 40, 36, 63, 64, 50,
   44, 48, 42, 66, 51, 67, 45, 52, 46, 72, 74, 62, 56, 60, 54, 78, 57, 58, 65, 70, 68, 73, 76, 83, 71, 82,
   69, 75, 77, 88, 80, 86, 79, 84, 85, 81, 90, 89, 87, 41, 37, 42, 49, 43, 63, 65, 40, 46, 53, 68, 47, 69,
   48, 54, 61, 55, 73, 75, 52, 58, 59, 79, 60, 71, 64, 66, 83, 70, 72, 77, 67, 81, 89, 80, 74, 76, 78, 84,
   86, 82, 87, 88, 90, 85, 38, 43, 44, 41, 47, 39, 45, 64, 65, 49, 50, 55, 56, 70, 71, 53, 59, 51, 57, 76,
   77, 61, 62, 80, 69, 67, 63, 68, 82, 66, 81, 74, 75, 79, 87, 78, 85, 72, 73, 83, 88, 89, 86, 84, 90
}; // 1.32 kb
static const cart_index_t iCartYY_ab12_a9[550] = {
   0, 16, 18, 23, 24, 3, 45, 5, 47, 9, 10, 28, 67, 30, 69, 33, 34, 15, 81, 17, 82, 21, 22, 27, 85, 29,
   87, 38, 55, 56, 74, 75, 39, 57, 41, 59, 49, 50, 61, 62, 88, 89, 90, 51, 53, 63, 78, 79, 72, 73, 66, 84,
   68, 86, 83, 15, 1, 20, 4, 46, 21, 26, 49, 7, 27, 66, 11, 13, 71, 32, 16, 81, 33, 35, 83, 19, 28, 84,
   23, 25, 89, 31, 53, 37, 54, 40, 58, 72, 77, 61, 43, 59, 86, 47, 87, 48, 60, 52, 90, 55, 78, 64, 80, 67,
   85, 74, 76, 88, 70, 82, 17, 19, 2, 48, 6, 50, 8, 22, 25, 68, 29, 70, 31, 12, 14, 82, 18, 83, 20, 34,
   35, 86, 30, 88, 32, 24, 26, 51, 52, 36, 60, 42, 62, 44, 73, 76, 84, 57, 85, 45, 58, 46, 54, 56, 90, 79,
   80, 65, 87, 69, 89, 71, 75, 77, 81, 9, 4, 42, 11, 65, 15, 46, 41, 37, 21, 63, 16, 40, 69, 54, 23, 75,
   27, 58, 68, 48, 33, 73, 28, 52, 79, 60, 49, 43, 71, 64, 77, 66, 81, 53, 47, 61, 83, 55, 89, 70, 80, 76,
   84, 59, 72, 67, 87, 74, 90, 78, 85, 86, 82, 88, 10, 40, 6, 64, 12, 39, 36, 17, 48, 63, 22, 67, 52, 18,
   42, 74, 24, 66, 46, 29, 60, 72, 34, 78, 58, 30, 54, 50, 70, 44, 76, 65, 51, 45, 68, 82, 83, 62, 88, 56,
   80, 71, 77, 57, 86, 73, 85, 69, 90, 75, 84, 81, 79, 87, 89, 3, 11, 44, 16, 45, 9, 65, 38, 43, 15, 39,
   23, 64, 56, 71, 28, 57, 21, 77, 50, 70, 27, 51, 33, 76, 62, 80, 41, 47, 69, 67, 81, 63, 75, 49, 55, 53,
   68, 59, 79, 82, 87, 85, 73, 61, 66, 74, 89, 78, 84, 72, 90, 83, 88, 86, 39, 13, 8, 40, 36, 63, 14, 50,
   19, 66, 51, 64, 25, 44, 20, 67, 45, 72, 26, 62, 31, 78, 57, 74, 35, 56, 32, 68, 48, 42, 52, 46, 73, 65,
   83, 70, 86, 79, 82, 69, 60, 54, 58, 75, 88, 84, 76, 71, 85, 81, 90, 77, 89, 80, 87, 5, 43, 12, 47, 18,
   38, 44, 10, 64, 41, 17, 55, 70, 24, 65, 59, 30, 49, 71, 22, 76, 53, 29, 61, 80, 34, 77, 39, 67, 45, 82,
   69, 50, 56, 63, 74, 66, 51, 78, 57, 85, 81, 87, 62, 72, 68, 88, 75, 86, 79, 83, 89, 73, 90, 84, 41, 7,
   14, 37, 42, 49, 20, 63, 13, 53, 68, 43, 19, 65, 26, 47, 69, 61, 32, 73, 25, 59, 79, 55, 31, 75, 35, 66,
   40, 46, 48, 54, 83, 71, 72, 64, 78, 84, 67, 81, 52, 58, 60, 89, 74, 86, 70, 77, 82, 87, 88, 80, 90, 76,
   85, 38, 37, 36, 43, 44, 41, 42, 39, 40, 49, 50, 47, 48, 45, 46, 55, 56, 53, 54, 51, 52, 61, 62, 59, 60,
   57, 58, 63, 64, 65, 70, 71, 68, 69, 66, 67, 72, 73, 74, 75, 76, 77, 80, 79, 78, 83, 82, 81, 88, 89, 86,
   87, 84, 85, 90
}; // 1.07 kb
static const cart_index_t iCartYY_ab12_a10[396] = {
   0, 11, 12, 3, 16, 5, 18, 43, 44, 9, 10, 23, 24, 64, 65, 15, 28, 17, 30, 70, 71, 21, 22, 33, 34, 76,
   77, 27, 29, 80, 45, 47, 38, 39, 67, 41, 69, 55, 56, 57, 81, 59, 82, 49, 50, 51, 85, 53, 87, 61, 62, 63,
   74, 75, 66, 78, 68, 79, 88, 89, 72, 73, 90, 84, 86, 83, 9, 1, 14, 15, 4, 41, 42, 7, 20, 21, 63, 11,
   65, 13, 26, 27, 16, 68, 69, 19, 32, 33, 73, 23, 75, 25, 35, 28, 79, 31, 46, 37, 49, 66, 40, 53, 54, 43,
   71, 81, 58, 47, 48, 61, 83, 84, 52, 59, 60, 55, 89, 72, 64, 77, 78, 67, 86, 87, 70, 80, 74, 90, 76, 85,
   82, 88, 10, 13, 2, 39, 40, 17, 6, 19, 8, 63, 22, 64, 12, 25, 14, 66, 67, 29, 18, 31, 20, 72, 34, 74,
   24, 35, 26, 78, 30, 32, 36, 48, 50, 51, 52, 68, 42, 70, 44, 45, 46, 82, 60, 83, 62, 57, 58, 86, 54, 88,
   56, 73, 76, 65, 84, 85, 79, 69, 80, 71, 90, 75, 77, 81, 87, 89, 3, 4, 36, 9, 11, 38, 44, 37, 42, 15,
   39, 16, 45, 40, 46, 21, 23, 50, 56, 48, 54, 27, 51, 28, 57, 52, 58, 33, 62, 60, 65, 43, 41, 63, 64, 49,
   71, 47, 69, 75, 77, 55, 70, 53, 68, 73, 76, 61, 80, 59, 79, 66, 67, 81, 72, 74, 83, 89, 82, 87, 78, 84,
   85, 90, 88, 86, 5, 37, 6, 38, 43, 10, 12, 40, 36, 41, 17, 47, 18, 48, 42, 49, 55, 22, 24, 52, 46, 53,
   29, 59, 30, 60, 54, 61, 34, 58, 44, 64, 39, 50, 70, 63, 65, 67, 45, 56, 71, 74, 76, 66, 51, 62, 80, 72,
   77, 78, 57, 68, 82, 69, 83, 88, 73, 75, 85, 81, 86, 79, 87, 89, 90, 84, 38, 7, 8, 41, 37, 39, 36, 13,
   14, 49, 50, 43, 44, 19, 20, 53, 47, 51, 45, 25, 26, 61, 62, 55, 56, 31, 32, 59, 57, 35, 42, 40, 63, 68,
   48, 66, 46, 64, 65, 69, 54, 67, 52, 72, 73, 79, 60, 78, 58, 74, 75, 83, 70, 71, 86, 82, 84, 81, 76, 77,
   88, 89, 80, 87, 85, 90
}; // 0.77 kb
static const cart_index_t iCartYY_ab12_a11[234] = {
   0, 4, 6, 11, 12, 3, 36, 5, 37, 9, 10, 16, 40, 18, 42, 23, 24, 15, 46, 17, 48, 21, 22, 28, 52, 30,
   54, 33, 34, 27, 58, 29, 60, 38, 43, 44, 64, 65, 39, 45, 41, 47, 49, 50, 55, 56, 70, 71, 76, 77, 51, 57,
   53, 59, 61, 62, 80, 63, 67, 69, 74, 75, 66, 81, 68, 82, 72, 73, 78, 85, 79, 87, 83, 88, 89, 90, 84, 86,
   3, 1, 8, 4, 36, 9, 14, 38, 7, 15, 39, 11, 13, 44, 20, 16, 45, 21, 26, 50, 19, 27, 51, 23, 25, 56,
   32, 28, 57, 33, 35, 62, 31, 41, 37, 42, 40, 46, 63, 65, 49, 43, 53, 68, 47, 69, 48, 54, 52, 58, 73, 75,
   61, 55, 59, 79, 60, 66, 64, 71, 67, 81, 72, 77, 83, 70, 78, 84, 74, 76, 89, 80, 86, 82, 87, 85, 90, 88,
   5, 7, 2, 37, 6, 38, 8, 10, 13, 41, 17, 43, 19, 12, 14, 47, 18, 49, 20, 22, 25, 53, 29, 55, 31, 24,
   26, 59, 30, 61, 32, 34, 35, 39, 40, 36, 48, 42, 50, 44, 63, 64, 66, 51, 67, 45, 52, 46, 60, 54, 62, 56,
   72, 74, 78, 57, 58, 68, 70, 65, 82, 69, 83, 71, 73, 76, 86, 79, 88, 80, 75, 77, 84, 85, 81, 87, 89, 90
}; // 0.46 kb
static const cart_index_t iCartYY_ab12_a12[91] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90
}; // 0.18 kb
static const cart_index_t iCartYY_ab13_a0[105] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104
}; // 0.21 kb
static const cart_index_t iCartYY_ab13_a1[273] = {
   0, 5, 7, 3, 1, 8, 4, 6, 2, 5, 9, 39, 11, 3, 40, 7, 39, 10, 13, 41, 4, 40, 8, 12, 41, 14,
   6, 9, 17, 46, 10, 44, 19, 15, 11, 47, 16, 45, 13, 42, 12, 20, 43, 18, 14, 17, 21, 48, 23, 15, 50, 19,
   49, 22, 25, 51, 16, 52, 20, 24, 53, 26, 18, 21, 29, 58, 22, 56, 31, 27, 23, 59, 28, 57, 25, 54, 24, 32,
   55, 30, 26, 29, 33, 60, 35, 27, 62, 31, 61, 34, 37, 63, 28, 64, 32, 36, 65, 38, 30, 33, 35, 68, 34, 67,
   37, 66, 36, 38, 45, 43, 41, 47, 40, 42, 39, 46, 44, 44, 69, 49, 70, 42, 52, 46, 48, 69, 71, 53, 43, 50,
   47, 70, 51, 71, 45, 57, 73, 51, 75, 55, 53, 59, 50, 72, 77, 52, 54, 48, 58, 74, 49, 76, 56, 56, 79, 61,
   81, 54, 64, 58, 60, 78, 83, 65, 55, 62, 59, 80, 63, 82, 57, 67, 85, 63, 87, 66, 65, 68, 62, 84, 89, 64,
   66, 60, 68, 86, 61, 88, 67, 69, 74, 76, 72, 70, 77, 73, 75, 71, 74, 78, 90, 80, 72, 91, 76, 90, 79, 82,
   92, 73, 91, 77, 81, 92, 83, 75, 78, 86, 97, 79, 95, 88, 84, 80, 98, 85, 96, 82, 93, 81, 89, 94, 87, 83,
   86, 84, 99, 88, 100, 85, 101, 89, 87, 96, 94, 92, 98, 91, 93, 90, 97, 95, 95, 102, 100, 103, 93, 101, 97, 99,
   102, 104, 101, 94, 99, 98, 103, 100, 104, 96, 102, 103, 104
}; // 0.53 kb
static const cart_index_t iCartYY_ab13_a2[468] = {
   0, 9, 10, 5, 7, 39, 11, 1, 12, 3, 40, 8, 13, 14, 2, 41, 4, 6, 15, 3, 42, 11, 47, 40, 16, 43,
   4, 45, 13, 41, 5, 17, 44, 9, 39, 46, 45, 18, 6, 43, 41, 14, 7, 46, 19, 39, 10, 44, 47, 8, 20, 40,
   42, 12, 9, 21, 69, 17, 46, 48, 10, 69, 22, 44, 19, 49, 23, 11, 70, 15, 50, 47, 70, 12, 24, 42, 52, 20,
   25, 71, 13, 51, 16, 45, 71, 26, 14, 53, 43, 18, 27, 15, 72, 23, 59, 50, 28, 73, 16, 57, 25, 51, 17, 29,
   74, 21, 48, 58, 75, 30, 18, 55, 53, 26, 19, 76, 31, 49, 22, 56, 77, 20, 32, 52, 54, 24, 21, 33, 78, 29,
   58, 60, 22, 79, 34, 56, 31, 61, 35, 23, 80, 27, 62, 59, 81, 24, 36, 54, 64, 32, 37, 82, 25, 63, 28, 57,
   83, 38, 26, 65, 55, 30, 33, 27, 84, 35, 68, 62, 34, 85, 28, 67, 37, 63, 29, 35, 86, 33, 60, 68, 87, 36,
   30, 66, 65, 38, 31, 88, 37, 61, 34, 67, 89, 32, 38, 64, 66, 36, 39, 48, 49, 46, 44, 69, 50, 40, 52, 47,
   70, 42, 51, 53, 41, 71, 45, 43, 72, 42, 54, 70, 77, 52, 73, 55, 43, 75, 71, 53, 44, 74, 56, 69, 49, 76,
   57, 75, 45, 73, 51, 71, 46, 58, 76, 48, 69, 74, 59, 47, 77, 50, 72, 70, 48, 60, 90, 58, 74, 78, 49, 90,
   61, 76, 56, 79, 62, 50, 91, 59, 80, 72, 63, 92, 51, 82, 57, 73, 91, 52, 64, 77, 81, 54, 92, 65, 53, 83,
   75, 55, 93, 54, 66, 81, 89, 64, 94, 66, 55, 87, 83, 65, 56, 95, 67, 79, 61, 88, 67, 96, 57, 85, 63, 82,
   58, 68, 97, 60, 78, 86, 68, 59, 98, 62, 84, 80, 60, 62, 99, 68, 86, 84, 61, 100, 63, 88, 67, 85, 101, 64,
   65, 89, 87, 66, 69, 78, 79, 74, 76, 90, 80, 70, 81, 72, 91, 77, 82, 83, 71, 92, 73, 75, 84, 72, 93, 80,
   98, 91, 85, 94, 73, 96, 82, 92, 74, 86, 95, 78, 90, 97, 96, 87, 75, 94, 92, 83, 76, 97, 88, 90, 79, 95,
   98, 77, 89, 91, 93, 81, 78, 84, 102, 86, 97, 99, 79, 102, 85, 95, 88, 100, 86, 80, 103, 84, 99, 98, 103, 81,
   87, 93, 101, 89, 88, 104, 82, 100, 85, 96, 104, 89, 83, 101, 94, 87, 90, 99, 100, 97, 95, 102, 99, 91, 101, 98,
   103, 93, 100, 101, 92, 104, 96, 94, 102, 93, 94, 103, 104, 101, 95, 103, 96, 102, 100, 104, 97, 98, 104, 99, 102, 103
}; // 0.91 kb
static const cart_index_t iCartYY_ab13_a3[660] = {
   0, 17, 19, 9, 10, 5, 44, 7, 46, 39, 15, 1, 20, 3, 42, 11, 12, 47, 8, 40, 16, 18, 2, 43, 4, 45,
   6, 13, 14, 41, 5, 21, 49, 17, 44, 9, 69, 39, 48, 46, 23, 3, 52, 11, 70, 15, 42, 50, 40, 47, 7, 48,
   22, 46, 19, 39, 49, 10, 69, 44, 25, 53, 4, 71, 13, 51, 41, 16, 43, 45, 50, 8, 24, 40, 52, 47, 20, 70,
   12, 42, 51, 26, 6, 53, 41, 71, 14, 45, 18, 43, 9, 29, 76, 21, 69, 17, 74, 46, 58, 48, 10, 74, 31, 69,
   22, 44, 56, 19, 76, 49, 27, 11, 77, 15, 72, 23, 70, 59, 47, 50, 28, 75, 13, 73, 16, 57, 45, 25, 71, 51,
   72, 12, 32, 42, 54, 70, 24, 77, 20, 52, 73, 30, 14, 55, 43, 75, 18, 71, 26, 53, 17, 33, 90, 29, 74, 21,
   78, 48, 60, 58, 35, 15, 91, 23, 80, 27, 72, 62, 50, 59, 19, 90, 34, 76, 31, 49, 61, 22, 79, 56, 37, 92,
   16, 82, 25, 63, 51, 28, 73, 57, 91, 20, 36, 52, 64, 77, 32, 81, 24, 54, 92, 38, 18, 65, 53, 83, 26, 75,
   30, 55, 21, 35, 97, 33, 78, 29, 86, 58, 68, 60, 22, 95, 37, 79, 34, 56, 67, 31, 88, 61, 33, 23, 98, 27,
   84, 35, 80, 68, 59, 62, 34, 96, 25, 85, 28, 67, 57, 37, 82, 63, 93, 24, 38, 54, 66, 81, 36, 89, 32, 64,
   94, 36, 26, 66, 55, 87, 30, 83, 38, 65, 29, 27, 99, 35, 86, 33, 84, 60, 62, 68, 31, 100, 28, 88, 37, 61,
   63, 34, 85, 67, 101, 32, 30, 64, 65, 89, 38, 87, 36, 66, 57, 55, 41, 75, 45, 73, 43, 51, 53, 71, 59, 40,
   54, 47, 77, 50, 52, 72, 42, 70, 39, 58, 56, 48, 49, 46, 76, 44, 74, 69, 44, 78, 61, 74, 56, 69, 79, 49,
   90, 76, 80, 42, 64, 70, 81, 72, 54, 91, 52, 77, 46, 60, 79, 58, 76, 48, 90, 69, 78, 74, 82, 65, 43, 83,
   71, 92, 53, 73, 55, 75, 62, 47, 81, 50, 91, 59, 77, 80, 70, 72, 63, 83, 45, 92, 51, 82, 71, 57, 75, 73,
   67, 94, 51, 96, 57, 85, 73, 63, 92, 82, 96, 66, 53, 87, 75, 94, 55, 92, 65, 83, 68, 50, 93, 59, 98, 62,
   91, 84, 72, 80, 98, 52, 66, 77, 89, 91, 64, 93, 54, 81, 48, 68, 95, 60, 90, 58, 97, 74, 86, 78, 49, 97,
   67, 90, 61, 76, 88, 56, 95, 79, 56, 102, 63, 95, 67, 79, 85, 61, 100, 88, 103, 54, 65, 81, 87, 93, 66, 101,
   64, 89, 58, 62, 102, 68, 97, 60, 99, 78, 84, 86, 104, 64, 55, 89, 83, 101, 65, 94, 66, 87, 60, 59, 103, 62,
   99, 68, 98, 86, 80, 84, 61, 104, 57, 100, 63, 88, 82, 67, 96, 85, 69, 86, 88, 78, 79, 74, 95, 76, 97, 90,
   84, 70, 89, 72, 93, 80, 81, 98, 77, 91, 85, 87, 71, 94, 73, 96, 75, 82, 83, 92, 74, 84, 100, 86, 95, 78,
   102, 90, 99, 97, 86, 72, 101, 80, 103, 84, 93, 99, 91, 98, 76, 99, 85, 97, 88, 90, 100, 79, 102, 95, 88, 101,
   73, 104, 82, 100, 92, 85, 94, 96, 99, 77, 87, 91, 101, 98, 89, 103, 81, 93, 100, 89, 75, 101, 92, 104, 83, 96,
   87, 94, 78, 80, 104, 84, 102, 86, 103, 97, 98, 99, 79, 103, 82, 102, 85, 95, 96, 88, 104, 100, 102, 81, 83, 93,
   94, 103, 87, 104, 89, 101, 95, 93, 92, 103, 96, 102, 94, 100, 101, 104, 97, 91, 94, 98, 104, 99, 101, 102, 93, 103,
   90, 98, 96, 99, 100, 97, 104, 95, 103, 102
}; // 1.29 kb
static const cart_index_t iCartYY_ab13_a4[825] = {
   0, 21, 22, 5, 17, 7, 19, 48, 49, 9, 10, 69, 44, 46, 39, 23, 1, 24, 15, 3, 50, 52, 8, 20, 11, 70,
   12, 42, 40, 47, 25, 26, 2, 51, 53, 16, 4, 18, 6, 71, 13, 14, 41, 43, 45, 27, 3, 54, 23, 11, 59, 77,
   40, 52, 15, 72, 42, 70, 47, 50, 28, 55, 4, 57, 75, 25, 13, 53, 41, 73, 16, 43, 45, 71, 51, 5, 29, 56,
   9, 21, 39, 49, 58, 76, 17, 44, 74, 69, 48, 46, 57, 30, 6, 73, 55, 51, 41, 26, 14, 75, 45, 18, 43, 53,
   71, 7, 58, 31, 39, 48, 10, 22, 74, 56, 46, 19, 76, 49, 69, 44, 59, 8, 32, 50, 40, 72, 54, 12, 24, 47,
   77, 20, 52, 42, 70, 9, 33, 79, 17, 29, 46, 76, 60, 90, 21, 69, 78, 74, 58, 48, 10, 78, 34, 44, 74, 19,
   31, 90, 61, 69, 22, 79, 56, 76, 49, 35, 11, 81, 27, 15, 62, 91, 47, 77, 23, 80, 70, 72, 50, 59, 80, 12,
   36, 72, 42, 91, 64, 20, 32, 70, 81, 24, 54, 52, 77, 37, 83, 13, 63, 92, 28, 16, 75, 45, 82, 25, 71, 51,
   73, 57, 82, 38, 14, 92, 65, 73, 43, 30, 18, 83, 71, 26, 53, 55, 75, 33, 15, 93, 35, 23, 68, 98, 50, 91,
   27, 84, 72, 80, 59, 62, 34, 94, 16, 67, 96, 37, 25, 92, 51, 85, 28, 73, 57, 82, 63, 17, 35, 95, 21, 33,
   48, 90, 68, 97, 29, 74, 86, 78, 60, 58, 96, 36, 18, 94, 66, 92, 53, 38, 26, 87, 75, 30, 55, 65, 83, 19,
   97, 37, 49, 90, 22, 34, 95, 67, 76, 31, 88, 61, 79, 56, 98, 20, 38, 91, 52, 93, 66, 24, 36, 77, 89, 32,
   64, 54, 81, 21, 27, 102, 29, 35, 58, 97, 62, 99, 33, 78, 84, 86, 68, 60, 22, 102, 28, 56, 95, 31, 37, 100,
   63, 79, 34, 85, 67, 88, 61, 29, 23, 103, 33, 27, 60, 99, 59, 98, 35, 86, 80, 84, 62, 68, 103, 24, 30, 93,
   54, 101, 65, 32, 38, 81, 87, 36, 66, 64, 89, 31, 104, 25, 61, 100, 34, 28, 96, 57, 88, 37, 82, 63, 85, 67,
   104, 32, 26, 101, 64, 94, 55, 36, 30, 89, 83, 38, 65, 66, 87, 39, 60, 61, 46, 58, 44, 56, 78, 79, 48, 49,
   90, 76, 74, 69, 62, 40, 64, 59, 47, 80, 81, 42, 54, 50, 91, 52, 77, 70, 72, 63, 65, 41, 82, 83, 57, 45,
   55, 43, 92, 51, 53, 71, 75, 73, 84, 42, 66, 80, 70, 98, 89, 52, 64, 72, 93, 54, 81, 77, 91, 85, 66, 43,
   96, 87, 82, 71, 65, 53, 94, 73, 55, 75, 83, 92, 44, 86, 67, 69, 78, 49, 61, 97, 88, 74, 56, 95, 79, 90,
   76, 67, 87, 45, 85, 94, 63, 51, 83, 71, 96, 57, 75, 73, 92, 82, 46, 68, 88, 48, 60, 69, 79, 86, 95, 58,
   76, 97, 90, 78, 74, 68, 47, 89, 62, 50, 84, 93, 70, 81, 59, 98, 77, 91, 72, 80, 48, 62, 100, 58, 68, 74,
   95, 84, 102, 60, 90, 99, 97, 86, 78, 49, 99, 63, 76, 97, 56, 67, 102, 85, 90, 61, 100, 88, 95, 79, 60, 50,
   101, 68, 59, 86, 103, 72, 93, 62, 99, 91, 98, 80, 84, 61, 101, 51, 88, 104, 67, 57, 94, 73, 100, 63, 92, 82,
   96, 85, 99, 52, 65, 98, 77, 103, 87, 54, 66, 91, 101, 64, 89, 81, 93, 100, 64, 53, 104, 89, 96, 75, 66, 55,
   101, 92, 65, 83, 87, 94, 102, 54, 55, 103, 81, 104, 83, 64, 65, 93, 94, 66, 87, 89, 101, 56, 103, 57, 79, 102,
   61, 63, 104, 82, 95, 67, 96, 85, 100, 88, 58, 59, 104, 60, 62, 78, 102, 80, 103, 68, 97, 98, 99, 84, 86, 69,
   84, 85, 74, 86, 76, 88, 99, 100, 78, 79, 102, 95, 97, 90, 86, 70, 87, 84, 72, 99, 101, 77, 89, 80, 103, 81,
   93, 91, 98, 88, 89, 71, 100, 101, 85, 73, 87, 75, 104, 82, 83, 92, 94, 96, 78, 72, 94, 86, 80, 97, 104, 91,
   101, 84, 102, 93, 103, 98, 99, 79, 93, 73, 95, 103, 88, 82, 101, 92, 102, 85, 94, 96, 104, 100, 74, 80, 96, 78,
   84, 90, 100, 98, 104, 86, 95, 103, 102, 99, 97, 95, 81, 75, 102, 93, 100, 92, 89, 83, 103, 96, 87, 94, 101, 104,
   76, 98, 82, 90, 99, 79, 85, 103, 96, 97, 88, 104, 100, 102, 95, 97, 77, 83, 99, 91, 102, 94, 81, 87, 98, 104,
   89, 101, 93, 103, 90, 91, 92, 97, 98, 95, 96, 93, 94, 99, 100, 101, 104, 103, 102
}; // 1.61 kb
static const cart_index_t iCartYY_ab13_a5[945] = {
   0, 29, 31, 21, 22, 5, 56, 7, 58, 9, 10, 17, 74, 19, 76, 39, 48, 49, 69, 44, 46, 27, 1, 32, 3, 54,
   23, 24, 59, 8, 15, 72, 11, 12, 77, 20, 50, 40, 52, 42, 70, 47, 28, 30, 2, 55, 4, 57, 6, 25, 26, 73,
   16, 75, 18, 13, 14, 51, 53, 41, 43, 45, 71, 5, 33, 61, 29, 56, 9, 79, 39, 60, 17, 44, 21, 78, 49, 90,
   46, 58, 76, 74, 69, 48, 35, 3, 64, 11, 81, 27, 54, 62, 40, 23, 80, 15, 42, 91, 52, 59, 47, 77, 70, 72,
   50, 7, 60, 34, 58, 31, 39, 61, 10, 78, 46, 19, 48, 90, 22, 79, 44, 74, 56, 76, 49, 69, 37, 65, 4, 83,
   13, 63, 41, 28, 55, 82, 25, 92, 53, 16, 43, 57, 75, 45, 71, 51, 73, 62, 8, 36, 40, 64, 59, 32, 80, 12,
   50, 91, 47, 20, 81, 24, 72, 42, 54, 52, 77, 70, 63, 38, 6, 65, 41, 82, 14, 57, 30, 92, 51, 83, 26, 45,
   18, 73, 55, 43, 53, 71, 75, 9, 35, 88, 33, 79, 17, 95, 46, 68, 21, 69, 29, 86, 76, 97, 48, 60, 90, 78,
   74, 58, 10, 86, 37, 78, 34, 44, 67, 19, 97, 69, 22, 74, 95, 31, 88, 49, 90, 61, 79, 56, 76, 33, 11, 89,
   15, 93, 35, 81, 68, 47, 27, 84, 23, 70, 98, 77, 62, 50, 91, 72, 80, 59, 34, 87, 13, 94, 16, 67, 45, 37,
   83, 85, 28, 96, 75, 25, 71, 63, 92, 51, 73, 57, 82, 84, 12, 38, 42, 66, 80, 36, 98, 20, 72, 93, 70, 24,
   89, 32, 91, 52, 64, 54, 81, 77, 85, 36, 14, 66, 43, 96, 18, 82, 38, 94, 73, 87, 30, 71, 26, 92, 65, 53,
   55, 75, 83, 17, 27, 100, 35, 95, 21, 102, 48, 62, 29, 74, 33, 84, 90, 99, 58, 68, 97, 86, 78, 60, 29, 15,
   101, 23, 103, 33, 93, 60, 50, 35, 86, 27, 72, 99, 91, 68, 59, 98, 80, 84, 62, 19, 99, 28, 97, 37, 49, 63,
   22, 102, 76, 31, 90, 100, 34, 85, 56, 95, 67, 88, 61, 79, 31, 101, 16, 104, 25, 61, 51, 34, 94, 88, 37, 100,
   92, 28, 73, 67, 96, 57, 82, 63, 85, 99, 20, 30, 52, 65, 98, 38, 103, 24, 91, 101, 77, 32, 87, 36, 93, 54,
   66, 64, 89, 81, 100, 32, 18, 64, 53, 104, 26, 96, 36, 101, 92, 89, 38, 75, 30, 94, 66, 55, 65, 83, 87, 21,
   23, 104, 27, 102, 29, 103, 58, 59, 33, 78, 35, 80, 97, 98, 60, 62, 99, 84, 86, 68, 22, 103, 25, 102, 28, 56,
   57, 31, 104, 79, 34, 95, 96, 37, 82, 61, 100, 63, 85, 67, 88, 102, 24, 26, 54, 55, 103, 30, 104, 32, 93, 94,
   81, 36, 83, 38, 101, 64, 65, 66, 87, 89, 67, 66, 41, 87, 45, 85, 43, 63, 65, 96, 57, 94, 55, 51, 53, 82,
   83, 71, 75, 73, 92, 68, 40, 66, 47, 89, 62, 64, 84, 42, 59, 98, 50, 52, 93, 54, 80, 70, 81, 77, 91, 72,
   39, 68, 67, 60, 61, 46, 88, 44, 86, 48, 49, 58, 97, 56, 95, 69, 78, 79, 90, 76, 74, 44, 84, 63, 86, 67,
   69, 85, 49, 99, 74, 56, 78, 102, 61, 100, 76, 97, 88, 95, 79, 90, 86, 42, 65, 70, 87, 84, 66, 99, 52, 80,
   103, 72, 54, 101, 64, 98, 77, 89, 81, 93, 91, 46, 62, 85, 68, 88, 48, 100, 69, 84, 58, 76, 60, 99, 79, 102,
   74, 86, 95, 97, 90, 78, 88, 64, 43, 89, 71, 100, 53, 85, 66, 104, 82, 101, 65, 73, 55, 96, 87, 75, 83, 92,
   94, 60, 47, 87, 50, 101, 68, 89, 86, 70, 62, 99, 59, 77, 103, 81, 84, 72, 93, 91, 98, 80, 61, 89, 45, 101,
   51, 88, 71, 67, 87, 100, 63, 104, 83, 57, 75, 85, 94, 73, 92, 82, 96, 56, 93, 51, 103, 57, 79, 73, 61, 101,
   95, 67, 102, 94, 63, 92, 88, 104, 82, 96, 85, 100, 95, 54, 53, 81, 75, 102, 55, 100, 64, 103, 96, 93, 66, 92,
   65, 104, 89, 83, 87, 94, 101, 58, 50, 94, 59, 104, 60, 101, 78, 72, 68, 97, 62, 91, 102, 93, 86, 80, 103, 98,
   99, 84, 97, 52, 55, 77, 83, 99, 65, 102, 54, 98, 104, 91, 64, 94, 66, 103, 81, 87, 89, 101, 93, 48, 59, 96,
   62, 100, 58, 104, 74, 80, 60, 90, 68, 98, 95, 103, 78, 84, 102, 99, 97, 86, 49, 98, 57, 99, 63, 76, 82, 56,
   103, 90, 61, 97, 104, 67, 96, 79, 102, 85, 100, 88, 95, 69, 80, 82, 84, 85, 74, 96, 76, 98, 78, 79, 86, 103,
   88, 104, 90, 99, 100, 102, 95, 97, 78, 70, 83, 72, 94, 86, 87, 97, 77, 84, 102, 80, 81, 104, 89, 99, 91, 101,
   93, 103, 98, 79, 81, 71, 93, 73, 95, 75, 88, 89, 102, 85, 103, 87, 82, 83, 100, 101, 92, 94, 96, 104, 74, 72,
   92, 80, 96, 78, 94, 90, 91, 86, 95, 84, 93, 100, 101, 97, 98, 104, 103, 102, 99, 76, 91, 73, 98, 82, 90, 92,
   79, 93, 97, 88, 99, 101, 85, 94, 95, 103, 96, 104, 100, 102, 90, 77, 75, 91, 92, 97, 83, 95, 81, 99, 100, 98,
   89, 96, 87, 102, 93, 94, 101, 104, 103
}; // 1.85 kb
static const cart_index_t iCartYY_ab13_a6[1008] = {
   0, 33, 34, 5, 29, 7, 31, 60, 61, 9, 10, 21, 22, 78, 79, 17, 19, 90, 56, 58, 39, 44, 74, 46, 76, 48,
   49, 69, 35, 1, 36, 27, 3, 62, 64, 8, 32, 23, 80, 11, 81, 12, 24, 15, 91, 20, 54, 40, 59, 72, 42, 50,
   52, 47, 77, 70, 37, 38, 2, 63, 65, 28, 4, 30, 6, 82, 25, 83, 13, 26, 14, 92, 16, 18, 41, 55, 57, 51,
   53, 73, 43, 75, 45, 71, 33, 3, 66, 35, 11, 68, 89, 40, 64, 27, 84, 15, 93, 42, 54, 23, 98, 52, 81, 47,
   62, 80, 70, 59, 77, 50, 91, 72, 34, 66, 4, 67, 87, 37, 13, 65, 41, 85, 28, 94, 16, 55, 43, 96, 25, 53,
   45, 83, 63, 57, 75, 82, 71, 92, 51, 73, 5, 35, 67, 9, 33, 39, 61, 68, 88, 17, 44, 29, 56, 86, 95, 21,
   49, 97, 79, 60, 46, 69, 78, 48, 90, 58, 76, 74, 67, 36, 6, 85, 66, 63, 41, 38, 14, 96, 57, 87, 45, 30,
   18, 94, 51, 26, 43, 65, 82, 73, 55, 92, 53, 83, 71, 75, 7, 68, 37, 39, 60, 10, 34, 86, 67, 46, 19, 58,
   31, 97, 88, 48, 22, 95, 61, 78, 44, 49, 90, 69, 79, 74, 56, 76, 68, 8, 38, 62, 40, 84, 66, 12, 36, 59,
   98, 47, 89, 20, 32, 50, 93, 24, 64, 42, 80, 91, 52, 72, 54, 70, 81, 77, 9, 27, 85, 17, 35, 46, 88, 62,
   100, 21, 69, 33, 79, 84, 102, 29, 76, 99, 95, 68, 48, 74, 86, 58, 97, 60, 90, 78, 10, 84, 28, 44, 86, 19,
   37, 99, 63, 69, 22, 78, 34, 102, 85, 74, 31, 100, 67, 97, 49, 56, 95, 76, 88, 90, 61, 79, 29, 11, 87, 33,
   15, 60, 101, 47, 89, 35, 86, 23, 103, 70, 81, 27, 99, 77, 93, 50, 68, 84, 72, 62, 91, 59, 98, 80, 86, 12,
   30, 84, 42, 99, 65, 20, 38, 80, 103, 70, 87, 24, 36, 72, 101, 32, 66, 52, 98, 93, 54, 91, 64, 77, 89, 81,
   31, 89, 13, 61, 101, 34, 16, 87, 45, 88, 37, 104, 25, 83, 71, 100, 28, 75, 51, 94, 67, 63, 92, 85, 73, 96,
   57, 82, 88, 32, 14, 100, 64, 85, 43, 36, 18, 104, 82, 89, 71, 38, 26, 101, 73, 30, 53, 66, 96, 92, 65, 94,
   55, 87, 75, 83, 21, 15, 94, 29, 23, 58, 104, 50, 101, 33, 78, 27, 102, 72, 93, 35, 97, 91, 103, 59, 60, 86,
   80, 68, 98, 62, 99, 84, 22, 93, 16, 56, 103, 31, 25, 101, 51, 79, 34, 102, 28, 94, 73, 95, 37, 92, 57, 104,
   61, 67, 96, 88, 82, 100, 63, 85, 17, 23, 96, 21, 27, 48, 100, 59, 104, 29, 74, 35, 95, 80, 103, 33, 90, 98,
   102, 62, 58, 78, 84, 60, 99, 68, 97, 86, 95, 24, 18, 102, 54, 100, 53, 32, 26, 103, 96, 81, 75, 36, 30, 93,
   92, 38, 55, 64, 104, 94, 66, 101, 65, 89, 83, 87, 19, 98, 25, 49, 99, 22, 28, 103, 57, 76, 31, 97, 37, 104,
   82, 90, 34, 96, 63, 102, 56, 61, 100, 79, 85, 95, 67, 88, 97, 20, 26, 99, 52, 102, 55, 24, 30, 98, 104, 77,
   83, 32, 38, 91, 94, 36, 65, 54, 103, 101, 64, 93, 66, 81, 87, 89, 39, 62, 63, 46, 68, 44, 67, 84, 85, 48,
   49, 60, 61, 99, 100, 58, 56, 102, 88, 86, 69, 76, 97, 74, 95, 78, 79, 90, 60, 40, 65, 68, 47, 86, 87, 42,
   66, 62, 99, 50, 101, 52, 64, 59, 103, 54, 89, 70, 84, 98, 77, 80, 81, 72, 93, 91, 61, 64, 41, 88, 89, 67,
   45, 66, 43, 100, 63, 101, 51, 65, 53, 104, 57, 55, 71, 87, 85, 82, 83, 96, 75, 94, 73, 92, 78, 42, 55, 86,
   70, 97, 83, 52, 65, 84, 102, 72, 94, 54, 66, 80, 104, 64, 87, 77, 99, 103, 81, 98, 89, 91, 101, 93, 79, 54,
   43, 95, 81, 88, 71, 64, 53, 102, 85, 93, 73, 66, 55, 103, 82, 65, 75, 89, 100, 96, 87, 104, 83, 101, 92, 94,
   44, 80, 57, 69, 84, 49, 63, 98, 82, 74, 56, 86, 67, 103, 96, 78, 61, 104, 85, 99, 76, 79, 102, 90, 100, 97,
   88, 95, 56, 81, 45, 79, 93, 61, 51, 89, 71, 95, 67, 103, 57, 87, 75, 102, 63, 83, 73, 101, 88, 85, 94, 100,
   92, 104, 82, 96, 46, 59, 82, 48, 62, 69, 85, 80, 96, 58, 76, 68, 88, 98, 104, 60, 79, 103, 100, 84, 74, 90,
   99, 78, 102, 86, 95, 97, 58, 47, 83, 60, 50, 78, 94, 70, 87, 68, 97, 59, 104, 77, 89, 62, 102, 81, 101, 72,
   86, 99, 91, 84, 93, 80, 103, 98, 48, 50, 92, 58, 59, 74, 96, 72, 94, 60, 90, 62, 100, 91, 101, 68, 95, 93,
   104, 80, 78, 97, 98, 86, 103, 84, 102, 99, 49, 91, 51, 76, 98, 56, 57, 93, 73, 90, 61, 99, 63, 101, 92, 97,
   67, 94, 82, 103, 79, 88, 104, 95, 96, 102, 85, 100, 90, 52, 53, 97, 77, 95, 75, 54, 55, 99, 100, 91, 92, 64,
   65, 98, 96, 66, 83, 81, 102, 104, 89, 103, 87, 93, 94, 101, 69, 72, 73, 74, 80, 76, 82, 91, 92, 78, 79, 84,
   85, 93, 94, 86, 88, 101, 96, 98, 90, 95, 103, 97, 104, 99, 100, 102, 74, 70, 75, 78, 72, 90, 92, 77, 83, 86,
   95, 80, 96, 81, 87, 84, 100, 89, 94, 91, 97, 102, 93, 99, 101, 98, 104, 103, 76, 77, 71, 90, 91, 79, 73, 81,
   75, 97, 88, 98, 82, 89, 83, 99, 85, 87, 92, 93, 95, 100, 101, 102, 94, 103, 96, 104
}; // 1.97 kb
static const cart_index_t iCartYY_ab13_a7[1008] = {
   0, 35, 37, 33, 34, 5, 67, 7, 68, 9, 10, 29, 86, 31, 88, 21, 22, 17, 95, 19, 97, 39, 60, 61, 78, 79,
   44, 56, 46, 58, 48, 49, 90, 69, 74, 76, 33, 1, 38, 3, 66, 35, 36, 68, 8, 27, 84, 11, 12, 89, 32, 15,
   93, 23, 24, 98, 20, 62, 40, 64, 42, 54, 80, 81, 59, 47, 50, 91, 52, 72, 70, 77, 34, 36, 2, 66, 4, 67,
   6, 37, 38, 85, 28, 87, 30, 13, 14, 94, 16, 96, 18, 25, 26, 63, 65, 41, 55, 43, 57, 45, 82, 83, 92, 51,
   53, 73, 75, 71, 5, 27, 63, 35, 67, 9, 85, 39, 62, 17, 44, 33, 84, 61, 100, 29, 56, 21, 102, 49, 99, 46,
   68, 88, 86, 95, 69, 79, 48, 60, 58, 76, 97, 74, 78, 90, 29, 3, 65, 11, 87, 33, 66, 60, 40, 35, 86, 15,
   42, 101, 64, 23, 103, 27, 54, 99, 52, 68, 47, 89, 70, 81, 84, 93, 62, 50, 59, 98, 77, 80, 72, 91, 7, 62,
   28, 68, 37, 39, 63, 10, 84, 46, 19, 60, 99, 34, 85, 58, 31, 48, 100, 22, 102, 44, 86, 67, 97, 88, 49, 61,
   69, 78, 74, 56, 95, 76, 90, 79, 31, 64, 4, 89, 13, 61, 41, 34, 66, 88, 37, 101, 65, 16, 43, 104, 25, 100,
   53, 28, 55, 67, 87, 45, 83, 71, 63, 51, 85, 94, 96, 57, 75, 82, 92, 73, 60, 8, 30, 40, 65, 68, 38, 86,
   12, 62, 99, 47, 20, 87, 36, 50, 101, 59, 32, 103, 24, 84, 42, 66, 52, 64, 98, 89, 80, 70, 72, 93, 54, 91,
   77, 81, 61, 32, 6, 64, 41, 88, 14, 67, 36, 100, 63, 89, 38, 45, 18, 101, 51, 104, 26, 57, 30, 85, 66, 43,
   65, 53, 82, 71, 96, 87, 94, 73, 55, 92, 83, 75, 9, 23, 82, 27, 85, 17, 96, 46, 59, 21, 69, 35, 80, 88,
   104, 33, 79, 29, 103, 76, 98, 48, 62, 100, 84, 102, 74, 95, 58, 68, 60, 90, 99, 78, 86, 97, 10, 80, 25, 84,
   28, 44, 57, 19, 98, 69, 22, 86, 103, 37, 82, 78, 34, 74, 96, 31, 104, 49, 99, 63, 102, 85, 56, 67, 76, 97,
   90, 61, 100, 79, 95, 88, 21, 11, 83, 15, 94, 29, 87, 58, 47, 33, 78, 23, 70, 104, 89, 27, 102, 35, 81, 97,
   77, 60, 50, 101, 72, 93, 86, 103, 68, 59, 62, 99, 91, 84, 80, 98, 22, 81, 13, 93, 16, 56, 45, 31, 89, 79,
   34, 103, 87, 25, 71, 102, 28, 95, 75, 37, 83, 61, 101, 51, 94, 73, 67, 57, 88, 104, 100, 63, 92, 85, 96, 82,
   78, 12, 26, 42, 55, 86, 30, 97, 20, 84, 102, 70, 24, 83, 38, 72, 94, 80, 36, 104, 32, 99, 52, 65, 54, 66,
   103, 87, 98, 77, 91, 101, 64, 93, 81, 89, 79, 24, 14, 54, 43, 95, 18, 88, 32, 102, 85, 81, 36, 71, 26, 93,
   73, 103, 30, 82, 38, 100, 64, 53, 66, 55, 96, 75, 104, 89, 101, 92, 65, 94, 87, 83, 17, 15, 92, 23, 96, 21,
   94, 48, 50, 29, 74, 27, 72, 100, 101, 35, 95, 33, 93, 90, 91, 58, 59, 104, 80, 103, 78, 102, 60, 62, 68, 97,
   98, 86, 84, 99, 19, 91, 16, 98, 25, 49, 51, 22, 93, 76, 31, 99, 101, 28, 73, 97, 37, 90, 92, 34, 94, 56,
   103, 57, 104, 82, 61, 63, 79, 102, 95, 67, 96, 88, 100, 85, 90, 20, 18, 52, 53, 97, 26, 95, 24, 99, 100, 77,
   32, 75, 30, 91, 92, 98, 38, 96, 36, 102, 54, 55, 64, 65, 104, 83, 103, 81, 93, 94, 66, 101, 89, 87, 56, 54,
   41, 81, 45, 79, 43, 61, 64, 95, 67, 93, 66, 51, 53, 103, 57, 102, 55, 63, 65, 88, 89, 71, 87, 75, 85, 73,
   100, 101, 104, 82, 83, 96, 94, 92, 58, 40, 55, 47, 83, 60, 65, 78, 42, 68, 97, 50, 52, 94, 66, 59, 104, 62,
   64, 102, 54, 86, 70, 87, 77, 89, 99, 101, 84, 72, 80, 103, 81, 98, 91, 93, 39, 59, 57, 62, 63, 46, 82, 44,
   80, 48, 49, 68, 98, 67, 96, 60, 61, 58, 104, 56, 103, 69, 84, 85, 99, 100, 76, 88, 74, 86, 78, 79, 102, 90,
   97, 95, 44, 72, 51, 80, 57, 69, 73, 49, 91, 74, 56, 84, 93, 63, 92, 86, 67, 78, 94, 61, 101, 76, 98, 82,
   103, 96, 79, 85, 90, 99, 97, 88, 104, 95, 102, 100, 74, 42, 53, 70, 75, 78, 55, 90, 52, 86, 95, 72, 54, 92,
   65, 80, 96, 84, 66, 100, 64, 97, 77, 83, 81, 87, 102, 94, 99, 91, 98, 104, 89, 103, 93, 101, 46, 50, 73, 59,
   82, 48, 92, 69, 72, 58, 76, 62, 91, 85, 94, 68, 88, 60, 101, 79, 93, 74, 80, 96, 98, 104, 90, 100, 78, 84,
   86, 95, 103, 97, 99, 102, 76, 52, 43, 77, 71, 90, 53, 79, 54, 97, 88, 91, 64, 73, 55, 98, 82, 99, 65, 85,
   66, 95, 81, 75, 89, 83, 100, 92, 102, 93, 103, 96, 87, 104, 101, 94, 48, 47, 75, 50, 92, 58, 83, 74, 70, 60,
   90, 59, 77, 96, 87, 62, 100, 68, 89, 95, 81, 78, 72, 94, 91, 101, 97, 104, 86, 80, 84, 102, 93, 99, 98, 103,
   49, 77, 45, 91, 51, 76, 71, 56, 81, 90, 61, 98, 89, 57, 75, 99, 63, 97, 83, 67, 87, 79, 93, 73, 101, 92,
   88, 82, 95, 103, 102, 85, 94, 100, 104, 96, 69, 70, 71, 72, 73, 74, 75, 76, 77, 78, 79, 80, 81, 82, 83, 84,
   85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104
}; // 1.97 kb
static const cart_index_t iCartYY_ab13_a8[945] = {
   0, 27, 28, 5, 35, 7, 37, 62, 63, 9, 10, 33, 34, 84, 85, 17, 29, 19, 31, 99, 100, 21, 22, 102, 67, 68,
   39, 44, 86, 46, 88, 60, 61, 56, 95, 58, 97, 48, 49, 69, 78, 79, 74, 76, 90, 29, 1, 30, 33, 3, 60, 65,
   8, 38, 35, 86, 11, 87, 12, 36, 27, 15, 99, 101, 20, 32, 23, 103, 24, 66, 40, 68, 84, 42, 62, 64, 47, 89,
   93, 54, 50, 52, 59, 98, 80, 70, 81, 72, 91, 77, 31, 32, 2, 61, 64, 34, 4, 36, 6, 88, 37, 89, 13, 38,
   14, 100, 101, 28, 16, 30, 18, 104, 25, 26, 41, 66, 67, 63, 65, 85, 43, 87, 45, 51, 53, 94, 55, 96, 57, 82,
   83, 71, 92, 73, 75, 21, 3, 55, 29, 11, 58, 83, 40, 65, 33, 78, 15, 94, 42, 66, 35, 23, 97, 104, 52, 64,
   27, 102, 54, 87, 47, 60, 86, 70, 68, 89, 50, 101, 103, 81, 59, 77, 62, 99, 84, 72, 93, 80, 98, 91, 22, 54,
   4, 56, 81, 31, 13, 64, 41, 79, 34, 93, 16, 66, 43, 95, 103, 37, 25, 65, 53, 102, 28, 55, 45, 89, 61, 67,
   87, 88, 71, 101, 51, 57, 75, 104, 83, 100, 63, 85, 94, 73, 96, 82, 92, 5, 23, 57, 9, 27, 39, 63, 59, 82,
   17, 44, 35, 67, 80, 96, 21, 33, 49, 61, 98, 104, 29, 56, 103, 85, 62, 46, 69, 84, 48, 100, 68, 88, 79, 102,
   60, 99, 58, 76, 74, 86, 95, 78, 90, 97, 56, 24, 6, 79, 54, 61, 41, 32, 14, 95, 67, 81, 45, 36, 18, 102,
   93, 63, 51, 38, 26, 103, 57, 30, 43, 64, 88, 85, 66, 100, 53, 89, 71, 73, 55, 101, 65, 104, 82, 96, 87, 75,
   94, 92, 83, 7, 59, 25, 39, 62, 10, 28, 80, 57, 46, 19, 68, 37, 98, 82, 48, 60, 22, 34, 103, 96, 58, 31,
   104, 63, 84, 44, 49, 99, 69, 85, 86, 67, 61, 100, 78, 102, 74, 56, 76, 97, 88, 90, 79, 95, 58, 8, 26, 60,
   40, 78, 55, 12, 30, 68, 97, 47, 83, 20, 38, 62, 50, 102, 94, 24, 36, 59, 104, 32, 65, 42, 86, 99, 52, 84,
   66, 70, 87, 101, 64, 72, 54, 80, 103, 98, 77, 89, 91, 93, 81, 9, 15, 73, 17, 23, 46, 82, 50, 92, 21, 69,
   27, 85, 72, 94, 29, 35, 76, 88, 91, 101, 33, 79, 93, 96, 59, 48, 74, 80, 58, 104, 62, 100, 95, 103, 68, 98,
   60, 90, 78, 84, 102, 86, 97, 99, 10, 72, 16, 44, 80, 19, 25, 91, 51, 69, 22, 84, 28, 93, 73, 74, 86, 31,
   37, 101, 92, 78, 34, 94, 57, 98, 49, 56, 103, 76, 82, 99, 63, 67, 96, 97, 104, 90, 61, 79, 102, 85, 95, 88,
   100, 17, 11, 75, 21, 15, 48, 92, 47, 83, 29, 74, 23, 96, 70, 87, 33, 27, 90, 100, 77, 89, 35, 95, 81, 94,
   50, 58, 78, 72, 60, 101, 59, 104, 102, 93, 62, 91, 68, 97, 86, 80, 103, 84, 99, 98, 74, 12, 18, 78, 42, 90,
   53, 20, 26, 86, 95, 70, 75, 24, 30, 84, 72, 100, 92, 32, 38, 80, 96, 36, 55, 52, 97, 102, 54, 99, 65, 77,
   83, 94, 66, 91, 64, 98, 104, 103, 81, 87, 93, 101, 89, 19, 77, 13, 49, 91, 22, 16, 81, 45, 76, 31, 98, 25,
   89, 71, 90, 99, 34, 28, 87, 75, 97, 37, 83, 51, 93, 56, 61, 101, 79, 73, 103, 57, 63, 92, 102, 94, 95, 67,
   88, 104, 82, 100, 85, 96, 76, 20, 14, 90, 52, 79, 43, 24, 18, 97, 88, 77, 71, 32, 26, 99, 91, 85, 73, 36,
   30, 98, 82, 38, 53, 54, 95, 100, 64, 102, 55, 81, 75, 92, 65, 93, 66, 103, 96, 104, 89, 83, 101, 94, 87, 39,
   50, 51, 46, 59, 44, 57, 72, 73, 48, 49, 62, 63, 91, 92, 58, 68, 56, 67, 93, 94, 60, 61, 101, 82, 80, 69,
   76, 98, 74, 96, 84, 85, 88, 104, 86, 103, 78, 79, 90, 99, 100, 97, 95, 102, 48, 40, 53, 58, 47, 74, 75, 42,
   55, 60, 90, 50, 92, 52, 65, 68, 59, 95, 96, 54, 66, 62, 100, 64, 83, 70, 78, 97, 77, 86, 87, 72, 94, 104,
   89, 80, 81, 84, 102, 99, 91, 101, 98, 103, 93, 49, 52, 41, 76, 77, 56, 45, 54, 43, 90, 61, 91, 51, 64, 53,
   97, 98, 67, 57, 66, 55, 99, 63, 65, 71, 81, 79, 88, 89, 95, 75, 93, 73, 82, 83, 103, 87, 102, 85, 100, 101,
   92, 104, 96, 94, 69, 42, 43, 74, 70, 76, 71, 52, 53, 78, 79, 72, 73, 54, 55, 86, 80, 88, 82, 64, 65, 84,
   85, 66, 75, 77, 90, 95, 81, 97, 83, 91, 92, 96, 87, 98, 89, 99, 100, 102, 93, 94, 103, 104, 101, 44, 70, 45,
   69, 72, 49, 51, 77, 71, 74, 56, 80, 57, 81, 75, 78, 84, 61, 63, 89, 83, 86, 67, 87, 73, 91, 76, 79, 93,
   90, 92, 98, 82, 85, 94, 99, 101, 97, 88, 95, 103, 96, 102, 100, 104, 46, 47, 71, 48, 50, 69, 73, 70, 75, 58,
   76, 59, 82, 77, 83, 60, 62, 79, 85, 81, 87, 68, 88, 89, 92, 72, 74, 90, 91, 78, 94, 80, 96, 100, 101, 84,
   93, 86, 95, 97, 98, 104, 99, 102, 103
}; // 1.85 kb
static const cart_index_t iCartYY_ab13_a9[825] = {
   0, 23, 25, 27, 28, 5, 57, 7, 59, 9, 10, 35, 80, 37, 82, 33, 34, 17, 96, 19, 98, 21, 22, 29, 103, 31,
   104, 39, 62, 63, 84, 85, 44, 67, 46, 68, 48, 49, 60, 61, 99, 100, 102, 56, 58, 69, 86, 88, 78, 79, 74, 95,
   76, 97, 90, 21, 1, 26, 3, 55, 29, 30, 58, 8, 33, 78, 11, 12, 83, 38, 15, 94, 35, 36, 97, 20, 27, 102,
   23, 24, 104, 32, 60, 40, 65, 42, 66, 86, 87, 68, 47, 62, 99, 50, 101, 52, 64, 54, 103, 59, 84, 70, 89, 72,
   93, 80, 81, 98, 77, 91, 22, 24, 2, 54, 4, 56, 6, 31, 32, 79, 34, 81, 36, 13, 14, 93, 16, 95, 18, 37,
   38, 102, 28, 103, 30, 25, 26, 61, 64, 41, 66, 43, 67, 45, 88, 89, 100, 63, 101, 51, 65, 53, 55, 57, 104, 85,
   87, 71, 94, 73, 96, 75, 82, 83, 92, 5, 15, 51, 23, 57, 9, 73, 39, 50, 17, 44, 27, 72, 63, 92, 35, 67,
   21, 94, 49, 91, 29, 56, 33, 93, 61, 101, 46, 59, 82, 80, 96, 69, 85, 48, 62, 58, 76, 68, 88, 98, 104, 103,
   79, 60, 74, 84, 100, 86, 95, 78, 102, 90, 99, 97, 17, 3, 53, 11, 75, 21, 55, 48, 40, 29, 74, 15, 42, 92,
   65, 23, 96, 33, 66, 90, 52, 35, 95, 27, 54, 100, 64, 58, 47, 83, 70, 87, 78, 94, 60, 50, 68, 97, 59, 104,
   77, 89, 81, 102, 62, 86, 72, 101, 80, 103, 84, 93, 99, 91, 98, 7, 50, 16, 59, 25, 39, 51, 10, 72, 46, 19,
   62, 91, 28, 73, 68, 37, 48, 92, 22, 93, 58, 31, 60, 101, 34, 94, 44, 80, 57, 98, 82, 49, 63, 69, 84, 74,
   56, 86, 67, 103, 96, 104, 61, 78, 76, 99, 85, 97, 88, 90, 100, 79, 102, 95, 19, 52, 4, 77, 13, 49, 41, 22,
   54, 76, 31, 91, 64, 16, 43, 98, 25, 90, 53, 34, 66, 97, 37, 99, 65, 28, 55, 56, 81, 45, 89, 71, 61, 51,
   79, 93, 95, 67, 103, 57, 87, 75, 83, 63, 102, 88, 101, 73, 104, 82, 100, 92, 85, 94, 96, 48, 8, 18, 40, 53,
   58, 26, 74, 12, 60, 90, 47, 20, 75, 30, 50, 92, 68, 38, 95, 24, 62, 100, 59, 32, 96, 36, 78, 42, 55, 52,
   65, 97, 83, 86, 70, 84, 102, 72, 94, 54, 66, 64, 104, 80, 99, 77, 87, 91, 101, 98, 89, 103, 81, 93, 49, 20,
   6, 52, 41, 76, 14, 56, 24, 90, 61, 77, 32, 45, 18, 91, 51, 97, 26, 67, 36, 99, 63, 98, 38, 57, 30, 79,
   54, 43, 64, 53, 88, 71, 95, 81, 102, 85, 93, 73, 66, 55, 65, 82, 103, 100, 89, 75, 101, 92, 104, 83, 96, 87,
   94, 9, 11, 71, 15, 73, 17, 75, 46, 47, 21, 69, 23, 70, 82, 83, 27, 85, 29, 87, 76, 77, 33, 79, 35, 81,
   88, 89, 48, 50, 92, 72, 94, 74, 96, 58, 59, 60, 90, 62, 100, 91, 101, 93, 95, 68, 78, 80, 104, 84, 102, 86,
   103, 97, 98, 99, 10, 70, 13, 72, 16, 44, 45, 19, 77, 69, 22, 80, 81, 25, 71, 84, 28, 74, 75, 31, 89, 78,
   34, 86, 87, 37, 83, 49, 91, 51, 93, 73, 56, 57, 76, 98, 90, 61, 99, 63, 101, 92, 94, 67, 97, 79, 103, 82,
   102, 85, 95, 96, 88, 104, 100, 69, 12, 14, 42, 43, 74, 18, 76, 20, 78, 79, 70, 24, 71, 26, 72, 73, 86, 30,
   88, 32, 84, 85, 80, 36, 82, 38, 90, 52, 53, 54, 55, 95, 75, 97, 77, 99, 100, 91, 92, 64, 65, 66, 96, 98,
   102, 81, 83, 93, 94, 103, 87, 104, 89, 101, 44, 42, 41, 70, 45, 69, 43, 49, 52, 74, 56, 72, 54, 51, 53, 80,
   57, 78, 55, 61, 64, 86, 67, 84, 66, 63, 65, 76, 77, 71, 81, 75, 79, 73, 90, 91, 97, 88, 98, 82, 89, 83,
   87, 85, 99, 95, 93, 92, 103, 96, 102, 94, 100, 101, 104, 46, 40, 43, 47, 71, 48, 53, 69, 42, 58, 76, 50, 52,
   73, 55, 59, 82, 60, 65, 79, 54, 68, 88, 62, 64, 85, 66, 74, 70, 75, 77, 83, 90, 92, 78, 72, 86, 95, 80,
   96, 81, 87, 89, 100, 84, 97, 91, 94, 98, 104, 99, 101, 102, 93, 103, 39, 47, 45, 50, 51, 46, 71, 44, 70, 48,
   49, 59, 77, 57, 75, 62, 63, 58, 83, 56, 81, 60, 61, 68, 89, 67, 87, 69, 72, 73, 91, 92, 76, 82, 74, 80,
   78, 79, 84, 85, 93, 94, 101, 88, 86, 90, 98, 96, 99, 100, 97, 104, 95, 103, 102
}; // 1.61 kb
static const cart_index_t iCartYY_ab13_a10[660] = {
   0, 15, 16, 5, 23, 7, 25, 50, 51, 9, 10, 27, 28, 72, 73, 17, 35, 19, 37, 91, 92, 21, 22, 33, 34, 93,
   94, 29, 31, 101, 57, 59, 39, 44, 80, 46, 82, 62, 63, 67, 96, 68, 98, 48, 49, 56, 103, 58, 104, 60, 61, 69,
   84, 85, 74, 86, 76, 88, 99, 100, 78, 79, 102, 95, 97, 90, 17, 1, 18, 21, 3, 48, 53, 8, 26, 29, 74, 11,
   75, 12, 30, 33, 15, 90, 92, 20, 38, 35, 95, 23, 96, 24, 36, 27, 100, 32, 55, 40, 58, 78, 42, 60, 65, 47,
   83, 94, 66, 50, 52, 68, 97, 102, 54, 62, 64, 59, 104, 86, 70, 87, 84, 72, 99, 101, 77, 89, 80, 103, 81, 93,
   91, 98, 19, 20, 2, 49, 52, 22, 4, 24, 6, 76, 31, 77, 13, 32, 14, 90, 91, 34, 16, 36, 18, 97, 37, 98,
   25, 38, 26, 99, 28, 30, 41, 54, 56, 61, 64, 79, 43, 81, 45, 51, 53, 93, 66, 95, 67, 63, 65, 102, 55, 103,
   57, 88, 89, 71, 100, 101, 85, 73, 87, 75, 104, 82, 83, 92, 94, 96, 9, 3, 43, 17, 11, 46, 71, 40, 53, 21,
   69, 15, 73, 42, 55, 29, 23, 76, 82, 52, 65, 33, 79, 27, 85, 54, 66, 35, 88, 64, 75, 47, 48, 74, 70, 58,
   83, 50, 92, 96, 87, 59, 77, 60, 90, 95, 81, 68, 89, 62, 100, 78, 72, 94, 86, 80, 97, 104, 91, 101, 84, 102,
   93, 103, 98, 99, 10, 42, 4, 44, 70, 19, 13, 52, 41, 69, 22, 72, 16, 54, 43, 74, 80, 31, 25, 64, 53, 78,
   34, 84, 28, 66, 55, 86, 37, 65, 45, 77, 49, 56, 81, 76, 71, 91, 51, 57, 75, 98, 89, 90, 61, 67, 87, 97,
   83, 99, 63, 79, 93, 73, 95, 103, 88, 82, 101, 92, 102, 85, 94, 96, 104, 100, 5, 11, 45, 9, 15, 39, 51, 47,
   71, 17, 44, 23, 57, 70, 75, 21, 27, 49, 63, 77, 83, 29, 56, 35, 67, 81, 87, 33, 61, 89, 73, 50, 46, 69,
   72, 48, 92, 59, 82, 85, 94, 62, 91, 58, 76, 79, 93, 60, 101, 68, 88, 74, 80, 96, 78, 84, 90, 100, 98, 104,
   86, 95, 103, 102, 99, 97, 44, 12, 6, 69, 42, 49, 41, 20, 14, 74, 56, 70, 45, 24, 18, 78, 72, 61, 51, 32,
   26, 86, 67, 80, 57, 36, 30, 84, 63, 38, 43, 52, 76, 79, 54, 90, 53, 77, 71, 73, 55, 91, 64, 97, 88, 85,
   66, 99, 65, 98, 82, 95, 81, 75, 102, 93, 100, 92, 89, 83, 103, 96, 87, 94, 101, 104, 7, 47, 13, 39, 50, 10,
   16, 70, 45, 46, 19, 59, 25, 77, 71, 48, 62, 22, 28, 81, 75, 58, 31, 68, 37, 89, 83, 60, 34, 87, 51, 72,
   44, 49, 91, 69, 73, 80, 57, 63, 92, 84, 93, 74, 56, 61, 101, 78, 94, 86, 67, 76, 98, 82, 90, 99, 79, 85,
   103, 96, 97, 88, 104, 100, 102, 95, 46, 8, 14, 48, 40, 69, 43, 12, 18, 58, 76, 47, 71, 20, 26, 60, 50, 79,
   73, 24, 30, 68, 88, 59, 82, 32, 38, 62, 85, 36, 53, 42, 74, 90, 52, 78, 55, 70, 75, 92, 65, 72, 54, 86,
   95, 100, 64, 84, 66, 80, 96, 97, 77, 83, 99, 91, 102, 94, 81, 87, 98, 104, 89, 101, 93, 103, 39, 40, 41, 46,
   47, 44, 45, 42, 43, 48, 49, 50, 51, 52, 53, 58, 59, 56, 57, 54, 55, 60, 61, 62, 63, 64, 65, 68, 67, 66,
   71, 70, 69, 76, 77, 74, 75, 72, 73, 82, 83, 80, 81, 78, 79, 88, 89, 86, 87, 84, 85, 90, 91, 92, 97, 98,
   95, 96, 93, 94, 99, 100, 101, 104, 103, 102
}; // 1.29 kb
static const cart_index_t iCartYY_ab13_a11[468] = {
   0, 11, 13, 15, 16, 5, 45, 7, 47, 9, 10, 23, 70, 25, 71, 27, 28, 17, 75, 19, 77, 21, 22, 35, 81, 37,
   83, 33, 34, 29, 87, 31, 89, 39, 50, 51, 72, 73, 44, 57, 46, 59, 48, 49, 62, 63, 91, 92, 93, 94, 56, 67,
   58, 68, 60, 61, 101, 69, 80, 82, 84, 85, 74, 96, 76, 98, 78, 79, 86, 103, 88, 104, 90, 99, 100, 102, 95, 97,
   9, 1, 14, 3, 43, 17, 18, 46, 8, 21, 69, 11, 12, 71, 26, 15, 73, 29, 30, 76, 20, 33, 79, 23, 24, 82,
   38, 27, 85, 35, 36, 88, 32, 48, 40, 53, 42, 55, 74, 75, 58, 47, 60, 90, 50, 92, 52, 65, 54, 66, 95, 96,
   68, 59, 62, 100, 64, 78, 70, 83, 72, 94, 86, 87, 97, 77, 84, 102, 80, 81, 104, 89, 99, 91, 101, 93, 103, 98,
   10, 12, 2, 42, 4, 44, 6, 19, 20, 69, 22, 70, 24, 13, 14, 72, 16, 74, 18, 31, 32, 78, 34, 80, 36, 25,
   26, 84, 28, 86, 30, 37, 38, 49, 52, 41, 54, 43, 56, 45, 76, 77, 90, 61, 91, 51, 64, 53, 66, 55, 67, 57,
   97, 98, 99, 63, 65, 79, 81, 71, 93, 73, 95, 75, 88, 89, 102, 85, 103, 87, 82, 83, 100, 101, 92, 94, 96, 104,
   5, 3, 41, 11, 45, 9, 43, 39, 40, 17, 44, 15, 42, 51, 53, 23, 57, 21, 55, 49, 52, 29, 56, 27, 54, 63,
   65, 35, 67, 33, 66, 61, 64, 46, 47, 71, 70, 75, 69, 73, 48, 50, 58, 76, 59, 82, 77, 83, 81, 87, 79, 85,
   60, 62, 68, 88, 89, 74, 72, 92, 80, 96, 78, 94, 90, 91, 86, 95, 84, 93, 100, 101, 97, 98, 104, 103, 102, 99,
   7, 40, 4, 47, 13, 39, 41, 10, 42, 46, 19, 50, 52, 16, 43, 59, 25, 48, 53, 22, 54, 58, 31, 62, 64, 28,
   55, 68, 37, 60, 65, 34, 66, 44, 70, 45, 77, 71, 49, 51, 69, 72, 74, 56, 80, 57, 81, 75, 89, 83, 61, 63,
   78, 84, 86, 67, 87, 76, 91, 73, 98, 82, 90, 92, 79, 93, 97, 88, 99, 101, 85, 94, 95, 103, 96, 104, 100, 102,
   39, 8, 6, 40, 41, 46, 14, 44, 12, 48, 49, 47, 20, 45, 18, 50, 51, 58, 26, 56, 24, 60, 61, 59, 32, 57,
   30, 62, 63, 68, 38, 67, 36, 69, 42, 43, 52, 53, 76, 71, 74, 70, 78, 79, 72, 73, 54, 55, 64, 65, 88, 82,
   86, 80, 84, 85, 66, 90, 77, 75, 91, 92, 97, 83, 95, 81, 99, 100, 98, 89, 96, 87, 102, 93, 94, 101, 104, 103
}; // 0.91 kb
static const cart_index_t iCartYY_ab13_a12[273] = {
   0, 3, 4, 5, 11, 7, 13, 40, 41, 9, 10, 15, 16, 42, 43, 17, 23, 19, 25, 52, 53, 21, 22, 27, 28, 54,
   55, 29, 35, 31, 37, 64, 65, 33, 34, 66, 45, 47, 39, 44, 70, 46, 71, 50, 51, 57, 75, 59, 77, 48, 49, 56,
   81, 58, 83, 62, 63, 67, 87, 68, 89, 60, 61, 69, 72, 73, 74, 80, 76, 82, 91, 92, 78, 79, 84, 85, 93, 94,
   86, 88, 101, 96, 98, 90, 95, 103, 97, 104, 99, 100, 102, 5, 1, 6, 9, 3, 39, 41, 8, 14, 17, 44, 11, 45,
   12, 18, 21, 15, 49, 51, 20, 26, 29, 56, 23, 57, 24, 30, 33, 27, 61, 63, 32, 38, 35, 67, 36, 43, 40, 46,
   69, 42, 48, 53, 47, 71, 73, 55, 50, 52, 58, 76, 79, 54, 60, 65, 59, 82, 85, 66, 62, 64, 68, 88, 74, 70,
   75, 78, 72, 90, 92, 77, 83, 86, 95, 80, 96, 81, 87, 84, 100, 89, 94, 91, 97, 102, 93, 99, 101, 98, 104, 103,
   7, 8, 2, 39, 40, 10, 4, 12, 6, 46, 19, 47, 13, 20, 14, 48, 50, 22, 16, 24, 18, 58, 31, 59, 25, 32,
   26, 60, 62, 34, 28, 36, 30, 68, 37, 38, 41, 42, 44, 49, 52, 69, 43, 70, 45, 51, 53, 72, 54, 74, 56, 61,
   64, 78, 55, 80, 57, 63, 65, 84, 66, 86, 67, 76, 77, 71, 90, 91, 79, 73, 81, 75, 97, 88, 98, 82, 89, 83,
   99, 85, 87, 92, 93, 95, 100, 101, 102, 94, 103, 96, 104
}; // 0.53 kb
static const cart_index_t iCartYY_ab13_a13[105] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104
}; // 0.21 kb
static const cart_index_t iCartYY_ab14_a0[120] = {
   0, 1, 2, 3, 4, 5, 6, 7, 8, 9, 10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22, 23, 24, 25,
   26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38, 39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51,
   52, 53, 54, 55, 56, 57, 58, 59, 60, 61, 62, 63, 64, 65, 66, 67, 68, 69, 70, 71, 72, 73, 74, 75, 76, 77,
   78, 79, 80, 81, 82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103,
   104, 105, 106, 107, 108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119
}; // 0.23 kb
static const cart_index_t iCartYY_ab14_a1[315] = {
   0, 3, 5, 4, 1, 7, 6, 8, 2, 11, 4, 43, 12, 42, 6, 3, 9, 44, 42, 14, 8, 5, 44, 10, 43, 7,
   13, 9, 15, 47, 10, 45, 17, 16, 11, 49, 46, 13, 19, 18, 50, 12, 48, 20, 14, 23, 16, 53, 24, 51, 18, 15,
   21, 55, 52, 26, 20, 17, 56, 22, 54, 19, 25, 21, 27, 59, 22, 57, 29, 28, 23, 61, 58, 25, 31, 30, 62, 24,
   60, 32, 26, 35, 28, 65, 36, 63, 30, 27, 33, 67, 64, 38, 32, 29, 68, 34, 66, 31, 37, 33, 39, 71, 34, 69,
   40, 39, 35, 73, 70, 37, 41, 40, 74, 36, 72, 41, 38, 44, 47, 45, 49, 43, 46, 50, 48, 42, 76, 46, 54, 77,
   52, 48, 45, 75, 56, 51, 77, 50, 47, 55, 75, 53, 49, 76, 55, 59, 78, 56, 80, 57, 61, 53, 79, 62, 81, 51,
   82, 54, 58, 83, 60, 52, 88, 58, 66, 89, 64, 60, 57, 85, 68, 63, 87, 62, 59, 67, 84, 65, 61, 86, 67, 71,
   90, 68, 92, 69, 73, 65, 91, 74, 93, 63, 94, 66, 70, 95, 72, 64, 98, 70, 72, 69, 97, 74, 71, 73, 96, 75,
   78, 80, 79, 76, 82, 81, 83, 77, 86, 79, 100, 87, 99, 81, 78, 84, 101, 99, 89, 83, 80, 101, 85, 100, 82, 88,
   84, 90, 104, 85, 102, 92, 91, 86, 106, 103, 88, 94, 93, 107, 87, 105, 95, 89, 96, 91, 110, 97, 108, 93, 90, 96,
   112, 109, 98, 95, 92, 113, 97, 111, 94, 98, 101, 104, 102, 106, 100, 103, 107, 105, 99, 115, 103, 111, 116, 109, 105, 102,
   114, 113, 108, 116, 107, 104, 112, 114, 110, 106, 115, 112, 110, 117, 113, 118, 108, 119, 111, 109, 114, 117, 118, 117, 115, 119,
   118, 119, 116
}; // 0.62 kb
static const cart_index_t iCartYY_ab14_a2[546] = {
   0, 9, 10, 3, 5, 44, 11, 1, 13, 4, 43, 7, 12, 14, 2, 42, 6, 8, 3, 15, 45, 9, 44, 47, 16, 4,
   46, 11, 49, 43, 5, 47, 17, 44, 10, 45, 18, 48, 6, 50, 12, 42, 49, 7, 19, 43, 46, 13, 50, 20, 8, 48,
   42, 14, 9, 21, 75, 15, 47, 55, 10, 75, 22, 45, 17, 56, 23, 11, 76, 16, 53, 49, 24, 77, 12, 51, 18, 50,
   76, 13, 25, 46, 54, 19, 77, 26, 14, 52, 48, 20, 15, 27, 78, 21, 55, 59, 28, 16, 79, 23, 61, 53, 17, 80,
   29, 56, 22, 57, 30, 81, 18, 62, 24, 51, 82, 19, 31, 54, 58, 25, 83, 32, 20, 60, 52, 26, 21, 33, 84, 27,
   59, 67, 22, 85, 34, 57, 29, 68, 35, 23, 86, 28, 65, 61, 36, 87, 24, 63, 30, 62, 88, 25, 37, 58, 66, 31,
   89, 38, 26, 64, 60, 32, 27, 39, 90, 33, 67, 71, 39, 28, 91, 35, 73, 65, 29, 92, 40, 68, 34, 69, 40, 93,
   30, 74, 36, 63, 94, 31, 41, 66, 70, 37, 95, 41, 32, 72, 64, 38, 33, 35, 96, 39, 71, 73, 34, 97, 36, 69,
   40, 74, 98, 37, 38, 70, 72, 41, 51, 52, 42, 77, 50, 48, 53, 43, 54, 49, 76, 46, 44, 55, 56, 47, 45, 75,
   45, 78, 57, 75, 56, 80, 79, 46, 58, 76, 82, 54, 47, 59, 80, 55, 75, 78, 81, 60, 48, 83, 77, 52, 61, 49,
   82, 53, 79, 76, 62, 83, 50, 81, 51, 77, 63, 99, 51, 87, 62, 81, 99, 64, 52, 89, 83, 60, 65, 53, 100, 61,
   86, 79, 100, 54, 66, 82, 88, 58, 55, 67, 101, 59, 78, 84, 56, 101, 68, 80, 57, 85, 57, 102, 69, 85, 68, 92,
   103, 58, 70, 88, 94, 66, 59, 71, 104, 67, 84, 90, 105, 72, 60, 95, 89, 64, 73, 61, 106, 65, 91, 86, 74, 107,
   62, 93, 63, 87, 69, 108, 63, 97, 74, 93, 109, 70, 64, 98, 95, 72, 71, 65, 110, 73, 96, 91, 111, 66, 72, 94,
   98, 70, 67, 73, 112, 71, 90, 96, 68, 113, 74, 92, 69, 97, 75, 84, 85, 78, 80, 101, 86, 76, 88, 79, 100, 82,
   87, 89, 77, 99, 81, 83, 78, 90, 102, 84, 101, 104, 91, 79, 103, 86, 106, 100, 80, 104, 92, 101, 85, 102, 93, 105,
   81, 107, 87, 99, 106, 82, 94, 100, 103, 88, 107, 95, 83, 105, 99, 89, 84, 96, 114, 90, 104, 112, 85, 114, 97, 102,
   92, 113, 96, 86, 115, 91, 110, 106, 97, 116, 87, 108, 93, 107, 115, 88, 98, 103, 111, 94, 116, 98, 89, 109, 105, 95,
   90, 91, 117, 96, 112, 110, 92, 118, 93, 113, 97, 108, 119, 94, 95, 111, 109, 98, 108, 109, 99, 116, 107, 105, 110, 100,
   111, 106, 115, 103, 101, 112, 113, 104, 102, 114, 102, 117, 108, 114, 113, 118, 117, 103, 109, 115, 119, 111, 104, 110, 118, 112,
   114, 117, 118, 111, 105, 119, 116, 109, 112, 106, 119, 110, 117, 115, 113, 119, 107, 118, 108, 116, 114, 115, 116, 117, 118, 119
}; // 1.07 kb
static const cart_index_t iCartYY_ab14_a3[780] = {
   0, 15, 17, 9, 10, 3, 45, 5, 47, 44, 16, 1, 19, 4, 46, 11, 13, 49, 7, 43, 18, 20, 2, 48, 6, 50,
   8, 12, 14, 42, 23, 4, 54, 11, 76, 16, 46, 53, 43, 49, 24, 52, 6, 77, 12, 51, 42, 18, 48, 50, 3, 21,
   56, 15, 45, 9, 75, 44, 55, 47, 51, 26, 8, 52, 42, 77, 14, 50, 20, 48, 5, 55, 22, 47, 17, 44, 56, 10,
   75, 45, 53, 7, 25, 43, 54, 49, 19, 76, 13, 46, 9, 27, 80, 21, 75, 15, 78, 47, 59, 55, 10, 78, 29, 75,
   22, 45, 57, 17, 80, 56, 28, 11, 82, 16, 79, 23, 76, 61, 49, 53, 79, 13, 31, 46, 58, 76, 25, 82, 19, 54,
   30, 83, 12, 81, 18, 62, 50, 24, 77, 51, 81, 32, 14, 60, 48, 83, 20, 77, 26, 52, 35, 16, 100, 23, 86, 28,
   79, 65, 53, 61, 36, 99, 18, 87, 24, 63, 51, 30, 81, 62, 15, 33, 101, 27, 78, 21, 84, 55, 67, 59, 99, 38,
   20, 64, 52, 89, 26, 83, 32, 60, 17, 101, 34, 80, 29, 56, 68, 22, 85, 57, 100, 19, 37, 54, 66, 82, 31, 88,
   25, 58, 21, 39, 104, 33, 84, 27, 90, 59, 71, 67, 22, 102, 40, 85, 34, 57, 69, 29, 92, 68, 39, 23, 106, 28,
   91, 35, 86, 73, 61, 65, 103, 25, 41, 58, 70, 88, 37, 94, 31, 66, 40, 107, 24, 93, 30, 74, 62, 36, 87, 63,
   105, 41, 26, 72, 60, 95, 32, 89, 38, 64, 33, 28, 110, 35, 96, 39, 91, 71, 65, 73, 34, 108, 30, 97, 36, 69,
   63, 40, 93, 74, 27, 35, 112, 39, 90, 33, 96, 67, 73, 71, 109, 37, 32, 70, 64, 98, 38, 95, 41, 72, 29, 113,
   36, 92, 40, 68, 74, 34, 97, 69, 111, 31, 38, 66, 72, 94, 41, 98, 37, 70, 44, 59, 57, 55, 56, 47, 80, 45,
   78, 75, 61, 43, 58, 49, 82, 53, 54, 79, 46, 76, 62, 60, 42, 83, 50, 81, 48, 51, 52, 77, 86, 46, 66, 76,
   88, 79, 58, 100, 54, 82, 87, 64, 48, 89, 77, 99, 52, 81, 60, 83, 45, 84, 68, 78, 57, 75, 85, 56, 101, 80,
   63, 89, 50, 99, 51, 87, 77, 62, 83, 81, 47, 67, 85, 59, 80, 55, 101, 75, 84, 78, 65, 49, 88, 53, 100, 61,
   82, 86, 76, 79, 55, 71, 102, 67, 101, 59, 104, 78, 90, 84, 56, 104, 69, 101, 68, 80, 92, 57, 102, 85, 73, 53,
   103, 61, 106, 65, 100, 91, 79, 86, 74, 105, 51, 107, 62, 93, 81, 63, 99, 87, 106, 54, 70, 82, 94, 100, 66, 103,
   58, 88, 107, 72, 52, 95, 83, 105, 60, 99, 64, 89, 115, 58, 72, 88, 98, 103, 70, 111, 66, 94, 116, 70, 60, 98,
   89, 109, 64, 105, 72, 95, 57, 114, 74, 102, 69, 85, 97, 68, 113, 92, 69, 116, 62, 108, 63, 97, 87, 74, 107, 93,
   59, 73, 114, 71, 104, 67, 112, 84, 96, 90, 71, 61, 115, 65, 110, 73, 106, 96, 86, 91, 67, 65, 117, 73, 112, 71,
   110, 90, 91, 96, 68, 118, 63, 113, 74, 92, 93, 69, 108, 97, 119, 66, 64, 94, 95, 111, 72, 109, 70, 98, 75, 90,
   92, 84, 85, 78, 102, 80, 104, 101, 91, 76, 94, 79, 103, 86, 88, 106, 82, 100, 93, 95, 77, 105, 81, 107, 83, 87,
   89, 99, 96, 79, 111, 86, 115, 91, 103, 110, 100, 106, 97, 109, 81, 116, 87, 108, 99, 93, 105, 107, 78, 96, 113, 90,
   102, 84, 114, 101, 112, 104, 108, 98, 83, 109, 99, 116, 89, 107, 95, 105, 80, 112, 97, 104, 92, 101, 113, 85, 114, 102,
   110, 82, 98, 100, 111, 106, 94, 115, 88, 103, 84, 91, 118, 96, 114, 90, 117, 104, 110, 112, 85, 117, 93, 114, 97, 102,
   108, 92, 118, 113, 90, 86, 119, 91, 117, 96, 115, 112, 106, 110, 117, 88, 95, 103, 109, 115, 98, 119, 94, 111, 92, 119,
   87, 118, 93, 113, 107, 97, 116, 108, 118, 94, 89, 111, 105, 119, 95, 116, 98, 109, 101, 110, 108, 112, 113, 104, 118, 102,
   117, 114, 112, 100, 109, 106, 119, 110, 111, 117, 103, 115, 113, 111, 99, 119, 107, 118, 105, 108, 109, 116, 114, 103, 105, 115,
   116, 117, 109, 118, 111, 119, 102, 115, 107, 117, 108, 114, 116, 113, 119, 118, 104, 106, 116, 110, 118, 112, 119, 114, 115, 117
}; // 1.52 kb
static const cart_index_t iCartYY_ab14_a4[990] = {
   0, 21, 22, 3, 15, 5, 17, 55, 56, 9, 10, 75, 45, 47, 44, 23, 1, 25, 16, 4, 53, 54, 7, 19, 11, 76,
   13, 46, 43, 49, 24, 26, 2, 51, 52, 18, 6, 20, 8, 77, 12, 14, 42, 48, 50, 3, 27, 57, 9, 21, 44, 56,
   59, 80, 15, 45, 78, 75, 55, 47, 28, 4, 58, 23, 11, 61, 82, 43, 54, 16, 79, 46, 76, 49, 53, 5, 59, 29,
   44, 55, 10, 22, 78, 57, 47, 17, 80, 56, 75, 45, 30, 60, 6, 62, 83, 24, 12, 52, 42, 81, 18, 48, 50, 77,
   51, 61, 7, 31, 53, 43, 79, 58, 13, 25, 49, 82, 19, 54, 46, 76, 62, 32, 8, 81, 60, 51, 42, 26, 14, 83,
   50, 20, 48, 52, 77, 9, 33, 85, 15, 27, 47, 80, 67, 101, 21, 75, 84, 78, 59, 55, 10, 84, 34, 45, 78, 17,
   29, 101, 68, 75, 22, 85, 57, 80, 56, 35, 11, 88, 28, 16, 65, 100, 49, 82, 23, 86, 76, 79, 53, 61, 36, 89,
   12, 63, 99, 30, 18, 83, 50, 87, 24, 77, 51, 81, 62, 86, 13, 37, 79, 46, 100, 66, 19, 31, 76, 88, 25, 58,
   54, 82, 87, 38, 14, 99, 64, 81, 48, 32, 20, 89, 77, 26, 52, 60, 83, 15, 39, 102, 21, 33, 55, 101, 71, 104,
   27, 78, 90, 84, 67, 59, 39, 16, 103, 35, 23, 73, 106, 53, 100, 28, 91, 79, 86, 61, 65, 17, 104, 40, 56, 101,
   22, 34, 102, 69, 80, 29, 92, 68, 85, 57, 40, 105, 18, 74, 107, 36, 24, 99, 51, 93, 30, 81, 62, 87, 63, 106,
   19, 41, 100, 54, 103, 70, 25, 37, 82, 94, 31, 66, 58, 88, 107, 41, 20, 105, 72, 99, 52, 38, 26, 95, 83, 32,
   60, 64, 89, 21, 35, 114, 27, 39, 59, 104, 73, 112, 33, 84, 96, 90, 71, 67, 22, 114, 36, 57, 102, 29, 40, 113,
   74, 85, 34, 97, 69, 92, 68, 33, 23, 115, 39, 28, 71, 110, 61, 106, 35, 96, 86, 91, 65, 73, 34, 116, 24, 69,
   108, 40, 30, 107, 62, 97, 36, 87, 63, 93, 74, 115, 25, 38, 103, 58, 111, 72, 31, 41, 88, 98, 37, 70, 66, 94,
   116, 37, 26, 109, 70, 105, 60, 41, 32, 98, 89, 38, 64, 72, 95, 27, 28, 117, 33, 35, 67, 112, 65, 110, 39, 90,
   91, 96, 73, 71, 29, 118, 30, 68, 113, 34, 36, 108, 63, 92, 40, 93, 74, 97, 69, 119, 31, 32, 111, 66, 109, 64,
   37, 38, 94, 95, 41, 72, 70, 98, 63, 64, 42, 87, 89, 62, 50, 60, 48, 99, 51, 52, 77, 83, 81, 65, 43, 66,
   61, 49, 86, 88, 46, 58, 53, 100, 54, 82, 76, 79, 44, 67, 68, 47, 59, 45, 57, 84, 85, 55, 56, 101, 80, 78,
   75, 45, 90, 69, 75, 84, 56, 68, 104, 92, 78, 57, 102, 85, 101, 80, 91, 46, 70, 86, 76, 106, 94, 54, 66, 79,
   103, 58, 88, 82, 100, 47, 71, 92, 55, 67, 75, 85, 90, 102, 59, 80, 104, 101, 84, 78, 93, 72, 48, 107, 95, 87,
   77, 64, 52, 105, 81, 60, 83, 89, 99, 73, 49, 94, 65, 53, 91, 103, 76, 88, 61, 106, 82, 100, 79, 86, 74, 95,
   50, 93, 105, 63, 51, 89, 77, 107, 62, 83, 81, 99, 87, 69, 109, 51, 97, 116, 74, 62, 105, 81, 108, 63, 99, 87,
   107, 93, 108, 70, 52, 116, 98, 107, 83, 72, 60, 109, 99, 64, 89, 95, 105, 71, 53, 111, 73, 61, 96, 115, 79, 103,
   65, 110, 100, 106, 86, 91, 110, 54, 72, 106, 82, 115, 98, 58, 70, 100, 111, 66, 94, 88, 103, 55, 73, 113, 59, 71,
   78, 102, 96, 114, 67, 101, 112, 104, 90, 84, 56, 112, 74, 80, 104, 57, 69, 114, 97, 101, 68, 113, 92, 102, 85, 57,
   117, 63, 85, 114, 68, 74, 118, 93, 102, 69, 108, 97, 113, 92, 117, 58, 64, 115, 88, 119, 95, 66, 72, 103, 109, 70,
   98, 94, 111, 59, 65, 118, 67, 73, 84, 114, 91, 117, 71, 104, 110, 112, 96, 90, 118, 66, 60, 119, 94, 116, 89, 70,
   64, 111, 105, 72, 95, 98, 109, 67, 61, 119, 71, 65, 90, 117, 86, 115, 73, 112, 106, 110, 91, 96, 68, 119, 62, 92,
   118, 69, 63, 116, 87, 113, 74, 107, 93, 108, 97, 75, 96, 97, 78, 90, 80, 92, 112, 113, 84, 85, 114, 102, 104, 101,
   96, 76, 98, 91, 79, 110, 111, 82, 94, 86, 115, 88, 103, 100, 106, 97, 98, 77, 108, 109, 93, 81, 95, 83, 116, 87,
   89, 99, 105, 107, 78, 91, 108, 84, 96, 101, 113, 110, 118, 90, 102, 117, 114, 112, 104, 90, 79, 109, 96, 86, 112, 119,
   100, 111, 91, 117, 103, 115, 106, 110, 80, 110, 93, 101, 112, 85, 97, 117, 108, 104, 92, 118, 113, 114, 102, 92, 111, 81,
   113, 119, 97, 87, 109, 99, 118, 93, 105, 107, 116, 108, 112, 82, 95, 110, 100, 117, 109, 88, 98, 106, 119, 94, 111, 103,
   115, 113, 94, 83, 118, 111, 108, 99, 98, 89, 119, 107, 95, 105, 109, 116, 84, 86, 116, 90, 91, 104, 118, 106, 119, 96,
   114, 115, 117, 110, 112, 85, 115, 87, 102, 117, 92, 93, 119, 107, 114, 97, 116, 108, 118, 113, 114, 88, 89, 117, 103, 118,
   105, 94, 95, 115, 116, 98, 109, 111, 119, 102, 103, 99, 114, 115, 113, 107, 111, 105, 117, 108, 109, 116, 119, 118, 104, 100,
   105, 112, 106, 114, 116, 103, 109, 110, 118, 111, 119, 115, 117, 101, 106, 107, 104, 110, 102, 108, 115, 116, 112, 113, 119, 118,
   117, 114
}; // 1.93 kb
static const cart_index_t iCartYY_ab14_a5[1155] = {
   0, 27, 29, 21, 22, 3, 57, 5, 59, 9, 10, 15, 78, 17, 80, 44, 55, 56, 75, 45, 47, 28, 1, 31, 4, 58,
   23, 25, 61, 7, 16, 79, 11, 13, 82, 19, 53, 43, 54, 46, 76, 49, 30, 32, 2, 60, 6, 62, 8, 24, 26, 81,
   18, 83, 20, 12, 14, 51, 52, 42, 48, 50, 77, 35, 4, 66, 11, 88, 28, 58, 65, 43, 23, 86, 16, 46, 100, 54,
   61, 49, 82, 76, 79, 53, 36, 64, 6, 89, 12, 63, 42, 30, 60, 87, 24, 99, 52, 18, 48, 62, 83, 50, 77, 51,
   81, 3, 33, 68, 27, 57, 9, 85, 44, 67, 15, 45, 21, 84, 56, 101, 47, 59, 80, 78, 75, 55, 63, 38, 8, 64,
   42, 87, 14, 62, 32, 99, 51, 89, 26, 50, 20, 81, 60, 48, 52, 77, 83, 5, 67, 34, 59, 29, 44, 68, 10, 84,
   47, 17, 55, 101, 22, 85, 45, 78, 57, 80, 56, 75, 65, 7, 37, 43, 66, 61, 31, 86, 13, 53, 100, 49, 19, 88,
   25, 79, 46, 58, 54, 82, 76, 9, 39, 92, 33, 85, 15, 102, 47, 71, 21, 75, 27, 90, 80, 104, 55, 67, 101, 84,
   78, 59, 10, 90, 40, 84, 34, 45, 69, 17, 104, 75, 22, 78, 102, 29, 92, 56, 101, 68, 85, 57, 80, 39, 11, 94,
   16, 103, 35, 88, 73, 49, 28, 91, 23, 76, 106, 82, 65, 53, 100, 79, 86, 61, 91, 13, 41, 46, 70, 86, 37, 106,
   19, 79, 103, 76, 25, 94, 31, 100, 54, 66, 58, 88, 82, 40, 95, 12, 105, 18, 74, 50, 36, 89, 93, 30, 107, 83,
   24, 77, 63, 99, 51, 81, 62, 87, 93, 41, 14, 72, 48, 107, 20, 87, 38, 105, 81, 95, 32, 77, 26, 99, 64, 52,
   60, 83, 89, 33, 16, 111, 23, 115, 39, 103, 71, 53, 35, 96, 28, 79, 110, 100, 73, 61, 106, 86, 91, 65, 34, 109,
   18, 116, 24, 69, 51, 40, 105, 97, 36, 108, 99, 30, 81, 74, 107, 62, 87, 63, 93, 15, 35, 113, 39, 102, 21, 114,
   55, 73, 27, 78, 33, 96, 101, 112, 59, 71, 104, 90, 84, 67, 108, 37, 20, 70, 52, 116, 26, 107, 41, 109, 99, 98,
   38, 83, 32, 105, 72, 60, 64, 89, 95, 17, 112, 36, 104, 40, 56, 74, 22, 114, 80, 29, 101, 113, 34, 97, 57, 102,
   69, 92, 68, 85, 110, 19, 38, 54, 72, 106, 41, 115, 25, 100, 111, 82, 31, 98, 37, 103, 58, 70, 66, 94, 88, 21,
   28, 118, 35, 114, 27, 117, 59, 65, 33, 84, 39, 91, 104, 110, 67, 73, 112, 96, 90, 71, 22, 117, 30, 114, 36, 57,
   63, 29, 118, 85, 34, 102, 108, 40, 93, 68, 113, 74, 97, 69, 92, 27, 23, 119, 28, 117, 33, 115, 67, 61, 39, 90,
   35, 86, 112, 106, 71, 65, 110, 91, 96, 73, 117, 25, 32, 58, 64, 115, 38, 119, 31, 103, 109, 88, 37, 95, 41, 111,
   66, 72, 70, 98, 94, 29, 119, 24, 118, 30, 68, 62, 34, 116, 92, 40, 113, 107, 36, 87, 69, 108, 63, 93, 74, 97,
   118, 31, 26, 66, 60, 119, 32, 116, 37, 111, 105, 94, 41, 89, 38, 109, 70, 64, 72, 95, 98, 44, 71, 69, 67, 68,
   47, 92, 45, 90, 55, 56, 59, 104, 57, 102, 75, 84, 85, 101, 80, 78, 73, 43, 70, 49, 94, 65, 66, 91, 46, 61,
   106, 53, 54, 103, 58, 86, 76, 88, 82, 100, 79, 74, 72, 42, 95, 50, 93, 48, 63, 64, 107, 62, 105, 60, 51, 52,
   87, 89, 77, 83, 81, 99, 96, 46, 72, 76, 98, 91, 70, 110, 54, 86, 115, 79, 58, 111, 66, 106, 82, 94, 88, 103,
   100, 97, 70, 48, 98, 77, 108, 52, 93, 72, 116, 87, 109, 64, 81, 60, 107, 95, 83, 89, 99, 105, 45, 96, 74, 90,
   69, 75, 97, 56, 112, 78, 57, 84, 114, 68, 113, 80, 104, 92, 102, 85, 101, 69, 98, 50, 109, 51, 97, 77, 74, 95,
   108, 63, 116, 89, 62, 83, 93, 105, 81, 99, 87, 107, 47, 73, 97, 71, 92, 55, 113, 75, 96, 59, 80, 67, 112, 85,
   114, 78, 90, 102, 104, 101, 84, 71, 49, 98, 53, 111, 73, 94, 96, 76, 65, 110, 61, 82, 115, 88, 91, 79, 103, 100,
   106, 86, 55, 65, 108, 73, 113, 59, 118, 78, 91, 67, 101, 71, 110, 102, 117, 84, 96, 114, 112, 104, 90, 56, 110, 63,
   112, 74, 80, 93, 57, 117, 101, 68, 104, 118, 69, 108, 85, 114, 97, 113, 92, 102, 67, 53, 109, 61, 119, 71, 111, 90,
   79, 73, 112, 65, 100, 117, 103, 96, 86, 115, 106, 110, 91, 68, 111, 51, 119, 62, 92, 81, 69, 109, 113, 74, 118, 105,
   63, 99, 97, 116, 87, 107, 93, 108, 112, 54, 64, 82, 95, 110, 72, 117, 58, 106, 119, 100, 66, 109, 70, 115, 88, 98,
   94, 111, 103, 113, 66, 52, 94, 83, 118, 60, 108, 70, 119, 107, 111, 72, 99, 64, 116, 98, 89, 95, 105, 109, 114, 58,
   60, 88, 89, 117, 64, 118, 66, 115, 116, 103, 70, 105, 72, 119, 94, 95, 98, 109, 111, 57, 115, 62, 117, 63, 85, 87,
   68, 119, 102, 69, 114, 116, 74, 107, 92, 118, 93, 108, 97, 113, 59, 61, 116, 65, 118, 67, 119, 84, 86, 71, 104, 73,
   106, 114, 115, 90, 91, 117, 110, 112, 96, 75, 91, 93, 96, 97, 78, 108, 80, 110, 84, 85, 90, 117, 92, 118, 101, 112,
   113, 114, 102, 104, 90, 76, 95, 79, 109, 96, 98, 112, 82, 91, 117, 86, 88, 119, 94, 110, 100, 111, 103, 115, 106, 92,
   94, 77, 111, 81, 113, 83, 97, 98, 118, 93, 119, 95, 87, 89, 108, 109, 99, 105, 107, 116, 84, 79, 105, 86, 116, 90,
   109, 104, 100, 96, 114, 91, 103, 118, 111, 112, 106, 119, 115, 117, 110, 85, 103, 81, 115, 87, 102, 99, 92, 111, 114, 97,
   117, 109, 93, 105, 113, 119, 107, 116, 108, 118, 78, 86, 107, 91, 108, 84, 116, 101, 106, 90, 102, 96, 115, 113, 119, 104,
   110, 118, 117, 114, 112, 102, 88, 83, 103, 99, 114, 89, 113, 94, 117, 108, 115, 98, 107, 95, 118, 111, 105, 109, 116, 119,
   80, 106, 87, 110, 93, 101, 107, 85, 115, 104, 92, 112, 119, 97, 116, 102, 117, 108, 118, 113, 114, 104, 82, 89, 100, 105,
   112, 95, 114, 88, 110, 118, 106, 94, 116, 98, 117, 103, 109, 111, 119, 115, 101, 100, 99, 106, 107, 104, 105, 102, 103, 112,
   113, 110, 111, 108, 109, 114, 115, 116, 119, 118, 117
}; // 2.26 kb
static const cart_index_t iCartYY_ab14_a6[1260] = {
   0, 33, 34, 3, 27, 5, 29, 67, 68, 9, 10, 21, 22, 84, 85, 15, 17, 101, 57, 59, 44, 45, 78, 47, 80, 55,
   56, 75, 35, 1, 37, 28, 4, 65, 66, 7, 31, 23, 86, 11, 88, 13, 25, 16, 100, 19, 58, 43, 61, 79, 46, 53,
   54, 49, 82, 76, 36, 38, 2, 63, 64, 30, 6, 32, 8, 87, 24, 89, 12, 26, 14, 99, 18, 20, 42, 60, 62, 51,
   52, 81, 48, 83, 50, 77, 3, 39, 69, 9, 33, 44, 68, 71, 92, 15, 45, 27, 57, 90, 102, 21, 56, 104, 85, 67,
   47, 75, 84, 55, 101, 59, 80, 78, 39, 4, 70, 35, 11, 73, 94, 43, 66, 28, 91, 16, 103, 46, 58, 23, 106, 54,
   88, 49, 65, 86, 76, 61, 82, 53, 100, 79, 5, 71, 40, 44, 67, 10, 34, 90, 69, 47, 17, 59, 29, 104, 92, 55,
   22, 102, 68, 84, 45, 56, 101, 75, 85, 78, 57, 80, 40, 72, 6, 74, 95, 36, 12, 64, 42, 93, 30, 105, 18, 60,
   48, 107, 24, 52, 50, 89, 63, 62, 83, 87, 77, 99, 51, 81, 73, 7, 41, 65, 43, 91, 70, 13, 37, 61, 106, 49,
   94, 19, 31, 53, 103, 25, 66, 46, 86, 100, 54, 79, 58, 76, 88, 82, 74, 41, 8, 93, 72, 63, 42, 38, 14, 107,
   62, 95, 50, 32, 20, 105, 51, 26, 48, 64, 87, 81, 60, 99, 52, 89, 77, 83, 9, 35, 97, 15, 39, 47, 92, 73,
   113, 21, 75, 33, 85, 96, 114, 27, 80, 112, 102, 71, 55, 78, 90, 59, 104, 67, 101, 84, 10, 96, 36, 45, 90, 17,
   40, 112, 74, 75, 22, 84, 34, 114, 97, 78, 29, 113, 69, 104, 56, 57, 102, 80, 92, 101, 68, 85, 33, 11, 98, 39,
   16, 71, 111, 49, 94, 35, 96, 23, 115, 76, 88, 28, 110, 82, 103, 53, 73, 91, 79, 65, 100, 61, 106, 86, 34, 98,
   12, 69, 109, 40, 18, 95, 50, 97, 36, 116, 24, 89, 77, 108, 30, 83, 51, 105, 74, 63, 99, 93, 81, 107, 62, 87,
   96, 13, 38, 91, 46, 110, 72, 19, 41, 86, 115, 76, 98, 25, 37, 79, 111, 31, 70, 54, 106, 103, 58, 100, 66, 82,
   94, 88, 97, 37, 14, 108, 70, 93, 48, 41, 20, 116, 87, 98, 77, 38, 26, 109, 81, 32, 52, 72, 107, 99, 64, 105,
   60, 95, 83, 89, 15, 28, 108, 21, 35, 55, 113, 65, 118, 27, 78, 39, 102, 91, 117, 33, 101, 110, 114, 73, 59, 84,
   96, 67, 112, 71, 104, 90, 27, 16, 109, 33, 23, 67, 119, 53, 111, 39, 90, 28, 117, 79, 103, 35, 112, 100, 115, 61,
   71, 96, 86, 73, 106, 65, 110, 91, 17, 110, 30, 56, 112, 22, 36, 117, 63, 80, 29, 104, 40, 118, 93, 101, 34, 108,
   74, 114, 57, 68, 113, 85, 97, 102, 69, 92, 29, 111, 18, 68, 119, 34, 24, 109, 51, 92, 40, 118, 30, 105, 81, 113,
   36, 99, 62, 116, 69, 74, 107, 97, 87, 108, 63, 93, 112, 19, 32, 110, 54, 117, 64, 25, 38, 106, 119, 82, 95, 31,
   41, 100, 109, 37, 72, 58, 115, 111, 66, 103, 70, 88, 98, 94, 113, 31, 20, 118, 66, 108, 52, 37, 26, 119, 107, 94,
   83, 41, 32, 111, 99, 38, 60, 70, 116, 105, 72, 109, 64, 98, 89, 95, 21, 23, 116, 27, 28, 59, 118, 61, 119, 33,
   84, 35, 114, 86, 115, 39, 104, 106, 117, 65, 67, 90, 91, 71, 110, 73, 112, 96, 22, 115, 24, 57, 117, 29, 30, 119,
   62, 85, 34, 114, 36, 116, 87, 102, 40, 107, 63, 118, 68, 69, 108, 92, 93, 113, 74, 97, 114, 25, 26, 117, 58, 118,
   60, 31, 32, 115, 116, 88, 89, 37, 38, 103, 105, 41, 64, 66, 119, 109, 70, 111, 72, 94, 95, 98, 69, 70, 42, 97,
   98, 74, 50, 72, 48, 108, 63, 109, 51, 64, 52, 116, 62, 60, 77, 95, 93, 87, 89, 107, 83, 105, 81, 99, 71, 43,
   72, 73, 49, 96, 98, 46, 70, 65, 110, 53, 111, 54, 66, 61, 115, 58, 94, 76, 91, 106, 82, 86, 88, 79, 103, 100,
   44, 73, 74, 47, 71, 45, 69, 96, 97, 55, 56, 67, 68, 112, 113, 59, 57, 114, 92, 90, 75, 80, 104, 78, 102, 84,
   85, 101, 45, 91, 63, 75, 96, 56, 74, 110, 93, 78, 57, 90, 69, 117, 108, 84, 68, 118, 97, 112, 80, 85, 114, 101,
   113, 104, 92, 102, 90, 46, 64, 96, 76, 112, 95, 54, 72, 91, 117, 79, 109, 58, 70, 86, 119, 66, 98, 82, 110, 115,
   88, 106, 94, 100, 111, 103, 47, 65, 93, 55, 73, 75, 97, 91, 108, 59, 80, 71, 92, 110, 118, 67, 85, 117, 113, 96,
   78, 101, 112, 84, 114, 90, 102, 104, 92, 66, 48, 113, 94, 97, 77, 70, 52, 118, 93, 111, 81, 72, 60, 119, 87, 64,
   83, 98, 108, 107, 95, 116, 89, 109, 99, 105, 67, 49, 95, 71, 53, 90, 109, 76, 98, 73, 112, 61, 119, 82, 94, 65,
   117, 88, 111, 79, 96, 110, 100, 91, 103, 86, 115, 106, 68, 94, 50, 92, 111, 69, 51, 98, 77, 113, 74, 119, 62, 95,
   83, 118, 63, 89, 81, 109, 97, 93, 105, 108, 99, 116, 87, 107, 57, 103, 51, 85, 115, 68, 62, 111, 81, 102, 69, 117,
   63, 109, 99, 114, 74, 105, 87, 119, 92, 97, 116, 113, 107, 118, 93, 108, 102, 58, 52, 114, 88, 113, 83, 66, 60, 117,
   108, 103, 99, 70, 64, 115, 107, 72, 89, 94, 118, 116, 98, 119, 95, 111, 105, 109, 59, 53, 105, 67, 61, 84, 116, 79,
   109, 71, 104, 65, 118, 100, 111, 73, 114, 103, 119, 86, 90, 112, 106, 96, 115, 91, 117, 110, 104, 54, 60, 112, 82, 114,
   89, 58, 64, 110, 118, 100, 105, 66, 72, 106, 116, 70, 95, 88, 117, 119, 94, 115, 98, 103, 109, 111, 55, 61, 107, 59,
   65, 78, 108, 86, 116, 67, 101, 73, 113, 106, 119, 71, 102, 115, 118, 91, 84, 104, 110, 90, 117, 96, 114, 112, 56, 106,
   62, 80, 110, 57, 63, 115, 87, 101, 68, 112, 74, 119, 107, 104, 69, 116, 93, 117, 85, 92, 118, 102, 108, 114, 97, 113,
   75, 86, 87, 78, 91, 80, 93, 106, 107, 84, 85, 96, 97, 115, 116, 90, 92, 119, 108, 110, 101, 102, 117, 104, 118, 112,
   113, 114, 84, 76, 89, 90, 79, 104, 105, 82, 95, 96, 114, 86, 116, 88, 98, 91, 118, 94, 109, 100, 112, 117, 103, 110,
   111, 106, 119, 115, 85, 88, 77, 102, 103, 92, 81, 94, 83, 114, 97, 115, 87, 98, 89, 117, 93, 95, 99, 111, 113, 108,
   109, 118, 105, 119, 107, 116, 78, 79, 99, 84, 86, 101, 107, 100, 105, 90, 102, 91, 108, 103, 109, 96, 113, 111, 116, 106,
   104, 114, 115, 112, 119, 110, 118, 117, 80, 100, 81, 101, 106, 85, 87, 103, 99, 104, 92, 110, 93, 111, 105, 112, 97, 109,
   107, 115, 102, 113, 119, 114, 116, 117, 108, 118, 101, 82, 83, 104, 100, 102, 99, 88, 89, 112, 113, 106, 107, 94, 95, 110,
   108, 98, 105, 103, 114, 118, 111, 117, 109, 115, 116, 119
}; // 2.46 kb
static const cart_index_t iCartYY_ab14_a7[1296] = {
   0, 39, 40, 33, 34, 3, 69, 5, 71, 9, 10, 27, 90, 29, 92, 21, 22, 15, 102, 17, 104, 44, 67, 68, 84, 85,
   45, 57, 47, 59, 55, 56, 101, 75, 78, 80, 39, 1, 41, 4, 70, 35, 37, 73, 7, 28, 91, 11, 13, 94, 31, 16,
   103, 23, 25, 106, 19, 65, 43, 66, 46, 58, 86, 88, 61, 49, 53, 100, 54, 79, 76, 82, 40, 41, 2, 72, 6, 74,
   8, 36, 38, 93, 30, 95, 32, 12, 14, 105, 18, 107, 20, 24, 26, 63, 64, 42, 60, 48, 62, 50, 87, 89, 99, 51,
   52, 81, 83, 77, 33, 4, 72, 11, 98, 39, 70, 71, 43, 35, 96, 16, 46, 111, 66, 23, 115, 28, 58, 110, 54, 73,
   49, 94, 76, 88, 91, 103, 65, 53, 61, 106, 82, 86, 79, 100, 34, 70, 6, 98, 12, 69, 42, 40, 72, 97, 36, 109,
   64, 18, 48, 116, 24, 108, 52, 30, 60, 74, 95, 50, 89, 77, 63, 51, 93, 105, 107, 62, 83, 87, 99, 81, 3, 35,
   74, 39, 69, 9, 97, 44, 73, 15, 45, 33, 96, 68, 113, 27, 57, 21, 114, 56, 112, 47, 71, 92, 90, 102, 75, 85,
   55, 67, 59, 80, 104, 78, 84, 101, 69, 37, 8, 70, 42, 97, 14, 74, 41, 108, 63, 98, 38, 50, 20, 109, 51, 116,
   26, 62, 32, 93, 72, 48, 64, 52, 87, 77, 107, 95, 105, 81, 60, 99, 89, 83, 5, 73, 36, 71, 40, 44, 74, 10,
   96, 47, 17, 67, 112, 34, 97, 59, 29, 55, 113, 22, 114, 45, 90, 69, 104, 92, 56, 68, 75, 84, 78, 57, 102, 80,
   101, 85, 71, 7, 38, 43, 72, 73, 41, 96, 13, 65, 110, 49, 19, 98, 37, 53, 111, 61, 31, 115, 25, 91, 46, 70,
   54, 66, 106, 94, 86, 76, 79, 103, 58, 100, 82, 88, 9, 28, 93, 35, 97, 15, 108, 47, 65, 21, 75, 39, 91, 92,
   118, 33, 85, 27, 117, 80, 110, 55, 73, 113, 96, 114, 78, 102, 59, 71, 67, 101, 112, 84, 90, 104, 10, 91, 30, 96,
   36, 45, 63, 17, 110, 75, 22, 90, 117, 40, 93, 84, 34, 78, 108, 29, 118, 56, 112, 74, 114, 97, 57, 69, 80, 104,
   101, 68, 113, 85, 102, 92, 27, 11, 95, 16, 109, 33, 98, 67, 49, 39, 90, 23, 76, 119, 94, 28, 117, 35, 88, 112,
   82, 71, 53, 111, 79, 103, 96, 115, 73, 61, 65, 110, 100, 91, 86, 106, 90, 13, 32, 46, 64, 96, 38, 112, 19, 91,
   117, 76, 25, 95, 41, 79, 109, 86, 37, 119, 31, 110, 54, 72, 58, 70, 115, 98, 106, 82, 100, 111, 66, 103, 88, 94,
   29, 94, 12, 111, 18, 68, 50, 34, 98, 92, 40, 119, 95, 24, 77, 118, 30, 113, 83, 36, 89, 69, 109, 51, 105, 81,
   74, 62, 97, 116, 108, 63, 99, 93, 107, 87, 92, 31, 14, 66, 48, 113, 20, 97, 37, 118, 93, 94, 41, 77, 26, 111,
   81, 119, 32, 87, 38, 108, 70, 52, 72, 60, 107, 83, 116, 98, 109, 99, 64, 105, 95, 89, 21, 16, 105, 23, 116, 27,
   109, 59, 53, 33, 84, 28, 79, 118, 111, 35, 114, 39, 103, 104, 100, 67, 61, 119, 86, 115, 90, 117, 71, 65, 73, 112,
   106, 96, 91, 110, 22, 103, 18, 115, 24, 57, 51, 29, 111, 85, 34, 117, 109, 30, 81, 114, 36, 102, 99, 40, 105, 68,
   119, 62, 116, 87, 69, 63, 92, 118, 113, 74, 107, 97, 108, 93, 15, 23, 107, 28, 108, 21, 116, 55, 61, 27, 78, 35,
   86, 113, 119, 39, 102, 33, 115, 101, 106, 59, 65, 118, 91, 117, 84, 114, 67, 73, 71, 104, 110, 90, 96, 112, 102, 25,
   20, 58, 52, 114, 26, 113, 31, 117, 108, 88, 37, 83, 32, 103, 99, 115, 38, 107, 41, 118, 66, 60, 70, 64, 116, 89,
   119, 94, 111, 105, 72, 109, 98, 95, 17, 106, 24, 110, 30, 56, 62, 22, 115, 80, 29, 112, 119, 36, 87, 104, 40, 101,
   107, 34, 116, 57, 117, 63, 118, 93, 68, 74, 85, 114, 102, 69, 108, 92, 113, 97, 104, 19, 26, 54, 60, 112, 32, 114,
   25, 110, 118, 82, 31, 89, 38, 100, 105, 106, 41, 116, 37, 117, 58, 64, 66, 72, 119, 95, 115, 88, 103, 109, 70, 111,
   94, 98, 44, 65, 63, 73, 74, 47, 93, 45, 91, 55, 56, 71, 110, 69, 108, 67, 68, 59, 118, 57, 117, 75, 96, 97,
   112, 113, 80, 92, 78, 90, 84, 85, 114, 101, 104, 102, 67, 43, 64, 49, 95, 71, 72, 90, 46, 73, 112, 53, 54, 109,
   70, 61, 119, 65, 66, 117, 58, 96, 76, 98, 82, 94, 110, 111, 91, 79, 86, 115, 88, 106, 100, 103, 68, 66, 42, 94,
   50, 92, 48, 69, 70, 113, 74, 111, 72, 51, 52, 119, 62, 118, 60, 63, 64, 97, 98, 77, 95, 83, 93, 81, 108, 109,
   116, 87, 89, 107, 105, 99, 84, 46, 60, 76, 89, 90, 64, 104, 54, 96, 114, 79, 58, 105, 72, 86, 116, 91, 70, 118,
   66, 112, 82, 95, 88, 98, 117, 109, 110, 100, 106, 119, 94, 115, 103, 111, 85, 58, 48, 88, 77, 102, 52, 92, 66, 114,
   97, 103, 70, 81, 60, 115, 87, 117, 64, 93, 72, 113, 94, 83, 98, 89, 108, 99, 118, 111, 119, 107, 95, 116, 109, 105,
   45, 86, 62, 91, 63, 75, 87, 56, 106, 78, 57, 96, 115, 74, 107, 90, 69, 84, 116, 68, 119, 80, 110, 93, 117, 108,
   85, 97, 101, 112, 104, 92, 118, 102, 114, 113, 57, 88, 50, 103, 51, 85, 77, 68, 94, 102, 69, 115, 98, 62, 83, 117,
   63, 114, 89, 74, 95, 92, 111, 81, 109, 99, 97, 87, 113, 119, 118, 93, 105, 108, 116, 107, 47, 61, 87, 65, 93, 55,
   107, 75, 86, 59, 80, 73, 106, 97, 116, 71, 92, 67, 119, 85, 115, 78, 91, 108, 110, 118, 101, 113, 84, 96, 90, 102,
   117, 104, 112, 114, 59, 49, 89, 53, 105, 67, 95, 84, 76, 71, 104, 61, 82, 116, 98, 65, 118, 73, 94, 114, 88, 90,
   79, 109, 100, 111, 112, 119, 96, 86, 91, 117, 103, 110, 106, 115, 55, 53, 99, 61, 107, 59, 105, 78, 79, 67, 101, 65,
   100, 108, 109, 73, 113, 71, 111, 102, 103, 84, 86, 116, 106, 119, 104, 118, 90, 91, 96, 114, 115, 112, 110, 117, 56, 100,
   51, 106, 62, 80, 81, 57, 103, 101, 68, 110, 111, 63, 99, 112, 74, 104, 105, 69, 109, 85, 115, 87, 119, 107, 92, 93,
   102, 117, 114, 97, 116, 113, 118, 108, 101, 54, 52, 82, 83, 104, 60, 102, 58, 112, 113, 100, 66, 99, 64, 106, 107, 110,
   72, 108, 70, 114, 88, 89, 94, 95, 118, 105, 117, 103, 115, 116, 98, 119, 111, 109, 75, 79, 81, 86, 87, 78, 99, 80,
   100, 84, 85, 91, 103, 93, 105, 96, 97, 90, 109, 92, 111, 101, 106, 107, 115, 116, 102, 108, 104, 110, 112, 113, 119, 114,
   117, 118, 78, 76, 83, 79, 99, 84, 89, 101, 82, 90, 102, 86, 88, 107, 95, 91, 108, 96, 98, 113, 94, 104, 100, 105,
   103, 109, 114, 116, 112, 106, 110, 118, 111, 117, 115, 119, 80, 82, 77, 100, 81, 101, 83, 85, 88, 104, 92, 106, 94, 87,
   89, 110, 93, 112, 95, 97, 98, 102, 103, 99, 111, 105, 113, 107, 114, 115, 117, 108, 109, 118, 119, 116
}; // 2.53 kb
static const cart_index_t iCartYY_ab14_a8[1260] = {
   0, 35, 36, 3, 39, 5, 40, 73, 74, 9, 10, 33, 34, 96, 97, 15, 27, 17, 29, 112, 113, 21, 22, 114, 69, 71,
   44, 45, 90, 47, 92, 67, 68, 57, 102, 59, 104, 55, 56, 75, 84, 85, 78, 80, 101, 33, 1, 38, 39, 4, 71, 72,
   7, 41, 35, 96, 11, 98, 13, 37, 28, 16, 110, 111, 19, 31, 23, 115, 25, 70, 43, 73, 91, 46, 65, 66, 49, 94,
   103, 58, 53, 54, 61, 106, 86, 76, 88, 79, 100, 82, 34, 37, 2, 69, 70, 40, 6, 41, 8, 97, 36, 98, 12, 38,
   14, 108, 109, 30, 18, 32, 20, 116, 24, 26, 42, 72, 74, 63, 64, 93, 48, 95, 50, 51, 52, 105, 60, 107, 62, 87,
   89, 77, 99, 81, 83, 3, 28, 63, 9, 35, 44, 74, 65, 93, 15, 45, 39, 69, 91, 108, 21, 33, 56, 68, 110, 118,
   27, 57, 117, 97, 73, 47, 75, 96, 55, 113, 71, 92, 85, 114, 67, 112, 59, 80, 78, 90, 102, 84, 101, 104, 27, 4,
   64, 33, 11, 67, 95, 43, 72, 39, 90, 16, 109, 46, 70, 35, 23, 112, 119, 54, 66, 28, 117, 58, 98, 49, 71, 96,
   76, 73, 94, 53, 111, 115, 88, 61, 82, 65, 110, 91, 79, 103, 86, 106, 100, 5, 65, 30, 44, 73, 10, 36, 91, 63,
   47, 17, 71, 40, 110, 93, 55, 67, 22, 34, 117, 108, 59, 29, 118, 74, 96, 45, 56, 112, 75, 97, 90, 69, 68, 113,
   84, 114, 78, 57, 80, 104, 92, 101, 85, 102, 29, 66, 6, 68, 94, 34, 12, 70, 42, 92, 40, 111, 18, 72, 48, 113,
   119, 36, 24, 64, 52, 118, 30, 60, 50, 98, 69, 74, 95, 97, 77, 109, 51, 62, 83, 116, 89, 108, 63, 93, 105, 81,
   107, 87, 99, 67, 7, 32, 71, 43, 90, 64, 13, 38, 73, 112, 49, 95, 19, 41, 65, 53, 117, 109, 25, 37, 61, 119,
   31, 72, 46, 96, 110, 54, 91, 70, 76, 98, 111, 66, 79, 58, 86, 115, 106, 82, 94, 100, 103, 88, 68, 31, 8, 92,
   66, 69, 42, 37, 14, 113, 74, 94, 50, 41, 20, 118, 111, 63, 51, 38, 26, 119, 62, 32, 48, 70, 97, 93, 72, 108,
   52, 98, 77, 81, 60, 109, 64, 116, 87, 107, 95, 83, 105, 99, 89, 9, 23, 87, 15, 28, 47, 93, 61, 107, 21, 75,
   35, 97, 86, 116, 27, 39, 80, 92, 106, 119, 33, 85, 115, 108, 65, 55, 78, 91, 59, 118, 73, 113, 102, 117, 71, 110,
   67, 101, 84, 96, 114, 90, 104, 112, 10, 86, 24, 45, 91, 17, 30, 106, 62, 75, 22, 96, 36, 115, 87, 78, 90, 29,
   40, 119, 107, 84, 34, 116, 63, 110, 56, 57, 117, 80, 93, 112, 74, 69, 108, 104, 118, 101, 68, 85, 114, 97, 102, 92,
   113, 21, 11, 89, 27, 16, 59, 105, 49, 95, 33, 84, 23, 116, 76, 98, 39, 28, 104, 118, 82, 94, 35, 114, 88, 109,
   53, 67, 90, 79, 71, 111, 61, 119, 117, 103, 65, 100, 73, 112, 96, 86, 115, 91, 110, 106, 22, 88, 12, 57, 103, 29,
   18, 94, 50, 85, 34, 115, 24, 98, 77, 102, 117, 40, 30, 95, 83, 114, 36, 89, 51, 111, 68, 69, 109, 92, 81, 119,
   62, 63, 99, 118, 105, 113, 74, 97, 116, 87, 108, 93, 107, 84, 13, 26, 90, 46, 104, 60, 19, 32, 96, 114, 76, 89,
   25, 38, 91, 79, 118, 105, 31, 41, 86, 116, 37, 64, 54, 112, 117, 58, 110, 72, 82, 95, 109, 70, 100, 66, 106, 119,
   115, 88, 98, 103, 111, 94, 85, 25, 14, 102, 58, 92, 48, 31, 20, 114, 97, 88, 77, 37, 26, 117, 103, 93, 81, 41,
   32, 115, 87, 38, 52, 66, 113, 108, 70, 118, 60, 94, 83, 99, 64, 111, 72, 119, 107, 116, 98, 89, 109, 105, 95, 15,
   16, 99, 21, 23, 55, 107, 53, 105, 27, 78, 28, 108, 79, 109, 33, 35, 101, 113, 100, 111, 39, 102, 103, 116, 61, 59,
   84, 86, 67, 119, 65, 118, 114, 115, 73, 106, 71, 104, 90, 91, 117, 96, 112, 110, 17, 100, 18, 56, 106, 22, 24, 103,
   51, 80, 29, 110, 30, 111, 81, 101, 112, 34, 36, 109, 99, 104, 40, 105, 62, 115, 57, 68, 119, 85, 87, 117, 63, 74,
   107, 114, 116, 102, 69, 92, 118, 93, 113, 97, 108, 101, 19, 20, 104, 54, 102, 52, 25, 26, 112, 113, 82, 83, 31, 32,
   110, 100, 108, 99, 37, 38, 106, 107, 41, 60, 58, 114, 118, 66, 117, 64, 88, 89, 105, 72, 103, 70, 115, 116, 119, 94,
   95, 111, 109, 98, 57, 58, 42, 85, 88, 68, 50, 66, 48, 102, 69, 103, 51, 70, 52, 114, 115, 74, 62, 72, 60, 117,
   63, 64, 77, 94, 92, 97, 98, 113, 83, 111, 81, 87, 89, 119, 95, 118, 93, 108, 109, 99, 116, 107, 105, 59, 43, 60,
   67, 49, 84, 89, 46, 64, 71, 104, 53, 105, 54, 72, 73, 61, 114, 116, 58, 70, 65, 118, 66, 95, 76, 90, 112, 82,
   96, 98, 79, 109, 119, 94, 86, 88, 91, 117, 110, 100, 111, 106, 115, 103, 44, 61, 62, 47, 65, 45, 63, 86, 87, 55,
   56, 73, 74, 106, 107, 59, 71, 57, 69, 115, 116, 67, 68, 119, 93, 91, 75, 80, 110, 78, 108, 96, 97, 92, 118, 90,
   117, 84, 85, 101, 112, 113, 104, 102, 114, 45, 79, 51, 75, 86, 56, 62, 100, 81, 78, 57, 91, 63, 103, 99, 84, 96,
   68, 74, 111, 105, 90, 69, 109, 87, 106, 80, 85, 115, 101, 107, 110, 93, 97, 116, 112, 119, 104, 92, 102, 117, 108, 114,
   113, 118, 78, 46, 52, 84, 76, 101, 83, 54, 60, 90, 102, 79, 99, 58, 64, 96, 86, 113, 107, 66, 72, 91, 108, 70,
   89, 82, 104, 114, 88, 112, 95, 100, 105, 116, 98, 106, 94, 110, 118, 117, 103, 109, 115, 119, 111, 47, 53, 81, 55, 61,
   75, 87, 79, 99, 59, 80, 65, 93, 100, 105, 67, 73, 85, 97, 103, 109, 71, 92, 111, 107, 86, 78, 101, 106, 84, 116,
   91, 108, 113, 119, 96, 115, 90, 102, 104, 110, 118, 112, 114, 117, 80, 54, 48, 101, 82, 85, 77, 58, 52, 104, 92, 100,
   81, 66, 60, 112, 106, 97, 87, 70, 64, 110, 93, 72, 83, 88, 102, 113, 94, 114, 89, 103, 99, 107, 95, 115, 98, 117,
   108, 118, 111, 105, 119, 116, 109, 55, 49, 83, 59, 53, 78, 99, 76, 89, 67, 101, 61, 107, 82, 95, 71, 65, 102, 108,
   88, 98, 73, 113, 94, 105, 79, 84, 104, 100, 90, 109, 86, 116, 118, 111, 91, 103, 96, 114, 112, 106, 119, 110, 117, 115,
   56, 82, 50, 80, 100, 57, 51, 88, 77, 101, 68, 106, 62, 94, 83, 104, 110, 69, 63, 98, 89, 112, 74, 95, 81, 103,
   85, 92, 111, 102, 99, 115, 87, 93, 105, 117, 109, 114, 97, 113, 119, 107, 118, 108, 116, 75, 76, 77, 78, 79, 80, 81,
   82, 83, 84, 85, 86, 87, 88, 89, 90, 91, 92, 93, 94, 95, 96, 97, 98, 99, 100, 101, 102, 103, 104, 105, 106, 107,
   108, 109, 110, 111, 112, 113, 114, 115, 116, 117, 118, 119
}; // 2.46 kb
static const cart_index_t iCartYY_ab14_a9[1155] = {
   0, 28, 30, 35, 36, 3, 63, 5, 65, 9, 10, 39, 91, 40, 93, 33, 34, 15, 108, 17, 110, 21, 22, 27, 117, 29,
   118, 44, 73, 74, 96, 97, 45, 69, 47, 71, 55, 56, 67, 68, 112, 113, 114, 57, 59, 75, 90, 92, 84, 85, 78, 102,
   80, 104, 101, 27, 1, 32, 4, 64, 33, 38, 67, 7, 39, 90, 11, 13, 95, 41, 16, 109, 35, 37, 112, 19, 28, 117,
   23, 25, 119, 31, 71, 43, 72, 46, 70, 96, 98, 73, 49, 65, 110, 53, 111, 54, 66, 58, 115, 61, 91, 76, 94, 79,
   103, 86, 88, 106, 82, 100, 29, 31, 2, 66, 6, 68, 8, 34, 37, 92, 40, 94, 41, 12, 14, 111, 18, 113, 20, 36,
   38, 118, 30, 119, 32, 24, 26, 69, 70, 42, 72, 48, 74, 50, 97, 98, 108, 63, 109, 51, 64, 52, 60, 62, 116, 93,
   95, 77, 105, 81, 107, 83, 87, 89, 99, 21, 4, 60, 11, 89, 27, 64, 59, 43, 33, 84, 16, 46, 105, 72, 23, 116,
   39, 70, 104, 54, 35, 114, 28, 58, 118, 66, 67, 49, 95, 76, 98, 90, 109, 71, 53, 73, 112, 61, 119, 82, 94, 88,
   117, 65, 96, 79, 111, 86, 115, 91, 103, 110, 100, 106, 22, 58, 6, 88, 12, 57, 42, 29, 66, 85, 34, 103, 70, 18,
   48, 115, 24, 102, 52, 40, 72, 114, 36, 117, 64, 30, 60, 68, 94, 50, 98, 77, 69, 51, 92, 111, 113, 74, 119, 62,
   95, 83, 89, 63, 118, 97, 109, 81, 116, 87, 108, 99, 93, 105, 107, 3, 23, 62, 28, 63, 9, 87, 44, 61, 15, 45,
   35, 86, 74, 107, 39, 69, 21, 116, 56, 106, 27, 57, 33, 115, 68, 119, 47, 65, 93, 91, 108, 75, 97, 55, 73, 59,
   80, 71, 92, 110, 118, 117, 85, 67, 78, 96, 113, 90, 102, 84, 114, 101, 112, 104, 57, 25, 8, 58, 42, 85, 14, 68,
   31, 102, 69, 88, 37, 50, 20, 103, 51, 114, 26, 74, 41, 117, 63, 115, 38, 62, 32, 92, 66, 48, 70, 52, 97, 77,
   113, 94, 118, 93, 111, 81, 72, 60, 64, 87, 119, 108, 98, 83, 109, 99, 116, 89, 107, 95, 105, 5, 61, 24, 65, 30,
   44, 62, 10, 86, 47, 17, 73, 106, 36, 87, 71, 40, 55, 107, 22, 115, 59, 29, 67, 119, 34, 116, 45, 91, 63, 110,
   93, 56, 74, 75, 96, 78, 57, 90, 69, 117, 108, 118, 68, 84, 80, 112, 97, 104, 92, 101, 113, 85, 114, 102, 59, 7,
   26, 43, 60, 67, 32, 84, 13, 71, 104, 49, 19, 89, 38, 53, 105, 73, 41, 114, 25, 65, 118, 61, 31, 116, 37, 90,
   46, 64, 54, 72, 112, 95, 96, 76, 91, 117, 79, 109, 58, 70, 66, 119, 86, 110, 82, 98, 100, 111, 106, 94, 115, 88,
   103, 9, 16, 81, 23, 87, 15, 99, 47, 53, 21, 75, 28, 79, 93, 105, 35, 97, 27, 109, 80, 100, 33, 85, 39, 103,
   92, 111, 55, 61, 107, 86, 116, 78, 108, 59, 65, 67, 101, 73, 113, 106, 119, 115, 102, 71, 84, 91, 118, 96, 114, 90,
   117, 104, 110, 112, 10, 79, 18, 86, 24, 45, 51, 17, 100, 75, 22, 91, 103, 30, 81, 96, 36, 78, 99, 29, 111, 84,
   34, 90, 109, 40, 105, 56, 106, 62, 115, 87, 57, 63, 80, 110, 101, 68, 112, 74, 119, 107, 116, 69, 104, 85, 117, 93,
   114, 97, 102, 108, 92, 118, 113, 15, 11, 83, 16, 99, 21, 89, 55, 49, 27, 78, 23, 76, 107, 95, 28, 108, 33, 98,
   101, 82, 39, 102, 35, 88, 113, 94, 59, 53, 105, 79, 109, 84, 116, 67, 61, 71, 104, 65, 118, 100, 111, 103, 114, 73,
   90, 86, 119, 91, 117, 96, 115, 112, 106, 110, 78, 13, 20, 46, 52, 84, 26, 101, 19, 90, 102, 76, 25, 83, 32, 79,
   99, 96, 38, 113, 31, 91, 108, 86, 37, 107, 41, 104, 54, 60, 58, 64, 114, 89, 112, 82, 110, 118, 100, 105, 66, 72,
   70, 116, 106, 117, 88, 95, 103, 109, 115, 98, 119, 94, 111, 17, 82, 12, 100, 18, 56, 50, 22, 88, 80, 29, 106, 94,
   24, 77, 110, 30, 101, 83, 34, 98, 104, 40, 112, 95, 36, 89, 57, 103, 51, 111, 81, 68, 62, 85, 115, 102, 69, 117,
   63, 109, 99, 105, 74, 114, 92, 119, 87, 118, 93, 113, 107, 97, 116, 108, 80, 19, 14, 54, 48, 101, 20, 85, 25, 104,
   92, 82, 31, 77, 26, 100, 81, 112, 32, 97, 37, 110, 93, 106, 41, 87, 38, 102, 58, 52, 66, 60, 113, 83, 114, 88,
   117, 108, 103, 99, 70, 64, 72, 107, 115, 118, 94, 89, 111, 105, 119, 95, 116, 98, 109, 44, 53, 51, 61, 62, 47, 81,
   45, 79, 55, 56, 65, 100, 63, 99, 73, 74, 59, 105, 57, 103, 67, 68, 71, 111, 69, 109, 75, 86, 87, 106, 107, 80,
   93, 78, 91, 84, 85, 96, 97, 115, 116, 119, 92, 90, 101, 110, 108, 112, 113, 104, 118, 102, 117, 114, 55, 43, 52, 49,
   83, 59, 60, 78, 46, 67, 101, 53, 54, 99, 64, 61, 107, 71, 72, 102, 58, 73, 113, 65, 66, 108, 70, 84, 76, 89,
   82, 95, 104, 105, 90, 79, 96, 114, 86, 116, 88, 98, 94, 118, 91, 112, 100, 109, 106, 119, 110, 111, 117, 103, 115, 56,
   54, 42, 82, 50, 80, 48, 57, 58, 101, 68, 100, 66, 51, 52, 106, 62, 104, 60, 69, 70, 112, 74, 110, 72, 63, 64,
   85, 88, 77, 94, 83, 92, 81, 102, 103, 114, 97, 115, 87, 98, 89, 95, 93, 117, 113, 111, 99, 119, 107, 118, 105, 108,
   109, 116, 75, 46, 48, 76, 77, 78, 52, 80, 54, 84, 85, 79, 58, 81, 60, 86, 87, 90, 64, 92, 66, 96, 97, 91,
   70, 93, 72, 101, 82, 83, 88, 89, 102, 99, 104, 100, 112, 113, 106, 107, 94, 95, 98, 108, 110, 114, 103, 105, 115, 116,
   117, 109, 118, 111, 119, 45, 76, 50, 79, 51, 75, 77, 56, 82, 78, 57, 86, 88, 62, 83, 91, 63, 84, 89, 68, 94,
   90, 69, 96, 98, 74, 95, 80, 100, 81, 103, 99, 85, 87, 101, 106, 104, 92, 110, 93, 111, 105, 109, 97, 112, 102, 115,
   107, 117, 108, 114, 116, 113, 119, 118, 47, 49, 77, 53, 81, 55, 83, 75, 76, 59, 80, 61, 82, 87, 89, 65, 93, 67,
   95, 85, 88, 71, 92, 73, 94, 97, 98, 78, 79, 99, 100, 105, 101, 107, 84, 86, 90, 102, 91, 108, 103, 109, 111, 113,
   96, 104, 106, 116, 110, 118, 112, 119, 114, 115, 117
}; // 2.26 kb
static const cart_index_t iCartYY_ab14_a10[990] = {
   0, 23, 24, 3, 28, 5, 30, 61, 62, 9, 10, 35, 36, 86, 87, 15, 39, 17, 40, 106, 107, 21, 22, 33, 34, 115,
   116, 27, 29, 119, 63, 65, 44, 45, 91, 47, 93, 73, 74, 69, 108, 71, 110, 55, 56, 57, 117, 59, 118, 67, 68, 75,
   96, 97, 78, 90, 80, 92, 112, 113, 84, 85, 114, 102, 104, 101, 21, 1, 26, 27, 4, 59, 60, 7, 32, 33, 84, 11,
   89, 13, 38, 39, 16, 104, 105, 19, 41, 35, 114, 23, 116, 25, 37, 28, 118, 31, 64, 43, 67, 90, 46, 71, 72, 49,
   95, 109, 70, 53, 54, 73, 112, 117, 58, 65, 66, 61, 119, 96, 76, 98, 91, 79, 110, 111, 82, 94, 86, 115, 88, 103,
   100, 106, 22, 25, 2, 57, 58, 29, 6, 31, 8, 85, 34, 88, 12, 37, 14, 102, 103, 40, 18, 41, 20, 114, 36, 115,
   24, 38, 26, 117, 30, 32, 42, 66, 68, 69, 70, 92, 48, 94, 50, 51, 52, 111, 72, 113, 74, 63, 64, 118, 60, 119,
   62, 97, 98, 77, 108, 109, 93, 81, 95, 83, 116, 87, 89, 99, 105, 107, 3, 16, 51, 9, 23, 44, 62, 53, 81, 15,
   45, 28, 63, 79, 99, 21, 35, 56, 74, 100, 105, 27, 57, 39, 69, 103, 109, 33, 68, 111, 87, 61, 47, 75, 86, 55,
   107, 65, 93, 97, 116, 73, 106, 59, 80, 85, 115, 67, 119, 71, 92, 78, 91, 108, 84, 96, 101, 113, 110, 118, 90, 102,
   117, 114, 112, 104, 15, 4, 52, 21, 11, 55, 83, 43, 60, 27, 78, 16, 99, 46, 64, 33, 23, 101, 107, 54, 72, 39,
   102, 28, 108, 58, 70, 35, 113, 66, 89, 49, 59, 84, 76, 67, 95, 53, 105, 116, 98, 61, 82, 71, 104, 114, 88, 73,
   94, 65, 118, 90, 79, 109, 96, 86, 112, 119, 100, 111, 91, 117, 103, 115, 106, 110, 5, 53, 18, 44, 61, 10, 24, 79,
   51, 47, 17, 65, 30, 100, 81, 55, 73, 22, 36, 103, 99, 59, 29, 71, 40, 111, 105, 67, 34, 109, 62, 86, 45, 56,
   106, 75, 87, 91, 63, 74, 107, 96, 115, 78, 57, 68, 119, 84, 116, 90, 69, 80, 110, 93, 101, 112, 85, 97, 117, 108,
   104, 92, 118, 113, 114, 102, 17, 54, 6, 56, 82, 22, 12, 58, 42, 80, 29, 100, 18, 66, 48, 101, 106, 34, 24, 70,
   52, 104, 40, 110, 30, 72, 60, 112, 36, 64, 50, 88, 57, 68, 94, 85, 77, 103, 51, 62, 83, 115, 98, 102, 69, 74,
   95, 114, 89, 117, 63, 92, 111, 81, 113, 119, 97, 87, 109, 99, 118, 93, 105, 107, 116, 108, 55, 7, 20, 59, 43, 78,
   52, 13, 26, 67, 101, 49, 83, 19, 32, 71, 53, 102, 99, 25, 38, 73, 113, 61, 107, 31, 41, 65, 108, 37, 60, 46,
   84, 104, 54, 90, 64, 76, 89, 105, 72, 79, 58, 96, 114, 118, 66, 91, 70, 86, 116, 112, 82, 95, 110, 100, 117, 109,
   88, 98, 106, 119, 94, 111, 103, 115, 56, 19, 8, 80, 54, 57, 42, 25, 14, 101, 68, 82, 50, 31, 20, 104, 100, 69,
   51, 37, 26, 112, 74, 106, 62, 41, 32, 110, 63, 38, 48, 58, 85, 92, 66, 102, 52, 88, 77, 81, 60, 103, 70, 114,
   97, 93, 72, 117, 64, 115, 87, 113, 94, 83, 118, 111, 108, 99, 98, 89, 119, 107, 95, 105, 109, 116, 9, 11, 77, 15,
   16, 47, 81, 49, 83, 21, 75, 23, 87, 76, 89, 27, 28, 80, 93, 82, 95, 33, 85, 35, 97, 88, 98, 39, 92, 94,
   99, 53, 55, 78, 79, 59, 105, 61, 107, 108, 109, 65, 100, 67, 101, 102, 103, 71, 111, 73, 113, 84, 86, 116, 90, 91,
   104, 118, 106, 119, 96, 114, 115, 117, 110, 112, 10, 76, 12, 45, 79, 17, 18, 82, 50, 75, 22, 86, 24, 88, 77, 78,
   91, 29, 30, 94, 83, 84, 34, 96, 36, 98, 89, 90, 40, 95, 51, 100, 56, 57, 103, 80, 81, 106, 62, 63, 99, 110,
   111, 101, 68, 69, 109, 104, 105, 112, 74, 85, 115, 87, 102, 117, 92, 93, 119, 107, 114, 97, 116, 108, 118, 113, 75, 13,
   14, 78, 46, 80, 48, 19, 20, 84, 85, 76, 77, 25, 26, 90, 79, 92, 81, 31, 32, 96, 97, 86, 87, 37, 38, 91,
   93, 41, 52, 54, 101, 102, 58, 104, 60, 82, 83, 99, 64, 100, 66, 112, 113, 108, 70, 110, 72, 106, 107, 114, 88, 89,
   117, 103, 118, 105, 94, 95, 115, 116, 98, 109, 111, 119, 45, 46, 42, 75, 76, 56, 50, 54, 48, 78, 57, 79, 51, 58,
   52, 84, 86, 68, 62, 66, 60, 90, 69, 91, 63, 70, 64, 96, 74, 72, 77, 82, 80, 85, 88, 101, 83, 100, 81, 87,
   89, 106, 94, 104, 92, 97, 98, 112, 95, 110, 93, 102, 103, 99, 114, 115, 113, 107, 111, 105, 117, 108, 109, 116, 119, 118,
   47, 43, 48, 55, 49, 75, 77, 46, 52, 59, 80, 53, 81, 54, 60, 67, 61, 85, 87, 58, 64, 71, 92, 65, 93, 66,
   72, 73, 97, 70, 83, 76, 78, 101, 82, 84, 89, 79, 99, 107, 95, 86, 88, 90, 102, 113, 94, 96, 98, 91, 108, 104,
   100, 105, 112, 106, 114, 116, 103, 109, 110, 118, 111, 119, 115, 117, 44, 49, 50, 47, 53, 45, 51, 76, 77, 55, 56, 61,
   62, 82, 83, 59, 65, 57, 63, 88, 89, 67, 68, 73, 74, 94, 95, 71, 69, 98, 81, 79, 75, 80, 100, 78, 99, 86,
   87, 93, 105, 91, 103, 84, 85, 92, 111, 90, 109, 96, 97, 101, 106, 107, 104, 110, 102, 108, 115, 116, 112, 113, 119, 118,
   117, 114
}; // 1.93 kb
static const cart_index_t iCartYY_ab14_a11[780] = {
   0, 16, 18, 23, 24, 3, 51, 5, 53, 9, 10, 28, 79, 30, 81, 35, 36, 15, 99, 17, 100, 21, 22, 39, 103, 40,
   105, 33, 34, 27, 109, 29, 111, 44, 61, 62, 86, 87, 45, 63, 47, 65, 55, 56, 73, 74, 106, 107, 115, 116, 57, 69,
   59, 71, 67, 68, 119, 75, 91, 93, 96, 97, 78, 108, 80, 110, 84, 85, 90, 117, 92, 118, 101, 112, 113, 114, 102, 104,
   15, 1, 20, 4, 52, 21, 26, 55, 7, 27, 78, 11, 13, 83, 32, 16, 99, 33, 38, 101, 19, 39, 102, 23, 25, 107,
   41, 28, 108, 35, 37, 113, 31, 59, 43, 60, 46, 64, 84, 89, 67, 49, 71, 104, 53, 105, 54, 72, 58, 70, 114, 116,
   73, 61, 65, 118, 66, 90, 76, 95, 79, 109, 96, 98, 112, 82, 91, 117, 86, 88, 119, 94, 110, 100, 111, 103, 115, 106,
   17, 19, 2, 54, 6, 56, 8, 22, 25, 80, 29, 82, 31, 12, 14, 100, 18, 101, 20, 34, 37, 104, 40, 106, 41, 24,
   26, 110, 30, 112, 32, 36, 38, 57, 58, 42, 66, 48, 68, 50, 85, 88, 102, 69, 103, 51, 70, 52, 72, 60, 74, 62,
   114, 115, 117, 63, 64, 92, 94, 77, 111, 81, 113, 83, 97, 98, 118, 93, 119, 95, 87, 89, 108, 109, 99, 105, 107, 116,
   9, 4, 48, 11, 77, 15, 52, 47, 43, 21, 75, 16, 46, 81, 60, 23, 87, 27, 64, 80, 54, 33, 85, 28, 58, 93,
   72, 35, 97, 39, 70, 92, 66, 55, 49, 83, 76, 89, 78, 99, 59, 53, 67, 101, 61, 107, 82, 95, 88, 98, 102, 108,
   71, 65, 73, 113, 94, 84, 79, 105, 86, 116, 90, 109, 104, 100, 96, 114, 91, 103, 118, 111, 112, 106, 119, 115, 117, 110,
   10, 46, 6, 76, 12, 45, 42, 17, 54, 75, 22, 79, 58, 18, 48, 86, 24, 78, 52, 29, 66, 84, 34, 91, 70, 30,
   60, 96, 36, 90, 64, 40, 72, 56, 82, 50, 88, 77, 57, 51, 80, 100, 101, 68, 106, 62, 94, 83, 98, 89, 69, 63,
   104, 110, 112, 74, 95, 85, 103, 81, 115, 87, 102, 99, 92, 111, 114, 97, 117, 109, 93, 105, 113, 119, 107, 116, 108, 118,
   3, 11, 50, 16, 51, 9, 77, 44, 49, 15, 45, 23, 76, 62, 83, 28, 63, 21, 89, 56, 82, 27, 57, 35, 88, 74,
   95, 39, 69, 33, 98, 68, 94, 47, 53, 81, 79, 99, 75, 87, 55, 61, 59, 80, 65, 93, 100, 105, 103, 109, 85, 97,
   67, 73, 71, 92, 111, 78, 86, 107, 91, 108, 84, 116, 101, 106, 90, 102, 96, 115, 113, 119, 104, 110, 118, 117, 114, 112,
   45, 13, 8, 46, 42, 75, 14, 56, 19, 78, 57, 76, 25, 50, 20, 79, 51, 84, 26, 68, 31, 90, 69, 86, 37, 62,
   32, 91, 63, 96, 38, 74, 41, 80, 54, 48, 58, 52, 85, 77, 101, 82, 104, 92, 100, 81, 66, 60, 70, 64, 97, 87,
   112, 106, 110, 93, 72, 102, 88, 83, 103, 99, 114, 89, 113, 94, 117, 108, 115, 98, 107, 95, 118, 111, 105, 109, 116, 119,
   5, 49, 12, 53, 18, 44, 50, 10, 76, 47, 17, 61, 82, 24, 77, 65, 30, 55, 83, 22, 88, 59, 29, 73, 94, 36,
   89, 71, 40, 67, 95, 34, 98, 45, 79, 51, 100, 81, 56, 62, 75, 86, 78, 57, 91, 63, 103, 99, 111, 105, 68, 74,
   84, 96, 90, 69, 109, 80, 106, 87, 110, 93, 101, 107, 85, 115, 104, 92, 112, 119, 97, 116, 102, 117, 108, 118, 113, 114,
   47, 7, 14, 43, 48, 55, 20, 75, 13, 59, 80, 49, 19, 77, 26, 53, 81, 67, 32, 85, 25, 71, 92, 61, 31, 87,
   38, 65, 93, 73, 41, 97, 37, 78, 46, 52, 54, 60, 101, 83, 84, 76, 90, 102, 79, 99, 58, 64, 66, 72, 113, 107,
   96, 86, 91, 108, 70, 104, 82, 89, 100, 105, 112, 95, 114, 88, 110, 118, 106, 94, 116, 98, 117, 103, 109, 111, 119, 115,
   44, 43, 42, 49, 50, 47, 48, 45, 46, 55, 56, 53, 54, 51, 52, 61, 62, 59, 60, 57, 58, 67, 68, 65, 66, 63,
   64, 73, 74, 71, 72, 69, 70, 75, 76, 77, 82, 83, 80, 81, 78, 79, 84, 85, 86, 87, 88, 89, 94, 95, 92, 93,
   90, 91, 96, 97, 98, 101, 100, 99, 106, 107, 104, 105, 102, 103, 112, 113, 110, 111, 108, 109, 114, 115, 116, 119, 118, 117
}; // 1.52 kb
static const cart_index_t iCartYY_ab14_a12[546] = {
   0, 11, 12, 3, 16, 5, 18, 49, 50, 9, 10, 23, 24, 76, 77, 15, 28, 17, 30, 82, 83, 21, 22, 35, 36, 88,
   89, 27, 39, 29, 40, 94, 95, 33, 34, 98, 51, 53, 44, 45, 79, 47, 81, 61, 62, 63, 99, 65, 100, 55, 56, 57,
   103, 59, 105, 73, 74, 69, 109, 71, 111, 67, 68, 75, 86, 87, 78, 91, 80, 93, 106, 107, 84, 85, 96, 97, 115, 116,
   90, 92, 119, 108, 110, 101, 102, 117, 104, 118, 112, 113, 114, 9, 1, 14, 15, 4, 47, 48, 7, 20, 21, 75, 11, 77,
   13, 26, 27, 16, 80, 81, 19, 32, 33, 85, 23, 87, 25, 38, 39, 28, 92, 93, 31, 41, 35, 97, 37, 52, 43, 55,
   78, 46, 59, 60, 49, 83, 99, 64, 53, 54, 67, 101, 102, 58, 71, 72, 61, 107, 108, 70, 65, 66, 73, 113, 84, 76,
   89, 90, 79, 104, 105, 82, 95, 96, 114, 86, 116, 88, 98, 91, 118, 94, 109, 100, 112, 117, 103, 110, 111, 106, 119, 115,
   10, 13, 2, 45, 46, 17, 6, 19, 8, 75, 22, 76, 12, 25, 14, 78, 79, 29, 18, 31, 20, 84, 34, 86, 24, 37,
   26, 90, 91, 40, 30, 41, 32, 96, 36, 38, 42, 54, 56, 57, 58, 80, 48, 82, 50, 51, 52, 100, 66, 101, 68, 69,
   70, 104, 60, 106, 62, 63, 64, 110, 72, 112, 74, 85, 88, 77, 102, 103, 92, 81, 94, 83, 114, 97, 115, 87, 98, 89,
   117, 93, 95, 99, 111, 113, 108, 109, 118, 105, 119, 107, 116, 3, 4, 42, 9, 11, 44, 50, 43, 48, 15, 45, 16, 51,
   46, 52, 21, 23, 56, 62, 54, 60, 27, 57, 28, 63, 58, 64, 33, 35, 68, 74, 66, 72, 39, 69, 70, 77, 49, 47,
   75, 76, 55, 83, 53, 81, 87, 89, 61, 82, 59, 80, 85, 88, 67, 95, 65, 93, 97, 98, 73, 94, 71, 92, 78, 79,
   99, 84, 86, 101, 107, 100, 105, 90, 102, 91, 108, 103, 109, 96, 113, 111, 116, 106, 104, 114, 115, 112, 119, 110, 118, 117,
   5, 43, 6, 44, 49, 10, 12, 46, 42, 47, 17, 53, 18, 54, 48, 55, 61, 22, 24, 58, 52, 59, 29, 65, 30, 66,
   60, 67, 73, 34, 36, 70, 64, 71, 40, 72, 50, 76, 45, 56, 82, 75, 77, 79, 51, 62, 83, 86, 88, 78, 57, 68,
   94, 84, 89, 91, 63, 74, 95, 96, 98, 90, 69, 80, 100, 81, 101, 106, 85, 87, 103, 99, 104, 92, 110, 93, 111, 105,
   112, 97, 109, 107, 115, 102, 113, 119, 114, 116, 117, 108, 118, 44, 7, 8, 47, 43, 45, 42, 13, 14, 55, 56, 49, 50,
   19, 20, 59, 53, 57, 51, 25, 26, 67, 68, 61, 62, 31, 32, 71, 65, 69, 63, 37, 38, 73, 74, 41, 48, 46, 75,
   80, 54, 78, 52, 76, 77, 81, 60, 79, 58, 84, 85, 92, 66, 90, 64, 86, 87, 93, 72, 91, 70, 96, 97, 101, 82,
   83, 104, 100, 102, 99, 88, 89, 112, 113, 106, 107, 94, 95, 110, 108, 98, 105, 103, 114, 118, 111, 117, 109, 115, 116, 119
}; // 1.07 kb
static const cart_index_t iCartYY_ab14_a13[315] = {
   0, 4, 6, 11, 12, 3, 42, 5, 43, 9, 10, 16, 46, 18, 48, 23, 24, 15, 52, 17, 54, 21, 22, 28, 58, 30,
   60, 35, 36, 27, 64, 29, 66, 33, 34, 39, 70, 40, 72, 44, 49, 50, 76, 77, 45, 51, 47, 53, 55, 56, 61, 62,
   82, 83, 88, 89, 57, 63, 59, 65, 67, 68, 73, 74, 94, 95, 